/*
    ettercap -- PlugIns module

    Copyright (C) 2001  ALoR <alor@users.sourceforge.net>, NaGA <crwm@freemail.it>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

    $Id: ec_plugins.c,v 1.2 2001/09/06 20:07:42 alor Exp $
*/

#include "include/ec_main.h"

#ifdef PERMIT_PLUGINS   // don't compile if plug-in not supported

#include <dlfcn.h>
#include <unistd.h>
#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <sys/time.h>
#include <dirent.h>
#ifdef SOLARIS
   #include "./missing/scandir.h"
#endif


#include "include/ec_plugins.h"
#include "include/ec_error.h"

#ifdef HAVE_NCURSES
   #ifdef HAVE_NCURSES_H
      #include <ncurses.h>
   #else
      #include <curses.h>
   #endif
   #include "include/ec_interface_plugins.h"
   extern WINDOW *plugin_window, *main_window;
   extern short scroll_yp;
   extern int W_MAINX1, W_MAINY1, W_MAINX2, W_MAINY2;
#endif

#ifdef DEBUG
   #include "include/ec_debug.h"
#endif

#ifdef OPENBSD
// The below define is a lie since we are really doing RTLD_LAZY since the
// system doesn't support RTLD_NOW.
   #define RTLD_NOW DL_LAZY
   #define RTLD_GLOBAL 0
#endif

PLUGINS *Plugins_list = NULL;

// protos....

int Plugin_Input(char *string, short size, short mode);
void Plugin_Output(char *message, ...);
void Plugin_SYS_Output(char *message, ...);

void Plugin_RunPlugIn(char *name, char **argv);
int Plugin_MakePlugList(void);
char *Plugin_Getname(char *plugin);
int Plugin_Input_GetChar_Block(char *string, short size);
int Plugin_Input_GetChar_NonBlock(char *string, short size);

// --------------------------

void Plugin_RunPlugIn(char *name, char **argv)
{
   static void *handle;
   char *Ette_Ver;
   void (*Start)(char **);

#ifdef DEBUG
   Debug_msg("Plugin_RunPlugIn -- starting %s plugin", name);
#endif

   handle = dlopen(name, RTLD_NOW | RTLD_GLOBAL);
   if (!handle)
   {
      #ifdef DEBUG
         Debug_msg("Plugin_RunPlugIn -- dlopen() | %s", dlerror());
      #endif
      Plugin_SYS_Output("Error Opening %s... (dlopen)", name);
      return;
   }

#if defined(OPENBSD) || defined(MACOSX)
   Ette_Ver = dlsym(handle, "_Ettercap_Version");
#else
   Ette_Ver = dlsym(handle, "Ettercap_Version");
#endif

   if (Ette_Ver == NULL || strcmp(Ette_Ver, VERSION) )
   {
      Plugin_SYS_Output("This plugin is NOT compatible with ettercap %s", VERSION);
      Plugin_SYS_Output("It was compiled under ettercap %s", Ette_Ver);
      dlclose(handle);
      return;
   }

#if defined(OPENBSD) || defined(MACOSX)
   Start = dlsym(handle, "_PlugIn_Start");
#else
   Start = dlsym(handle, "PlugIn_Start");
#endif

   if (Start == NULL)
   {
      #ifdef DEBUG
         Debug_msg("Plugin_RunPlugIn -- dlsym() | %s", dlerror());
      #endif
      Plugin_SYS_Output("Error Opening %s... (dlsym)", name);
      return;
   }

   Plugin_SYS_Output("Starting %s plugin...", name);

   (*Start)(argv);   // lounch the plugin

   if (!Options.normal)
      Plugin_SYS_Output("%s plugin ended.  (press 'q' to quit...)", name);
   else
      Plugin_SYS_Output("%s plugin ended.", name);

   dlclose(handle);

#ifdef DEBUG
   Debug_msg("Plugin_RunPlugIn -- shutting down %s plugin", name);
#endif


}


int Plugin_Input(char *string, short size, short mode)
{
   int nchars;

   if (mode == P_BLOCK)
      nchars = Plugin_Input_GetChar_Block(string, size);
   else // P_NONBLOCK
      nchars = Plugin_Input_GetChar_NonBlock(string, size);

   return nchars;

}



void Plugin_Output(char *message, ...)
{
   va_list ap;
   char plug_output[500];  // should be enough

   va_start(ap, message);
   vsnprintf(plug_output, 500, message, ap);
   va_end(ap);

#ifdef HAVE_NCURSES
   if (!Options.normal)
   {
      wprintw(plugin_window, "%s", plug_output);
      pnoutrefresh(plugin_window, scroll_yp, 0, W_MAINY1 + 3, 3, W_MAINY2 - 3 , W_MAINX2 - 2);
      doupdate();
   }
   else
#endif
   {
      fprintf(stdout, "%s", plug_output);
      fflush(stdout);
   }

}


void Plugin_SYS_Output(char *message, ...)
{
   va_list ap;
   char plug_output[500];  // should be enough

   va_start(ap, message);
   vsnprintf(plug_output, 500, message, ap);
   va_end(ap);


#ifdef HAVE_NCURSES
   if (!Options.normal)
   {
      #define HELP_COLOR 6
      #define NORM_COLOR 1
      wbkgdset(plugin_window, COLOR_PAIR(HELP_COLOR)); wattron(plugin_window, A_BOLD);
      wprintw(plugin_window, "\n%s\n", plug_output);
      wbkgdset(plugin_window, COLOR_PAIR(NORM_COLOR)); wattroff(plugin_window, A_BOLD);
      pnoutrefresh(plugin_window, scroll_yp, 0, W_MAINY1 + 3, 3, W_MAINY2 - 3 , W_MAINX2 - 2);
      doupdate();
   }
   else
#endif
   {
      fprintf(stdout, "\n\033[36m %s \033[0m\n", plug_output);
      fflush(stdout);
   }

}



int Plugin_MakePlugList(void)
{
   struct dirent **namelist;
   static void *handle;
   char name[100];
   char *Plug_Info;
   char *Ette_Ver;
   short *Plug_Ver;
   short i, k, j=0;

   #define MAX_PLUGIN 100

#ifdef DEBUG
      Debug_msg("Plugin_MakePlugList -- %s", INSTALL_PATH);
#endif

   if ( (Plugins_list = (PLUGINS *)calloc(MAX_PLUGIN, sizeof(PLUGINS))) == NULL)
      ERROR_MSG("calloc()");

   for(k=0; k<2; k++)
   {
      short n=0, m=0;

      if (k)
         m = scandir(INSTALL_PATH, &namelist, 0, alphasort);
      else
         n = scandir(".", &namelist, 0, alphasort);

      if ( (n > 0) || (m > 0) )
      {
         short num;
         if (k) num = m;
         else num = n;

          for(i=0; i<num; i++)
            if ( strstr(namelist[i]->d_name, "ec_") && strstr(namelist[i]->d_name, ".so") )
            {
               Plugins_list[j].name = strdup(namelist[i]->d_name);

               if (k) strcpy(name, INSTALL_PATH);
               else strcpy(name, "./");

               Plugins_list[j].path = strdup(name);

               strcat(name, namelist[i]->d_name);

               #ifdef DEBUG
                  Debug_msg("\t %s  %s", Plugins_list[j].path, Plugins_list[j].name );
               #endif

               handle = dlopen(name, RTLD_NOW | RTLD_GLOBAL);
               if (handle)
               {

                  #if defined(OPENBSD) || defined(MACOSX)
                     Plug_Info = dlsym(handle, "_PlugIn_Info");
                  #else
                     Plug_Info = dlsym(handle, "PlugIn_Info");
                  #endif

                  if (Plug_Info == NULL)
                     Plugins_list[j].description = strdup("No info available...");
                  else
                     Plugins_list[j].description = strdup(Plug_Info);

                  #if defined(OPENBSD) || defined(MACOSX)
                     Plug_Ver = dlsym(handle, "_PlugIn_Version");
                  #else
                     Plug_Ver = dlsym(handle, "PlugIn_Version");
                  #endif

                  if (Plug_Ver == NULL)
                     Plugins_list[j].version = 0;
                  else
                     Plugins_list[j].version = (float)(*Plug_Ver)/10;

                  #if defined(OPENBSD) || defined(MACOSX)
                     Ette_Ver = dlsym(handle, "_Ettercap_Version");
                  #else
                     Ette_Ver = dlsym(handle, "Ettercap_Version");
                  #endif

                  if (Ette_Ver == NULL || strcmp(Ette_Ver, VERSION) )
                     Plugins_list[j].description = strdup("Compiled under a different ettercap version");

                  dlclose(handle);
               }
               else
               {
                  Plugins_list[j].version = -1;
                  Plugins_list[j].description = strdup("Seems to be an invalid plugin...");
               }

               j++;
               if (j == MAX_PLUGIN) return j;
            }
      }
   }

   return j;

}

char *Plugin_Getname(char *plugin)     // parses  "ec_dummy.so"  and return  "dummy"
{
   char *name;

   name = strdup(plugin);
   name[strlen(name)-3] = 0;

   return name+3;
}


int Plugin_Input_GetChar_Block(char *string, short size)   // the real input
{
   int nchars=0;

   #ifdef HAVE_NCURSES
      if (!Options.normal)
      {
         pnoutrefresh(plugin_window, scroll_yp, 0, W_MAINY1 + 3, 3, W_MAINY2 - 3 , W_MAINX2 - 2);
         doupdate();
      }
   #endif

   memset(string, 0, size);

   loop
   {
      #ifdef HAVE_NCURSES
         if (!Options.normal)
         {
            int c = 0;
            static int p_text = 0;
            static char text[200] = "";

            c = wgetch(main_window);
            if ( c == 8 || c == 263 || c == KEY_BACKSPACE)  // BACKSPACE
            {
               int x=0,y=0;
               getyx(plugin_window, y, x);
               wmove(plugin_window, y, --x);
               pechochar(plugin_window, ' ');
               wmove(plugin_window, y, x);
               text[p_text] = 0;
               if ( p_text > 0 ) p_text--;
            }
            else
            {
               pechochar(plugin_window, c);
               if (p_text < 200) text[p_text++] = c;
            }

            if ( c == '\n')
            {
               strcpy(string, text);
               memset(text, 0, 200);
               nchars = p_text;
               p_text = 0;
               string[strlen(string)-1] = 0;  // remove the \n
               break;
            }
         }
         else
      #endif
         {
            read(0, string + nchars, 1);
            if (string[nchars] == 8)
            {
               nchars -= 2;
               string[nchars+1] = 0;
            }
         }

      if (nchars++ >= size || string[nchars-1] == '\n')
         break;
   }

   return nchars;

}


int Plugin_Input_GetChar_NonBlock(char *string, short size)   // the real input
{
   int nchars=0;
   fd_set msk_fd;
   struct timeval TimeOut;

   memset(&TimeOut, 0, sizeof(TimeOut));  //  timeout = 0
   FD_ZERO(&msk_fd);

   loop
   {
      FD_SET(0, &msk_fd);

      select(FOPEN_MAX, &msk_fd, (fd_set *) 0, (fd_set *) 0, &TimeOut);

      if (FD_ISSET(0, &msk_fd))
      {
         #ifdef HAVE_NCURSES
            if (!Options.normal)
            {
               int c = 0;
               static int p_text = 0;
               static char text[200] = "";

               c = wgetch(main_window);
               if ( c == 8 || c == 263 || c == KEY_BACKSPACE)  // BACKSPACE
               {
                  int x=0,y=0;
                  getyx(plugin_window, y, x);
                  wmove(plugin_window, y, --x);
                  pechochar(plugin_window, ' ');
                  wmove(plugin_window, y, x);
                  text[p_text] = 0;
                  if ( p_text > 0 ) p_text--;
               }
               else
               {
                  pechochar(plugin_window, c);
                  if (p_text < 200) text[p_text++] = c;
               }

               if ( c == '\n')
               {
                  strcpy(string, text);
                  memset(text, 0, 200);
                  nchars = p_text;
                  p_text = 0;
                  string[strlen(string)-1] = 0;  // remove the \n
                  break;
               }
            }
            else
         #endif
            {
               read(0, string + nchars, 1);
               if (string[nchars] == 8)
               {
                  nchars -= 2;
                  string[nchars+1] = 0;
               }
            }

         if (nchars++ >= size || string[nchars-1] == '\n')
            break;
      }
      else                    // no input
          break;
   }

   return nchars;

}



#endif   // PERMIT_PLUGINS

/* EOF */
