/*
    ettercap -- data decoding module

    Copyright (C) 2001  ALoR <alor@users.sourceforge.net>, NaGA <crwm@freemail.it>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

    $Id: ec_decodedata.c,v 1.6 2001/09/13 20:15:48 alor Exp $
*/

#include "include/ec_main.h"

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#ifdef HAVE_CTYPE_H
   #include <ctype.h>
#endif

#include "include/ec_inet_structures.h"
#include "include/ec_inet.h"
#include "include/ec_error.h"

#ifdef DEBUG
   #include "include/ec_debug.h"
#endif

typedef struct
{
   int port;
   char proto;
   char desc[18];
   struct database *next;
} database;

database *d_list = NULL;

typedef struct
{
   char fingerprint[FINGER_LEN+1];
   char os[60];
   struct os_database *next;
} os_database;

os_database *os_list = NULL;

#ifndef HAVE_CTYPE_H
   int isprint(int c);
#endif
int Decodedata_MakeConnectionList(CONNECTION data);
int Decodedata_RefreshConnectionList(void);
void Decodedata_UpdateInfo(CONNECTION *ptr, CONNECTION *data);

int Decodedata_MakePassiveList(PASSIVE_DATA data);
int Decodedata_FreePassiveList(void);
void Decodedata_UpdatePassiveInfo(PASSIVE_DATA *ptr, PASSIVE_DATA *data);

int Decodedata_GetPassiveOS(char *fingerprint, char *os);

char * Decodedata_GetType(char proto, int port1, int port2);
char * Decodedata_GetAsciiData(char *buffer, int buff_len);
char * Decodedata_GetHexData(char *buffer, int buff_len, short dimX);
char * Decodedata_GetEnhanchedHexData(char *buffer, int buff_len, short cr);
char * Decodedata_TCPFlags(char flags);

void Decodedata_ConvertPassiveToHost(void);
void Decodedata_Passive_SortList(void);
int Decodedata_Compare_Host(PASSIVE_DATA *h1, PASSIVE_DATA *h2);

//--------------------------

#ifndef HAVE_CTYPE_H

   int isprint(int c)
   {
      return ( (c>31 && c<127) ? 1 : 0 );
   }

#endif


int Decodedata_MakeConnectionList(CONNECTION data)
{
   int num_conn = 1;
   char filter[45];
   char cur_conn[45];
   char rev_cur_conn[45];
   char found = 0;
   CONNECTION *ptr;

   sprintf(filter, "%c%s%d%s%d", data.proto, data.source_ip, data.source_port, data.dest_ip, data.dest_port);

   if (number_of_connections != 0)
   {
      for(ptr=&Conn_Between_Hosts[number_of_connections-1]; num_conn <= number_of_connections; ptr--)
      {
         sprintf(cur_conn, "%c%s%d%s%d", ptr->proto, ptr->source_ip, ptr->source_port, ptr->dest_ip, ptr->dest_port);
         sprintf(rev_cur_conn, "%c%s%d%s%d", ptr->proto, ptr->dest_ip, ptr->dest_port, ptr->source_ip, ptr->source_port);

         num_conn++;

         if (!strcmp(filter, cur_conn))
         {
            if (ptr->proto == 'T')
            {
               ptr->source_seq = data.source_seq;
               ptr->flags = data.flags;
            }
            found = 1;
         }

         if (!strcmp(filter, rev_cur_conn))
         {
            if (ptr->proto == 'T')
            {
               ptr->dest_seq = data.dest_seq;
               ptr->flags = data.flags;
            }
            found = 1;
         }

         if (found)
         {
            Decodedata_UpdateInfo(ptr, &data);

            if (ptr->proto == 'T')
            {
               if (ptr->flags & TH_RST)
                  strcpy(ptr->status,  "KILLED");
               else if (ptr->flags & TH_SYN)
                  strcpy(ptr->status,  "OPENING");
               else if (ptr->flags & TH_FIN)
                  strcpy(ptr->status,  "CLOSING");
               else if (ptr->flags & TH_PSH)
                  strcpy(ptr->status,  "ACTIVE");
               else
                  if (!strcmp(ptr->status, "CLOSING"))     // FIN ACK
                     strcpy(ptr->status,  "CLOSED");       // ACK
                  else if (strcmp(ptr->status, "CLOSED"))
                     strcpy(ptr->status,  "silent");
            }
            else if (ptr->proto == 'U')
            {
               strcpy(ptr->status,  "  UDP ");
            }

            if (strcmp(data.type, ""))
               strcpy(ptr->type, data.type);

            return number_of_connections;

         }
      }
   }

   if (!found)
   {
      #ifdef DEBUG
         Debug_msg("Decodedata_MakeConnectionList - new node ! %d ! %c %s:%d - %s:%d ", num_conn, data.proto, data.source_ip, data.source_port, data.dest_ip, data.dest_port);
      #endif

      Conn_Between_Hosts = (CONNECTION *)realloc(Conn_Between_Hosts, num_conn*sizeof(CONNECTION));
      if ( Conn_Between_Hosts == NULL )
         ERROR_MSG("realloc()");
      else
         memset(&Conn_Between_Hosts[num_conn-1], 0, sizeof(CONNECTION));

      memcpy(Conn_Between_Hosts[num_conn-1].source_ip, &data.source_ip, sizeof(data.source_ip));
      memcpy(Conn_Between_Hosts[num_conn-1].dest_ip, &data.dest_ip, sizeof(data.dest_ip));

      Inet_PutMACinString(Conn_Between_Hosts[num_conn-1].source_mac, data.source_mac);
      Inet_PutMACinString(Conn_Between_Hosts[num_conn-1].dest_mac, data.dest_mac);

      Conn_Between_Hosts[num_conn-1].source_port = data.source_port;
      Conn_Between_Hosts[num_conn-1].dest_port = data.dest_port;
      Conn_Between_Hosts[num_conn-1].proto = data.proto;

      if (Conn_Between_Hosts[num_conn-1].proto == 'T')
      {
         Conn_Between_Hosts[num_conn-1].source_seq = data.source_seq;
         Conn_Between_Hosts[num_conn-1].dest_seq = data.dest_seq;
         Conn_Between_Hosts[num_conn-1].flags = data.flags;

         if (Conn_Between_Hosts[num_conn-1].flags & TH_RST)
            strcpy(Conn_Between_Hosts[num_conn-1].status, "KILLED");
         else if (Conn_Between_Hosts[num_conn-1].flags & TH_FIN)
            strcpy(Conn_Between_Hosts[num_conn-1].status, "CLOSING");
         else if (Conn_Between_Hosts[num_conn-1].flags & TH_SYN)
            strcpy(Conn_Between_Hosts[num_conn-1].status, "OPENING");
         else if (Conn_Between_Hosts[num_conn-1].flags & TH_PSH)
            strcpy(Conn_Between_Hosts[num_conn-1].status, "ACTIVE");
         else
            strcpy(Conn_Between_Hosts[num_conn-1].status, "silent");
      }
      else if (Conn_Between_Hosts[num_conn-1].proto == 'U')
         strcpy(Conn_Between_Hosts[num_conn-1].status, "  UDP ");


      if (strcmp(data.type, ""))
         strcpy(Conn_Between_Hosts[num_conn-1].type, data.type);
      else
         strncpy(Conn_Between_Hosts[num_conn-1].type, Decodedata_GetType(Conn_Between_Hosts[num_conn-1].proto,
                                                                         Conn_Between_Hosts[num_conn-1].source_port,
                                                                         Conn_Between_Hosts[num_conn-1].dest_port),
                                                                         18);

      if (Conn_Between_Hosts[num_conn-1].proto == 'T')
      {
         if (!(Conn_Between_Hosts[num_conn-1].flags & TH_SYN) &&
              ( data.dest_port == 23 || data.source_port == 23 ||
                data.dest_port == 513 || data.source_port == 513) )
                  Conn_Between_Hosts[num_conn-1].user[1] = -1;  // flag for the "waiting for syn" only for telnet and rlogin

         Decodedata_UpdateInfo(&Conn_Between_Hosts[num_conn-1], &data);
      }
      else if (Conn_Between_Hosts[num_conn-1].proto == 'U')
      {
         if (strcmp(data.user, ""))
         {
            strtok(data.user, "\n");
            snprintf(Conn_Between_Hosts[num_conn-1].user, 30, "USER: %s", data.user);
         }
         if (strcmp(data.pass, ""))
         {
            strtok(data.pass, "\n");
            snprintf(Conn_Between_Hosts[num_conn-1].pass, 30, "PASS: %s", data.pass);
         }
         if (strcmp(data.info, "")) strncpy(Conn_Between_Hosts[num_conn-1].info, data.info, 100);
      }

   }

   return num_conn;

}


void Decodedata_UpdateInfo(CONNECTION *ptr, CONNECTION *data)
{

   if (ptr->user[1] != -1)    // waiting for syn for some protocols like telnet
   {
      if ( ptr->user[0] == 0 )      // the string is under construction
      {
         strncpy(ptr->user + 1 + strlen(ptr->user+1), data->user, 29 - strlen(ptr->user+1) );
         if (strchr(data->user, '\n'))
         {
            char str[30];

            ptr->user[0] = ' ';
            strtok(ptr->user, "\n");

            if ( data->dest_port == 23 || data->source_port == 23 ||
                 data->dest_port == 513 || data->source_port == 513 )
               data->pass[0] = 0;        // evil workaround for telnet... we assume that pass always come AFTER login

            snprintf(str, 30, "USER:%s", ptr->user);
            strcpy(ptr->user, str);
         }
      }
      if ( ptr->user[0] != 0 && ptr->pass[0] == 0 )      // the string is under construction
      {
         strncpy(ptr->pass + 1 + strlen(ptr->pass+1), data->pass, 29 - strlen(ptr->pass+1) );
         if (strchr(data->pass, '\n'))
         {
            char str[30];

            ptr->pass[0] = ' ';
            strtok(ptr->pass, "\n");
            snprintf(str, 30, "PASS:%s", ptr->pass);
            strcpy(ptr->pass, str);
         }
      }

      if (strlen(data->info) && !strchr(ptr->info, '\n'))
         strlcat(ptr->info, data->info, 150);
   }
   else
   {
      if ( (data->flags & TH_SYN) ||
           ( data->dest_port != 23 && data->source_port != 23 &&
             data->dest_port != 513 && data->source_port != 513) )  // telnet and rlogin are enabled only on syn
               ptr->user[1] = 0;                                  // ok, start collecting user and pass...
   }
}



int Decodedata_RefreshConnectionList(void)
{

#ifdef DEBUG
   Debug_msg("Decodedata_RefreshConnectionList");
#endif

   if (Conn_Between_Hosts) free(Conn_Between_Hosts);
   Conn_Between_Hosts = NULL;
   number_of_connections = 0;

   return 0;
}



void Decodedata_UpdatePassiveInfo(PASSIVE_DATA *ptr, PASSIVE_DATA *data)
{

   char OS[60];

   if (!strcmp(ptr->ip, ""))
      strcpy(ptr->ip, data->ip);

   if (!strcmp(ptr->mac, "") && strcmp(data->type, "NL"))   // no mac for non local hosts
      strcpy(ptr->mac, data->mac);

   if (!strcmp(ptr->type, ""))
      strcpy(ptr->type, data->type);

   ptr->hop = data->hop;

   if (strcmp(data->fingerprint, ""))
   {
      if (!strcmp(ptr->fingerprint, ""))
         strcpy(ptr->fingerprint, data->fingerprint);

      if (!strcmp(strrchr(ptr->fingerprint, ':'), ":A") || !strcmp(ptr->os, "") )
      {
         strcpy(ptr->fingerprint, data->fingerprint);

         #ifdef DEBUG
            Debug_msg("Decodedata_UpdatePassiveInfo -- %s %s", data->ip, data->fingerprint);
         #endif

         if (Decodedata_GetPassiveOS(data->fingerprint, OS) == 0)
            strcpy(ptr->os, OS);
         else
         {
            ptr->os[0] = 0;
            strncpy(ptr->os + 1, OS, 58);
         }
      }
   }

   if (data->port != 0)
   {
      struct open_ports *current, *newelem;

      newelem = (struct open_ports *) calloc(1, sizeof(struct open_ports));
      newelem->port = data->port;

      if (data->proto == 'T')
      {
         if (LIST_EMPTY(&ptr->tcp_ports))
            LIST_INSERT_HEAD(&ptr->tcp_ports, newelem, next);
         else
            LIST_FOREACH(current, &ptr->tcp_ports, next)
            {
               if (current->port == data->port)
               {
                  strncpy(current->banner, data->banner, 149);
                  free(newelem);
                  break;
               }
               else if (current->port > data->port)
               {
                  if (current == LIST_FIRST(&ptr->tcp_ports))
                     LIST_INSERT_HEAD(&ptr->tcp_ports, newelem, next);
                  else
                     LIST_INSERT_BEFORE(current, newelem, next);
                  break;
               }
               else if (LIST_NEXT(current, next) == LIST_END(&ptr->tcp_ports))
               {
                  LIST_INSERT_AFTER(current, newelem, next);
                  break;
               }
            }
      }
      else if (data->proto == 'U')
      {
         if (LIST_EMPTY(&ptr->udp_ports))
            LIST_INSERT_HEAD(&ptr->udp_ports, newelem, next);
         else
            LIST_FOREACH(current, &ptr->udp_ports, next)
            {
               if (current->port == data->port)
               {
                  free(newelem);
                  break;
               }
               else if (current->port > data->port)
               {
                  if (current == LIST_FIRST(&ptr->udp_ports))
                     LIST_INSERT_HEAD(&ptr->udp_ports, newelem, next);
                  else
                     LIST_INSERT_BEFORE(current, newelem, next);
                  break;
               }
               else if (LIST_NEXT(current, next) == LIST_END(&ptr->udp_ports))
               {
                  LIST_INSERT_AFTER(current, newelem, next);
                  break;
               }
            }
      }
   }

}



int Decodedata_MakePassiveList(PASSIVE_DATA data)
{
   int num_conn = 1;
   char found = 0;
   PASSIVE_DATA *ptr;

   if (number_of_passive_hosts != 0)
   {
      for(ptr=Passive_Host; num_conn <= number_of_passive_hosts; ptr++)
      {
         num_conn++;

         if (!strcmp(data.type, "NL") && !strcmp(ptr->mac, data.mac))      // the packet is from the GW...
         {                                                                 // search and mark it
            strcpy(ptr->type, "GW");
            strcpy(ptr->gwforthis, data.ip);
         }

         if (!strcmp(ptr->ip, data.ip))
         {
            Decodedata_UpdatePassiveInfo(ptr, &data);
            found = 1;
         }
      }
   }

   if (!found)
   {
      int j;
      struct open_ports *cur;

      #ifdef DEBUG
         Debug_msg("Decodedata_MakePassiveList - new node ! %d ! %s - %s", num_conn, data.ip, data.fingerprint);
      #endif

      Passive_Host = (PASSIVE_DATA *)realloc(Passive_Host, num_conn*sizeof(PASSIVE_DATA));
      if ( Passive_Host == NULL )
         ERROR_MSG("realloc()");
      else
         memset(&Passive_Host[num_conn-1], 0, sizeof(PASSIVE_DATA));

      for (j=0; j<number_of_passive_hosts; j++)    // do a consistency check on the list because we have realloc'ed it.
      {
         cur = LIST_FIRST(&Passive_Host[j].tcp_ports);
         if (cur)
            cur->next.le_prev = &Passive_Host[j].tcp_ports.lh_first;
         cur = LIST_FIRST(&Passive_Host[j].udp_ports);
         if (cur)
            cur->next.le_prev = &Passive_Host[j].udp_ports.lh_first;
      }

      Decodedata_UpdatePassiveInfo(&Passive_Host[num_conn-1], &data);
   }
   else
      num_conn = number_of_passive_hosts;

   return num_conn;

}


int Decodedata_FreePassiveList(void)
{
   int i;
   struct open_ports *cur;

#ifdef DEBUG
   Debug_msg("Decodedata_FreePassiveList");
#endif

   for(i=0; i<number_of_passive_hosts; i++)
   {
      while (!LIST_EMPTY(&Passive_Host[i].tcp_ports))
      {
         cur = LIST_FIRST(&Passive_Host[i].tcp_ports);
         LIST_REMOVE(cur, next);
         free(cur);
      }
      while (!LIST_EMPTY(&Passive_Host[i].udp_ports))
      {
         cur = LIST_FIRST(&Passive_Host[i].udp_ports);
         LIST_REMOVE(cur, next);
         free(cur);
      }
   }

   if (Passive_Host) free(Passive_Host);
   Passive_Host = NULL;
   number_of_passive_hosts = 0;

   return 0;
}



char * Decodedata_GetHexData(char *buffer, int buff_len, short dimX)
{
   short octets;

   for(octets = 0; octets < dimX; octets++)
   {
      if ( (octets*3.5 + 12) >= dimX ) break;
   }

   if (octets > 16) octets = 16;
   if (octets % 2 == 1) octets--;

   return Decodedata_GetEnhanchedHexData(buffer, buff_len, octets);

}


char * Decodedata_GetEnhanchedHexData(char *buffer, int buff_len, short cr)
{
   static char *hexdata;
   int i, j, jm;
   int c, dim = 0;

   if (buff_len == 0) return "";

   c = cr*3.5 + 11;
   dim = c;

   for (i = 0; i < buff_len; i++)   // approximately
      if ( i % cr == 0)             // approximately
         dim += c;                  // approximately


   if (hexdata) free(hexdata);
   if ( (hexdata = (char *)calloc(dim, sizeof(char))) == NULL)
      ERROR_MSG("calloc()");

   // adapted from dsniff by Dug Song <dugsong@monkey.org>
   sprintf(hexdata,"\n");
   for (i = 0; i < buff_len; i += cr) {
           sprintf(hexdata, "%s %04x: ", hexdata, i );
           jm = buff_len - i;
           jm = jm > cr ? cr : jm;

           for (j = 0; j < jm; j++) {
                   if ((j % 2) == 1) sprintf(hexdata,"%s%02x ", hexdata, (unsigned char) buffer[i+j]);
                   else sprintf(hexdata,"%s%02x", hexdata, (unsigned char) buffer[i+j]);
           }
           for (; j < cr; j++) {
                   if ((j % 2) == 1) strcat(hexdata,"   ");
                   else strcat(hexdata,"  ");
           }
           strcat(hexdata," ");

           for (j = 0; j < jm; j++) {
                   c = buffer[i+j];
                   c = isprint(c) ? c : '.';
                   sprintf(hexdata,"%s%c", hexdata, c);
           }
           strcat(hexdata,"\n");
   }

   return hexdata;
}



char * Decodedata_GetAsciiData(char *buffer, int buff_len)
{

   int i = 0;

   if (buff_len == 0) return "";

   for(i = 0; i < buff_len; i++)
   {
      if ( !( isprint((int)buffer[i]) || buffer[i] == '\n' || buffer[i] == '\t') )
         buffer[i] = '.';
   }

   return buffer;
}


char * Decodedata_TCPFlags(char flags)
{
   static char string[8];
   char *p;

   memset(string, 0, 8);
   p = string;

   if (flags & TH_SYN) *p++ = 'S';
   if (flags & TH_FIN) *p++ = 'F';
   if (flags & TH_RST) *p++ = 'R';
   if (flags & TH_ACK) *p++ = 'A';
   if (flags & TH_PSH) *p++ = 'P';

   return string;

}


char * Decodedata_GetType(char proto, int port1, int port2)
{

   static char type[18];
   database *d_index;

   if (d_list == NULL)  // only the first time
   {
      FILE *f_ser;
      char line[1024], desc[18], stype[4];
      int port;

#ifdef DEBUG
   Debug_msg("Decodedata_GetType - loading from /etc/services");
#endif

      if ( (d_index = (database *)calloc(1,sizeof(database))) == NULL)
         ERROR_MSG("calloc()");

      d_list = d_index;

      if (!(f_ser = fopen ("/etc/services", "r")))
         ERROR_MSG("fopen(\"/etc/services\")");

      while (fgets (line, 1024, f_ser))
      {
         if ((sscanf (line, "%16s%u/%s", desc, &port, stype) == 3) && (!strstr (desc, "#")) )
         {
            if ( (d_index->next = ( struct database *) calloc (1, sizeof(database))) == NULL)
               ERROR_MSG("calloc()");

            d_index->port = port;
            if (strstr (stype, "tcp")) d_index->proto = 'T';
            if (strstr (stype, "udp")) d_index->proto = 'U';

            strcpy (d_index->desc, desc);

            d_index = (database *) d_index->next;
         }
      }

      fclose (f_ser);
      d_index->next = NULL;
   }

   d_index = d_list;
   for( ; d_index; d_index = (database *)d_index->next)
   {
      if ( d_index->proto == proto && (port1 == d_index->port || port2 == d_index->port) )
      {
         strcpy(type, d_index->desc);
         return type;
      }
   }

   return "";
}


int Decodedata_GetPassiveOS(char *fingerprint, char *os)
{

   os_database *os_index;
   int ret = -1;

//   #ifdef DEBUG
//      Debug_msg("Decodedata_GetPassiveOS - %s", fingerprint);
//   #endif

   if (!strcmp(fingerprint, "")) // no fingerprint, no os...
   {
      strcpy(os, "");
      return 1;
   }

   if (os_list == NULL)  // only the first time
   {
      FILE *f_os;
      char line[1024];
      char *ptr;

      if ( (os_index = (os_database *)calloc(1,sizeof(os_database))) == NULL)
         ERROR_MSG("calloc()");

      os_list = os_index;

      f_os = fopen( "./" OS_FILE, "r");
      if (f_os == NULL)
      {
         f_os = fopen( INSTALL_PATH OS_FILE, "r");
         if (f_os == NULL)
            Error_msg("\nCan't find " OS_FILE " in ./ or " INSTALL_PATH);
         else
         {
            #ifdef DEBUG
               Debug_msg("Decodedata_GetPassiveOS - loading from " INSTALL_PATH OS_FILE);
            #endif
         }
      }
      else
      {
         #ifdef DEBUG
            Debug_msg("Decodedata_GetPassiveOS - loading from ./" OS_FILE);
         #endif
      }

      while (fgets (line, 1024, f_os))
      {
         if ( (ptr = strchr(line, '#')) )
            *ptr = 0;

         if (!strlen(line))   // skip 0 length line
            continue;

         if ( (os_index->next = ( struct os_database *) calloc (1, sizeof(os_database))) == NULL)
            ERROR_MSG("calloc()");

         strncpy(os_index->fingerprint, line, FINGER_LEN);
         strncpy(os_index->os, line+FINGER_LEN+1, 60);
         os_index->os[59]=0;
         os_index->os[strlen(os_index->os)-1] = 0;

         os_index = (os_database *) os_index->next;

      }

      fclose (f_os);
      os_index->next = NULL;
   }

   os_index = os_list;
   for( ; os_index; os_index = (os_database *)os_index->next)
   {
      if ( strcmp(os_index->fingerprint, fingerprint) == 0)
      {
         strcpy(os, os_index->os);
         ret = 0;
         break;
      }
      if ( strcmp(os_index->fingerprint, fingerprint) > 0)  // take the nearest entry
      {
         strcpy(os, os_index->os);
         ret = 1;
         break;
      }
   }

   #ifdef DEBUG
      Debug_msg("Decodedata_GetPassiveOS - result: %d %s", ret, os);
   #endif

   return ret;
}




void Decodedata_ConvertPassiveToHost(void)
{
   int num_conn = 1;
   PASSIVE_DATA *ptr;

#ifdef DEBUG
   Debug_msg("Decodedata_ConvertPassiveToHost -- %d", number_of_hosts_in_lan);
#endif

   if (number_of_hosts_in_lan > 1) number_of_hosts_in_lan = 1;

   if (number_of_passive_hosts != 0)
   {
      for(ptr=Passive_Host; num_conn <= number_of_passive_hosts; ptr++)
      {
         num_conn++;

         if (strcmp(ptr->type, "NL") && strcmp(ptr->ip, Host_In_LAN[0].ip)) // don't add Non Local IPs and my ip
         {
            number_of_hosts_in_lan++;
            Host_In_LAN = (HOST *)realloc(Host_In_LAN, number_of_hosts_in_lan*sizeof(HOST));
            if ( Host_In_LAN == NULL )
               ERROR_MSG("realloc()");
            else
               memset(&Host_In_LAN[number_of_hosts_in_lan-1], 0, sizeof(HOST));

            strcpy(Host_In_LAN[number_of_hosts_in_lan-1].ip, ptr->ip);
            strcpy(Host_In_LAN[number_of_hosts_in_lan-1].mac, ptr->mac);
         }
      }
   }
#ifdef DEBUG
   Debug_msg("Decodedata_ConvertPassiveToHost -- %d host(s) converted ", number_of_hosts_in_lan );
#endif
}


int Decodedata_Compare_Host(PASSIVE_DATA *h1, PASSIVE_DATA *h2)
{
   u_long ip1, ip2;

   inet_aton(h1->ip, (struct in_addr *)&ip1);
   inet_aton(h2->ip, (struct in_addr *)&ip2);

   if (ntohl(ip1) < ntohl(ip2))
      return -1;
   else if (ntohl(ip1) == ntohl(ip2))
      return 0;
   else
      return 1;
}


void Decodedata_Passive_SortList(void)
{
   int i;
   struct open_ports *cur;

#ifdef DEBUG
   Debug_msg("Decodedata_Passive_SortList");
#endif

   qsort(Passive_Host, number_of_passive_hosts, sizeof(PASSIVE_DATA), (int (*)(const void *, const void *))Decodedata_Compare_Host);

   for (i=0; i<number_of_passive_hosts; i++)    // do a consistency check on the list
   {
      cur = LIST_FIRST(&Passive_Host[i].tcp_ports);
      if (cur)
         cur->next.le_prev = &Passive_Host[i].tcp_ports.lh_first;
      cur = LIST_FIRST(&Passive_Host[i].udp_ports);
      if (cur)
         cur->next.le_prev = &Passive_Host[i].udp_ports.lh_first;
   }

}


/* EOF */
