/*
    ettercap -- module for NON ncurses interface

    Copyright (C) 2001  ALoR <alor@users.sourceforge.net>, NaGA <crwm@freemail.it>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <time.h>
#include <sys/time.h>
#include <signal.h>

#include "include/ec_main.h"
#include "include/ec_error.h"
#include "include/ec_inet.h"
#include "include/ec_inet_structures.h"
#include "include/ec_decodedata.h"
#include "include/ec_dissector.h"
#include "include/ec_illithid.h"
#include "include/ec_doppleganger.h"
#include "include/ec_grell.h"
#include "include/ec_buffer.h"
#include "include/ec_plugins.h"
#include "include/ec_fingerprint.h"
#include "include/ec_logtofile.h"

#ifdef DEBUG
   #include "include/ec_debug.h"
#endif

#ifdef HAVE_TERMIOS_H

   #include <termios.h>

   struct termios old_tc;
   struct termios new_tc;

   void set_raw_term(void);
   void reset_term(void);

#endif

#define ASCII_VIEW      0
#define HEX_VIEW        1

extern char *Execute_Plugin;

char logtofile = 0;
FILE *logcollect;

// protos...
#ifdef PERMIT_PLUGINS
   void Simple_Plugin(void);
#endif
void Simple_HostList(void);
void Simple_Run(void);
void Simple_CheckForPoisoner(void);
void Simple_FingerPrint(void);
void Simple_CheckForSwitch(void);
void Simple_Demonize(void);
void Simple_CreateCertFile(void);

//---------------------------------

void Simple_HostList(void)
{
   int j;

#ifdef DEBUG
   Debug_msg("Simple_HostList");
#endif

   printf("\nHost in this LAN:\n\n");

   for(j=0; j<number_of_hosts_in_lan; j++)
   {
      printf("%3d) %s\t%s\t%s\n", j+1, Host_In_LAN[j].ip, Host_In_LAN[j].mac, Host_In_LAN[j].name);
   }

   printf("\n");
}


void Simple_FingerPrint(void)
{
   int IP;
   char MAC[6];
   char MACs[17];

#ifdef DEBUG
   Debug_msg("Simple_FingerPrint -- [%s]", Host_Dest.ip);
#endif

   IP = inet_addr(Host_Dest.ip);
   memcpy(MAC, Inet_MacFromIP(IP), 6);
   Inet_PutMACinString(MACs, MAC);

   printf("\n\nFingerprinting %s...\n\n", Host_Dest.ip);

   printf("\n\033[36mOperating System:\033[0m\n\n%s\n", Fingerprint_OS(Host_Dest.ip) );
   printf("\n\033[36mNetwork Adapter :\033[0m\n\n%s\n", Fingerprint_MAC(MACs) );

   printf("\n");
}



void Simple_CreateCertFile(void)
{

#ifdef DEBUG
   Debug_msg("Simple_CreateCertFile");
#endif

#if defined (HAVE_OPENSSL) && defined (PERMIT_HTTPS)

	// XXX - i am cheap and dirty...  as dug song said in the webmitm source code...
	// ripped form it

	if (system("openssl genrsa -out " CERT_FILE " 1024") != 0)
		Error_msg("ec_simple:%d system() | could not launch openssl (check the path)", __LINE__);

	if (system("openssl req -new -key " CERT_FILE " -out " CERT_FILE ".csr") != 0)
		Error_msg("ec_simple:%d system() | could not launch openssl (check the path)", __LINE__);

	if (system("openssl x509 -req -days 365 -in " CERT_FILE ".csr -signkey " CERT_FILE " -out " CERT_FILE ".new"))
		Error_msg("ec_simple:%d system() | could not launch openssl (check the path)", __LINE__);

	if (system("cat " CERT_FILE ".new >> " CERT_FILE) != 0)
		Error_msg("ec_simple:%d system() | could not launch openssl (check the path)", __LINE__);

	unlink(CERT_FILE ".new");
	unlink(CERT_FILE ".csr");

	fprintf(stdout, "\nOpenssl [%s] certificate generated in ./\n\n", CERT_FILE);
#else
	Error_msg("ettercap was compiled without HTTPS support...");
#endif
	exit(0);
}



void Simple_CheckForSwitch(void)
{
   short type;

   switch(type = Inet_CheckSwitch())
   {
      case 0: printf("\n Link Type: unknown\n");
              break;
      case 1: printf("\n Link Type: HUB\n");
              break;
      case 2: printf("\n Link Type: SWITCH\n");
              break;
   }

}



void Simple_CheckForPoisoner(void)
{
   SniffingHost *SniffList;
   int i;
   short found = 0;


#ifdef DEBUG
   Debug_msg("Simple_CheckForPoisoner");
#endif


   printf("\nChecking for poisoners...\n\n");

   SniffList = Inet_NoSniff();

   for (i=0; i<number_of_hosts_in_lan; i++)
   {
      if (SniffList[i].mode == 0) break;
      if (SniffList[i].mode == 1)
      {
         found = 1;
         printf("%s is replying for %s", Host_In_LAN[SniffList[i].Host_Index1].ip, Host_In_LAN[SniffList[i].Host_Index2].ip);
      }

      if (SniffList[i].mode == 2)
      {
         found = 1;
         printf("MAC of %s and %s are identical !",Host_In_LAN[SniffList[i].Host_Index1].ip,Host_In_LAN[SniffList[i].Host_Index2].ip);
      }

   }

   if (!found)
      printf("No poisoners found in this lan (seems to be safe)\n\n");

   free(SniffList);
}

#ifdef PERMIT_PLUGINS

void Simple_Plugin(void)
{
   char plugin[strlen(Execute_Plugin)+6];
   short i, n, found=-1;
   #include "include/ec_install_path.h"

#ifdef DEBUG
   Debug_msg("Simple_Plugin -- %s", Execute_Plugin);
#endif


   if (strcmp(Host_Source.name, ""))
      strcpy(Host_Source.ip, Inet_NameToIp(Host_Source.name));

   if (strcmp(Host_Dest.name, ""))
      strcpy(Host_Dest.ip, Inet_NameToIp(Host_Dest.name));

#ifdef DEBUG
   Debug_msg("Simple_Plugin -- [%s] [%s] [%s] [%s]", Host_Source.name, Host_Source.ip, Host_Dest.name, Host_Dest.ip );
#endif

   sprintf(plugin, "ec_%s.so", Execute_Plugin);

   n = Plugin_MakePlugList();

   if (!strcmp(Execute_Plugin, "list"))
   {
      #include "include/ec_install_path.h"

      if (n == 0)
         Error_msg("\n NO available plugin found in %s or ./ !!\n", path);

      fprintf(stdout, "\n Available Plugins :\n\n");

      for(i=0; i<n; i++)
         fprintf(stdout, "%2d) %8s     v %.1f -- %s\n", i+1, Plugin_Getname(Plugins_list[i].name), Plugins_list[i].version, Plugins_list[i].description);
      return;
   }

   for(i=0; i<n; i++)
   {
      if ( !strcmp(Plugins_list[i].name, plugin) ) found = i;
   }

   if (found == -1)
   {
      fprintf(stderr, "\nPlugin %s NOT found in %s or in ./ !\n", plugin, path);
      fprintf(stderr, "\nuse -p list to view the available ones.\n\n");
   }
   else
   {
      char toexec[strlen(plugin)+strlen(Plugins_list[found].path)];
      strcpy(toexec, Plugins_list[found].path);
      strcat(toexec, plugin);
      Plugin_RunPlugIn(toexec, NULL);
   }

}

#endif


#ifdef HAVE_TERMIOS_H

void set_raw_term()        // taken from readchar.c, by M. Andreoli (2000)
{
    tcgetattr(0, &old_tc);
    new_tc = old_tc;
    new_tc.c_lflag &= ~(ECHO | ICANON);   /* raw output */
    new_tc.c_cc[VTIME] = 1;

    tcsetattr(0, TCSANOW, &new_tc);
}

void reset_term()          // taken from readchar.c, by M. Andreoli (2000)
{
    tcsetattr(0, TCSANOW, &old_tc);
}

#endif

void Simple_Demonize(void)
{
 pid_t pid;

#ifdef DEBUG
   Debug_msg("Simple_Demonize");
#endif

   if((signal(SIGTTOU, SIG_IGN)) == SIG_ERR)
      Error_msg("ec_simple:%d signal() | ERRNO : %d | %s", __LINE__, errno, strerror(errno));

   if((signal(SIGTTIN, SIG_IGN)) == SIG_ERR)
      Error_msg("ec_simple:%d signal() | ERRNO : %d | %s", __LINE__, errno, strerror(errno));

   if((signal(SIGTSTP, SIG_IGN)) == SIG_ERR)
      Error_msg("ec_simple:%d signal() | ERRNO : %d | %s", __LINE__, errno, strerror(errno));

   if((signal(SIGHUP, SIG_IGN)) == SIG_ERR)
      Error_msg("ec_simple:%d signal() | ERRNO : %d | %s", __LINE__, errno, strerror(errno));

   if((pid = fork()) < 0)
      Error_msg("ec_simple:%d demonization failed | ERRNO : %d | %s", __LINE__, errno, strerror(errno));
   else if(pid != 0)
      Error_msg("ettercap demonized with PID: %d", pid);

   if(setsid() == -1)
      Error_msg("ec_simple:%d setsid() | ERRNO : %d | %s", __LINE__, errno, strerror(errno));


}


void Simple_Run(void)
{
   int Illithid_pid = 0, Dopple_pid = 0, Illithid_pid2 = 0;
   #if defined (HAVE_OPENSSL) && defined (PERMIT_HTTPS)
   	int Grell_pid = 0;
   #endif
   int i, index_source=-1, index_dest=-1;
   char view = ASCII_VIEW;
   char stop = 0;
   fd_set msk_fd;
   struct timeval TimeOut;
   SNIFFED_DATA data_from_illithid;
   CONNECTION data_from_illithid_conn;
   int datalen;
   char source[16], dest[16];
   char proto = 'T';
   TIME_DECLARE;

   if (Options.udp) proto = 'U';

#ifdef DEBUG
   Debug_msg("Simple_Run - name - [%s][%s]", Host_Dest.name, Host_Source.name);
   Debug_msg("Simple_Run -  IP  - [%s][%s]", Host_Dest.ip, Host_Source.ip);
   Debug_msg("Simple_Run - port - [%d][%d]", Host_Dest.port, Host_Source.port);
   Debug_msg("Simple_Run -  MAC - [%s][%s]", Host_Dest.mac, Host_Source.mac);
   if (Options.sniff)      Debug_msg("Simple_Run - %c - sniff", proto);
   if (Options.arpsniff)   Debug_msg("Simple_Run - %c - arpsniff", proto);
   if (Options.macsniff)   Debug_msg("Simple_Run - %c - macsniff", proto);
#endif

   if (Options.hexview)
      view = HEX_VIEW;

   if (Options.quiet)
      Simple_Demonize();

   if (Options.logtofile)
   {
      if (Options.collect)
      {
         logcollect = LogToFile_open_collect();
         logtofile = 2;
      }
      else
         logtofile = 1;
   }

   if (strcmp(Host_Source.name, ""))
   {
      strcpy(Host_Source.ip, Inet_NameToIp(Host_Source.name));
      strcpy(source, Host_Source.ip);
   }
   else
      strcpy(source, "ANY");

   if (strcmp(Host_Dest.name, ""))
   {
      strcpy(Host_Dest.ip, Inet_NameToIp(Host_Dest.name));
      strcpy(dest, Host_Dest.ip);
   }
   else
      strcpy(dest, "ANY");


   if (!Options.silent)
   {
      for(i=0; i<number_of_hosts_in_lan; i++)
      {
         if ( !strcmp(Host_Source.ip, Host_In_LAN[i].ip) )
         {
            index_source = i;
            strcpy(Host_Source.mac, Host_In_LAN[i].mac);
         }
         if ( !strcmp(Host_Dest.ip, Host_In_LAN[i].ip) )
         {
            index_dest = i;
            strcpy(Host_Dest.mac, Host_In_LAN[i].mac);
         }
      }

      if ( index_source < 0 && strcmp(Host_Source.name, ""))
         Error_msg("Source host %s (%s) not found !!", Host_Source.name, Host_Source.ip);


      if ( index_dest < 0 && strcmp(Host_Dest.name, ""))
         Error_msg("Dest host %s (%s) not found !!", Host_Dest.name, Host_Dest.ip);
   }

  	if ( !strcmp(Host_Source.ip, Host_Dest.ip) && strcmp(Host_Source.ip, "") )
  		Error_msg("SOURCE and DEST IP address must be different !!");

   if (Options.arpsniff)
      if ( (!strcmp(Host_Source.ip, Inet_MyIPAddress())) || (!strcmp(Host_Dest.ip, Inet_MyIPAddress())) )
         Error_msg("CAN'T arpsniff yourself !!");

   // it is ok...

   if (Options.arpsniff)      // Doppleganger born...
   {
   	Inet_DisableForwarding();
	   Dopple_pid = Doppleganger_Run(Options.netiface, Host_Source.ip, Host_Dest.ip, Host_Source.mac, Host_Dest.mac);
	}

   if (!Options.quiet) printf("\nPress 'h' for help...\n\n");

   pipe_with_illithid_data = Buffer_Create(1.0e6); // 1Mb in shared memory !!!


   if (Options.arpsniff)
   {
   	if (strcmp(Host_Source.ip, ""))	// arp based
   	{
      	if (!Options.quiet) printf(" Sniffing (ARP based) : %s:%d <--> %s <--> %s:%d\n\n", source, Host_Source.port, Inet_MyIPAddress(), dest, Host_Dest.port);
      	if (!Options.collect) Connection_Mode=0;
      	Illithid_pid = Illithid_ARPBased_GetConnections(Options.netiface, Host_Source.ip,  Host_Dest.ip, Host_Source.mac, Host_Dest.mac);
         #if defined (HAVE_OPENSSL) && defined (PERMIT_HTTPS)
         	Grell_pid = Grell_Run();
         #endif
      }
      else	// public arp
      {
			if (!Options.quiet) printf(" Sniffing (PUBLIC ARP) : %s:%d --> %s --> %s:%d  (half-duplex)\n\n", source, Host_Source.port, Inet_MyIPAddress(), dest, Host_Dest.port);
      	if (!Options.collect) Connection_Mode=0;
      	Illithid_pid = Illithid_PublicARP_GetConnections(Options.netiface, Host_Source.ip,  Host_Dest.ip, Host_Source.mac, Host_Dest.mac);
         #if defined (HAVE_OPENSSL) && defined (PERMIT_HTTPS)
         	Grell_pid = Grell_Run();
         #endif
      }
   }
   else if (Options.sniff)
   {
      if (!Options.quiet) printf(" Sniffing (IP based): %s:%d <--> %s:%d\n\n", source, Host_Source.port, dest, Host_Dest.port );
      if (!Options.collect) Connection_Mode=0;
      Illithid_pid = Illithid_IPBased_GetConnections(Options.netiface, Host_Source.ip, Host_Dest.ip);
   }
   else if (Options.macsniff)
   {
      if (!Options.quiet) printf(" Sniffing (MAC based): %s <--> %s\n\n", Host_Dest.mac, Host_Source.mac);
      if (!Options.collect) Connection_Mode=0;
      Illithid_pid = Illithid_MACBased_GetConnections(Options.netiface, Host_Dest.mac, Host_Source.mac);
   }


      memset(&TimeOut, 0, sizeof(TimeOut));  //  timeout = 0
      FD_ZERO(&msk_fd);

      if(!Options.quiet)
      {
         #ifdef HAVE_TERMIOS_H
            set_raw_term();   // non blocking stdin... yes this work !!
         #else
            #ifdef DEBUG
               Debug_msg("Simple_Run -- NO TERMIOS_H");
            #endif
            fcntl(0, F_SETFL, O_NONBLOCK);   // stdin non blocking... seems to be a non working method for me...
         #endif
      }

		TIME_START;

      loop
      {

         FD_SET(0, &msk_fd);

         select(FOPEN_MAX, &msk_fd, (fd_set *) 0, (fd_set *) 0, &TimeOut);

         if (FD_ISSET(0, &msk_fd))
         {
            char ch = 0;
            ch = getchar();
            switch(ch)
            {
               case 'A':
               case 'a':
                     if (!Options.collect)
                     {
                        printf("\n\nAscii mode...\n\n");
                        view = ASCII_VIEW;
                     }
                     break;

               case 'X':
               case 'x':
                     if (!Options.collect)
                     {
                        printf("\n\nHex mode...\n\n");
                        view = HEX_VIEW;
                     }
                     break;

               case 'L':
               case 'l':
                     if (logtofile)
                     {
                        printf("\n\nStop logging to file(s)...\n\n");
                        if (logtofile == 2) fclose(logcollect);
                        if (logtofile == 1) LogToFile_close();
                        logtofile = 0;
                     }
                     else
                     {
                        printf("\n\nLogging to file(s)...\n\n");
                        if (!Options.collect)
                           logtofile = 1;
                        else
                        {
                           logcollect = LogToFile_open_collect();
                           logtofile = 2;
                        }
                     }
                     break;

               case 'S':
               case 's':
                     if (stop == 1)
                     {
                        printf("\n\nLet's go...\n\n");
                        stop = 0;
                     }
                     else
                     {
                        printf("\n\nStopped...\n\n");
                        stop = 1;
                     }
                     break;

               case 'H':
               case 'h':
                     printf("\n\n[qQ]  - quit\n");
                     if (!Options.collect)
                     {
                        printf(    "[aA]  - dump data in ascii mode\n");
                        printf(    "[xX]  - dump data in hex mode\n");
                     }
                     printf(    "[lL]  - log all trafic to file(s)\n");
                     printf(    "[sS]  - stop/cont sniffing\n\n");
                     break;

               case 'Q':
               case 'q':
#ifdef HAVE_TERMIOS_H
                     if(!Options.quiet) reset_term();
#endif
                     printf("\n\n\nQuitting...\n\n");
                     kill(Illithid_pid, SIGTERM);
                     if (logtofile == 2) fclose(logcollect);
                     if (logtofile == 1) LogToFile_close();
                     if (Illithid_pid2) kill(Illithid_pid2,SIGTERM);
                     if (Dopple_pid) kill(Dopple_pid, SIGTERM);
                     #if defined (HAVE_OPENSSL) && defined (PERMIT_HTTPS)
         					if (Grell_pid) kill(Grell_pid, SIGTERM);
         				#endif
                     exit(0);
                     break;
            }
         }

         if (Options.collect)
            datalen = Buffer_Get(pipe_with_illithid, &data_from_illithid_conn, sizeof(CONNECTION));
         else
            datalen = Buffer_Get(pipe_with_illithid_data, &data_from_illithid, sizeof(SNIFFED_DATA));

         if (datalen>0)
         {
            time_t tt = time(NULL);
            struct tm *dd = localtime(&tt);

            if (Options.collect)
            {
               int i;
               number_of_connections = Decodedata_MakeConnectionList(data_from_illithid_conn);

               for (i=0; i < number_of_connections; i++)
               {
                  if (Conn_Between_Hosts[i].user[0] != 0 &&  Conn_Between_Hosts[i].pass[0] != 0)
                  {
                     if (logtofile == 2)
                     {
                        fprintf(logcollect, "\n\n%02d:%02d:%02d  %s:%d <--> %s:%d\t\t%s\n\n", dd->tm_hour, dd->tm_min, dd->tm_sec,
                                             Conn_Between_Hosts[i].source_ip,
                                             Conn_Between_Hosts[i].source_port,
                                             Conn_Between_Hosts[i].dest_ip,
                                             Conn_Between_Hosts[i].dest_port,
                                             Conn_Between_Hosts[i].type
                                             );

                        fprintf(logcollect, "%s\n", Conn_Between_Hosts[i].user);
                        fprintf(logcollect, "%s\n", Conn_Between_Hosts[i].pass);
                        if (strlen(Conn_Between_Hosts[i].info))
                           fprintf(logcollect, "\n%s\n", Conn_Between_Hosts[i].info);
                        fflush(logcollect);
                     }

                     if (!Options.quiet)
                     {
                        printf("\n\n%02d:%02d:%02d  %s:%d <--> %s:%d\t\t%s\n\n", dd->tm_hour, dd->tm_min, dd->tm_sec,
                              Conn_Between_Hosts[i].source_ip,
                              Conn_Between_Hosts[i].source_port,
                              Conn_Between_Hosts[i].dest_ip,
                              Conn_Between_Hosts[i].dest_port,
                              Conn_Between_Hosts[i].type
                              );

                        printf("%s\n", Conn_Between_Hosts[i].user);
                        printf("%s\n", Conn_Between_Hosts[i].pass);
                        if (strlen(Conn_Between_Hosts[i].info))
                          printf("\n%s\n", Conn_Between_Hosts[i].info);
                     }
                     memset(&Conn_Between_Hosts[i], 0, sizeof(CONNECTION));  // dont display in the future...
                  	#ifdef DEBUG
               			Debug_msg("\tConnection %d displayed and deleted...", i+1);
               		#endif
                  }
               }	// end for
               TIME_FINISH;
               if (TIME_ELAPSED >= 300)
               {
               	#ifdef DEBUG
               		Debug_msg("\tConnection List refreshed after %.0f minutes...", (TIME_ELAPSED)/60);
               	#endif
               	TIME_START;									// after 300 seconds (5 min)...
               	Decodedata_RefreshConnectionList(); // periodically refresh the list...
               }
            }
            else // !collect
            if (proto == data_from_illithid.proto && Illithid_ToBeSniffed("", Host_Source.port, "", Host_Dest.port, &data_from_illithid))
            {

               if (logtofile) LogToFile(&data_from_illithid);
               if (!stop && !Options.quiet )
               {
                  switch (view)
                  {
                     case ASCII_VIEW:
                        printf("\n\n\n%02d:%02d:%02d  %s:%d -> %s:%d\n\n%s", dd->tm_hour, dd->tm_min, dd->tm_sec,
                           data_from_illithid.source_ip,
                           data_from_illithid.source_port,
                           data_from_illithid.dest_ip,
                           data_from_illithid.dest_port,
                           Decodedata_GetAsciiData(data_from_illithid.data, data_from_illithid.datasize)
                        );
                        break;

                     case HEX_VIEW:
                        if (data_from_illithid.proto == 'T')
                           printf("\n\n%02d:%02d:%02d  %s:%d -> %s:%d | seq %lx ack %lx | flags %s |\n%s ", dd->tm_hour, dd->tm_min, dd->tm_sec,
                              data_from_illithid.source_ip,
                              data_from_illithid.source_port,
                              data_from_illithid.dest_ip,
                              data_from_illithid.dest_port,
                              data_from_illithid.seq,
                              data_from_illithid.ack_seq,
                              Decodedata_TCPFlags(data_from_illithid.flags),
                              Decodedata_GetHexData(data_from_illithid.data, data_from_illithid.datasize, 80)
                           );
                        else
                           printf("\n\n%02d:%02d:%02d  %s:%d -> %s:%d | UDP |\n%s ", dd->tm_hour, dd->tm_min, dd->tm_sec,
                              data_from_illithid.source_ip,
                              data_from_illithid.source_port,
                              data_from_illithid.dest_ip,
                              data_from_illithid.dest_port,
                              Decodedata_GetHexData(data_from_illithid.data, data_from_illithid.datasize, 80)
                           );
                        break;
                  }
                  fflush(stdout);
               }
            }
         }
         else
             usleep(1000);
      }
}

/* EOF */
