/*
    ettercap -- ncurses interface for connection list

    Copyright (C) 2001  ALoR <alor@users.sourceforge.net>, NaGA <crwm@freemail.it>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include "include/ec_main.h"

#ifdef HAVE_NCURSES  // don't compile if ncurses interface is not supported

#include <string.h>
#include <ncurses.h>
#include <unistd.h>
#include <signal.h>
#include <sys/time.h>

#include "include/ec_interface.h"
#include "include/ec_interface_sniff_data.h"
#include "include/ec_interface_factory.h"
#include "include/ec_decodedata.h"
#include "include/ec_illithid.h"
#include "include/ec_doppleganger.h"
#include "include/ec_inet.h"
#include "include/ec_inet_forge.h"
#include "include/ec_inet_structures.h"
#include "include/ec_buffer.h"

#ifdef DEBUG
   #include "include/ec_debug.h"
#endif

#ifdef PERMIT_PLUGINS
   #include "include/ec_interface_plugins.h"
   #include "include/ec_plugins.h"
#endif

#define BOTTOM_COLOR 1        // color schemes
#define TITLE_COLOR  2
#define MAIN_COLOR   3
#define POINT_COLOR  4
#define SEL_COLOR    5
#define HELP_COLOR   6
#define SNIFF_COLOR  7

#define KEY_RETURN   10       // they aren't defined in ncurses.h :(
#define KEY_CTRL_L   12


// protos...
void Interface_Sniff_Run(short mode);
void Interface_Sniff_PointItem(char direction);
void Interface_Sniff_InitList(void);
void Interface_Sniff_RefreshList(void);
void Interface_Sniff_KillConn(void);
void Interface_Sniff_ActiveDissecor(short mode);

// global variables

extern WINDOW *main_window, *bottom_window, *top_window;

extern int W_MAINX1, W_MAINY1, W_MAINX2, W_MAINY2;
extern int W_BOTTOMX1, W_BOTTOMY1, W_BOTTOMX2, W_BOTTOMY2;
extern int W_SELECTCONN;

short LeftMargin = 0;

int Conn_Base_Pointer = 0;
int Conn_Pointer = 0;
extern int Sel_Number;


//---------------------------


void Interface_Sniff_InitList(void)
{
   int j;
   int Conn_Top_Pointer;
   char info[100];

//#ifdef DEBUG
// Debug_msg("Interface_Sniff_InitList");
//#endif

   if (has_colors())
      wbkgdset(main_window, COLOR_PAIR(MAIN_COLOR));
   else
      wattroff(main_window,A_REVERSE);

   werase(main_window);

   if (number_of_connections == 0)     // no connection... no action... ;)
   {
      wnoutrefresh(main_window);
      doupdate();
      return;
   }

   Conn_Top_Pointer = (Conn_Base_Pointer+Sel_Number < number_of_connections) ? Conn_Base_Pointer + Sel_Number : number_of_connections ;

   for(j=Conn_Base_Pointer; j<Conn_Top_Pointer; j++)     // prints connections within the main_window height
   {
      wmove(main_window, j-Conn_Base_Pointer, LeftMargin );
      wprintw(main_window, "%3d) %15s:%d", j+1, Conn_Between_Hosts[j].source_ip, Conn_Between_Hosts[j].source_port);
      mvwprintw(main_window, j-Conn_Base_Pointer, 28," <--> %15s:%d", Conn_Between_Hosts[j].dest_ip , Conn_Between_Hosts[j].dest_port);
      wmove(main_window, j-Conn_Base_Pointer, LeftMargin + 56);  waddch(main_window, ACS_VLINE);
      wprintw(main_window, " %s", Conn_Between_Hosts[j].status);
      wmove(main_window, j-Conn_Base_Pointer, LeftMargin + 65);  waddch(main_window, ACS_VLINE);
      wprintw(main_window, " %s", Conn_Between_Hosts[j].type);
   }

   if (has_colors())
      wbkgdset(main_window, COLOR_PAIR(POINT_COLOR));
   else
      wattron(main_window,A_REVERSE);


   wmove(main_window, Conn_Pointer - Conn_Base_Pointer, LeftMargin );
   whline(main_window, ' ', W_SELECTCONN);
   wprintw(main_window, "%3d) %15s:%d", Conn_Pointer+1, Conn_Between_Hosts[Conn_Pointer].source_ip, Conn_Between_Hosts[Conn_Pointer].source_port);
   mvwprintw(main_window, Conn_Pointer-Conn_Base_Pointer, 28," <--> %15s:%d", Conn_Between_Hosts[Conn_Pointer].dest_ip , Conn_Between_Hosts[Conn_Pointer].dest_port);
   wmove(main_window, Conn_Pointer-Conn_Base_Pointer, LeftMargin + 56);  waddch(main_window, ACS_VLINE);
   wprintw(main_window, " %s", Conn_Between_Hosts[Conn_Pointer].status);
   wmove(main_window, Conn_Pointer - Conn_Base_Pointer, LeftMargin + 65); waddch(main_window, ACS_VLINE);
   wprintw(main_window, " %s", Conn_Between_Hosts[Conn_Pointer].type);

   werase(bottom_window);
   mvwprintw(bottom_window, 0, 1, " %s", Conn_Between_Hosts[Conn_Pointer].user);
   mvwprintw(bottom_window, 1, 1, " %s", Conn_Between_Hosts[Conn_Pointer].pass);
   snprintf(info, W_MAINX2 - 23, "%s", Conn_Between_Hosts[Conn_Pointer].info);
   mvwprintw(bottom_window, 1, 25, " %s", info);

   wnoutrefresh(bottom_window);
   wnoutrefresh(main_window);
   doupdate();
}



void Interface_Sniff_PointItem(char direction)
{

   int Old_Conn_Pointer;
   char info[100];

   if (number_of_connections == 0) return;   // no connection... no action... ;)

   Old_Conn_Pointer = Conn_Pointer;

   Conn_Pointer += direction;

   if (Conn_Pointer > number_of_connections -1 ) Conn_Pointer = number_of_connections - 1;
   if (Conn_Pointer < 0) Conn_Pointer = 0;


   if ( (Conn_Pointer - Conn_Base_Pointer + direction  >= Sel_Number) && (direction > 0) )      // scroll down
   {
      if (Conn_Base_Pointer + Sel_Number <= number_of_connections)
         Conn_Base_Pointer = (Conn_Base_Pointer + direction < number_of_connections) ? Conn_Base_Pointer + direction : number_of_connections - Sel_Number;

      Interface_Sniff_InitList();
   }
   else if ( (Conn_Pointer - Conn_Base_Pointer + direction < 0) && (direction < 0) )         // scroll up
   {
      if (Conn_Base_Pointer > 0)
         Conn_Base_Pointer = (Conn_Base_Pointer + direction > 0) ? Conn_Base_Pointer + direction : 0;

      Interface_Sniff_InitList();
   }


   if (has_colors())
      wbkgdset(main_window, COLOR_PAIR(MAIN_COLOR));
   else
      wattroff(main_window,A_REVERSE);


   if ( (Old_Conn_Pointer >= Conn_Base_Pointer) && (Old_Conn_Pointer <= Conn_Base_Pointer + Sel_Number -1)) // DON'T redraw previous selected item if it is out of view
   {
      wmove(main_window, Old_Conn_Pointer - Conn_Base_Pointer, LeftMargin);
      whline(main_window,' ', W_SELECTCONN);                         //deletes the previous position

      wprintw(main_window, "%3d) %15s:%d", Old_Conn_Pointer+1, Conn_Between_Hosts[Old_Conn_Pointer].source_ip, Conn_Between_Hosts[Old_Conn_Pointer].source_port);
      mvwprintw(main_window, Old_Conn_Pointer-Conn_Base_Pointer, 28," <--> %15s:%d", Conn_Between_Hosts[Old_Conn_Pointer].dest_ip , Conn_Between_Hosts[Old_Conn_Pointer].dest_port);
      wmove(main_window, Old_Conn_Pointer - Conn_Base_Pointer, LeftMargin + 56);  waddch(main_window, ACS_VLINE);
      wprintw(main_window, " %s", Conn_Between_Hosts[Old_Conn_Pointer].status);
      wmove(main_window, Old_Conn_Pointer - Conn_Base_Pointer, LeftMargin + 65);  waddch(main_window, ACS_VLINE);
      wprintw(main_window, " %s", Conn_Between_Hosts[Old_Conn_Pointer].type);
   }

   if (has_colors())
      wbkgdset(main_window, COLOR_PAIR(POINT_COLOR));
   else
      wattron(main_window,A_REVERSE);

   wmove(main_window, Conn_Pointer - Conn_Base_Pointer, LeftMargin);
   whline(main_window, ' ', W_SELECTCONN);                           //select new position

   wprintw(main_window, "%3d) %15s:%d", Conn_Pointer+1, Conn_Between_Hosts[Conn_Pointer].source_ip, Conn_Between_Hosts[Conn_Pointer].source_port);
   mvwprintw(main_window, Conn_Pointer-Conn_Base_Pointer, 28," <--> %15s:%d", Conn_Between_Hosts[Conn_Pointer].dest_ip , Conn_Between_Hosts[Conn_Pointer].dest_port);
   wmove(main_window, Conn_Pointer - Conn_Base_Pointer, LeftMargin + 56);  waddch(main_window, ACS_VLINE);
   wprintw(main_window, " %s", Conn_Between_Hosts[Conn_Pointer].status);
   wmove(main_window, Conn_Pointer - Conn_Base_Pointer, LeftMargin + 65);  waddch(main_window, ACS_VLINE);
   wprintw(main_window, " %s", Conn_Between_Hosts[Conn_Pointer].type);

   werase(bottom_window);
   mvwprintw(bottom_window, 0, 1, " %s", Conn_Between_Hosts[Conn_Pointer].user);
   mvwprintw(bottom_window, 1, 1, " %s", Conn_Between_Hosts[Conn_Pointer].pass);
   snprintf(info, W_MAINX2 - 23, "%s", Conn_Between_Hosts[Conn_Pointer].info);
   mvwprintw(bottom_window, 1, 25, " %s", info);

   wnoutrefresh(bottom_window);
   wnoutrefresh(main_window);
   doupdate();
}



void Interface_Sniff_RefreshList(void)
{
   WINDOW *message_window;

#ifdef DEBUG
   Debug_msg("Interface_Sniff_RefreshList");
#endif

   message_window = newwin(5, strlen("updating list...") + 4,0,0);
   mvwin(message_window, W_MAINY1+(W_MAINY2-W_MAINY1)/2-2, W_MAINX2/2 - (strlen("updating list...") + 4)/2 );
   wbkgdset(message_window, COLOR_PAIR(TITLE_COLOR));
   wattron(message_window, A_BOLD);
   box(message_window,ACS_VLINE,ACS_HLINE);
   mvwprintw(message_window,  2, 2, "updating list...");
   wnoutrefresh(message_window);
   doupdate();

   number_of_connections = Decodedata_RefreshConnectionList();
   Conn_Pointer = 0;
   Conn_Base_Pointer = 0;

   delwin(message_window);
   werase(bottom_window);
   wnoutrefresh(bottom_window);
   wnoutrefresh(main_window);

   doupdate();

}


void Interface_Sniff_KillConn(void)
{

   u_char *buf;
   char *MACD, MACS[6];
   u_long IPS, IPD;
   int sock;
//   char answer;

   if (Conn_Between_Hosts[Conn_Pointer].proto == 'U')
   {
      Interface_PopUp("Trying to kill an UDP connection ?!? Ehi Kiddie, go home !!");
      return;
   }


//   answer = Interface_PopUp("Do U really want to kill this connection (y/n)?");
//   if ((answer != 'y') && (answer != 'Y'))
//      return;


#ifdef DEBUG
   Debug_msg("Interface_Sniff_KillConn -- %s:%d -> %s:%d  %lu %lu", Conn_Between_Hosts[Conn_Pointer].source_ip,
                                                                    Conn_Between_Hosts[Conn_Pointer].source_port,
                                                                    Conn_Between_Hosts[Conn_Pointer].dest_ip,
                                                                    Conn_Between_Hosts[Conn_Pointer].dest_port,
                                                                    Conn_Between_Hosts[Conn_Pointer].source_seq,
                                                                    Conn_Between_Hosts[Conn_Pointer].dest_seq );
#endif

   Inet_GetMACfromString(Inet_MyMACAddress(), MACS);
   IPS = inet_addr(Conn_Between_Hosts[Conn_Pointer].source_ip);
   IPD = inet_addr(Conn_Between_Hosts[Conn_Pointer].dest_ip);

   MACD = Inet_MacFromIP(IPD);

   buf = Inet_Forge_packet( ETH_HEADER + IP_HEADER + TCP_HEADER );

   Inet_Forge_ethernet( buf, MACS, MACD, ETH_P_IP );

   Inet_Forge_ip( buf + ETH_HEADER, IPS, IPD, TCP_HEADER, 0xe77e, 0, IPPROTO_TCP );    // to source

   Inet_Forge_tcp( buf + ETH_HEADER + IP_HEADER, Conn_Between_Hosts[Conn_Pointer].source_port,
                                                 Conn_Between_Hosts[Conn_Pointer].dest_port,
                                                 Conn_Between_Hosts[Conn_Pointer].source_seq,
                                                 0,
                                                 TH_RST,
                                                 0, 0 );

   sock = Inet_OpenRawSock(Options.netiface);

   MACD = Inet_MacFromIP(IPS);

   Inet_SendRawPacket(sock, buf, ETH_HEADER + IP_HEADER + TCP_HEADER );

   Inet_Forge_ethernet( buf, MACS, MACD, ETH_P_IP );

   Inet_Forge_ip( buf + ETH_HEADER, IPD, IPS, TCP_HEADER, 0xe77e, 0, IPPROTO_TCP );    // to dest

   Inet_Forge_tcp( buf + ETH_HEADER + IP_HEADER, Conn_Between_Hosts[Conn_Pointer].dest_port,
                                                 Conn_Between_Hosts[Conn_Pointer].source_port,
                                                 Conn_Between_Hosts[Conn_Pointer].dest_seq,
                                                 0,
                                                 TH_RST,
                                                 0, 0 );

   Inet_SendRawPacket(sock, buf, ETH_HEADER + IP_HEADER + TCP_HEADER );

   Inet_Forge_packet_destroy( buf );

   strcpy(Conn_Between_Hosts[Conn_Pointer].status, "KILLED");

   close(sock);

}


void Interface_Sniff_ActiveDissecor(short mode)
{

#ifdef DEBUG
   Debug_msg("Interface_Sniff_ActiveDissecor %d", active_dissector);
#endif

   wbkgdset(top_window, COLOR_PAIR(HELP_COLOR));
   mvwprintw(top_window, 2, 31, "Active Dissector: ");

   switch(active_dissector)
   {
      case 1:  wbkgdset(top_window, COLOR_PAIR(BOTTOM_COLOR));
               mvwprintw(top_window, 2, 49, "ON ");
               break;

      case 0:  wbkgdset(top_window, COLOR_PAIR(BOTTOM_COLOR));
               mvwprintw(top_window, 2, 49, "OFF");
               break;
   }

   wnoutrefresh(top_window);
   doupdate();

}



void Interface_Sniff_Run(short mode)
{
   int KeyPress;
   int Illithid_pid = 0, Dopple_pid = 0;
   fd_set msk_fd;
   struct timeval TimeOut;
   CONNECTION data_from_illithid;

#ifdef DEBUG
   Debug_msg("Interface_Sniff_Run -- mode %d", mode);
#endif


   wbkgdset(main_window, COLOR_PAIR(MAIN_COLOR));
   werase(main_window);
   werase(bottom_window);

   wmove(bottom_window, 0, 0);
   wnoutrefresh(main_window);
   wnoutrefresh(bottom_window);
   doupdate();

   if (mode != ARPBASED )
      active_dissector = 0;

   Interface_Sniff_ActiveDissecor(mode);

   number_of_connections = 0;
   Interface_Sniff_RefreshList();
   Interface_Sniff_InitList();

   Buffer_Flush(pipe_with_illithid);

   switch (mode)
   {
      case ARPBASED:
      case PUBLICARP:
                     Dopple_pid = Doppleganger_Run(Options.netiface, Host_Source.ip, Host_Dest.ip, Host_Source.mac, Host_Dest.mac);

                     if ( mode == ARPBASED )
                        Illithid_pid = Illithid_ARPBased_GetConnections(Options.netiface, Host_Source.ip, Host_Dest.ip, Host_Source.mac, Host_Dest.mac);
                     else
                        Illithid_pid = Illithid_PublicARP_GetConnections(Options.netiface, Host_Source.ip, Host_Dest.ip, Host_Source.mac, Host_Dest.mac);
                     break;
      case IPBASED:
                     Illithid_pid = Illithid_IPBased_GetConnections(Options.netiface, Host_Source.ip, Host_Dest.ip);
                     break;
      case MACBASED:
                     Illithid_pid = Illithid_MACBased_GetConnections(Options.netiface, Host_Source.mac, Host_Dest.mac);
                     break;
   }


   memset(&TimeOut, 0, sizeof(TimeOut));  //  timeout = 0
   FD_ZERO(&msk_fd);

   loop
   {
      int datalen;
      FD_SET(0, &msk_fd);

      select(FOPEN_MAX, &msk_fd, (fd_set *) 0, (fd_set *) 0, &TimeOut);

      memset(&data_from_illithid, 0, sizeof(CONNECTION));
      datalen = Buffer_Get(pipe_with_illithid, &data_from_illithid, sizeof(CONNECTION));

      if (datalen > 0)
      {
         number_of_connections = Decodedata_MakeConnectionList(data_from_illithid);
         Interface_Sniff_InitList();
      }
      else  // workaround for the 100% CPU usage...
         usleep(5);

      if (FD_ISSET(0, &msk_fd))
      {

         KeyPress = wgetch(main_window);

         switch (KeyPress)
         {
            case KEY_DOWN:
                     Interface_Sniff_PointItem(1);
                     break;

            case KEY_UP:
                     Interface_Sniff_PointItem(-1);
                     break;

            case KEY_NPAGE:
                     Interface_Sniff_PointItem(Sel_Number-1);  //PGDOWN
                     break;

            case KEY_PPAGE:
                     Interface_Sniff_PointItem(-Sel_Number+1); //PGUP
                     break;

            case KEY_RETURN:
                     if (Conn_Between_Hosts[Conn_Pointer].source_ip == NULL) break;   // no connections
                     wmove(bottom_window, 0, 0);
                     if ( mode == PUBLICARP )
                         kill(Dopple_pid,SIGTERM);    // kill publicARP doppleganger...

                     kill(Illithid_pid, SIGUSR2);
                     Interface_Sniff_Data_Run(Conn_Between_Hosts[Conn_Pointer].source_ip,
                                              Conn_Between_Hosts[Conn_Pointer].source_port,
                                              Conn_Between_Hosts[Conn_Pointer].dest_ip,
                                              Conn_Between_Hosts[Conn_Pointer].dest_port,
                                              Conn_Between_Hosts[Conn_Pointer].source_mac,
                                              Conn_Between_Hosts[Conn_Pointer].dest_mac,
                                              Conn_Between_Hosts[Conn_Pointer].proto,
                                              Conn_Between_Hosts[Conn_Pointer].type,
                                              mode);
                     // wait and then...
                     kill(Illithid_pid, SIGUSR2);
                     if ( mode == PUBLICARP )
                         Dopple_pid = Doppleganger_Run(Options.netiface, Host_Source.ip, Host_Dest.ip, Host_Source.mac, Host_Dest.mac);

                     Interface_Sniff_InitList();
                     break;

            case 'K':
            case 'k':
                     if (Conn_Between_Hosts[Conn_Pointer].source_ip == NULL) break;   // no connections
                     Interface_Sniff_KillConn();      // with TH_RST !  i'm too tired to implement FIN handshaking
                     Interface_Sniff_InitList();
                     break;

            case 'D':
            case 'd':
                     {
                        char source[40];
                        char dest[40];
                        char mesg[80];
                        if (Conn_Between_Hosts[Conn_Pointer].source_ip == NULL) break;   // no connections
                        snprintf(source, 40, " %s (%s) <-->", Inet_HostName(Conn_Between_Hosts[Conn_Pointer].source_ip),
                                                              Conn_Between_Hosts[Conn_Pointer].source_ip);
                        snprintf(dest, 40, " %s (%s)", Inet_HostName(Conn_Between_Hosts[Conn_Pointer].dest_ip),
                                                       Conn_Between_Hosts[Conn_Pointer].dest_ip);
                        snprintf(mesg, 80, "%s%s", source, dest);
                        Interface_PopUp(mesg);
                     }
                     break;

            case 'R':
            case 'r':
                     Interface_Sniff_RefreshList();
                     Interface_Sniff_InitList();
                     break;

            case 'X':
            case 'x':
                     Interface_Factory_Run();
                     Interface_Redraw();
                     break;

#ifdef PERMIT_PLUGINS
	         case 'P':
   	      case 'p':
      	            Interface_Plugins_Run();
         	         break;
#endif

            case 'A':
            case 'a':
                     if (mode == ARPBASED)
                     {
                        kill(Illithid_pid, SIGUSR1);  // activate/deactivate special dissector
                        kill(getpid(), SIGUSR1);
                        Interface_Sniff_ActiveDissecor(mode);
                     }
                     else
                     {
                        Interface_PopUp("ACTIVE dissector is available only in ARP BASED mode !!");
                     }
                     break;

            case KEY_CTRL_L:  // CTRL+L refresh the screen
                     Interface_Redraw();
                     break;

            case KEY_F(1):
            case 'H':
            case 'h':{
                        static char *help[] = {
                           "[qQ][F10] - quit",
                           "[return]  - sniff the selected connection",
                           "[xX]      - Packet Forge",
                           "[aA]      - enable/disable ACTIVE password collectors",
                           "[kK]      - kill the connection (be careful !)",
#ifdef PERMIT_PLUGINS
                       		"[pP]      - run a plugin",
#endif
                           "[dD]      - resolve ip via DNS",
                           "[rR]      - refresh the list",
                           NULL};
                        Interface_HelpWindow(help);
                     }
                     Interface_Redraw();
                     break;

            case 'Q':
            case 'q':
                     kill(Illithid_pid, SIGTERM);
                     if (Dopple_pid) kill(Dopple_pid, SIGTERM);
                     #ifdef DEBUG
                        Debug_msg("Interface_Sniff_END");
                     #endif
                     if (Options.silent) Interface_WExit("They are safe!!  for now... ");
                     werase(bottom_window);
                     wnoutrefresh(bottom_window);
                     doupdate();
                     return;
                     break;

            case KEY_F(10):
                     #ifdef DEBUG
                        Debug_msg("Interface_Sniff_END");
                        Debug_msg("Interface_Run_END");
                     #endif
                     kill(Illithid_pid, SIGTERM);
                     if (Dopple_pid) kill(Dopple_pid, SIGTERM);
                     Interface_WExit("They are safe!!  for now... ");
                     break;
         }
      }

   }

}



#endif

/* EOF */
