/*
    ettercap -- ncurses interface for data injector

    Copyright (C) 2001  ALoR <alor@users.sourceforge.net>, NaGA <crwm@freemail.it>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include "include/ec_main.h"

#ifdef HAVE_NCURSES  // don't compile if ncurses interface is not supported

#include <string.h>
#include <ncurses.h>
#include <unistd.h>
#ifdef HAVE_CTYPE_H
   #include <ctype.h>
#endif
#ifdef HAVE_FORM
   #include <form.h>
#endif



#include "include/ec_interface.h"
#include "include/ec_interface_sniff_data.h"
#include "include/ec_decodedata.h"
#include "include/ec_error.h"
#include "include/ec_filterdrop.h"

#ifdef DEBUG
   #include "include/ec_debug.h"
#endif


#define BOTTOM_COLOR 1        // color schemes
#define TITLE_COLOR  2
#define MAIN_COLOR   3
#define POINT_COLOR  4
#define SEL_COLOR    5
#define HELP_COLOR   6
#define SNIFF_COLOR  7

#define CTRL(x)      ((x) & 0x1f)
#define QUIT         CTRL('Q')
#define ESCAPE       CTRL('[')
#define KEY_RETURN   10
#define KEY_INS		331

// protos...

void Interface_Inject_Redraw(void);
int Interface_Inject_Run(u_char *inject_data, char proto, char *app);
int Interface_Inject_Filter(DROP_FILTER *filter);

// global variables

extern WINDOW *main_window;
extern WINDOW *data_source_win, *data_dest_win, *data_source, *data_dest, *win_pointer;

extern int W_MAINX1, W_MAINY1, W_MAINX2, W_MAINY2;
extern int W_BOTTOMY2;



//---------------------------


void Interface_Inject_Redraw(void)
{

   Interface_Sniff_Data_Redraw();

#ifdef DEBUG
   Debug_msg("Interface_Inject_Redraw");
#endif


   doupdate();
}



int Interface_Inject_Run(u_char *inject_data, char proto, char *app)
{
   WINDOW *inject_window, *i_win;
   int dimY = 10;
   int dimX = 60;
   char inject_sequence[MAX_INJECT];
   int len;

#ifdef DEBUG
   Debug_msg("Interface_Inject_Run -- %c %s", proto, app);
#endif

   i_win = newwin(dimY+2, dimX+2, W_BOTTOMY2/2 - dimY/2, W_MAINX2/2 - dimX/2);
   inject_window = newwin(dimY, dimX, W_BOTTOMY2/2 - dimY/2 +1, W_MAINX2/2 - dimX/2 +1);
   wbkgdset(i_win, COLOR_PAIR(HELP_COLOR));
   wattron(i_win, A_BOLD);
   box(i_win,ACS_VLINE,ACS_HLINE);
   mvwprintw(i_win,  0, 2, "Type characters to be injected (max %d):", MAX_INJECT);
   wbkgdset(inject_window, COLOR_PAIR(BOTTOM_COLOR));
   wmove(inject_window, 0, 0);
   echo();
   scrollok(inject_window, TRUE);
   keypad(inject_window, TRUE);
   curs_set(TRUE);
   wnoutrefresh(i_win);
   wnoutrefresh(inject_window);
   doupdate();

   mvwgetnstr(inject_window, 1, 0, inject_sequence, MAX_INJECT);

#ifdef DEBUG
   Debug_msg("Interface_Inject_Run -- inject_sequence len -- [%d]", strlen(inject_sequence));
#endif

   noecho();
   curs_set(FALSE);
   delwin(i_win);
   delwin(inject_window);
   doupdate();

   len = FilterDrop_strescape(inject_data, inject_sequence);

#ifdef DEBUG
   Debug_msg("Interface_Inject_Run -- inject_data len -- [%d]", len);
#endif

   return len;

}


#ifdef HAVE_FORM


static FIELD *make_label(int frow, int fcol, NCURSES_CONST char *label)
{
   FIELD *f = new_field(1, strlen(label), frow, fcol, 0, 0);

   if (f)
   {
      set_field_buffer(f, 0, label);
      set_field_opts(f, field_opts(f) & ~O_ACTIVE);
   }
   return(f);
}

static FIELD *make_field(int frow, int fcol, int rows, int cols, bool secure)
{
   FIELD *f = new_field(rows, cols, frow, fcol, 0, secure ? 1 : 0);

   if (f) {
      set_field_back(f, A_UNDERLINE);
      set_field_userptr(f, (void *)0);
      field_opts_off(f, O_WRAP | O_BLANK);
   }
   return(f);
}

static void display_form(FORM *f)
{
   WINDOW *w;
   int rows, cols;

   scale_form(f, &rows, &cols);

   if ((w = newwin(rows+6, cols+4, W_BOTTOMY2/2 - rows/2 - 1, W_MAINX2/2 - cols/2)) != (WINDOW *)0)
   {
      wbkgdset(w, COLOR_PAIR(HELP_COLOR));
      set_form_win(f, w);
      set_form_sub(f, derwin(w, rows, cols, 1, 2));
      box(w, 0, 0);
      keypad(w, TRUE);
      mvwprintw(w, rows+2, 2, "Enter - set the filter  F10/ESC - exit form");
      mvwprintw(w, rows+3, 2, " ^N   - next field        ^P    - previous field");
      mvwprintw(w, rows+4, 2, " ^H   - del prev char     ^Y    - delete line");
   }

   if (post_form(f) != E_OK)
      wrefresh(w);
}

static void erase_form(FORM *f)
{
   WINDOW *w = form_win(f);
   WINDOW *s = form_sub(f);

   unpost_form(f);
   werase(w);
   wrefresh(w);
   delwin(s);
   delwin(w);
}


static int form_virtualize(FORM *f, WINDOW *w)
{

// "Defined form-traversal keys:   F10/ESC- exit form"
// "^N   -- go to next field       ^P  -- go to previous field"
// "Home -- go to first field      End -- go to last field"
// "^L   -- go to field to left    ^R  -- go to field to right"
// "^U   -- move upward to field   ^D  -- move downward to field"
// "^W   -- go to next word        ^B  -- go to previous word"
// "^S   -- go to start of field   ^E  -- go to end of field"
// "^H   -- delete previous char   ^Y  -- delete line"
// "^G   -- delete current word    ^C  -- clear to end of line"
// "^K   -- clear to end of field  ^X  -- clear field"


    static const struct {
      int code;
      int result;
    } lookup[] = {
         { CTRL('A'),     REQ_NEXT_CHOICE },
         { CTRL('B'),     REQ_PREV_WORD },
         { CTRL('C'),     REQ_CLR_EOL },
         { CTRL('D'),     REQ_DOWN_FIELD },
         { CTRL('E'),     REQ_END_FIELD },
         { CTRL('F'),     REQ_NEXT_PAGE },
         { CTRL('G'),     REQ_DEL_WORD },
         { CTRL('H'),     REQ_DEL_PREV },
         { CTRL('I'),     REQ_INS_CHAR },
         { CTRL('K'),     REQ_CLR_EOF },
         { CTRL('M'),     REQ_NEW_LINE },
         { CTRL('N'),     REQ_NEXT_FIELD },
         { CTRL('O'),     REQ_INS_LINE },
         { CTRL('P'),     REQ_PREV_FIELD },
         { CTRL('S'),     REQ_BEG_FIELD },
         { CTRL('U'),     REQ_UP_FIELD },
         { CTRL('V'),     REQ_DEL_CHAR },
         { CTRL('W'),     REQ_NEXT_WORD },
         { CTRL('X'),     REQ_CLR_FIELD },
         { CTRL('Y'),     REQ_DEL_LINE },
         { CTRL('Z'),     REQ_PREV_CHOICE },
         { ESCAPE,        MAX_FORM_COMMAND + 1 },
         { KEY_F(10),     MAX_FORM_COMMAND + 1 },
         { KEY_BACKSPACE, REQ_DEL_PREV },
         { KEY_DOWN,      REQ_DOWN_CHAR },
         { KEY_END,       REQ_LAST_FIELD },
         { KEY_HOME,      REQ_FIRST_FIELD },
         { KEY_LEFT,      REQ_LEFT_CHAR },
         { KEY_LL,        REQ_LAST_FIELD },
         { KEY_NEXT,      REQ_NEXT_FIELD },
         { KEY_NPAGE,     REQ_NEXT_PAGE },
         { KEY_PPAGE,     REQ_PREV_PAGE },
         { KEY_PREVIOUS,  REQ_PREV_FIELD },
         { KEY_RIGHT,     REQ_RIGHT_CHAR },
         { KEY_UP,        REQ_UP_CHAR },
         { KEY_RETURN,    MAX_FORM_COMMAND + 2 },
         { QUIT,          MAX_FORM_COMMAND + 1 }
    };

   static int  mode = REQ_OVL_MODE;
   int c = wgetch(w);
   unsigned n;
   FIELD *me = current_field(f);

	if (c == CTRL(']') || c == KEY_INS )
   {
      if (mode == REQ_INS_MODE)
         mode = REQ_OVL_MODE;
      else
         mode = REQ_INS_MODE;
      c = mode;
   }
   else
   {
      for (n = 0; n < sizeof(lookup)/sizeof(lookup[0]); n++)
      {
         if (lookup[n].code == c)
         {
            c = lookup[n].result;
            break;
         }
      }
   }

   /*
    * Force the field that the user is typing into to be in reverse video,
    * while the other fields are shown underlined.
    */
   if (c <= KEY_MAX)
      set_field_back(me, A_REVERSE);
   else if (c <= MAX_FORM_COMMAND)
      set_field_back(me, A_UNDERLINE);

   return c;
}

static int my_form_driver(FORM *form, int c)
{
   if (c == (MAX_FORM_COMMAND + 1) && form_driver(form, REQ_VALIDATION) == E_OK)
      return(TRUE);
   else if (c == (MAX_FORM_COMMAND + 2) && form_driver(form, REQ_VALIDATION) == E_OK)
      return 2;
   else
   {
      beep();
      return(FALSE);
   }
}

static void trim_buffer(char *buffer, char trim)
{

   int i;

   for(i = strlen(buffer)-1; i >= 0; i--)
      if (buffer[i] == trim) buffer[i] = 0;
      else break;

}



int Interface_Inject_Filter(DROP_FILTER *filter)
{
   WINDOW *w;
   FORM *form;
   FIELD *f[12];
   int finished = 0, c;
   unsigned n = 0;
   int status = 0;

#ifdef DEBUG
   Debug_msg("Interface_Inject_Filter");
#endif

   refresh();

   f[n++] = make_label(0, 15, "Define a filter");
   f[n++] = make_label(1, 0, "Search :");
   f[n++] = make_field(2, 0, 4, MAX_FILTER/4, FALSE);
   f[n++] = make_label(7, 0, "Action (Drop/Replace) :");
   f[n++] = make_field(7, 25, 1, 7, FALSE);
   f[n++] = make_label(9, 0, "Replace :");
   f[n++] = make_field(10, 0, 4, MAX_FILTER/4, FALSE);
   f[n++] = (FIELD *)0;


   form = new_form(f);
   display_form(form);
   w = form_win(form);

   if (win_pointer == data_dest_win)
   {
      char type[2];
      sprintf(type, "%c",  ARP_Filter_Source.type);
      set_field_buffer(f[2], 0, ARP_Filter_Source.search);
      set_field_buffer(f[6], 0, ARP_Filter_Source.replace);
      set_field_buffer(f[4], 0, type);
   }
   else
   {
      char type[2];
      sprintf(type, "%c",  ARP_Filter_Dest.type);
      set_field_buffer(f[2], 0, ARP_Filter_Dest.search);
      set_field_buffer(f[6], 0, ARP_Filter_Dest.replace);
      set_field_buffer(f[4], 0, type);
   }

   curs_set(1);

   while (!finished)
   {
      switch(form_driver(form, c = form_virtualize(form, w)))
      {
         case E_OK:
            clrtoeol();
            refresh();
            break;
         case E_UNKNOWN_COMMAND:
            finished = my_form_driver(form, c);
            break;
         default:
            beep();
            break;
      }
   }

   erase_form(form);
   free_form(form);

   if (finished == 2)
   {
      memcpy(&filter->search, field_buffer(f[2], 0), MAX_FILTER);
      trim_buffer(filter->search, ' ');
      memcpy(&filter->replace, field_buffer(f[6], 0), MAX_FILTER);
      trim_buffer(filter->replace, ' ');

      filter->slen = strlen(filter->search);
      filter->rlen = strlen(filter->replace);

      memcpy(&filter->type, field_buffer(f[4], 0), 1);
      filter->type = toupper(filter->type);
      if (filter->type == ' ') filter->type = 0;
      status = 1;
   }

   for (c = 0; f[c] != 0; c++)
      free_field(f[c]);

   curs_set(0);

   return status;
}

#else    // DOESN'T HAVE_FORM

int Interface_Inject_Filter(DROP_FILTER *filter)
{
   WINDOW *filter_window, *f_win;
   int dimY = 10;
   int dimX = 60;
   char type[7];

#ifdef DEBUG
   Debug_msg("Interface_Inject_Filter -- NO FORM");
#endif

   f_win = newwin(dimY+2, dimX+2, W_BOTTOMY2/2 - dimY/2, W_MAINX2/2 - dimX/2);
   filter_window = newwin(dimY, dimX, W_BOTTOMY2/2 - dimY/2 +1, W_MAINX2/2 - dimX/2 +1);
   wbkgdset(f_win, COLOR_PAIR(HELP_COLOR));
   wattron(f_win, A_BOLD);
   box(f_win,ACS_VLINE,ACS_HLINE);
   mvwprintw(f_win,  0, 2, "Define a filter:", MAX_INJECT);
   wbkgdset(filter_window, COLOR_PAIR(BOTTOM_COLOR));
   wmove(filter_window, 0, 0);
   echo();
   scrollok(filter_window, TRUE);
   keypad(filter_window, TRUE);
   curs_set(TRUE);

   wprintw(filter_window, "Search : ");
      wnoutrefresh(f_win);
      wnoutrefresh(filter_window);
      doupdate();
   wgetnstr(filter_window, filter->search, MAX_FILTER);
   filter->slen = strlen(filter->search);

   wprintw(filter_window, "\nAction (Drop/Replace) : ");
      wnoutrefresh(f_win);
      wnoutrefresh(filter_window);
      doupdate();
   wgetnstr(filter_window, type, 7);
   filter->type = toupper(type[0]);

   if (filter->type == 'R')
   {
      wprintw(filter_window, "\nReplace : ");
         wnoutrefresh(f_win);
         wnoutrefresh(filter_window);
         doupdate();
      wgetnstr(filter_window, filter->replace, MAX_FILTER);
      filter->rlen = strlen(filter->replace);
   }

   noecho();
   curs_set(FALSE);
   delwin(f_win);
   delwin(filter_window);
   doupdate();

   return 1;
}

#endif   // HAVE_FORM


#endif

/* EOF */
