/* ``The contents of this file are subject to the Erlang Public License,
 * Version 1.0, (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.erlang.org/EPL1_0.txt
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 * 
 * The Original Code is Erlang-4.7.3, December, 1998.
 * 
 * The Initial Developer of the Original Code is Ericsson Telecom
 * AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
 * Telecom AB. All Rights Reserved.
 * 
 * Contributor(s): ______________________________________.''
 */
/* Copyright (C) 1991-94, Ellemtel Telecommunications Systems Laboratories 
 * Copyright (C) 1995, 1996 Ericsson Telecom
 * 
 * Author: Claes Wikstrom  klacke@erix.ericsson.se 
 * 
 * Purpose:read and write stdio with prepended twobytes headers    
 *
 * Mod: 3 Nov 1997 by gordon@erix.ericsson.se 
 *      Changed erl_thb_write so that global buffer is no longer used
 *      (do two writes instead of copying to own buffer). Semantics of
 *      erl_thb_read had to change, now we just return the size
 *      determined from the thb and let the user read the data himself.
 * 
 * Mod: 20 Oct 1997 by gordon@erix.ericsson.se
 *      Removed USE_LOCAL_HEAP
 * 
 * Mod: 8 May 1996 by tobbe@erix.ericsson.se
 *      Introduced USE_LOCAL_HEAP
 * 
 * Mod: 12 Mar 1996 by tobbe@erix.ericsson.se
 *      Turned it into erl_rport.c
 * 
 * Mod: 2 Aug 1995 by Tobbe (tobbe@erix.ericsson.se) 
 *      Fixed a bug both in tbh_read and in "erl_tbh_write" */
#ifdef __WIN32__
#include <winsock2.h>

#elif VXWORKS
#include <sys/types.h>
#include <ioLib.h>
#include <unistd.h>

#else /* other unix */
#include <sys/types.h>
#include <sys/uio.h>
#include <unistd.h>
#endif

/* common includes */
#include <stdio.h> 
#include <stdlib.h>
#include <string.h>
#include "erl_rport.h"
#include "erl_interface.h"

#if defined(HAVE_WRITEV) && defined(SUNOS4)
extern int writev();
#endif

int erl_read_fill(int,char*,int);
int erl_write_fill(int,char*,int);

/* Fill buffer, return buffer length, 0 for EOF, < 0 for error. 
 */
int erl_read_fill(int fd, char* buf, int len)
{
  int i,got=0;
  
  do {
    i = readsocket(fd, buf+got, len-got);
    if (i <= 0)
      return (i);
    got += i;
  } while (got < len);
  return (len);

} /* read_fill */

/* write entire buffer on fd  or fail
 */
int erl_write_fill(int fd, char *buf, int len)
{
  int i,done=0;
  
  do {
    i = writesocket(fd, buf+done, len-done);
    if (i <= 0)
      return (i);
    done += i;
  } while (done < len);
  return (len);
}
/*
 * Whenever a program sits in the other end of an Erlang open_port
 * command that program has its stdio connected to the Erlang-port
 * When data comes from erlang that data is always prepended with a
 * a two byte header in binary integer format msb first.
 * Theese routines provide a way to read and write data to Erlang
 * without having to worry about the two byte headers
 * 
 */

/*
 *  This routine writes buf to an Erlang port in one single write 
 *  The writev does this
 *  Returns: ERL_RPORT_OK when successful, and ERL_RPORT_ERROR on failure
 */
#ifdef HAVE_WRITEV

int erl_tbh_write(int fd, char *buf,int len)
{
  unsigned char lenbuf[2];
  struct iovec vector[2];
  
  if (len > 0xffff) return(ERL_RPORT_ERROR);

  lenbuf[1] = len & 0xff;
  lenbuf[0] = (len >>8) & 0xff;

  vector[0].iov_base = (char*) lenbuf;
  vector[0].iov_len = 2;
  vector[1].iov_base =  buf;
  vector[1].iov_len = len;
  
  if (writev(fd,&vector[0],2) != len + 2) 
    return (ERL_RPORT_ERROR);
  else
    return (ERL_RPORT_OK);
  
} /* tbh_write */

#else

/* This is a version for systems that don't have writev. 
 * We simply do two writes.
 */
int erl_tbh_write(int fd,char *buf,int len) 
{
  unsigned char lenbuf[2];

  if (len > 0xffff) return(ERL_RPORT_ERROR);

  lenbuf[1] = len & 0xff;
  lenbuf[0] = (len >>8) & 0xff;

  if ((erl_write_fill(fd,lenbuf,2) < 0) || (erl_write_fill(fd,buf,len) < 0))
    return (ERL_RPORT_ERROR);
  else 
    return (ERL_RPORT_OK);
} /* tbh_write */

#endif


/* Returns an integer that indicates the number of bytes waiting to be
 * read. To actually read the data, it is (now) the caller's
 * resposibility to call erl_read_fill with a sufficiently large
 * buffer after calling this function to determine the size of buffer
 * needed. 
 */
int erl_tbh_read(int fd) 
{
  unsigned char lenbuf[2];
  int len;

  erl_read_fill(fd,lenbuf,2);
  len = (lenbuf[0] << 8) | lenbuf[1];

  return len;
}

