/* ``The contents of this file are subject to the Erlang Public License,
 * Version 1.0, (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.erlang.org/EPL1_0.txt
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 * 
 * The Original Code is Erlang-4.7.3, December, 1998.
 * 
 * The Initial Developer of the Original Code is Ericsson Telecom
 * AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
 * Telecom AB. All Rights Reserved.
 * 
 * Contributor(s): ______________________________________.''
 */
/* Copyright (C) 1994,1995 Ellemtel Telecommunications Systems Laboratories
 * Copyright (C) 1996 Ericsson Telecom
 * Created: Around 15 Dec 1995
 * Author: {tony,tobbe}@erix.ericsson.se
 * 
 * Purpose: Connect to any node at any host.
 */
#include <stdlib.h>
#include <sys/types.h>
#include <stdarg.h>
#include <fcntl.h>

#ifdef __WIN32__
#include <winsock2.h>
#include <windows.h>
#include <winbase.h>
#include <sys/timeb.h>

#elif VXWORKS
#include <vxWorks.h>
#include <hostLib.h>
#include <selectLib.h>
#include <ifLib.h>
#include <sockLib.h>
#include <taskLib.h>
#include <inetLib.h>
#include <time.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/times.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h> 
#include "netdb.h" /* local file */

#define getpid() taskIdSelf()
extern int h_errno;

#else /* some other unix */
#include <unistd.h>
#include <sys/types.h>
#include <sys/times.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h> 
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/utsname.h>  /* for gen_challenge (NEED FIX?) */
#endif

/* common includes */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

/* the relics from proto.h */
#define PASS_THROUGH 'p'
#define MY_NAME      'M'
#define MY_HIDDEN_NAME 'H'
/* end proto.h relics */

#include "dist.h" 

#include "erl_interface.h"
#include "erl_config.h"
#include "putget.h"
#include "ei.h"
#include "erl_md5.h"

#if !defined(__WIN32__) && !defined(VXWORKS) 
extern int gethostname();
#endif

extern int erl_read_fill(int,char*,int);
extern int erl_write_fill(int,char*,int);

#define SELF(fd) erl_mk_pid(ec.thisnodename, fd, 0, ec.creation)

#define COOKIE_FILE "/.erlang.cookie"

#ifdef __WIN32__
static void initWinSock(void);
#endif

typedef unsigned short  uint16;

#ifndef MAX_COOKIE_SIZE
#define MAX_COOKIE_SIZE 512
#endif

/* rpc_from() uses a buffer this size */
#ifndef MAX_RECEIVE_BUF
#define MAX_RECEIVE_BUF 32*1024
#endif

/* some send() functions use buffer on heap for "small" messages */
/* messages larger than this require call to malloc() */
#define SMALLBUF 2048

typedef struct {
  char thishostname[MAXHOSTLEN+1];
  char thisnodename[MAXHOSTLEN+1+MAXALIVELEN+1];
  char thisalivename[MAXALIVELEN+1];
  struct in_addr this_ipaddr;             /* stored in network byte order */
  char erl_connect_cookie[MAX_COOKIE_SIZE+1];
  short creation;
  int conns[256];
  erlang_pid self;
} internal_stuff;


internal_stuff ec;
static int init_done = 0; /* write-once flag for above global data */

/* forward */
static int name_send(int fd, int dist);
static int recv_name(int fd, ErlConnect *oconn);
static int handshake(int fd, int dist, ErlConnect* oconn);
static int shakehand(int fd, ErlConnect* oconn);


int erl_distversion(int fd)
{
  return ec.conns[fd];
}

const char *erl_thisnodename(void)
{
  return (const char *) ec.thisnodename;
}

const char *erl_thishostname(void)
{
  return (const char *) ec.thishostname;
}

const char *erl_thisalivename(void)
{
  return (const char *) ec.thisalivename;
}

const char *erl_thiscookie(void)
{
  return (const char *) ec.erl_connect_cookie;
}

short erl_thiscreation(void)
{
  return ec.creation;
}

void erl_set_thiscreation(short creation)
{
  ec.creation = creation;
  return;
}

erlang_pid *erl_self(void)
{
  return &ec.self;
}

static int get_int32(unsigned char *s)
{
  return ((s[0] << 24) | (s[1] << 16) | (s[2] << 8) | (s[3] ));
}

static int get_home(char *buf, int size)
{
  char* homedrive;
  char* homepath;

#ifdef __WIN32__
  homedrive = getenv("HOMEDRIVE");
  homepath = getenv("HOMEPATH");
#else
  homedrive = "";
  homepath = getenv("HOME");
#endif

  if (!homedrive || !homepath) {
    buf[0] = '.';
    buf[1] = '\0';
    return 1;
  } else if (strlen(homedrive)+strlen(homepath) < size-1) {
    strcpy(buf, homedrive);
    strcat(buf, homepath);
    return 1;
  }

  return 0;
}


static void get_cookie(void)
{
  char fname[1024+sizeof(COOKIE_FILE)+1];
  int fd;
  int len;
    
  if (!get_home(fname, sizeof(fname))) {
    erl_err_sys("<ERROR> too long home");
  }
  strcat(fname, COOKIE_FILE);
  if ((fd = open(fname, O_RDONLY,0777)) < 0)
    erl_err_sys("<ERROR> open cookie file");
    

  if ((len = read(fd, ec.erl_connect_cookie,
		  sizeof(ec.erl_connect_cookie)-1)) < 0)
    erl_err_sys("<ERROR> reading cookie file (1)");
  else if (len == BUFSIZ)
    erl_err_sys("<ERROR> reading cookie file (2)");
  ec.erl_connect_cookie[len] = '\0';
  if (ec.erl_connect_cookie[len-1] == '\n')
    ec.erl_connect_cookie[len-1] = '\0';
}

#ifdef __WIN32__
static void win32_error(char *buf, int buflen)
{
  FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM|FORMAT_MESSAGE_IGNORE_INSERTS,
		0,	/* n/a */
		WSAGetLastError(), /* error code */
		MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), /* language */
		buf,
		buflen,
		NULL);
  return;
}

static void initWinSock(void)
{
  WORD wVersionRequested;  
  WSADATA wsaData; 
  int i; 
  static int initialized;

  wVersionRequested = MAKEWORD(1, 1); 
  if (!initialized) {
    initialized = 1;
    if ((i = WSAStartup(wVersionRequested, &wsaData))) {
      erl_err_msg("<ERROR> erl_connect_init: Can't initialize windows sockets: %d",i);
    }
  
    if ( LOBYTE( wsaData.wVersion ) != 1 || HIBYTE( wsaData.wVersion ) != 1 ) { 
      erl_err_msg("<ERROR> erl_connect_init: This version of windows sockets "
		  "not supported");
      WSACleanup(); 
    }
  }
}
#endif

/*
 * Perhaps run this routine instead of erl_connect_init/2 ?
 * Initailize by setting:
 * thishostname, thisalivename, thisnodename and thisipaddr
 */
int erl_connect_xinit(char *thishostname, char *thisalivename, char *thisnodename,
		      Erl_IpAddr thisipaddr, char *cookie, short creation)
{

  if (init_done) return 1;
  init_done = 1;

  ec.creation = creation;

#ifdef __WIN32__
  initWinSock();
#endif

  if (!cookie) get_cookie();
  else if (strlen(cookie) >= sizeof(ec.erl_connect_cookie)) 
    erl_err_quit("<ERROR> erl_connect_xinit: Cookie size too large");
  else strcpy(ec.erl_connect_cookie, cookie);

  if (strlen(thishostname) >= sizeof(ec.thishostname)) 
    erl_err_quit("<ERROR> erl_connect_init: Thishostname too long");
  strcpy(ec.thishostname, thishostname);

  if (strlen(thisalivename) >= sizeof(ec.thisalivename)) 
    erl_err_quit("<ERROR> erl_connect_init: Thisalivename too long");
  strcpy(ec.thisalivename, thisalivename);

  if (strlen(thisnodename) >= sizeof(ec.thisnodename)) 
    erl_err_quit("<ERROR> erl_connect_init: Thisnodename too long");
  strcpy(ec.thisnodename, thisnodename);

  memmove(&ec.this_ipaddr, thisipaddr, sizeof(ec.this_ipaddr)); 

  strcpy(ec.self.node,thisnodename);
  ec.self.num = 0;
  ec.self.serial = 0;
  ec.self.creation = creation;

  memset(&ec.conns,0,sizeof(ec.conns));
  return 1;
}

int erl_connect_init_ex(int this_node_number, char *cookie,
			short creation, int use_long_name)
{
  struct hostent *hp;
  char* ct;
  char thishostname[MAXHOSTLEN+1];
  char thisnodename[MAXHOSTLEN+1+MAXALIVELEN+1];
  char thisalivename[MAXALIVELEN+1];

#ifdef __WIN32__
  initWinSock();
#endif /* win32 */

  if (gethostname(thishostname, MAXHOSTLEN) ==  -1) {
#ifdef __WIN32__
    erl_err_quit("erl_connect_init: Failed to get host name: %d",
		 WSAGetLastError());
#else
    erl_err_quit("erl_connect_init: Failed to get host name: %d",
		 errno);
#endif /* win32 */
  }

  sprintf(thisalivename, "c%d",
	  this_node_number < 0 ?  (int) getpid() : this_node_number);
  
  if ((hp = erl_gethostbyname(thishostname)) == 0) {
#ifdef __WIN32__
    char reason[1024];

    win32_error(reason,sizeof(reason));
    erl_err_msg("erl_connect_init: Can't get ip address for host %s: %s\n",
		thishostname, reason);
#else
    erl_err_msg("erl_connect_init: Can't get ip address for host %s: %d\n",
		thishostname,h_errno);
#endif /* win32 */
    return 0;
  }
  
  if ((use_long_name == 0) &&  /* shortnames */
      ((ct = strchr(hp->h_name, '.')) != NULL)) *ct = '\0';

  strcpy(thishostname, hp->h_name);
  sprintf(thisnodename, "%s@%s", thisalivename, hp->h_name);

  return erl_connect_xinit(thishostname, thisalivename, thisnodename,
			   (struct in_addr *)*hp->h_addr_list, cookie, creation);
}

/*
 * Initialize by set: thishostname, thisalivename, 
 * thisnodename and thisipaddr. At success return 1,
 * otherwise return 0.
 */
int erl_connect_init(int this_node_number, char *cookie, short creation) {
  return erl_connect_init_ex(this_node_number, cookie, creation, 0);
}


/* connects to port at ip-address ip_addr 
 * and returns fd to socket 
 * port has to be in host byte order 
 */
static int cnct(uint16 port, struct in_addr *ip_addr, int addr_len)
{
  int s;
  struct sockaddr_in iserv_addr;

  if ((s = socket(AF_INET, SOCK_STREAM, 0)) < 0)
    return ERL_ERROR;
  
  memset((char*)&iserv_addr, 0, sizeof(struct sockaddr_in));
  memcpy((char*)&iserv_addr.sin_addr, (char*)ip_addr, addr_len);
  iserv_addr.sin_family = AF_INET;
  iserv_addr.sin_port = htons(port);

  if (connect(s, (struct sockaddr*)&iserv_addr, sizeof(iserv_addr)) < 0) {
    closesocket(s);
    return ERL_ERROR;
  }

  return s;
} /* cnct */


/* 
 * Set up a connection to a given Node, and 
 * interchange hand shake messages with it.
 * Returns a valid file descriptor at success,
 * otherwise a negative error code.
 */
int erl_connect(char *nodename)
{
  char *hostname, alivename[BUFSIZ];
  struct hostent *hp;
#if !defined (__WIN32__) 
  /* these are needed for the call to gethostbyname_r */
  struct hostent host;
  char buffer[1024];
  int h_errno;
#endif /* win32 */

  /* extract the host and alive parts from nodename */
  if (!(hostname = strchr(nodename,'@'))) return ERL_ERROR;
  else {
    strncpy(alivename, nodename, hostname - nodename);
    alivename[hostname - nodename] = 0x0;
    hostname++;
  }

#ifndef __WIN32__
  if ((hp = erl_gethostbyname_r(hostname,&host,buffer,1024,&h_errno)) == NULL) {
    erl_err_msg("<ERROR> erl_connect: Can't find host for %s: %d\n", nodename, h_errno);
  return ERL_ERROR;
  }

#else
  if ((hp = erl_gethostbyname(hostname)) == NULL) {
    char reason[1024];
    win32_error(reason,sizeof(reason));
    erl_err_msg("<ERROR> erl_connect: Can't find host for %s: %s\n", nodename, reason);
    return ERL_ERROR;
  }
#endif /* win32 */

  return erl_xconnect((Erl_IpAddr) *hp->h_addr_list, alivename);
} /* erl_connect */


/* ip_addr is now in network byte order 
 *
 * first we have to get hold of the portnumber to
 *  the node through epmd at that host 
 *
 */
int erl_xconnect(Erl_IpAddr adr, char *alivename)
{
  struct in_addr *ip_addr=(struct in_addr *) adr;
  int rport = 0;
  int sockd;
  int one = 1;
  int dist = 0;

  if ((rport = erl_epmd_port(ip_addr,alivename,&dist)) < 0) return ERL_NO_PORT;

  /* we now have port number to enode, try to connect */
  if((sockd = cnct(rport, ip_addr, sizeof(struct in_addr))) < 0)
    return ERL_CONNECT_FAIL;

  if (dist < ERL_DIST_LOW)
      return ERL_ERROR;

  if (!handshake(sockd, dist, NULL)) {
      closesocket(sockd);
      return ERL_ERROR;
  }

  setsockopt(sockd, IPPROTO_TCP, TCP_NODELAY, (char *)&one, sizeof(one));
  setsockopt(sockd, SOL_SOCKET, SO_KEEPALIVE, (char *)&one, sizeof(one));

  return sockd;
} /* erl_xconnect */

/* 
 * For symmetry reasons
 */
int erl_close_connection(int fd)
{
  return closesocket(fd);
} /* erl_close_connection */


/*
 * Accept and initiate a connection from an other
 * Erlang node. Return a file descriptor at success,
 * otherwise -1;
 */
int erl_accept(int lfd, ErlConnect *conp)
{
  int fd;
  struct sockaddr_in cli_addr;
  int cli_addr_len=sizeof(struct sockaddr_in);

  if ((fd = accept(lfd, (struct sockaddr*) &cli_addr, &cli_addr_len )) < 0)
      return ERL_ERROR;
  if (!shakehand(fd, conp)) {
      closesocket(fd);
      return ERL_ERROR;      
  }
  return fd;
} /* erl_accept */


/* Receives a message from an Erlang socket.
 * If the message was a TICK it is immediately
 * answered. Returns: ERL_ERROR, ERL_TICK or
 * the number of bytes read.
 */
int erl_receive(int s, unsigned char *bufp, int bufsize) 
{
  int len;
  unsigned char fourbyte[4]={0,0,0,0};
  
  if (erl_read_fill(s, (char *) bufp, 4)  != 4) {
    return ERL_ERROR;
  }

  /* Tick handling */
  if ((len = get_int32(bufp)) == ERL_TICK) {
    erl_write_fill(s, (char *) fourbyte, 4);
    return ERL_TICK;
  }
  else if (len > bufsize) {
    erl_err_quit("<ERROR> Receive buffer in erl_receive overwritten, len = %d", len);
  }
  else if (erl_read_fill(s, (char *) bufp, len) != len)
    return ERL_ERROR;

  return len;

} /* erl_receive */

/* 
 * Send an Erlang message to a registered process
 * at the Erlang node, connected with a socket.
 */
int erl_reg_send(int fd, char *server_name, ETERM *msg)
{
  char sbuf[SMALLBUF]; /* use this for short messages */
  char *dbuf = NULL;   /* use this for longer ones */
  char *msgbuf;
  int msglen;
  erlang_pid *self = erl_self();

  /* get large enough buffer */
  if ((msglen = erl_term_len(msg)) > SMALLBUF)
    if (!(dbuf = malloc(msglen)))
      return -1;
  msgbuf = (dbuf ? dbuf : sbuf);

  if (!erl_encode(msg,msgbuf)) return -1;

  self->num = fd;
  if (ei_send_reg_encoded(fd,self,server_name,msgbuf,msglen)) {
    if (dbuf) free(dbuf);
    return -1;
  }

  if (dbuf) free(dbuf);
  return 1;
}

/* 
 * Sends an Erlang message to a process at an Erlang node
 */
int erl_send(int fd, ETERM *to ,ETERM *msg)
{
  char sbuf[SMALLBUF]; /* use this for short messages */
  char *dbuf = NULL;   /* use this for longer ones */
  char *msgbuf = NULL;
  int msglen;
  erlang_pid topid;

  /* make the to-pid */
  if (!(ERL_IS_PID(to))) return -1;
  
  strcpy(topid.node,ERL_PID_NODE(to));
  topid.num = ERL_PID_NUMBER(to);
  topid.serial = ERL_PID_SERIAL(to);
  topid.creation = ERL_PID_CREATION(to);
  
  if ((msglen = erl_term_len(msg))  >  SMALLBUF) 
    if (!(dbuf = malloc(msglen))) 
      return -1;
  msgbuf = (dbuf ? dbuf : sbuf);
  
  if (!erl_encode(msg,msgbuf)) return -1;

  if (ei_send_encoded(fd,&topid,msgbuf,msglen)) {
    if (dbuf) free(dbuf);
    return -1;
  }

  if (dbuf) free(dbuf);
  return 1;
}

/* 
 * Try to receive an Erlang message on a given
 * socket. Returns ERL_TICK, ERL_MSG, or ERL_ERROR.
 */
int erl_receive_msg(int fd, unsigned char *buf, int bufsize, ErlMessage *emsg)
{
  int size = bufsize;
  erlang_msg msg;
  int i;

  if (!(i=ei_receive_encoded(fd,buf,&size,&msg))) return ERL_TICK;
  if (i<0) return ERL_ERROR;
  if (size >= bufsize) return ERL_ERROR;
  
  emsg->type = msg.msgtype;
  emsg->to_name[0] = 0x0;
  
  switch (msg.msgtype) {
  case ERL_SEND:
    emsg->to = erl_mk_pid(msg.to.node,msg.to.num,msg.to.serial,msg.to.creation);
    emsg->from = NULL;
    emsg->msg = erl_decode(buf);
    return ERL_MSG;
    break;
    
  case ERL_REG_SEND:
    emsg->from = erl_mk_pid(msg.from.node,msg.from.num,msg.from.serial,msg.from.creation);
    emsg->to = NULL;
    emsg->msg = erl_decode(buf);
    strcpy(emsg->to_name,msg.toname);
    return ERL_MSG;
    break;

  case ERL_LINK:
  case ERL_UNLINK:
  case ERL_GROUP_LEADER:
    emsg->from = erl_mk_pid(msg.from.node,msg.from.num,msg.from.serial,msg.from.creation);
    emsg->to = erl_mk_pid(msg.to.node,msg.to.num,msg.to.serial,msg.to.creation);
    emsg->msg = NULL;
    return ERL_MSG;
    

  case ERL_EXIT:
  case ERL_EXIT2:
    emsg->from = erl_mk_pid(msg.from.node,msg.from.num,msg.from.serial,msg.from.creation);
    emsg->to = erl_mk_pid(msg.to.node,msg.to.num,msg.to.serial,msg.to.creation);
    emsg->msg = erl_decode(buf); /* contains reason */
    return ERL_MSG;

  case ERL_NODE_LINK:
    emsg->to = NULL;
    emsg->from = NULL;
    emsg->msg = NULL;
    return ERL_MSG;
  }

  /* else (error) */
  if (emsg->to) erl_free_term(emsg->to);
  if (emsg->from) erl_free_term(emsg->from);
  if (emsg->msg) erl_free_term(emsg->msg);
  emsg->to = NULL;
  emsg->from = NULL;
  emsg->msg = NULL;

  return ERL_ERROR;
} /* erl_receive_msg */


/* 
 * The RPC consists of two parts, send and receive.
 * Here is the send part ! 
 * { PidFrom, { call, Mod, Fun, Args, user }} 
 */
void erl_rpc_to(int fd, char *mod, char *fun, ETERM *args)
{
  char sbuf[SMALLBUF];
  char *dbuf = NULL;
  char *msgbuf = NULL;
  erlang_pid *self = erl_self();
  int index = 0;
  int msglen = 0;
  char tmp;

  msglen = erl_term_len(args);

  /* determine how much buffer we need for this */
  index = 0;
  ei_encode_version(NULL,&index);
  ei_encode_tuple_header(NULL,&index,2);
  
  self->num = fd;
  ei_encode_pid(NULL,&index,self);

  ei_encode_tuple_header(NULL,&index,5);
  ei_encode_atom(NULL,&index,"call");
  ei_encode_atom(NULL,&index,mod);
  ei_encode_atom(NULL,&index,fun);
  index += msglen-1; /* see coment below for explanation */
  ei_encode_atom(NULL,&index,"user");

  /* get a large enough buffer to encode this into */
  if (index > SMALLBUF)
    if (!(dbuf = malloc(index)))
      return;
  msgbuf = (dbuf ? dbuf : sbuf);
  
  /* now encode the message into the message buffer */
  index = 0;
  ei_encode_version(msgbuf,&index);
  ei_encode_tuple_header(msgbuf,&index,2);
  
  self->num = fd;
  ei_encode_pid(msgbuf,&index,self);

  ei_encode_tuple_header(msgbuf,&index,5);
  ei_encode_atom(msgbuf,&index,"call");
  ei_encode_atom(msgbuf,&index,mod);
  ei_encode_atom(msgbuf,&index,fun);
  /* erl_encode would write an unwanted version byte at msgbuf[index], so
   * we pass it msgbuf[index-1] instead, saving the byte at that position
   * first and restoring it afterwards
   */
  tmp = msgbuf[index-1];
  erl_encode(args,msgbuf+index-1);
  msgbuf[index-1] = tmp;
  index += msglen-1;
  ei_encode_atom(msgbuf,&index,"user");

  ei_send_reg_encoded(fd, self, "rex", msgbuf, index);

  if (dbuf) free(dbuf);
} /* rpc_to */

/*
 * And here is the rpc receiving part. A negative
 * timeout means 'infinity'. Returns either of: ERL_MSG,
 * ERL_TICK, ERL_ERROR or ERL_TIMEOUT.
 */
int erl_rpc_from(int fd, int timeout, ErlMessage *emsg) 
{
  char rbuf[MAX_RECEIVE_BUF];
  fd_set readmask;
  struct timeval tv;
  struct timeval *t = NULL;

  if (timeout >= 0) {
    tv.tv_sec = timeout / 1000;
    tv.tv_usec = (timeout % 1000) * 1000;
    t = &tv;
  }

  FD_ZERO(&readmask);
  FD_SET(fd,&readmask);

  switch (select(FD_SETSIZE,&readmask,NULL,NULL,t)) {
  case -1: 
    return ERL_ERROR;

  case 0:
    return ERL_TIMEOUT;

  default:
    if (FD_ISSET(fd, &readmask)) 
      return erl_receive_msg(fd, (unsigned char *) rbuf, sizeof(rbuf), emsg);
  }

  return ERL_ERROR;
} /* rpc_from */

/*
 * A true RPC. It return a NULL pointer
 * in case of failure, otherwise a valid
 * (ETERM *) pointer containing the reply
 */
ETERM *erl_rpc(int fd, char *mod, char *fun, ETERM *args)
{
  int i;
  ETERM *ep;
  ErlMessage emsg;

  erl_rpc_to(fd, mod, fun, args);
  while ((i=erl_rpc_from(fd, ERL_NO_TIMEOUT, &emsg)) == ERL_TICK);
  if (i == ERL_ERROR)  return NULL;

  ep = erl_element(2,emsg.msg); /* {RPC_Tag, RPC_Reply} */
  erl_free_term(emsg.msg);
  erl_free_term(emsg.to);
  return ep;
} /* rpc */

/* FROM RTP RFC 1889  (excpet we use all bit? bug in RFC?) */
static u_int32 md_32(char* string, int length)
{
    MD5_CTX ctx;
    union {
	char c[16];
	u_int32 x[4];
    } digest;
    MD5Init(&ctx);
    MD5Update(&ctx, string, length);
    MD5Final((u_int8*) &digest, &ctx);
    return (digest.x[0] ^ digest.x[1] ^ digest.x[2] ^ digest.x[3]);
}

#if defined(__WIN32__)
static u_int32 gen_challenge(int type)
{
    struct {
	int type;
	struct _timeb tv;
	clock_t cpu;
	int pid;
    } s;
    _ftime(&s.tv);
    s.type = type;
    s.cpu  = clock();
    s.pid  = getpid();
    return md_32((char*) &s, sizeof(s));
}
#elif  defined(VXWORKS)
static u_int32 gen_challenge(int type)
{
    struct {
	int type;
	struct timespec tv;
	clock_t cpu;
	int pid;
    };
    s.cpu  = clock();
    clock_gettime(CLOCK_REALTIME, &s.tv);
    s.pid = getpid();
    return md_32((char*) &s, sizeof(s));
}

#else  /* some unix */
static u_int32 gen_challenge(int type)
{
    struct {
	int type;
	struct timeval tv;
	clock_t cpu;
	pid_t pid;
	u_long hid;
	uid_t uid;
	gid_t gid;
	struct utsname name;
    } s;
    gettimeofday(&s.tv, 0);
    uname(&s.name);
    s.type = type;
    s.cpu  = clock();
    s.pid  = getpid();
    s.hid  = gethostid();
    s.uid  = getuid();
    s.gid  = getgid();
    return md_32((char*) &s, sizeof(s));
}
#endif


/* given challenge & cookie generate a 128 bit digest */
static void gen_digest(u_int32 challenge, const char* cookie, 
		       u_int8 digest[16])
{
    MD5_CTX ctx;
    u_int8  ch[10];

    sprintf(ch, "%lu", challenge);
    MD5Init(&ctx);
    MD5Update(&ctx, (char*)cookie, strlen(cookie));
    MD5Update(&ctx, ch, strlen(ch));
    MD5Final(digest, &ctx);
}

static int challenge_recv(int fd, int* type, 
			  u_int32* challenge, ErlConnect *oconn)
{
    ErlConnect conn;
    u_int8 buf[SMALLBUF];
    u_int8* ptr;
    int len;
    int dist;
    /* [Type,Version(2),Ip(4),challenge(4) | Node] */
    if (erl_read_fill(fd, (char*) buf, 2) <= 0) return 0;
    ptr = buf;
    len = get16be(ptr);

    if (len >= SMALLBUF)  return 0;
    if (erl_read_fill(fd, (char*) buf, len) < len) return 0;
    ptr = buf;

    if (*ptr == 'm') *type = 'm';
    else if (*ptr == 'h') *type = 'h';
    else return 0;
    ptr++;

    dist = get16be(ptr);
    memmove(&conn.ipadr, ptr, sizeof(conn.ipadr));
    ptr += sizeof(conn.ipadr);
    *challenge = get32be(ptr);
    len -= (1+2+4+4);  /* rest is node part */
    if (len >= MAXNODELEN) return 0;
    memcpy(conn.nodename, ptr, len);
    conn.nodename[len] = '\0';
    ec.conns[fd] = dist;
    if (oconn != NULL) *oconn = conn;
    return 1;
}

static int challenge_send_reply(int fd, u_int32 challenge, u_int8 digest[16])
{
    u_int8 buf[23];
    u_int8* ptr = buf;

    /* send [len(2) 'r', challenge(4), digest(16) ] */
    put16be(ptr, 21);   /* 1 + 4 + 16 */
    *ptr++ = 'r';
    put32be(ptr, challenge);
    memcpy((void*)ptr, (void*)digest, 16);
    return erl_write_fill(fd, (char*) buf, 23) == 23;
}


static int challenge_recv_ack(int fd, u_int32 challenge, 
			      const char* cookie)
{
    u_int8 buf[17];
    u_int8* ptr;
    u_int8 digest[16];
    int len;

    /* receive ['a' | digest ] */
    if (erl_read_fill(fd, (char*) buf, 2) <= 0) return 0;
    ptr = buf;
    len = get16be(ptr);
    if (len != 17) return 0;
    if (erl_read_fill(fd, (char*)buf, 17) < 17) return 0;
    if (buf[0] != 'a') return 0;

    gen_digest(challenge, cookie, digest);
    if (memcmp((void*) digest, (void*)&buf[1], 16) != 0) return 0;
    return 1;
}

static int challenge_send(int fd, char* node, u_int32 challenge, int dist)
{
    u_int8  buf[SMALLBUF];
    u_int8* ptr;
    int len;

    /* send [$m,version(16),Ip1,Ip2,Ip3,Ip4,challenge(32),node */
    ptr = buf + 2;
    put8(ptr, 'm');
    put16be(ptr, dist);
    memmove((void*)ptr, (void*)&ec.this_ipaddr.s_addr, 4);
    ptr += 4;
    put32be(ptr, challenge);
    strcpy((char*)ptr, node);
    ptr += strlen(node);
    len = ptr - buf;
    ptr = buf;
    put16be(ptr, len);
    len += 2;
    if (erl_write_fill(fd, (char*) buf, len) != len) return 0;
    return 1;
}

static int challenge_recv_reply(int fd, u_int32 challenge, 
				u_int32* challengeB, const char* cookie)
{
    u_int8 buf[21];
    u_int8* ptr;
    u_int8 digest[16];
    u_int32 temp;
    int len;

    /* receive [$r,challengeB(32), digestB] */
    if (erl_read_fill(fd, (char*) buf, 2) <= 0) return 0;
    ptr = buf;
    len = get16be(ptr);
    if (len != 21) return 0;
    if (erl_read_fill(fd, (char*) buf, 21) < 21) return 0;
    if (buf[0] != 'r') return 0;
    ptr = &buf[1];
    temp = get32be(ptr);
    gen_digest(challenge, cookie, digest);
    if (memcmp((void*)digest, (void*) ptr, 16) != 0) return 0;
    *challengeB = temp;
    return 1;
}

static int challenge_send_ack(int fd, u_int8 digest[16])
{
    u_int8 buf[19];
    u_int8* ptr = buf;
    /* send [$a,Digest] */
    put16be(ptr, 17);
    *ptr++ = 'a';
    memcpy((void*)ptr, (void*) digest, 16);
    return erl_write_fill(fd, (char*) buf, 19) == 19;
}



#ifdef DEBUG_DIST
void print_digest(FILE* f, u_int8 digest [16])
{
    int i;
    for (i = 0; i < 16; i++)
	fprintf(f, "%02x", digest[i]);
}
#endif


static int handshake(int fd, int dist, ErlConnect* oconn)
{
    u_int32 challenge;
    u_int32 challengeA;
    int type;
    u_int8 digest[16];
    ErlConnect conn;

    if (!name_send(fd, dist)) return 0;
    if (!challenge_recv(fd, &type, &challengeA, &conn)) return 0;
    gen_digest(challengeA, erl_thiscookie(), digest);
    challenge = gen_challenge(dist);
    if (!challenge_send_reply(fd, challenge, digest)) return 0;
    if (!challenge_recv_ack(fd, challenge,
			    erl_thiscookie())) return 0;
    if (oconn != NULL) *oconn = conn;
    return 1;
}

static int shakehand(int fd, ErlConnect* oconn)
{
    u_int8 digest[16];
    u_int32 challenge;
    u_int32 challengeB;
    ErlConnect conn;
    int dist;

    if (!recv_name(fd, &conn)) return 0;
    dist = erl_distversion(fd);
    if ((dist < ERL_DIST_LOW) || (dist > ERL_DIST_HIGH)) return 0;

    challenge = gen_challenge(dist);
    if (!challenge_send(fd, ec.thisnodename, challenge, dist)) return 0;
    if (!challenge_recv_reply(fd, challenge, &challengeB, erl_thiscookie()))
	return 0;
    gen_digest(challengeB, erl_thiscookie(), digest);
    if (!challenge_send_ack(fd, digest)) return 0;
    if (oconn != NULL) *oconn = conn;
    return 1;
}


static int name_send(int fd, int dist)
{
  char rbuf[SMALLBUF];
  char *s, *s0;
  int len;
  int ntype;
#if defined(VXWORKS) && defined(DEBUG_DIST)
  char ntoabuf[32];
#endif

  s0 = s = rbuf+2; /* save two bytes for length  */
  ntype = 'h'; /* new form of MY_HIDDEN_NAME */
  put8(s,ntype); 
  put16be(s,dist); /* extra field (new in R4) */
  memmove(s, &ec.this_ipaddr.s_addr, 4);
  s += 4;
  strcpy(s, ec.thisnodename);
  s += strlen(ec.thisnodename);
  len = s-s0; /* size of payload */
  s = rbuf;
  put16be(s,len);
  len += 2;   /* size of packet */

#ifdef DEBUG_DIST
#ifdef VXWORKS
  inet_ntoa_b(ec.this_ipaddr,ntoabuf);
  fprintf(stderr,"-> TELLNAME ntype=%d dist=%d ip=%s node=%s\n",
	  ntype,dist,ntoabuf,ec.thisnodename);
#else
  fprintf(stderr,"-> TELLNAME ntype=%d dist=%d ip=%s node=%s\n",
	  ntype,dist,inet_ntoa(ec.this_ipaddr),ec.thisnodename);
#endif
#endif
    if (erl_write_fill(fd, rbuf, len) != len) return 0;
  return 1;
} /* name_send */

static int recv_name(int fd, ErlConnect *oconn)
{
  ErlConnect conn;
  char rbuf[SMALLBUF];
  char *s;
  int i,len;
  int ntype;
  int dist;
#ifdef DEBUG_DIST
  struct in_addr adr;
#ifdef VXWORKS
  char ntoabuf[32];
#endif
#endif

  /* now get his ip-address */
  if (erl_read_fill(fd, rbuf, 2) <= 0) return 0;

  s = rbuf;
  i = get16be(s);
  len = erl_read_fill(fd, rbuf, i);
  if (len <= 0) return 0;

  rbuf[len] = 0x0; /* terminate the nodename */
  s = rbuf;
  ntype = get8(s);

  switch (ntype) {
  case 'h': /* r4 types */
  case 'm':
    dist = get16be(s);
    memmove(&conn.ipadr,s,sizeof(conn.ipadr));
    s += sizeof(conn.ipadr);
    strncpy(conn.nodename, s, MAXNODELEN);
    conn.nodename[MAXNODELEN] = 0x0;
    break;

  default: /* unknown */
#ifdef DEBUG_DIST
    fprintf(stderr,"<- TELLNAME ntype=%d (unknown)\n",ntype);
#endif
    return 0;
  }

#ifdef DEBUG_DIST
  memmove(&adr.s_addr,conn.ipadr,4);
#ifdef VXWORKS
  inet_ntoa_b(adr,ntoabuf);
  fprintf(stderr,"<- RECVNAME type=%d dist=%d ip=%s node=%s\n",
	  ntype,dist,ntoabuf,conn.nodename);
#else
  fprintf(stderr,"<- RECVNAME ntype=%d dist=%d ip=%s node=%s\n",
	  ntype,dist,inet_ntoa(adr),conn.nodename); 
#endif
#endif
  
  /* transfer to callers buffer */
  if (oconn) *oconn = conn;

  /* remember what distribution version we use on this connection */
  ec.conns[fd] = dist;
  
  return 1;
} /* get_name */
