/* ``The contents of this file are subject to the Erlang Public License,
 * Version 1.0, (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.erlang.org/EPL1_0.txt
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 * 
 * The Original Code is Erlang-4.7.3, December, 1998.
 * 
 * The Initial Developer of the Original Code is Ericsson Telecom
 * AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
 * Telecom AB. All Rights Reserved.
 * 
 * Contributor(s): ______________________________________.''
 */
/*
 * This file is copyright (c) Ellemtel in April 1990
 *
 * Author Mike Williams
 *
 * Tony:
 * 950227 Added heap_change for fast increase & decrease of heap size
 * 950227 Added parameter to stack_change to allow shrink of stack
 * 950227 Added heap replacment heaps for small heaps
 * 950310 VERBOSE macro 
 * 960307 Removed heap replacement, introduced realloc on heaps,
 *        and fibonacci heap sizes, stack sizes
 *
 * While stack size are f(n) - 2 since we add two fence patterns for
 * consistency check
 *
 * 960308 Added stack move for garbage_collect(0)
 * 960308 Added grow_flag for garbage collect, the grow will not
 *        take place until next gc
 *
 * Klacke:
 * 970925 made the collector generational, using two generations
 *
 */

#include "sys.h"
#include "config.h"
#include "global.h"
#include "erl_process.h"
#include "db.h"
#include "jtab.h"

/*
 * Returns number of elements in an array.
 */
#define ALENGTH(a) (sizeof(a)/sizeof(a[0]))

#ifdef DEBUG
/* #define HARDDEBUG  1 */
/* #define GC_REALLOC 1 */
/* #define GC_HEAP_TRACE 1 */
/* #define GC_STACK_TRACE 1 */
/* #define GC_SWITCH_TRACE 1 */
/* #define OLD_HEAP_CREATION_TRACE 1 */
#endif

/*
 * This structure describes the rootset for the GC.
 */
typedef struct {
    uint32* v[6];		/* Pointers to vectors with terms to GC
				 * (e.g. the stack).
				 */
    uint32 sz[6];		/* Size of each vector. */
    uint32* v_msg;		/* Pointer to messages to GC. */
    uint32 def_msg[32];		/* Default storage for messages (to avoid malloc). */
} Rootset;

#if defined(JAM)
static FUNCTION(void, offset_stack, (uint32 *, uint32 *, sint32));
static FUNCTION(void, stack_move, (Process *));
#endif
static FUNCTION(int, setup_rootset, (Process *, uint32 *, int , Rootset* ));
static FUNCTION(int, gen_gc, (Process*, int, uint32*, int));
static FUNCTION(char*, print_pid, (Process* p));

#ifdef HARDDEBUG
static FUNCTION(void, check_heap, (Process*, char*));
static FUNCTION(void, check_stack, (Process*, char*));
static FUNCTION(int,  within, (uint32*, Process*));
static FUNCTION(void, check_bins, (Process*));
#define CHECK(p) \
    check_stack((p), "check"); \
    check_bins(p); \
    check_heap((p), "check")
#else
#define within(x,y) 1
#define CHECK(p) ((void) 1)
#endif

/* Use this if we want to force realloc to return a  */
/* new memory area, i.e. with offset != 0            */
#ifdef GC_REALLOC
static void *gc_copy_realloc(void *ptr, int sz)
{
    void *mem = safe_alloc(sz);
    sys_memcpy(mem, ptr, sz);
    sys_free(ptr);
    return mem;
}
#define REALLOC(ptr, sz) gc_copy_realloc((ptr), (sz))
#else
#define REALLOC(ptr, sz) safe_realloc((ptr), (sz))
#endif

#define ptr_within(ptr, x, y) (((ptr) >= (x) && (ptr) < (y)))

static int heap_sizes[64];	/* Suitable heap sizes. */
static int num_heap_sizes;	/* Number of heap sizes. */

/*
** Initialize GC global data
*/
void init_gc(void)
{
    int i;

    switch (heap_series) {
    case HS_FIBONACCI:
	heap_sizes[0] = 34;
	heap_sizes[1] = 55;
	for (i = 2; i < ALENGTH(heap_sizes) && heap_sizes[i-1] < MAX_SMALL; i++) {
	    heap_sizes[i] = heap_sizes[i-1] + heap_sizes[i-2];
	}
	break;
    case HS_POWER_TWO:
	heap_sizes[0] = 32;
	for (i = 1; i < ALENGTH(heap_sizes) && heap_sizes[i-1] < MAX_SMALL; i++) {
	    heap_sizes[i] = 2 * heap_sizes[i-1];
	}
	break;
    case HS_POWER_TWO_MINUS_ONE:
	heap_sizes[0] = 31;
	for (i = 1; i < ALENGTH(heap_sizes) && heap_sizes[i-1] < MAX_SMALL; i++) {
	    heap_sizes[i] = 2 * (heap_sizes[i-1]+1) - 1;
	}
	break;
    }
    num_heap_sizes = i;
}

/*
** Offset pointer to heap from stack (Beam is special)
*/
#if defined(BEAM)
static void offset_heap_ptr(uint32 *hp, uint32 sz, sint32 offs, 
			    uint32 *low, uint32 *high)
{
    DECL_JVALUE(v, CP0_DEF)
    DECL_JVALUE(v, CP4_DEF)
    DECL_JVALUE(v, CP8_DEF)
    DECL_JVALUE(v, CP12_DEF)
    DECL_JVALUE(v, SMALL_DEF)
    DECL_JVALUE(v, BIG_DEF)
    DECL_JVALUE(v, FLOAT_DEF)
    DECL_JVALUE(v, ATOM_DEF)
    DECL_JVALUE(v, REFER_DEF)
    DECL_JVALUE(v, PORT_DEF)
    DECL_JVALUE(v, PID_DEF)
    DECL_JVALUE(v, TUPLE_DEF)
    DECL_JVALUE(v, CATCH_DEF)
    DECL_JVALUE(v, LIST_DEF)
    DECL_JVALUE(v, BLANK_DEF)
    DECL_JVALUE(v, BINARY_DEF)
    DECL_JTABLE(v, 16)

    int ignore_hilo = (high == 0 && low == 0);

    if (JTABLE_NEED_INIT(v)) {
	DEFINE_LOCATION(v, CP0_DEF);
	DEFINE_LOCATION(v, CP4_DEF);
	DEFINE_LOCATION(v, CP8_DEF);
	DEFINE_LOCATION(v, CP12_DEF);
	DEFINE_LOCATION(v, SMALL_DEF);
	DEFINE_LOCATION(v, BIG_DEF);
	DEFINE_LOCATION(v, FLOAT_DEF);
	DEFINE_LOCATION(v, ATOM_DEF);
	DEFINE_LOCATION(v, REFER_DEF);
	DEFINE_LOCATION(v, PORT_DEF);
	DEFINE_LOCATION(v, PID_DEF);
	DEFINE_LOCATION(v, TUPLE_DEF);
	DEFINE_LOCATION(v, CATCH_DEF);
	DEFINE_LOCATION(v, LIST_DEF);
	DEFINE_LOCATION(v, BLANK_DEF);
	DEFINE_LOCATION(v, BINARY_DEF);
	DEFINE_JTABLE(v);
    }

    while(sz--) {
	uint32 val = *hp;
	JUMP(v, tag_val_def(val));

	LOCATION(v,BIG_DEF);
#if defined(BEAM)
	if (is_nil(val)) {
	    hp++;
	    continue;
	}
#endif
	/* fall through */
	LOCATION(v,FLOAT_DEF);
	LOCATION(v,LIST_DEF);
	LOCATION(v,TUPLE_DEF) {
	    if (ignore_hilo || ptr_within(ptr_val(val), low, high))
		*hp = offset_ptr(val, offs);
	    hp++;
	    continue;
	}
	LOCATION(v, SMALL_DEF);
	LOCATION(v, CP0_DEF);
	LOCATION(v, CP4_DEF);
	LOCATION(v, CP8_DEF);
	LOCATION(v, CP12_DEF);
	LOCATION(v, CATCH_DEF);
	LOCATION(v, ATOM_DEF);
	LOCATION(v, PID_DEF);
	LOCATION(v, REFER_DEF);
	LOCATION(v, PORT_DEF);
	LOCATION(v, BLANK_DEF);
	LOCATION(v, BINARY_DEF) {
	    hp++;
	    continue;
	}
	JUMP_END
    }
}
#endif

/*
** Offset pointers into heap (works for both stack and heap)
** (BEAM MUST CALL ofset_heap_ptr ON STACK)
**  Only offset pointers that point into the interval of low and high 
** unless both low and high == 0, when we offset all pointers
*/

static void offset_heap(uint32 *hp, uint32 sz, sint32 offs, 
			uint32 *low, uint32 *high)
{
#if defined(JAM)
    DECL_JVALUE(v, FRAME_DEF)
#endif /* JAM */
    DECL_JVALUE(v, SMALL_DEF)
    DECL_JVALUE(v, BIG_DEF)
    DECL_JVALUE(v, FLOAT_DEF)
    DECL_JVALUE(v, ATOM_DEF)
    DECL_JVALUE(v, REFER_DEF)
    DECL_JVALUE(v, PORT_DEF)
    DECL_JVALUE(v, PID_DEF)
    DECL_JVALUE(v, TUPLE_DEF)
#if defined(JAM)
    DECL_JVALUE(v, NIL_DEF)
    DECL_JVALUE(v, CATCH_DEF)
#endif /* JAM */
    DECL_JVALUE(v, LIST_DEF)
    DECL_JVALUE(v, ARITYVAL_DEF)
    DECL_JVALUE(v, MOVED_DEF)
    DECL_JVALUE(v, THING_DEF)
    DECL_JVALUE(v, BINARY_DEF)
    DECL_JTABLE(v, 16)

    int ignore_hilo = (high == 0 && low == 0);

    if (JTABLE_NEED_INIT(v)) {
#if defined(JAM)
	DEFINE_LOCATION(v, FRAME_DEF);
#endif /* JAM */
	DEFINE_LOCATION(v, SMALL_DEF);
	DEFINE_LOCATION(v, BIG_DEF);
	DEFINE_LOCATION(v, FLOAT_DEF);
	DEFINE_LOCATION(v, ATOM_DEF);
	DEFINE_LOCATION(v, REFER_DEF);
	DEFINE_LOCATION(v, PORT_DEF);
	DEFINE_LOCATION(v, PID_DEF);
	DEFINE_LOCATION(v, TUPLE_DEF);
#if defined(JAM)
	DEFINE_LOCATION(v, NIL_DEF);
	DEFINE_LOCATION(v, CATCH_DEF);
#endif /* JAM */
	DEFINE_LOCATION(v, LIST_DEF);
	DEFINE_LOCATION(v, ARITYVAL_DEF);
	DEFINE_LOCATION(v, MOVED_DEF);
	DEFINE_LOCATION(v, THING_DEF);
	DEFINE_LOCATION(v, BINARY_DEF);
	DEFINE_JTABLE(v);
    }

    while(sz--) {
	uint32 val = *hp;
	JUMP(v, tag_val_def(val));

	LOCATION(v,BIG_DEF);
#if defined(BEAM)
	if (is_nil(val)) {
	    hp++;
	    continue;
	}
#endif
	/* fall through */
	LOCATION(v,FLOAT_DEF);
	LOCATION(v,LIST_DEF);
	LOCATION(v,TUPLE_DEF) {
	    if (ignore_hilo || ptr_within(ptr_val(val), low, high))
		*hp = offset_ptr(val, offs);
	    hp++;
	    continue;
	}
	LOCATION(v,THING_DEF) {
	    uint32 tari = thing_arityval(val);
	    hp += (tari + 1);
	    sz -= tari;
	    continue;
	}
	LOCATION(v, SMALL_DEF);
#if defined(JAM)
	LOCATION(v, NIL_DEF);
#endif /* JAM */
	LOCATION(v, ATOM_DEF);
	LOCATION(v, PID_DEF);
	LOCATION(v, REFER_DEF);
	LOCATION(v, PORT_DEF);
	LOCATION(v, ARITYVAL_DEF);
	LOCATION(v, BINARY_DEF) {
	    hp++;
	    continue;
	}
#if defined(JAM)
	LOCATION(v, CATCH_DEF) {
	    hp += 2;
	    sz--;
	    continue;
	}
	LOCATION(v, FRAME_DEF) {
	    hp += 4;
	    sz -= 3;
	    continue;
	}
#endif /* JAM */
	LOCATION(v, MOVED_DEF) {
	    erl_exit(1, "move mark found: 0x%08x at 0x%08x\n",
		     val, hp);
	}
	JUMP_END
    }
}


/*
** Offset pointer in message queue
*/
static void offset_mqueue(Process *p, sint32 offs, 
			  uint32 *low, uint32 *high) 
{
    ErlMessage* mp = p->msg.first;
    int ignore_hilo = (high == 0 && low == 0);

    while(mp != NULL) {
        uint32 mesg = mp->mesg;
        switch(tag_val_def(mesg)) {
        case BIG_DEF:
#if defined(BEAM)
            if (is_nil(mesg))
                break;
#endif
            /* fall through */
        case FLOAT_DEF:
        case LIST_DEF:
        case TUPLE_DEF:
	    if (ignore_hilo || (ptr_within(ptr_val(mesg), low, high)))
		mp->mesg = offset_ptr(mesg, offs);
            break;
        }

#ifdef SEQ_TRACE
	ASSERT((is_nil(mp->seq_trace_token) || is_tuple(mp->seq_trace_token)));
	mesg = mp->seq_trace_token;
	if (is_tuple(mesg) && (ignore_hilo || (ptr_within(ptr_val(mesg), low, high)))) {
	    mp->seq_trace_token = offset_ptr(mesg, offs);
        }
#endif
        mp = mp->next;
    }
}


static void
restore_rootset(Process *p, Rootset *rootset)
{
    uint32 *v_ptr;
    ErlMessage* mp;
    ErlMessageBuffer* bp;

    /*
     * Restore all message pointers.
     */
    mp = p->msg.first;
    v_ptr = rootset->v_msg;
    while (mp != NULL) {
	mp->mesg = *v_ptr++;
#ifdef SEQ_TRACE
	ASSERT((is_nil(*v_ptr) || is_tuple(*v_ptr)));
	mp->seq_trace_token = *v_ptr++;
#endif
	mp = mp->next;
    }
    
    if (rootset->v_msg != rootset->def_msg) {
	sys_free(rootset->v_msg);
    }

    /*
     * Remove all message buffers.
     */
    bp = p->mbuf;
    p->mbuf = NULL;
    p->mbuf_sz = 0;
    p->mbuf_struct_sz = 0;
    while (bp != NULL) {
	ErlMessageBuffer* next_bp = bp->next;
#ifdef DEBUG
	sys_memset(bp->mem, 0xff, bp->size*sizeof(uint32));
#endif 
	free_message_buffer(bp);
	bp = next_bp;
    }
}


/*
**  Garbage collect a heap, copy all data to a new fresh heap
**  Add garbage collect for objects in
**  objv as well. And update them.
**  Returns 0 if OK, <0 on error.
**  Parameters:
**  p: The process to build rootset from.
**  n_hstart: The start of the new heap (where saved objects 
**  are moved).
**  pn_htop: [in-out] Address of pointer to the new heap top,
**  should contain new value upon return from this function.
**  objv: Vector of "extra" objects to be "saved".
**  nobj: Number of objects in objv. 
*/

static int do_fullsweep_gc(Process *p, uint32 *n_hstart, 
			   uint32 **pn_htop, 
			   uint32 *objv, int nobj)
{
    /* Stack scan jump table */
#if defined(JAM)
    DECL_JVALUE(vs, FRAME_DEF)
#endif /* JAM */
#if defined(BEAM)
    DECL_JVALUE(vs, CP0_DEF)
    DECL_JVALUE(vs, CP4_DEF)
    DECL_JVALUE(vs, CP8_DEF)
    DECL_JVALUE(vs, CP12_DEF)
    DECL_JVALUE(vs, BLANK_DEF)
#endif /* BEAM */
    DECL_JVALUE(vs, SMALL_DEF)
    DECL_JVALUE(vs, BIG_DEF)
    DECL_JVALUE(vs, FLOAT_DEF)
    DECL_JVALUE(vs, ATOM_DEF)
    DECL_JVALUE(vs, REFER_DEF)
    DECL_JVALUE(vs, PORT_DEF)
    DECL_JVALUE(vs, PID_DEF)
    DECL_JVALUE(vs, TUPLE_DEF)
#if defined(JAM)
    DECL_JVALUE(vs, NIL_DEF)
#endif /* JAM */
    DECL_JVALUE(vs, LIST_DEF)
#if defined(JAM)
    DECL_JVALUE(vs, ARITYVAL_DEF)
    DECL_JVALUE(vs, MOVED_DEF)
    DECL_JVALUE(vs, THING_DEF)
#endif /* JAM */
    DECL_JVALUE(vs, CATCH_DEF)
    DECL_JVALUE(vs, BINARY_DEF)
    /* Heap scan jump table */
#if defined(JAM)
    DECL_JVALUE(hs, FRAME_DEF)
#endif /* JAM */
#if defined(BEAM)
    DECL_JVALUE(hs, CP0_DEF)
    DECL_JVALUE(hs, CP4_DEF)
    DECL_JVALUE(hs, CP8_DEF)
      /* DECL_JVALUE(hs, CP12_DEF) save as MOVED_DEF */
#endif /* BEAM */
    DECL_JVALUE(hs, SMALL_DEF)
    DECL_JVALUE(hs, BIG_DEF)
    DECL_JVALUE(hs, FLOAT_DEF)
    DECL_JVALUE(hs, ATOM_DEF)
    DECL_JVALUE(hs, REFER_DEF)
    DECL_JVALUE(hs, PORT_DEF)
    DECL_JVALUE(hs, PID_DEF)
    DECL_JVALUE(hs, TUPLE_DEF)
#if defined(JAM)
    DECL_JVALUE(hs, NIL_DEF)
    DECL_JVALUE(hs, CATCH_DEF)
#endif /* JAM */
    DECL_JVALUE(hs, LIST_DEF)
    DECL_JVALUE(hs, ARITYVAL_DEF)
    DECL_JVALUE(hs, MOVED_DEF)
    DECL_JVALUE(hs, THING_DEF)
    DECL_JVALUE(hs, BINARY_DEF)

    DECL_JTABLE(vs, 16)
    DECL_JTABLE(hs, 16)

    Rootset rootset;
    uint32 *n_htop;
    uint32 *n_hp;
    int tmp;
    int n;

#ifdef GC_HEAP_TRACE
    fprintf(stderr, "Fullsweep GC proc: %d\n", (int)get_number(p->id));
#endif
    if (JTABLE_NEED_INIT(vs) || JTABLE_NEED_INIT(hs)) {
#if defined(JAM)
	DEFINE_LOCATION(vs, FRAME_DEF);
#endif /* JAM */
#if defined(BEAM)
	DEFINE_LOCATION(vs, CP0_DEF);
	DEFINE_LOCATION(vs, CP4_DEF);
	DEFINE_LOCATION(vs, CP8_DEF);
	DEFINE_LOCATION(vs, CP12_DEF);
	DEFINE_LOCATION(vs, BLANK_DEF);
#endif /* BEAM */
	DEFINE_LOCATION(vs, SMALL_DEF);
	DEFINE_LOCATION(vs, BIG_DEF);
	DEFINE_LOCATION(vs, FLOAT_DEF);
	DEFINE_LOCATION(vs, ATOM_DEF);
	DEFINE_LOCATION(vs, REFER_DEF);
	DEFINE_LOCATION(vs, PORT_DEF);
	DEFINE_LOCATION(vs, PID_DEF);
	DEFINE_LOCATION(vs, TUPLE_DEF);
#if defined(JAM)
	DEFINE_LOCATION(vs, NIL_DEF);
#endif /* JAM */
	DEFINE_LOCATION(vs, LIST_DEF);
#if defined(JAM)
	DEFINE_LOCATION(vs, ARITYVAL_DEF);
	DEFINE_LOCATION(vs, MOVED_DEF);
	DEFINE_LOCATION(vs, THING_DEF);
#endif /* JAM */
	DEFINE_LOCATION(vs, CATCH_DEF);
	DEFINE_LOCATION(vs, BINARY_DEF);
	DEFINE_JTABLE(vs);

#if defined(JAM)
	DEFINE_LOCATION(hs, FRAME_DEF);
#endif /* JAM */
#if defined(BEAM)
	DEFINE_LOCATION(hs, CP0_DEF);
	DEFINE_LOCATION(hs, CP4_DEF);
	DEFINE_LOCATION(hs, CP8_DEF);
	/* DEFINE_LOCATION(hs, CP12_DEF); Same as MOVED_DEF */
#endif /* BEAM */
	DEFINE_LOCATION(hs, SMALL_DEF);
	DEFINE_LOCATION(hs, BIG_DEF);
	DEFINE_LOCATION(hs, FLOAT_DEF);
	DEFINE_LOCATION(hs, ATOM_DEF);
	DEFINE_LOCATION(hs, REFER_DEF);
	DEFINE_LOCATION(hs, PORT_DEF);
	DEFINE_LOCATION(hs, PID_DEF);
	DEFINE_LOCATION(hs, TUPLE_DEF);
#if defined(JAM)
	DEFINE_LOCATION(hs, NIL_DEF);
	DEFINE_LOCATION(hs, CATCH_DEF);
#endif /* JAM */
	DEFINE_LOCATION(hs, LIST_DEF);
	DEFINE_LOCATION(hs, ARITYVAL_DEF);
	DEFINE_LOCATION(hs, MOVED_DEF);
	DEFINE_LOCATION(hs, THING_DEF);
	DEFINE_LOCATION(hs, BINARY_DEF);
	DEFINE_JTABLE(hs);
    }
    n_htop = *pn_htop; /* Hopefully optimized into register */
    n = setup_rootset(p, objv, nobj, &rootset);

    while (n--) {
	uint32* g_ptr = rootset.v[n];
	uint32 g_sz = rootset.sz[n];
	
	while(g_sz--) {
	    uint32 *ptr;
	    uint32 val;
	    uint32 gval = *g_ptr;

	    JUMP(vs, tag_val_def(gval));

	    LOCATION(vs, FLOAT_DEF) {
		ptr = ptr_val(gval);
		val = *ptr;
		if (is_moved(val))
		    *g_ptr++ = make_float(ptr_val(val));
		else {
		    ASSERT(within(ptr, p));
		    ASSERT(is_thing(val));
		    *g_ptr++ = gval = make_float(n_htop);
		    *ptr++ = make_moved(gval);   /* Store move address */
		    *n_htop++ = val;             /* Store thing */
		    *n_htop++ = *ptr++;          /* Copy float part 1 */
		    *n_htop++ = *ptr;            /* Copy float part 2 */
		}
		continue;
	    }

	    LOCATION(vs, BIG_DEF) {
#if defined(BEAM)
	      if (is_nil(gval)) {
		g_ptr++;
		continue;
	      }
#endif /* BEAM */
		ptr = ptr_val(gval);
		val = *ptr;
		if (is_moved(val))
		    *g_ptr++ = make_big(ptr_val(val));
		else {
		    ASSERT(within(ptr, p));
		    ASSERT(is_thing(val));
		    *g_ptr++ = gval = make_big(n_htop);
		    *ptr++ = make_moved(gval);   /* Store move address */
		    *n_htop++ = val;             /* Store thing */
		    tmp = thing_arityval(val);    /* Get arity value */
		    while(tmp--)
			*n_htop++ = *ptr++;
		}
		continue;
	    }

	    LOCATION(vs, TUPLE_DEF) {
		ptr = ptr_val(gval);
		val = *ptr;
		if (is_moved(val))
		    *g_ptr++ = make_tuple(ptr_val(val));
		else {
		    ASSERT(within(ptr, p));
		    ASSERT(is_arity_value(val));
		    *g_ptr++ = gval = make_tuple(n_htop);
		    *ptr++ = make_moved(gval);   /* Store move address */
		    *n_htop++ = val;             /* Store arity val */
		    tmp = arityval(val);          /* Get arity value */
		    while(tmp--)
			*n_htop++ = *ptr++;
		}
		continue;
	    }

	    LOCATION(vs, LIST_DEF) {
		ptr = ptr_val(gval);
		val = *ptr;
		if (is_moved(val))
		    *g_ptr++ = make_list(ptr_val(val));
		else {
		    ASSERT(within(ptr, p));
		    *g_ptr++ = gval = make_list(n_htop);
		    *ptr++ = make_moved(gval);   /* Store move address */
		    *n_htop++ = val;             /* Store head */
		    *n_htop++ = *ptr;            /* Store tail */
		}
		continue;
	    }

#if defined(BEAM)
	    LOCATION(vs, CP0_DEF);
	    LOCATION(vs, CP4_DEF);
	    LOCATION(vs, CP8_DEF);
	    LOCATION(vs, CP12_DEF);
	    LOCATION(vs, BLANK_DEF);
	    LOCATION(vs, CATCH_DEF);
#endif /* BEAM */
	    LOCATION(vs, SMALL_DEF);
#if defined(JAM)
	    LOCATION(vs, NIL_DEF);
#endif /* JAM */
	    LOCATION(vs, ATOM_DEF);
	    LOCATION(vs, PID_DEF);
	    LOCATION(vs, REFER_DEF);
	    LOCATION(vs, PORT_DEF) {
		g_ptr++;
		continue;
	    }

#if defined(JAM)
	    LOCATION(vs, CATCH_DEF)  {
		g_ptr += 2;
		g_sz--;
		continue;
	    }

	    LOCATION(vs, FRAME_DEF)  {
		g_ptr += 4;
		g_sz -= 3;
		continue;
	    }
#endif /* JAM */

	    LOCATION(vs, BINARY_DEF) {
		((ProcBin*) ptr_val(gval))->mark = BIN_FULLSWEAP_MARKED;
		g_ptr++;
		continue;
	    }

#if defined(JAM)
	    LOCATION(vs, MOVED_DEF);
	    LOCATION(vs, ARITYVAL_DEF);
	    LOCATION(vs, THING_DEF) {
		erl_exit(1, "%s: bad data on stack GC pass 1 0x%08x at 0x%08x\n",
			 print_pid(p), gval, g_ptr);
	    }
#endif /* JAM */
	    JUMP_END;
	}
    }


    /* now all references on the stack point to the new heap. However 
       most references on the new heap point to the old heap so the next stage
       is to scan through the new heap evacuating data from the old heap
       until all is changed */

    n_hp = n_hstart;
    
    while (n_hp != n_htop) {
	uint32 *ptr;
	uint32 val;
	uint32 gval = *n_hp;

	JUMP(hs, tag_val_def(gval));
	LOCATION(hs, FLOAT_DEF) {
	    ptr = ptr_val(gval);
	    val = *ptr;
	    if (is_moved(val))
		*n_hp++ = make_float(ptr_val(val));
	    else {
		ASSERT(within(ptr, p));
		ASSERT(is_thing(val));
		*n_hp++ = gval = make_float(n_htop);
		*ptr++ = make_moved(gval);   /* Store move address */
		*n_htop++ = val;             /* Store thing */
		*n_htop++ = *ptr++;          /* Copy float part 1 */
		*n_htop++ = *ptr;            /* Copy float part 2 */
	    }
	    continue;
	}

	LOCATION(hs, BIG_DEF) {
#if defined(BEAM)
	    if (is_nil(gval)) {
	      n_hp++;
	      continue;
	    }
#endif /* BEAM */
	    ptr = ptr_val(gval);
	    val = *ptr;
	    if (is_moved(val))
		*n_hp++ = make_big(ptr_val(val));
	    else {
		ASSERT(within(ptr, p));
		ASSERT(is_thing(val));
		*n_hp++ = gval = make_big(n_htop);
		*ptr++ = make_moved(gval);   /* Store move address */
		*n_htop++ = val;             /* Store thing */
		tmp = thing_arityval(val);    /* Get arity value */
		while(tmp--) 
		    *n_htop++ = *ptr++;
	    }
	    continue;
	}

	LOCATION(hs, TUPLE_DEF) {
	    ptr = ptr_val(gval);
	    val = *ptr;
	    if (is_moved(val))
		*n_hp++ = make_tuple(ptr_val(val));
	    else {
		ASSERT(within(ptr, p));
		ASSERT(is_arity_value(val));
		*n_hp++ = gval = make_tuple(n_htop);
		*ptr++ = make_moved(gval);   /* Store move address */
		*n_htop++ = val;             /* Store arity val */
		tmp = arityval(val);          /* Get arity value */
		while(tmp--)
		    *n_htop++ = *ptr++;
	    }
	    continue;
	}

	LOCATION(hs, LIST_DEF) {
	    ptr = ptr_val(gval);
	    val = *ptr;
	    if (is_moved(val))
		*n_hp++ = make_list(ptr_val(val));
	    else {
		ASSERT(within(ptr, p));
		*n_hp++ = gval = make_list(n_htop);
		*ptr++ = make_moved(gval);   /* Store move address */
		*n_htop++ = val;             /* Store head */
		*n_htop++ = *ptr;            /* Store tail */
	    }
	    continue;
	}

	LOCATION(hs, THING_DEF) {
	    n_hp += (thing_arityval(gval)+1);
	    continue;
	}

	LOCATION(hs, SMALL_DEF);
#if defined(JAM)
	LOCATION(hs, NIL_DEF);
#endif /* JAM */
	LOCATION(hs, ATOM_DEF);
	LOCATION(hs, PID_DEF);
	LOCATION(hs, REFER_DEF);
	LOCATION(hs, PORT_DEF);
	LOCATION(hs, ARITYVAL_DEF) {
	    n_hp++;
	    continue;
	}
	LOCATION(hs, BINARY_DEF) {
	    ((ProcBin*) ptr_val(gval))->mark = BIN_FULLSWEAP_MARKED;
	    n_hp++;
	    continue;
	}

#if defined(BEAM)
	LOCATION(hs, CP0_DEF);
	LOCATION(hs, CP4_DEF);
	LOCATION(hs, CP8_DEF);
	/* LOCATION(hs, CP12_DEF);  Same as MOVED_DEF */
#endif /* BEAM */
#if defined(JAM)
	LOCATION(hs, FRAME_DEF);
	LOCATION(hs, CATCH_DEF);
#endif /* JAM */
	LOCATION(hs, MOVED_DEF) {
	    display(p->id, CERR);
	    erl_exit(1, "%s: GC: bad data on heap - pass 2 0x%08x at 0x%08x\n",
		     print_pid(p), gval, n_hp);
	}
	JUMP_END;
    }

    restore_rootset(p, &rootset);
    *pn_htop = n_htop;

    return(0);
}


static int create_old_heap(Process *p, int new_sz)
{
    uint32 *n_old;
   
    /* Create new,empty old_heap */
    n_old = (uint32 *) safe_alloc(sizeof(uint32)*new_sz);

    p->old_hend = n_old + new_sz;
    p->old_heap = n_old;
    p->old_htop = n_old;
    return(0);
}
/* 
** Set up parameters and call do_fullsweep_gc.
** This function is used when using gen_gc and the old_heap
** needs garbing.
** Returns:
** 0 if OK < 0 on error.
** Parameters:
** p: The process who is being garbage collected.
** new_sz: The wanted size of the old_heap after collecting.
** objv: Vector of "extra" objects to be "saved".
** nobj: Number of objects in objv.
*/
static int fullsweep_old_heap(Process *p, int new_sz,
			      uint32 *objv, int nobj)
{
    uint32 *n_hstart;     /* Start of new (old_)heap */
    uint32 *n_htop;       /* Top of new (old_)heap */
    uint32 *n_old;
    uint32 saved_status;  /* Save the process status.*/
    int ret;              /* return value from fullsweep, 
		             currently always 0 */
   
    /* Create new,empty old_heap */
    n_old = (uint32 *) safe_alloc(sizeof(uint32)*new_sz);
    /* high_water, low_water and hend are set up after fullsweep, 
       when new_heap is empty. */

    n_hstart = n_htop = n_old;
    saved_status = p->status;
    p->status = P_GARBING;

    /* This should never fail */
    if((ret = do_fullsweep_gc(p, n_hstart, &n_htop, 
			  objv, nobj)) != 0)
	erl_exit(1, "%s: Fullsweep GC failed with status %d.\n",
		 print_pid(p), ret);
    
    /* new_heap is empty and old old_heap is to be discarded. */
    p->htop = p->heap;
    SET_HEAP_MARGIN(p, p->hend - H_MARGIN);

    if(p->old_heap != NULL){
#ifdef DEBUG
	sys_memset(p->old_heap, 0xff,
		   (p->old_hend - p->old_heap)*sizeof(uint32));
#endif
	sys_free(p->old_heap);
    }
    p->old_hend = n_old + new_sz;
    p->old_heap = n_old;
    p->old_htop = n_htop;
    p->low_water = p->high_water = p->heap;
    p->flags &= ~F_GCFLIP; /* Reset generational state
			      whatever it was before this. */

    p->status = saved_status;
    if (p->mso || p->old_mso) 
	fullsweap_bin_gc(p);
    return(0);
}

/* 
** Set up parameters and call do_fullsweep_gc.
** This function is used when using fullsweep gc.
** This replaces gen_gc for processes that use the
** fullsweep algorithm and don't have an old_heap.
** Returns:
** 0 if OK < 0 on error.
** Parameters:
** p: The process who is being garbage collected.
** new_sz: The wanted size of the heap after collecting.
** objv: Vector of "extra" objects to be "saved".
** nobj: Number of objects in objv.
*/
static int fullsweep_heap(Process *p, int new_sz,
			      uint32 *objv, int nobj)
{
    uint32 *n_hstart;     /* Start of new heap */
    uint32 *n_htop;       /* Top of new heap */
    uint32 *n_heap;   /* The new heap */
    uint32 saved_status;  /* Save the process status.*/
    int ret;              /* return value from fullsweep, 
		             currently always 0 */
#if defined(BEAM)
    int n;
#endif

    /* Create new,empty heap */
    n_heap = (uint32 *) safe_alloc(sizeof(uint32)*new_sz);

    n_hstart = n_htop = n_heap;
    saved_status = p->status;
    p->status = P_GARBING;

    /* This should never fail */
    if((ret = do_fullsweep_gc(p, n_hstart, &n_htop, 
			  objv, nobj)) != 0)
	erl_exit(1, "%s: Fullsweep GC failed with status %d.\n",
		 print_pid(p), ret);

#if defined(BEAM)
    /* Move the stack, the beam stack is "in the heap" */
    n = p->stack - p->stop;
    sys_memcpy(n_heap + new_sz - n, p->stop, n * sizeof(uint32));
    p->stack = n_heap + new_sz;
    p->stop = p->stack - n;
#endif /* BEAM */
    
#ifdef DEBUG
    sys_memset(p->heap, 0xff, (p->hend - p->heap)*sizeof(uint32));
#endif    
    sys_free(p->heap);
    p->hend = n_heap + new_sz;
    p->heap = n_heap;
    p->htop = n_htop;
    p->heap_sz = new_sz;

    p->low_water = n_heap;    /* These are used to switch */
    p->high_water = p->htop;  /* GC algorithm */

    SET_HEAP_MARGIN(p, p->hend - H_MARGIN);

    p->status = saved_status;
    if (p->mso) 
	proc_bin_gc(p);
    return(0);
}

#if defined(JAM)
/*
** Offset stack
*/
static void offset_stack(uint32 *sp, uint32 *sp_end, sint32 offs) /****I****/
     /*uint32* sp; uint32* sp_end; sint32 offs;*/
{
    while (sp < sp_end) {
	if (is_frame(*sp)) {
	    sp[FRAME_FP] = offset_frame(sp[FRAME_FP], offs);
	    sp[FRAME_AP] = offset_frame(sp[FRAME_AP], offs);
	    sp += FRAME_SIZE;
	}
	else if (is_catch(*sp)) {
	    if (ptr_val(sp[CATCH_PTR]) != ENULL)
		sp[CATCH_PTR] = offset_ptr(sp[CATCH_PTR], offs);
	    sp += CATCH_SIZE;
	}
	else
	    sp++;
    }
}



/*
** Change the stack size to 'new_sz'
**
*/

static void stack_change(Process *p, uint32 new_sz)
{
    uint32 *new_stack;
    sint32 offs;
    uint32 stack_size = p->stop - p->stack;
    uint32 sz = p->stack_sz;

    if (stack_size > sz)
	erl_exit(1, "%s: Overrun stack, panic\n", print_pid(p));
    if (p->stack[sz] != S_PATTERN || p->stack[sz+1] != S_PATTERN)
	erl_exit(1, "%s: Overrun stack (bad pattern), panic\n",
		 print_pid(p));

    sz = new_sz;
    new_stack = (uint32 *) REALLOC((char*)p->stack,
				   sizeof(uint32)*(sz+2));
    new_stack[sz] = S_PATTERN;
    new_stack[sz+1] = S_PATTERN;

    if ((offs = new_stack - p->stack) != 0) {
	offset_stack(new_stack, new_stack+stack_size, offs);

	if (p->catches != ENULL)
	    p->catches += offs;
	p->fp += offs;
	p->ap += offs;
	p->stop += offs;
	p->stack = new_stack;
    }
    p->stack_sz = new_sz;
    p->stack_margin = p->stack + sz - S_MARGIN;

}

/* We need to grow and need at least 'need' number of words */
void stack_grow(Process *p, uint32 need)
{
    uint32 sz;

#ifdef HARDDEBUG
    check_stack(p, "grow");
#endif
    need += (p->stop - p->stack) + 2;

    sz = next_heap_size(need,0);
    
#ifdef GC_STACK_TRACE
    fprintf(stderr, "STACK: GROW (%d) FROM %d TO %d\n",
	    get_number(p->id), p->stack_sz, sz-2);
#endif
    stack_change(p, sz-2);

#ifdef HARDDEBUG
    check_stack(p, "grow2");
#endif

}

/* We need at least 'need' words on the stack */
void stack_shrink(Process *p, uint32 need)
{
    uint32 stack_sz = p->stack_sz + 2;
    uint32 sz;

#ifdef HARDDEBUG
    check_stack(p, "shrink");
#endif

    need += (p->stop - p->stack) + 2;
    if (need < S_MIN_SIZE)
	need = S_MIN_SIZE;

    sz = next_heap_size(need,0);

    if (sz != stack_sz) {
#ifdef GC_STACK_TRACE
	fprintf(stderr, "STACK: SHRINK (%d) FROM %d DOWNTO %d (used %d)\n",
		get_number(p->id), p->stack_sz, sz-2, p->stop-p->stack);
#endif
	stack_change(p, sz-2);
    }
#ifdef HARDDEBUG
    check_stack(p, "shrink2");
#endif

}

static void stack_move(Process *p)
{
    uint32 *new_stack;
    sint32 offs;
    uint32 stack_size = p->stop - p->stack;
    uint32 sz = p->stack_sz;

    new_stack = (uint32 *) safe_alloc(sizeof(uint32)*(sz+2));
    sys_memcpy(new_stack, p->stack, sz*sizeof(uint32));

    new_stack[sz] = S_PATTERN;
    new_stack[sz+1] = S_PATTERN;

    VERBOSE({
	display(p->id,COUT);
	erl_printf(COUT," moving stack, size = %d\n", sz);
    });

    offs = new_stack - p->stack;
    offset_stack(new_stack, new_stack+stack_size, offs);
    if (p->catches != ENULL)
	p->catches += offs;
    p->fp += offs;
    p->ap += offs;
    p->stop += offs;
    sys_free(p->stack);
    p->stack = new_stack;
    p->stack_margin = p->stack + sz - S_MARGIN;
}
#endif /* JAM */

/*
 * Find the next heap size equal to or greater than the given size (if offset == 0).
 *
 * If offset is 1, the next higher heap size is returned (always greater than size).
 */
int
next_heap_size(int size, int offset)
{
    if (size < heap_sizes[0]) {
	return heap_sizes[0];
    } else {
	int* low = heap_sizes;
	int* high = heap_sizes + num_heap_sizes;
	int* mid;

	while (low < high) {
	    mid = low + (high-low) / 2;
	    if (size < mid[0]) {
		high = mid;
	    } else if (size == mid[0]) {
		ASSERT(mid+offset-heap_sizes < num_heap_sizes);
		return mid[offset];
	    } else if (size < mid[1]) {
		ASSERT(mid[0] < size && size <= mid[1]);
		ASSERT(mid+offset-heap_sizes < num_heap_sizes);
		return mid[offset+1];
	    } else {
		low = mid + 1;
	    }
	}
	erl_exit(1, "no next heap size found: %d, offset %d\n");
    }
    return 0;
}

    
static void offset_rootset(Process *p, int offs, 
			   uint32 *low, uint32 *high, 
			   uint32 *objv, int nobj)
{
    offset_heap(&p->dictionary, 1, offs, low, high);
#ifdef SEQ_TRACE
    offset_heap(&p->seq_trace_token, 1, offs, low, high);
#endif
#if defined(JAM)
    offset_heap(&p->fvalue, 1, offs, low, high);
#endif
    offset_mqueue(p, offs, low, high);
#if defined(BEAM)
    offset_heap_ptr(p->stop, (p->stack - p->stop), offs, low, high);
#elif defined(JAM)
    offset_heap(p->stack, (p->stop - p->stack), offs, low, high);
#endif
    if (nobj > 0)
	offset_heap(objv, nobj, offs, low, high);
}



/*
** Change the new heap size to 'new_sz'
** If p->htop - p->heap ==0, all live data resides on the old heap
** and we need not do any offsetting at all, this is typically 
** the case after a fullsweep collection.
*/
static void heap_change(Process *p, uint32 new_sz, 
			uint32 *objv, int nobj)
{
    uint32* new_heap;
    int heap_size = p->htop - p->heap;

#if defined(BEAM)
    uint32 *prev_stop = p->stop;
    int stack_size;
#endif
    sint32 offs;

#if defined(BEAM) 
    if (new_sz < p->heap_sz) { 
	new_heap = (uint32*) safe_alloc(sizeof(uint32)*new_sz);
	sys_memcpy(new_heap, p->heap, new_sz * sizeof(uint32));
    }
    else
#endif
	new_heap = (uint32*) REALLOC((void*)p->heap, sizeof(uint32)*new_sz);

#ifdef GC_HEAP_TRACE
    fprintf(stderr, "NEWHEAP: CHANGE (%d) FROM %d TO %d (used %d)\n",
	    get_number(p->id), p->heap_sz, new_sz, heap_size);
#endif

    if ((offs = new_heap - p->heap) != 0) {
	uint32 low_water;
	uint32 high_water;

	offset_heap(new_heap, heap_size, offs, p->heap, p->htop);

	/* Even if we are using fullsweep, we need to keep the
	   water marks OK, we DO use them when switching...*/
	low_water = p->low_water - p->heap;
	high_water = p->high_water - p->heap;
	p->low_water = new_heap + low_water;
	p->high_water = new_heap + high_water;

#if defined(BEAM)
	stack_size = p->stack - p->stop;
	if (new_sz < p->heap_sz) { /* Shrinking */
	    sys_memcpy(new_heap + new_sz - stack_size,
		       p->stop, 
		       stack_size * sizeof(uint32));
	    p->stack = new_heap + new_sz;
	    p->stop = new_heap + new_sz - stack_size;
	    sys_free(p->heap);
	}
	else {
	    prev_stop = new_heap + (p->stop - p->heap);
	    p->stack = new_heap + new_sz;
	    p->stop = new_heap + new_sz - stack_size;
	    sys_memmove(p->stop, prev_stop, stack_size * sizeof(uint32));
	}
	
#endif
	if (heap_size > 0)
	    offset_rootset(p, offs, p->heap, p->htop, objv, nobj);
	p->htop = new_heap + heap_size;
	p->heap = new_heap;
    }
#if defined(BEAM)
    else {  /* offs == 0 */
	stack_size = p->stack - p->stop;
	p->stack = new_heap + new_sz;
	p->stop = new_heap + new_sz - stack_size;
	sys_memmove(p->stop, prev_stop, stack_size * sizeof(uint32));
    }
#endif
    p->heap_sz = new_sz;
    p->hend = p->heap + new_sz;
    SET_HEAP_MARGIN(p, p->hend - H_MARGIN);
}



/* New heap is empty when this function gets called */

static void shrink_old_heap(Process *p, uint32 new_sz, 
			    uint32 *objv, int nobj)
{
    uint32* new_heap;
    sint32 used, offs;
    uint32 *old = p->old_heap;
    uint32 *old_heap_ptr = p->old_heap;
#ifdef GC_HEAP_TRACE
    uint32 heap_size = (p->old_hend - p->old_heap);
#endif

    used = p->old_htop - p->old_heap;

    new_heap = (uint32 *) REALLOC((void*) old, sizeof(uint32) * new_sz);
    offs = new_heap - old_heap_ptr;

#ifdef GC_HEAP_TRACE
    fprintf(stderr, "OLDHEAP: CHANGE (%d) FROM %d DOWNTO %d \n",
	    get_number(p->id), heap_size, new_sz);
#endif
    if (offs != 0) {
	offset_heap(new_heap, used, offs, 0, 0);
	offset_rootset(p, offs, 0, 0, objv, nobj);
    }
    p->old_hend = new_heap + new_sz;
    p->old_htop = new_heap + used;
    p->old_heap = new_heap;
}

/*
 * Garbage collect a process.  For Beam, this must not be the running process
 * (the p->arity field is NOT valid when a process is running).
 */
int
do_gc(Process *p, int need)
{
#if defined(BEAM)
    return do_gc2(p, need, p->arg_reg, p->arity);
#elif defined(JAM)
    return do_gc2(p, need, 0, 0);
#endif
}


/*
** Garbage collect a process.  This entry point is used by Beam 
** when the running process is garbage collected.
** Parameters:
** p: Pointer to the process structure.
** need: Number of (erlang) words needed on the heap.
** objv: Array of terms to add to rootset, that is to preserve.
** nobj: Number of objects in objv.
*/

/* This macro is used before return from do_gc2 in order to
   detect as early as possible if something has gone wrong with 
   stop or htop
*/
#ifdef BEAM
#define OverRunCheck() \
    if (p->stop < p->htop) { \
	erl_exit(1, "%s: Overrun stack and heap at line %d\n", print_pid(p),__LINE__); \
    } 
#else
#define OverRunCheck()
#endif

int do_gc2(Process *p, int need, 
	   uint32 *objv, int nobj) 
{
    int size_before;
    int size_after;
    int need_after;

    int old_heap_need;
    int old_heap_sz;

    int wanted;
    int stack_size = 0;		/* Size of stack ON HEAP (0 for Jam). */
    int sz;


#if defined(BEAM)
    p->arith_avail = 0;
#ifdef DEBUG
    p->arith_check_me = NULL;
#endif
#endif

    garbage_cols++;
    CHECK(p);

    if (IS_TRACED_FL(p, F_TRACE_GC)) {
	trace_gc(p, am_gc_start);
    }

#if defined(JAM)
    sz = p->stack_sz;
    if (p->stop - p->stack > sz) {
	erl_exit(1, "%s: Overrun stack\n", print_pid(p));
    } 
    if (p->stack[sz] != S_PATTERN || p->stack[sz+1] != S_PATTERN) {
	erl_exit(1, "%s: Overrun stack (bad pattern)\n", p);
    }
    if (p->htop > p->hend) {
	erl_exit(1, "%s: Overrun heap\n"
		 "heap start 0x%08x, heap end 0x%08x, size 0x%08x\n",
		 print_pid(p), p->heap, p->htop, p->heap_sz);
    }
#else
#ifdef DEBUG
    { 
	int acc = 0;
	static int maxwrong = 0;
	ErlMessageBuffer* mb= p->mbuf;
	while (mb) {
	    acc+=mb->size;
	    mb=mb->next;
	}
	if (acc != p->mbuf_sz) {
	    if (acc - p->mbuf_sz > maxwrong){
		maxwrong = acc - p->mbuf_sz;
		erl_printf(CERR,"%s: Messagebuffer size"
			   " miscalculation shouldbe %d was %d\n", 
			   print_pid(p),acc,p->mbuf_sz);
	    }
	}
    }
#endif /* DEBUG */
    stack_size = p->stack - p->stop;
    if (p->stop < p->htop) {
	erl_exit(1, "%s: Overrun stack and heap\n", print_pid(p));
    }
#endif

    /* Size of heap before first GC */
    size_before = p->mbuf_sz + (p->htop - p->heap);

    /* Fullsweep GC */
    if (!IS_GEN_GC(p)) {
	if(p->high_water - p->low_water < p->gc_switch){
	
	    /* How much do we need (adding for message buffers) */
	    /* to just do the GC?                               */
	    sz = next_heap_size(p->heap_sz + p->mbuf_sz,0);

	    /* Should we grow although we don't actually need to? */
	    if(sz == p->heap_sz && (p->flags & F_HEAP_GROW)){
		sz = next_heap_size(p->heap_sz,1);
		p->flags &= ~F_HEAP_GROW;
	    }

	    /* Do fullsweep GC */
	    fullsweep_heap(p, sz, objv, nobj); 
	    size_after = (p->htop - p->heap);
	    reclaimed += (size_before - size_after);

	    /* Resize heap */
	    /* Calculate the actual need, stack_size is 0 for JAM */
	    need_after = size_after + need + stack_size + H_MARGIN;

	    if(p->heap_sz < need_after){
		/* To small, grow to match requested need */
		sz = next_heap_size(need_after,0);
		heap_change(p, sz, objv, nobj);
	    } else if(3 * p->heap_sz < 4 * need_after){
		/* Need more than 75% of current, postpone to next GC.*/
		p->flags |= F_HEAP_GROW;
	    } else if(4 * need_after < p->heap_sz &&
		      p->heap_sz > H_MIN_SIZE){
		/* We need less than 25% of the current heap, shrink.*/
		/* XXX - This is how it was done in the old GC:
		   wanted = 4 * need_after; 
		   I think this is better as fullsweep is used mainly on
		   small memory systems, but I could be wrong... */
		wanted = 2 * need_after; 
		if (wanted < p->min_heap_size) {
		    sz = p->min_heap_size;
		} else {
		    sz = next_heap_size(wanted, 0);
		}
		if (sz != p->heap_sz) {
		    heap_change(p, sz, objv, nobj);
		}
	    }

	    /* The heap is resized, one thing left to do, a special
	       case for JAM. A need of 0 means reallocate stack. */
#if defined(JAM)
	    if(need == 0)
		stack_move(p);
#endif /* JAM */
	    
	    /* Done. */
	    CHECK(p);
	    p->flags &= ~F_NEED_GC;
	    ASSERT(p->heap_sz == next_heap_size(p->heap_sz, 0));
	    if (IS_TRACED_FL(p, F_TRACE_GC)) {
		trace_gc(p, am_gc_end);
	    }
	    OverRunCheck();
	    return ((int) (p->htop - p->heap) / 10);
	} else {
	    /* Migrate GC algorithm */
	    if(p->high_water != p->low_water)
		p->flags |= F_GCFLIP; /* Tenure now. */
	    SET_GEN_GC(p);
#if defined(GC_SWITCH_TRACE)
	    fprintf(stderr, 
		    "Switched gc for %s, live data = %d "
		    "(switch at %d).\r\n",
		    print_pid(p), 
		    (int) (p->high_water - p->low_water),
		    (int) p->gc_switch);
#endif
	    /* Fall through to generational GC */
	}
    }

    /* Generational GC from here on*/
    if (p->old_heap == NULL && (p->flags & F_GCFLIP) != 0 &&
	p->old_mso == NULL) {
	create_old_heap(p, next_heap_size(p->high_water - p->low_water,0));
#ifdef OLD_HEAP_CREATION_TRACE
	fprintf(stderr,"Created old_heap for %s.\r\n",
		print_pid(p));
#endif
    }

    if (p->high_water - p->low_water <= p->old_hend - p->old_htop &&
	p->old_mso == NULL) {
	
	/*
	 * There is space enough in old_heap for everything
	 * below the high water mark AND there are no old binaries.
	 */
	
	gen_gc(p, next_heap_size(p->heap_sz + p->mbuf_sz, 0), objv, nobj);
	size_after = p->htop - p->heap;
	need_after = size_after + need + stack_size + H_MARGIN;
	reclaimed += (size_before - size_after);

	/* Don't even bother on reasonable small heaps */
	/* The reason for this is that after tenuring, we often */
	/* use a really small portion of new heap, therefore, unless */
	/* the heap size is substantial, we don't want to shrink */

	if ((p->heap_sz > 300) && 
	    (8 * need_after < p->heap_sz) &&
	    (p->heap_sz > (p->old_hend - p->old_heap))) {
	    if ((3 * need_after) > (p->old_hend - p->old_heap))
		wanted = 3 * need_after;
	    else
		wanted = (p->old_hend - p->old_heap);
		
	    if (wanted < p->min_heap_size) {
		wanted = p->min_heap_size;
	    } else {
		wanted = next_heap_size(wanted, 0);
	    }
	    
	    heap_change(p, wanted, objv, nobj);
	    CHECK(p);
	    p->flags &= ~F_NEED_GC;  
	    ASSERT(p->heap_sz == next_heap_size(p->heap_sz, 0));
	    if (IS_TRACED_FL(p, F_TRACE_GC)) {
		trace_gc(p, am_gc_end);
	    }
	    OverRunCheck();
	    return ((int) (p->htop - p->heap) / 10);
	}

	if (p->heap_sz >= need_after) {
	    p->flags &= ~F_NEED_GC;  /* Reset GC flag */
	    CHECK(p);
	    ASSERT(p->heap_sz == next_heap_size(p->heap_sz, 0));
	    if (IS_TRACED_FL(p, F_TRACE_GC)) {
		trace_gc(p, am_gc_end);
	    }
	    OverRunCheck();
	    return ((int) (p->htop - p->heap) / 10);
	}
#ifdef GC_HEAP_TRACE
	fprintf(stderr, "Did a gen_gc, still not enough room\n");
#endif
	/* The last gen_gc did not leave us enough room for need */
	/* We therefore run a full sweep on everything in order to make room */
    }
    
    /* We're going to do a fullsweep gc: we garb everything to the */
    /* old heap, and continue with a fresh (empty) new heap. */
    /* First we need to figure out how big the (new) old_heap is */
    /* going to be. */

    old_heap_need = p->heap_sz + p->mbuf_sz  + (p->old_htop - p->old_heap);
    old_heap_sz = next_heap_size(old_heap_need, 0);
    fullsweep_old_heap(p, old_heap_sz, objv, nobj);

    /* We have an empty new heap and all live data now resides on old */
    /* We first possibly adjust the size of old heap */

    size_after = (p->old_htop - p->old_heap);
    reclaimed += (size_before - size_after);

    /* Shrink old if we're using less than 25 % of old heap */

    if ((4 * (p->old_htop - p->old_heap)) < old_heap_sz && 
	old_heap_sz != H_MIN_SIZE) {
	int nsz = next_heap_size(2 * (p->old_htop - p->old_heap), 0);  /* shrink */
	shrink_old_heap(p, nsz, objv, nobj);
    }


    /* That was old heap, now check that the new heap has a reasonable size */
    /* The new heap is empty, since we have just done a full sweep */
    /* We want the new heap to be appr. the size of live data on old heap */

    need += stack_size;  /* zero for JAM */
    wanted = next_heap_size((p->old_hend - p->old_heap) + need + H_MARGIN, 0);
    if ((need  > p->heap_sz) ||        /* grow */
	((2 * wanted) > p->heap_sz) || /* grow */
	((4 * wanted) < p->heap_sz))   /* shrink */

	heap_change(p, wanted, objv, nobj);
    CHECK(p);
    p->flags &= ~F_NEED_GC;  /* Reset GC flags. */
    ASSERT(p->heap_sz == next_heap_size(p->heap_sz, 0));
    if (IS_TRACED_FL(p, F_TRACE_GC)) {
	trace_gc(p, am_gc_end);
    }
    OverRunCheck();
    return ((int) (p->htop - p->heap) / 10);
}
    

/*
** This function is used to sweep both the remainer of the new heap
** as well as the remainer of the old heap after the first pass
** of the generational collector gen_gc().
** Parameters:
** p: Pointer to the process structure
** from: Pointer into heap, sweep from here.
** to: [out] Pointer to pointer that gets set to the new heap top. 
** tenure: Flag that indicates if objects should be tenured, that 
** is saved on the old heap.
** low_water: Pointer into new heap, start of preserved data.
** high_water: Pointer into new heap, end of preserved data.
** old_htop_ptr: [out] Pointer to pointer that gets set to the
** top of the old heap.
** sweeping_old_heap: True if we're sweeping the new heap.
*/
static void gen_cheney(Process *p, uint32 *from, uint32 **to, 
		       int tenure, 
		       uint32 *low_water, uint32 *high_water, 
		       uint32 **old_htop_ptr, 
		       int sweeping_new_heap) 
{

    /* Generation Heap scan jump table */
#if defined(JAM)
    DECL_JVALUE(ghs, FRAME_DEF)
#endif /* JAM */
#if defined(BEAM)
    DECL_JVALUE(ghs, CP0_DEF)
    DECL_JVALUE(ghs, CP4_DEF)
    DECL_JVALUE(ghs, CP8_DEF)
	/* DECL_JVALUE(ghs, CP12_DEF) save as MOVED_DEF */
#endif /* BEAM */
    DECL_JVALUE(ghs, SMALL_DEF)
    DECL_JVALUE(ghs, BIG_DEF)
    DECL_JVALUE(ghs, FLOAT_DEF)
    DECL_JVALUE(ghs, ATOM_DEF)
    DECL_JVALUE(ghs, REFER_DEF)
    DECL_JVALUE(ghs, PORT_DEF)
    DECL_JVALUE(ghs, PID_DEF)
    DECL_JVALUE(ghs, TUPLE_DEF)
#if defined(JAM)
    DECL_JVALUE(ghs, NIL_DEF)
    DECL_JVALUE(ghs, CATCH_DEF)
#endif /* JAM */
    DECL_JVALUE(ghs, LIST_DEF)
    DECL_JVALUE(ghs, ARITYVAL_DEF)
    DECL_JVALUE(ghs, MOVED_DEF)
    DECL_JVALUE(ghs, THING_DEF)
    DECL_JVALUE(ghs, BINARY_DEF)
    
    DECL_JTABLE(ghs, 16)

    uint32 *old_htop, *n_hp, *n_htop;
    uint32 *oh_start, *oh_end;

    uint32 tmp;
    uint32 *ptr;
    uint32 val;
    uint32 gval;
    uint32 do_tenure = tenure && sweeping_new_heap;

    n_hp = from; 
    n_htop = *to; 
    old_htop = *old_htop_ptr;
    oh_start = p->old_heap;
    oh_end = p->old_hend;
    old_htop = *old_htop_ptr;

    if (JTABLE_NEED_INIT(ghs)) {
	
#if defined(JAM)
	DEFINE_LOCATION(ghs, FRAME_DEF);
#endif /* JAM */
#if defined(BEAM)
	DEFINE_LOCATION(ghs, CP0_DEF);
	DEFINE_LOCATION(ghs, CP4_DEF);
	DEFINE_LOCATION(ghs, CP8_DEF);
	/* DEFINE_LOCATION(ghs, CP12_DEF); Same as MOVED_DEF */
#endif /* BEAM */
	DEFINE_LOCATION(ghs, SMALL_DEF);
	DEFINE_LOCATION(ghs, BIG_DEF);
	DEFINE_LOCATION(ghs, FLOAT_DEF);
	DEFINE_LOCATION(ghs, ATOM_DEF);
	DEFINE_LOCATION(ghs, REFER_DEF);
	DEFINE_LOCATION(ghs, PORT_DEF);
	DEFINE_LOCATION(ghs, PID_DEF);
	DEFINE_LOCATION(ghs, TUPLE_DEF);
#if defined(JAM)
	DEFINE_LOCATION(ghs, NIL_DEF);
	DEFINE_LOCATION(ghs, CATCH_DEF);
#endif /* JAM */
	DEFINE_LOCATION(ghs, LIST_DEF);
	DEFINE_LOCATION(ghs, ARITYVAL_DEF);
	DEFINE_LOCATION(ghs, MOVED_DEF);
	DEFINE_LOCATION(ghs, THING_DEF);
	DEFINE_LOCATION(ghs, BINARY_DEF);
	DEFINE_JTABLE(ghs);
    }

    while (n_hp != n_htop) {
	
	gval = *n_hp;

	JUMP(ghs, tag_val_def(gval));
	LOCATION(ghs, FLOAT_DEF) {
	    ptr = ptr_val(gval);
	    val = *ptr;
	    if (ptr_within(ptr, oh_start, oh_end))
		n_hp++;
	    else if (is_moved(val))
		*n_hp++ = make_float(ptr_val(val));
	    else if (do_tenure && ptr_within(ptr, low_water, high_water)) {
		/* Make object old */
		*n_hp++ = gval = make_float(old_htop);
		*ptr++ = make_moved(gval);   /* Store move address */
		*old_htop++ = val;             /* Store thing */
		*old_htop++ = *ptr++;          /* Copy float part 1 */
		*old_htop++ = *ptr;            /* Copy float part 2 */
	    }
	    /* else our pointers are already on old_heap, we may continue */
	    else {
		ASSERT(within(ptr, p));
		ASSERT(is_thing(val));
		*n_hp++ = gval = make_float(n_htop);
		*ptr++ = make_moved(gval);   /* Store move address */
		*n_htop++ = val;             /* Store thing */
		*n_htop++ = *ptr++;          /* Copy float part 1 */
		*n_htop++ = *ptr;            /* Copy float part 2 */
	    }
	    continue;
	}

	LOCATION(ghs, BIG_DEF) {
#if defined(BEAM)
	    if (is_nil(gval)) {
	      n_hp++;
	      continue;
	    }
#endif /* BEAM */
	    ptr = ptr_val(gval);
	    val = *ptr;
	    if (ptr_within(ptr, oh_start, oh_end))
		n_hp++;
	    else if (is_moved(val))
		*n_hp++ = make_big(ptr_val(val));
	    else if (do_tenure && ptr_within(ptr, low_water, high_water)) {
		*n_hp++ = gval = make_big(old_htop);
		*ptr++ = make_moved(gval);   /* Store move address */
		*old_htop++ = val;             /* Store thing */
		tmp = thing_arityval(val);    /* Get arity value */
		while(tmp--) 
		    *old_htop++ = *ptr++;
	    }
	    else {
		ASSERT(within(ptr, p));
		ASSERT(is_thing(val));
		*n_hp++ = gval = make_big(n_htop);
		*ptr++ = make_moved(gval);   /* Store move address */
		*n_htop++ = val;             /* Store thing */
		tmp = thing_arityval(val);    /* Get arity value */
		while(tmp--) 
		    *n_htop++ = *ptr++;
	    }
	    continue;
	}

	LOCATION(ghs, TUPLE_DEF) {
	    ptr = ptr_val(gval);
	    val = *ptr;
	    if (ptr_within(ptr, oh_start, oh_end))
		n_hp++;
	    else if (is_moved(val))
		*n_hp++ = make_tuple(ptr_val(val));
	    else if (do_tenure && ptr_within(ptr, low_water, high_water)) {
		/* Make object old */
		*n_hp++ = gval = make_tuple(old_htop);
		*ptr++ = make_moved(gval);   /* Store move address */
		*old_htop++ = val;             /* Store arity val */
		tmp = arityval(val);          /* Get arity value */
		while(tmp--)
		    *old_htop++ = *ptr++;
	    }
	    else {
		ASSERT(within(ptr, p));
		ASSERT(is_arity_value(val));
		*n_hp++ = gval = make_tuple(n_htop);
		*ptr++ = make_moved(gval);   /* Store move address */
		*n_htop++ = val;             /* Store arity val */
		tmp = arityval(val);          /* Get arity value */
		while(tmp--)
		    *n_htop++ = *ptr++;
	    }
	    continue;
	}

	LOCATION(ghs, LIST_DEF) {
	    ptr = ptr_val(gval);
	    val = *ptr;
	    if (ptr_within(ptr, oh_start, oh_end))
		n_hp++;
	    else if (is_moved(val))
		*n_hp++ = make_list(ptr_val(val));
	    else if (do_tenure && ptr_within(ptr, low_water, high_water)) {
		/* Make object old */
		*n_hp++ = gval = make_list(old_htop);
		*ptr++ = make_moved(gval);   /* Store move address */
		*old_htop++ = val;             /* Store head */
		*old_htop++ = *ptr;            /* Store tail */
	    }
	    else {
		ASSERT(within(ptr, p));
		*n_hp++ = gval = make_list(n_htop);
		*ptr++ = make_moved(gval);   /* Store move address */
		*n_htop++ = val;             /* Store head */
		*n_htop++ = *ptr;            /* Store tail */
	    }
	    continue;
	}

	LOCATION(ghs, THING_DEF) {
	    n_hp += (thing_arityval(gval)+1);
	    continue;
	}

	LOCATION(ghs, SMALL_DEF);
#if defined(JAM)
	LOCATION(ghs, NIL_DEF);
#endif /* JAM */
	LOCATION(ghs, ATOM_DEF);
	LOCATION(ghs, PID_DEF);
	LOCATION(ghs, REFER_DEF);
	LOCATION(ghs, PORT_DEF);
	LOCATION(ghs, ARITYVAL_DEF) {
	    n_hp++;
	    continue;
	}
	LOCATION(ghs, BINARY_DEF) {
	    ProcBin *bp = (ProcBin*) ptr_val(gval);
	    if (tenure)
		bp->mark = BIN_OLD;
	    else if (bp->mark != BIN_OLD) /* Don't mark OLD bins */
		bp->mark = BIN_MARKED;
	    n_hp++;
	    continue;
	}

#if defined(BEAM)
	LOCATION(ghs, CP0_DEF);
	LOCATION(ghs, CP4_DEF);
	LOCATION(ghs, CP8_DEF);
	/* LOCATION(ghs, CP12_DEF);  Same as MOVED_DEF */
#endif /* BEAM */
#if defined(JAM)
	LOCATION(ghs, FRAME_DEF);
	LOCATION(ghs, CATCH_DEF);
#endif /* JAM */
	LOCATION(ghs, MOVED_DEF) {
	    erl_exit(1, "%s: GC: bad data on heap - pass 2 0x%08x at 0x%08x\n",
		     print_pid(p), gval, n_hp);
	}
	JUMP_END;
    }
    /* Now set the parameter pointers for the caller */
    *to = n_htop;
    if (sweeping_new_heap)
	*old_htop_ptr = old_htop;
}

static int
setup_rootset(Process *p, uint32 *objv, 
	      int nobj, Rootset *rootset) 
{
    int n;
    ErlMessage* mp;
    uint32* v_ptr;
    int v_msg_len = p->msg.len;

#if defined(SEQ_TRACE)
    v_msg_len *= 2;
#endif

    /*
     * Move pointers for all messages into an array pointed to by p->v_msg.
     */
    if (v_msg_len > ALENGTH(rootset->def_msg)) {
	rootset->v_msg = (uint32 *) safe_alloc(sizeof(uint32) * v_msg_len);
    } else {
	rootset->v_msg = rootset->def_msg;
    }
    mp = p->msg.first;
    v_ptr = rootset->v_msg;
    while (mp != NULL) {
        *v_ptr++ = mp->mesg;
#if defined(SEQ_TRACE)
	ASSERT((is_nil(mp->seq_trace_token) || is_tuple(mp->seq_trace_token)));
	*v_ptr++ = mp->seq_trace_token;
#endif
        mp = mp->next;
    }

#if defined(JAM)
    rootset->v[0]  = p->stack;
    rootset->sz[0] = p->stop - p->stack;
#elif defined(BEAM)
    rootset->v[0]  = p->stop;
    rootset->sz[0] = p->stack - p->stop;
#endif
    rootset->v[1]  = &p->dictionary;
    rootset->sz[1] = 1;
    rootset->v[2]  = rootset->v_msg;
    rootset->sz[2] = v_msg_len;
    n = 3;
    if (nobj > 0) {
	rootset->v[n]  = objv;
	rootset->sz[n] = nobj;
	n++;
    }
#if defined(JAM)
    /*
     * XXX Beam never does a GC when p->fvalue contains something useful.
     * Jam, however, still does in its exit processing; until that is fixed
     * this code must remain.
     */
    rootset->v[n]  = &p->fvalue;
    rootset->sz[n] = 1;
    n++;
#endif

#if defined(SEQ_TRACE)
    ASSERT((is_nil(p->seq_trace_token) || is_tuple(p->seq_trace_token)));
    rootset->v[n] = &p->seq_trace_token;
    rootset->sz[n] = 1;
    n++;
#endif

    ASSERT(n <= ALENGTH(rootset->v));
    return n;
}

/* 
** Garbage collect the heap, however all objects pointing 
** to the old generation heap may be left as is 
** Every second turn, we remember the position on the heap
** that turned out to be the heap top, every other second turn
** we tenure all live objects that reside below that water mark
** This means that we only tenure objects that have survived at 
** least one collection 
*/
static int gen_gc(Process *p, int new_sz, uint32 *objv, int nobj)
{
    /* Stack scan jump table */
#if defined(JAM)
    DECL_JVALUE(gvs, FRAME_DEF)
#endif /* JAM */
#if defined(BEAM)
    DECL_JVALUE(gvs, CP0_DEF)
    DECL_JVALUE(gvs, CP4_DEF)
    DECL_JVALUE(gvs, CP8_DEF)
    DECL_JVALUE(gvs, CP12_DEF)
    DECL_JVALUE(gvs, BLANK_DEF)
#endif /* BEAM */
    DECL_JVALUE(gvs, SMALL_DEF)
    DECL_JVALUE(gvs, BIG_DEF)
    DECL_JVALUE(gvs, FLOAT_DEF)
    DECL_JVALUE(gvs, ATOM_DEF)
    DECL_JVALUE(gvs, REFER_DEF)
    DECL_JVALUE(gvs, PORT_DEF)
    DECL_JVALUE(gvs, PID_DEF)
    DECL_JVALUE(gvs, TUPLE_DEF)
#if defined(JAM)
    DECL_JVALUE(gvs, NIL_DEF)
#endif /* JAM */
    DECL_JVALUE(gvs, LIST_DEF)
#if defined(JAM)
    DECL_JVALUE(gvs, ARITYVAL_DEF)
    DECL_JVALUE(gvs, MOVED_DEF)
    DECL_JVALUE(gvs, THING_DEF)
#endif /* JAM */
    DECL_JVALUE(gvs, CATCH_DEF)
    DECL_JVALUE(gvs, BINARY_DEF)

    DECL_JTABLE(gvs, 16)

    uint32 *n_hstart;
    uint32 *n_htop;
    uint32 saved_status;
    int tmp;
    Rootset rootset;		/* Rootset for GC (stack, dictionary, etc). */
    int tenure, n;
    uint32 *ptr;
    uint32 val;
    uint32 gval;
    uint32* low_water;
    uint32* high_water;
    uint32* oh_start;
    uint32* old_htop;
    uint32* oh_end;
    uint32* prev_old_htop;

    oh_start = p->old_heap;
    oh_end = p->old_hend;
    low_water = p->low_water;
    high_water = p->high_water;
    tenure = (p->flags & F_GCFLIP);
    prev_old_htop = p->old_htop;

#ifdef GC_HEAP_TRACE
    fprintf(stderr, "Generational GC (proc = %d) ", (int)get_number(p->id));
#endif

    /* If flip is true, we need to tenure all (live) objects */
    /* within the watermarks, if flip is 0, we need to alloc a */
    /* new new_heap and copy all live objects to the new new_heap */
    /* that is to not tenure any objects at all */

    if (JTABLE_NEED_INIT(gvs)) {
#if defined(JAM)
	DEFINE_LOCATION(gvs, FRAME_DEF);
#endif /* JAM */
#if defined(BEAM)
	DEFINE_LOCATION(gvs, CP0_DEF);
	DEFINE_LOCATION(gvs, CP4_DEF);
	DEFINE_LOCATION(gvs, CP8_DEF);
	DEFINE_LOCATION(gvs, CP12_DEF);
	DEFINE_LOCATION(gvs, BLANK_DEF);
#endif /* BEAM */
	DEFINE_LOCATION(gvs, SMALL_DEF);
	DEFINE_LOCATION(gvs, BIG_DEF);
	DEFINE_LOCATION(gvs, FLOAT_DEF);
	DEFINE_LOCATION(gvs, ATOM_DEF);
	DEFINE_LOCATION(gvs, REFER_DEF);
	DEFINE_LOCATION(gvs, PORT_DEF);
	DEFINE_LOCATION(gvs, PID_DEF);
	DEFINE_LOCATION(gvs, TUPLE_DEF);
#if defined(JAM)
	DEFINE_LOCATION(gvs, NIL_DEF);
#endif /* JAM */
	DEFINE_LOCATION(gvs, LIST_DEF);
#if defined(JAM)
	DEFINE_LOCATION(gvs, ARITYVAL_DEF);
	DEFINE_LOCATION(gvs, MOVED_DEF);
	DEFINE_LOCATION(gvs, THING_DEF);
#endif /* JAM */
	DEFINE_LOCATION(gvs, CATCH_DEF);
	DEFINE_LOCATION(gvs, BINARY_DEF);
	DEFINE_JTABLE(gvs);
    }

    n_hstart = (uint32*) safe_alloc(sizeof(uint32)*new_sz);
    n_htop = n_hstart;
    old_htop = p->old_htop;
    saved_status = p->status;
    p->status = P_GARBING;
    n = setup_rootset(p, objv, nobj, &rootset);

    while (n--) {
	uint32* g_ptr = rootset.v[n];
	uint32 g_sz = rootset.sz[n];
	
	while (g_sz--) {
	    gval = *g_ptr;

	    JUMP(gvs, tag_val_def(gval));

	    LOCATION(gvs, FLOAT_DEF) {
		ptr = ptr_val(gval);
		val = *ptr;
		if (ptr_within(ptr, oh_start, oh_end))
		    g_ptr++;
		else if (is_moved(val))
		    *g_ptr++ = make_float(ptr_val(val));
		else if (tenure && ptr_within(ptr, low_water, high_water)) { /* copy to old*/
		    *g_ptr++ = gval = make_float(old_htop);
		    *ptr++ = make_moved(gval);
		    *old_htop++ = val;             /* Store thing */
		    *old_htop++ = *ptr++;          /* Copy float part 1 */
		    *old_htop++ = *ptr;            /* Copy float part 2 */
		}
		else {
		    ASSERT(within(ptr, p));
		    ASSERT(is_thing(val));
		    *g_ptr++ = gval = make_float(n_htop);
		    *ptr++ = make_moved(gval);   /* Store move address */
		    *n_htop++ = val;             /* Store thing */
		    *n_htop++ = *ptr++;          /* Copy float part 1 */
		    *n_htop++ = *ptr;            /* Copy float part 2 */
		}
		continue;
	    }

	    LOCATION(gvs, BIG_DEF) {
#if defined(BEAM)
		if (is_nil(gval)) {
		  g_ptr++;
		continue;
	    }
#endif /* BEAM */
		ptr = ptr_val(gval);
		val = *ptr;
		if (ptr_within(ptr, oh_start, oh_end))
		    g_ptr++;
		else if (is_moved(val))
		    *g_ptr++ = make_big(ptr_val(val));
		else if (tenure && ptr_within(ptr, low_water, high_water)) {
		    *g_ptr++ = gval = make_big(old_htop);
		    *ptr++ = make_moved(gval);   /* Store move address */
		    *old_htop++ = val;
		    tmp = thing_arityval(val);    /* Get arity value */
		    while(tmp--)
			*old_htop++ = *ptr++;
		}
		else {
		    ASSERT(within(ptr, p));
		    ASSERT(is_thing(val));
		    *g_ptr++ = gval = make_big(n_htop);
		    *ptr++ = make_moved(gval);   /* Store move address */
		    *n_htop++ = val;             /* Store thing */
		    tmp = thing_arityval(val);    /* Get arity value */
		    while(tmp--)
			*n_htop++ = *ptr++;
		}
		continue;
	    }

	    LOCATION(gvs, TUPLE_DEF) {
		ptr = ptr_val(gval);
		val = *ptr;
		if (ptr_within(ptr, oh_start, oh_end))
		    g_ptr++;
		else if (is_moved(val))
		    *g_ptr++ = make_tuple(ptr_val(val));
		else if (tenure && ptr_within(ptr, low_water, high_water)) {
		    *g_ptr++ = gval = make_tuple(old_htop);
		    *ptr++ = make_moved(gval);
		    *old_htop++ = val;             /* Store arity val */
		    tmp = arityval(val);          /* Get arity value */
		    while(tmp--)
			*old_htop++ = *ptr++;
		}
		else {
		    ASSERT(within(ptr, p));
		    ASSERT(is_arity_value(val));
		    *g_ptr++ = gval = make_tuple(n_htop);
		    *ptr++ = make_moved(gval);   /* Store move address */
		    *n_htop++ = val;             /* Store arity val */
		    tmp = arityval(val);          /* Get arity value */
		    while(tmp--)
			*n_htop++ = *ptr++;
		}
		continue;
	    }

	    LOCATION(gvs, LIST_DEF) {
		ptr = ptr_val(gval);
		val = *ptr;
		if (ptr_within(ptr, oh_start, oh_end))
		    g_ptr++;
		else if (is_moved(val))
		    *g_ptr++ = make_list(ptr_val(val));
		else if (tenure && ptr_within(ptr, low_water, high_water)) {
		    *g_ptr++ = gval = make_list(old_htop);
		    *ptr++ = make_moved(gval);     /* Store move address */
		    *old_htop++ = val;             /* Store head */
		    *old_htop++ = *ptr;            /* Store tail */
		}
		else {
		    ASSERT(within(ptr, p));
		    *g_ptr++ = gval = make_list(n_htop);
		    *ptr++ = make_moved(gval);   /* Store move address */
		    *n_htop++ = val;             /* Store head */
		    *n_htop++ = *ptr;            /* Store tail */
		}
		continue;
	    }

#if defined(BEAM)
	    LOCATION(gvs, CP0_DEF);
	    LOCATION(gvs, CP4_DEF);
	    LOCATION(gvs, CP8_DEF);
	    LOCATION(gvs, CP12_DEF);
	    LOCATION(gvs, BLANK_DEF);
	    LOCATION(gvs, CATCH_DEF);
#endif /* BEAM */
	    LOCATION(gvs, SMALL_DEF);
#if defined(JAM)
	    LOCATION(gvs, NIL_DEF);
#endif /* JAM */
	    LOCATION(gvs, ATOM_DEF);
	    LOCATION(gvs, PID_DEF);
	    LOCATION(gvs, REFER_DEF);
	    LOCATION(gvs, PORT_DEF) {
		g_ptr++;
		continue;
	    }

#if defined(JAM)
	    LOCATION(gvs, CATCH_DEF)  {
		g_ptr += 2;
		g_sz--;
		continue;
	    }

	    LOCATION(gvs, FRAME_DEF)  {
		g_ptr += 4;
		g_sz -= 3;
		continue;
	    }
#endif /* JAM */

	    LOCATION(gvs, BINARY_DEF) {
		ProcBin *bp = (ProcBin*) ptr_val(gval);
		if (tenure)
		    bp->mark = BIN_OLD;
		else if (bp->mark != BIN_OLD) /* Don't mark OLD bins */
		    bp->mark = BIN_MARKED;
		g_ptr++;
		continue;
	    }

#if defined(JAM)
	    LOCATION(gvs, MOVED_DEF);
	    LOCATION(gvs, ARITYVAL_DEF);
	    LOCATION(gvs, THING_DEF) {
		erl_exit(1, "%s: bad data on stack GC pass 1 0x%08x at 0x%08x\n",
			 print_pid(p), gval, g_ptr);
	    }
#endif /* JAM */
	    JUMP_END;
	}
    }

#if defined(BEAM)
    /* 
     * Now we got to move the stack to the top of the new heap...
     */
    n = p->stack - p->stop;

    sys_memcpy(n_hstart + new_sz - n, p->stop, n * sizeof(uint32));

    p->stack = n_hstart + new_sz;
    p->stop = p->stack - n;

#endif /* BEAM */

    /* now all references on the stack point to the new heap. However 
       most references on the new heap point to the old heap so the next stage
       is to scan through the new heap evacuating data from the old heap
       until all is changed */
    
    gen_cheney(p, n_hstart, &n_htop, tenure, low_water, 
	       high_water, &old_htop, 1);
    
    /* And also if we have been tenuring,references on the second generation */
    /* may point to the old (soon to be deleted) heap */

    if (tenure)
	gen_cheney(p, prev_old_htop, &old_htop, tenure, low_water, 
		   high_water,&old_htop, 0);

#ifdef GC_HEAP_TRACE
    if (tenure) 
	fprintf(stderr,"Tenured %d words Kept %d words \n", 
		old_htop - prev_old_htop,
		n_htop - n_hstart);
    else
	fprintf(stderr, "No tenuring Kept %d words \n",
		n_htop - n_hstart);
#endif

    /* Clobber the old heap */
#ifdef DEBUG
    sys_memset(p->heap, 0xff, p->heap_sz*sizeof(uint32));
#endif
    sys_free((void*)p->heap);

    restore_rootset(p, &rootset);
    
    p->heap = n_hstart;
    p->hend = n_hstart + new_sz;
    p->htop = n_htop;
    p->heap_sz = new_sz;
    SET_HEAP_MARGIN(p, p->hend - H_MARGIN);

    p->old_htop = old_htop;
    p->status = saved_status;
    
    if (tenure) {
	p->low_water = p->high_water = n_hstart;
	p->flags &= ~F_GCFLIP;
    }
    else {
	p->low_water = n_hstart;
	p->high_water = n_htop;
	p->flags |= F_GCFLIP;
    }
    if (p->mso)
	generation_bin_gc(p);
    
    return(0);
}


/************************************************************** */
/*  DEBUG routines                                              */
/****************************************************************/


#ifdef HARDDEBUG

static int within(uint32 *ptr, Process *p)
{
    ErlMessageBuffer* bp = p->mbuf;

    if (IS_GEN_GC(p) && (ptr >= p->old_heap) && (ptr < p->old_hend))
	return 1;
    if ((ptr >= p->heap) && (ptr < p->hend))
	return 1;
    while (bp != NULL) {
	if ((ptr >= bp->mem) && (ptr < (bp->mem + bp->size)))
	    return 1;
	bp = bp->next;
    }
    return 0;
}

/* Different check whether we check old_heap or not */
/* since we want to ensure that old heap contains no pointers */
/* to the new heaps */

static void check_binary(Process *p, uint32 obj, char *msg)
{
    ProcBin *bp = (ProcBin*)ptr_val(obj);

    if (  * ((uint32*)bp) == 0xffffffff) {
	erl_exit(1, "%s: Clobbered binary left\n", print_pid(p));
    }
    if (bp->size > 0xfffffff) {
	erl_exit(1, "%s: check_heap: %s: LARGE binary found %x\n",
		 print_pid(p), msg, obj);
    }
    if (bp->bytes == 0) {
	erl_exit(1, "%s: check_heap: %s: NULL binary found %x\n",
		 print_pid(p), msg, obj);
    }
}

static int check_heap_obj(Process *p, char *msg, uint32 obj, 
			  int back_pointers_allowed)
{
    int tmp;

    switch (tag_val_def(obj)) {
    case ATOM_DEF:
	if (unsigned_val(obj) >= atom_table_size) {
	    erl_exit(1, "%s: check_heap: bad atom on heap %d\n",
		     print_pid(p), unsigned_val(obj));
	}
	return 0;
    case SMALL_DEF:
    case ARITYVAL_DEF:
    case PID_DEF:
    case REFER_DEF:
#if defined(JAM)
    case NIL_DEF:
#endif /* JAM */
    case PORT_DEF:
	return 0;
    case BINARY_DEF:
	check_binary(p, obj, msg);
	return 0;
    case THING_DEF:
	tmp = thing_arityval(obj); /* get thing value */
	return tmp;
    case BIG_DEF:
#if defined(BEAM)
	if(is_nil(obj)) {
	    return 0;
	}
#endif /* BEAM */
	/* fall through */
    case FLOAT_DEF:
    case LIST_DEF:
    case TUPLE_DEF:
	if (back_pointers_allowed && !within(ptr_val(obj), p)) {
	    erl_exit(1, "%s: check_heap: %s: bad address %x\n",
		     print_pid(p), msg, obj);
	}
	else if (!back_pointers_allowed && 
		 IS_GEN_GC(p) && !ptr_within(ptr_val(obj), p->old_heap, p->old_htop)) {
	    if (within(ptr_val(obj), p)) 
		erl_exit(1, "%s: check_heap: %s: back pointer %x\n",
			 print_pid(p), msg, obj);
	    else
		erl_exit(1, "%s: check_heap: %s: bad address %x\n", 
			 print_pid(p), msg, obj);
	}
	return 0;
    default:
	erl_exit(1, "%s: check_heap: %s: bad tag %x\n",
		 print_pid(p), msg, obj);
    }
    return 0;
}

static void check_heap(Process *p, char *msg)
{
    uint32 *hp;
    uint32 obj;
    int heap_size = p->htop - p->heap;

    if (heap_size > p->heap_sz)
	erl_exit(1, "%s: check_heap: %s: bad heap size\n",
		 print_pid(p), msg);
#if defined(BEAM)
    if (p->stop < p->htop)
	erl_exit(1, "%s: check_heap: %s: overflow\n",
		 print_pid(p), msg);
#endif
    for (hp = p->heap; hp < p->htop; hp++) {
	obj = *hp;
	hp += check_heap_obj(p, msg, obj, 1);
    }
    if (IS_GEN_GC(p)) {
	for (hp=p->old_heap; hp < p->old_htop; hp++) {
	    uint32 obj = *hp;
	    hp += check_heap_obj(p, msg, obj, 0);
	}
    }
}



static void stack_element_check(Process *p, char *msg, uint32 x)
{
    switch (tag_val_def(x)) {
    case ATOM_DEF:
	if (unsigned_val(x) >= atom_table_size) {
	    erl_exit(1, "%s: check_stack: bad atom on stack %d\n",
		     print_pid(p), unsigned_val(x));
	}
	return;
    case SMALL_DEF:
    case PID_DEF:
    case REFER_DEF:
#if defined(JAM)
    case NIL_DEF:
#endif /* JAM */
#if defined(BEAM)
    case CP0_DEF:
    case CP4_DEF:
    case CP8_DEF:
    case CP12_DEF:
    case BLANK_DEF:
#endif /* BEAM */
    case PORT_DEF:
	return;
    case BINARY_DEF:
	check_binary(p, x, msg);
	return;
    case BIG_DEF:
#if defined(BEAM)
      if(is_nil(x)) {
	return;
      }
      /* fall through */
#endif /* BEAM */
    case FLOAT_DEF:
    case LIST_DEF:
    case TUPLE_DEF:
	if (!within(ptr_val(x),p)) {
	    erl_exit(1, "%s: check_stack: %s: bad address %x\n",
		     print_pid(p), msg, x);
	}
	return;
    default:
	erl_printf(1, "%s: stack_check: %s: bad tag %x\n",
		   print_pid(p), msg, x);
    }
}

static void check_stack(Process *p, char *msg)
{
    uint32 *sp;
    uint32 *sp_end;
    int stack_size, sz;

#if defined(JAM)
    sp = p->stack;
    sp_end = p->stop;
    sz = p->stack_sz;
#elif defined(BEAM)
    sp = p->stop;
    sp_end = p->stack;
    sz = p->stack - p->htop;
#endif /* JAM */

    stack_size = sp_end - sp;
    if (stack_size > sz)
	erl_exit(1, "%s: Overrun stack, panic\n", print_pid(p));

#if defined(JAM)
    if (p->stack[sz] != S_PATTERN || p->stack[sz+1] != S_PATTERN)
	erl_exit(1, "%s: check_stack: %s: overrun stack (bad pattern), panic\n",
		 print_pid(p), msg);
    while (sp < sp_end) {
	if (is_frame(*sp))
	    sp += FRAME_SIZE;   /* Check frame ? */
	else if (is_catch(*sp))
	    sp += CATCH_SIZE;   /* Check catch ? */
	else {
	    stack_element_check(p, msg, *sp);
	    sp++;
	}
    }
#elif defined(BEAM)
    for(sp = p->stack-1; sp >= p->stop; sp--) {
	if (!is_catch(*sp))
	    stack_element_check(p, msg, *sp);
    }
#endif /* JAM */
}


int where_pointer(Process *p, uint32 *from, uint32 *to, uint32 obj)
{
    int count = 0;
    uint32 *ptr = ptr_val(obj);

    fprintf(stderr, " at ");
    if (ptr_within(ptr, from, to)) 
	fprintf(stderr,"This interval   ");
    if (IS_GEN_GC(p) && ptr_within(ptr, p->old_heap, p->old_htop)) {
	fprintf(stderr, "Old heap \n");
	count++;
    }
    if (ptr_within(ptr, p->heap, p->htop)) {
	count++;
	fprintf(stderr, "New heap \n");
    }
    if (count == 0) 
	fprintf(stderr, "Unknown pointer %x !!!!\n", (unsigned int) ptr);
    return 1;
}



void heap_dump(Process *p, uint32 *from, uint32 *to)
{
    uint32 *hp;
    uint32 obj;
    int tmp;

    for (hp = from; hp < to; hp++) {
	obj = *hp;
	fprintf(stderr,"%x :", (unsigned int) hp);
	switch (tag_val_def(obj)) {
	case ATOM_DEF:
	    if (unsigned_val(obj) >= atom_table_size) {
		erl_printf(CERR, "heap_dump: bad atom on heap %d\n",
			   unsigned_val(obj));
	    }
	case SMALL_DEF:
	case PID_DEF:
	case REFER_DEF:
#if defined(JAM)
	case NIL_DEF:
#endif /* JAM */
	case PORT_DEF:
	case BINARY_DEF:
	    td(obj);
	    break;

	case THING_DEF:
	    tmp = thing_arityval(obj); /* get thing value */
	    fprintf(stderr, "THING size = %d\n", tmp);
	    hp += tmp;	/* skip the binary data */
	    /* note the hp++ in the "for" will also add 1 */
	    break;
	
	case BIG_DEF:
#if defined(BEAM)
	    if(is_nil(obj)) {
		td(obj);
		break;
	    }
#endif /* BEAM */
	    /* fall through */
	case FLOAT_DEF:
	    fprintf(stderr,"Float pointer %x ", 
		    (unsigned int) ptr_val(obj));
	    where_pointer(p, from, to, obj);
	    break;
	case LIST_DEF:
	    fprintf(stderr,"List pointer %x", 
		    (unsigned int)ptr_val(obj)); 
	    where_pointer(p, from, to, obj);
	    break;
	case TUPLE_DEF:
	    fprintf(stderr,"Tuple pointer %x ", 
		    (unsigned int)ptr_val(obj)); 
	    where_pointer(p, from, to, obj);
	    break;
	case ARITYVAL_DEF:
	    fprintf(stderr,"Arity val = %d \n", (int)arityval(obj));
	    break;
	default:
	    fprintf(stderr, "Unknown object on heap \n");
	}
    }
}


void stack_element_dump(Process *p, uint32 *sp)
{
    uint32 x = *sp;
    fprintf(stderr,"%x :", (unsigned int) sp);
    
    switch (tag_val_def(x)) {
    case ATOM_DEF:
	if (unsigned_val(x) >= atom_table_size) {
	    erl_exit(1, "%s: check_stack: bad atom on stack %d\n",
		     print_pid(p), unsigned_val(x));
	}
	return;
    case SMALL_DEF:
    case PID_DEF:
    case REFER_DEF:
#if defined(JAM)
    case NIL_DEF:
#endif /* JAM */
	td(x);
	break;
#if defined(BEAM)
    case CP0_DEF:
    case CP4_DEF:
    case CP8_DEF:
    case CP12_DEF:
	fprintf(stderr, "Continuation pointer %x \n", (uint32*) x);
	break;
    case BLANK_DEF:
	fprintf(stderr,"Blank  %x \n", (uint32*) x);
	break;
#endif /* BEAM */
    case PORT_DEF:
    case BINARY_DEF:
	td(x);
	break;
    case BIG_DEF:
#if defined(BEAM)
	if(is_nil(x)) {
	    td(x);
	    break;
	}
	/* fall through */
#endif /* BEAM */
    case FLOAT_DEF:
	fprintf(stderr,"Float pointer %x ", 
		(unsigned int) ptr_val(x));
	where_pointer(p, 0, 0, x);
	break;
    case LIST_DEF:
	fprintf(stderr,"List pointer %x", 
		(unsigned int)ptr_val(x)); 
	where_pointer(p, 0, 0, x);
	break;
    case TUPLE_DEF:
	fprintf(stderr,"Tuple pointer %x ", 
		(unsigned int)ptr_val(x)); 
	where_pointer(p, 0, 0 , x);
	break;
    default:
	erl_printf(CERR, "stack_dump: : bad tag %x\n", x);
    }
}


void stack_dump(Process *p)
{
    uint32 *sp;

#if defined(JAM)
    uint32 *sp_end;
    int sz;
#endif

#if defined(JAM)
    sp = p->stack;
    sp_end = p->stop;
    sz = p->stack_sz;

    if (p->stack[sz] != S_PATTERN || p->stack[sz+1] != S_PATTERN) {
	erl_exit(1, "%s: GC: stack_dump: bad_pattern\n", print_pid(p));
    }
    while (sp < sp_end) {
	if (is_frame(*sp)) {
	    fprintf(stderr,"%x : Frame \n", (unsigned int) sp );
	    sp += FRAME_SIZE;   /* Check frame ? */
	}
	else if (is_catch(*sp))
	    sp += CATCH_SIZE;   /* Check catch ? */
	else {
	    stack_element_dump(p, sp);
	    sp++;
	}
    }
#elif defined(BEAM)
    for(sp = p->stack-1; sp >= p->stop; sp--)
        stack_element_dump(p, sp);
#endif /* JAM */
}


int chk_sys(void)
{
    Process *p = *process_tab;
    int i, res = 0;
    for (i = 0; i < max_process; i++) {
	if ((p = process_tab[i]) != NULL) {
	    res++;
	    check_heap(p, "chk");
	    check_stack(p, "chk");
	}
    }
    return res;
}
	    
void check_bins(Process *p)
{

    ProcBin* ms = p->mso;
    while(ms) {
	if (ms->mark == BIN_MARKED) {
	    erl_exit(1, "%s: Found marked object: Size %d\n",
		     print_pid(p), ms->size);
	}
	ms = ms->next;
    }
    if (IS_GEN_GC(p)) {
	ms = p->old_mso;
	while(ms) {
	    if (ms->mark != BIN_OLD) {
		erl_exit(1, "%s: Found non-old object: Size %d\n",
			 print_pid(p), ms->size);
	    }
	    ms = ms->next;
	}
    }
}

    
#endif  /* HARDDEBUG  */

static char*
print_pid(Process *p)
{
    char static buf[64];

    uint32 obj = p->id;
    sprintf(buf, "<%ld.%ld.%ld>", get_node(obj), get_number(obj), get_serial(obj));
    return buf;
}
