/* ``The contents of this file are subject to the Erlang Public License,
 * Version 1.0, (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.erlang.org/EPL1_0.txt
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 * 
 * The Original Code is Erlang-4.7.3, December, 1998.
 * 
 * The Initial Developer of the Original Code is Ericsson Telecom
 * AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
 * Telecom AB. All Rights Reserved.
 * 
 * Contributor(s): ______________________________________.''
 */
/*
** This file is copyright (c) Ellemtel in January 1991
**
** File: process.h
*/
#ifndef __PROCESS_H__
#define __PROCESS_H__

#include "erl_message.h"

/*
** Timer entry:
*/
typedef struct erl_timer {
    struct erl_timer* next;   /* next entry tiw slot or chain */
    uint32 slot;              /* slot in timer wheel */
    uint32 count;             /* number of loops reamining */
    int    active;            /* 1=activated, 0=deactivated */
    /* called when timeout */
    FUNCTION(void, (*timeout), (void*));
    /* called when cancel (may be NULL) */
    FUNCTION(void, (*cancel), (void*));
    void* arg;        /* argument to timeout/cancel procs */
} ErlTimer;

typedef FUNCTION(void, (*ErlTimeoutProc), (void*));
typedef FUNCTION(void, (*ErlCancelProc), (void*));


#define INITIAL_MOD 1
#define INITIAL_FUN 2
#define INITIAL_ARI 3

typedef struct process {
    uint32* htop;           /* Heap top */
    uint32* stop;           /* Stack top */
    uint32* stack;          /* Stack start */
    uint32* heap;	    /* Heap start */
    uint32* hend;           /* Heap end */

    uint32  gc_switch;      /* Switch to generational GC when live
			       data is more than or equal to 
			       this many words */
    uint32 min_heap_size;   /* Minimum size of heap (in words). */

    uint32 status;	    /* process STATE */
    uint32 rstatus;	    /* process resume STATE */
    int rcount;             /* suspend count */

    uint32 id;		    /* The pid of this process */
    int    prio;            /* Priority of process */
    uint32 reds;            /* No of reductions for this process  */
    uint32 error_handler;   /* module atom for the error handler */
    uint32 tracer_proc;     /* If proc is traced, this is the tracer */
    uint32 group_leader;    /* pid in charge */
    uint32 flags;           /* Trap exit, trace  flags etc */
    uint32 fvalue;          /* Exit & Throw value (failure reason) */
    uint32 freason;	    /* Reason for detected failure */
    sint32 fcalls;          /* Jam: Number of reductions executed in this call
			     * call to process_main().
			     * Beam: Number of reductions left to execute.
			     * For both Jam and Beam, only valid for the current
			     * process.
			     */
    int    dslot;           /* Distribution slot to use if F_DISTRIBUTION */

    ErlTimer tm;            /* Timer entry */

    struct process *next;    /* Pointer to next process in list */
    uint32 mso_weight;       /* GC weight of binaries */
    struct proc_bin *mso;    /* List of mark-sweep objects */
    struct proc_bin *old_mso; /* Tenured binaries */
    struct reg_proc* reg;    /* NULL iff not registerd */

    uint32   heap_sz;       /* Size of heap in words */

    uint32 *high_water;
    uint32 *low_water;
    uint32 *old_hend;
    uint32 *old_htop;
    uint32 *old_heap;

    struct erl_link* links;         /* List of links */

    ErlMessageQueue msg;     /* Message queue */
    ErlMessageBuffer* mbuf;  /* Pointer to message buffer list */
    uint32 mbuf_sz;          /* Size of all message buffers */
    uint32 mbuf_struct_sz;   /* Sum of sizeof(ErlangMessageBuffer)/4 - 1 */ 

    uint32 dictionary;       /* Process dictionary list */
    /* beam_apply expect mod,fun,arity to be placed like this */
#ifdef SEQ_TRACE
    uint32 seq_trace_clock;
    uint32 seq_trace_lastcnt;
    uint32 seq_trace_token;  /* sequential trace token (tuple size 5 see below) */
#endif
    uint32 initial[4];        /* dummy(0), module(1), function(2), arity(3) */

#if defined(JAM)
    uint32* fp;			/* Frame pointer */
    uint32* ap;			/* Function argument(s) pointer */
    byte*   pc;			/* Current program counter */
    byte*   cc;			/* Current call */

    uint32* stack_margin;	/* Stack margin */
    uint32  stack_sz;		/* Size of stack in words */

    uint32 *catches;		/* For storing the address of the last catch */
    uint32* heap_margin;	/* Heap margin */
#elif defined(BEAM)
    uint32* call;               /* Current call (points to C code). */
    uint32* cp;                 /* Continuation pointer (for threaded code). */
    uint32* i;                  /* Program counter for threaded code. */
    uint32 beam_apply[5];       /* Beam code for apply/spawn. */
    sint32 catches;             /* Number of catches on stack */
    uint32* current;		/* Current Erlang function: module(0), function(0),
				 * arity(0) (module and functions are tagged atoms;
				 * arity an untagged integer).
				 */
    uint32* action_time_out;    /* Where to go when timed out (copied to p->call
				 * on timeout).
				 */
    uint32* arith_heap;		/* Secondary heap for arithmethic operations. */
    uint32 arith_avail;		/* Available space on arithmetic heap. */
#ifdef DEBUG
    char* arith_file;		/* Filename of last ArithAlloc(). */
    int arith_line;		/* And linenumber. */
    uint32* arith_check_me;	/* Address to check for overwrite. */
#endif
    /*
     * Saved x registers.
     */
    uint32 arity;               /* Number of live argument registers (only valid
				 * when process is *not* running).
				 */
    uint32* arg_reg;		/* Pointer to argument registers. */
    unsigned max_arg_reg;	/* Maximum number of argument registers available. */
    uint32 def_arg_reg[6];	/* Default array for argument registers. */
#endif
} Process;


/*
 * The MBUF_GC_FACTOR descides how easily a process is subject to GC 
 * due to message buffers allocated outside the heap.
 * The larger the factor, the easier the process gets GCed.
 * On a small memory system with lots of processes, this makes a significant 
 * difference, especially since the GCs help fragmentation quite a bit too.
 */
#if defined(SMALL_MEMORY)
#define MBUF_GC_FACTOR 4
#else
#define MBUF_GC_FACTOR 1
#endif


#ifdef SEQ_TRACE
#define SEQ_TRACE_TOKEN(p)  ((p)->seq_trace_token)

/* The sequential tracing token is a tuple of size 5:
 *
 *    {Flags, Label, Serial, Sender}
 */

#define SEQ_TRACE_TOKEN_ARITY(p)    (unsigned_val(*(ptr_val(SEQ_TRACE_TOKEN(p)))))
#define SEQ_TRACE_TOKEN_FLAGS(p)    (*(ptr_val(SEQ_TRACE_TOKEN(p)) + 1))
#define SEQ_TRACE_TOKEN_LABEL(p)    (*(ptr_val(SEQ_TRACE_TOKEN(p)) + 2))
#define SEQ_TRACE_TOKEN_SERIAL(p)   (*(ptr_val(SEQ_TRACE_TOKEN(p)) + 3))
#define SEQ_TRACE_TOKEN_SENDER(p)   (*(ptr_val(SEQ_TRACE_TOKEN(p)) + 4))
#define SEQ_TRACE_TOKEN_LASTCNT(p)  (*(ptr_val(SEQ_TRACE_TOKEN(p)) + 5))

/* used when we have unit32 token */
#define SEQ_TRACE_T_ARITY(token)    (unsigned_val(*(ptr_val(token))))
#define SEQ_TRACE_T_FLAGS(token)    (*(ptr_val(token) + 1))
#define SEQ_TRACE_T_LABEL(token)    (*(ptr_val(token) + 2))
#define SEQ_TRACE_T_SERIAL(token)   (*(ptr_val(token) + 3))
#define SEQ_TRACE_T_SENDER(token)   (*(ptr_val(token) + 4))
#define SEQ_TRACE_T_LASTCNT(token)  (*(ptr_val(token) + 5))


#endif

/*
 * Possible flags for the flags field in ErlSpawnOpts below.
 */

#define SPO_LINK 1
#define SPO_USE_ARGS 2

/*
 * The following struct contains options for a process to be spawned.
 */
typedef struct {
    uint32 flags;
    int error_code;		/* Error code returned from create_process(). */

    /*
     * The following items are only initialized if the SPO_USE_ARGS flag is set.
     */
    uint32 min_heap_size;	/* Minimum heap size (must be a valued returned
				 * from next_heap_size()).
				 */
    uint32  gc_switch;		/* Threshold for GC switch: 0 - always gen_gc,
				 * MAX_SMALL - fullsweep
				 */
    int priority;		/* Priority for process. */
    uint32 process_flags;	/* Initial value for process flags. */
} ErlSpawnOpts;

/*
 * The KILL_CATCHES(p) macro kills pending catches for process p.
 */

#if defined(JAM)
#  define KILL_CATCHES(p) (p)->catches = ENULL
#elif defined(BEAM)
#  define KILL_CATCHES(p) (p)->catches = 0
#endif

/*
 * The SET_HEAP_MARGIN() macro sets heap margin for the process (if heap
 * margins are used).
 */

#if defined(JAM)
#  define SET_HEAP_MARGIN(p, margin) (p)->heap_margin = (margin)
#elif defined(BEAM)
#  define SET_HEAP_MARGIN(p, margin)
#endif

#if defined(BEAM)
extern uint32* arith_alloc(Process* p, uint32 need);
#endif

extern Process** process_tab;
extern uint32 max_process;

#define INVALID_PID(p, pid) (p == NULL || \
			     p->id != pid || \
			     p->status == P_EXITING)

#define IS_TRACED(p)        ( (p)->tracer_proc != NIL)
#define IS_TRACED_FL(p,tf)  ( IS_TRACED(p) && ( ((p)->flags & (tf)) == (tf)) )

/* process priorities */
#define PRIORITY_MAX          0
#define PRIORITY_HIGH         1
#define PRIORITY_NORMAL       2
#define PRIORITY_LOW          3
#define NPRIORITY_LEVELS      4

/* process flags */
#define F_TRAPEXIT           (1 << 0)
#define F_INSLPQUEUE         (1 << 1) /* Set if in timer queue */
#define F_TIMO               (1 << 2) /* Set if timeout */
#define F_USING_DB           (1 << 3) /* If have created tables */
#define F_TRACE_SEND         (1 << 4)   
#define F_TRACE_RECEIVE      (1 << 5)
#define F_TRACE_SOS          (1 << 6) /* Set on spawn       */
#define F_TRACE_SOS1         (1 << 7) /* Set on first spawn */
#define F_TRACE_SOL          (1 << 8) /* Set on link        */
#define F_TRACE_SOL1         (1 << 9) /* Set on first link  */
#define F_TRACE_CALLS        (1 << 10)
#define F_TIMESTAMP          (1 << 11)
#define F_DONT_PRE_EMPT      (1 << 12) /* Don't preempt this process */
#define F_TRACE_BIFS         (1 << 13) 
#define F_TRACE_PROCS        (1 << 14)
#define F_TRACE_FIRST_CHILD  (1 << 15)
#define F_TRACE_SCHED        (1 << 16)
#define F_TRACE_GC           (1 << 17)
#define F_IS_TRACING         (1 << 18)  /* We are tracing another process */
#define F_NEED_GC            (1 << 19)  /* Time to GC */
#define F_DISTRIBUTION       (1 << 20)  /* Process used in distribution */
#define F_GCFLIP             (1 << 21)  /* Flag for the generational gc */
#define F_GEN_GC             (1 << 22)  /* If set, generational GC is used */
#define F_HEAP_GROW          (1 << 23)

#define TRACE_FLAGS (  F_TRACE_PROCS | F_TRACE_BIFS | F_TRACE_CALLS \
		     | F_TRACE_SOS |  F_TRACE_SOS1| F_TRACE_RECEIVE  \
		     | F_TRACE_SOL | F_TRACE_SOL1 | F_TRACE_SEND | \
		     F_TRACE_SCHED | F_TIMESTAMP | F_TRACE_GC )

#ifdef SEQ_TRACE
/* Sequential trace flags */
#define SEQ_TRACE_SEND     (1 << 0)
#define SEQ_TRACE_RECEIVE  (1 << 1)
#define SEQ_TRACE_PRINT    (1 << 2)
#define SEQ_TRACE_TIMESTAMP (1 << 3)
#endif


/* Process status values */
#define P_FREE      0
#define P_RUNABLE   1
#define P_WAITING   2
#define P_RUNNING   3
#define P_EXITING   4
#define P_GARBING   5
#define P_SUSPENDED 6

/*
 * Every process starts out doing fullswep GC, and it might switch to doing
 * generation gc.  IS_GEN_GC(p) tests if generation GC is active for the given
 * process, and SET_GEN_GC(p) turns on generation GC.
 */

#define IS_GEN_GC(p) ((p)->flags & F_GEN_GC)
#define SET_GEN_GC(p) ((p)->flags |= F_GEN_GC)


#define CANCEL_TIMER(p) \
    do { \
	if ((p)->flags & (F_INSLPQUEUE)) \
	    erl_cancel_timer(&(p)->tm); \
	(p)->flags &= ~(F_INSLPQUEUE|F_TIMO); \
    } while (0)

EXTERN_FUNCTION(void, init_scheduler, (_VOID_));
EXTERN_FUNCTION(int,  sched_q_len, (_VOID_));
EXTERN_FUNCTION(void, add_to_schedule_q, (Process*));
EXTERN_FUNCTION(int,  remove_proc_from_sched_q,  (Process*));
EXTERN_FUNCTION(int,  schedule, (_VOID_));
EXTERN_FUNCTION(uint32, erl_create_process, (Process*, uint32, uint32, uint32,
					   ErlSpawnOpts*));
EXTERN_FUNCTION(void, delete_process, (Process*));
EXTERN_FUNCTION(void, schedule_exit, (Process*, uint32));
EXTERN_FUNCTION(void, do_exit, (Process*, uint32));
EXTERN_FUNCTION(void, set_timer, (Process*, uint32));
EXTERN_FUNCTION(void, cancel_timer, (Process*));

#ifdef BEAM
EXTERN_FUNCTION(void, trace_proc_call, (Process*, uint32));
EXTERN_FUNCTION(void, trace_proc_ret, (Process*, uint32));
#endif

#endif











