/* ``The contents of this file are subject to the Erlang Public License,
 * Version 1.0, (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.erlang.org/EPL1_0.txt
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 * 
 * The Original Code is Erlang-4.7.3, December, 1998.
 * 
 * The Initial Developer of the Original Code is Ericsson Telecom
 * AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
 * Telecom AB. All Rights Reserved.
 * 
 * Contributor(s): ______________________________________.''
 */
/*
** File: erl_message.c
** Author: Tony Rogvall
** Description:
**     Message passing primitives
*/

#include "sys.h"
#include "config.h"
#include "global.h"
#include "erl_message.h"
#include "erl_process.h"

#define MSO_WEIGHT 1024
static int  d_4;
static int  d_8;
static int d_16;
static int d_32;
static int d_64;
static int d_buf[64+1];

static void atexit_message(arg)
void* arg;
{
#if !defined(NO_FIX_ALLOC)
    fix_release(d_4);
    fix_release(d_8);
    fix_release(d_16);
    fix_release(d_32);
    fix_release(d_64);
#endif
}

/* initialize messages */
void init_message()
{
    int i;

    /* Handle free lists etc */
    d_4 = new_fix_size(sizeof(ErlMessageBuffer) + sizeof(uint32)*(4-1));
    d_8 = new_fix_size(sizeof(ErlMessageBuffer) + sizeof(uint32)*(8-1));
    d_16 = new_fix_size(sizeof(ErlMessageBuffer) + sizeof(uint32)*(16-1));
    d_32 = new_fix_size(sizeof(ErlMessageBuffer) + sizeof(uint32)*(32-1));
    d_64 = new_fix_size(sizeof(ErlMessageBuffer) + sizeof(uint32)*(64-1));

    for (i = 0; i <= 4; i++) {
	d_buf[i] = d_4;
    }
    for (i = 5; i <= 8; i++) {
	d_buf[i] = d_8;
    }
    for (i = 9; i <= 16; i++) {
	d_buf[i] = d_16;
    }
    for (i = 17; i <= 32; i++) {
	d_buf[i] = d_32;
    }
    for (i = 33; i <= 64; i++) {
	d_buf[i] = d_64;
    }
    erl_at_exit(atexit_message, NULL);
}

void free_message(mp)
ErlMessage* mp;
{
    fix_free(mesg_desc, (uint32*) mp);
}

/* Allocate message buffer (size in words) */
ErlMessageBuffer* new_message_buffer(size)
uint32 size;
{
    ErlMessageBuffer* bp;


#ifdef INSTRUMENT
    alloc_from(33);
#endif


    if (size > 64) {
	bp = (ErlMessageBuffer*) safe_alloc(sizeof(ErlMessageBuffer) +
					    ((size-1)*sizeof(uint32)));
    } else {
	bp = (ErlMessageBuffer*) fix_alloc(d_buf[size]);
    }
    bp->next = NULL;
    bp->size = size;
    bp->mso = NULL;
    return bp;
}

void free_message_buffer(bp)
ErlMessageBuffer* bp;
{
    ProcBin* pb = bp->mso;
    /* Currently we only use mso for temporary message build !!! */
    while(pb != NULL) {
	ProcBin* pb_next = pb->next;
	maybe_delete_contents(pb);
	fix_free(proc_bin_desc, (uint32*) pb);	
	pb = pb_next;
    }
    if (bp->size > 64)
	sys_free(bp);
    else
	fix_free(d_buf[bp->size], (uint32*) bp);
}

/* Add a message last in message queue */
#ifdef SEQ_TRACE
void queue_message_tt(receiver, bp, message, seq_trace_token)
Process* receiver; ErlMessageBuffer* bp; uint32 message; uint32 seq_trace_token;
#else
void queue_message(receiver, bp, message)
Process* receiver; ErlMessageBuffer* bp; uint32 message;
#endif
{
    ErlMessage* mp;
    ProcBin* pb;

    mp = (ErlMessage*) fix_alloc(mesg_desc);
    mp->mesg = message;
    mp->next = NULL;
#ifdef SEQ_TRACE
    mp->seq_trace_token = seq_trace_token;
#endif
    LINK_MESSAGE(receiver, mp);

    if (bp != NULL) {
	/* Link the message buffer */
	bp->next = receiver->mbuf;
	receiver->mbuf = bp;
	receiver->mbuf_sz += bp->size;
	receiver->mbuf_struct_sz += 
	    (sizeof(ErlMessageBuffer)/sizeof(uint32) - 1); 
	/* mbuf_struct_sz is the administrative overhead caused by 
	   message buffers, used together with mbuf_sz to indicate 
	   that GC is needed */ 

	/* We don't set the F_NEED_GC flag here, as long as the size 
	   (in receiver->mbuf_sz)
	   is updated, the scheduler will notice that GC is needed. */
	/* Move binaries into process */
	if ((pb = bp->mso) != NULL) {
	    bp->mso = NULL;
	    while(pb != NULL) {
		ProcBin* pb_next = pb->next;
		pb->next = receiver->mso;
		receiver->mso = pb;
		receiver->mso_weight += (MSO_WEIGHT + pb->size);
		maybe_gc_binary(receiver, &message);
		pb = pb_next;
	    }
	}
    }
    if (receiver->status == P_WAITING) {
	add_to_schedule_q(receiver);
	ASSERT(&(*receiver->msg.save)->next == receiver->msg.last);
    }
    else if (receiver->status == P_SUSPENDED)
	receiver->rstatus = P_RUNABLE;
    /* Note: a tracer is never traced !!! */
    if (IS_TRACED_FL(receiver, F_TRACE_RECEIVE))
	trace_receive(receiver, message);
}

/*
 * Send a local message when sender & receiver processes are known.
 *
 * For clarity and efficiency reasons, there are two version of this
 * function.
 */

#ifdef SEQ_TRACE

void send_message(sender, receiver, message)
Process* sender; Process* receiver; uint32 message;
{
    uint32* hp;
    ErlMessageBuffer* bp = NULL;
    uint32 msize;
    uint32 token = NIL;

    msize = size_object(message);
    if (SEQ_TRACE_TOKEN(sender) != NIL) {
	seq_trace_update_send(sender);
	seq_trace_output(SEQ_TRACE_TOKEN(sender), message, SEQ_TRACE_SEND, receiver->id);
	bp = new_message_buffer(msize + 6 /* TUPLE5 */);
	hp = bp->mem;
	token = copy_struct(SEQ_TRACE_TOKEN(sender), 6 /* TUPLE5 */, 
			    &hp, &bp->mso);
    } else {
	bp = new_message_buffer(msize);
	hp = bp->mem;
    }
    message = copy_struct(message, msize, &hp, &bp->mso);
    queue_message_tt(receiver, bp, message, token);
}

#else

void send_message(sender, receiver, message)
Process* sender; Process* receiver; uint32 message;
{
    ErlMessageBuffer* bp = NULL;

    if (sender != receiver) {
	uint32* hp;
	uint32 msize;

	msize = size_object(message);
	bp = new_message_buffer(msize);
	hp = bp->mem;
	message = copy_struct(message, msize, &hp, &bp->mso);
    }
    queue_message(receiver, bp, message);
}

#endif

/* Called from dist */
void send_msg(to, message)
uint32 to; uint32 message;
{
    Process* rp;
    ErlMessageBuffer* bp = NULL;
    uint32* hp;
    uint32 size;

    rp = process_tab[get_number(to)];
    if (INVALID_PID(rp, to))
	return;
    size = size_object(message);
    bp = new_message_buffer(size);
    hp = bp->mem;
    message = copy_struct(message, size, &hp, &bp->mso);
    queue_message(rp, bp, message);
}

/* this function delivers an exit message to a processs which is trapping
   exits 
 */

#ifdef SEQ_TRACE
void deliver_exit_message_tt(from, to, reason, token)
uint32 from; Process *to; uint32 reason; uint32 token;
{
    uint32 mess;
    uint32 save;
    uint32 sz_reason;
    ErlMessageBuffer* bp;
    uint32* hp;
    uint32 sz_token;
    uint32 temptoken;

/* We should propagate the tracetoken of from  to to here when
*  we are ready with seq_trace impl
*/
/* don't use from to check seq_trace_token it can be
a remote pid or a port, we need a new deliver_exit_message_tt
for the port and remote case
*/
    
    if (token != NIL) {
	ASSERT(is_tuple(token));
	sz_reason = size_object(reason);
	sz_token = size_object(token);
	bp = new_message_buffer(sz_reason + 4 + sz_token);
	hp = bp->mem;
	mess = copy_struct(reason, sz_reason, &hp, &to->mso);
	save = TUPLE3(hp, am_EXIT, from, mess);
	/* the trace token must in this case be updated by the caller */
	seq_trace_output(token, save, SEQ_TRACE_SEND, to->id);
	temptoken = copy_struct(token, sz_token, &hp, &to->mso);
	queue_message_tt(to, bp, save, temptoken);
    } else {
	sz_reason = size_object(reason);
	bp = new_message_buffer(sz_reason + 4);
	hp = bp->mem;
	mess = copy_struct(reason, sz_reason, &hp, &to->mso);
	save = TUPLE3(hp, am_EXIT, from, mess);
	queue_message(to, bp, save);
    }
}
#else
void deliver_exit_message(from, to, reason)
uint32 from; Process *to; uint32 reason;
{
    uint32 mess;
    uint32 save;
    uint32 sz_reason;
    ErlMessageBuffer* bp;
    uint32* hp;
	
    sz_reason = size_object(reason);
    bp = new_message_buffer(sz_reason + 4);
    hp = bp->mem;
    mess = copy_struct(reason, sz_reason, &hp, &to->mso);
    save = TUPLE3(hp, am_EXIT, from, mess);
    queue_message(to, bp, save);
}

#endif /* SEQ_TRACE */

void deliver_result(sender, pid, res)
uint32 sender; uint32 pid; uint32 res;
{
    uint32 tuple;
    Process *rp;
    ErlMessageBuffer* bp;
    uint32* hp;
    uint32 sz_res;


    rp = process_tab[get_number(pid)];
    if (INVALID_PID(rp, pid))
	return;

    sz_res = size_object(res);
    bp = new_message_buffer(sz_res + 3);
    hp = bp->mem;
    res = copy_struct(res, sz_res, &hp, &rp->mso);
    tuple = TUPLE2(hp, sender, res);
    hp += 3;

    queue_message(rp, bp, tuple);
}













