/* ``The contents of this file are subject to the Erlang Public License,
 * Version 1.0, (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.erlang.org/EPL1_0.txt
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 * 
 * The Original Code is Erlang-4.7.3, December, 1998.
 * 
 * The Initial Developer of the Original Code is Ericsson Telecom
 * AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
 * Telecom AB. All Rights Reserved.
 * 
 * Contributor(s): ______________________________________.''
 */
/*
 * This file is copyright (c) Ellemtel in January 1993
 *
 * Author Claes Wikstrom
 *
 *
 */

#include "sys.h"
#include "config.h"
#include "global.h"
#include "erl_process.h"
#include "error.h"
#include "bif.h"

#define MSO_WEIGHT 1024
#define MSO_GC_LIMIT (64*1024)


void insert_pb(msoh, pb)
ProcBin** msoh; ProcBin *pb;
{
    pb->mark = 0;       /* We'll always pass through here */
    pb->next = *msoh;
    *msoh = pb;
}

ProcBin* alloc_binary(msoh, buf, len)
ProcBin** msoh; byte* buf; int len;
{
    ProcBin *pb;
    Binary *bptr;

    bptr = (Binary*) safe_alloc_from(61,len+sizeof(Binary));
    bptr->orig_size = len;
    bptr->refc = 1;
    if (buf != NULL)
	sys_memcpy(bptr->orig_bytes,buf,len);
    pb = (ProcBin*) fix_alloc_from(62,proc_bin_desc);
    tot_bin_allocated += len;
    pb->next = NULL;
    pb->val = bptr;
    pb->bytes = bptr->orig_bytes;
    pb->size = len;
    insert_pb(msoh, pb);
    return pb;
}


/* create a brand new binary from scratch */

uint32 new_binary(p,buf,len)	
Process *p; byte *buf; int len;
{
    ProcBin *pb = alloc_binary(&p->mso, buf, len);

    p->mso_weight += (MSO_WEIGHT + pb->size);
    return make_binary(pb);
}


/* copy binary to process p */

ProcBin* copy_binary(msoh, pb)
ProcBin** msoh; ProcBin* pb;
{
    ProcBin *newpb;
    
    newpb = (ProcBin*) fix_alloc_from(62,proc_bin_desc);
    newpb->next = NULL;
    newpb->bytes = pb->bytes;
    newpb->size = pb->size;
    newpb->val = pb->val;
    newpb->val->refc++;
    insert_pb(msoh, newpb);
    return newpb;
}


/* create a new binary that points straight into the data of an old.
   This can be implemented due to the fact that
   no destructive ops are allowed towards a binary */


static uint32 offset_binary(p, old, offset, len)
Process *p; ProcBin* old; int offset; int len;
{
    ProcBin *pb;

    if (offset+len-1 > old->size)
	return 0;
    pb = (ProcBin*) fix_alloc_from(62,proc_bin_desc);
    pb->val = old->val;
    pb->val->refc++;
    pb->bytes = old->bytes + offset - 1;
    pb->size = len;
    insert_pb(&p->mso, pb);
    return make_binary(pb);
}
	
BIF_RETTYPE binary_to_list_1(BIF_ALIST_1)
BIF_ADECL_1
{
    int i;
    uint32 previous;
    ProcBin *pb;
    uint32* hp;

    if (is_not_binary(BIF_ARG_1)) {
	BIF_ERROR1(BADARG, am_binary_to_list, BIF_ARG_1);
    }
    pb = (ProcBin*) ptr_val(BIF_ARG_1);
    i = pb->size;
    hp = HAlloc(BIF_P, i * 2);
    previous = NIL;
    while (i--) {
	previous = CONS(hp, make_small(pb->bytes[i]), previous);
	hp += 2;
    }
    BIF_RET(previous);
}

BIF_RETTYPE binary_to_list_3(BIF_ALIST_3)
BIF_ADECL_3
{
    int i;
    uint32 previous;
    ProcBin *pb;
    int start,stop;
    uint32* hp;

    if (is_not_binary(BIF_ARG_1)) {
	BIF_ERROR3(BADARG, am_binary_to_list, 
		   BIF_ARG_1, BIF_ARG_2, BIF_ARG_3);
    }
    pb = (ProcBin*) ptr_val(BIF_ARG_1);
    i = pb->size;
    if (is_not_small(BIF_ARG_3) || is_not_small(BIF_ARG_2)) {
	BIF_ERROR3(BADARG, am_binary_to_list, 
		   BIF_ARG_1, BIF_ARG_2, BIF_ARG_3);
    }
    start = signed_val(BIF_ARG_2);
    stop = signed_val(BIF_ARG_3);
    if (start < 1 || start > i || stop < 1 || stop > i || stop < start) {
	BIF_ERROR3(BADARG, am_binary_to_list, 
		   BIF_ARG_1, BIF_ARG_2, BIF_ARG_3);
    }

    hp = HAlloc(BIF_P, (stop-start+1) * 2);
    previous = NIL;
    for (i = stop - 1; i >= start -1 ; i--) {
	previous = CONS(hp, make_small(pb->bytes[i]), previous);
	hp += 2;
    }
    BIF_RET(previous);
}


/* Turn a possibly deep list of ints (and binaries) into */
/* One large binary object                               */

BIF_RETTYPE list_to_binary_1(BIF_ALIST_1)
BIF_ADECL_1
{
    uint32 bin;
    int j,i;
    ProcBin *bp;
    int gcalls;

    if (is_nil(BIF_ARG_1))
	BIF_RET(new_binary(BIF_P,(byte*)"",0));
    if (is_not_list(BIF_ARG_1)) {
	BIF_ERROR1(BADARG, am_list_to_binary, BIF_ARG_1);
    }
    if((i = io_list_len(BIF_ARG_1)) < 0) {
	BIF_ERROR1(BADARG, am_list_to_binary, BIF_ARG_1);
    }
    j = 0;
    bin = new_binary(BIF_P, (byte *)NULL, i);
    gcalls = maybe_gc_binary(BIF_P, &bin);
    bp = (ProcBin *) ptr_val(bin);
    if (io_list_to_buf(BIF_ARG_1, (char*) bp->bytes, &j, i+1) != 0) {
	BIF_ERROR1(BADARG, am_list_to_binary, BIF_ARG_1);
    }
    BIF_RET2(bin,gcalls);
}


BIF_RETTYPE concat_binary_1(BIF_ALIST_1)
BIF_ADECL_1
{
    return list_to_binary_1(BIF_ALIST_1);  /* Not meaningful any longer */
}


BIF_RETTYPE split_binary_2(BIF_ALIST_2)
BIF_ADECL_2
{
    int pos;
    ProcBin *pb;
    uint32 tmp1, tmp2;
    uint32* hp;

    if (is_not_small(BIF_ARG_2)) {
	BIF_ERROR2(BADARG, am_split_binary, BIF_ARG_1, BIF_ARG_2);
    }
    if (is_not_binary(BIF_ARG_1)) {
	BIF_ERROR2(BADARG, am_split_binary, BIF_ARG_1, BIF_ARG_2);
    }
    if ((pos = signed_val(BIF_ARG_2)) < 0) {
	BIF_ERROR2(BADARG, am_split_binary, BIF_ARG_1, BIF_ARG_2);
    }

    hp = HAlloc(BIF_P, 3);
    pb = (ProcBin*) ptr_val(BIF_ARG_1);
    if((tmp1 = offset_binary(BIF_P, pb, 1, pos)) == 0) {
	BIF_ERROR2(BADARG, am_split_binary, BIF_ARG_1, BIF_ARG_2);
    }
    if((tmp2 = offset_binary(BIF_P, pb, pos+1, pb->size - pos)) == 0) {
	BIF_ERROR2(BADARG, am_split_binary, BIF_ARG_1, BIF_ARG_2);
    }
    BIF_RET(TUPLE2(hp, tmp1, tmp2));
}


void maybe_delete_contents(pb)
ProcBin *pb;
{
    pb->val->refc--;
    if (pb->val->refc == 0) {
	tot_bin_allocated -= pb->val->orig_size;
	sys_free((char*)pb->val);
	return;
    }
}

/* Do we want to garbage collect ??? */

int maybe_gc_binary(p, obj)
Process* p; uint32* obj;
{
    if (p->mso_weight > MSO_GC_LIMIT) {
	p->flags |= F_NEED_GC;
	return 1;
    }
    return 0;
}

/* The generational collector calls this function */
/* It only removes bins that still have the INITIAL mark */
/* The gen gc may move bins to the old heap, in that case */
/* it also changes the mark to BIN_OLD */

void generation_bin_gc(p)
Process* p;
{
    ProcBin **prev, *ptr;

    prev = &p->mso;
    ptr = p->mso;
    p->mso_weight = 0;
    while(ptr) {
	switch (ptr->mark) {
	case BIN_INITIAL:
            *prev = ptr->next;
            maybe_delete_contents(ptr);
            fix_free(proc_bin_desc, (uint32*) ptr);
            ptr = *prev;
	    break;
	case BIN_MARKED:
	    ptr->mark = BIN_INITIAL;
	    prev = &ptr->next;
            ptr = ptr->next;
	    break;
	case BIN_OLD:  /* Tenured object, move it to old_proc_bins */
	    *prev = ptr->next;
	    ptr->next = p->old_mso;
	    p->old_mso = ptr;
	    ptr = *prev;
	    break;
	default:
	    erl_exit(1, "Found bad mark in proc_bin_list %d\n",ptr->mark);
	}
    }
}

/* The fullsweep collector calls this function */
/* It collects *all* binary objects that are not */
/* marked by the fullsweep collector, it also  */
/* moves all remaining binary objects to the old_mso list */

void fullsweap_bin_gc(p)
Process* p;
{
    ProcBin **prev, *ptr;
    int dels = 0;
#ifdef DEBUG
    int after_len, len , old_len;
    len = old_len = after_len = 0;
    ptr = p->mso;
    while(ptr) {
	len++;
	ptr = ptr->next;
    }
    ptr = p->old_mso;
    while(ptr) {
	old_len++;
	ptr = ptr->next;
    }
#endif

    prev = &p->mso;
    ptr = p->mso;
    p->mso_weight = 0;
    while(ptr) {
	if (ptr->mark == BIN_FULLSWEAP_MARKED) {
	    ptr->mark = BIN_OLD;
	    prev = &ptr->next;
            ptr = ptr->next;
	}
	else {
            *prev = ptr->next;
	    dels++;
            maybe_delete_contents(ptr);
            fix_free(proc_bin_desc, (uint32*) ptr);
            ptr = *prev;
	}
    }

    prev = &p->old_mso;    /* And do the same thing again */
    ptr = p->old_mso;
    while(ptr) {
	if (ptr->mark == BIN_FULLSWEAP_MARKED) {
	    ptr->mark = BIN_OLD;
	    prev = &ptr->next;
            ptr = ptr->next;
	}
	else {
	    dels++;
            *prev = ptr->next;
            maybe_delete_contents(ptr);
            fix_free(proc_bin_desc, (uint32*) ptr);
            ptr = *prev;
	}
    }
    *prev = p->mso;   /* Link together both chains in old_mso */
    p->mso = NULL;    /* and drop mso                         */
#ifdef DEBUG
    ptr = p->old_mso;
    while(ptr) {
	after_len++;
	ptr = ptr->next;
    }
    if (after_len != (len +old_len - dels) )
	abort();
#endif
}


/* This one is used by the old collector */


void proc_bin_gc(p)
Process* p;
{
    ProcBin **prev, *ptr;

    prev = &p->mso;
    ptr = p->mso;
    p->mso_weight = 0;
    while(ptr) {
        if (ptr->mark == 0) {
            *prev = ptr->next;
            maybe_delete_contents(ptr);
            fix_free(proc_bin_desc, (uint32*) ptr);
            ptr = *prev;
        }
	else {
	    ptr->mark = 0;
            prev = &ptr->next;
            ptr = ptr->next;
        }
    }
}
