/* ``The contents of this file are subject to the Erlang Public License,
 * Version 1.0, (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.erlang.org/EPL1_0.txt
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 * 
 * The Original Code is Erlang-4.7.3, December, 1998.
 * 
 * The Initial Developer of the Original Code is Ericsson Telecom
 * AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
 * Telecom AB. All Rights Reserved.
 * 
 * Contributor(s): ______________________________________.''
 */
/*
** File: atom.c
**
*/
#include "sys.h"
#include "config.h"
#include "global.h"
#include "hash.h"
#include "atom.h"

#define ATOM_SIZE  1000
#define ATOM_LIMIT (64*1024)
#define ATOM_RATE  100

IndexTable atom_table;    /* The index table */

/* functions for allocating space for the ext of atoms. We do not
** use malloc for each atom to prevent excessive memory fragmentation
*/

typedef struct _atom_text {
    struct _atom_text* next;
    char text[ATOM_TEXT_SIZE];
} AtomText;

static AtomText* text_list;  /* list of text buffers */

static byte *atom_text_pos;
static byte *atom_text_end;
uint32 reserved_atom_space;      /* Total amount of atom text space */
uint32 atom_space;	         /* Amount of atom text space used */

/*
 * Special atoms.
 */
uint32 am_noname;
uint32 am_underscore;

/*
** Print info about atom tables
*/
void atom_info(to)
CIO to;
{
    index_info(to, &atom_table);
    erl_printf(to,"Atom space  %d/%d\n", atom_space, reserved_atom_space);
}

/*
** Allocate an atom text segments
*/
static void more_atom_space()
{
    AtomText* ptr;

    if ((ptr = (AtomText*) sys_alloc_from(1,sizeof(AtomText))) == NULL)
	erl_exit(1, "out of memory -- panic");
    ptr->next = text_list;
    text_list = ptr;

    atom_text_pos = ptr->text;
    atom_text_end = atom_text_pos + ATOM_TEXT_SIZE;
    reserved_atom_space += sizeof(AtomText);

    VERBOSE(erl_printf(COUT,"Allocated %d atom space\n",
		       ATOM_TEXT_SIZE););
}

/*
** Allocate string space with in an atom text segment
*/

static byte *atom_text_alloc(bytes)
int bytes;
{
    byte *res;

    if (bytes >= ATOM_TEXT_SIZE)
	erl_exit(1, "absurdly large atom --- panic\n");

    if (atom_text_pos + bytes >= atom_text_end)
	more_atom_space();
    res = atom_text_pos;
    atom_text_pos += bytes;
    atom_space    += bytes;
    return res;
}

/*
** Calculate atom hash value
** use hash algorrithm hashpjw (from Dragon Book)
*/

static HashValue atom_hash(obj)
Atom* obj;
{
    byte* p = obj->name;
    int len = obj->len;
    HashValue h = 0, g;

    while(len--) {
	h = (h << 4) + *p++;
	if ((g = h & 0xf0000000)) {
	    h ^= (g >> 24);
	    h ^= g;
	}
    }
    return h;
}


static int atom_cmp(tmpl, obj)
Atom* tmpl; Atom* obj;
{
    if (tmpl->len == obj->len &&
	sys_memcmp(tmpl->name, obj->name, tmpl->len) == 0)
	return 0;
    return 1;
}


static Atom* atom_alloc(tmpl)
Atom* tmpl;
{
    Atom* obj = (Atom*) fix_alloc_from(11, atom_desc);

    if (tmpl->slot.index != -2) {
	obj->name = atom_text_alloc(tmpl->len);
	sys_memcpy(obj->name, tmpl->name, tmpl->len);
    }
    else
	obj->name = tmpl->name;
    obj->len = tmpl->len;
    obj->slot.index = -1;
    return obj;
}

/* Reuse atom  text ??? */

static void atom_free(obj)
Atom* obj;
{
    fix_free(atom_desc, (void*) obj);
}


int atom_get(name, len)
byte* name; int len;
{
    Atom a;

    a.len = len;
    a.name = name;

    return index_get(&atom_table, (void*) &a);
}

int atom_put(name, len)
byte* name; int len;
{
    Atom a;

    a.len = len;
    a.name = name;
    a.slot.index = -1;

    return index_put(&atom_table, (void*) &a);
}

/* Insert atom but do not allocate memory for name */


int atom_static_put(name, len)
byte* name; int len;
{
    Atom a;

    a.len = len;
    a.name = name;
    a.slot.index = -2;

    return index_put(&atom_table, (void*) &a);
}

static void atexit_atom(arg)
void* arg;
{
    AtomText* ptr = text_list;

    index_delete(&atom_table);

    while(ptr != NULL) {
	AtomText* ptr_next = ptr->next;
	sys_free(ptr);
	ptr = ptr_next;
    }
}

void init_atom_table()
{
    HashFunctions f;
    int i;

    f.hash = (H_FUN) atom_hash;
    f.cmp  = (HCMP_FUN) atom_cmp;
    f.alloc = (HALLOC_FUN) atom_alloc;
    f.free = (HFREE_FUN) atom_free;

    atom_text_pos = NULL;
    atom_text_end = NULL;
    reserved_atom_space = 0;
    atom_space = 0;
    text_list = NULL;

    index_init(&atom_table, "atom_tab",
	       ATOM_SIZE, ATOM_LIMIT, ATOM_RATE, f);

    more_atom_space();

    /* Special atoms */
    am_noname = am_magic_word("nonode@nohost");
    am_underscore = am_magic_word("_");

    /* Ordinary atoms */
    for (i = 0; i < AM_SIZE; i++)
	am_ix[i] = am_magic_word(am_nm[i]);
    erl_at_exit(atexit_atom, NULL);
}
