/* ``The contents of this file are subject to the Erlang Public License,
 * Version 1.0, (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.erlang.org/EPL1_0.txt
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 * 
 * The Original Code is Erlang-4.7.3, December, 1998.
 * 
 * The Initial Developer of the Original Code is Ericsson Telecom
 * AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
 * Telecom AB. All Rights Reserved.
 * 
 * Contributor(s): ______________________________________.''
 */
/*
 * This file is copyright (c) Ellemtel in January 1991
 *
 * Author Mike Williams
 * and Claes Wikstrom changed the loader to load a binary
 * instead of reading a file ...
 *
 * Mike merged this with load.c and remove load.c
 */

#include "sys.h"
#include "config.h"
#include "global.h"
#include "jam_load.h"
#include "magic.h"
#include "erl_process.h"
#include "index.h"
#include "jam_opcodes.h"


#define LDEBUG

static const char *load_errs[] = {
    "", 
    "Bad string",
    "Bad magic",
    "Bad function",
    "Truncated",
    "Too many functions",
    "Bad format",
    "Bad float", 
    "Too long function names",
    "No module name",
    "Bad module name" };


typedef struct b_file {
    byte* cp;       /* current position */
    byte* start;    /* point to the beginning of the jam code */
    byte* end;      /* points to one position behind the jam code */
} BFile;

#define GET8(f) (((f)->cp+1 >= (f)->end) ? END_OF_FILE : (((f)->cp += 1),\
		 ((f)->cp[-1])))
#define GET16(f) (((f)->cp+2 > (f)->end) ? END_OF_FILE : (((f)->cp += 2),\
		  ((f)->cp[-2] <<8 | (f)->cp[-1])))
#define GET24(f) (((f)->cp+3 > (f)->end) ? END_OF_FILE : (((f)->cp += 3), \
		  ((f)->cp[-3] <<16 | (f)->cp[-2]<<8 | (f)->cp[-1])))

#define SKIP8(f)     (f)->cp += 1
#define SKIP16(f)    (f)->cp += 2
#define SKIP24(f)    (f)->cp += 3
#define SKIP_STR(f)  do { int __len; \
                          SKIP8(f); \
			__len = GET16(f); (f)->cp += __len; \
		     } while(0)

static FUNCTION(void, delete_code, (int));
static FUNCTION(void, delete_export_references, (int));


static int b_fread (ptr, size, nitems, f)
char *ptr; int size; int nitems; BFile *f;
{
    int n = size*nitems;

    if (f->cp+n > f->end)
	return 0;
    sys_memcpy(ptr,f->cp, n);
    f->cp += n;
    return n;
}

static int b_fseek(f, offset, ptrname)
BFile *f; long offset; int ptrname;
{
    switch(ptrname) {
    case 0:
	if (f->start + offset > f->end)
	    return -1;
	f->cp = f->start + offset;
	break;
    case 1:
	if (f->cp + offset > f->end)
	    return -1;
	f->cp += offset;
	break;
    case 2:
	if (f->end - offset < f->start)
	    return -1;
	f->cp = f->end - offset;
	break;
    default:
	return -1;
    }
    return 0;
}


static int b_get_string(f, cpp)
BFile *f; char** cpp;
{
    if (GET8(f) != L_STRING)
	return -1;
    else {
	int len = GET16(f);
	*cpp = (char*) f->cp;          /* the string pointer */
	if (f->cp + len >= f->end)
	    return 0;
	f->cp += len;
	return len;
    }
}

/*****************************************************************

   Function defintions when loading jam code
  
*******************************************************************/

#define FUNDEF_SIZE_INITIAL    100
#define FUNDEF_SIZE_LIMIT      64*1024
#define FUNDEF_SIZE_INCREAMENT 10

typedef struct function_def 
{
    IndexSlot slot;
    byte *name;
    int len;
    int address;
    int arity;
} FunDef;

static IndexTable* fundef_table;
static int fundef_desc;

#define function_addr(i) (((FunDef*)fundef_table->table[i])->address)

/*
** Calculate atom hash value
** use hash algorrithm hashpjw (from Dragon Book)
*/

static HashValue fundef_hash(obj)
FunDef* obj;
{
    byte* p = obj->name;
    int len = obj->len;
    HashValue h = 0;

    while(len--)
	h += *p++;
    h += obj->arity;
    return h;
}


static int fundef_cmp(tmpl, obj)
FunDef* tmpl; FunDef* obj;
{
    if ((tmpl->len == obj->len) &&
	(tmpl->arity == obj->arity) &&
	(sys_memcmp(tmpl->name, obj->name, tmpl->len) == 0))
	return 0;
    return 1;
}

static FunDef* fundef_alloc(tmpl)
FunDef* tmpl;
{
    FunDef* obj = (FunDef*) fix_alloc_from(171,fundef_desc);

    obj->name = tmpl->name;
    obj->len = tmpl->len;
    obj->arity = tmpl->arity;
    obj->address = tmpl->address;
    obj->slot.index = -1;
    return obj;
}


static void fundef_free(obj)
FunDef* obj;
{
#if defined(NO_FIX_ALLOC)
    fix_free(fundef_desc,obj);
#else
    /* Free everything with fix_release(fundef_desc); */
#endif
}



int fundef_get(name, len, arity)
byte* name; int len; int arity;
{
    FunDef f;

    f.name = name;
    f.len = len;
    f.arity = arity;

    return index_get(fundef_table, (void*) &f);
}


int fundef_put(name, len, arity, offset)
byte* name; int len; int arity; int offset;
{
    FunDef f;

    f.len = len;
    f.name = name;
    f.arity = arity;
    f.address = offset;
    f.slot.index = -1;

    return index_put(fundef_table, (void*) &f);
}


#define LOAD_ERR(x)       { done = x; break; }

#define LOAD_BUF_SIZE 256

static char load_buf[LOAD_BUF_SIZE];

static char* copy_str(str, len)
byte* str; int len;
{
    if (len >= LOAD_BUF_SIZE) {
	sys_memcpy(load_buf, str, LOAD_BUF_SIZE);
	load_buf[LOAD_BUF_SIZE-1] = '\0';
	load_buf[LOAD_BUF_SIZE-2] = '.';
	load_buf[LOAD_BUF_SIZE-3] = '.';
	load_buf[LOAD_BUF_SIZE-4] = '.';
    }
    else {
	sys_memcpy(load_buf, str, len);
	load_buf[len] = '\0';
    }
    return load_buf;
}


static int bin_load_done(code)
int code;
{
    index_delete(fundef_table);
#if !defined(NO_FIX_ALLOC)
    fix_release(fundef_desc);
#endif
    return code;
}


int bin_load(group_leader, mod, module,size)
uint32 group_leader;
uint32 mod;     /* module name as an atom */
byte *module;   /* points to the jam code */
int size;
{
    BFile mfile;
    byte module_name[MAX_LOAD_FILE_PATH];
    byte *code;
    byte patch0;
    byte patch1;
    byte *last_function_code;
    char* str_p;
    int fpatch;
    byte *b;
    double float_patch;
    uint16 offset;
    sint32 offset_3;
    int module_length;
    int field_length;
    int function_number;
    int max_functions;
    int this_module;
    int module_atom_no;
    int i;
    int done;
    int arity;
    int module_name_length;
    int errors;
    int local_or_exported;
    int function_atom_no;
    int export_index;
    int patch_type;
    int patch_address;
    int other_module;
    int this_atom_no;
    int exports;
    HashFunctions f;

    f.hash = (H_FUN) fundef_hash;
    f.cmp  = (HCMP_FUN) fundef_cmp;
    f.alloc = (HALLOC_FUN) fundef_alloc;
    f.free = (HFREE_FUN) fundef_free;
    fundef_table = index_new("fun_def",
			     FUNDEF_SIZE_INITIAL,
			     FUNDEF_SIZE_LIMIT,
			     FUNDEF_SIZE_INCREAMENT, f);

    /* Setup the pointer to the module */
    mfile.cp = module;
    mfile.start = module;
    mfile.end = module + size;

    /* pass 1 check file for correctness, work out code length 
       register function name etc */
    exports = 0;
    errors = 0;
    done = 0;
    max_functions = -1;
    this_module = -1;
    module_length = 0;
    function_number = -1;
    while (!done) {
	switch (GET8(&mfile)) {
	case L_CODE_LENGTH:
	    field_length = GET16(&mfile);
	    module_length += field_length; 
	    /* and skip over the code */
	    if (b_fseek(&mfile, field_length, 1) < 0) 
		LOAD_ERR(TRUNCATED);
	    break;
	case L_CODE_LENGTH_3:
	    field_length = GET24(&mfile);
	    module_length += field_length; 
	    /* and skip over the code */
	    if (b_fseek(&mfile, field_length, 1) < 0) 
		LOAD_ERR(TRUNCATED);
	    break;
	case L_MAGIC_STRING:
	    if ((field_length = b_get_string(&mfile, &str_p)) < 0 ||
		(field_length >= LOAD_BUF_SIZE))
		LOAD_ERR(BAD_STRING);
	    copy_str(str_p, field_length); 
	    if (sys_strcmp(load_buf, MAGICSTR_1_0) == 0) {
#ifdef LDEBUG
		VERBOSE(erl_printf(COUT,"Magic string OK\n"););
#endif
		break;
	    }
	    LOAD_ERR(BAD_MAGIC);
	case L_COPYRIGHT:
	    if ((field_length = b_get_string(&mfile, &str_p)) < 0 ||
		(field_length >= LOAD_BUF_SIZE))
		LOAD_ERR(BAD_STRING);
#ifdef LDEBUG
	    VERBOSE({
		erl_printf(COUT,"Copyright on file\n%s\n",
			   		copy_str(str_p, field_length));
	    });
#endif
	    break;
	case L_FUNCTION:
	    if ((field_length = b_get_string(&mfile,&str_p)) < 0)
		LOAD_ERR(BAD_STRING);
	    if (this_module == -1) {
		sys_memcpy(module_name,str_p, field_length);
		module_name_length = field_length;
		module_name[field_length] = '\0';
		if (field_length != atom_tab(unsigned_val(mod))->len ||
		    (sys_strncmp(str_p, 
				 (char*)atom_tab(unsigned_val(mod))->name,
				 field_length)
		     != 0))
		    LOAD_ERR(BAD_MODULE_NAME);
#ifdef LDEBUG
		VERBOSE(erl_printf(COUT,"Module = %s\n", module_name););
#endif
		this_module = 0;
	    }
	    i = GET8(&mfile);
	    if ((i != L_EXPORTED ) && (i != L_LOCAL))
		LOAD_ERR(BAD_FUNCTION);
	    if ((field_length = b_get_string(&mfile,&str_p)) < 0)
		LOAD_ERR(BAD_STRING);

	    function_number++;
	    arity = GET8(&mfile);

	    if (fundef_put(str_p,field_length,arity,module_length) !=
		function_number)
		erl_exit(1, "Bad function number while loading\n");
	    
#ifdef LDEBUG
	    VERBOSE({
		erl_printf(COUT,"Function = %s/%d\n", 
			   copy_str(str_p, field_length), arity);
	    });
#endif
	    max_functions = function_number;
	    break;
	case L_PATCH_AT:
	    SKIP16(&mfile);
	    break;
	case L_PATCH_AT_3:
	    SKIP24(&mfile);
	    break;
	case L_PATCH_CONSTANT:
	    if (b_get_string(&mfile, &str_p) < 0)
		LOAD_ERR(BAD_STRING);
	    break;
	case L_PATCH_LOCAL:
	    if (b_get_string(&mfile, &str_p) < 0)
		LOAD_ERR(BAD_STRING);
	    SKIP8(&mfile); /* arity */
	    break;
	case L_PATCH_FLOAT:
	    if ((field_length = b_get_string(&mfile, &str_p)) < 0 ||
		(field_length >= LOAD_BUF_SIZE))
		LOAD_ERR(BAD_STRING);
	    copy_str(str_p, field_length);
	    if (sys_chars_to_double(load_buf, &float_patch) != 0)
		LOAD_ERR(BAD_FLOAT);
	    break;
	case L_IMPORT:
	    if (b_get_string(&mfile,&str_p) < 0)
		LOAD_ERR(BAD_STRING);
	    if (b_get_string(&mfile, &str_p) < 0)
		LOAD_ERR(BAD_STRING);
	    SKIP8(&mfile); /* arity */
	    break;
	case END_OF_FILE:
	    done = 1;
	    break;
	default: 
	    LOAD_ERR(BAD_FORMAT);
	}
    }

    if (done < 0) {
	if (done == BAD_MODULE_NAME) {
	    erl_printf(CBUF,"** Modulename in objectcode differs from ");
	    erl_printf(CBUF,"Modulename in argument **\n");
	}
	else
	    erl_printf(CBUF,"Format error <%s> in loading\n", 
		       load_errs[-done]);
	send_error_to_logger(group_leader);
#ifdef DEBUG
	erl_printf(CERR,"Tried to load file with size %d\n",size);
	bin_write(CERR, module,size > 100 ? 100 : size );
#endif
	return bin_load_done(-1);
    }

    if (this_module == -1) {
 	erl_printf(CBUF,"No module name found in binary\n");
	send_error_to_logger(group_leader);
 	done = NO_MODULE_NAME;
	return bin_load_done(-1);
   }

#ifdef LDEBUG
    VERBOSE(erl_printf(COUT,"File consistency OK\n"););
#endif
    /* pass 2 - check for undefined functions. In this pass we know
     * that the format of the file if OK
     */

    (void) b_fseek(&mfile, 0, 0);
    done = 0;
    while (!done) {
	switch (GET8(&mfile)) {
	case L_CODE_LENGTH:
	    field_length = GET16(&mfile);
	    b_fseek(&mfile, field_length, 1);
	    break;
	case L_CODE_LENGTH_3:
	    field_length = GET24(&mfile);
	    b_fseek(&mfile, field_length, 1);
	    break;
	case L_COPYRIGHT:
	case L_MAGIC_STRING:
	    SKIP_STR(&mfile);
	    break;
	case L_FUNCTION:
	    SKIP_STR(&mfile);     /* skip module name */
	    SKIP8(&mfile);	  /* Local or exported */
	    SKIP_STR(&mfile);     /* function name */
	    SKIP8(&mfile);	  /* arity */
	    break;
	case L_PATCH_AT:
	    SKIP16(&mfile);
	    break;
	case L_PATCH_AT_3:
	    SKIP24(&mfile);
	    break;
	case L_PATCH_CONSTANT:
	    SKIP_STR(&mfile);
	    break;
	case L_PATCH_LOCAL:
	    field_length = b_get_string(&mfile,&str_p);
	    arity = GET8(&mfile); /* arity */

	    if ((i = fundef_get(str_p, field_length, arity)) == -1) {
		byte tmp_name[128];
		if (field_length > 127)
		    field_length = 127;
		sys_memcpy((char*)tmp_name, str_p, field_length);
		tmp_name[field_length] = '\0';
		erl_printf(CBUF, "Missing local function: %s:%s/%d\n",
			   module_name, tmp_name, arity);
		send_error_to_logger(group_leader);
		errors++;
	    }
	    break;
	case L_PATCH_FLOAT:
	    SKIP_STR(&mfile);
	    break;
	case L_IMPORT:
	    SKIP_STR(&mfile);
	    SKIP_STR(&mfile);
	    SKIP8(&mfile);
	    break;
	case END_OF_FILE:
	    done = 1;
	    break;
	}
    }
    if (errors) {
	erl_printf(CBUF, "%d missing functions - load failed\n", errors);
	send_error_to_logger(group_leader);
	return bin_load_done(-1);
    }
#ifdef LDEBUG
    VERBOSE(erl_printf(COUT,"No undefined functions\n"););
#endif

    /* pass 3 all OK - nothing undefined */
    (void) b_fseek(&mfile, 0, 0);
    done = 0;
    module_atom_no = atom_put(module_name,module_name_length);
    if ((this_module = module_get(module_atom_no)) != -1) {
	if (module_code(this_module)->old_code_length != 0 &&
	    module_code(this_module)->code_length != 0 ) {
	    /* we only keep one version of old code for a module */
	    erl_printf(CBUF, "Module ");
	    print_atom(module_atom_no, CBUF);
	    erl_printf(CBUF, " must be purged before loading\n");
	    send_error_to_logger(group_leader);
	    return bin_load_done(-3);
	}
	else if (module_code(this_module)->old_code_length == 0) {
	    /* store the current version */
	    if (display_loads)
		erl_printf(COUT,"saving old code\n");
	    delete_code(this_module);
	    delete_export_references(module_atom_no);
	}
    }
    this_module = module_put(module_atom_no);

    /* XXX Do we need safe memory here NO ??? */
    code = (byte *) safe_alloc_from(170,module_length);
    
    VERBOSE(erl_printf(COUT,"Allocated %d bytes of memory at 0x%x\n",
		       module_length, code););

    module_code(this_module)->code = code;
    module_code(this_module)->code_length = module_length;
    function_number = -1;
    while (!done) {
	switch (GET8(&mfile)) {
	case L_CODE_LENGTH:
	    field_length = GET16(&mfile);
	    last_function_code = code;
	    b_fread(code, 1, field_length, &mfile);
	    code += field_length;
#ifdef LDEBUG
	    VERBOSE(erl_printf(COUT,"Read %d bytes at %x\n",
			       field_length, last_function_code););
#endif
	    break;
	case L_CODE_LENGTH_3:
	    field_length = GET24(&mfile);
	    last_function_code = code;
	    b_fread(code, 1, field_length, &mfile);
	    code += field_length;
#ifdef LDEBUG
	    VERBOSE(erl_printf(COUT,"Read %d bytes at %x\n",
			       field_length, last_function_code););
#endif
	    break;
	case L_COPYRIGHT:
	case L_MAGIC_STRING:
	    SKIP_STR(&mfile);
	    break;
	case L_FUNCTION:
	    function_number++;
	    SKIP_STR(&mfile);                           /* skip module name */
	    local_or_exported = GET8(&mfile);
	    field_length = b_get_string(&mfile,&str_p);	/* function name */
	    arity = GET8(&mfile);	/* arity */
#ifdef LDEBUG
	    VERBOSE({
		erl_printf(COUT,"%s/%d\n", 
			   copy_str(str_p, field_length), arity);
	    });
#endif
	    if (local_or_exported == L_EXPORTED) {
		exports++;
		function_atom_no = atom_put((byte*)str_p,field_length);
		export_index = export_put(module_atom_no,
					  function_atom_no, arity);
		export_list(export_index)->address = code;
#ifdef LDEBUG
		VERBOSE({
		    erl_printf(COUT,"Export = %s/%d\n",
			       copy_str(str_p, field_length), arity);
		});
#endif
	    }
	    break;
	case L_PATCH_AT:
	    patch_address = GET16(&mfile); /* in bytes */
#ifdef LDEBUG
	    VERBOSE(erl_printf(COUT,"function relative patch address = %d\n",
			       patch_address););
#endif
	    if ((patch_type == PATCH_CSA) ||
		(patch_type == PATCH_CONST)) {
		*(last_function_code +  patch_address) = patch1;
		*(last_function_code +  patch_address + 1) = patch0;
		break;
	    }
	    if (patch_type == PATCH_APPLY) {
		if (*(last_function_code +  patch_address -2) ==
		    OP_enter_remote) 
		    *(last_function_code +  patch_address -2) = OP_apply_enter;
		else 
		    /* assume it is remote call */
		    *(last_function_code +  patch_address -2) = OP_apply_call;
		break;
	    }
	    if (patch_type == PATCH_F) {
		b = last_function_code + patch_address;
		float_to_bytes(b, float_patch);
		break;
	    }
	    if (patch_type == PATCH_BIF) {
		/*
		  BIF calls are of the form kernel:bif_name(......)
		  We have already checked that it is a BIF so we now
		  just have to patch the BIF index (in patch) and
		  munge the opcode 
		 */
		*(last_function_code +  patch_address) = patch1;
		*(last_function_code +  patch_address + 1) = patch0;
		if (*(last_function_code +  patch_address -2) ==
		    OP_enter_remote) 
		    *(last_function_code +  patch_address -2) = OP_bif_enter;
		else 
		    /* assume it is remote call */
		    *(last_function_code +  patch_address -2) = OP_bif_call;
		break;
	    }
	    
	    /* must be a local function patch - work out offset */
	    offset = function_addr(fpatch) - 
		function_addr(function_number) - patch_address + 1;
	    *(last_function_code +  patch_address) = 
		hi_byte((uint16) offset);
	    *(last_function_code +  patch_address +1) =
		lo_byte((uint16) offset);
	    break;
	case L_PATCH_AT_3:
	    patch_address = GET24(&mfile); /* patch address is 3 bytes */
#ifdef LDEBUG
	    VERBOSE(erl_printf(COUT,"function relative patch address = %d\n",
			       patch_address););
#endif
	    if ((patch_type == PATCH_CSA) ||
		(patch_type == PATCH_CONST)) {
		*(last_function_code +  patch_address) = patch1;
		*(last_function_code +  patch_address + 1) = patch0;
		break;
	    }
	    if (patch_type == PATCH_APPLY) {
		if (*(last_function_code +  patch_address -2) ==
		    OP_enter_remote) 
		    *(last_function_code +  patch_address -2) = OP_apply_enter;
		else 
		    /* assume it is remote call */
		    *(last_function_code +  patch_address -2) = OP_apply_call;
		break;
	    }
	    if (patch_type == PATCH_F) {
		b = last_function_code + patch_address;
		float_to_bytes(b, float_patch);
		break;
	    }
	    if (patch_type == PATCH_BIF) {
		/*
		  BIF calls are of the form kernel:bif_name(......)
		  We have already checked that it is a BIF so we now
		  just have to patch the BIF index (in patch) and
		  munge the opcode 
		 */
		*(last_function_code +  patch_address) = patch1;
		*(last_function_code +  patch_address + 1) = patch0;
		if (*(last_function_code +  patch_address -2) ==
		    OP_enter_remote) 
		    *(last_function_code +  patch_address -2) = OP_bif_enter;
		else 
		    /* assume it is remote call */
		    *(last_function_code +  patch_address -2) = OP_bif_call;
		break;
	    }
	    /* must be a local function patch - work out offset  (3 bytes !)*/
	    offset_3 =  function_addr(fpatch) -
		function_addr(function_number) - patch_address + 1;
	    *(last_function_code +  patch_address + 2) = offset_3;
	    offset_3 = offset_3 >> 8;
	    *(last_function_code +  patch_address + 1) = offset_3;
	    offset_3 = offset_3 >> 8;
	    *(last_function_code +  patch_address) = offset_3;
	    break;
	case L_PATCH_CONSTANT:
	    field_length = b_get_string(&mfile, &str_p);
	    this_atom_no = atom_put((byte*)str_p, field_length);
#ifdef LDEBUG
	    VERBOSE({
		erl_printf(COUT,"Patching atom %s atom table pos %d\n",
			   copy_str(str_p, field_length), this_atom_no);
	    });
#endif
	    patch1 = (byte)  hi_byte(this_atom_no);
	    patch0 = (byte)  lo_byte(this_atom_no);
#ifdef LDEBUG
	    VERBOSE(erl_printf(COUT,"hi byte = %d, lo byte = %d\n", 
			       patch1, patch0););
#endif
	    patch_type = PATCH_CONST;
	    break;
	case L_PATCH_LOCAL:
	    field_length = b_get_string(&mfile,&str_p);
	    arity = GET8(&mfile); /* arity */

	    fpatch = fundef_get(str_p, field_length, arity);
	    patch_type = PATCH_LOC;
	    break;
	case L_PATCH_FLOAT:
	    field_length = b_get_string(&mfile, &str_p);
	    copy_str(str_p, field_length);
	    sys_chars_to_double(load_buf, &float_patch);
	    patch_type = PATCH_F;
	    break;
	case L_IMPORT:
	    field_length = b_get_string(&mfile, &str_p);
	    other_module = atom_put((byte*)str_p, field_length);
	    field_length = b_get_string(&mfile, &str_p);
	    function_atom_no = atom_put((byte*) str_p, field_length);
	    arity = GET8(&mfile);
	    if ((other_module == unsigned_val(am_erlang))
		&& (function_atom_no == unsigned_val(am_apply))
		&& (arity == 3)) {
		/* this is not an imported function, this an apply */
		patch_type = PATCH_APPLY;
		break;
	    }
	    if (other_module == unsigned_val(am_erlang)) {
		/* this might be a BIF */
		int findex;
		if ((findex = find_bif(function_atom_no, arity)) != -1) {
		    patch_type = PATCH_BIF;
		    patch1 = (byte) hi_byte(findex);
		    patch0 = (byte) lo_byte(findex);
#ifdef LDEBUG
		    VERBOSE({
			erl_printf(COUT,"BIF = %s/%d\n", 
				   copy_str(str_p, field_length), arity);
		    });
#endif
		    break;
		}
	    }
#ifdef LDEBUG
	    VERBOSE({
		erl_printf(COUT,"Import = %s/%d\n",
			   copy_str(str_p, field_length), arity);
	    });
#endif
	    /* OK, it was an imported function */
	    offset = (uint16) export_put(other_module,
					 function_atom_no, arity);
	    patch1 = (byte) hi_byte(offset);
	    patch0 = (byte) lo_byte(offset);
	    patch_type = PATCH_CSA;
	    break;
	case END_OF_FILE:
	    done = 1;
	    break;
	}
	
    }
    if (display_loads) {
	erl_printf(COUT,"Loaded module %s, ", module_name);
	erl_printf(COUT,"Functions %d, ", max_functions);
	erl_printf(COUT,"Exported %d, ", exports);
	erl_printf(COUT,"Code size %d\r\n", module_length);
    }
    return bin_load_done(module_atom_no);
}

/* Cleaup all code & relase fix allocator */
static void atexit_load(arg)
void* arg;
{
    int i;
#if !defined(NO_FIX_ALLOC)
    /* Does not work if NO_FIX_ALLOC is defined */
    fix_release(fundef_desc);
#endif
    /* Module.c will release the module table (but not code) */
    for (i = 0; i < module_code_size; i++) {
	Module* m = module_code(i);
	if (m->old_code != NULL) {
	    sys_free(m->old_code);
	    m->old_code = NULL;
	    m->old_code_length = 0;
	}
	if (m->code != NULL) {
	    sys_free(m->code);
	    m->code = NULL;
	    m->code_length = 0;
	}
    }
#if defined(NO_FIX_ALLOC)
    /* This actually frees what we need to free without using fix_release */
    bin_load_done(0);
#endif
}


void init_load()
{
    int i;
    Preload* pre;
    uint32 module_name;
    byte* code;
    char* name;
    int length;
    
    fundef_desc = new_fix_size(sizeof(FunDef));

    erl_at_exit(atexit_load, NULL);

    if ((pre = sys_preloaded()) == NULL)
	return;

    i = 0;
    while((name = pre[i].name) != NULL) {
	if ((code = sys_preload_begin(&pre[i])) == 0)
	    erl_exit(1, "Failed to find preloaded code for module %s\n", name);
	length = pre[i].size;
	module_name = am_atom_put(name, sys_strlen(name));
	if (do_load(0, module_name, code, length) < 0)
	    erl_exit(1,"Failed loading preloaded module %s\n", name);
	sys_preload_end(&pre[i]);
	i++;
    }
}



/* Move code from current to old */
static void delete_code(mi)
int mi;
{
    module_code(mi)->old_code_length = module_code(mi)->code_length;
    module_code(mi)->old_code = module_code(mi)->code;
    module_code(mi)->code_length = 0;
    module_code(mi)->code = NULL;
}

/* null all references on the export table for the module called with the
   atom index below */

static void delete_export_references(module)
int module;
{
    int i;

    for (i = 0; i < export_list_size; i++) {
        if (export_list(i) != NULL && (export_list(i)->module == module))
	    export_list(i)->address = (byte*) 0;
    }
}

/* delete module does not delete the module,
   what it does is to delete the export refences to the module
   and move the code to the old code area.
   If you wish to delete a module you must first do delete module
   and then purge_module (with the appropriate check_process_code
   calls
   */


int delete_module(group_leader, module)
uint32 group_leader;
int module;
{
    int mod_index;

    if ((mod_index = module_get(module)) == -1)
	return 1; /* not found */

    /* must be purged first before deleteing */
    if (module_code(mod_index)->old_code != NULL) {
	erl_printf(CBUF, "Module ");
	print_atom(module, CBUF);
	erl_printf(CBUF, " must be purged before deleting\n");
	send_error_to_logger(group_leader);
	return 2;
    }

    delete_export_references(module);
    delete_code(mod_index);

    return 0;
}


/*
  Check that a process is not executing (or has return value to)
  code in a replaced module. Returns < 0 for errors, 0 if the
  process is OK or 1 if the process is executing old code
  */

/* NOTE argument cp is used in BEAM ! */
int check_process_code(cp, pid, module)
Process* cp; uint32 pid; int module; 
{
    Process *p;
    int mod_index;
    byte* start;
    byte* end;
    byte* pc;
    uint32 *fp;

    /* check that pid is a valid process */
    if (is_not_pid(pid) || (get_node(pid) != THIS_NODE) ||
	(get_number(pid) >= max_process))
	return(-1);
    p = process_tab[get_number(pid)];
    if (INVALID_PID(p, pid))
	return(0);

    /* correct module ? */
    if ((mod_index = module_get(module)) == -1)
	return 0;

    /* any code to purge ? */
    if (module_code(mod_index)->old_code == (byte*) 0)
	return 0;

    /* Now check the processes stack */
    start = module_code(mod_index)->old_code;
    end   = module_code(mod_index)->old_code +
	module_code(mod_index)->old_code_length;

    fp = p->fp;
    while(fp != p->stack) {
	pc = (byte*) fp[FRAME_PC];
	if ((pc >= start) && (pc <= end)) 
	    return 1;
	fp = frame_val(fp[FRAME_FP]);
    }
    /* check the PC and the last_pc of the process */
    if ((p->pc >= start) && (p->pc <= end))
	return 1;
    return 0;
}

int purge_module(module)
int module;
{
    int mod_index;

   /* correct module ? */
    if ((mod_index = module_get(module)) == -1)
	return -2;

    /* any code to purge ? */
    if (module_code(mod_index)->old_code == NULL) {
	if (display_loads) {
	    erl_printf(COUT,"No code to purge for ");
	    print_atom(module, COUT);
	    erl_printf(COUT,"\n");
	}
	return -1;
    }
    if (display_loads) {
	erl_printf(COUT,"Purging code for ");
	print_atom(module,COUT);
	erl_printf(COUT,"\n");
    }
    sys_free((char *) module_code(mod_index)->old_code);
    module_code(mod_index)->old_code = (byte*) 0;
    module_code(mod_index)->old_code_length = 0;
    return 0;
}
