/* ``The contents of this file are subject to the Erlang Public License,
 * Version 1.0, (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.erlang.org/EPL1_0.txt
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 * 
 * The Original Code is Erlang-4.7.3, December, 1998.
 * 
 * The Initial Developer of the Original Code is Ericsson Telecom
 * AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
 * Telecom AB. All Rights Reserved.
 * 
 * Contributor(s): ______________________________________.''
 */
/*
 * This file is copyright (c) Ellemtel in January 1991
 * 
 * Author Mike Williams
 * 
 * Modified by Tony Rogvall 950101 - 950301
 *  Can use GCC jump table if compiled with WANT_JUMP_TABLE
 *  The dispatcher is macrofied.
 */

#define WANT_JUMP_TABLE

#include "sys.h"
#include "config.h"
#include "global.h"
#include "erl_message.h"
#include "erl_process.h"
#include "error.h"
#include "big.h"
#include "bif.h"
#include "dist.h"
#include "jam_opcodes.h"

/* #define HARDDEBUG */

#define SWAPIN \
    htop = p->htop; \
    stop = p->stop; \
    ap = p->ap; \
    fp = p->fp; \
    pc = p->pc; \
    cc = p->cc

#define SWAPOUT \
    p->htop = htop; \
    p->stop = stop; \
    p->ap = ap; \
    p->fp = fp; \
    p->pc = pc; \
    p->cc = cc

/* Failure */

#define FAIL          { goto L_fail; }
#define FAIL1(Reason) { tmp = (Reason); goto L_fail1; }
#define EXIT(Reason)  { fail_pc = NULL; fail_reason = (Reason); goto L_exit; }

#if defined(__GNUC__) && defined(WANT_JUMP_TABLE) && !defined(QUANTIFY)

#define DECL_IREG             register const void* ireg
#define FETCH(n)              ireg = jmptabp[*(pc+(n))];  pc += ((n)+1)
#define DISPATCH_BEGIN()      goto *ireg
#define DISPATCH_END
#define CASE(op)              L_##op
#define CASE_DEFAULT          L_default


#if defined(HARDDEBUG) || defined(OPPROF) || defined(OPTRACE)

#define LBL_NEXT              L_next:
#define NEXT                  goto L_next

#else

#define LBL_NEXT
#define NEXT                  goto *ireg

#endif


#else

#define DECL_IREG             register uint32 ireg
#define FETCH(n)              ireg = *(pc+(n));  pc += ((n)+1)
#define DISPATCH_BEGIN()      switch(ireg) {
#define DISPATCH_END          }
#define CASE(op)              case op
#define CASE_DEFAULT          default

#define LBL_NEXT              L_next:
#define NEXT                  goto L_next

#endif


#define NTH(i)  *(stop - (i))
#define VAR(i)  fp[FRAME_SIZE+(i)]
#define ARG(i)  ap[(i)]

#define FRAME_SIZE 4

#define LINK(Len, Arity) \
    fcalls++; \
    stop[FRAME_FP] = make_frame(fp); \
    stop[FRAME_AP] = make_frame(ap); \
    stop[FRAME_PC] = (uint32) (pc+(Len)); \
    stop[FRAME_CC] = (uint32) cc; \
    fp = stop; \
    ap = stop - (Arity); \
    stop += FRAME_SIZE

#define UNLINK() \
    tmp = NTH(1); \
    stop = ap; \
    cc = (byte*) fp[FRAME_CC]; \
    pc = (byte*) fp[FRAME_PC]; \
    ap = frame_val(fp[FRAME_AP]); \
    fp = frame_val(fp[FRAME_FP]); \
    *stop++ = tmp; \
    if (cc == NULL) { EXIT(NORMAL); }


#define ELINK(Arity) do { \
    uint32 __tmp_fp = fp[FRAME_FP]; \
    uint32 __tmp_ap = fp[FRAME_AP]; \
    uint32 __tmp_pc = fp[FRAME_PC]; \
    uint32 __tmp_cc = fp[FRAME_CC]; \
    uint32 __n = (Arity); \
    fcalls++; \
    stop -= __n; \
    while(__n--) *ap++ = *stop++; \
    if (ap != fp) { \
        fp = ap; \
        fp[FRAME_FP] = __tmp_fp; \
        fp[FRAME_AP] = __tmp_ap; \
        fp[FRAME_PC] = __tmp_pc; \
        fp[FRAME_CC] = __tmp_cc; \
    } \
    stop = fp + FRAME_SIZE; \
    ap -= (Arity); \
    cc = NULL; \
} while(0)

#ifdef HARDDEBUG
#define H_NEED(x) { \
    if ((p->heap_margin - htop) < 0) { \
	display(p->id, CERR); \
	erl_printf(CERR, " Overrun heap (emul), words=%d, panic\n", \
		   (p->heap_margin - htop)); \
	erl_exit(1, " Overrun heap (emul), words=%d, panic\n", (p->heap_margin - htop)); \
    } \
    if ((htop + (x)) > (p->heap_margin)) {  \
        SWAPOUT; \
        VERBOSE(erl_printf(COUT,"<GC in process_main>");); \
        reds -= do_gc(p, x); \
        SWAPIN; \
    } }
#else
#define H_NEED(x) \
    if ((htop + (x)) > (p->heap_margin)) {  \
        SWAPOUT; \
        VERBOSE(erl_printf(COUT,"<GC in process_main>");); \
        reds -= do_gc(p, x); \
        SWAPIN; \
    }
#endif


#define S_NEED(x) \
    if ((stop + (x)) > (p->stack_margin)) { \
        SWAPOUT; \
        VERBOSE(erl_printf(COUT,"<stack increament in process_main>");); \
	stack_grow(p, (x)+S_MARGIN); \
        SWAPIN; \
    }

#define CHKSTK do { \
        if ((stop - p->stack) < p->stack_sz / 4) { \
            stack_shrink(p, S_MARGIN); \
        } \
    } while(0)
  
/*
** We may add heap compactation
**
** #define CHKHEAP do { \
**	heap_compact(p, H_MARGIN); \
**    } while(0)
**
** No we check if we need to gc 
*/
#define CHKHEAP do { \
    if (p->flags & F_NEED_GC) \
	do_gc(p, 0); \
    } while(0)

#define CHKREDS \
    if ((--reds <= 0) && (!(p->flags & F_DONT_PRE_EMPT))) { \
        pc--; \
        SWAPOUT; \
	add_to_schedule_q(p); \
        return fcalls; \
    }


#define MATCH(x)   { FETCH(0); if (NTH(1) == (x)) { stop--; NEXT;} else FAIL; }
#define PUSH(x)    { FETCH(0); *stop++ = (x); NEXT; }
#define STORE(at)  { FETCH(0); (at) = *--stop; NEXT; }
#define EQ(x)      { FETCH(0); if (NTH(1) == (x)) { stop--; NEXT; } \
		     if (eq(x, NTH(1))) { stop--; NEXT; } else FAIL; }

#define MATCH1(x)  { FETCH(1); if (NTH(1) == (x)) { stop--; NEXT;} else FAIL; }
#define PUSH1(x)   { FETCH(1); *stop++ = (x); NEXT; }
#define STORE1(at) { FETCH(1); (at) = *--stop; NEXT; }
#define EQ1(x)     { FETCH(1); if (NTH(1) == (x)) { stop--; NEXT; } \
		     if (eq(x, NTH(1))) { stop--; NEXT; } else FAIL; }

#define RELOAD(k, arg1, arg2) \
   if ((k) & 1) arg1 = NTH(1); \
   if ((k) & 2) arg2 = NTH(2)


#define small_plus(x, y)  ((x)+(y))
#define small_minus(x,y)  ((x)-(y))
#define small_times(x,y)  ((x)*(y))
#define small_div(x,y)    ((x)/(y))
#define small_rem(x,y)    ((x)%(y))
#define small_band(x,y)   ((x)&(y))
#define small_bor(x,y)    ((x)|(y))
#define small_bxor(x,y)   ((x)^(y))
#define small_bsl(x,y)    ((x)<<(y))
#define small_bsr(x,y)    ((x)>>(y))
#define small_bnot(x)     ~(x)
#define small_neg(x)      -(x)
#define small_comp(x,y)   ((x)-(y))


#define float_plus(x,y)    ((x)+(y))
#define float_minus(x,y)   ((x)-(y))
#define float_times(x,y)   ((x)*(y))
#define float_div(x,y)     ((x)/(y))
#define float_neg(x)       -(x)
#define float_comp(x,y)    (((x)-(y)) < 0.0 ? -1 : (((x)==(y)) ? 0 : 1))


#define BAND_OP 0
#define BOR_OP  1
#define BXOR_OP 2

#define GT_OP     0
#define LT_OP     1
#define GEQ_OP    2
#define LEQ_OP    3
#define EQEQ_OP   4
#define NEQ_OP    5
#define PLUS_OP   6
#define MINUS_OP  7
#define TIMES_OP  8
#define DIVIDE_OP 9

#define MAX_NEED(x,y) (((x)>(y)) ? (x) : (y))


/*
** Type values map for the type op
** 1-10
*/
int typeval[16] = {
    -1,          /* FRAME_DEF */
    1,           /* SMALL_DEF */
    1,           /* BIG_DEF */
    2,           /* FLOAT_DEF */
    3,           /* ATOM_DEF */
    4,           /* REFER_DEF */
    5,           /* PORT_DEF */
    6,           /* PID_DEF */
    7,           /* TUPLE_DEF */
    8,           /* NIL_DEF */
    9,           /* LIST_DEF */
    -1,          /* ARITYVAL_DEF */
    -1,          /* MOVED_DEF (only when GC) */
    -1,          /* CATCH_DEF */
    -1,          /* THING_DEF */
    10           /* BINARY_DEF */
};

extern void trace_bif();  /* multi arguments */

DECL_OP_TABLE

#ifdef HARDDEBUG
static void schk(p, top)
Process *p; uint32 *top;
{
    uint32 *ptr = p->stack;

    while (ptr < top) {
	if (is_frame(ptr[0])) {
	    if (is_not_frame(ptr[1])) {
		erl_printf(COUT,"Bad stack frame at %08lx for ", ptr);
		display(p->id, COUT);
		erl_printf(COUT, "\nstack printout follows:\n");
		ps(p, top);
		abort();
	    }
	    ptr += FRAME_SIZE;
	}
	else if (is_catch(ptr[0]))
	    ptr += CATCH_SIZE;
	else
	    ptr++;
    }
}
#endif

#ifdef OPTRACE
int trace_on = 0;

/*
** Instruction trace output
*/
void optrace(f, pc)
CIO f; byte* pc;
{
    int op_code = *pc;
    int i;

    erl_printf(f, "%08x : %s ", pc, opc[op_code].name);
    switch(op_code) {
    case OP_pushVarN:     /* u 1 */
    case OP_storeVarN:    /* u 1 */
    case OP_allocN:       /* u 1 */
    case OP_eqVarN:       /* u 1 */
    case OP_eqArgN:       /* u 1 */
    case OP_argN:         /* u 1 */
    case OP_unpkTupleN:   /* u 1 */
    case OP_mkTupleN:     /* u 1 */
    case OP_getInt1:      /* u 1 */
    case OP_pushInt1:     /* u 1 */
	erl_printf(f, "%d", *(pc+1));
	break;
    case OP_getAtom:      /* u 2 */
    case OP_pushAtom:     /* u 2 */
    case OP_heap_need:    /* u 2 */
    case OP_stack_need:   /* u 2 */
    case OP_hash:         /* u 2 */
	erl_printf(f, "%d", (uint16) make_16(*(pc+1), *(pc + 2)));
	break;
    case OP_type:         /* u 2 */
	erl_printf(f, "%04x", (uint16) make_16(*(pc+1), *(pc + 2)));
	break;
    case OP_try_me_else: /* s 3 */
    case OP_goto:        /* s 3 */
    case OP_call_local:  /* s 3 */
    case OP_enter_local: /* s 3 */
    case OP_pushCatch:   /* s 3 */
    case OP_wait1:       /* s 3 */
	erl_printf(f, "%d", make_signed_24(*(pc+1),*(pc+2),*(pc+3)));
	break;
    case OP_getInt4:       /* s 4 */
    case OP_pushInt4:      /* s 4 */
	erl_printf(f, "%d", make_signed_32(*(pc+1),*(pc+2),*(pc+3),*(pc+4)));
	break;
    case OP_getFloat:     /* b 8 */
	erl_printf(f, "%.20le", bytes_to_float(pc+1));
	break;
    case OP_pushFloat:    /* b 8 */
	erl_printf(f, "%.20le", bytes_to_float(pc+1));
	break;
    case OP_nodebug_info: /* b 4 */
	break;
    case OP_debug_info:   /* b 4 */
	break;
    case OP_call_remote:   /* u 1, u 2 */
    case OP_enter_remote:  /* u 1, u 2 */
    case OP_bif_enter:     /* u 1, u 2 */
    case OP_bif_call:      /* u 1, u 2 */
	erl_printf(f, "%d, %d", *(pc+1), make_16(*(pc+2), *(pc+3)));
	break;
    case OP_pushIntN:
    case OP_getIntN:
	i = make_32(*(pc+1),*(pc+2),*(pc+3),*(pc+4));
	pc += 4;
	erl_printf(f, "%d %c", i, (*pc)?'-':'+');
	pc++;
	while(i--)
	    erl_printf(f, "%02x", (*pc++));
	break;
    case OP_pushStr:
    case OP_getStr:
	i = make_16(*(pc+1), *(pc+2));
	erl_printf(f, "%d: ", i);
	pc += 3;
	while(i--)
	    erl_putc(*pc++, f);
	break;
    case OP_gotoix:
	i = make_16(*(pc+1),*(pc+2));
	erl_printf(f, "%d %d : ", i, make_16(*(pc+3),*(pc+4)));
	pc += 5;
	while(i--) {
	    erl_printf(f, "%ld ", make_signed_24(*pc,*(pc+1),*(pc+2)));
	    pc += 3;
	}
	break;
    }
    erl_putc('\n', f);
}
#endif

static uint32  tmp_big1[2];
static uint32  tmp_big2[2];

/* since pc is no longer coded on frame in JAM 
** it is OK to allocate jam_apply & jam_exit in process global storage
*/
byte jam_apply[5];
byte jam_exit[1];

void init_emulator()
{
    jam_apply[0] = OP_apply_call;
    jam_apply[1] = 0;
    jam_apply[2] = 0;
    jam_apply[3] = 0;
    jam_apply[4] = OP_ret;
    jam_exit[0] = OP_die;
}


static void bif_error(p, index, inguard, arity)
Process* p; int index; int inguard; int arity;
{
    /* the BIF has failed we do not wish to print and error if:
       It was a exit/1 (p->freason = USER_EXIT)
       It was in a guard (fail != 0)
       It was a throw (p->freason = THROWN)
       It was within a catch (catches != ENULL)
       */
    if ((p->freason == USER_EXIT) || (p->freason == THROWN)) {
	*(p->stop - arity) = p->fvalue;  /* Return error code */
	p->fvalue = NIL;
	p->stop = p->stop - arity + 1;   /* Update stack */
    }
    else if (!inguard && (p->catches == ENULL)) {
	display(p->id, CBUF);
	erl_printf(CBUF, "Error in BIF %s/%d(",
		   bif_name(index), arity);
	switch(arity) {
	case 0:
	    break;
	case 1:
	    ldisplay(*(p->stop-1), CBUF, display_items);
	    break;
	case 2:
	    ldisplay(*(p->stop-2), CBUF, display_items);
	    erl_printf(CBUF,",");
	    ldisplay(*(p->stop-1), CBUF, display_items);
	    break;
	case 3:
	    ldisplay(*(p->stop-3), CBUF, display_items);
	    erl_printf(CBUF,",");
	    ldisplay(*(p->stop-2), CBUF, display_items);
	    erl_printf(CBUF,",");
	    ldisplay(*(p->stop-1), CBUF, display_items);
	    break;
	}
	erl_printf(CBUF,")\n");
	send_error_to_logger(p->group_leader);
    }
}

/*
** Emulator loop
** Reductions is either function call, gc-call, send, receive ...
** Function calls are calculated separatly as number of real function calls
*/

int process_main(p, reds)
register Process *p; int reds;
{
    DECL_JMPTAB
    DECL_IREG;
    register byte  *pc;
    register uint32 *stop;
    register uint32 *htop;
    register uint32 *ap;
    register uint32 *fp;
    int      inguard;
    int      fcalls;       /* number of function calls */
    byte*    cc;           /* current call */
    byte*    resched_pc;   /* set to pc when reschedule bif */
    uint32*  resched_sp;
    uint32*  fail_stack;
    byte*    fail_pc;
    uint32   fail_reason;
    uint32   tmp;
    uint32*  tp;
    byte*    tb;
    uint32   res;
    FloatDef f1;
    FloatDef f2;
    uint32   barg1, barg2;
    digit_t  sz1, sz2, sz;
    int      i, j, k, need_heap;
    sint32   iarg, ires;
    uint32   lastpos;		/* Used by OP_apply */
    uint32   function;		/* Used by OP_apply */

    /* Default reason after context switch to BADMATCH 
       Since a context switch can only occur before
       a function call. */

    fail_reason = BADMATCH;
    fail_pc = NULL;
    inguard = 0;
    fail_stack = 0;
    fcalls = 0;

    SWAPIN;
    FETCH(0);    /* fetch first instruction */

 LBL_NEXT;

#ifdef HARDDEBUG
    if ((p->heap_margin - htop) < 0) {
	display(p->id, CERR);
	erl_printf(CERR, " Overrun heap (emul), words=%d, panic\n",
		   (p->heap_margin - htop));
	erl_exit(1, " Overrun heap (emul), words=%d, panic\n", (p->heap_margin - htop));
    }
    tmp = p->stack_sz;
    /* if (((stop - p->stack_margin) < -S_MARGIN) || */
    if ((stop > (p->stack + tmp)) ||
	p->stack[tmp] != S_PATTERN ||
	p->stack[tmp+1] != S_PATTERN) {
	display(p->id, CERR);
	erl_printf(CERR, " Overrun stack (emul), word=%d, panic\n",
		   (p->stack_margin-stop));
	erl_exit(1, " Overrun stack (emul), word=%d, panic\n", (p->stack_margin-stop));
    }
    schk(p, stop);
#endif
#ifdef OPPROF
    opc[*(pc-1)].count++;
#endif
#ifdef OPTRACE
    if (trace_on)
	optrace(COUT, pc-1);
#endif
    DISPATCH_BEGIN();

    /****************************************************
      1) THE MOST COMMON INSTRUCTIONS ARE STORED HERE 
     *****************************************************/

    CASE(OP_getNil):  MATCH(NIL);
    CASE(OP_pushNil): PUSH(NIL);

    CASE(OP_getInt_0):    MATCH(make_small(0));
    CASE(OP_getInt_1):    MATCH(make_small(1));
    CASE(OP_getInt_2):    MATCH(make_small(2));
    CASE(OP_getInt_3):    MATCH(make_small(3));
    CASE(OP_getInt_4):    MATCH(make_small(4));
    CASE(OP_getInt_5):    MATCH(make_small(5));
    CASE(OP_getInt_6):    MATCH(make_small(6));
    CASE(OP_getInt_7):    MATCH(make_small(7));
    CASE(OP_getInt_8):    MATCH(make_small(8));
    CASE(OP_getInt_9):    MATCH(make_small(9));
    CASE(OP_getInt_10):   MATCH(make_small(10));
    CASE(OP_getInt_11):   MATCH(make_small(11));
    CASE(OP_getInt_12):   MATCH(make_small(12));
    CASE(OP_getInt_13):   MATCH(make_small(13));
    CASE(OP_getInt_14):   MATCH(make_small(14));
    CASE(OP_getInt_15):   MATCH(make_small(15));
    CASE(OP_getInt1):  { tmp = *pc; MATCH1(make_small(tmp)); }

    CASE(OP_pushInt_0):   PUSH(make_small(0));
    CASE(OP_pushInt_1):   PUSH(make_small(1));
    CASE(OP_pushInt_2):   PUSH(make_small(2));
    CASE(OP_pushInt_3):   PUSH(make_small(3));
    CASE(OP_pushInt_4):   PUSH(make_small(4));
    CASE(OP_pushInt_5):   PUSH(make_small(5));
    CASE(OP_pushInt_6):   PUSH(make_small(6));
    CASE(OP_pushInt_7):   PUSH(make_small(7));
    CASE(OP_pushInt_8):   PUSH(make_small(8));
    CASE(OP_pushInt_9):   PUSH(make_small(9));
    CASE(OP_pushInt_10):  PUSH(make_small(10));
    CASE(OP_pushInt_11):  PUSH(make_small(11));
    CASE(OP_pushInt_12):  PUSH(make_small(12));
    CASE(OP_pushInt_13):  PUSH(make_small(13));
    CASE(OP_pushInt_14):  PUSH(make_small(14));
    CASE(OP_pushInt_15):  PUSH(make_small(15));
    CASE(OP_pushInt1): { tmp = *pc; PUSH1(make_small(tmp)); }

    CASE(OP_pushVar_0):   PUSH(VAR(0));
    CASE(OP_pushVar_1):   PUSH(VAR(1));
    CASE(OP_pushVar_2):   PUSH(VAR(2));
    CASE(OP_pushVar_3):   PUSH(VAR(3));
    CASE(OP_pushVar_4):   PUSH(VAR(4));
    CASE(OP_pushVar_5):   PUSH(VAR(5));
    CASE(OP_pushVar_6):   PUSH(VAR(6));
    CASE(OP_pushVar_7):   PUSH(VAR(7));

    CASE(OP_storeVar_0):  STORE(VAR(0));
    CASE(OP_storeVar_1):  STORE(VAR(1));
    CASE(OP_storeVar_2):  STORE(VAR(2));
    CASE(OP_storeVar_3):  STORE(VAR(3));
    CASE(OP_storeVar_4):  STORE(VAR(4));
    CASE(OP_storeVar_5):  STORE(VAR(5));
    CASE(OP_storeVar_6):  STORE(VAR(6));
    CASE(OP_storeVar_7):  STORE(VAR(7));

    CASE(OP_eqVar_0):  EQ(VAR(0));
    CASE(OP_eqVar_1):  EQ(VAR(1));
    CASE(OP_eqVar_2):  EQ(VAR(2));
    CASE(OP_eqVar_3):  EQ(VAR(3));
    CASE(OP_eqVar_4):  EQ(VAR(4));
    CASE(OP_eqVar_5):  EQ(VAR(5));
    CASE(OP_eqVar_6):  EQ(VAR(6));
    CASE(OP_eqVar_7):  EQ(VAR(7));

    CASE(OP_eqArg_0):  EQ(ARG(0));
    CASE(OP_eqArg_1):  EQ(ARG(1));
    CASE(OP_eqArg_2):  EQ(ARG(2));
    CASE(OP_eqArg_3):  EQ(ARG(3));
    CASE(OP_eqArg_4):  EQ(ARG(4));
    CASE(OP_eqArg_5):  EQ(ARG(5));
    CASE(OP_eqArg_6):  EQ(ARG(6));
    CASE(OP_eqArg_7):  EQ(ARG(7));

    CASE(OP_arg_0):    PUSH(ARG(0));
    CASE(OP_arg_1):    PUSH(ARG(1));
    CASE(OP_arg_2):    PUSH(ARG(2));
    CASE(OP_arg_3):    PUSH(ARG(3));
    CASE(OP_arg_4):    PUSH(ARG(4));
    CASE(OP_arg_5):    PUSH(ARG(5));
    CASE(OP_arg_6):    PUSH(ARG(6));
    CASE(OP_arg_7):    PUSH(ARG(7));

    CASE(OP_unpkTuple_0): {
	FETCH(0);
	if (is_not_tuple(NTH(1))) FAIL;
	tp = ptr_val(NTH(1));
	if (*tp != make_arityval(0)) FAIL;
	stop--;
	NEXT;
    }

    CASE(OP_unpkTuple_1):
    {
	FETCH(0);
	if (is_not_tuple(NTH(1))) FAIL;
	tp = ptr_val(NTH(1));
	if (*tp != make_arityval(1)) FAIL;
	*(stop-1) = tp[1];
	NEXT;
    }

    CASE(OP_unpkTuple_2): 
    {
	FETCH(0);
	if (is_not_tuple(NTH(1))) FAIL;
	tp = ptr_val(NTH(1));
	if (*tp != make_arityval(2)) FAIL;
	*(stop-1) = tp[2];
	*(stop-0) = tp[1];
	stop += 1;
	NEXT;
    }

    CASE(OP_unpkTuple_3):  {
	FETCH(0);
	if (is_not_tuple(NTH(1))) FAIL;
	tp = ptr_val(NTH(1));
	if (*tp != make_arityval(3)) FAIL;
	*(stop-1) = tp[3];
	*(stop-0) = tp[2];
	*(stop+1) = tp[1];
	stop += 2;
	NEXT;
    }

    CASE(OP_unpkTuple_4): {
	FETCH(0);
	if (is_not_tuple(NTH(1))) FAIL;
	tp = ptr_val(NTH(1));
	if (*tp != make_arityval(4)) FAIL;
	*(stop-1) = tp[4];
	*(stop-0) = tp[3];
	*(stop+1) = tp[2];
	*(stop+2) = tp[1];
	stop += 3;
	NEXT;
    }

    CASE(OP_unpkTuple_5):  {
	FETCH(0);
	if (is_not_tuple(NTH(1))) FAIL;
	tp = ptr_val(NTH(1));
	if (*tp != make_arityval(5)) FAIL;
	*(stop-1) = tp[5];
	*(stop-0) = tp[4];
	*(stop+1) = tp[3];
	*(stop+2) = tp[2];
	*(stop+3) = tp[1];
	stop += 4;
	NEXT;
    }

    CASE(OP_mkTuple_0): {
	FETCH(0);
	H_NEED(1);
	*stop++ = TUPLE0(htop);
	htop++;
	NEXT;
    }

    CASE(OP_mkTuple_1): {
	FETCH(0);
	H_NEED(2);
	NTH(1) = TUPLE1(htop, NTH(1));
	htop += 2;
	NEXT;
    }

    CASE(OP_mkTuple_2): {
	FETCH(0);
	H_NEED(3);
	NTH(2) = TUPLE2(htop, NTH(2), NTH(1));
	htop += 3;
	stop -= 1;
	NEXT;
    }

    CASE(OP_mkTuple_3):  {
	FETCH(0);
	H_NEED(4);
	NTH(3) = TUPLE3(htop, NTH(3), NTH(2), NTH(1));
	stop -= 2;
	htop += 4;
	NEXT;
    }

    CASE(OP_mkTuple_4):  {
	FETCH(0);
	H_NEED(5);
	NTH(4) = TUPLE4(htop, NTH(4), NTH(3), NTH(2), NTH(1));
	stop -= 3;
	htop += 5;
	NEXT;
    }

    CASE(OP_mkTuple_5):  {
	FETCH(0);
	H_NEED(6);
	NTH(5) = TUPLE5(htop, NTH(5), NTH(4), NTH(3), NTH(2), NTH(1));
	stop -= 4;
	htop += 6;
	NEXT;
    }

    CASE(OP_mkList): {
	FETCH(0);
	H_NEED(2);
	NTH(2) = CONS(htop, NTH(1), NTH(2)),
	htop += 2;
	stop--;
	NEXT;
    }

    CASE(OP_unpkList): {
	FETCH(0);
	if (is_not_list(NTH(1))) FAIL;
	tp = ptr_val(NTH(1));
	NTH(1) = CDR(tp);
	*stop++ = CAR(tp);
	NEXT;
    }

    CASE(OP_try_me_else): {
	fail_pc = pc;
	inguard = 1;
	FETCH(3);
	fail_stack = stop;
	NEXT;
    }

    CASE(OP_try_me_else_fail): {
	FETCH(0);
	fail_reason = FUNCTION_CLAUSE;
	fail_pc = NULL;
	inguard = 1;
	NEXT;
    }

    CASE(OP_goto): {
	iarg = make_signed_24(pc[0],pc[1],pc[2]);
	FETCH(iarg);
	NEXT;
    }

    CASE(OP_commit): {
	FETCH(0);
	fail_reason = BADMATCH;
	fail_pc = NULL;
	inguard = 0;
	NEXT;
    }

    CASE(OP_popCommit): {
	FETCH(0);
	stop--;
	fail_pc = NULL;
	inguard = 0;
	NEXT;
    }

    CASE(OP_pop): {
	FETCH(0);
	stop--;
	NEXT;
    }

    CASE(OP_dup): {
	FETCH(0);
	*stop = NTH(1);
	stop++;
	NEXT;
    }

    CASE(OP_alloc_0): {
	FETCH(0);
	NEXT;
    }

    CASE(OP_alloc_1):  { FETCH(0); tmp = 1; goto L_alloc; }
    CASE(OP_alloc_2):  { FETCH(0); tmp = 2; goto L_alloc; }
    CASE(OP_alloc_3):  { FETCH(0); tmp = 3; goto L_alloc; }
    CASE(OP_alloc_4):  { FETCH(0); tmp = 4; goto L_alloc; }
    CASE(OP_alloc_5):  { FETCH(0); tmp = 5; goto L_alloc; }
    CASE(OP_alloc_6):  { FETCH(0); tmp = 6; goto L_alloc; }
    CASE(OP_alloc_7):  { FETCH(0); tmp = 7; goto L_alloc; }
    CASE(OP_alloc_8):  { FETCH(0); tmp = 8; goto L_alloc; }
    CASE(OP_alloc_9):  { FETCH(0); tmp = 9; goto L_alloc; }
    CASE(OP_alloc_10): { FETCH(0); tmp = 10; goto L_alloc; }
    CASE(OP_alloc_11): { FETCH(0); tmp = 11; goto L_alloc; }
    CASE(OP_alloc_12): { FETCH(0); tmp = 12; goto L_alloc; }
    CASE(OP_alloc_13): { FETCH(0); tmp = 13; goto L_alloc; }
    CASE(OP_alloc_14): { FETCH(0); tmp = 14; goto L_alloc; }
    CASE(OP_alloc_15): { FETCH(0); tmp = 15; goto L_alloc; }
    CASE(OP_allocN): {
	tmp = *pc;
	FETCH(1);

    L_alloc:
	tp = stop;
	S_NEED(tmp);
	fail_stack += (stop - tp);

	while(tmp > 0) {
	    k = (tmp > 15) ? 15 : tmp;
	    switch(k) {
	    case 15: stop[14] = NIL;
	    case 14: stop[13] = NIL;
	    case 13: stop[12] = NIL;
	    case 12: stop[11] = NIL;
	    case 11: stop[10] = NIL;
	    case 10: stop[9] = NIL;
	    case 9:  stop[8] = NIL;
	    case 8:  stop[7] = NIL;
	    case 7:  stop[6] = NIL;
	    case 6:  stop[5] = NIL;
	    case 5:  stop[4] = NIL;
	    case 4:  stop[3] = NIL;
	    case 3:  stop[2] = NIL;
	    case 2:  stop[1] = NIL;
	    case 1:  stop[0] = NIL;
	    }
	    stop += k;
	    tmp -= k;
	}
	NEXT;
    }

    CASE(OP_stack_need): {
	tb = pc;
	FETCH(2);
	i = (uint16) make_16(tb[0], tb[1]);
	tp = stop;
	S_NEED(i);
	/* In this case we may have a failure pointer to offset !!!! */
	fail_stack += (stop - tp);
	NEXT;
    }

    CASE(OP_call_local): {
	S_NEED(FRAME_SIZE);
	CHKREDS;
	iarg = make_signed_24(*(pc+1), *(pc+2), *(pc+3));
	LINK(4, *pc);
	FETCH(iarg);
	NEXT;
    }

    CASE(OP_enter_local): {
	S_NEED(FRAME_SIZE);
	CHKREDS;
	iarg = make_signed_24(*(pc+1), *(pc+2), *(pc+3));
	ELINK(*pc);
	FETCH(iarg);
	NEXT;
    }

    /* for old non debug compiled code */
    CASE(OP_nodebug_info): {
	tb = pc-1;
	FETCH(5);
	cc = tb;
	NEXT;
    }

    CASE(OP_ret): {
	UNLINK();

	/* This test is here in OP_ret since we don't have any more ops */
	/* It really should only be in (the yet undef) OP_debug_ret */

	if (IS_TRACED_FL(p, F_TRACE_CALLS | F_TIMESTAMP)) {
	    H_NEED(4);
	    tb = cc+1;
	    tmp = make_small(tb[0]);
	    tmp = TUPLE3(htop,
			 make_atom(make_16(tb[1], tb[2])),
			 make_atom(make_16(tb[3], tb[4])),
			 tmp);
	    htop += 4;
	    trace_proc(p, am_return, tmp);
	}
	FETCH(0);
	NEXT;
    }


    /****************************************************

      2) NOT SO COMMON CASES ARE PUT HERE

     *****************************************************/


    CASE(OP_heap_need): {
	tb = pc;
	FETCH(2);
	i = (uint16) make_16(tb[0], tb[1]);
	H_NEED(i);
	NEXT;
    }

    CASE(OP_self): {
	FETCH(0);
	*stop++ = p->id;
	NEXT;
    }

    CASE(OP_pushFloat): {
	tb = pc;
	FETCH(8);
	H_NEED(3);
	f1.fd = bytes_to_float(tb);
	*stop++ = make_float(htop);
	PUT_DOUBLE(f1, htop);
	htop += 3;
	NEXT;
    }

    CASE(OP_getFloat): {
	tb = pc;
	FETCH(8);
	if (is_not_float(NTH(1)))
	    FAIL;
	GET_DOUBLE(NTH(1), f1);
	f2.fd = bytes_to_float(tb);
	if (f1.fd != f2.fd)
	    FAIL;
	stop--;
	NEXT;
    }

    CASE(OP_pushVar_8):   PUSH(VAR(8));
    CASE(OP_pushVar_9):   PUSH(VAR(9));
    CASE(OP_pushVar_10):  PUSH(VAR(10));
    CASE(OP_pushVar_11):  PUSH(VAR(11));
    CASE(OP_pushVar_12):  PUSH(VAR(12));
    CASE(OP_pushVar_13):  PUSH(VAR(13));
    CASE(OP_pushVar_14):  PUSH(VAR(14));
    CASE(OP_pushVar_15):  PUSH(VAR(15));
    CASE(OP_pushVarN): { tmp = *pc; PUSH1(VAR(tmp)); }


    CASE(OP_storeVar_8):  STORE(VAR(8));
    CASE(OP_storeVar_9):  STORE(VAR(9));
    CASE(OP_storeVar_10): STORE(VAR(10));
    CASE(OP_storeVar_11): STORE(VAR(11));
    CASE(OP_storeVar_12): STORE(VAR(12));
    CASE(OP_storeVar_13): STORE(VAR(13));
    CASE(OP_storeVar_14): STORE(VAR(14));
    CASE(OP_storeVar_15): STORE(VAR(15));
    CASE(OP_storeVarN): { tmp = *pc; STORE1(VAR(tmp)); }


    CASE(OP_eqVar_8):  EQ(VAR(8));
    CASE(OP_eqVar_9):  EQ(VAR(9));
    CASE(OP_eqVar_10): EQ(VAR(10));
    CASE(OP_eqVar_11): EQ(VAR(11));
    CASE(OP_eqVar_12): EQ(VAR(12));
    CASE(OP_eqVar_13): EQ(VAR(13));
    CASE(OP_eqVar_14): EQ(VAR(14));
    CASE(OP_eqVar_15): EQ(VAR(15));
    CASE(OP_eqVarN): { tmp = *pc; EQ1(VAR(tmp)); }


    CASE(OP_eqArg_8):  EQ(ARG(8));
    CASE(OP_eqArg_9):  EQ(ARG(9));
    CASE(OP_eqArg_10): EQ(ARG(10));
    CASE(OP_eqArg_11): EQ(ARG(11));
    CASE(OP_eqArg_12): EQ(ARG(12));
    CASE(OP_eqArg_13): EQ(ARG(13));
    CASE(OP_eqArg_14): EQ(ARG(14));
    CASE(OP_eqArg_15): EQ(ARG(15));
    CASE(OP_eqArgN):   { tmp = *pc; EQ1(ARG(tmp)); }


    CASE(OP_arg_8):    PUSH(ARG(8));
    CASE(OP_arg_9):    PUSH(ARG(9));
    CASE(OP_arg_10):   PUSH(ARG(10));
    CASE(OP_arg_11):   PUSH(ARG(11));
    CASE(OP_arg_12):   PUSH(ARG(12));
    CASE(OP_arg_13):   PUSH(ARG(13));
    CASE(OP_arg_14):   PUSH(ARG(14));
    CASE(OP_arg_15):   PUSH(ARG(15));
    CASE(OP_argN):     { tmp = *pc; PUSH1(ARG(tmp)); }

    CASE(OP_unpkTuple_6):  {
	FETCH(0);
	tmp = 6; goto L_unpkTuple;
    }
    CASE(OP_unpkTuple_7):  {
	FETCH(0);
	tmp = 7; goto L_unpkTuple;
    }
    CASE(OP_unpkTuple_8): { 
	FETCH(0);
	tmp = 8; goto L_unpkTuple;
    }
    CASE(OP_unpkTuple_9):  {
	FETCH(0);
	tmp = 9; goto L_unpkTuple;
    }
    CASE(OP_unpkTuple_10): {
	FETCH(0);
	tmp = 10; goto L_unpkTuple;
    }
    CASE(OP_unpkTuple_11): { 
	FETCH(0);
	tmp = 11; goto L_unpkTuple;
    }
    CASE(OP_unpkTuple_12): { 
	FETCH(0);
	tmp = 12; goto L_unpkTuple;
    }
    CASE(OP_unpkTuple_13): { 
	FETCH(0);
	tmp = 13; goto L_unpkTuple;
    }
    CASE(OP_unpkTuple_14): { 
	FETCH(0);
	tmp = 14; goto L_unpkTuple;
    }
    CASE(OP_unpkTuple_15): { 
	FETCH(0);
	tmp = 15; goto L_unpkTuple;
    }

    CASE(OP_unpkTupleN): {
	tmp = *pc;
	FETCH(1);

    L_unpkTuple:
	if (is_not_tuple(NTH(1))) FAIL;
	tp = ptr_val(NTH(1));
	if (*tp != make_arityval(tmp)) FAIL;
	tp += tmp;
	stop--;
	while(tmp--)
	    *stop++ = *tp--;
	NEXT;
    }

    CASE(OP_mkTuple_6): {
	FETCH(0);
	tmp = 6; goto L_mkTuple;
    }
    CASE(OP_mkTuple_7): {
	FETCH(0); 
	tmp = 7; goto L_mkTuple;
    }
    CASE(OP_mkTuple_8): { 
	FETCH(0); 
	tmp = 8; goto L_mkTuple;
    }
    CASE(OP_mkTuple_9): {
	FETCH(0); 
	tmp = 9; goto L_mkTuple;
    }
    CASE(OP_mkTuple_10): { 
	FETCH(0); 
	tmp = 10; goto L_mkTuple;
    }
    CASE(OP_mkTuple_11): { 
	FETCH(0); 
	tmp = 11; goto L_mkTuple;
    }
    CASE(OP_mkTuple_12): { 
	FETCH(0); 
	tmp = 12; goto L_mkTuple;
    }
    CASE(OP_mkTuple_13): { 
	FETCH(0); 
	tmp = 13; goto L_mkTuple;
    }
    CASE(OP_mkTuple_14): { 
	FETCH(0); 
	tmp = 14; goto L_mkTuple;
    }
    CASE(OP_mkTuple_15): {
	FETCH(0); 
	tmp = 15; goto L_mkTuple;
    }
    CASE(OP_mkTupleN): { 
	tmp = *pc;
	FETCH(1);

    L_mkTuple:
	H_NEED(tmp+1);
	res = make_tuple(htop);
	*htop = make_arityval(tmp);
	tp = htop + tmp;
	htop = tp + 1;
	stop--;
	while(tmp--)
	    *tp-- = *stop--;
	stop++;
	*stop++ = res;
	NEXT;
    }

    CASE(OP_getInt4): {
	tb = pc;
	FETCH(4);
	if (is_not_small(NTH(1)))
	    FAIL;
	if (NTH(1) != make_small(make_signed_32(tb[0],tb[1],
						tb[2],tb[3])))
	    FAIL;
	stop--;
	NEXT;
    }

    CASE(OP_getIntN): {
	tb = pc;
	sz1 = make_32(tb[0],tb[1],tb[2],tb[3]); /* number of digits */
	FETCH(sz1+5);
	if (is_not_integer(NTH(1)))
	    FAIL;
	if (!bytes_eq_big(tb+5, sz1, tb[4], NTH(1)))
	    FAIL;
	stop--;
	NEXT;
    }

    CASE(OP_pushInt4): {
	tb = pc;
	FETCH(4);
	*stop++ = make_small(make_signed_32(tb[0],tb[1],
					    tb[2],tb[3]));
	NEXT;
    }

    CASE(OP_pushIntN): {
	tb = pc;
	sz1 = make_32(tb[0],tb[1],tb[2],tb[3]);
	FETCH(sz1+5);
	sz = (sz1+1)/2;  /* number of digits */
	need_heap = BIG_NEED_SIZE(sz);
	H_NEED(need_heap);
	*stop++ = res = bytes_to_big(tb+5, sz1, tb[4], htop);
	if (is_big(res))
	    htop += (big_arity(res)+1);
	NEXT;
    }

    CASE(OP_send): {
	fcalls++;
	SWAPOUT;
	res = send_2(p, NTH(2), NTH(1));
	SWAPIN;
	if (res == 0) {
	    if (p->freason == RESCHEDULE) {
		p->pc--;        /* Set pc for resend */
		return fcalls;  /* we are suspended !! return !! */
	    }
	    else if (p->freason == TRAP) {  /* Trap function */
		k = 2;                      /* set arity */
		pc = pc+1;	/* Pretend we did a FETCH, since
				   L_trap_call compensates for one. */
		goto L_trap_call;
	    }
	    FAIL1(p->freason);
	}
	else {
	    NTH(2) = res;
	    stop--;
	    FETCH(0);
	    NEXT;
	}
    }

    CASE(OP_pushAtom): {
	tb = pc;
	FETCH(2);
	*stop++ = make_atom(make_16(tb[0], tb[1]));
	NEXT;
    }

    CASE(OP_getAtom): {
	tb = pc;
	FETCH(2);
	if (NTH(1) != make_atom(make_16(tb[0], tb[1])))
	    FAIL;
	stop--;
	NEXT;
    }

    CASE(OP_pushStr): {
	tb = pc+2;
	k = make_16(pc[0], pc[1]);  /* Length of string */
	FETCH(k+2);

	H_NEED(2*k);
	p->htop = htop;
	*stop++ = buf_to_intlist(&p->htop, tb, k, NIL);
	htop = p->htop;
	NEXT;
    }

    CASE(OP_getStr): {
	tb = pc+2;
	k = make_16(pc[0], pc[1]);
	FETCH(k+2);

	tmp = NTH(1);

	while(k--) {
	    if (is_not_list(tmp)) FAIL;
	    if (!is_byte(CAR(ptr_val(tmp)))) FAIL;
	    if (unsigned_val(CAR(ptr_val(tmp))) != *tb++) FAIL;
	    tmp = CDR(ptr_val(tmp));
	}
	if (is_not_nil(tmp))
	    FAIL;
	stop--;
	NEXT;
    }

    CASE(OP_list_length): {
	FETCH(0);
	if ((i = list_length(NTH(1))) == -1) {
	    FAIL1(BADARG);
	}
	NTH(1) = make_small(i);
	NEXT;
    }

    CASE(OP_head): {
	FETCH(0);
	if (is_not_list(NTH(1))) { FAIL1(BADARG); }
	NTH(1) = CAR(ptr_val(NTH(1)));
	NEXT;
    }

    CASE(OP_tail): {
	FETCH(0);
	if (is_not_list(NTH(1))) { FAIL1(BADARG); }
	NTH(1) = CDR(ptr_val(NTH(1)));
	NEXT;
    }

    CASE(OP_hash): {
	tb = pc;
	FETCH(2);
	k = make_16(tb[0], tb[1]);
	tmp = make_hash(NTH(1), 0);
	NTH(1) = make_small(1+(tmp % k));
	NEXT;
    }

    CASE(OP_type): {
	tmp = make_16(pc[0], pc[1]);
	FETCH(2);
	k = typeval[tag_val_def(NTH(1))]; /* map to type value bit */
	/* Check type mask */
	if ((k < 1) || (k > 10) || (((1 << k) & tmp) == 0)) 
	    FAIL;
	/* The type number is depending on the mask, we is only
	   intressted in the ordinal from the types we are 
	   interested in (i.e a mask with only two types 
	   always return 1 or 2) */
	i = k-1;
	while (i > 0) {
	    if (((1 << i) & tmp) == 0)
		k--;
	    i--;
	}
	NTH(1) = make_small(k);
	NEXT;
    }

    CASE(OP_call_remote): {
	S_NEED(FRAME_SIZE);
	CHKREDS;
	k = *pc;  /* the arity */
	tmp = make_16(*(pc + 1), *(pc + 2));
	if (export_list(tmp)->address == (void*) 0) {
	    int function, module;

	    if ((module = export_list(tmp)->module) == -1) {
		erl_printf(CERR, "No import or export declaration\r\n");
		EXIT(INTERNAL_ERROR);
	    }
	    function = export_list(tmp)->function;
	    if ((i = find_function(p->error_handler,
				   am_undefined_function, 3)) < 0)
		EXIT(UNDEF);

	    need_heap = 2*k + 1;
	    H_NEED(need_heap);
	    if (k == 0)
		tmp = NIL;
	    else {
		tp = stop - k;
		tmp = make_list(htop);
		stop -= k;
		while (k--) {
		    *htop++ = *tp++;
		    *htop = make_list(htop + 1);
		    htop++;
		}
		*(htop - 1) = NIL;
	    }
	    *stop++ = make_atom(module);
	    *stop++ = make_atom(function);
	    *stop++ = tmp;
	    k = 3;	/* the undefined function handler has arity 3 */
	    tmp = i;
	}
	LINK(3, k);
	pc = (byte*) export_list(tmp)->address;
	FETCH(0);
	NEXT;
    }

    CASE(OP_enter_remote): {
	S_NEED(FRAME_SIZE);
	CHKREDS;
	k = *pc;   /* the arity */
	tmp = make_16(*(pc + 1), *(pc + 2));
	if (export_list(tmp)->address == (void*) 0) {
	    int function, module;

	    if ((module = export_list(tmp)->module) == -1) {
		erl_printf(CERR, "No import or export declaration\r\n");
		EXIT(INTERNAL_ERROR);
	    }
	    function = export_list(tmp)->function;
	    if ((i = find_function(p->error_handler,
				   am_undefined_function, 3)) < 0)
		EXIT(UNDEF);

	    need_heap = 2*k + 1;
	    H_NEED(need_heap);
	    if (k == 0)
		tmp = NIL;
	    else {
		tp = stop - k;
		tmp = make_list(htop);
		stop -= k;
		while (k--) {
		    *htop++ = *tp++;
		    *htop = make_list(htop + 1);
		    htop++;
		}
		*(htop - 1) = NIL;
	    }
	    *stop++ = make_atom(module);
	    *stop++ = make_atom(function);
	    *stop++ = tmp;
	    k = 3;	/* the undefined function handler has arity 3 */
	    tmp = i;
	}
	ELINK(k);
	pc = (byte*) export_list(tmp)->address;
	FETCH(0);
	NEXT;
    }

    CASE(OP_failIf): {
	EXIT(IF_CLAUSE); 
    }

    CASE(OP_failCase): {
	EXIT(CASE_CLAUSE); 
    }

    CASE(OP_failFunction): {
	EXIT(FUNCTION_CLAUSE);
    }

    CASE(OP_pushCatch): {
	tb = pc;
	FETCH(3);
	stop[CATCH_PTR] = make_catch(p->catches);
	stop[CATCH_PC]  = (uint32) (tb+make_signed_24(tb[0], tb[1], tb[2]));
	p->catches = stop;
	stop += CATCH_SIZE;
	NEXT;
    }

    CASE(OP_popCatch): {
	FETCH(0);
	if (is_not_catch(NTH(3))) {
	    erl_printf(CERR, "poping non existent catch\n");
	    EXIT(INTERNAL_ERROR);
	}
	p->catches = ptr_val(NTH(3));
	NTH(3) = NTH(1);
	stop -= CATCH_SIZE;
	NEXT;
    }

    CASE(OP_setTimeout): {
	FETCH(0);
	if (is_not_small(NTH(1))) {
	    if (NTH(1) != am_infinity) { EXIT(TIMEOUT_VALUE); }
	    stop--;  /* pop timer value */
	    NEXT;
	}

	/* Integer timeout, milliseconds */
	if (signed_val(NTH(1)) < 0) { EXIT(TIMEOUT_VALUE); }
	set_timer(p, signed_val(NTH(1)));
	stop--;  /* pop timer value */
	NEXT;
    }

    CASE(OP_wait): {
	/* this is a wait without a timeout */
	fcalls++;
	if (PEEK_MESSAGE(p) == NULL) {
	    p->status = P_WAITING;
	    pc--;
	    SWAPOUT;
	    CHKSTK;
	    CHKHEAP;
	    return fcalls;
	}
	*stop++ = PEEK_MESSAGE(p)->mesg;
	FETCH(0);
	NEXT;
    }

    CASE(OP_wait1): {
	/* this is a wait with a timeout */
	fcalls++;
	if (PEEK_MESSAGE(p) == NULL) {
	    if (p->flags & F_TIMO) {
		pc = pc + make_signed_24(pc[0], pc[1], pc[2]);
		fail_pc = NULL;	/* commit */
		inguard = 0;
		p->flags &= ~F_TIMO;
#ifdef SEQ_TRACE
		SEQ_TRACE_TOKEN(p) = NIL;
#endif

		JOIN_MESSAGE(p);
		if (IS_TRACED(p))
		    trace_receive(p, am_timeout);
		FETCH(0);
		NEXT;
	    }
	    p->status = P_WAITING;
	    pc--;
	    SWAPOUT;
	    CHKSTK;
	    CHKHEAP;
	    return fcalls;
	}
	*stop++ = PEEK_MESSAGE(p)->mesg;
	pc += 3;
	FETCH(0);
	NEXT;
    }

    CASE(OP_popCommitJoin): {
	ErlMessage* mp;

	FETCH(0);
	cancel_timer(p);
#ifdef SEQ_TRACE
	SEQ_TRACE_TOKEN(p) = PEEK_MESSAGE(p)->seq_trace_token;
	if (SEQ_TRACE_TOKEN(p) != NIL) {
	    uint32 msg;
	    ASSERT(is_tuple(SEQ_TRACE_TOKEN(p)));
	    ASSERT(SEQ_TRACE_TOKEN_ARITY(p) == 5);
	    ASSERT(is_small(SEQ_TRACE_TOKEN_SERIAL(p)));
	    ASSERT(is_small(SEQ_TRACE_TOKEN_LASTCNT(p)));
	    ASSERT(is_small(SEQ_TRACE_TOKEN_FLAGS(p)));
	    ASSERT(is_pid(SEQ_TRACE_TOKEN_SENDER(p)));
	    p -> seq_trace_lastcnt = unsigned_val(SEQ_TRACE_TOKEN_SERIAL(p));
	    if (p -> seq_trace_clock < unsigned_val(SEQ_TRACE_TOKEN_SERIAL(p)))
		p -> seq_trace_clock = unsigned_val(SEQ_TRACE_TOKEN_SERIAL(p));
	    msg = PEEK_MESSAGE(p)->mesg;
	    seq_trace_output(SEQ_TRACE_TOKEN(p), msg, SEQ_TRACE_RECEIVE, p->id);
	}
#endif
	
	mp = PEEK_MESSAGE(p);  /* get message */
	UNLINK_MESSAGE(p);     /* unlink */
	JOIN_MESSAGE(p);       /* join */
	free_message(mp);      /* free message link */
	fail_pc = NULL;        /* commit */
	inguard = 0;
	stop--;                /* pop */
	NEXT;
    }

    CASE(OP_save): {
	FETCH(0);
	SAVE_MESSAGE(p);
	stop--;
	NEXT;
    }

    CASE(OP_arith_plus): {
	FETCH(0);
	if (is_small(NTH(1)) && is_small(NTH(2))) {
	    ires = small_plus(signed_val(NTH(2)),signed_val(NTH(1)));
	    stop -= 2;
	    if (IS_SSMALL(ires))
		*stop++ = make_small(ires);
	    else {
		H_NEED(2);
		*stop++ = small_to_big(ires, htop);
		htop += 2;
	    }
	    NEXT;
	}
	switch(i = NUMBER_CODE(NTH(1), NTH(2))) {
	case SMALL_BIG:
	    if (NTH(1)==SMALL_ZERO) { stop--; NEXT; }
	    barg1 = small_to_big(signed_val(NTH(1)), tmp_big1);
	    barg2 = NTH(2);
	    j = 2;
	    goto L_big_plus;
	case BIG_SMALL:
	    if (NTH(2)==SMALL_ZERO) { NTH(2)=NTH(1); stop--; NEXT; }
	    barg1 = NTH(1);
	    barg2 = small_to_big(signed_val(NTH(2)), tmp_big2);
	    j = 1;
	    goto L_big_plus;
	case BIG_BIG:
	    barg1 = NTH(1);
	    barg2 = NTH(2);
	    j = 3;
	L_big_plus:
	    sz1 = big_size(barg1);
	    sz2 = big_size(barg2);
	    sz = MAX_NEED(sz1,sz2)+1;
	    need_heap = BIG_NEED_SIZE(sz);
	    H_NEED(need_heap);
	    RELOAD(j, barg1, barg2);
	    res = big_plus(barg2, barg1, htop);
	    if (is_big(res))
		htop += (big_arity(res)+1);
	    else if (is_nil(res)) { EXIT(SYSTEM_LIMIT); }
	    stop -= 2;
	    *stop++ = res;
	    NEXT;
	default:
	    k = PLUS_OP;
	    goto L_float_op;
	}
    }

    CASE(OP_arith_minus): {
	FETCH(0);
	if (is_small(NTH(1)) && is_small(NTH(2))) {
	    ires = small_minus(signed_val(NTH(2)),signed_val(NTH(1)));
	    stop -= 2;
	    if (IS_SSMALL(ires))
		*stop++ = make_small(ires);
	    else
	    {
		H_NEED(2);
		*stop++ = small_to_big(ires, htop);
		htop += 2;
	    }
	    NEXT;
	}
	switch(i = NUMBER_CODE(NTH(1), NTH(2))) {
	case SMALL_BIG: /* small - big */
	    if (NTH(1) == SMALL_ZERO) { stop--; NEXT; }
	    barg1 = small_to_big(signed_val(NTH(1)), tmp_big1);
	    barg2 = NTH(2);
	    j = 2;
	    goto L_big_minus;
	case BIG_SMALL:   /* negate big - small (no optimize) */
	    barg1 = NTH(1);
	    barg2 = small_to_big(signed_val(NTH(2)), tmp_big2);
	    j = 1;
	    goto L_big_minus;
	case BIG_BIG:
	    barg1 = NTH(1);
	    barg2 = NTH(2);
	    j = 3;
	L_big_minus:
	    sz1 = big_size(barg1);
	    sz2 = big_size(barg2);
	    sz = MAX_NEED(sz1, sz2)+1;
	    need_heap = BIG_NEED_SIZE(sz);
	    H_NEED(need_heap);
	    RELOAD(j, barg1, barg2);
	    res = big_minus(barg2, barg1, htop);
	    if (is_big(res))
		htop += (big_arity(res)+1);
	    else if (is_nil(res)) { EXIT(SYSTEM_LIMIT); }
	    stop -= 2;
	    *stop++ = res;
	    NEXT;
	default:
	    k = MINUS_OP;
	    goto L_float_op;
	}
    }

    CASE(OP_arith_times): {
	FETCH(0);
	switch(i = NUMBER_CODE(NTH(1), NTH(2))) {
	case SMALL_SMALL:
	    if (NTH(1) == SMALL_ZERO) { NTH(2)=SMALL_ZERO; stop--; NEXT; }
	    if (NTH(1) == SMALL_ONE)  { stop--; NEXT; }
	    if (NTH(2) == SMALL_ZERO) { stop--; NEXT; }
	    if (NTH(2) == SMALL_ONE)  { NTH(2)=NTH(1); stop--; NEXT; }
	    barg1 = small_to_big(signed_val(NTH(1)), tmp_big1);
	    barg2 = small_to_big(signed_val(NTH(2)), tmp_big2);
	    j = 0;
	    goto L_big_times;
	case SMALL_BIG:
	    if (NTH(1) == SMALL_ZERO) { NTH(2)=SMALL_ZERO; stop--; NEXT; }
	    if (NTH(1) == SMALL_ONE)  { stop--; NEXT; }
	    barg1 = small_to_big(signed_val(NTH(1)), tmp_big1);
	    barg2 = NTH(2);
	    j = 2;
	    goto L_big_times;
	case BIG_SMALL:
	    if (NTH(2) == SMALL_ZERO) { stop--; NEXT; }
	    if (NTH(2) == SMALL_ONE)  { NTH(2)=NTH(1); stop--; NEXT; }
	    barg1 = NTH(1);
	    barg2 = small_to_big(signed_val(NTH(2)), tmp_big2);
	    j = 1;
	    goto L_big_times;
	case BIG_BIG:
	    barg1 = NTH(1);
	    barg2 = NTH(2);
	    j = 3;
	L_big_times:
	    sz1 = big_size(barg1);
	    sz2 = big_size(barg2);
	    sz = sz1 + sz2;
	    need_heap = BIG_NEED_SIZE(sz);
	    H_NEED(need_heap);
	    RELOAD(j, barg1, barg2);
	    res = big_times(barg2, barg1, htop);
	    if (is_big(res))
		htop += (big_arity(res)+1);
	    else if (is_nil(res)) { EXIT(SYSTEM_LIMIT); }
	    stop -= 2;
	    *stop++ = res;
	    NEXT;
	default:
	    k = TIMES_OP;
	    goto L_float_op;
	}
    }

    CASE(OP_arith_div): {
	FETCH(0);
	if (NTH(1) == SMALL_ZERO)
	    FAIL1(BADARITH);
	
	k = DIVIDE_OP;
	i = NUMBER_CODE(NTH(1),NTH(2));
	goto L_float_op;
    }

    CASE(OP_arith_intdiv): {
	FETCH(0);
	if (NTH(1) == SMALL_ZERO)
	    FAIL1(BADARITH);
	if (is_small(NTH(1)) && is_small(NTH(2))) {
	    ires = small_div(signed_val(NTH(2)), signed_val(NTH(1)));
	    stop -= 2;
	    *stop++ = make_small(ires);
	    NEXT;
	}
	switch(NUMBER_CODE(NTH(1), NTH(2))) {
	case BIG_SMALL:
	    stop -= 2;
	    *stop++ = SMALL_ZERO;
	    NEXT;
	case SMALL_BIG:
	    barg1 = small_to_big(signed_val(NTH(1)), tmp_big1);
	    barg2 = NTH(2);
	    j = 2;
	    goto L_big_div;
	case BIG_BIG:
	    barg1 = NTH(1);
	    barg2 = NTH(2);
	    j = 3;
	L_big_div:
	    ires = big_ucomp(barg2, barg1);
	    if (ires < 0)
	    {
		stop -= 2;
		*stop++ = SMALL_ZERO;
		NEXT;
	    }
	    else if (ires == 0)
	    {
		stop -= 2;
		if (big_sign(barg1) == big_sign(barg2))
		    *stop++ = SMALL_ONE;
		else
		    *stop++ = SMALL_MINUS_ONE;
		NEXT;
	    }
	    sz1 = big_size(barg1);
	    sz2 = big_size(barg2);
	    /* space for quotient & temporary remainder */
	    need_heap = BIG_NEED_SIZE(sz2-sz1+1) + BIG_NEED_SIZE(sz2);
	    H_NEED(need_heap);
	    RELOAD(j, barg1, barg2);
	    res = big_div(barg2, barg1, htop);
	    if (is_big(res))
		htop += (big_arity(res)+1);
	    else if (is_nil(res)) { EXIT(SYSTEM_LIMIT); }
	    stop -= 2;
	    *stop++ = res;
	    NEXT;
	}
	FAIL1(BADARITH);
    }

    CASE(OP_arith_rem): {
	FETCH(0);
	if (NTH(1) == SMALL_ZERO)
	    FAIL1(BADARITH);
	if (is_small(NTH(1)) && is_small(NTH(2))) {
	    ires = small_rem(signed_val(NTH(2)), signed_val(NTH(1)));
	    stop -= 2;
	    *stop++ = make_small(ires);
	    NEXT;
	}
	switch(NUMBER_CODE(NTH(1), NTH(2))) {
	case BIG_SMALL:
	    stop--;
	    NEXT;
	case SMALL_BIG:
	    barg1 = small_to_big(signed_val(NTH(1)), tmp_big1);
	    barg2 = NTH(2);
	    j = 2;
	    goto L_big_rem;
	case BIG_BIG:
	    barg1 = NTH(1);
	    barg2 = NTH(2);
	    j = 3;
	L_big_rem:
	    ires = big_ucomp(barg2, barg1);
	    if (ires < 0)
	    {
		stop--;
		NEXT;
	    }
	    else if (ires == 0)
	    {
		stop -= 2;
		*stop++ = SMALL_ZERO;
		NEXT;
	    }
	    sz = big_size(barg2);
	    need_heap = BIG_NEED_SIZE(sz);
	    H_NEED(need_heap);
	    RELOAD(j, barg1, barg2);
	    res = big_rem(barg2, barg1, htop);
	    if (is_big(res))
		htop += (big_arity(res)+1);
	    else if (is_nil(res)) { EXIT(SYSTEM_LIMIT); }
	    stop -= 2;
	    *stop++ = res;
	    NEXT;
	}
	FAIL1(BADARITH);
    }

    CASE(OP_arith_band):  {
	FETCH(0);
	if (is_small(NTH(1)) && is_small(NTH(2))) {
	    ires = small_band(signed_val(NTH(2)),signed_val(NTH(1)));
	    stop -= 2;
	    *stop++ = make_small(ires);
	    NEXT;	    
	}
	k = BAND_OP;
	goto L_binary_op;
    }

    CASE(OP_arith_bor): {
	FETCH(0);
	if (is_small(NTH(1)) && is_small(NTH(2))) {
	    ires = small_bor(signed_val(NTH(2)),signed_val(NTH(1)));
	    stop -= 2;
	    *stop++ = make_small(ires);
	    NEXT;	    
	}
	k = BOR_OP;
	goto L_binary_op;
    }

    CASE(OP_arith_bxor): {
	FETCH(0);
	if (is_small(NTH(1)) && is_small(NTH(2))) {
	    ires = small_bxor(signed_val(NTH(2)),signed_val(NTH(1)));
	    stop -= 2;
	    *stop++ = make_small(ires);
	    NEXT;	    
	}
	k = BXOR_OP;
	goto L_binary_op;
    }

    CASE(OP_arith_neg): {
	FETCH(0);
	switch(tag_val_def(NTH(1))) {
	case SMALL_DEF:
	    ires = small_neg(signed_val(NTH(1)));
	    if (IS_SSMALL(ires))
		NTH(1) = make_small(ires);
	    else {
		H_NEED(2);
		NTH(1) = small_to_big(ires, htop);
		htop += 2;
	    }
	    NEXT;
	case BIG_DEF:
	    sz = big_size(NTH(1));
	    need_heap = BIG_NEED_SIZE(sz);
	    H_NEED(need_heap);
	    res = big_neg(NTH(1),htop);
	    if (is_big(res))
		htop += (big_arity(res)+1);
	    else if (is_nil(res)) { EXIT(SYSTEM_LIMIT); }
	    NTH(1) = res;
	    NEXT;
	case FLOAT_DEF:
	    GET_DOUBLE(NTH(1), f1);
	    f1.fd = float_neg(f1.fd);
	    H_NEED(3);
	    NTH(1) = make_float(htop);
	    PUT_DOUBLE(f1, htop);
	    htop += 3;
	    NEXT;
	}
	FAIL1(BADARITH);
    }


    CASE(OP_arith_bnot): {
	FETCH(0);
	if (is_small(NTH(1))) {
	    NTH(1) = make_small(small_bnot(signed_val(NTH(1))));
	    NEXT;
	}
	else if (is_big(NTH(1))) {
	    sz = big_size(NTH(1))+1;
	    need_heap = BIG_NEED_SIZE(sz);
	    H_NEED(need_heap);
	    res = big_bnot(NTH(1), htop);
	    NTH(1) = res;
	    if (is_big(res))
		htop += (big_arity(res)+1);
	    else if (is_nil(res)) { EXIT(SYSTEM_LIMIT); }
	    NEXT;
	}
	FAIL1(BADARITH);
    }

    CASE(OP_arith_bsr): {
	FETCH(0);
	if (is_small(NTH(1))) {
	    iarg = -signed_val(NTH(1));
	    if (is_small(NTH(2))) {
		ires = signed_val(NTH(2));
		goto L_small_lshift;  /* common shift case */
	    }
	    else if (is_big(NTH(2))) {
		barg2 = NTH(2);
		j = 2;
		goto L_big_lshift;	/* common shift case */
	    }
	}
	FAIL1(BADARITH);
    }

    CASE(OP_arith_bsl): {
	FETCH(0);
	if (is_small(NTH(1))) {
	    iarg = signed_val(NTH(1));
	    if (is_small(NTH(2))) {
		ires = signed_val(NTH(2));
		goto L_small_lshift;
	    }
	    else if (is_big(NTH(2))) {
		barg2 = NTH(2);
		j = 2;
		goto L_big_lshift;
	    }
	}
	FAIL1(BADARITH);

    L_small_lshift:
	if (ires == 0) {
	    stop -= 2;
	    *stop++ = SMALL_ZERO;
	    NEXT;
	}
	if (iarg == 0) {
	    stop--;
	    NEXT;
	}
	else if (iarg < 0)  { /* right shift */
	    iarg = -iarg;
	    stop -= 2;
	    if (iarg >= 27)
	    {
		if (ires < 0)
		    *stop++ = SMALL_MINUS_ONE;
		else
		    *stop++ = SMALL_ZERO;
	    }
	    else
		*stop++ = make_small(ires >> iarg);
	    NEXT;
	}
	else  /* left shift */
	{
	    if (iarg < 27) {
		if (ires > 0) {
		    if (((-1 << (27-iarg)) & ires) == 0) {
			stop -= 2;
			*stop++ = make_small(ires << iarg);
			NEXT;
		    }
		}
		else {
		    if (((-1 << (27-iarg)) & ~ires) == 0) {
			stop -= 2;
			*stop++ = make_small(ires << iarg);
			NEXT;
		    }
		}
	    }
	}
	barg2 = small_to_big(ires, tmp_big2);
	j = 0;
    L_big_lshift:
	if (iarg == 0) {
	    stop--;
	    NEXT;
	}
	else if (iarg > 0)
	    sz = big_size(barg2) + (iarg / D_EXP);
	else {
	    sz = big_size(barg2);
	    if (sz <= (-iarg / D_EXP))
		sz = 3;
	    else
		sz -= (-iarg / D_EXP);
	}
	sz += 1;
	need_heap = BIG_NEED_SIZE(sz);
	H_NEED(need_heap);
	RELOAD(j, barg1, barg2);
	res = big_lshift(barg2, iarg, htop);
	if (is_big(res))
	    htop += (big_arity(res)+1);
	else if (is_nil(res)) { EXIT(SYSTEM_LIMIT); }
	stop -= 2;
	*stop++ = res;
	NEXT;
    }


    CASE(OP_exact_eqeq): {
	FETCH(0);
	if (eq(NTH(2), NTH(1)) == 0)
	    FAIL;
	stop -= 2;
	NEXT;
    }

    CASE(OP_exact_neq): {
	FETCH(0);
	if (eq(NTH(2), NTH(1)) != 0)
	    FAIL;
	stop -= 2;
	NEXT;
    }

    CASE(OP_comp_gt):      k=GT_OP;   goto L_comp;
    CASE(OP_comp_lt):      k=LT_OP;   goto L_comp;
    CASE(OP_comp_geq):     k=GEQ_OP;  goto L_comp;
    CASE(OP_comp_leq):     k=LEQ_OP;  goto L_comp;
    CASE(OP_comp_eqeq):    k=EQEQ_OP; goto L_comp;
    CASE(OP_comp_neq):     k=NEQ_OP; {
    L_comp:
	FETCH(0);
	ires = cmp(NTH(2), NTH(1));
	if (inguard) {
	    switch(k) {
	    case LT_OP: if (ires >= 0) FAIL; break;
	    case LEQ_OP: if (ires > 0) FAIL; break;
	    case GT_OP: if (ires <= 0) FAIL; break;
	    case GEQ_OP:  if (ires < 0) FAIL; break;
	    case EQEQ_OP: if (ires != 0) FAIL; break;
	    case NEQ_OP:  if (ires == 0) FAIL; break;
	    }
	    stop -= 2;
	}
	else {
	    switch(k) {
	    case LT_OP:  ires = (ires < 0); break;
	    case LEQ_OP: ires = (ires <= 0); break;
	    case GT_OP:  ires = (ires > 0); break;
	    case GEQ_OP: ires = (ires >= 0); break;
	    case EQEQ_OP: ires = (ires == 0); break;
	    case NEQ_OP:  ires =(ires != 0); break;
	    }
	    NTH(2) = ires ? am_true : am_false;
	    stop -= 1;
	}
	NEXT;
    }

    CASE(OP_test_float):     tmp = FLOAT_BIT; goto L_test;
    CASE(OP_test_atom):      tmp = ATOM_BIT; goto L_test;
    CASE(OP_test_tuple):     tmp = TUPLE_BIT; goto L_test;
    CASE(OP_test_pid):       tmp = PID_BIT; goto L_test;
    CASE(OP_test_port):      tmp = PORT_BIT; goto L_test;
    CASE(OP_test_reference): tmp = REFER_BIT; goto L_test;
    CASE(OP_test_binary):    tmp = BINARY_BIT; goto L_test;
    CASE(OP_test_list):      tmp = (LIST_BIT|NIL_BIT); goto L_test;
    CASE(OP_test_integer):   tmp = (SMALL_BIT|BIG_BIT); goto L_test;
    CASE(OP_test_number):    tmp = (SMALL_BIT|BIG_BIT|FLOAT_BIT); goto L_test;
    CASE(OP_test_constant): {
	tmp = (SMALL_BIT | BIG_BIT | FLOAT_BIT | BINARY_BIT |
	       ATOM_BIT | REFER_BIT | PORT_BIT | PID_BIT);
    L_test:
	FETCH(0);
	if (is_not_type(NTH(1), tmp))
	    FAIL;
	stop--;
	NEXT;
    }

    CASE(OP_bif_call): {
	BifFunction bf;
	iarg = make_16(*(pc+1), *(pc+2));
	resched_pc = pc - 1;
	FETCH(3);
	lastpos = 0;		/* This is not an apply */
	/* EXECUTE BIF CALL: iarg = index to bif, 
	   pc must be set to next instruction to execute 
	   after call */
    L_bif_call:
	SWAPOUT;
	p->fcalls = fcalls;  /* save current value */
	bif_gc = 0; 
	/* the bif sets a value in bif_gc to indicate how many reds it 
	   corresponds to. This should be solved in another way as soon
	   as possible (no global variable please) */
	k = bif_table[iarg].arity;
	bf = bif_table[iarg].f;
	if (IS_TRACED(p)) {
	    switch(k) {
	    case 0: trace_bif(p, bf); break;
	    case 1: trace_bif(p, bf, *(stop-1)); break;
	    case 2: trace_bif(p, bf, *(stop-2),*(stop-1)); break;
	    case 3: trace_bif(p, bf, *(stop-3),*(stop-2),*(stop-1)); break;
	    }
	}
	switch (k) {
	case 0: res = (*bf)(p); break;
	case 1: res = (*bf)(p,*(stop-1)); break;
	case 2: res = (*bf)(p,*(stop-2),*(stop-1)); break;
	case 3: res = (*bf)(p,*(stop-3),*(stop-2),*(stop-1));
	    break;
	}
	if (res == 0) {
	    if (p->freason == RESCHEDULE) {
		if (lastpos) {	/* Entered through apply; must
				 * restore stack. */
		    stop = stop - k + 3;
		    NTH(3) = am_erlang;
		    NTH(2) = function;
		    NTH(1) = lastpos;
		    p->stop = stop;
		}
		p->pc = resched_pc;   /* Set pc for reschedule */
		return fcalls;  /* we are suspended !! return !! */
	    }
	    else if (p->freason == TRAP) { /* TRAP CALL TO ERLANG */
		/* SWAPIN ? */
	    L_trap_call:
		tb = (byte*) export_list(unsigned_val(p->fvalue))->address;
		if (tb == NULL)
		    EXIT(UNDEF);
		S_NEED(FRAME_SIZE);  /* We need a frame */
		pc = pc-1;	/* Compensate for the FETCH done earlier. */
		LINK(0, k);
		pc = tb;
		FETCH(0);
		NEXT;
	    }
	    bif_error(p, iarg, inguard, k);
	    SWAPIN;
	    FAIL1(p->freason);
	}
	SWAPIN;
	*(stop - k) = res;      /* Handle return */
	stop = stop - k + 1;    /* Update stack */
	reds -= bif_gc;
	fcalls += bif_gc;
	NEXT;
    }

    /* like bif_call + return */
    CASE(OP_bif_enter): {
	BifFunction bf;
	iarg = make_16(*(pc+1), *(pc+2));
	resched_pc = pc - 1;
	resched_sp = stop;
	/* EXECUTE BIF ENTER: iarg = index to bif,
	   bif_enter works like call/return 
	   even for trapped bifs */

    L_bif_enter:
	SWAPOUT;
	p->fcalls = fcalls;
	bif_gc = 0; 
	/* the bif sets a value in bif_gc to indicate how many reds it 
	   corresponds to. This should be solved in another way as soon
	   as possible (no global variable please) */
	k = bif_table[iarg].arity;
	bf = bif_table[iarg].f;
	if (IS_TRACED(p)) {
	    switch(k) {
	    case 0: trace_bif(p, bf); break;
	    case 1: trace_bif(p, bf, *(stop-1)); break;
	    case 2: trace_bif(p, bf, *(stop-2),*(stop-1)); break;
	    case 3: trace_bif(p, bf, *(stop-3),*(stop-2),*(stop-1)); break;
	    }
	}
	switch (k) {
	case 0: res = (*bf)(p); break;
	case 1: res = (*bf)(p, *(stop-1)); break;
	case 2: res = (*bf)(p, *(stop-2), *(stop-1)); break;
	case 3: res = (*bf)(p, *(stop-3), *(stop-2), *(stop-1));
	    break;
	}
	if (res == 0) {
	    if (p->freason == RESCHEDULE) {
		p->stop = resched_sp;
		p->pc = resched_pc;   /* Set pc for reschedule */
		return fcalls;  /* we are suspended !! return !! */
	    }
	    else if (p->freason == TRAP) { /* TRAP ENTER TO ERLANG */
		/* SWAPIN ? (we should not build anything when trapping) */
		tb = (byte*) export_list(unsigned_val(p->fvalue))->address;
		if (tb == NULL)
		    EXIT(UNDEF);
		ELINK(k);
		pc = tb;  /* set after link */
		FETCH(0);
		NEXT;
	    }
	    bif_error(p, iarg, inguard, k);
	    SWAPIN;
	    FAIL1(p->freason);
	}
	SWAPIN;
	*(stop - k) = res;    /* Handle return */
	stop = stop - k + 1;  /* Update stack */
	UNLINK();
	reds -= bif_gc;
	fcalls += bif_gc;
	FETCH(0);
	NEXT;
    }

    CASE(OP_apply_enter): {
	uint32* sp;
	resched_pc = pc - 1;

    L_apply_enter:
	if (is_not_atom(NTH(3)))
	    FAIL1(BADARG);
	if (is_not_atom(NTH(2)))
	    FAIL1(BADARG);
	if ((is_not_nil(NTH(1))) && is_not_list(NTH(1)))
	    FAIL1(BADARG);

	if ((i = list_length(NTH(1))) == -1) {
	    FAIL1(BADARG);
	}

	CHKREDS;
	S_NEED(FRAME_SIZE+i);

	sp = stop;

	tmp = NTH(1);
	while (is_list(tmp)) {
	    tp = ptr_val(tmp);
	    *sp++ = CAR(tp);
	    tmp   = CDR(tp);
	}

	if (NTH(3) == am_erlang) {
	    if (NTH(2) == am_apply && i == 3) {
		stop = sp;
		goto L_apply_enter;
	    }
	    else if ((iarg = find_bif(unsigned_val(NTH(2)), i)) != -1) {
		resched_sp = stop;
		stop = sp;
		goto L_bif_enter;
	    }
	}

	if ((j = find_function(NTH(3), NTH(2), i)) < 0) {
	    if ((j = find_function(p->error_handler,
				   am_undefined_function, 3)) < 0)
		EXIT(UNDEF);
	    /*
	     * fix things up to comply with the normal
	     * case - looks wierd but optimises the most
	     * common case
	     */
	    i = 3;
	    sp = stop;
	}
	stop = sp;
	ELINK(i);
	pc = (byte*) export_list(j)->address;
	FETCH(0);
	NEXT;
    }

    CASE(OP_apply_call): {
	uint32* sp;
	uint32 module;

	resched_pc = pc - 1;
    L_apply_call:

	if (is_not_atom(NTH(3)))
	    FAIL1(BADARG);
	if (is_not_atom(NTH(2)))
	    FAIL1(BADARG);
	if (is_not_nil(NTH(1)) && is_not_list(NTH(1)))
	    FAIL1(BADARG);

	if ((i = list_length(NTH(1))) == -1) {
	    FAIL1(BADARG);
	}
	module = NTH(3);
	function = NTH(2);

	CHKREDS;
	S_NEED(FRAME_SIZE+i);

	/*
	 * list of arguments to the applied func is saved in
	 * lastpos
	 */

	lastpos = tmp = NTH(1);
	sp = stop - 3;	/* overwrite the mod func args */

	while (is_list(tmp)) {
	    tp = ptr_val(tmp);
	    *sp++ = CAR(tp);
	    tmp = CDR(tp);
	}

	if (module == am_erlang) {
	    if ((function == am_apply) && (i == 3)) {
		stop = sp;
		goto L_apply_call;
	    }
	    else if ((iarg = find_bif(unsigned_val(function), i)) != -1) {
		FETCH(3);
		stop = sp;
		goto L_bif_call;
	    }
	}
	if ((j = find_function(module, function, i)) < 0) {
	    /* try to call the error handler */
	    if ((j = find_function(p->error_handler,
				   am_undefined_function, 3)) < 0)
	    { EXIT(UNDEF); }

	    /* restore module & function */
	    NTH(3) = module;
	    NTH(2) = function;
	    NTH(1) = lastpos;
	    sp = stop;		/* so that stack pointer is
				 * OK below */
	    i = 3;		/* undef handler has arity 3 */
	}
	stop = sp;
	LINK(3, i);
	pc = (byte*) export_list(j)->address;
	FETCH(0);
	NEXT;
    }

    CASE(OP_debug_info): {
	tb = pc-1;
	FETCH(6);
	cc = tb;
	if (IS_TRACED_FL(p, F_TRACE_CALLS)) {
	    tb = cc+1;
	    need_heap = (2*tb[0]) + 11;
	    H_NEED(need_heap);
	    if (p->flags & F_TIMESTAMP)
		tmp = make_small(tb[0]);
	    else {
		if (tb[0] == 0)
		    tmp = NIL;
		else {
		    tmp = make_list(htop);
		    for (i = 0; i < tb[0]; i++) {
			*htop++ = ARG(i);
			*htop = make_list(htop + 1);
			htop++;
		    }
		    *(htop - 1) = NIL;
		}
	    }
	    tmp = TUPLE3(htop,
			 make_atom(make_16(tb[1], tb[2])),
			 make_atom(make_16(tb[3], tb[4])),
			 tmp);
	    htop += 4;
	    trace_proc(p, am_call, tmp);
	}
	NEXT;
    }

    CASE(OP_gotoix): {
	if (is_not_small(NTH(1))) FAIL;
	i = signed_val(NTH(1));     /* jump index */
	k = make_16(pc[0], pc[1]);  /* number of labels */
        j = make_16(pc[2], pc[3]);  /* low index */
	i -= j;
	if (i < 0 || i >= k) FAIL;
	tb = pc + 4 + 3*i;
	pc += make_signed_24(tb[0], tb[1], tb[2]);
	FETCH(0);
	NEXT;
    }

    CASE(OP_die): {  EXIT(KILLED);  }

    CASE(OP_dummy): {
	goto bad_opcode;
    }

    CASE_DEFAULT: {
    bad_opcode:
	erl_printf(CERR, " Unknown op code %d in process ", *(pc - 1));
	display(p->id, CERR);
	erl_putc('\n', CERR);
	EXIT(INTERNAL_ERROR);
    }

    DISPATCH_END;


/****************************************************
  All bin-op floating point operations is done here
    i is the argument code, and k is the operation
*****************************************************/

 L_float_op:

    if (!FP_PRE_CHECK_OK()) {
	FAIL1(BADARITH);
    }

    /* Convert to float arguments */
    switch(i) {
    case SMALL_SMALL:
	f1.fd = signed_val(NTH(1));
	f2.fd = signed_val(NTH(2));
	break;
    case SMALL_BIG:
	f1.fd = signed_val(NTH(1));
	f2.fd = big_to_double(NTH(2));
	if (!FP_RESULT_OK(f2.fd)) { FAIL1(BADARITH) }
	break;
    case BIG_SMALL:
	f1.fd = big_to_double(NTH(1));
	if (!FP_RESULT_OK(f1.fd)) { FAIL1(BADARITH) }
	f2.fd = signed_val(NTH(2));
	break;
    case BIG_BIG:
	f1.fd = big_to_double(NTH(1));
	if (!FP_RESULT_OK(f1.fd)) { FAIL1(BADARITH) }
	f2.fd = big_to_double(NTH(2));
	if (!FP_RESULT_OK(f2.fd)) { FAIL1(BADARITH) }
	break;
    case SMALL_FLOAT:
	f1.fd = signed_val(NTH(1));
	GET_DOUBLE(NTH(2), f2);
	break;
    case BIG_FLOAT:
	f1.fd = big_to_double(NTH(1));
	if (!FP_RESULT_OK(f1.fd)) { FAIL1(BADARITH) }
	GET_DOUBLE(NTH(2), f2);
	break;
    case FLOAT_SMALL:
	GET_DOUBLE(NTH(1), f1);
	f2.fd = signed_val(NTH(2));
	break;
    case FLOAT_BIG:
	GET_DOUBLE(NTH(1), f1);
	f2.fd = big_to_double(NTH(2));
	if (!FP_RESULT_OK(f2.fd)) { FAIL1(BADARITH) }
	break;
    case FLOAT_FLOAT:
	GET_DOUBLE(NTH(1), f1);
	GET_DOUBLE(NTH(2), f2);
	break;
    default:
	FAIL1(BADARITH);
    }

    stop -= 2;
    H_NEED(3);

    switch(k) {
    case PLUS_OP:
	f1.fd = float_plus(f2.fd, f1.fd);
	break;
    case MINUS_OP:
	f1.fd = float_minus(f2.fd, f1.fd);
	break;
    case TIMES_OP:
	f1.fd = float_times(f2.fd, f1.fd);
	break;
    case DIVIDE_OP:
	if (f1.fd == 0.0) {
	    FAIL1(BADARITH);
	}
	f1.fd = float_div(f2.fd,f1.fd);
	break;
    default:
	EXIT(INTERNAL_ERROR);
    }

    if (FP_RESULT_OK(f1.fd)) {
	*stop++ = make_float(htop);
	PUT_DOUBLE(f1, htop);
	htop += 3;
	NEXT;
    } else {
	FAIL1(BADARITH);
    }

/****************************************************
  Binary operations band, bor, bxor

****************************************************/

 L_binary_op:

    switch(NUMBER_CODE(NTH(1),NTH(2))) {
    case SMALL_BIG:
	barg1 = small_to_big(signed_val(NTH(1)), tmp_big1);
	barg2 = NTH(2);
	j = 2;
	break;
    case BIG_SMALL:
	barg1 = NTH(1);
	barg2 = small_to_big(signed_val(NTH(2)), tmp_big1);
	j = 1;
	break;
    case BIG_BIG:
	barg1 = NTH(1);
	barg2 = NTH(2);
	j = 3;
	break;
    default:
	FAIL1(BADARITH);
    }
    
    sz1 = big_size(barg1);
    sz2 = big_size(barg2);
    sz = MAX_NEED(sz1, sz2) + 1;
    need_heap = BIG_NEED_SIZE(sz);
    H_NEED(need_heap);
    RELOAD(j, barg1, barg2);
    switch(k) {
    case BAND_OP: 
	res = big_band(barg2,barg1,htop);
	break;
    case BOR_OP:
	res = big_bor(barg2,barg1,htop);
	break;
    case BXOR_OP:
	res = big_bxor(barg2,barg1,htop);
	break;
    default:
	EXIT(INTERNAL_ERROR);
    }
    if (is_big(res))
	htop += (big_arity(res)+1);
    else if (is_nil(res))
	EXIT(SYSTEM_LIMIT);
    stop -= 2;
    *stop++ = res;
    NEXT;

 L_fail1: /* We endup here if we FAIL1, tmp = reason */
    if (!inguard) {
	fail_reason = tmp;
	goto L_exit;
    }

 L_fail:  /* We endup here if we FAIL */

    if (fail_pc != NULL) {
	stop = fail_stack;
	pc = fail_pc + make_signed_24(fail_pc[0],fail_pc[1],fail_pc[2]);
	FETCH(0);
	NEXT;
    }

 L_exit:  /* We go here if we EXIT */

    switch (fail_reason) {
    case NORMAL:
	*stop++ = am_normal;
	break;
    case BADMATCH:
	H_NEED(3);
	NTH(1) = TUPLE2(htop, am_badmatch, NTH(1));
	htop += 3;
	break;
    case CASE_CLAUSE:
	H_NEED(3);
	NTH(1) = TUPLE2(htop, am_case_clause, NTH(1));
	htop += 3;
	break;
    case IF_CLAUSE:
	*stop++ = am_if_clause;
	break;
    case UNDEF:
	*stop++ = am_undef;
	break;
    case BADARITH:
	*stop++ = am_badarith;
	break;
    case BADARG:
	*stop++ = am_badarg;
	break;
    case FUNCTION_CLAUSE:
	*stop++ = am_function_clause;
	break;
    case BADSIGNAL:
	*stop++ = am_badsig;
	break;
    case TIMEOUT_VALUE:
	*stop++ = am_timeout_value;
	break;
    case NOPROC:
	p->catches = ENULL;
	*stop++ = am_noproc;
	break;
    case NOTALIVE:
	*stop++ = am_notalive;
	break;
    case SYSTEM_LIMIT:
	*stop++ = am_system_limit;
	break;
    case KILLED:
	*stop++ = p->fvalue;
	p->fvalue = NIL;
	p->catches = ENULL;
	break;
    case USER_ERROR:
	*stop++ = p->fvalue;
	p->fvalue = NIL;
	break;
    case USER_EXIT:
	/*
	 * stack already contains error code. We special case
	 * exit(normal)
	 */
	if (NTH(1) == am_normal)
	    fail_reason = NORMAL;
	break;
    case THROWN:
	if (p->catches != ENULL)
	    break;
	*stop++ = am_nocatch;
	break;
    default:
	p->catches = ENULL;
	erl_printf(CERR, "unknown error %x\n", fail_reason);
	*stop++ = am_internal_error;
	break;
    }

    /*
     * create an exit value which we will use in all cases
     * *except* a throw and a normal exit: {Reason, Module,
     * Function, ArgumentList}
     * tmp is set to reason
     */
	
    if ((fail_reason != THROWN) && (fail_reason != NORMAL) &&
	(fail_reason != USER_EXIT) && (fail_reason != KILLED)) {
	if (cc != NULL) {
	    tb = cc+1;

	    need_heap = (2*tb[0])+10;     /* May build am_EXIT +3 */
	    H_NEED(need_heap);

	    if (tb[0] == 0)
		tmp = NIL;
	    else {
		tmp = make_list(htop);
		for (i = 0; i < tb[0]; i++) {
		    (void) CONS(htop, ARG(i), make_list(htop+2));
		    htop += 2;
		}
		*(htop - 1) = NIL;
	    }
	    tmp = TUPLE3(htop,
			 make_atom(make_16(tb[1],tb[2])),
			 make_atom(make_16(tb[3],tb[4])),
			 tmp);
	    htop += 4;
	    tmp = TUPLE2(htop, NTH(1), tmp);
	    htop += 3;
	}
	else {
	   need_heap = 3+3;	/* May build am_EXIT */
	   H_NEED(need_heap);
	   /* We can get MFA from 'initial', and we may even
	      be able to get the initial arg list from the stack,
	      but currently we make do with 'undefined', in analogy
	      with process_info/2 */
	   tmp = am_undefined;
	   tmp = TUPLE2(htop, NTH(1), tmp);
	   htop += 3;
	}
    }
    else {
	H_NEED(3);    /* May build am_EXIT */
	tmp = NTH(1);
    }


    if (p->catches != ENULL) {
	tp = NULL;
	if (fail_reason != THROWN) {
	    NTH(1) = TUPLE2(htop, am_EXIT, tmp);
	    htop += 3;
	}
	/*
	 * We try to unwind the stack to find the catch. This
	 * will contain an address to which to jump when we
	 * fail. 
	 */
	while(1) {
	    if (p->catches > fp) {    /* is catch in the frame? */
		tp = p->catches;
		break;	              /* got it so break */
	    }
	    /* didn't get it so unwind a frame */
	    pc = (byte*) fp[FRAME_PC];
	    cc = (byte*) fp[FRAME_CC];
	    ap = frame_val(fp[FRAME_AP]);
	    fp = frame_val(fp[FRAME_FP]);
	    /*
	     * we don't really need pc since we will set
	     * it anyway by what is in the catch data on
	     * the stack, but the easiest way to check
	     * that we haven't bombed out and gone to the
	     * top of the stack is to check that pc !=
	     * ENULL
	     */
	    if (pc == NULL) {
		fail_reason = INTERNAL_ERROR;
		display(p->id, CBUF);
		erl_printf(CERR, " Improper catch (1)\n");
		tmp = am_internal_error;
		break;
	    }
	}

	if (pc != NULL) {
	    /* yes, it was the catch */
	    pc = (byte*) tp[CATCH_PC];
	    p->catches = ptr_val(tp[CATCH_PTR]);
	    *tp = NTH(1);
	    stop = tp + 1;
	    fail_reason = BADMATCH;	/* default value for a next error */
	    inguard = 0;
	    if (*pc++ != OP_popCatch) {
		erl_printf(CERR, " Improper catch (2)\n");
		EXIT(INTERNAL_ERROR);
	    }
	    FETCH(0);
	    NEXT;
	}
    }
    /*
     * we now know that we have a true exit, mess contains an
     * exit value
     */
    if ((cc != NULL) &&
	(fail_reason != NORMAL) &&
	(fail_reason != USER_EXIT) &&
	(fail_reason != KILLED)) {
	erl_printf(CBUF, "!!! Error in process ");
	display(p->id, CBUF);
	if (this_node != am_noname) {
	    erl_printf(CBUF, " on node ");
	    print_atom(unsigned_val(this_node), CBUF);
	}
	erl_printf(CBUF, " with exit value: ");
	ldisplay(tmp, CBUF, display_items);
	erl_printf(CBUF, "\n");
	    
	send_error_to_logger(p->group_leader);
    }

    /* deliver the exit message to linked processes & exit */
    do_exit(p, tmp);

    return fcalls;
}

