/* ``The contents of this file are subject to the Erlang Public License,
 * Version 1.0, (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.erlang.org/EPL1_0.txt
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 * 
 * The Original Code is Erlang-4.7.3, December, 1998.
 * 
 * The Initial Developer of the Original Code is Ericsson Telecom
 * AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
 * Telecom AB. All Rights Reserved.
 * 
 * Contributor(s): ______________________________________.''
 */
#ifndef __CONFIG_H__
#define __CONFIG_H__

#define JAM 1
#define EMULATOR "JAM"
#define SEQ_TRACE 1 /* Enables SEQ_TRACE only for JAM at the moment */

/* some constants for various  table sizes etc */
/* "constants" which are declared as 'extern int' have been moved to
   mkconfig.c, to allow users to set them in config.c */

#define ATOM_TEXT_SIZE  32768	/* Increment for allocating atom text space */

extern int MAX_PORTS;		/* Maximum number of ports                  */
extern int TMP_BUF_SIZE;	/* temp buff size - diverse usages          */
#define ITIME 100		/* Number of milliseconds per clock tick    */
#define MAX_LOAD_FILE_PATH 1024 /* Max no of chars in path to a    jam file */
#define TMP_BUF_MIN 2048	/* minimum size allowed for same            */
#define BG_PROPORTION 8		/* Do bg processes after this # fg          */

#ifdef SLOW_PROCESSOR           /* E.g. VxWorks/cpu32 needs this            */
#define CONTEXT_REDS 250        /* Swap process out after this number       */
#else
#define CONTEXT_REDS 500	/* Swap process out after this number       */
#endif

#define INPUT_REDUCTIONS   (4 * CONTEXT_REDS)

/* Heap and stack constants */

#define H_MARGIN 0          /* heap margin in words (test 0) */
extern int H_MIN_SIZE;      /* minimum heap messured in words */
#define H_DEFAULT_SIZE 34   /* 34 words as default heap min size */

#define S_MARGIN 32         /* stack margin in words */
extern int S_MIN_SIZE;      /* minimum stack messured in words */
#define S_DEFAULT_SIZE  34  /* 34 words as default stack min size */

/* This bit pattern is for stack check,
   It's a THING_DEF (not a stack object) and a mirrored 16 bit code */

#define S_PATTERN  0x751248AE

#ifdef __WIN32__
/*
 * Due to an optimizer bug in Microsoft C, pointers may be shifted
 * *arithmetically* right.  To avoid that problem, we will put tags
 * in the high end of word (to extract the tag an AND operation is
 * used, which is harder to get wrong even by a smart optimizer).
 */
#  define HIGH_TAGS
#else
#  undef HIGH_TAGS
#endif

/* tag size */

#define TAGSIZE       4
#define BODY          28

#ifdef HIGH_TAGS
#  define TAGMASK       0xf0000000
#  define ARITY_MASK    0x0fffffff  /* arity for tuples max 2^28-1 */
#  define TARITY_MASK   0x0000ffff  /* arity for thing max 2^16-1 */
#else
#  define TAGMASK       0xf
#  define ARITY_MASK    0xfffffff0  /* arity for tuples max 2^28-1 */
#  define TARITY_MASK   0x000ffff0  /* arity for thing max 2^16-1 */
#endif

#define MAX_SMALL     ((1 << 27)-1)
#define MIN_SMALL     -(1 << 27)

#define BIG_ARITY_MAX ((1 << 16)-1)
#define BIG_SIGN_BIT  0x08000000


/* definition of tags. NB the order of these is used for lexical comparison */

#define FRAME_DEF	0
#define SMALL_DEF 	1
#define BIG_DEF         2
#define FLOAT_DEF	3
#define ATOM_DEF	4
#define REFER_DEF	5
#define PORT_DEF	6
#define PID_DEF     	7
#define TUPLE_DEF   	8
#define NIL_DEF     	9
#define LIST_DEF    	10
#define ARITYVAL_DEF   	11
#define MOVED_DEF	12
#define CATCH_DEF       13
#define THING_DEF	14
#define BINARY_DEF      15

#ifdef HIGH_TAGS
#  define _CONSTRUCT_TAG(x) (x << BODY)
#else
#  define _CONSTRUCT_TAG(x) x
#endif

#define FRAME		_CONSTRUCT_TAG(FRAME_DEF)
#define NIL     	_CONSTRUCT_TAG(NIL_DEF)
#define SMALL 	        _CONSTRUCT_TAG(SMALL_DEF)
#define BIG 	        _CONSTRUCT_TAG(BIG_DEF)
#define ATOM		_CONSTRUCT_TAG(ATOM_DEF)
#define REFER		_CONSTRUCT_TAG(REFER_DEF)
#define PORT		_CONSTRUCT_TAG(PORT_DEF)
#define PID     	_CONSTRUCT_TAG(PID_DEF)
#define TUPLE   	_CONSTRUCT_TAG(TUPLE_DEF)
#define LIST    	_CONSTRUCT_TAG(LIST_DEF)
#define ARITYVAL   	_CONSTRUCT_TAG(ARITYVAL_DEF)
#define MOVED		_CONSTRUCT_TAG(MOVED_DEF)
#define CATCH           _CONSTRUCT_TAG(CATCH_DEF)
#define FLOAT		_CONSTRUCT_TAG(FLOAT_DEF)
#define THING		_CONSTRUCT_TAG(THING_DEF)
#define BINARY          _CONSTRUCT_TAG(BINARY_DEF)

#define FRAME_BIT       (1 << FRAME_DEF)
#define SMALL_BIT       (1 << SMALL_DEF)
#define BIG_BIT         (1 << BIG_DEF)
#define FLOAT_BIT       (1 << FLOAT_DEF)
#define ATOM_BIT        (1 << ATOM_DEF)
#define REFER_BIT       (1 << REFER_DEF)
#define PORT_BIT        (1 << PORT_DEF)
#define PID_BIT         (1 << PID_DEF)
#define TUPLE_BIT       (1 << TUPLE_DEF)
#define NIL_BIT         (1 << NIL_DEF)
#define LIST_BIT        (1 << LIST_DEF)
#define ARITYVAL_BIT    (1 << ARITYVAL_DEF)
#define MOVED_BIT       (1 << MOVED_DEF)
#define CATCH_BIT       (1 << CATCH_DEF)
#define THING_BIT       (1 << THING_DEF)
#define BINARY_BIT      (1 << BINARY_DEF)

#define ORIG_CREATION 0

/* This macro depend on SMALL=1, BIG=2, FLOAT=3 */

#define NUMBER_CODE(x, y)  (((tag_val_def(x) < 4 ) && \
			     (tag_val_def(y) < 4 )) ? \
			    ((3 * (tag_val_def(x) - 1)) + \
			     (tag_val_def(y) - 1)) : -1)

#define SMALL_SMALL   0
#define SMALL_BIG     1
#define SMALL_FLOAT   2
#define BIG_SMALL     3
#define BIG_BIG       4
#define BIG_FLOAT     5
#define FLOAT_SMALL   6
#define FLOAT_BIG     7
#define FLOAT_FLOAT   8

/* macros for extracting bytes from uint16's */

#define hi_byte(a)  ((a) >> 8)
#define lo_byte(a)  ((a) & 255)

/* macros for combining bytes */

#define make_16(x, y) (((x) << 8) | (y))
#define make_24(x,y,z) (((x) << 16) | ((y) << 8) | (z))
#define make_32(x3,x2,x1,x0) (((x3)<<24) | ((x2)<<16) | ((x1)<<8) | (x0))

#define make_signed_24(x,y,z) ((sint32) (((x) << 24) | ((y) << 16) | ((z) << 8)) >> 8)
#define make_signed_32(x3,x2,x1,x0) ((sint32) (((x3) << 24) | ((x2) << 16) | ((x1) << 8) | (x0)))

#define is_list(x)        (((x) & TAGMASK) == LIST)
#define is_not_list(x)    (((x) & TAGMASK) != LIST)

#define is_nil(x)         ((x) == NIL)
#define is_not_nil(x)     ((x) != NIL)

#define is_atom(x)        (((x) & TAGMASK) == ATOM)
#define is_not_atom(x)    (((x) & TAGMASK) != ATOM)

#define is_small(x)       (((x) & TAGMASK) == SMALL)
#define is_not_small(x)   (((x) & TAGMASK) != SMALL)

#define is_big(x)         (((x) & TAGMASK) == BIG)
#define is_not_big(x)     (((x) & TAGMASK) != BIG)

#define is_tuple(x)       (((x) & TAGMASK) == TUPLE)
#define is_not_tuple(x)   (((x) & TAGMASK) != TUPLE)

#define is_arity_value(x) (((x) & TAGMASK) == ARITYVAL)
#define is_not_arity_value(x) (((x) & TAGMASK) != ARITYVAL)

#define is_refer(x)       (((x) & TAGMASK) == REFER)
#define is_not_refer(x)   (((x) & TAGMASK) != REFER)

#define is_pid(x)         (((x) & TAGMASK) == PID)
#define is_not_pid(x)     (((x) & TAGMASK) != PID)

#define is_frame(x)       (((x) & TAGMASK) == FRAME)
#define is_not_frame(x)   (((x) & TAGMASK) != FRAME)

#define is_port(x)        (((x) & TAGMASK) == PORT)
#define is_not_port(x)    (((x) & TAGMASK) != PORT)

#define is_float(x)       (((x) & TAGMASK) == FLOAT)
#define is_not_float(x)   (((x) & TAGMASK) != FLOAT)

#define is_thing(x)       (((x) & TAGMASK) == THING)
#define is_not_thing(x)   (((x) & TAGMASK) != THING)

#define is_binary(x)      (((x) & TAGMASK) == BINARY)
#define is_not_binary(x)  (((x) & TAGMASK) != BINARY)

#define is_moved(x)       (((x) & TAGMASK) == MOVED)
#define is_not_moved(x)   (((x) & TAGMASK) != MOVED)

#define is_catch(x)       (((x) & TAGMASK) == CATCH)
#define is_not_catch(x)   (((x) & TAGMASK) != CATCH)

/* number tests */

#define is_integer(x)      (is_small(x) || is_big(x))
#define is_not_integer(x)  (is_not_small(x) && is_not_big(x))

#define is_number(x)       (is_integer(x) || is_float(x))
#define is_not_number(x)   (is_not_integer(x) && is_not_float(x))

#ifdef HIGH_TAGS
#  define is_byte(x)         (((x) & 0xffffff00) == SMALL)
#  define arityval(x)        ((x) & ARITY_MASK)
#  define thing_arityval(x)  ((x) & TARITY_MASK)
#else
#  define is_byte(x)         (((x) & 0xfffff00f) == SMALL)
#  define arityval(x)        (((x) & ARITY_MASK) >> TAGSIZE)
#  define thing_arityval(x)  (((x) & TARITY_MASK) >> TAGSIZE)
#endif

#define tag(x)             ((x) & TAGMASK)

/* bit version tests */
#define is_type(x, mask)     (((1<<(tag_val_def(x))) & (mask)) != 0)
#define is_not_type(x, mask) (((1<<(tag_val_def(x))) & (mask)) == 0)

#define is_table_id(x) (is_small(x) || is_atom(x))

#if 0
#define IS_INTEGER(x)  is_type(x, SMALL_BIT | BIG_BIT)
#define IS_NUMBER(x)   is_type(x, SMALL_BIT | BIG_BIT | FLOAT_BIT)
#endif

#define IS_CONST(x)    is_type(x, SMALL_BIT | ATOM_BIT | REFER_BIT | PORT_BIT | PID_BIT | NIL_BIT)

#define IS_ONE_CELL(x) is_type(x, SMALL_BIT | ATOM_BIT | REFER_BIT | PORT_BIT | PID_BIT | NIL_BIT | BINARY_BIT)

#if defined(HIGH_TAGS) && defined(EXTRA_POINTER_BITS)
#  define ptr_val(x) ((uint32 *) (((uint32)(x) & ~TAGMASK) | EXTRA_POINTER_BITS))
#  define ENULL ((void*) EXTRA_POINTER_BITS)
#elif defined(EXTRA_POINTER_BITS)
#  define ptr_val(x) ((uint32 *) (((uint32)(x) >> TAGSIZE) | EXTRA_POINTER_BITS))
#  define ENULL ((void*) EXTRA_POINTER_BITS)
#elif defined(HIGH_TAGS)
#  define ptr_val(x) ((uint32 *) ((x) & ~TAGMASK))
#  define ENULL NULL
#else
#  define ptr_val(x) ((uint32 *) ((x) >> TAGSIZE))
#  define ENULL NULL
#endif

#ifdef HIGH_TAGS
#  define make_ptr(x)  ((uint32)(x))
#  define unsigned_val(x)  ((uint32) ((x) & ~TAGMASK))
#  define signed_val(X)  (((sint32) ((X) << TAGSIZE)) >> TAGSIZE)
#  define tag_val_def(x) ((uint32) (x) >> BODY)
#else
#  define make_ptr(x)  ((uint32)(x) << TAGSIZE)
#  define unsigned_val(x)  ((uint32) ((x) >> TAGSIZE))
#  define signed_val(x)    (((sint32) (x)) >> TAGSIZE)
#  define tag_val_def(x) ((x) & TAGMASK)
#endif

#define frame_val(x) ptr_val(x)
#define tag_val(x)     ((x) & TAGMASK)

#define not_eq_tags(x,y) (((x) ^ (y)) & TAGMASK)

#define offset_ptr(x, offs)   (make_ptr(ptr_val(x)+offs) | tag_val(x))
#define offset_frame(x, offs) make_frame(frame_val(x)+offs)

#define make_frame(x)	 (make_ptr(x) | FRAME)
#define make_list(x)	 (make_ptr(x) | LIST)
#define make_tuple(x)	 (make_ptr(x) | TUPLE)
#define make_moved(x)    (((uint32)(x) & ~TAGMASK) | MOVED)
#define make_catch(x)	 (make_ptr(x) | CATCH)
#define make_float(x)	 (make_ptr(x) | FLOAT)
#define make_binary(x)   (make_ptr(x) | BINARY)
#define make_big(x)      (make_ptr(x) | BIG)

#ifdef HIGH_TAGS
#  define _ADD_TAG(x, tag) (((x) & ~TAGMASK) | tag)
#else
#  define _ADD_TAG(x, tag) (((x) << TAGSIZE) | tag)
#endif

#define make_small(x)    _ADD_TAG(x, SMALL)
#define make_atom(x)     _ADD_TAG(x, ATOM)
#define make_arityval(x) _ADD_TAG(x, ARITYVAL)
#define make_thing(x)    _ADD_TAG(x, THING)

#define SMALL_MINUS_TWO  make_small(-2)
#define SMALL_MINUS_ONE  make_small(-1)
#define SMALL_ZERO       make_small(0)
#define SMALL_ONE        make_small(1)
#define SMALL_TWO        make_small(2)

/* Float definition for byte and word access */
typedef double ieee754_8;

typedef union float_def
{
    ieee754_8 fd;
    byte   fb[sizeof(ieee754_8)];
    uint16 fs[sizeof(ieee754_8) / sizeof(uint16)];
    uint32 fw[sizeof(ieee754_8) / sizeof(uint32)];
} FloatDef;


#define GET_DOUBLE(x, f) (f).fw[0] = *(ptr_val(x)+1), \
                         (f).fw[1] = *(ptr_val(x)+2)

#define PUT_DOUBLE(f, x)  *(x) = make_thing(2), \
                          *((x)+1) = (f).fw[0], \
			  *((x)+2) = (f).fw[1]

#define CONS(hp, car, cdr) \
        (*(hp) = (car), \
         *((hp)+1) = (cdr), \
          make_list(hp))

#define CAR(x)  *(x)
#define CDR(x)  *((x)+1)

#define TUPLE0(t) \
        ((t)[0] = make_arityval(0), \
        make_tuple(t))
#define TUPLE1(t,e1) \
        ((t)[0] = make_arityval(1), \
        (t)[1] = (e1), \
        make_tuple(t))
#define TUPLE2(t,e1,e2) \
        ((t)[0] = make_arityval(2), \
        (t)[1] = (e1), \
        (t)[2] = (e2), \
        make_tuple(t))
#define TUPLE3(t,e1,e2,e3) \
        ((t)[0] = make_arityval(3), \
        (t)[1] = (e1), \
        (t)[2] = (e2), \
        (t)[3] = (e3), \
        make_tuple(t))
#define TUPLE4(t,e1,e2,e3,e4) \
        ((t)[0] = make_arityval(4), \
        (t)[1] = (e1), \
        (t)[2] = (e2), \
        (t)[3] = (e3), \
        (t)[4] = (e4), \
        make_tuple(t))
#define TUPLE5(t,e1,e2,e3,e4,e5) \
        ((t)[0] = make_arityval(5), \
        (t)[1] = (e1), \
        (t)[2] = (e2), \
        (t)[3] = (e3), \
        (t)[4] = (e4), \
        (t)[5] = (e5), \
        make_tuple(t))
#define TUPLE6(t,e1,e2,e3,e4,e5,e6) \
        ((t)[0] = make_arityval(6), \
        (t)[1] = (e1), \
        (t)[2] = (e2), \
        (t)[3] = (e3), \
        (t)[4] = (e4), \
        (t)[5] = (e5), \
	(t)[6] = (e6), \
        make_tuple(t))

#ifdef DEBUG
#define VERBOSE(x) do { if (verbose) x } while(0)
#else
#define VERBOSE(x)
#endif

/* THIS (HAlloc) MACRO OBSOLETES ALL PNEED BIF_NEED etc */
#define HAlloc(p, sz) \
    ((((p)->htop + (sz)) > ((p)->heap_margin)) ? \
          halloc((p),(sz)) : ((p)->htop = (p)->htop + (sz), (p)->htop - (sz)))

#define ArithAlloc(p, sz) HAlloc(p, sz)

/* Offsets from vars to frame values */
#define FRAME_SIZE 4
#define FRAME_FP  0
#define FRAME_AP  1
#define FRAME_PC  2
#define FRAME_CC  3

#define CATCH_SIZE 2
#define CATCH_PTR  0
#define CATCH_PC   1

#define INIT_FRAME(p) do { \
    (p)->ap = (p)->fp = (p)->stop; \
    (p)->fp[FRAME_FP] = (uint32) make_frame((p)->stack); \
    (p)->fp[FRAME_AP] = (uint32) make_frame((p)->stack); \
    (p)->fp[FRAME_PC] = 0; \
    (p)->fp[FRAME_CC] = 0; \
    (p)->stop += FRAME_SIZE; \
} while(0)


#define P_SERIAL 3
#define P_NODE   8
#define P_CREAT  2
#define P_NUMBER 15

/* Minimum NUMBER of processes for a small system to start */
#define MIN_PROCESS  16

/* Maximum NUMBER of process identifiers */
#define MAX_PROCESS (1 << P_NUMBER)

/* Maximum NUMBER of serial numbers */
#define MAX_SERIAL   (1 << P_SERIAL)

#ifdef HIGH_TAGS
/* pid layout
**
**     Tag Serial  Number     Node      Creat
**   +---------------------------------------+
**   | 4   | 3  |  15       |   8      |   2 |
**   +---------------------------------------+
*/

#define make_pid3(Ser,Node,Number,Creation) \
  ((uint32) (PID | ((Ser) << 25) | ((Number) << 10) | \
	     ((Node) << 2) | (Creation)))

#define make_pid(Ser,Node,Number) \
  make_pid3(Ser,Node,Number,ORIG_CREATION)

#define GETBITS(X,Pos,Size) (((X) >> (Pos)) & ~(~0 << (Size)))

/* This macro get Size bits starting at low order position Pos
   and adjusts the bits to the right 
   bits are numbered from 1 - 32 */

#define get_serial(Pid)   GETBITS(Pid,25,P_SERIAL)
#define get_node(Pid)     GETBITS(Pid,2,P_NODE)
#define get_creation(Pid) GETBITS(Pid,0,P_CREAT)
#define get_number(Pid)   GETBITS(Pid,10,P_NUMBER)


/*
**  Port and ref layout
**
** Tag  Node   Number           Creat
** +------------------------------------+
** |  4 |    8   |  18              | 2 |
** +------------------------------------+
**
*/
#define R_NUMBER	18

/* Maximum number of references in the system */
#define MAX_REFERENCE  (1 << R_NUMBER)

#define get_node_reference(X)   GETBITS(X,20,P_NODE)
#define get_number_reference(X) GETBITS(X,2,R_NUMBER)

#define make_port2(Node,Number) \
 ((uint32) (PORT | ((Node) << 20) | ((Number) << 2) | ORIG_CREATION))

#define make_port(Number) make_port2(node, Number)


#define make_refer(Node,Number) \
 ( (uint32) (REFER | ((Node) << 20) | ((Number)  << 2) | ORIG_CREATION))

/* note that the get_creation works on
   ports and ref's as well as on pids */
#else

/* pid layout
**
**    Serial  Number     Node    Creat  Tag 
**   +---------------------------------------+
**   |  3  |  15       |   8    |   2 |  4   |
**   +---------------------------------------+
*/

#define make_pid3(Ser,Node,Number,Creation) \
  ((uint32) (PID | ((Ser) << 29) | ((Number) << 14) | \
	     ((Node) << 6) | ((Creation) << 4)))

#define make_pid(Ser,Node,Number) \
  make_pid3(Ser,Node,Number,ORIG_CREATION)

#define GETBITS(X,Pos,Size) (((X) >> (Pos)) & ~(~0 << (Size)))

/* This macro get Size bits starting at low order position Pos
   and adjusts the bits to the right 
   bits are numbered from 1 - 32 */


#define get_serial(Pid)   GETBITS(Pid,29,P_SERIAL)
#define get_node(Pid)     GETBITS(Pid,6,P_NODE)
#define get_creation(Pid) GETBITS(Pid,4,P_CREAT)
#define get_number(Pid)   GETBITS(Pid,14,P_NUMBER)

/*
**  Port and ref layout
**
**  Node   Number           Creat   Tag
** +------------------------------------+
** |   8   |  18              | 2 |  4  |
** +------------------------------------+
**
*/
#define R_NUMBER	18

/* Maximum number of references in the system */
#define MAX_REFERENCE  (1 << R_NUMBER)

#define get_node_reference(X)   GETBITS(X,24,P_NODE)
#define get_number_reference(X) GETBITS(X,6,R_NUMBER)

#define make_port2(Node,Number) \
 ((uint32) (PORT | ((Node) << 24) | ((Number) << 6) | \
           ((ORIG_CREATION) << 4)))

#define make_port(Number) make_port2(node, Number)

#define make_refer(Node,Number) \
 ((uint32) (REFER | ((Node) << 24) | ((Number)  << 6) | \
           ((ORIG_CREATION) << 4)))

#endif

#endif
