/* ``The contents of this file are subject to the Erlang Public License,
 * Version 1.0, (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.erlang.org/EPL1_0.txt
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 * 
 * The Original Code is Erlang-4.7.3, December, 1998.
 * 
 * The Initial Developer of the Original Code is Ericsson Telecom
 * AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
 * Telecom AB. All Rights Reserved.
 * 
 * Contributor(s): ______________________________________.''
 */
/*
 * Author: Bjorn Gustavsson
 */

#include "sys.h"
#include "config.h"
#include "global.h"
#include "erl_process.h"
#include "error.h"
#include "driver.h"
#include "bif.h"
#include "external.h"
#include "beam_opcodes.h"

#define MAX_OPARGS 8
#define CALLED    0
#define DEFINED   1
#define EXPORTED  2

#define MI_NUM_FUNCTIONS 0
#define MI_NUM_ATTRIBUTES 1
#define MI_FUNCTIONS 2

/*
 * XXX This should go to a header file together with the
 * companion macros in "beam_emu.c".
 */

#ifdef NO_JUMP_TABLE
#  define BeamOpCode(Op) ((uint32)(Op))
#else
#  define BeamOpCode(Op) ((uint32)beam_ops[Op])
#endif

/*
 * Errors returned from tranform_engine().
 */
#define TE_OK 0
#define TE_FAIL (-1)
#define TE_SHORT_WINDOW (-2)

typedef struct {
    uint32 value;		/* Value of label (NULL if not known yet). */
    uint32 patches;		/* Index (into code buffer) to first location
				 * which must be patched with the value of this label.
				 */
} Label;

union arg {
    uint32 u;			/* Any term or pointer. */
};

/*
 * A generic operation.
 */
typedef struct {
    int op;			/* Opcode. */
    int arity;			/* Number of arguments. */
    int type[MAX_OPARGS];	/* Type for each arguments. */
    union arg arg[MAX_OPARGS];	/* Arguments for operation. */
    uint32 sign_mask;		/* Mask for signature. */
} GenOp;


typedef struct {
    uint32 module;		/* Tagged atom for module name. */
    int mod_index;		/* Index of module name in atom table. */
    byte* bytes;		/* Pointer to code to load. */
    unsigned loaded_size;	/* Size of loaded coded (from header). */
    char module_name[256];	/* Module name as a string. */
    char function[256];		/* Current function as a string. */
    unsigned arity;		/* Arity for current function. */
    int specific_op;		/* Specific opcode (-1 if not found). */
    uint32 group_leader;	/* Group leader (for diagnostics). */

    uint32 flags;		/* Flags from header. */
    int num_functions;		/* Number of functions in module. */

    int num_labels;		/* Number of labels. */
    int code_buffer_size;	/* Size of code buffer in words.  */
    uint32* code;		/* Loaded code. */
    int ci;			/* Current index into loaded code. */
    Label* labels;

    int num_atoms;		/* Number of atoms in atom table. */
    uint32* atom;		/* Atom table. */

    int num_exps;		/* Number of exports. */
    byte* exp_table;		/* Pointer to export table (in binary). */

    int num_imports;		/* Number of imports. */
    byte* imp_table;		/* Pointer to import table (in binary). */
    uint32* import_patches;	/* All locations needed to patch. */

    byte* string_table;		/* Pointer to string table. */
    uint32 strtab_offset;	/* Offset to beginning of string table. */
    uint32 strtab_size;		/* Size of string table. */
    uint32 put_strings;		/* Linked list of put_string instructions. */

    /*
     * Generic instructions.
     */
    GenOp pushback[TE_MAX_WINDOW]; /* Pushed back generic instructions.. */
    GenOp tr_window[TE_MAX_WINDOW]; /* Instructions in transformation window. */
    int putback;		/* Number of instructions put back. */
    int win_index;		/* Index of next free slot in window or next
				 * transformed instruction.
				 */
    int left_to_load;		/* Number of instructions left to load in window. */
    GenOp* gen_op;		/* The last generic instruction seen. */
} LoaderState;

typedef struct {
    uint32* start;		/* Pointer to start of module. */
    uint32* end;		/* Points one word beyond last function in module. */
} Range;

#define LoadError0(Fmt) \
    do { \
	load_printf(__LINE__, state_p, Fmt); \
	goto load_error; \
    } while (0)

#define LoadError1(Fmt, Arg1) \
    do { \
	load_printf(__LINE__, state_p, Fmt, Arg1); \
	goto load_error; \
    } while (0)

#define LoadError2(Fmt, Arg1, Arg2) \
    do { \
	load_printf(__LINE__, state_p, Fmt, Arg1, Arg2); \
	goto load_error; \
    } while (0)

#define LoadError3(Fmt, Arg1, Arg2, Arg3) \
    do { \
	load_printf(__LINE__, state_p, Fmt, Arg1, Arg2, Arg3); \
	goto load_error; \
    } while (0)

#define GetInt(N, Dest) \
    if (bytes-orig_bytes+(N) > unloaded_size) { \
       LoadError1("unexpected end of binary (%d bytes needed)", N); \
    } else { \
       int __n = (N); \
       unsigned __result = 0; \
       while (__n-- > 0) { \
          __result = __result << 8 | *bytes++; \
       } \
       Dest = __result; \
    } while (0)

#define GetByte(Dest) \
    if (bytes-orig_bytes+1 > unloaded_size) { \
       LoadError0("unexpected end of binary (1 byte needed)"); \
    } else { \
       Dest = *bytes++; \
    }

#define GetTagAndValue(Tag, Val) \
   do { \
      uint32 __w; \
      GetByte(__w); \
      Tag = __w & 0x07; \
      if ((__w & 0x08) == 0) { \
	 Val = __w >> 4; \
      } else if ((__w & 0x10) == 0) { \
	 Val = ((__w >> 5) << 8); \
	 GetByte(__w); \
	 Val |= __w; \
      } else { \
	GetInt(__w >> 5, Val); \
      } \
   } while (0)

#define VerifyTag(Actual, Expected) \
    if (Actual != Expected) { \
       LoadError2("bad tag %d; expected %d", Actual, Expected);	\
    } else {}

#define UPDATE_SIGN(op_p, tag) \
    (op_p)->sign_mask = ((op_p)->sign_mask << 8) | (1 << (tag))

#define Need(w) \
    ASSERT(ci <= code_buffer_size); \
    if (code_buffer_size < ci+(w)) { \
        code_buffer_size = next_heap_size(ci+(w), 0); \
	code = (uint32 *) safe_realloc((char *) code, code_buffer_size * sizeof(uint32)); \
    }

static FUNCTION(void, delete_code, (int));
static FUNCTION(void, delete_export_references, (int));
static int setup_for_loading(LoaderState* state_p, uint32 group_leader,
			     uint32 module_to_load, byte* bytes, int unloaded_size);
static int finish_loading(LoaderState* state_p);
static void load_printf(int line, LoaderState* context, char *fmt, ...);
static int transform_engine(LoaderState* st, uint32* pc);

/*
 * Offsets of header fields.
 */
#define BH_VERSION 6
#define BH_FLAGS 8
#define BH_ATTRIBUTES 12
#define BH_CODE_START 16
#define BH_NUM_FUNCTIONS 20
#define BH_NUM_LABELS 22
#define BH_NUM_ATOMS 26
#define BH_NUM_EXPORTS 28
#define BH_NUM_IMPORTS 30
#define BH_STRING_SIZE 32
#define BH_ATOM_TABLE 36
#define BH_EXPORT_TABLE 40
#define BH_IMPORT_TABLE 44
#define BH_STRING_TABLE 48

#define HEADER_SIZE 52

#define FLAG_TRACE 1

static char beam_magic[6] = {0x7F, 'B', 'E', 'A', 'M', '!'};
static int must_swap_floats;

/*
 * The following variables keep a sorted list of address ranges for
 * each module.  It allows us to quickly find a function given an
 * instruction pointer.
 */
static Range* modules = NULL;	/* Sorted lists of module addresses. */
static int num_loaded_modules;	/* Number of loaded modules. */
static int allocated_modules;	/* Number of slots allocated. */

/**********************************************************************/

/* Cleaup all code & relase fix allocator */
static void atexit_load(arg)
void* arg;
{
    /* XXX Nothing is freed here. */
}

void init_load(void)
{
    int i;
    int res;
    Preload* preload_p;
    uint32 module_name;
    byte* code;
    char* name;
    int length;
    FloatDef f;

    erl_at_exit(atexit_load, NULL);

    f.fd = 1.0;
    must_swap_floats = (f.fw[0] == 0);

    if ((preload_p = sys_preloaded()) == NULL)
	return;

    allocated_modules = 128;
    modules = (Range *) sys_alloc(allocated_modules * sizeof(Range));
    num_loaded_modules = 0;

    i = 0;
    while ((name = preload_p[i].name) != NULL) {
	length = preload_p[i].size;
	module_name = am_atom_put(name, sys_strlen(name));
	if ((code = sys_preload_begin(&preload_p[i])) == 0)
	    erl_exit(1, "Failed to find preloaded code for module %s\n", 
		     name);
	res = do_load(0, module_name, code, length);
	sys_preload_end(&preload_p[i]);
	if (res < 0)
	    erl_exit(1,"Failed loading preloaded module %s\n", name);
	i++;
    }
}

/* Move code from current to old */
static void delete_code(mod_index)
int mod_index;
{
    module_code(mod_index)->old_code = module_code(mod_index)->code;
    module_code(mod_index)->old_code_length = module_code(mod_index)->code_length;
    
    module_code(mod_index)->code = NULL;
    module_code(mod_index)->code_length = 0;
}

/* null all references on the export table for the module called with the
   atom index below */

static void delete_export_references(module)
int module;
{
    int i;

    for (i = 0; i < export_list_size; i++) {
        if (export_list(i) != NULL && (export_list(i)->module == module)) {
	    export_list(i)->address = &(export_list(i)->op_error_handler);
	}
    }
}

/**********************************************************************/

int purge_module(module)
int module;
{
    uint32* code;
    int mod_index;
    int i;

    /* correct module? */
    if ((mod_index = module_get(module)) == -1)
	return -2;

    /*
     * Any code to purge?
     */
    if (module_code(mod_index)->old_code == 0) {
	if (display_loads) {
	    erl_printf(COUT,"No code to purge for ");
	    print_atom(module, COUT);
	    erl_printf(COUT,"\n");
	}
	return -1;
    }
    if (display_loads) {
	erl_printf(COUT,"Purging code for ");
	print_atom(module, COUT);
	erl_printf(COUT,"\n");
    }

    /*
     * Remove the old code.
     */
    code = module_code(mod_index)->old_code;
    sys_free((char *) code);
    module_code(mod_index)->old_code = NULL;
    module_code(mod_index)->old_code_length = 0;

    /*
     * Remove the code from the address table too.
     */
    for (i = 0; i < num_loaded_modules; i++) {
	if (modules[i].start == code) {
	    num_loaded_modules--;
	    while (i < num_loaded_modules) {
		modules[i] = modules[i+1];
		i++;
	    }
	    return 0;
	}
    }

    ASSERT(0);			/* Not found? */
    return 0;
}

/**********************************************************************/
/*
 * Check that a process is executing (or has return value to) code in an old
 * version of module.
 */

int check_process_code(cp, pid, module)
Process* cp; uint32 pid; int module;
{
    Process *p;
    int mod_index;
    uint32* start;
    uint32* end;
    uint32* sp;

#define INSIDE(a) (start <= (a) && (a) < end)

    /* Check that pid is a valid process */
    if (is_not_pid(pid) || (get_node(pid) != THIS_NODE) ||
	(get_number(pid) >= max_process))
	return -1;
    p = process_tab[get_number(pid)];
    if (INVALID_PID(p, pid))
	return 0;

    /* Does module exist? */
    if ((mod_index = module_get(module)) == -1)
	return 0;
    
    /* Any old version of code? */
    if (module_code(mod_index)->old_code == 0)
	return 0;

    /*
     * Pick up limits for the module.
     */
    start = module_code(mod_index)->old_code;
    end = (uint32 *)((char *)start + module_code(mod_index)->old_code_length);

    /*
     * Check if current instruction or continuation pointer points into module.
     */
    if (INSIDE(p->i) || INSIDE(p->cp))
	return 1;

    /*
     * Check all continuation pointers stored on the stack.
     */
    for (sp = p->stop; sp < p->stack; sp++) {
	if (is_CP(*sp) && INSIDE(cp_ptr_val(*sp)))
	    return 1;
    }
    return 0;
#undef INSIDE
}

/**********************************************************************/
/*
 * delete module does not delete the module, what it does is to delete the
 * export refences to the module Import references in all compiled code are
 * reset e.g. refences to the module are removed. If you wish to delete a
 * module you must first do delete module and then purge_module (with the
 * appropriate check_process_code calls
 */

int delete_module(group_leader, module)
uint32 group_leader; int module;
{
    int mod_index;
    
    if ((mod_index = module_get(module)) == -1)
	return 1;

    if (module_code(mod_index)->old_code != 0) {
	erl_printf(CBUF, "Module ");
	print_atom(module, CBUF);
	erl_printf(CBUF, " must be purged before loading\n");
	send_error_to_logger(group_leader);
	return 2;
    }

    delete_export_references(module); /* delete current references */
    delete_code(mod_index);           /* move current to old */
    return 0;
}

/* 
 *  Compare function for qsort
 */
static int
intcompare(i,j)
    int *i, *j;
{
    return(*i - *j);
}

static int
setup_for_loading(state_p, group_leader, module_to_load, bytes, unloaded_size)
LoaderState* state_p;		/* Pointer to state for loader. */
uint32 group_leader;		/* Group leader to use for error reporting. */
uint32 module_to_load;		/* Tagged atom for module name. */
byte* bytes;			/* Pointer to code for module. */
int unloaded_size;		/* Size of code. */
{
    int this_module;		/* Module index for module to load. */
    Atom* ap;			/* Pointer to atom. */
    unsigned version;		/* Version of module. */
    int atom_table_offset;	/* Offset of unloaded atom table. */
    byte* atoms;		/* Pointer to unloaded atoms. */
    byte* s;			/* General string pointer. */
    int i;
    int seen_eq;

    state_p->mod_index = unsigned_val(module_to_load);
    state_p->module = module_to_load;

    /*
     * Check if the previous code has been already deleted;
     * if not, delete old code; error if old code already exists.
     */
    if ((this_module = module_get(state_p->mod_index)) != -1) {
	if ((module_code(this_module)->code != 0) &&
	    (module_code(this_module)->old_code != 0)) {
	    erl_printf(CBUF, "Module ");
	    print_atom(state_p->mod_index, CBUF);
	    erl_printf(CBUF, " must be purged before loading\n");
	    send_error_to_logger(group_leader);
	    return -3;
	} else if (module_code(this_module)->old_code == 0) {
	    /*
	     * Make the current version old.
	     */
	    if (display_loads) {
		erl_printf(COUT, "saving old code\n");
	    }
	    delete_code(this_module);
	    delete_export_references(state_p->mod_index);
	}
    }

    /*
     * Initialize context for error diagnostics.
     */

    ap = atom_tab(state_p->mod_index);
    memcpy(state_p->module_name, ap->name, ap->len);
    state_p->module_name[ap->len] = 0;
    state_p->function[0] = 0;	/* Function not known yet */
    state_p->arity = 0;
    state_p->gen_op = 0;
    state_p->specific_op = -1;
    state_p->group_leader = group_leader;
    state_p->putback = 0;
    state_p->win_index = 0;
    state_p->left_to_load = 0;
    state_p->gen_op = NULL;
    state_p->atom = NULL;
    state_p->import_patches = NULL;

    if (unloaded_size < HEADER_SIZE) {
	LoadError0("module too short");
    }

    /*
     * Check header for magic and correct version.
     */
    if (memcmp(bytes, beam_magic, sizeof(beam_magic)) != 0) {
	LoadError0("not a Beam file (bad magic)");
    }
    version = bytes[BH_VERSION] << 8 | bytes[BH_VERSION+1];
    if (version != BEAM_FORMAT_NUMBER) {
	LoadError2("wrong version %d; expected %d", version, BEAM_FORMAT_NUMBER);
    }
    state_p->num_functions = get_int16(bytes+BH_NUM_FUNCTIONS);
    state_p->flags = get_int32(bytes+BH_FLAGS);

    /*
     * Read the atom table.
     */
    
    state_p->num_atoms = get_int16(bytes+BH_NUM_ATOMS);
    atom_table_offset = get_int32(bytes+BH_ATOM_TABLE);
    if (atom_table_offset >= unloaded_size) {
	LoadError1("invalid offset to atom table: %d", atom_table);
    }
    if (atom_table_offset + state_p->num_atoms > unloaded_size) {
	LoadError1("absurd number of atoms: %d", state_p->num_atoms);
    }
    state_p->atom = safe_alloc(next_heap_size(state_p->num_atoms * sizeof(uint32), 0));
    atoms = bytes + atom_table_offset;
    for (i = 0; i < state_p->num_atoms; i++) {
	uint32 n;
	if (atoms + 1 > bytes + unloaded_size) {
	    LoadError1("short atom table (reading length of atom #%d)", i);
	}
	n = *atoms;
	if (atoms + n + 1 > bytes + unloaded_size) {
	    LoadError1("short atom table (reading atom #%d)", i);
	}
	state_p->atom[i] = am_atom_put(atoms+1, n);
	atoms += 1 + n;
    }

    /*
     * Check the module name.
     */
    if (state_p->atom[0] != module_to_load) {
	char sbuf[256];

	ap = atom_tab(unsigned_val(state_p->atom[0]));
	memcpy(sbuf, ap->name, ap->len);
	sbuf[ap->len] = '\0';
	LoadError1("module name in object code is %s", sbuf);
    }

    state_p->num_labels = get_int32(bytes+BH_NUM_LABELS);

    /*
     * Validate the export table.
     */
    state_p->num_exps = get_int16(bytes+BH_NUM_EXPORTS);
    i = get_int32(bytes+BH_EXPORT_TABLE);
    if (i >= unloaded_size) {
	LoadError1("start of export table '%d' outside module", i);
    }
    if (i + state_p->num_exps * 8 > unloaded_size) {
	LoadError0("export table extends beyond end of module");
    }
    state_p->exp_table = bytes + i;
    for (i = 0; i < state_p->num_exps; i++) {
	uint32 n;
	n = get_int16(state_p->exp_table+i*8);
	if (n >= state_p->num_atoms) {
	    LoadError2("export entry %d: invalid atom number %d", i, n);
	}
	n = get_int16(state_p->exp_table+i*8+2);
	if (n > MAX_REG) {
	    LoadError2("export entry %d: invalid arity %d", i, n);
	}
	n = get_int32(state_p->exp_table+i*8+4);
	if (n >= state_p->num_labels) {
	    LoadError2("export entry %d: invalid label %d", i, n);
	}
    }

    /*
     * Validate the import table.
     */
    state_p->num_imports = get_int16(bytes+BH_NUM_IMPORTS);
    i = get_int32(bytes+BH_IMPORT_TABLE);
    if (i >= unloaded_size) {
	LoadError1("start of import table '%d' outside module", i);
    }
    if (i + state_p->num_imports * 6 > unloaded_size) {
	LoadError0("import table extends beyond end of module");
    }
    state_p->imp_table = bytes + i;
    state_p->import_patches = safe_alloc(next_heap_size(state_p->num_imports
							* sizeof(uint32), 0));
    for (i = 0; i < state_p->num_imports; i++) {
	uint32 n;

	state_p->import_patches[i] = 0;
	n = get_int16(state_p->imp_table+i*6);
	if (n >= state_p->num_atoms) {
	    LoadError2("import entry %d: invalid atom number %d", i, n);
	}
	n = get_int16(state_p->imp_table+i*6+2);
	if (n >= state_p->num_atoms) {
	    LoadError2("import entry %d: invalid atom number %d", i, n);
	}
	n = get_int16(state_p->imp_table+i*6+4);
	if (n > MAX_REG) {
	    LoadError2("import entry %d: invalid arity %d", i, n);
	}
    }

    /*
     * Validate the string table.
     */
    state_p->strtab_offset = get_int32(bytes+BH_STRING_TABLE);
    state_p->strtab_size = get_int32(bytes+BH_STRING_SIZE);
    if (state_p->strtab_offset > unloaded_size ||
	state_p->strtab_offset + state_p->strtab_size > unloaded_size) {
	LoadError2("string table (offset %d, size %d) outside module",
		   state_p->strtab_offset, state_p->strtab_size);
    }
    state_p->put_strings = 0;
    state_p->string_table = bytes + state_p->strtab_offset;

    /*
     * Initialize label table.
     */
    state_p->labels = (Label *) safe_alloc(state_p->num_labels * sizeof(Label));
    for (i = 0; i < state_p->num_labels; i++) {
	state_p->labels[i].value = 0;
	state_p->labels[i].patches = 0;
    }

    /*
     * Validate code start.
     */
    i = get_int32(bytes+BH_CODE_START);
    if (i >= unloaded_size) {
	LoadError1("code start %d outside module", i);
    }
    state_p->bytes = bytes + i;

    state_p->code_buffer_size = next_heap_size(2048 + state_p->num_functions, 0);
    state_p->code = (uint32*) safe_alloc(sizeof(uint32) * state_p->code_buffer_size);

    state_p->code[MI_NUM_FUNCTIONS] = state_p->num_functions;
    state_p->ci = MI_FUNCTIONS + state_p->num_functions + 1;

    /*
     * Validate and read the attribute table.
     */
    i = get_int32(bytes+BH_ATTRIBUTES);
    if (i >= unloaded_size) {
	LoadError1("attribute table outside module", i);
    }
    s = bytes + i;
    seen_eq = 0;
    for (i = 0; *s; i++) {
	byte* before = s;
	
	if (state_p->ci >= state_p->code_buffer_size) {
	    LoadError0("absurd number of attributes");
	}
	if (!seen_eq) {
	    while (s-bytes < unloaded_size && *s != '=') {
		s++;
	    }
	    seen_eq = 1;
	} else {
	    while (s-bytes < unloaded_size && *s != '\0') {
		s++;
	    }
	    seen_eq = 0;
	}

	state_p->code[state_p->ci++] = am_atom_put(before, s-before);
	s++;
    }
    state_p->code[MI_NUM_ATTRIBUTES] = i;
    return 1;

 load_error:
    return 0;
}

int
bin_load(group_leader, module_to_load, bytes, unloaded_size)
uint32 group_leader;		/* Group leader to use for error reporting. */
uint32 module_to_load;		/* Tagged atom for module name. */
byte* bytes;			/* Pointer to code for module. */
int unloaded_size;		/* Size of code. */
{
    LoaderState state;
    LoaderState* state_p = &state; /* For LoadError macros. */
    int i;
    int tmp;
    int ci;
    byte* orig_bytes = bytes;
    char* sign;
    int arg;			/* Number of current argument. */
    int num_specific;		/* Number of specific ops for current. */
    int tag;			/* Current tag. */
    uint32* code;
    int code_buffer_size;
    int specific;
    uint32 last_label = 0;	/* Number of last label. */
    uint32 function_number = 0;

    if (!setup_for_loading(&state, group_leader, module_to_load, bytes, unloaded_size))
	return -1;

    code = state.code;
    code_buffer_size = state.code_buffer_size;
    bytes = state.bytes;
    ci = state.ci;
    
    for (;;) {
	int new_op;
	GenOp* cur_op;		/* Current operation. */
	int transform_me;	/* Transform current operation. */

	/*
	 * Get a new generic instruction from one of the following sources:
	 * [1] From tr_window[], if a previous transformation left a result there
	 *     (left_to_load > 0).
	 * [2] From the pushback buffer (putback > 0); instructions were placed
	 *     there by an unsuccessful match operation.
	 * [3] From the binary.
	 */

	ASSERT(ci <= code_buffer_size);

    get_next_instr:
	if (state.left_to_load > 0) {
	    state.gen_op = state.tr_window + state.win_index++;
	    state.left_to_load--;
	    if (state.left_to_load == 0) { /* Last saved. */
		state.win_index = 0;
	    }
	    transform_me = 0;
	} else if (state.putback > 0) {
	    state.tr_window[state.win_index] = state.pushback[--state.putback];
	    state.gen_op = state.tr_window + state.win_index++;
	    transform_me = 1;
	} else {
	    GetByte(new_op);
	    if (new_op >= NUM_GENERIC_OPS) {
		LoadError1("invalid opcode %d", new_op);
	    }
	    if (gen_opc[new_op].name[0] == '\0') {
		LoadError1("invalid opcode %d", new_op);
	    }
	    state.gen_op = state.tr_window+state.win_index++;
	    cur_op = state.gen_op;
	    cur_op->op = new_op;
	    transform_me = 1;

	    /*
	     * Read all arguments for the current operation.
	     */
	    cur_op->arity = gen_opc[cur_op->op].arity;
	    ASSERT(cur_op->arity <= MAX_OPARGS);
	    cur_op->sign_mask = 0;
	    for (arg = 0; arg < cur_op->arity; arg++) {
		uint32 val;

		GetTagAndValue(tag, val);
		cur_op->type[arg] = tag;
		switch (cur_op->type[arg]) {
		case TAG_r:
		case TAG_n:
		case TAG_x:
		case TAG_i:
		    cur_op->arg[arg].u = val;
		    break;
		case TAG_y:
		    cur_op->arg[arg].u = val + 1;
		    break;
		case TAG_a:
		    if (val  >= state.num_atoms) {
			LoadError1("bad atom index: %d", val);
		    }
		    cur_op->arg[arg].u = state.atom[val];
		    break;
		case TAG_f:
		case TAG_p:
		    if (val  >= state.num_labels) {
			LoadError1("bad label: %d", val);
		    }
		    cur_op->arg[arg].u = val;
		    break;
		}
		UPDATE_SIGN(cur_op, tag);
	    }
	}
	
	/*
	 * Unless we are loading the results of a previous transformation,
	 * we will try to transform the instructions in the window.
	 */

	if (transform_me) {
	    int tr;

	    tr = gen_opc[state.tr_window[0].op].transform;
	    if (tr == -1) {	/* No transformation, simply load. */
		state.win_index = 0;
	    } else if (state.win_index < gen_opc[state.tr_window[0].op].min_window) {
		goto get_next_instr;
	    } else if (!transform_engine(&state, op_transform+tr)) {
		goto get_next_instr;
	    }
	    state.gen_op = state.tr_window;
	}
	
	/*
	 * From the collected generic instruction, find the specific
	 * instruction.
	 */
	
	cur_op = state.gen_op;
	specific = gen_opc[cur_op->op].specific;
	num_specific = gen_opc[cur_op->op].num_specific;
	for (i = 0; i < num_specific; i++) {
	    if ((opc[specific].mask & cur_op->sign_mask) == cur_op->sign_mask) {
		break;
	    }
	    specific++;
	}
	if (i == num_specific) {
	    state.specific_op = -1;
	    LoadError0("no specific operation found");
	}

	state.specific_op = specific;
	Need(opc[state.specific_op].sz+2); /* Extra margin for packing */
	code[ci++] = BeamOpCode(state.specific_op);
	
	/*
	 * Load the found specific operation.
	 */
	sign = opc[state.specific_op].sign;
	ASSERT(sign != NULL);
	arg = 0;
	while (*sign) {
	    ASSERT(arg < cur_op->arity);
	    tag = cur_op->type[arg];
	    switch (*sign) {
	    case 'r':	/* x(0) */
	    case 'n':	/* Nil */
		VerifyTag(tag_to_letter[tag], *sign);
		break;
	    case 'x':	/* x(N) */
	    case 'y':	/* y(N) */
		VerifyTag(tag_to_letter[tag], *sign);
		code[ci++] = cur_op->arg[arg].u << 2;
		break;
	    case 'a':	/* Tagged atom */
		VerifyTag(tag_to_letter[tag], *sign);
		code[ci++] = cur_op->arg[arg].u;
		break;
	    case 'i':	/* Tagged integer */
		VerifyTag(tag_to_letter[tag], *sign);
		code[ci++] = make_small(cur_op->arg[arg].u);
		break;
	    case 'c':	/* Tagged constant */
		switch (tag) {
		case TAG_i:
		    code[ci++] = make_small(cur_op->arg[arg].u);
		    break;
		case TAG_a:
		    code[ci++] = cur_op->arg[arg].u;
		    break;
		case TAG_n:
		    code[ci++] = NIL;
		    break;
		default:
		    LoadError1("bad tag %d for tagged constant",
			       cur_op->type[arg]);
		    break;
		}
		break;
	    case 's':	/* Any source (tagged constant or register) */
		switch (tag) {
		case TAG_r:
		    code[ci++] = make_rreg();
		    break;
		case TAG_x:
		    code[ci++] = make_xreg(cur_op->arg[arg].u << 2);
		    break;
		case TAG_y:
		    code[ci++] = make_yreg(cur_op->arg[arg].u << 2);
		    break;
		case TAG_i:
		    code[ci++] = make_small(cur_op->arg[arg].u);
		    break;
		case TAG_a:
		    code[ci++] = cur_op->arg[arg].u;
		    break;
		case TAG_n:
		    code[ci++] = NIL;
		    break;
		default:
		    LoadError1("bad tag %d for general source",
			       cur_op->type[arg]);
		    break;
		}
		break;
	    case 'd':	/* Destination (x(0), x(N), y(N) */
		switch (tag) {
		case TAG_r:
		    code[ci++] = make_rreg();
		    break;
		case TAG_x:
		    code[ci++] = make_xreg(cur_op->arg[arg].u << 2);
		    break;
		case TAG_y:
		    code[ci++] = make_yreg(cur_op->arg[arg].u << 2);
		    break;
		default:
		    LoadError1("bad tag %d for destination",
			       cur_op->type[arg]);
		    break;
		}
		break;
	    case 'I':	/* Untagged integer. */
		VerifyTag(tag_to_letter[tag], 'i');
		code[ci++] = cur_op->arg[arg].u;
		break;
	    case 'A':	/* Arity value. */
		VerifyTag(tag, TAG_i);
		code[ci++] = make_arityval(cur_op->arg[arg].u);
		break;
	    case 'f':		/* Destination label */
	    case 'p':		/* Pointer (to label) */
		VerifyTag(tag_to_letter[tag], *sign);
		code[ci] = state.labels[cur_op->arg[arg].u].patches;
		state.labels[cur_op->arg[arg].u].patches = ci;
		ci++;
		break;
	    case 'j':		/* 'f' or 'p' */
		if (tag != TAG_f && tag != TAG_p) {
		    LoadError3("bad tag %d; expected %d or %d", tag, TAG_f, TAG_p);
		}
		code[ci] = state.labels[cur_op->arg[arg].u].patches;
		state.labels[cur_op->arg[arg].u].patches = ci;
		ci++;
		break;
	    case 'L':		/* Define label */
		ci--;		/* Remove label from loaded code */
		ASSERT(state.specific_op == op_label_L);
		VerifyTag(tag, TAG_i);
		last_label = cur_op->arg[arg].u;
		if (!(last_label < state.num_labels)) {
		    LoadError1("invalid label %d", cur_op->arg[arg].u);
		}
		state.labels[last_label].value = ci;
		ASSERT(state.labels[last_label].patches < ci);
		break;
	    case 'e':		/* Export entry */
		VerifyTag(tag, TAG_i);
		if (cur_op->arg[arg].u >= state_p->num_imports) {
		    LoadError1("invalid import table index %d", cur_op->arg[arg].u);
		}
		code[ci] = state.import_patches[cur_op->arg[arg].u];
		state.import_patches[cur_op->arg[arg].u] = ci;
		ci++;
		break;
	    case 'b':
		VerifyTag(tag, TAG_i);
		code[ci++] = (uint32) bif_table[cur_op->arg[arg].u].f;
		break;
	    case 'N':		/* Big number (arity and sign) */
		{
		    uint32 arity;

		    VerifyTag(tag, TAG_i);
		    arity = make_thing(cur_op->arg[arg].u >> 1);
		    if (cur_op->arg[arg].u & 1) { /* Negative. */
			arity |= BIG_SIGN_BIT;
		    }
		    code[ci++] = arity;
		}
		break;
	    case 'u':		/* Big number (part of value) */
		VerifyTag(tag, TAG_i);
		tmp = cur_op->arg[arg].u;
		(((unsigned short *)(code+ci)))[0] = tmp & 0xffff;
		(((unsigned short *)(code+ci)))[1] = tmp >> 16;
		ci++;
		break;
	    case 'P':		/* Byte offset into tuple */
		VerifyTag(tag, TAG_i);
		tmp = cur_op->arg[arg].u;
		code[ci++] = (uint32) ((cur_op->arg[arg].u+1) * sizeof(uint32 *));
		break;
	    default:
		LoadError1("bad argument tag: %d", *sign);
	    }
	    sign++;
	    arg++;
	}

	/*
	 * The packing engine.
	 */
	if (opc[state.specific_op].pack[0]) {
	    char* prog;		/* Program for packing engine. */
	    uint32 stack[8];	/* Stack. */
	    uint32* sp = stack;	/* Points to next free position. */
	    uint32 packed = 0;	/* Accumulator for packed operations. */
	    
	    for (prog = opc[state.specific_op].pack; *prog; prog++) {
		switch (*prog) {
		case 'g':	/* Get instruction; push on stack. */
		    *sp++ = code[--ci];
		    break;
		case 'i':	/* Initialize packing accumulator. */
		    packed = code[--ci];
		    break;
		case '0':	/* Shift 10 steps */
		    packed = (packed << 10) | code[--ci];
		    break;
		case '2':	/* Shift 12 steps */
		    packed = (packed << 12) | code[--ci];
		    break;
		case '6':	/* Shift 16 steps */
		    packed = (packed << 16) | code[--ci];
		    break;
		case 'p':	/* Put instruction (from stack). */
		    code[ci++] = *--sp;
		    break;
		case 'P':	/* Put packed operands. */
		    *sp++ = packed;
		    packed = 0;
		    break;
		default:
		    ASSERT(0);
		}
	    }
	    ASSERT(sp == stack); /* Incorrect program? */
	}

	/*
	 * Handle a few special cases.
	 */
	switch (state.specific_op) {
	case op_func_info_aaI:
	    {
		Atom* ap;
		uint32 offset;

		if (function_number >= state_p->num_functions) {
		    LoadError1("too many functions in module (header said %d)",
			       state_p->num_functions); 
		}

		/*
		 * Save context for error messages.
		 *
		 * XXX This could be done smarter.
		 */
		ap = atom_tab(unsigned_val(code[ci-2]));
		memcpy(state.function, ap->name, ap->len);
		state.function[ap->len] = 0;
		state.arity = code[ci-1];

		offset = MI_FUNCTIONS + function_number;
		code[offset] = state.labels[last_label].patches;
		state.labels[last_label].patches = offset;
		function_number++;
#ifdef DEBUG
		ASSERT(state.labels[0].patches == 0); /* Should not be referenced. */
		for (i = 1; i < state.num_labels; i++) {
		    ASSERT(state.labels[i].patches < ci);
		}
#endif
	    }
	    break;

	case op_put_string_IId:
	    {
		/*
		 * At entry:
		 *
		 * code[ci-4]	&&lb_put_string_IId
		 * code[ci-3]	length of string
		 * code[ci-2]   offset into string table
		 * code[ci-1]   destination register
		 *
		 * Since we don't know the address of the string table yet,
		 * just check the offset and length for validity, and use
		 * the instruction field as a link field to link all put_string
		 * instructions into a single linked list.  At exit:
		 *
		 * code[ci-4]	pointer to next put_string instruction (or 0
		 *		if this is the last)
		 */
		uint32 offset = code[ci-2];
		uint32 len = code[ci-3];
		if (offset > state.strtab_size || offset + len > state.strtab_size) {
		    LoadError2("invalid string reference %d, size %d", offset, len);
		}
		code[ci-4] = state.put_strings;
		state.put_strings = ci - 4;
	    }
	    break;
	case op_put_float_IIr:
	    /*
	     * code[ci-3]   &&lb_put_float_IIr
	     * code[ci-2]   high word
	     * code[ci-1]   low word
	     */
	    if (must_swap_floats) {
		uint32 t = code[ci-2];
		code[ci-2] = code[ci-1];
		code[ci-1] = t;
	    }
	    break;
	case op_put_float_IIx:
	case op_put_float_IIy:
	    /*
	     * code[ci-4]   &&lb_put_float_*
	     * code[ci-3]   high word
	     * code[ci-2]   low word
	     * code[ci-1]   destination register
	     */
	    if (must_swap_floats) {
		uint32 t = code[ci-3];
		code[ci-3] = code[ci-2];
		code[ci-2] = t;
	    }
	    break;
	case op_atom_switch_sI:
	    {
		int size;
		uint32 *start;

		size = code[ci-1];
		Need(size*2);
		start = &code[ci];

		/*
		 * Pick up all atoms and label numbers.
		 */
		for (i = 0; i < size; i++) {
		    uint32 tag;
		    uint32 val;

		    GetTagAndValue(tag, val);
		    VerifyTag(tag, TAG_a);
		    if (val  >= state.num_atoms) {
			LoadError1("bad atom index: %d", val);
		    }
		    code[ci++] = state.atom[val];
		    GetTagAndValue(tag, code[ci]);
		    VerifyTag(tag, TAG_p);
		    ci++;
		}
		
		/* 
		 * Sort list of atoms so we can do a binary search.
		 */
		qsort(start, size, 2 * sizeof(uint32), intcompare);

#ifdef DEBUG
		/*
		 * Verify that the atoms were correctly sorted.
		 */
		for (i = 0; i < size-1; i++) {
		    ASSERT(start[2*i] < start[2*i+2]);
		}
#endif

		/*
		 * Go through the list and register the label references.
		 * (This must be done after the sort.)
		 */
		for (i = 0; i < 2*size; i += 2) {
		    int label = code[ci-i-1];
		    code[ci-i-1] = state.labels[label].patches;
		    state.labels[label].patches = ci-i-1;
		}
	    }
	    break;
	case op_lookup_switch_sI:
	    {
		int size;
		uint32 *start;

		size = code[ci-1];
		Need(size*2);
		start = &code[ci];
		i = size;
		while (i-- > 0) {
		    uint32 tag;

		    GetTagAndValue(tag, code[ci]);
		    VerifyTag(tag, TAG_i);
		    ci++;
		    GetTagAndValue(tag, tmp);
		    VerifyTag(tag, TAG_p);
		    code[ci] = state.labels[tmp].patches;
		    state.labels[tmp].patches = ci;
		    ci++;
		}
	    }
	    break;
	case op_jmp_switch_sI:
	    {
		int size;
		uint32 *start;

		size = code[ci-1];
		Need(size);

		start = &code[ci];
		i = size;
		while (i-- > 0) {
		    GetTagAndValue(tag, tmp);
		    VerifyTag(tag, TAG_p);
		    code[ci] = state.labels[tmp].patches;
		    state.labels[tmp].patches = ci;
		    ci++;
		}
	    }
	    break;
	case op_element_jIsd:
	    /*
	     * Make sure that this instruction is safe.
	     */
	    if (code[ci-3] < 1) {
		code[ci-5] = BeamOpCode(op_element_jssd);
		code[ci-3] = make_small(code[ci-3]);
	    }
	    break;

	case op_int_code_end:
	    state.code = code;
	    state.code_buffer_size = code_buffer_size;
	    state.ci = ci;
	    if (finish_loading(&state) != 0) {
		goto load_error;
	    }
	    return 0;
	}
    }
	
 load_error:
    if (code != 0) {
	sys_free(code);
    }
    if (state.labels != NULL) {
	sys_free(state.labels);
    }
    if (state.atom != NULL) {
	sys_free(state.atom);
    }
    if (state.import_patches != NULL) {
	sys_free(state.import_patches);
    }
    return -1;
}

static int
finish_loading(LoaderState* state_p)
{
    uint32* code = state_p->code;
    uint32 index;
    int i;
    uint32 size = state_p->ci * sizeof(uint32) + state_p->strtab_size;
    byte* str_table;

    /*
     * Move the code to its final location and place the string table after
     * the code.
     */
    code = (uint32 *) safe_realloc((char *) code, size);
    memcpy(code+state_p->ci, state_p->string_table, state_p->strtab_size);
    str_table = (byte *) (code+state_p->ci);

    /*
     * Place a pointer to the op_int_code_end instruction in the
     * function table in the beginning of the file.
     */
    code[MI_FUNCTIONS+state_p->num_functions] = (uint32) (code + state_p->ci - 1);

    /*
     * Go through all put_strings instructions, restore the pointer to
     * the instruction and convert string offsets to pointers (to the
     * last character).
     */
    index = state_p->put_strings;
    while (index != 0) {
	uint32 next = code[index];
	code[index] = BeamOpCode(op_put_string_IId);
	code[index+2] = (uint32) (str_table + code[index+2] + code[index+1] - 1);
	index = next;
    }

    /*
     * Resolve all labels.
     */
    for (i = 0; i < state_p->num_labels; i++) {
	uint32 this_patch;
	uint32 next_patch;
	uint32 value = state_p->labels[i].value;
	
	if (value == 0 && state_p->labels[i].patches != 0) {
	    LoadError1("label %d not resolved", i);
	}
	ASSERT(value < state_p->ci);
	this_patch = state_p->labels[i].patches;
	while (this_patch != 0) {
	    ASSERT(this_patch < state_p->ci);
	    next_patch = code[this_patch];
	    ASSERT(next_patch < state_p->ci);
	    code[this_patch] = (uint32) (code + value);
	    this_patch = next_patch;
	}
    }

    /*
     * Export functions.
     */
    for (i = 0; i < state_p->num_exps; i++) {
	uint32 name;
	uint32 arity;
	int label;
	uint32 value;
	uint32* address;

	name = state_p->atom[get_int16(state_p->exp_table+i*8)];
	arity = get_int16(state_p->exp_table+i*8+2);
	label = get_int32(state_p->exp_table+i*8+4);
	value = state_p->labels[label].value;
	if (value == 0) {
	    LoadError2("label %d not resolved (in export entry %d)", label, i);
	}
	address = code + value;
	export_function(state_p->module, name, arity, address);
    }


    /*
     * Import functions and patch all callers.
     */
    for (i = 0; i < state_p->num_imports; i++) {
	uint32 mod;
	uint32 func;
	uint32 arity;
	uint32 import;
	int import_index;
	uint32 current;
	uint32 next;

	mod = state_p->atom[get_int16(state_p->imp_table+i*6)];
	func = state_p->atom[get_int16(state_p->imp_table+i*6+2)];
	arity = get_int16(state_p->imp_table+i*6+4);
	import_index = export_put(unsigned_val(mod), unsigned_val(func), arity);
	import = (uint32) export_list(import_index);
	current = state_p->import_patches[i];
	while (current != 0) {
	    next = code[current];
	    code[current] = import;
	    current = next;
	}
    }

    /*
     * Update module table.
     */
    i = module_put(state_p->mod_index);
    module_code(i)->code = code;
    module_code(i)->code_length = size;
    
    sys_free(state_p->labels);
    sys_free(state_p->atom);
    sys_free(state_p->import_patches);

    /*
     * Update address table.
     */
    if (num_loaded_modules == allocated_modules) {
	allocated_modules *= 2;
	modules = (Range *) sys_realloc(modules, allocated_modules * sizeof(Range));
    }
    for (i = num_loaded_modules; i > 0; i--) {
	if (code > modules[i-1].start) {
	    break;
	}
	modules[i] = modules[i-1];
    }
    modules[i].start = code;
    modules[i].end = code + state_p->ci;
    num_loaded_modules++;
    return 0;

 load_error:
    return -1;
}

typedef struct var {
    uint32 type;
    union arg val;
} Var;

static int
transform_engine(LoaderState* st, uint32* pc)
{
    uint32 op;
    int ip;			/* Current input instruction. */
    int ap;			/* Current argument. */
    uint32* restart = pc;	/* Where to restart if current match fails. */
    Var var[TE_MAX_VARS];	/* Variables. */
    int i;			/* General index. */
    uint32 mask;
    GenOp* instr = st->tr_window;
    int num_instr = st->win_index;

 restart:
    ASSERT(restart != NULL);
    pc = restart;
    ASSERT(*pc < NUM_TOPS);	/* Valid instruction? */
#ifdef DEBUG
    restart = NULL;
#endif
    ip = ap = 0;
    for (;;) {
	op = *pc++;

	switch (op) {
	case TOP_is_op:
	    if (num_instr <= ip) {
		/*
		 * We'll need at least one more instruction to decide whether
		 * this combination matches or not.
		 */
		return 0;
	    }
	    if (*pc++ != instr[ip].op)
		goto restart;
	    break;
	case TOP_is_type:
	    mask = *pc++;

	    ASSERT(ap < instr[ip].arity);
	    ASSERT(instr[ip].type[ap] < 15);
	    if (((1 << instr[ip].type[ap]) & mask) == 0)
		goto restart;
	    break;
	case TOP_test:
	    mask = *pc++;
	    if ((st->flags & mask) != *pc++)
		goto restart;
	    break;
	case TOP_is_eq:
	    ASSERT(ap < instr[ip].arity);
	    if (*pc++ != instr[ip].arg[ap].u)
		goto restart;
	    break;
	case TOP_is_same_var:
	    ASSERT(ap < instr[ip].arity);
	    i = *pc++;
	    ASSERT(i < TE_MAX_VARS);
	    if (var[i].type != instr[ip].type[ap])
		goto restart;
	    switch (var[i].type) {
	    case TAG_r: case TAG_n: break;
	    default:
		if (var[i].val.u != instr[ip].arg[ap].u)
		    goto restart;
	    }
	    break;
	case TOP_set_var:
	    ASSERT(ap < instr[ip].arity);
	    i = *pc++;
	    ASSERT(i < TE_MAX_VARS);
	    var[i].type = instr[ip].type[ap];
	    var[i].val = instr[ip].arg[ap];
	    break;
	case TOP_next_arg:
	    ap++;
	    break;
	case TOP_next_instr:
	    ip++;
	    ap = 0;
	    break;
	case TOP_commit:
	    ip++;		/* The next_instr was optimized away. */

	    /*
	     * The left-hand side of this transformation matched.  Before storing
	     * the new instruction(s), we must push back any instructions left
	     * over beyond the ones matched (a previous transformation could have
	     * expanded the window).
	     */
	    while (num_instr > ip) {
		st->pushback[st->putback++] = st->tr_window[--num_instr];
	    }
	    ip = 0, ap = 0;
	    break;
	case TOP_store_op:
	    instr[ip].op = *pc++;
	    instr[ip].arity = *pc++;
	    instr[ip].sign_mask = 0;
	    break;
	case TOP_store_type:
	    i = *pc++;
	    UPDATE_SIGN(instr+ip, i);
	    instr[ip].type[ap] = i;
	    break;
	case TOP_store_var:
	    i = *pc++;
	    ASSERT(i < TE_MAX_VARS);
	    UPDATE_SIGN(instr+ip, var[i].type);
	    instr[ip].type[ap] = var[i].type;
	    instr[ip].arg[ap] = var[i].val;
	    break;
	case TOP_try_me_else:
	    restart = pc + 1;
	    restart += *pc++;
	    ASSERT(*pc < NUM_TOPS); /* Valid instruction? */
	    break;
	case TOP_end:
	    /*
	     * Success.
	     */
	    if (ip == 0) {		/* Success, but nothing to load. */
		st->win_index = 0;
		return 0;		/* Get a new instruction. */
	    } 

	    /*
	     * Load the first instruction, save the others (if any).
	     */
	       
	    st->left_to_load = ip-1;
	    st->win_index = (ip > 1) ? 1 : 0;
	    return 1;
	case TOP_fail:
	    /*
	     * No possible match.  Push back all instructions in the window
	     * except the first one, which will be loaded.
	     */
	    while (num_instr > 1) {
		st->pushback[st->putback++] = st->tr_window[--num_instr];
	    }
	    st->win_index = 0;
	    return 1;
	default:
	    ASSERT(0);
	}
    }
}

static void
load_printf(int line, LoaderState* context, char *fmt,...)
{
    char sbuf[1024];
    char error[1024];
    va_list va;
    char* ep = error;

    va_start(va, fmt);
    vsprintf(sbuf, fmt, va);
    va_end(va);

    sprintf(ep, "%s(%d): Error loading ", __FILE__, line);
    ep += strlen(ep);
    if (context->function[0]) {
	sprintf(ep, "function %s:%s/%d", context->module_name,
		   context->function, context->arity);
    } else {
	sprintf(ep, "module %s", context->module_name);
    }
    ep += strlen(ep);
    if (context->gen_op) {
	sprintf(ep, ": op %s", gen_opc[context->gen_op->op].name);
    }
    ep += strlen(ep);
    if (context->specific_op != -1) {
	sprintf(ep, ": %s", opc[context->specific_op].sign);
    } else if (context->gen_op) {
	int i;
	for (i = 0; i < context->gen_op->arity; i++) {
	    sprintf(ep, " %c", tag_to_letter[context->gen_op->type[i]]);
	    ep += strlen(ep);
	}
    }
    ep += strlen(ep);
    sys_printf(CBUF, "%s:\n  %s\n", error, sbuf);
#ifdef DEBUG
    sys_printf(CERR, "%s:\n  %s\n", error, sbuf);
#endif
    send_error_to_logger(context->group_leader);
}

/*
 * Builds a list of all exported functions in the given module:
 *     [{Name, Arity},...]
 *
 * Returns a tagged term, or 0 on error.
 */

uint32
exported_from_module(Process* p, /* Process whose heap to use. */
		     uint32 mod) /* Tagged atom for module. */
{
    int mod_index;
    uint32* code;
    int i;
    uint32* hp = NULL;
    uint32* hend = NULL;
    uint32 result = NIL;

    if (is_not_atom(mod))
	return 0;
    mod_index = module_get(unsigned_val(mod));
    if (mod_index == -1)
	return 0;
    code = module_code(mod_index)->code;
    for (i = code[MI_NUM_FUNCTIONS]-1; i >= 0 ; i--) {
	uint32* func_info = (uint32 *) code[MI_FUNCTIONS+i];
	uint32 name = func_info[2];
	uint32 arity = func_info[3];
	uint32 tuple;

	if (find_function(mod, name, arity) != -1) {
	    if (hp == hend) {
		uint32 need = 10 * 5;
		hp = HAlloc(p, need);
		hend = hp + need;
	    }
	    tuple = TUPLE2(hp, name, make_small(arity));
	    hp += 3;
	    result = CONS(hp, tuple, result);
	    hp += 2;
	}
    }
    return result;
}

/*
 * Builds a list of all of atom-valued attributes for the module:
 *     [{Name, [Atom]}, ...]
 *
 * Returns a tagged term, or 0 on error.
 */

uint32
attributes_for_module(Process* p, /* Process whose heap to use. */
		      uint32 mod, /* Tagged atom for module. */
		      uint32 result) /* Tail of list to build. */
{
    int mod_index;
    uint32* code;
    int i;
    uint32* hp = NULL;
    uint32* hend = NULL;

    if (is_not_atom(mod) || (is_not_list(result) && is_not_nil(result))) {
	return 0;
    }
    mod_index = module_get(unsigned_val(mod));
    if (mod_index == -1)
	return 0;
    code = module_code(mod_index)->code;
    i = code[MI_NUM_ATTRIBUTES]-2;
    code += MI_FUNCTIONS + code[MI_NUM_FUNCTIONS] + 1;
    while (i >= 0) {
	uint32 cons;
	uint32 tuple;

	if (hp == hend) {
	    uint32 need = 10 * 7;
	    hp = HAlloc(p, need);
	    hend = hp + need;
	}
	cons = CONS(hp, code[i+1], NIL);
	hp += 2;
	tuple = TUPLE2(hp, code[i], cons);
	hp += 3;
	result = CONS(hp, tuple, result);
	hp += 2;
	i -= 2;
    }
    return result;
}

/*
 * Returns a pointer to {module, function, arity}, or NULL if not found.
 */
uint32*
find_function_from_pc(uint32* pc)
{
    Range* low = modules;
    Range* high = low + num_loaded_modules;
    Range* mid;

    while (low < high) {
	mid = low + (high-low) / 2;
	if (pc < mid->start) {
	    high = mid;
	} else if (pc > mid->end) {
	    low = mid + 1;
	} else {
	    uint32** low1 = (uint32 **) (mid->start + MI_FUNCTIONS);
	    uint32** high1 = low1 + mid->start[MI_NUM_FUNCTIONS];
	    uint32** mid1;

	    while (low1 < high1) {
		mid1 = low1 + (high1-low1) / 2;
		if (pc < mid1[0]) {
		    high1 = mid1;
		} else if (pc < mid1[1]) {
		    return mid1[0]+1;
		} else {
		    low1 = mid1 + 1;
		}
	    }
	    return NULL;
	}
    }
    return NULL;
}
