/* ``The contents of this file are subject to the Erlang Public License,
 * Version 1.0, (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.erlang.org/EPL1_0.txt
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 * 
 * The Original Code is Erlang-4.7.3, December, 1998.
 * 
 * The Initial Developer of the Original Code is Ericsson Telecom
 * AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
 * Telecom AB. All Rights Reserved.
 * 
 * Contributor(s): ______________________________________.''
 */
/*
 * Author: Bjorn Gustavsson
 * Purpos: Basic debugging support.
 */

#include "sys.h"
#include "config.h"
#include "global.h"
#include "erl_process.h"
#include "error.h"
#include "driver.h"
#include "bif.h"
#include "external.h"
#include "beam_opcodes.h"

static int print_op(int op, int size, uint32* addr);

void
dis(uint32* address, int instructions)
{
    uint32 instr;
    int i;

    while (instructions-- > 0) {
	sys_printf(CERR, "%08X: ", address);
	instr = (uint32) address[0];
	for (i = 0; i < NUM_SPECIFIC_OPS; i++) {
	    if (instr == (uint32) beam_ops[i] && opc[i].name[0] != '\0') {
		address += print_op(i, opc[i].sz-1, address+1) + 1;
		break;
	    }
	}
	if (i >= NUM_SPECIFIC_OPS) {
	    sys_printf(CERR, "unknown %d\n", address[i]);
	    address++;
	}
    }
}

void
dbg_bt(Process* p, uint32* sp)
{
    uint32* stack = p->stack;

    while (sp < stack) {
	if (is_CP(*sp)) {
	    uint32* addr = find_function_from_pc(cp_ptr_val(*sp));
	    if (addr) {
		sys_printf(CERR, "%08X: ", addr);
		display(addr[0], CERR);
		sys_printf(CERR, ":");
		display(addr[1], CERR);
		sys_printf(CERR, "/%d", addr[2]);
		sys_printf(CERR, "\n");
	    }
	}
	sp++;
    }
}

void
dbg_where(uint32* addr, uint32 x0, uint32* reg)
{
    uint32* f = find_function_from_pc(addr);

    if (f == NULL) {
	sys_printf(CERR, "???\n");
    } else {
	uint32 arity;
	char* sep = "";
	uint32 i;

	addr = f;
	arity = addr[2];
	sys_printf(CERR, "%08X: ", addr);
	display(addr[0], CERR);
	sys_printf(CERR, ":");
	display(addr[1], CERR);
	sys_printf(CERR, "(");
	for (i = 0; i < arity; i++) {
	    sys_printf(CERR, "%s", sep);
	    sep = ", ";
	    if (i == 0) {
		display(x0, CERR);
	    } else {
		display(reg[i], CERR);
	    }
	}
	sys_printf(CERR, ")\n");
    }
}

static int
print_op(int op, int size, uint32* addr)
{
    int i;
    uint32 tag;
    char* sign;
    char* start_prog;		/* Start of program for packer. */
    char* prog;			/* Current position in packer program. */
    uint32 stack[8];		/* Stack for packer. */
    uint32* sp = stack;		/* Points to next free position. */
    uint32 packed = 0;		/* Accumulator for packed operations. */
    uint32 args[8];		/* Arguments for this instruction. */
    uint32* ap;			/* Pointer to arguments. */

    /*
     * Copy all arguments to a local buffer.
     */

    ASSERT(size <= sizeof(args)/sizeof(args[0]));
    ap = args;
    for (i = 0; i < size; i++) {
	*ap++ = addr[i];
    }

    /*
     * Undo any packing done by the loader.  This is easily done by running
     * the packing program backwards and in reverse.
     */

    start_prog = opc[op].pack;
    prog = start_prog + strlen(start_prog);
    while (start_prog < prog) {
	prog--;
	switch (*prog) {
	case 'g':
	    *ap++ = *--sp;
	    break;
	case 'i':	/* Initialize packing accumulator. */
	    *ap++ = packed;
	    break;
	case 's':
	    *ap++ = packed & 0x3ff;
	    packed >>= 10;
	    break;
	case '0':	/* Shift 10 steps */
	    *ap++ = packed & 0x3ff;
	    packed >>= 10;
	    break;
	case '2':	/* Shift 12 steps */
	    *ap++ = packed & 0x3ff;
	    packed >>= 12;
	    break;
	case '6':	/* Shift 16 steps */
	    *ap++ = packed & 0x3ff;
	    packed >>= 16;
	    break;
	case 'p':
	    *sp++ = *--ap;
	    break;
	case 'P':
	    packed = *--sp;
	    break;
	default:
	    ASSERT(0);
	}
    }
    
    /*
     * Print the name and all operands of the instructions.
     */
	
    sys_printf(CERR, "%s ", opc[op].name);
    ap = args;
    sign = opc[op].sign;
    while (*sign) {
	switch (*sign) {
	case 'r':		/* x(0) */
	    sys_printf(CERR, "x(0)");
	    break;
	case 'x':		/* x(N) */
	    sys_printf(CERR, "x(%d)", ap[0]/4);
	    ap++;
	    break;
	case 'y':		/* y(N) */
	    sys_printf(CERR, "y(%d)", ap[0]/4-1);
	    ap++;
	    break;
	case 'n':		/* Nil */
	    sys_printf(CERR, "[]");
	    break;
	case 's':		/* Any source (tagged constant or register) */
	    tag = tag_val_def(*ap);
	    if (tag == X_REG_DEF) {
		sys_printf(CERR, "x(%d)", signed_val(*ap)/4);
		ap++;
		break;
	    } else if (tag == Y_REG_DEF) {
		sys_printf(CERR, "y(%d)", signed_val(*ap)/4-1);
		ap++;
		break;
	    } else if (tag == R_REG_DEF) {
		sys_printf(CERR, "x(0)");
		ap++;
		break;
	    }
	case 'a':		/* Tagged constant */
	case 'i':		/* Tagged integer */
	case 'c':		/* Tagged constant */
	    display(*ap, CERR);
	    ap++;
	    break;
	case 'A':
	    sys_printf(CERR, "%d", unsigned_val(ap[0]));
	    ap++;
	    break;
	case 'd':		/* Destination (x(0), x(N), y(N) */
	    switch (tag_val_def(*ap)) {
	    case X_REG_DEF: sys_printf(CERR, "x(%d)", signed_val(*ap)/4); break;
	    case Y_REG_DEF: sys_printf(CERR, "y(%d)", signed_val(*ap)/4-1); break;
	    case R_REG_DEF: sys_printf(CERR, "x(0)");
	    }
	    ap++;
	    break;
	case 'I':		/* Untagged integer. */
	    sys_printf(CERR, "%d", *ap);
	    ap++;
	    break;
	case 'f':		/* Destination label */
	    sys_printf(CERR, "f(%X)", *ap);
	    ap++;
	    break;
	case 'p':		/* Pointer (to label) */
	    {
		uint32* f = find_function_from_pc((uint32 *)*ap);

		if (f+3 != (uint32 *) *ap) {
		    sys_printf(CERR, "p(%X)", *ap);
		} else {
		    display(f[0], CERR);
		    sys_printf(CERR, ":");
		    display(f[1], CERR);
		    sys_printf(CERR, "/%d", f[2]);
		}
		ap++;
	    }
	    break;
	case 'j':		/* Pointer (to label) */
	    sys_printf(CERR, "j(%X)", *ap);
	    ap++;
	    break;
	case 'e':		/* Export entry */
	    {
		Export* exp = (Export *) *ap;
		display(make_atom(exp->module), CERR);
		sys_printf(CERR, ":");
		display(make_atom(exp->function), CERR);
		sys_printf(CERR, "/%d", exp->arity);
		ap++;
	    }
	    break;
	case 'F':		/* Function definition */
	    break;
	case 'b':
	    for (i = 0; i < BIF_SIZE; i++) {
		BifFunction bif = (BifFunction) *ap;
		if (bif == bif_table[i].f) {
		    break;
		}
	    }
	    if (i == BIF_SIZE) {
		sys_printf(CERR, "b(%d)", (uint32) *ap);
	    } else {
		uint32 name = *bif_table[i].name;
		uint32 arity = bif_table[i].arity;
		display(name, CERR);
		sys_printf(CERR, "/%d", arity);
	    }
	    ap++;
	    break;
	case 'P':		/* Untagged integer. */
	    sys_printf(CERR, "%d", *ap/4);
	    ap++;
	    break;
	default:
	    sys_printf(CERR, "???");
	    ap++;
	    break;
	}
	sys_printf(CERR, " ");
	sign++;
    }
    sys_printf(CERR, "\n");
    return size;
}
