%% ``The contents of this file are subject to the Erlang Public License,
%% Version 1.0, (the "License"); you may not use this file except in
%% compliance with the License. You may obtain a copy of the License at
%% http://www.erlang.org/EPL1_0.txt
%% 
%% Software distributed under the License is distributed on an "AS IS"
%% basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
%% the License for the specific language governing rights and limitations
%% under the License.
%% 
%% The Original Code is Erlang-4.7.3, December, 1998.
%% 
%% The Initial Developer of the Original Code is Ericsson Telecom
%% AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
%% Telecom AB. All Rights Reserved.
%% 
%% Contributor(s): ______________________________________.''
%%
%%%----------------------------------------------------------------------
%%% File    : erl_atom_cache.erl
%%% Author  : Magnus Fr|berg <magnus@erix.ericsson.se>
%%% Purpose : 
%%% Created : 13 Jun 1997 by Magnus Fr|berg <magnus@erix.ericsson.se>
%%%----------------------------------------------------------------------

-module(erl_atom_cache).
-copyright('Copyright (c) 1991-97 Ericsson Telecom AB').
-vsn('$Revision: /main/release/free/1').
-author('magnus@erix.ericsson.se').

-behaviour(gen_server).

%% External exports
-export([start/0, start_link/0, add_node/2, insert/3,
	 fetch/2, lookup/2, delete_node/1]).

%% gen_server callbacks
-export([init/1, handle_call/3, handle_cast/2, handle_info/2, terminate/2]).

-record(state, {cache = ets:new(erl_atom_cache, [public, {keypos, 2},
						 named_table]),
	        node_owners = []}).

-record(node_owner, {pid, node}).

-record(atom, {node_key, value}).

%%%----------------------------------------------------------------------
%%% API
%%%----------------------------------------------------------------------
start() ->  gen_server:start({local, erl_atom_cache}, erl_atom_cache, [], []).

start_link() ->  gen_server:start_link({local, erl_atom_cache},
				       erl_atom_cache, [], []).

%%
%% Link the Owner. If the Owner terminates all cached atoms
%% for that node are atomatically deleted.
%% Also start the erl_atom_cache server if not started.
%%
add_node(Node, Owner) ->
    start(),
    gen_server:call(erl_atom_cache, {add_node, Node, Owner}, infinity).

%%
%% A Key can either be an Index (for the incomming cache) or the
%% atom itself (for outgoing cache).
%% For the outgoing cache an already existing atom with the same
%% index must be deleted first.
%%
insert(Atom, Node, Index) when list(Atom) ->  %% Outgoing cache
    ets:match_delete(erl_atom_cache, #atom{node_key = {Node, '_'},
					   value = Index}),
    ets:insert(erl_atom_cache, #atom{node_key = {Node, Atom},
				     value = Index});
insert(Index, Node, Atom) when integer(Index) ->  %% Incomming cache
    ets:insert(erl_atom_cache, #atom{node_key = {Node, Index},
				     value = Atom}).

%%
%% A Key is supposed to exist !
%%
fetch(Key, Node) ->
    [Atom] = ets:lookup(erl_atom_cache, {Node, Key}),
    Atom#atom.value.

lookup(Key, Node) ->
    case ets:lookup(erl_atom_cache, {Node, Key}) of
	[Atom] -> {ok, Atom#atom.value};
	_      -> undefined
    end.

delete_node(Node) ->
    ets:match_delete(erl_atom_cache, #atom{node_key = {Node, '_'},
					   value = '_'}).

%%%----------------------------------------------------------------------
%%% Callback functions from gen_server
%%%----------------------------------------------------------------------

%%----------------------------------------------------------------------
%% 
%%----------------------------------------------------------------------
init([]) ->
    process_flag(trap_exit, true),
    {ok, #state{}}.

%%----------------------------------------------------------------------
%%
%%----------------------------------------------------------------------
handle_call({add_node, Node, Owner}, From, State) ->
    link(Owner),
    Owns = State#state.node_owners,
    {reply, ok, State#state{node_owners =
			    [#node_owner{pid = Owner,
					 node = Node} | Owns]}}.

%%----------------------------------------------------------------------
%%
%%----------------------------------------------------------------------
handle_cast(_, State) ->
    {noreply, State}.

%%----------------------------------------------------------------------
%% 
%%----------------------------------------------------------------------
handle_info({'EXIT', Pid, _}, State) ->
    Owners = State#state.node_owners,
    NewOwn = delete_nodes(Owners, Pid),
    {noreply, State#state{node_owners = NewOwn}};
handle_info(Info, State) ->
    {noreply, State}.

%%----------------------------------------------------------------------
%% 
%%----------------------------------------------------------------------
terminate(Reason, State) ->
    ok.

%%%----------------------------------------------------------------------
%%% Internal functions
%%%----------------------------------------------------------------------

delete_nodes([#node_owner{pid = Pid, node = Node}|Owners], Pid) ->
    delete_node(Node),
    delete_nodes(Owners, Pid);
delete_nodes([Own|Owners], Pid) ->
    [Own|delete_nodes(Owners, Pid)];
delete_nodes([], _) ->
    [].
    



