%% ``The contents of this file are subject to the Erlang Public License,
%% Version 1.0, (the "License"); you may not use this file except in
%% compliance with the License. You may obtain a copy of the License at
%% http://www.erlang.org/EPL1_0.txt
%% 
%% Software distributed under the License is distributed on an "AS IS"
%% basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
%% the License for the specific language governing rights and limitations
%% under the License.
%% 
%% The Original Code is Erlang-4.7.3, December, 1998.
%% 
%% The Initial Developer of the Original Code is Ericsson Telecom
%% AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
%% Telecom AB. All Rights Reserved.
%% 
%% Contributor(s): ______________________________________.''
%%
%% Copyright (C) 1996 Ericsson Telecommunications AB
%% File    : sys_pre_expand.erl
%% Author  : Robert Virding
%% Purpose : Expand some source Erlang constructions. This is part of the
%%           pre-processing phase.

%% N.B. Although structs (tagged tuples) are not yet allowed in the
%% language there is code included in pattern/2 and expr/3 (commented out)
%% that handles them by transforming them to tuples.

-module(sys_pre_expand).
-copyright('Copyright (c) 1991-97 Ericsson Telecom AB').
-vsn('$Revision: /main/release/free/1').

-author('rv@cslab.ericsson.se').
-copyright('Copyright (c) 1996 Ericsson Telecommunications AB').

%% Main entry point.
-export([module/2]).

-import(ordsets, [list_to_set/1,add_element/2,
		  union/1,union/2,intersection/1,intersection/2,subtract/2]).
-import(lists,   [member/2,map/2,foldl/3,foldr/3,reverse/1]).

-record(expand, {status=ok,
		 module=[],			%Module name
		 exports=[],			%Exports
		 imports=[],			%Imports
		 compile=[],			%Compile flags
		 records=dict:new(),		%Record definitions
		 attributes=[],			%Attributes
		 defined=[],			%Defined functions
		 fcount=0,			%Gen function counter
		 vcount=0,			%Variable counter
		 lcount=0,			%Lambda (fun) counter
		 lclauses=[],			%Lambda (fun) clauses
		 genfuncs=[],			%Generated functions
		 func=[]}).			%Current function

%% module(Forms, CompileOptions)
%%	{ModuleName,Exports,TransformedForms}
%%  Expand the forms in one module. N.B. the lists of predefined functions
%%  and exports are really ordsets!

module(Fs, Opts) ->
    St0 = #expand{exports=[{module_info,0},
			   {module_info,1},
			   {module_lambdas,4}],
		  compile=Opts,
		  defined=[{apply_lambda,2},
			   {module_info,0},
			   {module_info,1},
			   {module_lambdas,4},
			   {record_index,2}]},
    {Tfs,St1} = forms(Fs, foldl(fun define_function/2, St0, Fs)),
    %% Get the correct list of exported functions.
    Exp = case member(export_all, St1#expand.compile) of
	      true -> St1#expand.defined;
	      false -> St1#expand.exports
	  end,
    {Efs,St2} = extra_funcs(St1),
    {Mfs,St3} = module_info_funcs(Exp, St2),
    {St3#expand.module,Exp,Tfs ++ Mfs ++ Efs,St3#expand.compile}.

%% -type define_function(Form, State) -> State.
%%  Add function to defined if form a function.

define_function({attribute,L,asm,{function,N,A,Code}}, St) ->
    St#expand{defined=add_element({N,A}, St#expand.defined)};
define_function({function,L,N,A,Cs}, St) ->
    St#expand{defined=add_element({N,A}, St#expand.defined)};
define_function(F, St) -> St.

extra_funcs(St) ->
    %% Function module_lambdas/4 has already been transformed.
    Mlf = {function,0,module_lambdas,4,
	   St#expand.lclauses ++
	   [{clause,0,[{var,0,'I'},{var,0,'Uniq'},
		       {var,0,'Args'},{var,0,'Free'}],[],
	     [call_exit(0, {tuple,0,[{atom,0,undef_lambda},
				     {atom,0,St#expand.module}]})]}
	   ]},
    %% Function apply_lambda/2 has already been transformed.
    Alf = {function,0,apply_lambda,2,
	   [{clause,0,[make_fun(0, {var,0,'I'}, {var,0,'Uniq'},
				{var,0,'Free'}, St),
		       {var,0,'Args'}],
	     [],
	     [{call,0,{atom,0,module_lambdas},
	       [{var,0,'I'},{var,0,'Uniq'},{var,0,'Args'},{var,0,'Free'}]}
	     ]},
	    {clause,0,
	     [make_fun(0, {var,0,'M'}, {var,0,'I'},
		       {var,0,'Uniq'}, {var,0,'Free'}, St),
	      {var,0,'Args'}],
	     [],
	     [{call,0,
	       {remote,0,{atom,0,erlang},{atom,0,apply}},
	       [{var,0,'M'},{atom,0,module_lambdas},
		make_list([{var,0,'I'},{var,0,'Uniq'},
			   {var,0,'Args'},{var,0,'Free'}], 0)]}
	     ]},
	    {clause,0,
	     [{var,0,'F'},{var,0,'Args'}],
	     [],
	     [{call,0,{remote,0,{atom,0,erlang},{atom,0,apply}},
	       [{var,0,'F'},{var,0,'Args'}]}]}
	   ]},
    %% Function record_index/2 has already been transformed.
    Rif = {function,0,record_index,2,
	   [{clause,0,[{var,0,'R'},{var,0,'F'}], [],
	     [call_exit(0, {tuple,0,[{atom,0,badfield},
				     {var,0,'R'},{var,0,'F'}]})]}
	   ]},
    {[Mlf,Alf,Rif],St}.

module_info_funcs(Exp, St) ->
    {Y,Mo,D} = date(),
    {H,Mi,S} = time(),
    {AtomAttr, OtherAttr} = split_attr(St),
    %% Function module_info/0 has already been transformed.
    {[{function,0,module_info,0,
       [{clause,0,[],[],
	 [foldr(fun (A, T) -> {cons,0,mi0_call(A),T} end, {nil,0},
		[exports,imports,attributes,compile])]}]},
      {function,0,module_info,1,
       [mi1_clause(module, St#expand.module),
	mi1_clause(exports, Exp, St),
	mi1_clause(imports, St#expand.imports),
	mi1_clause(attributes, OtherAttr, St),
	mi1_clause(compile, [{time,{Y,Mo,D,H,Mi,S}},
			     {options,St#expand.compile}])]}],
     St#expand{compile=[{'_atom_attributes_', AtomAttr}|St#expand.compile]}}.

mi0_call(Name) ->
    {tuple,0,[{atom,0,Name},{call,0,{atom,0,module_info},[{atom,0,Name}]}]}.

mi1_clause(Name, Val) ->
    {clause,0,[{atom,0,Name}],[],[erl_parse:abstract(Val)]}.

mi1_clause(exports, Val, St) ->
    case member(beam, St#expand.compile) of
	true ->
	    {clause,0,[{atom,0,exports}],[],
	     [{call,0,{remote,0,{atom,0,erlang},{atom,0,info}},
	       [{tuple,0,[{atom,0,exports},
			  {atom,0,St#expand.module}]}]}]};
	false ->
	    mi1_clause(exports, Val)
    end;
mi1_clause(attributes, Val, St) ->
    case member(beam, St#expand.compile) of
	true ->
	    {clause,0,[{atom,0,attributes}],[],
	     [{call,0,{remote,0,{atom,0,erlang},{atom,0,info}},
	       [{tuple,0,[{atom,0,attributes},
			  {atom,0,St#expand.module},
			  erl_parse:abstract(Val)]}]}]};
	false ->
	    mi1_clause(attributes, Val)
    end.

%% For Beam, atom-valued attributes should be separated from other attributes,
%% so that they can be put into the file in a format which allows manipulation
%% by external tools.

split_attr(St) ->
    case member(beam, St#expand.compile) of
	true ->
	    split_attr(St#expand.attributes, [], []);
	false ->
	    {[], St#expand.attributes}
    end.

split_attr([{Name, Val}|T], Atoms, Others) ->
    case {good_value(Val), good_atom(Name)} of
	{true, true} ->
	    split_attr(T, [{Name, hd(Val)}|Atoms], Others);
	_ ->
	    split_attr(T, Atoms, [{Name, Val}|Others])
    end;
split_attr([], Atoms, Others) ->
    {reverse(Atoms), reverse(Others)}.

%% A good value should be a single value and be a good atom.
good_value([Val]) ->
    good_atom(Val);
good_value(_) ->
    false.

%% A good atom should be an atom and not contain $\0 or $=.
good_atom(A) when atom(A) ->
    no_funny_chars(atom_to_list(A));
good_atom(_) ->
    false.

no_funny_chars([0|_])  -> false;
no_funny_chars([$=|_]) -> false;
no_funny_chars([C|T])  ->  no_funny_chars(T);
no_funny_chars([])     -> true.

%% forms(Forms, State) ->
%%	{TransformedForms,State'}
%%  Process the forms. Attributes are lost and just affect the state.
%%  Asm things are funny, just pass them through. Ignore uninteresting forms
%%  like eof and type.

forms([{attribute,L,asm,{function,N,A,Code}}|Fs0], St0) ->
    {Fs,St1} = forms(Fs0, St0),
    {[{asm,N,A,Code}|Fs],St1};
forms([{attribute,L,Name,Val}|Fs0], St0) ->
    St1 = attribute(L, Name, Val, St0),
    forms(Fs0, St1);
forms([{function,L,N,A,Cs}|Fs0], St0) ->
    {Ffs,St1} = function(L, N, A, Cs, St0),
    {Fs,St2} = forms(Fs0, St1),
    {Ffs ++ Fs,St2};
forms([F|Fs], St) -> forms(Fs, St);
forms([], St) -> {[],St}.

%% -type attribute(Line, Attribute, Value, State) ->
%%	State.
%%  Process an attribute, this just affects the state.

attribute(L, module, Module, St) ->
    St#expand{module=Module};
attribute(L, export, Es, St) ->
    St#expand{exports=union(list_to_set(Es), St#expand.exports)};
attribute(L, import, Is, St) ->
    import(L, Is, St);
attribute(L, compile, C, St) when list(C) ->
    St#expand{compile=St#expand.compile ++ C};
attribute(L, compile, C, St) ->
    St#expand{compile=St#expand.compile ++ [C]};
attribute(L, record, {Name,Defs}, St) ->
    St#expand{records=dict:store(Name, normalise_fields(Defs),
				 St#expand.records)};
attribute(L, file, File, St) ->	St;		%This is ignored
attribute(L, Name, Val, St) when list(Val) ->
    St#expand{attributes=St#expand.attributes ++ [{Name,Val}]};
attribute(L, Name, Val, St) ->
    St#expand{attributes=St#expand.attributes ++ [{Name,[Val]}]}.

function(L, N, A, Cs, St) ->
    function([{function,L,N,A,Cs}], [], St#expand{func=N}).

%%function([{function,L,N,A,Cs0}|Fs0], Fss, St0) ->
%%    {Cs1,St1} = clauses(Cs0, St0),
%%    Gfs = St1#expand.genfuncs,
%%    {Fs1,St2} = function(Fs0, Fss, St1),
%%    {[{function,L,N,A,Cs1}|Gfs ++ Fs1],St2};
function([{function,L,N,A,Cs0}|Fs], Fss, St0) ->
    %%{Cs,St1} = clauses(Cs0, St0#expand{func=N}),
    {Cs,St1} = clauses(Cs0, St0),
    {Gfs,St2} = function(St1#expand.genfuncs, [Fs|Fss],
			 St1#expand{genfuncs=[]}),
    {[{function,L,N,A,Cs}|Gfs],St2};
function([], [Fs|Fss], St) ->
    function(Fs, Fss, St);
function([], [], St) ->
    {[],St}.

%% -type clauses([Clause], State) ->
%%	{[TransformedClause],State}.
%%  Expand function clauses.

clauses([{clause,Line,H0,G0,B0}|Cs0], St0) ->
    {H,Hvs,St1} = head(H0, St0),
    {G,Gvs,Gus,St2} = guard(G0, Hvs, St1),
    {B,Bvs,Bus,St3} = exprs(B0, union(Hvs, Gvs), St2),
    {Cs,St4} = clauses(Cs0, St3),
    {[{clause,Line,H,G,B}|Cs],St4};
clauses([], St) -> {[],St}.

%% head(HeadPatterns, State) ->
%%	{TransformedPatterns,Variables,State'}

head([A0|As0], St0) ->
    {A,Avs,St1} = pattern(A0, St0),
    {As,Asvs,St2} = head(As0, St1),
    {[A|As],union(Avs, Asvs),St2};
head([], St) -> {[],[],St}.

%% pattern(Pattern, State) ->
%%	{TransformedPattern,Variables,State'}

pattern({var,Line,'_'}, St) ->			%Ignore anonymous variable.
    {{var,Line,'_'},[],St};
pattern({var,Line,V}, St) ->
    {{var,Line,V},[V],St};
pattern({integer,Line,I}, St) ->
    {{integer,Line,I},[],St};
pattern({float,Line,F}, St) ->
    {{float,Line,F},[],St};
pattern({atom,Line,A}, St) ->
    {{atom,Line,A},[],St};
pattern({string,Line,S}, St) ->
    {{string,Line,S},[],St};
pattern({nil,Line}, St) ->
    {{nil,Line},[],St};
pattern({cons,Line,H,T}, St0) ->
    {TH,THvs,St1} = pattern(H, St0),
    {TT,TTvs,St2} = pattern(T, St1),
    {{cons,Line,TH,TT},union(THvs, TTvs),St2};
pattern({tuple,Line,Ps}, St0) ->
    {TPs,TPsvs,St1} = pattern_list(Ps, St0),
    {{tuple,Line,TPs},TPsvs,St1};
%%pattern({struct,Line,Tag,Ps}, St0) ->
%%    {TPs,TPsvs,St1} = pattern_list(Ps, St0),
%%    {{tuple,Line,[{atom,Line,Tag}|TPs]},TPsvs,St1};
pattern({record_index,Line,Name,Field}, St) ->
    {index_expr(Line, Field, Name, record_fields(Name, St)),[],St};
pattern({record,Line,Name,Pfs}, St0) ->
    Fs = record_fields(Name, St0),
    {TMs,TMsvs,St1} = pattern_list(pattern_fields(Fs, Pfs), St0),
    {{tuple,Line,[{atom,Line,Name}|TMs]},TMsvs,St1};
%% The following are necessary to be able to handle unary +,- in patterns.
pattern({op,Line,'+',{integer,Li,I}}, St) ->
    {{integer,Li,I},[],St};
pattern({op,Line,'-',{integer,Li,I}}, St) ->
    {{integer,Li,-I},[],St};
pattern({op,Line,'+',{float,Lf,F}}, St) ->
    {{float,Lf,F},[],St};
pattern({op,Line,'-',{float,Lf,F}}, St) ->
    {{float,Lf,-F},[],St}.

pattern_list([P|Ps], St0) ->
    {TP,TPvs,St1} = pattern(P, St0),
    {TPs,TPsvs,St2} = pattern_list(Ps, St1),
    {[TP|TPs],union(TPvs, TPsvs),St2};
pattern_list([], St) -> {[],[],St}.

%% guard(Guard, VisibleVariables, State) ->
%%	{TransformedGuard,NewVariables,UsedVariables,State'}
%%  Transform a list of guard tests. We KNOW that this has been checked
%%  and what the guards test are. Use expr for transforming the guard
%%  expressions.

guard([{op,Line,Op,L0,R0}|Gs0], Vs, St0) ->
    {L,Lvs,Lus,St1} = expr(L0, Vs, St0),
    {R,Rvs,Rus,St2} = expr(R0, Vs, St1),
    {Gs,Gsvs,Gsus,St3} = guard(Gs0, union(Lvs, Rvs), St2),
    {[{op,Line,Op,L,R}|Gs],union([Lvs,Rvs,Gsvs]),union([Lus,Rus,Gsus]),St3};
guard([{call,Line,{atom,Lr,record},[A,{atom,Ln,Name}]}|Gs], Vs, St) ->
    Fs = record_fields(Name, St),
    guard([{op,Line,'==',{call,Line,{atom,Ln,size},[A]},
	    {integer,Ln,length(Fs)+1}},
	   {op,Line,'==',{call,Line,{atom,Ln,element},[{integer,Ln,1},A]},
	    {atom,Ln,Name}}|Gs], Vs, St);
guard([{call,Line,{atom,Lf,function},[A]}|Gs], Vs, St) ->
    guard([{op,Line,'==',{call,Line,{atom,Lf,size},[A]},{integer,Lf,5}},
	   {op,Line,'==',{call,Line,{atom,Lf,element},[{integer,Lf,1},A]},
	    {atom,Lf,'fun'}}|Gs], Vs, St);
guard([{call,Line,Test,As0}|Gs0], Vs, St0) ->
    {As,Asvs,Asus,St1} = expr_list(As0, Vs, St0),
    {Gs,Gsvs,Gsus,St2} = guard(Gs0, union(Asvs, Vs), St1),
    {[{call,Line,Test,As}|Gs],union(Asvs, Gsvs),union(Asus, Gsus),St2};
guard([{atom,Line,true}|Gs], Vs, St) ->
    guard(Gs, Vs, St);
guard([], Vs, St) ->
    {[],[],[],St}.

%% exprs(Expressions, VisibleVariables, State) ->
%%	{TransformedExprs,NewVariables,UsedVariables,State'}

exprs([E0|Es0], Vs, St0) ->
    {E,Evs,Eus,St1} = expr(E0, Vs, St0),
    {Es,Esvs,Esus,St2} = exprs(Es0, union(Evs, Vs), St1),
    {[E|Es],union(Evs, Esvs),union(Eus, Esus),St2};
exprs([], Vs, St) ->
    {[],[],[],St}.

%% expr(Expression, VisibleVariables, State) ->
%%	{TransformedExpression,NewVariables,UsedVariables,State'}

expr({var,Line,V}, Vs, St) ->
    {{var,Line,V},[],[V],St};
expr({integer,Line,I}, Vs, St) ->
    {{integer,Line,I},[],[],St};
expr({float,Line,F}, Vs, St) ->
    {{float,Line,F},[],[],St};
expr({atom,Line,A}, Vs, St) ->
    {{atom,Line,A},[],[],St};
expr({string,Line,S}, Vs, St) ->
    {{string,Line,S},[],[],St};
expr({nil,Line}, Vs, St) ->
    {{nil,Line},[],[],St};
expr({cons,Line,H0,T0}, Vs, St0) ->
    {H,Hvs,Hus,St1} = expr(H0, Vs, St0),
    {T,Tvs,Tus,St2} = expr(T0, Vs, St1),
    {{cons,Line,H,T},union(Hvs, Tvs),union(Hus, Tus),St2};
expr({lc,Line,E,Qs}, Vs, St) ->
    lc_tq(Line, E, Qs, {nil,Line}, Vs, St);
expr({tuple,Line,Es0}, Vs, St0) ->
    {Es1,Esvs,Esus,St1} = expr_list(Es0, Vs, St0),
    {{tuple,Line,Es1},Esvs,Esus,St1};
%%expr({struct,Line,Tag,Es0}, Vs, St0) ->
%%    {Es1,Esvs,Esus,St1} = expr_list(Es0, Vs, St0),
%%    {{tuple,Line,[{atom,Line,Tag}|Es1]},Esvs,Esus,St1};
expr({record_index,Line,Name,F}, Vs, St) ->
    I = index_expr(Line, F, Name, record_fields(Name, St)),
    expr(I, Vs, St);
expr({record,Line,Name,Is}, Vs, St) ->
    expr({tuple,Line,[{atom,Line,Name}|
		      record_inits(record_fields(Name, St), Is)]},
	 Vs, St);
expr({record_field,Line,R,Name,F}, Vs, St) ->
    I = index_expr(Line, F, Name, record_fields(Name, St)),
    expr({call,Line,{atom,Line,element},[I,R]}, Vs, St);
expr({record,Line,R,Name,Us}, Vs, St0) ->
    {Ue,St1} = record_update(R, Name, record_fields(Name, St0), Us, St0),
    expr(Ue, Vs, St1);
expr({block,Line,Es0}, Vs, St0) ->
    {Es,Esvs,Esus,St1} = exprs(Es0, Vs, St0),
    {{block,Line,Es},Esvs,Esus,St1};
expr({'if',Line,Cs0}, Vs, St0) ->
    {Cs,Csvss,Csuss,St1} = icr_clauses(Cs0, Vs, St0),
    {All,Some} = new_in(Vs, Csvss),
    {{'if',Line,Cs},All,union(Csuss),St1};
expr({'case',Line,E0,Cs0}, Vs, St0) ->
    {E,Evs,Eus,St1} = expr(E0, Vs, St0),
    {Cs,Csvss,Csuss,St2} = icr_clauses(Cs0, union(Evs, Vs), St1),
    {All,Some} = new_in(Vs, Csvss),
    {{'case',Line,E,Cs},union(Evs, All),union([Eus|Csuss]),St2};
expr({'receive',Line,Cs0}, Vs, St0) ->
    {Cs,Csvss,Csuss,St1} = icr_clauses(Cs0, Vs, St0),
    {All,Some} = new_in(Vs, Csvss),
    {{'receive',Line,Cs},All,union(Csuss),St1};
expr({'receive',Line,Cs0,To0,ToEs0}, Vs, St0) ->
    {To,Tovs,Tous,St1} = expr(To0, Vs, St0),
    {ToEs,ToEsvs,ToEsus,St2} = exprs(ToEs0, Vs, St1),
    {Cs,Csvss,Csuss,St3} = icr_clauses(Cs0, Vs, St2),
    {All,Some} = new_in(Vs, [ToEsvs|Csvss]),
    {{'receive',Line,Cs,To,ToEs},union(Tovs, All),union([Tous|Csuss]),St3};
expr({'fun',Line,Body}, Vs, St) ->
    fun_tq(Line, Body, Vs, St);
expr({call,Line,{atom,La,apply},As0}, Vs, St0) when length(As0) == 2 ->
    {As,Asvs,Asus,St1} = expr_list(As0, Vs, St0),
    {{call,Line,{atom,La,apply_lambda},As},Asvs,Asus,St1};
expr({call,Line,{atom,La,record_info},[{atom,Li,Info},{atom,Ln,Name}]}, Vs, St) ->
    case Info of
	size -> {{integer,Line,1+length(record_fields(Name, St))},[],[],St};
	fields -> {make_list(field_names(record_fields(Name, St)), Line),
		   [],[],St}
    end;
expr({call,Line,{atom,La,N},As0}, Vs, St0) ->
    {As,Asvs,Asus,St1} = expr_list(As0, Vs, St0),
    Ar = length(As),
    case erl_internal:bif(N, Ar) of
	true ->
	    {{call,Line,{remote,La,{atom,La,erlang},{atom,La,N}},As},
	     Asvs,Asus,St1};
	false ->
	    case imported(N, Ar, St1) of
		{yes,Mod} ->
		    {{call,Line,{remote,La,{atom,La,Mod},{atom,La,N}},As},
		     Asvs,Asus,St1};
		no ->
		    {{call,Line,{atom,La,N},As},Asvs,Asus,St1}
	    end
    end;
expr({call,Line,{remote,Lr,{atom,Lm,M},{atom,Lf,F}},As0}, Vs, St0) ->
    {As1,Asvs,Asus,St1} = expr_list(As0, Vs, St0),
    {{call,Line,{remote,Lr,{atom,Lm,M},{atom,Lf,F}},As1},Asvs,Asus,St1};
expr({call,Line,{remote,Lr,M,F},As}, Vs, St) ->
    expr({call,Line,{atom,Line,apply},[M,F,make_list(As, Line)]}, Vs, St);
expr({call,Line,F,As}, Vs, St) ->
    expr({call,Line,{atom,Line,apply},[F,make_list(As, Line)]}, Vs, St);
expr({'catch',Line,E0}, Vs, St0) ->
    %% Catch exports no new variables.
    {E,Evs,Eus,St1} = expr(E0, Vs, St0),
    {{'catch',Line,E},[],Eus,St1};
expr({match,Line,P0,E0}, Vs, St0) ->
    {E,Evs,Eus,St1} = expr(E0, Vs, St0),
    {P,Pvs,St2} = pattern(P0, St1),
    {{match,Line,P,E},
     union(subtract(Pvs, Vs), Evs),
     union(intersection(Pvs, Vs), Eus),St2};
expr({op,Line,'++',{lc,Ll,E,Qs},L2}, Vs, St) ->
    lc_tq(Ll, E, Qs, L2, Vs, St);
expr({op,Ll,'++',{cons,Lc,H,T},L2}, Vs, St) ->
    expr({cons,Ll,H,{op,Lc,'++',T,L2}}, Vs, St);
expr({op,Ll,'++',{nil,Ln},L2}, Vs, St) ->
    expr(L2, Vs, St);
expr({op,Line,Op,A0}, Vs, St0) ->
    {A,Avs,Aus,St1} = expr(A0, Vs, St0),
    {{op,Line,Op,A},Avs,Aus,St1};
expr({op,Line,Op,L0,R0}, Vs, St0) ->
    {L,Lvs,Lus,St1} = expr(L0, Vs, St0),
    {R,Rvs,Rus,St2} = expr(R0, Vs, St1),
    {{op,Line,Op,L,R},union(Lvs, Rvs),union(Lus, Rus),St2}.

expr_list([E0|Es0], Vs, St0) ->
    {E,Evs,Eus,St1} = expr(E0, Vs, St0),
    {Es,Esvs,Esus,St2} = expr_list(Es0, Vs, St1),
    {[E|Es],union(Evs, Esvs),union(Eus, Esus),St2};
expr_list([], Vs, St) ->
    {[],[],[],St}.

%% icr_clauses([Clause], [VisibleVariable], State) ->
%%	{[TransformedClause],[[NewVariable]],[[UsedVariable]],State'}
%%  Be very careful here to return the variables that are really used
%%  and really new.

icr_clauses([{clause,Line,H0,G0,B0}|Cs0], Vs, St0) ->
    {H,Hvs,St1} = head(H0, St0),		%Hvs is really used!
    {G,Gvs,Gus,St2} = guard(G0, union(Hvs, Vs), St1),
    {B,Bvs,Bus,St3} = exprs(B0, union([Vs,Hvs,Gvs]), St2),
    New = subtract(union([Hvs,Gvs,Bvs]), Vs),	%Really new
    Used = intersection(union([Hvs,Gus,Bus]), Vs), %Really used
    {Cs,Csvs,Csus,St4} = icr_clauses(Cs0, Vs, St3),
    {[{clause,Line,H,G,B}|Cs],[New|Csvs],[Used|Csus],St4};
icr_clauses([], Vs, St) ->
    {[],[],[],St}.

%% lc_tq(Line, Expr, Qualifiers, More, [VisibleVar], State) ->
%%	{ListComp,[NewVar],[UsedVar],State'}
%%
%% This TQ from Simon PJ pp 127-138.  No need to call ourselves
%% recursively on append as this is special cased in expr as an
%% optimisation. As Erlang doesn't have letrec's we explicitly pass
%% the lambda as an extra argument to be able to do the recursive
%% call. We also do the argument matching in the head of the lambda
%% instead of in a 'case' so as to automatically get the correct
%% shadowing.

lc_tq(Line, E, Qs, More, Vs, St0) ->
    {Lc,St1} = lc_tq(Line, E, Qs, More, St0),
    expr(Lc, Vs, St1).

lc_tq(Line, E, [{generate,Lg,P,G}|Qs], More, St0) ->
    {F,St1} = new_var(St0),
    {T,St2} = new_var(St1),
    Fun = {var,Lg,F},
    Tail = {var,Lg,T},
    NewMore = {call,Lg,Fun,[Tail,Fun]},
    {Lc,St3} = lc_tq(Line, E, Qs, NewMore, St2),
    {{block,Lg,
      [{match,Lg,Fun,
	{'fun',Lg,
	 {clauses,[{clause,Lg,[{cons,Lg,P,Tail},Fun],[],[Lc]},
		   {clause,Lg,[{cons,Lg,{var,Lg,'_'},Tail},Fun],[],[NewMore]},
		   {clause,Lg,[{nil,Lg},Fun],[],[More]}]}}},
       {call,Lg,Fun,[G,Fun]}]},
     St3};
lc_tq(Line, E, [F|Qs], More, St0) ->
    Lf = element(2, F),
    {Lc,St1} = lc_tq(Line, E, Qs, More, St0),
    case erl_lint:is_guard_test(F) of
	true ->
	    {{'if',Lf,
	      [{clause,Lf,[],[F],[Lc]},
	       {clause,Lf,[],[],[More]}]},St1};
	false ->
	    {{'case',Lf,F,
	      [{clause,Lf,[{atom,Lf,true}],[],[Lc]},
	       {clause,Lf,[{atom,Lf,false}],[],[More]}]},St1}
    end;
lc_tq(Line, E, [], More, St) ->
    {{cons,Line,E,More},St}.

%% fun_tq(Line, Body, VisibleVariables, State) ->
%%	{Fun,NewVariables,UsedVariables,State'}
%%  Transform a fun into into a tuple {'fun',Mod,I,Uniq,Free} and add
%%  clauses to module_lambdas/4 to hold the code. Process the body
%%  sequence directly to get the new and used variables, this also
%%  allows recursives call for the function case to correctly handle
%%  BIFs. N.B. erl_lint disallows imports so we don't have to check.

fun_tq(Lf, {function,F,A}, Vs, St0) ->
    {Avs,St1} = new_vars(A, St0),
    As = make_vars(Avs, Lf),
    fun_tq(Lf, {clauses,[{clause,Lf,As,[],[{call,Lf,{atom,Lf,F},As}]}]},
	   Vs, St1);
fun_tq(Lf, {clauses,Cs0}, Vs, St0) ->
    I = St0#expand.lcount,
    St1 = St0#expand{lcount=I+1},
    %% The added clauses are transformed directly to get new/used vars.
    {Cs1,Hvss,Frees,St2} = fun_clauses(Cs0, Vs, St1),
    Free = intersection(union(Frees), Vs),
    Uniq = uniq(St2),
    %%io:format("Vs   :~p~nFrees:~p~nFree :~p~n" ,[Vs,Frees,Free]),
    FunClauses = fun_clauses(Cs1, Hvss, I, Free, Uniq, St2),
    St3 = St2#expand{lclauses=FunClauses ++ St2#expand.lclauses},
    {make_fun(Lf, {integer,Lf,I}, {integer,Lf,Uniq},
	      {tuple,Lf,make_vars(Free, Lf)}, St1),
     [],union(Frees),St3}.

fun_clauses([{clause,L,H0,G0,B0}|Cs0], Vs, St0) ->
    {H,Hvs,St1} = head(H0, St0),
    {G,Gvs,Gus,St2} = guard(G0, union(Hvs, Vs), St1),
    {B,Bvs,Bus,St3} = exprs(B0, union([Vs,Hvs,Gvs]), St2),
    %% Free variables cannot be new anywhere in the clause.
    Free = subtract(union(Gus, Bus), union([Hvs,Gvs,Bvs])),
    %%io:format(" Gus :~p~n Bvs :~p~n Bus :~p~n Free:~p~n" ,[Gus,Bvs,Bus,Free]),
    {Cs,Hvss,Frees,St4} = fun_clauses(Cs0, Vs, St3),
    {[{clause,L,H,G,B}|Cs],[Hvs|Hvss],[Free|Frees],St4};
fun_clauses([], Vs, St) -> {[],[],[],St}.

fun_clauses([{clause,L,H,G,B}|Cs], [Hvs|Hvss], Index, Free, Uniq, St) ->
    [{clause,L,
      [{integer,L,Index},{integer,L,Uniq},make_list(H, L),
       {tuple,L,free_vars(Free, Hvs, L)}],
      G,
      B}|fun_clauses(Cs, Hvss, Index, Free, Uniq, St)];
fun_clauses([], [], Index, Free, Uniq, St) ->
    [{clause,0,
      [{integer,0,Index},{integer,0,Uniq},{var,0,'Args'},{var,0,'Free'}],
      [],
      [call_exit(0, {tuple,0,[{atom,0,lambda_clause},
			      {atom,0,St#expand.module}]})]}].

free_vars(Vs, Hvs, Line) ->
    [ case member(V, Hvs) of
	  true -> {var,Line,'_'};
	  false -> {var,Line,V}
      end || V <- Vs ].

make_fun(Line, I, Uniq, Free, St) ->
    make_fun(Line, {atom,Line,St#expand.module}, I, Uniq, Free, St).

make_fun(Line, Mod, I, Uniq, Free, St) ->
    {tuple,Line,[{atom,Line,'fun'},Mod,I,Uniq,Free]}.

uniq(St) ->
    {M,S,U} = now(),
    ((M bsl 8) bxor (S bsl 4) bxor U).

%% normalise_fields([RecDef]) -> [Field].
%%  Normalise the field definitions to always have a default value. If
%%  none has been given then use 'undefined'.

normalise_fields(Fs) ->
    map(fun ({record_field,Lf,Field}) ->
		{record_field,Lf,Field,{atom,Lf,undefined}};
	    (F) -> F end, Fs).

%% record_fields(RecordName, State)
%% find_field(FieldName, Fields)

record_fields(R, St) -> dict:fetch(R, St#expand.records).

find_field(F, [{record_field,Lf,{atom,La,F},Val}|Fs]) -> {ok,Val};
find_field(F, [_|Fs]) -> find_field(F, Fs);
find_field(F, []) -> error.

%% field_names(RecFields) -> [Name].
%%  Return a list of the field names structures.

field_names(Fs) ->
    [ Field || {record_field,Lf,Field,Val} <- Fs ].

%% index_expr(Line, FieldExpr, Name, Fields) -> IndexExpr.
%%  Return an expression which evaluates to the index of a
%%  field. Currently only handle the case where the field is an
%%  atom. This expansion must be passed through expr again.

index_expr(Line, {atom,La,F}, Name, Fs) ->
    {integer,Line,index_expr(F, Fs, 2)}.
%%index_expr(Line, F, Name, Fs) ->
%%    {call,Line,{atom,Line,record_index},[{atom,Line,Name},F]}.

index_expr(F, [{record_field,Lf,{atom,La,F},Val}|Fs], I) -> I;
index_expr(F, [_|Fs], I) ->
    index_expr(F, Fs, I+1).

%% pattern_fields([RecDefField], [Match]) -> [Pattern].
%%  Build a list of match patterns for the record tuple elements.
%%  This expansion must be passed through pattern again. N.B. We are
%%  scanning the record definition field list!

pattern_fields(Fs, Ms) ->
    map(fun ({record_field,L,{atom,La,F},D}) ->
		case find_field(F, Ms) of
		    {ok,Match} -> Match;
		    error -> {var,L,'_'}
		end end,
	Fs).

%% record_inits([RecDefField], [Init]) -> [InitExpr].
%%  Build a list of initialisation expressions for the record tuple
%%  elements. This expansion must be passed through expr
%%  again. N.B. We are scanning the record definition field list!

record_inits(Fs, Is) ->
    map(fun ({record_field,L,{atom,La,F},D}) ->
		case find_field(F, Is) of
		    {ok,Init} -> Init;
		    error -> D
		end end,
	Fs).

%% record_update(Record, RecordName, [RecDefField], [Update], State) ->
%%	{Expr,State'}
%%  Build an expression to update fields in a record returning a new
%%  record.  Try to be smart and optimise this. This expansion must be
%%  passed through expr again.

record_update(R, Name, Fs, Us, St) ->
    Nf = length(Fs),				%# of record fields
    Nu = length(Us),				%# of update fields
    Nc = Nf - Nu,				%# of copy fields
    %% Try to be intelligent about which method of updating record to use.
    if
	Nu == 0 ->
	    {record_el(R, Name, Fs, Us),St};
	Nu == 1 ->
	    {record_setel(R, Name, Fs, Us),St};
	Nc == 1 ->
	    {record_el(R, Name, Fs, Us),St};
	true ->
	    record_match(R, Name, Fs, Us, St)
    end.	

%% record_match(Record, RecordName, [RecDefField], [Update], State)
%%  Build a 'case' expression to modify record fields.

record_match(R, Name, Fs, Us, St0) ->
    {Ps,News,St1} = record_upd_fs(Fs, Us, St0),
    Lr = element(2, hd(Us)),
    {{'case',Lr,R,
      [{clause,Lr,[{tuple,Lr,[{atom,Lr,Name}|Ps]}],[],
	[{tuple,Lr,[{atom,Lr,Name}|News]}]},
       {clause,Lr,[{var,Lr,'_'}],[],
	[call_exit(Lr, {tuple,Lr,[{atom,Lr,badrecord},{atom,Lr,Name}]})]}
      ]},
     St1}.

record_upd_fs([{record_field,Lf,{atom,La,F},Val}|Fs], Us, St0) ->
    case find_field(F, Us) of
	{ok,New} ->
	    {Ps,News,St1} = record_upd_fs(Fs, Us, St0),
	    {[{var,Lf,'_'}|Ps],[New|News],St1};
	error ->
	    {Pv,St1} = new_var(St0),
	    P = {var,Lf,Pv},
	    {Ps,News,St2} = record_upd_fs(Fs, Us, St1),
	    {[P|Ps],[P|News],St2}
    end;
record_upd_fs([], Us, St) -> {[],[],St}.

%% record_el(Record, RecordName, [RecDefField], [Update])
%%  Build a new record using Rec#name.field expression for unchanged
%%  values.

record_el(R, Name, Fs, []) ->
    record_el1(R, Name, Fs, [], element(2, R));
record_el(R, Name, Fs, Us) ->
    record_el1(R, Name, Fs, Us, element(2, hd(Us))).

record_el1(R, Name, Fs, Us, Lr) ->
    {tuple,Lr,[{atom,Lr,Name}|
	       map(fun (F) -> record_el2(R, Name, F, Us) end, Fs)]}.

record_el2(R, Name, {record_field,Lf,{atom,La,F},Val}, Us) ->
    case find_field(F, Us) of
	{ok,New} -> New;
	error -> {record_field,Lf,R,Name,{atom,La,F}}
    end.

%% record_setel(Record, RecordName, [RecDefField], [Update])
%%  Build a nested chain of setelement calls to build updated record tuple.

record_setel(R, Name, Fs, Us) ->
    foldr(fun ({record_field,Lf,Field,Val}, Acc) ->
		  I = index_expr(Lf, Field, Name, Fs),
		  {call,Lf,{atom,Lf,setelement},[I,Acc,Val]} end,
	  R, Us).

new_var(St) ->
    C = St#expand.vcount,
    V = list_to_atom("%" ++ integer_to_list(C)),
    {V,St#expand{vcount=C+1}}.

new_vars(N, St) -> new_vars(N, St, []).

new_vars(N, St0, Vs) when N > 0 ->
    {V,St1} = new_var(St0),
    new_vars(N-1, St1, [V|Vs]);
new_vars(0, St, Vs) -> {Vs,St}.

%% make_vars([VariableName], Line) -> [VariableTerm].

make_vars(Vs, Line) -> [ {var,Line,V} || V <- Vs ].

%% make_list(TermList, Line) ->	ConsTerm.

make_list(Ts, Line) ->
    foldr(fun (H, T) -> {cons,Line,H,T} end, {nil,Line}, Ts).

%% call_exit(Line, Reason) -> Expr.
%%  Build a call to erlang:exit/1 with reason Reason.

call_exit(L, R) ->
    {call,L,{remote,L,{atom,L,erlang},{atom,L,exit}},[R]}.

%% new_in(Before, RegionList) ->
%%	{NewInAll,NewInSome}
%%  Return the variables new in all clauses and those new in some clauses.

new_in(Before, Region) ->
    InAll = intersection(Region),
    InSome = union(Region),
    NewInAll = subtract(InAll, Before),
    NewInSome = subtract(subtract(InSome, Before), NewInAll),
    {NewInAll,NewInSome}.

%% For storing the import list we use our own version of the module dict.
%% This is/was the same as the original but we must be sure of the format
%% (sorted list of pairs) so we can do ordset operations on them (see the
%% the function eof/2. We know an empty set is [].

%% import(Line, Imports, State) ->
%%	State'
%% imported(Name, Arity, State) ->
%%	{yes,Module} | no
%%  Handle import declarations and est for imported functions. No need to
%%  check when building imports as code is correct.

import(Line, {Mod,Fs}, St) ->
    Mfs = list_to_set(Fs),
    St#expand{imports=add_imports(Mod, Mfs, St#expand.imports)}.

add_imports(Mod, [F|Fs], Is) ->
    add_imports(Mod, Fs, store(F, Mod, Is));
add_imports(Mod, [], Is) ->
    Is.

imported(F, A, St) ->
    case find({F,A}, St#expand.imports) of
	{ok,Mod} -> {yes,Mod};
	error -> no
    end.

%% This is our own version of the module dict which is/was the same as
%% the original but we must be sure of the format (sorted list of pairs)
%% so we can do ordset operations on them. We know an empty set is [].

%% find(Key, Dictionary) -> {ok,Value} | error

find(Key, [{K,Value}|D]) when Key > K -> find(Key, D);
find(Key, [{K,Value}|_]) when Key == K -> {ok,Value};
find(Key, [{K,Value}|_]) when Key < K -> error;
find(Key, []) -> error.

%% store(Key, Value, Dictionary) -> Dictionary.

store(Key, New, [{K,Old}|Dict]) when Key > K ->
    [{K,Old}|store(Key, New, Dict)];
store(Key, New, [{K,Old}|Dict]) when Key == K ->
    [{Key,New}|Dict];
store(Key, New, [{K,Old}|Dict]) when Key < K ->
    [{Key,New},{K,Old}|Dict];
store(Key, New, []) -> [{Key,New}].
