%% ``The contents of this file are subject to the Erlang Public License,
%% Version 1.0, (the "License"); you may not use this file except in
%% compliance with the License. You may obtain a copy of the License at
%% http://www.erlang.org/EPL1_0.txt
%% 
%% Software distributed under the License is distributed on an "AS IS"
%% basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
%% the License for the specific language governing rights and limitations
%% under the License.
%% 
%% The Original Code is Erlang-4.7.3, December, 1998.
%% 
%% The Initial Developer of the Original Code is Ericsson Telecom
%% AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
%% Telecom AB. All Rights Reserved.
%% 
%% Contributor(s): ______________________________________.''
%%
-module(jam_optimize).
-copyright('Copyright (c) 1991-97 Ericsson Telecom AB').
-vsn('$Revision: /main/release/free/1').

-export([optimse/1,jump_through/1,succ/1]).

-import(jam_compile,[print_code_on_stream/2]).

-import(lists, [reverse/1, reverse/2]).

copyright() ->
    "Copyright (C) 1991, Ellemtel Telecommunications Systems Laboratories".

optimse(L) -> 
	% io:format("optimse~n",[]),print_code_on_stream(tty,L),
	L1 = opt(L),
	% io:format("optimse result~n",[]),print_code_on_stream(tty,L1),
	L1.

%______________________________________________________________________
%% opt(Code) -> Code' 
%%    optimise Code

opt(L) ->
	opt(L,[jump_through,succ]).

%______________________________________________________________________
%% opt(Code, Opts) -> Code' 
%%    optimise Code (Opts is a list of optimisations to try)

opt(L,Opts) -> opt(L,Opts,false).

opt(L,[H|T],Flag) ->
    case apply(jam_optimize,H,[L]) of
	{true,L1} ->
	    opt(L1,T,true);
	false ->
	    opt(L,T,Flag)
    end;
opt(L,[],true) -> 
    opt(L);
opt(L,[],false) ->
    L.

%______________________________________________________________________

% each optimisation step must return
% {true,OptimisedCode} or false

jump_through(L) -> 
	case goto_replacement(L,[]) of
	    {true,Label,Instr,L1} ->
	        replace_goto(L1,Label,Instr,[]);
	    false ->
	        false
	end.

%______________________________________________________________________
%% goto_replacement(Code, Before) ->
%%   {true, Label, Instr, Code'} | false
%%   if true then all instances of {goto, Label} in the input
%%   can be replaced by Instr

goto_replacement([{label,A},{goto,B}|T],Before) ->
    {true,A,{goto,B},reverse(Before,[{goto,B}|T])};
goto_replacement([{label,A},ret|T],Before) ->
    {true,A,ret,reverse(Before,[ret|T])};
goto_replacement([H|T],Before) ->
    goto_replacement(T,[H|Before]);
goto_replacement([],_) -> false.

replace_goto([{goto,L}|T],L,Instr,Before) ->
    report_opt(goto_replacement),
    replace_goto(T,L,Instr,[Instr|Before]);
replace_goto([H|T],L,Instr,Before) ->
    replace_goto(T,L,Instr,[H|Before]);
replace_goto([],_,_,Before) ->
    {true,reverse(Before)}.

%______________________________________________________________________
%% succ(Code) ->
%%   {true, Code'} | false

succ(L) -> succ(L,[], false).

succ([{call,Where,Mod,Func,Arity},ret|T], Before, _) ->
    report_opt(lco),
    succ([{enter,Where,Mod,Func,Arity}|T], Before, true);
succ([{pushInt,N},pop|T],Before, _) ->
    report_opt(pushInt_pop),
    succ(T, Before, true);
succ([{pushAtom,N},pop|T], Before, _) ->
    report_opt(pushAtom_pop),
    succ(T, Before, true);
succ([{goto,L},{X,Y}|T], Before, _) when X /= label ->
    report_opt(dead_code),
    succ([{goto,L}|T], Before, true);
succ([ret,{X,Y}|T], Before, _) when X /= label ->
    report_opt(dead_code),
    succ([ret|T], Before, true);
succ([{enter,M,F,A},{X,Y}|T], Before, _) when X /= label ->
    report_opt(dead_code),
    succ([{enter,M,F,A}|T],Before, true);
succ([{arg,N},{storeVar,{_,{arg,N}}}|T], Before, _) ->
    report_opt(push_store_save_var),
    succ(T, Before, true);
succ([{arg,N},{eqVar,{_,{arg,N}}}|T], Before, _) ->
    report_opt(push_eq_var),
    succ(T, Before, true);
succ([{arg,N},pop|T], Before, _) ->
    report_opt(arg_pop),
    succ(T, Before, true);
succ([{alloc,0}|T], Before, _) ->
    report_opt(alloc0),
    succ(T, Before, true);
succ([dup,{storeVar,X}, pop|T], Before, _) ->
    report_opt(dup_store_pop),
    succ([{storeVar,X}|T], Before, true);
succ([{storeVar,X}, {pushVar,X}, ret|T], Before, _) ->
    report_opt(store_push_ret),
    succ([ret|T], Before, true);
succ([H|T], Before, Flag) ->
    succ(T, [H|Before], Flag);
succ([],Before, true) ->
    {true,reverse(Before)};
succ([], _,  _) ->
    false.

report_opt(_) ->
    true;
report_opt(Opt) ->
    io:format("~w ",[Opt]).



