%% ``The contents of this file are subject to the Erlang Public License,
%% Version 1.0, (the "License"); you may not use this file except in
%% compliance with the License. You may obtain a copy of the License at
%% http://www.erlang.org/EPL1_0.txt
%% 
%% Software distributed under the License is distributed on an "AS IS"
%% basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
%% the License for the specific language governing rights and limitations
%% under the License.
%% 
%% The Original Code is Erlang-4.7.3, December, 1998.
%% 
%% The Initial Developer of the Original Code is Ericsson Telecom
%% AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
%% Telecom AB. All Rights Reserved.
%% 
%% Contributor(s): ______________________________________.''
%%
%%
%% jam_encode.erl
%%
%%              Copyright (c) 1988 Ellemtel Utveklings AB
%%
%% Author: Joe Armstrong
%% Creation Date: 1989-11-08
%% Purpose:
%%      opcode descriptions
%%       This file must be munged through make_opcdoes.awk to turn
%%       it into a proper erlang file !

%% Modifications:
%%   930318 -- added bumps
%%   930611 -- 3 byte opcodes
%%     	Old   New  Instruction
%%   	176   242  try_me_else
%%	188   243  goto
%%	190   244  callLocal
%%      192   245  enterLocal
%%      198   246  pushCatch
%%      202   247  wait
%%   930907 -- loosing 3 bits of integer precision for integers > 24 bits
%%      ---   248  pushInt4
%%      ---   249  getInt4
%% Tony:
%%   950201 -- Added pushIntN and getIntN for bignum constants.
%%   950301 -- Used the -define(OP, Number) for all opodes from
%%             opcodes.erl
%%             Added code generation for pushStr and getStr
%%   950302 -- Added Ops:
%%             type
%%             hash
%%             head
%%             tail
%%             list_length
%%             heap_need
%%             stack_need
%%             gotoix
%%
%% Joe:
%%   950420 -- Removed Ops:
%%             bump
%%     

-module(jam_encode).
-copyright('Copyright (c) 1991-97 Ericsson Telecom AB').
-vsn('$Revision: /main/release/free/1').
-export([form/1, magic/0]).

%% encode:form(X) -> {CodeBytes,Len,Actions}
%%   encode a form
%%   CodeBytes = list of integers
%%   Len = length of code
%%   Actions = list of actions to perform

-import(jam_asm,[as_bytes2/1, as_bytes3/1, as_bytes4/1]).
-import(lists, [append/2, append/1]).

-include("jam_opcodes.hrl").
%%
copyright() ->
    "Copyright (C) 1991, Ellemtel Telecommunications Systems Laboratories".

%______________________________________________________________________
%% encode:form(Form) -> {Bytes,Length,Patches} | error

form(X) ->
	case form1(X) of
	    N when integer(N) ->
		{[N],1,[]};
	    {error,Wot} ->
		{error,Wot};
	    {B,A} ->
		{B,length(B),A}
	end.

%% the N's in the following are >= 0

form1({pushVar,{_,{var,N}}}) when N =< 15 	-> ?pushVar_0 + N;
form1({pushVar,{_,{var,N}}}) when N =< 255     	-> {[?pushVarN,N],[]};
form1({storeVar, {_,{var,N}}}) when N =< 15 	-> ?storeVar_0 + N;
form1({storeVar, {_,{var,N}}}) when N =< 255    -> {[?storeVarN,N],[]};
form1({alloc,  N}) when N =< 15 		-> ?alloc_0 + N;
form1({alloc,  N}) when N =< 255 		-> {[?allocN,N],[]};
form1({eqVar,{_,{var,N}}})when N=<15 		-> ?eqVar_0 + N;
form1({eqVar,{_,{var,N}}}) when N =< 255    	-> {[?eqVarN,N],[]};
form1({eqVar,{_,{arg,N}}})when N=<15 		-> ?eqArg_0 + N;
form1({eqVar,{_,{arg,N}}}) when N =< 255    	-> {[?eqArgN,N],[]};

form1({arg,N}) when N =< 15 		-> ?arg_0 + N;
form1({arg,N}) when N =< 255        	-> {[?argN,N],[]};
form1({unpkTuple,N})when N =< 15 	-> ?unpkTuple_0 + N;
form1({unpkTuple,N})when N =< 255	-> {[?unpkTupleN,N],[]};
form1({mkTuple,N})when N =< 15		-> ?mkTuple_0 + N;
form1({mkTuple,N})when N =< 255		-> {[?mkTupleN,N],[]};

%% getInt: 137+N (0  <= N <= 15
%%         153    16 <= N <= 255
%%         249    N is 32 bit (28 bit)
%%         222    N is K bit  (K > 28)
%%
form1({getInt,N}) -> encode_int_op(?getInt_0,
				   ?getInt1,
				   ?getInt4,
				   ?getIntN, N);

form1({getFloat,N}) -> {[?getFloat,0,0,0,0,0,0,0,0],[{1,{float,N}}]};
form1({getStr,Str}) -> encode_str_op(?getNil, ?getStr, Str);
form1({getAtom,A})  -> {[?getAtom,0,0],[{1,{const,A}}]};
form1(getNil) -> ?getNil;

%%
%% pushInt: 156 + N   0 <= N <= 15
%%          172       16 <= N <= 255
%%          248       N is 32 bit
%%          221       N is K bit (K > 28)
%%
form1({pushInt,N}) -> encode_int_op(?pushInt_0, 
				    ?pushInt1,
				    ?pushInt4,
				    ?pushIntN, N);

form1({pushFloat,N}) -> {[?pushFloat,0,0,0,0,0,0,0,0],[{1,{float,N}}]};
form1({pushStr,Str}) -> encode_str_op(?pushNil, ?pushStr, Str);
form1({pushAtom,A})  -> {[?pushAtom,0,0],[{1,{const,A}}]};
form1(pushNil) -> ?pushNil;

form1({debug_info,Mod,Func,Arity}) ->
    {[?debug_info,Arity,0,0,0,0,0],
     [{2,{const,Mod}},{4,{const,Func}}]};
form1({info,Mod,Func,Arity}) ->
    {[?nodebug_info,Arity,0,0,0,0],
     [{2,{const,Mod}},{4,{const,Func}}]};

form1({gotoix,K,L,Ls}) ->
    {[?gotoix| append([as_bytes2(K),as_bytes2(L),label_list(K,[])])],
     label_patch(Ls,5)};

form1({hash,N})                 -> {[?hash | as_bytes2(N)],[]};
form1({type,Ts})                -> {[?type | as_bytes2(type_mask(Ts,0))],[]};
form1({heap_need,N})            -> {[?heap_need | as_bytes2(N)], []};
form1({stack_need,N})           -> {[?stack_need | as_bytes2(N)], []};
form1({try_me_else,Label})      -> {[?try_me_else,0,0,0],[{1,{label,Label}}]};
form1(try_me_else_fail)	        -> ?try_me_else_fail;
form1(commit)			-> ?commit;
form1(ret)			-> ?ret;
form1(pop) 			-> ?pop;
form1(head)                     -> ?head;
form1(tail)                     -> ?tail;
form1(list_length)              -> ?list_length;
form1(mkList)			-> ?mkList;
form1(unpkList) 		-> ?unpkList;
form1(send) 			-> ?send;
form1(self) 			-> ?self;
form1({goto,Label})		-> {[?goto,0,0,0],[{1,{label,Label}}]};
form1({label,X}) 		-> {[],[{0,{noteLabel,X}}]};
form1({call,local,_,F,Arity})	-> {[?call_local,Arity,0,0,0],
				    [{2,{local,F,Arity}}]};
form1({call,remote,M,F,Arity})	-> {[?call_remote,Arity,0,0],
				    [{2,{csa,M,F,Arity}}]};
form1({enter,local,_,F,Arity})  -> {[?enter_local,Arity,0,0,0],
				    [{2,{local,F,Arity}}]};
form1({enter,remote,M,F,Arity}) -> {[?enter_remote,Arity,0,0],
			            [{2,{csa,M,F,Arity}}]};
form1(dup)			-> ?dup;
form1(popCommit)		-> ?popCommit;
form1(failIf)			-> ?failIf;
form1(failCase)			-> ?failCase;
form1({pushCatch,Label})	-> {[?pushCatch,0,0,0],[{1,{label,Label}}]};
form1(popCatch)			-> ?popCatch;
form1(setTimeout)		-> ?setTimeout;
form1(wait)			-> ?wait;
form1({wait,Label})		-> {[?wait1,0,0,0],[{1,{label,Label}}]};
form1(popCommitJoin)            -> ?popCommitJoin;
form1(save)			-> ?save;
form1({arith,'+'})		-> ?arith_plus;
form1({arith,'-'})		-> ?arith_minus;
form1({arith,'*'})		-> ?arith_times;
form1({arith,'/'})		-> ?arith_div;
form1({arith,'neg'})            -> ?arith_neg;
form1({arith,'div'})		-> ?arith_intdiv;
form1({arith,'band'})		-> ?arith_band;
form1({arith,'bor'})		-> ?arith_bor;
form1({arith,'bxor'})		-> ?arith_bxor;
form1({arith,'bnot'})		-> ?arith_bnot;
form1({arith,'bsl'})		-> ?arith_bsl;
form1({arith,'bsr'})		-> ?arith_bsr;
form1({arith,'rem'})		-> ?arith_rem;
form1({comp,'>'})		-> ?comp_gt;
form1({comp,'<'})		-> ?comp_lt;
form1({comp,'>='})		-> ?comp_geq;
form1({comp,'=<'})		-> ?comp_leq;
form1({comp,'=:='}) 		-> ?exact_eqeq;
form1({comp,'=/='}) 		-> ?exact_neq;
form1({comp,'=='}) 		-> ?comp_eqeq;
form1({comp,'/='}) 		-> ?comp_neq;

form1({test,integer,1}) 	-> ?test_integer;
form1({test,float,1}) 		-> ?test_float;
form1({test,number,1}) 		-> ?test_number;
form1({test,atom,1}) 		-> ?test_atom;
form1({test,constant,1}) 	-> ?test_constant;
form1({test,list,1}) 		-> ?test_list;
form1({test,tuple,1}) 		-> ?test_tuple;
form1({test,pid,1}) 	        -> ?test_pid;
form1({test,reference,1}) 	-> ?test_reference;
form1({test,port,1})	 	-> ?test_port;
form1({test,binary,1})	 	-> ?test_binary;

form1(X) 			-> {error,X}.

%%
%% Encode the string as a pushNil,pushStr or getNil,getStr
%%
encode_str_op(OP, _, "") -> {[OP],[]};
encode_str_op(_, OP, Str) ->
    L = length(Str),
    {append([OP|as_bytes2(L)], Str), []}.

encode_int_op(OP0, OP1, OP4, OPN, N) ->
    if
	0 =< N, N =< 15 -> OP0 + N;
	N > 15, N =< 255 -> {[OP1,N],[]};
	N >= -16#08000000, N =< 16#07ffffff -> {[OP4|as_bytes4(N)], []};
	true ->  {big_op(OPN, N), []}
    end.

big_op(OPN, N) when N < 0 ->
    L = as_bytesN(-N),
    [OPN | append(as_bytes4(length(L)), [1|L])];
big_op(OPN, N) ->
    L = as_bytesN(N),
    [OPN | append(as_bytes4(length(L)), [0|L])].

as_bytesN(0) -> [0];
as_bytesN(N) -> bytesN(N).

bytesN(0) -> [];
bytesN(N) -> [N band 16#ff | bytesN(N bsr 8)].

%%
%% Contruct a type mask for the type op
%% 2 bytes
%%
type_mask(['integer'|Ts],M)   -> type_mask(Ts, M bor (1 bsl ?integer_bit));
type_mask(['float'|Ts],M)     -> type_mask(Ts, M bor (1 bsl ?float_bit));
type_mask(['atom'|Ts],M)      -> type_mask(Ts, M bor (1 bsl ?atom_bit));
type_mask(['reference'|Ts],M) -> type_mask(Ts, M bor (1 bsl ?reference_bit));
type_mask(['port'|Ts],M)      -> type_mask(Ts, M bor (1 bsl ?port_bit));
type_mask(['pid'|Ts],M)       -> type_mask(Ts, M bor (1 bsl ?pid_bit));
type_mask(['tuple'|Ts],M)     -> type_mask(Ts, M bor (1 bsl ?tuple_bit));
type_mask(['nil'|Ts],M)       -> type_mask(Ts, M bor (1 bsl ?nil_bit));
type_mask(['list'|Ts],M)      -> type_mask(Ts, M bor (1 bsl ?list_bit));
type_mask(['binary'|Ts],M)    -> type_mask(Ts, M bor (1 bsl ?binary_bit));
type_mask([], M) -> M.

%%
%% Construct a position list for labels (jump table)
%%
label_list(0, L) ->
    L;
label_list(K, L) ->
    label_list(K-1,[0,0,0 | L]).

%%
%% Construct a label patch list with offsets (jump table)
%%
label_patch([L | Ls], Offs) ->
    [{Offs,{label,L}} | label_patch(Ls, Offs+3)];
label_patch([], _) -> [].

magic() ->
    '1.0 Fri Feb 3 09:55:56 MET 1995'.

