%% ``The contents of this file are subject to the Erlang Public License,
%% Version 1.0, (the "License"); you may not use this file except in
%% compliance with the License. You may obtain a copy of the License at
%% http://www.erlang.org/EPL1_0.txt
%% 
%% Software distributed under the License is distributed on an "AS IS"
%% basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
%% the License for the specific language governing rights and limitations
%% under the License.
%% 
%% The Original Code is Erlang-4.7.3, December, 1998.
%% 
%% The Initial Developer of the Original Code is Ericsson Telecom
%% AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
%% Telecom AB. All Rights Reserved.
%% 
%% Contributor(s): ______________________________________.''
%%
%% Copyright (C) 1993, Ellemtel Telecommunications Systems Laboratories
%% File    : jam_compile.erl
%% Author  : Joe Armstrong
%% Purpose : Compiler

-module(jam_compile).
-copyright('Copyright (c) 1991-97 Ericsson Telecom AB').
-vsn('$Revision: /main/release/free/1').

-copyright(
 'Copyright (C) 1995, Ellemtel Telecommunications Systems Laboratories').

-define(stackMargin, 32).

-export([module/4]).
-export([format_error/1]).

-import(lists,    	[member/2, reverse/1, reverse/2, append/1, append/2]).

module(Mod, Exports, Forms, Opts) ->
    %% io:format("Here1: ~p\n", [{compile, Mod, Exports, Forms, Opts}]),
    Forms2 = do_lambdas(Mod, Forms, []),
    do_compile(Mod, Exports, Forms2, Opts, [], []).

do_compile(Mod, Exports, [{function, Line, Name, Arity, Clauses}|T], 
	   Opts, Errs, CodeBuff) ->
    Form = {function, Line, Name, Arity, Clauses},
    %% io:format("compiling: ~w:~w/~w\n", [Mod, Name, Arity]),
    Type = case lists:member({Name,Arity}, Exports) of
	       true  -> exported;
	       false -> local
	   end,
    case function(Mod, Form, Opts) of
	{code, C} ->
	    C1 = {code, Mod, Name, Arity, Type, C},
	    do_compile(Mod, Exports, T, Opts, Errs, [C1|CodeBuff]);
	{error, What} ->
	    do_compile(Mod, Exports, T, Opts, What++Errs, [])
    end;
do_compile(Mod, Exports, [{asm, Name, Arity, C} | T],
	   Opts, Errs, CodeBuff) ->
    Type = case lists:member({Name,Arity}, Exports) of
	       true  -> exported;
	       false -> local
	   end,
    C1 = {code, Mod, Name, Arity, Type, C},
    do_compile(Mod, Exports, T, Opts, Errs, [C1 | CodeBuff]);
do_compile(Mod, Exports, [H|T], Opts, Errs, CodeBuff) ->
    do_compile(Mod, Exports, T, Opts, 
	       [{none,jam_compile,{unknown_form,H}}|Errs], CodeBuff);
do_compile(Mod, Exports, [], Opts, [], CodeBuff) ->
    {ok, CodeBuff};
do_compile(Mod, Exports, [], Opts, Errs, CodeBuff) ->
    {error, Errs}.

%%
%% Translate module_lambdas/4 into several local functions
%% i.e
%%    module_lambdas(Ni, Unique, Args, Free) ->  Body1
%%
%%    module_lambda_Ni(Unique, Args, Free) -> Body1
%%
%%    module_lambdas(I,Unique,Args,Free) ->
%%        select I of
%%           0 -> module_lambdas_0(Unique,Args,Free);
%%           1 -> module_lambdas_1(Unique,Args,Free)
%%           ...
%%           K -> module_lambdas_K(Unique,Args,Free);
%%           _ -> erlang:exit({undef_lambda,lam})
%%        end
%%
do_lambdas(Mod, [{function,Line,module_lambdas,4, Clauses} | Fs], Fs1) ->
    Fs2 = do_lambda_funs(Mod, Clauses),
    Fs ++ Fs2 ++ Fs1;
do_lambdas(Mod, [F | Fs], Fs1) ->
    do_lambdas(Mod, Fs, [F|Fs1]);
do_lambdas(Mod, [], Fs1) ->
    Fs1.

do_lambda_funs(Mod, [{clause,Line,[{var,Line,I}|Hs],G,B}]) ->
    [{function,Line,module_lambdas,4,[{clause,Line,[{var,Line,I}|Hs],G,B}]}];
do_lambda_funs(Mod, Cs) ->
    do_lambda_funs(Mod, Cs, []).
     
    
%% create local lambda functions
do_lambda_funs(Mod, [{clause,Line,H,G,B} | Cs], Fs) ->
    [{integer,_,N} | H2] = H,  %% get index
    {FN,Cs2} = do_lambda_fun(Cs, N, [{clause,Line,H2,G,B}]),
    do_lambda_funs(Mod, Cs2, [FN | Fs]);
do_lambda_funs(Mod, [], Fs) ->
    %% check that all indexes are present [0..N-1]
    N = length(Fs),
    Ns = lists:map(fun({K,_}) -> K end, Fs),
    Funs = lists:map(fun({_,F}) -> F end, Fs),
    Ix = lists:seq(0, N-1),
    [] = Ns -- Ix,
    %% generate the asm index function 
    Calls = lists:map(
	      fun(K) ->
		      Name = list_to_atom("module_lambda_" ++ 
					  integer_to_list(K)),
		      [{label,K},
		       popCommit,
		       {enter,local,[],Name,3}]
	      end, Ns),
    Fail = 
	[{label,N},
	 {pushAtom,lambda_clause},
	 {pushAtom,lam},
	 {mkTuple,2},
	 {enter,remote,erlang,exit,1},
	 ret ],

    Cases = append(Calls) ++ Fail,

    [ {asm, module_lambdas, 4,
       [{info, Mod, module_lambdas, 4},
	{try_me_else, N},
	{arg,1},
	{arg,2},
	{arg,3},
	{arg,0},
	{gotoix, N, 0, Ix} | Cases]} | Funs].


%% gather one lambda function (matching the index N)
do_lambda_fun([{clause,Line,[{integer,_,N} | H2],G,B} | Cs], N, LCs) ->
    do_lambda_fun(Cs, N, [{clause,Line,H2,G,B} | LCs]);
do_lambda_fun([{clause,Line,[{var,_,_} | _],_,_}], N, LCs) ->
    { make_lambda_fun(N, reverse(LCs)), []};
do_lambda_fun(Cs, N, LCs) ->
    { make_lambda_fun(N, reverse(LCs)), Cs }.

%% make the local function
make_lambda_fun(N, Clauses) ->
    Name = list_to_atom("module_lambda_" ++ integer_to_list(N)),
    [{clause,Line,_,_,_} | _] = Clauses,
    {N, {function, Line, Name, 3, Clauses}}.



%% usage:  jam_compile:function(Func)
%%   returns abstract form of code
%%   Func is a {function1,Module,Type,Name,Arity,Clause2} 
%%   Form as retured by sys_procompile
%%   Module   = the module name of the function
%%   Type     = exported | local
%%   Name     = function name (atom)
%%   Arity    = function arity (integer)
%%   Clauses2 = clauses

%% Modifications:
%%   911001 -- adding stub generation
%%   930318 -- added bumps
%%   930326 -- Joe Armstrong
%%      Changed {number, N} wrappers to
%%      {integer, N} or {float, N}
%%
%%   Tony:
%%
%%   950301 -- added opcodes for strings pushStr and getStr
%%
%%   Joe:
%%   9504230 -- removed bumps


%%______________________________________________________________________
%% function(Mod, Func, Options) -  compile a function
%%   Func is an abstract form returned by the parser
%%   Options is a list of options from the user
%%   -> {code,Code}
%%   -> {error, Type, What}

function(Mod, Func, Options) ->
    case 
	catch % can be commented out when debugging !
	function1(Mod, Func, Options) of
	    {code,X} 		-> {code,X};
	    {error, X}          -> {error, [X]};
	    {'EXIT',X} 		-> {error, [{jam_compile, {system, X}}]}
    end.

function1(Mod, Func, Options) ->
    {Code,E1} = compile_clauses(Mod, Func,new_env(Func, Options)),
    {code,Code}.

%%______________________________________________________________________
%% new_env(...) -> Env

new_env(Func,Options) ->
    MaxLabel = 0,
    {env,alloc,MaxLabel,Options,Func}.

%%______________________________________________________________________
%% The Environment -- usually abbreviated to E !

alloc(E) -> 		element(2,E).
set_alloc(Alloc,E) -> 	setelement(2,E,Alloc).
new_label(E) ->  	Label = element(3,E) + 1,
	       		{Label,setelement(3,E,Label)}.
debug_flag(E) ->	member(debug, element(4,E)).
trace_flag(E) ->        member(trace, element(4, E)).

%% get_function(E) ->	element(5,E).
%% get_module_name(E) ->   element(2,get_function(E)).

clear_alloc(Alloc, E) ->
    clear_alloc(Alloc, [], E).

clear_alloc([{VarName, Where}|T], L, E) ->
    clear_alloc(T, [{VarName, {Where, unset}}|L], E);
clear_alloc([], L, E) ->
    set_alloc(L, E).

%%______________________________________________________________________
%% tset_var(Var, Env} -> {VarStatus, Where, Env'}
%%   VarStatus =  (set | unset) 
%%   Where = {arg, N} | {var, N} 
%%   If unset the variable is set in Env'

tset_var(VarName, E) -> 
    case variable_status(VarName, E) of
	{{Where, unset}, L1, L2} ->
	    Alloc = reverse([{VarName, {Where, set}}|L1], L2),
	    {unset, Where, set_alloc(Alloc,E)};
        {{Where, set}, L1, L2} ->
	    {set, Where, E}
    end.

variable_status(VarName, E) ->
    variable_status(VarName, alloc(E), []).

variable_status(VarName, [{VarName, Status}|T], L) ->
    {Status, T, L};
variable_status(VarName, [H|T], L) ->
    variable_status(VarName, T, [H|L]).

%%______________________________________________________________________
%%%  all the routine in compile are of the form
%%%    are Compile(Form,Env) -> {Code,Env'}

%%______________________________________________________________________
%% compile_clauses(Func,Env) -> {Code, Env'}

compile_clauses(Module,{function,Line,Name,Arity,Clauses},E0) ->
    Clauses2 = alloc_vars(Clauses),
    {Code1,E1} = compile_clause_list(Clauses2,E0),
    Instr = case trace_flag(E0) of
		true ->
		    debug_info;
		false ->
		    info
	    end,
    Code2 = [{Instr,Module,Name,Arity}|Code1],
    Code3 = optimise(Code2),
    {Code3,E1}.

compile_clause_list([X],E0) ->
    {Code,E1} = compile_clause(X,E0),
    {append([try_me_else_fail|Code],[ret]),E1};
compile_clause_list([H|T],E0) ->
    {Label,E1} = new_label(E0),
    {Code1,E2} = compile_clause(H,E1),
    {Code2,E3} = compile_clause_list(T,E2),
    {append([{try_me_else,Label}|Code1],[ret,{label,Label}|Code2]),E3}.

compile_clause(Clause,E0) ->
    {clause2,Line,Head,Guard,Body,{Max,Alloc}} = Clause,
    E1 = clear_alloc(Alloc,E0),
    dformat("compile head:~w~n",[Head],E0),
    {CodeH,E2} = compile_head(Head,E1),
    StackNeedMatch = stack_depth_match(CodeH),
    CodeH1 = add_stack_need_instruction(StackNeedMatch, CodeH),
    {CodeG,E3} = compile_guard(Guard,E2),
    CodeG1 = add_stack_need(CodeG),
    {CodeB,E4} = compile_seq(Body,E3),
    CodeB1 = add_stack_need(CodeB),
    {append([[{alloc,Max}|CodeH1],CodeG1,[commit|CodeB1]]),E4}.

add_stack_need_instruction(Need, Code) when Need > ?stackMargin ->
    [{stack_need, Need}|Code];
add_stack_need_instruction(_, Code) ->
    Code.

%%______________________________________________________________________
%% compile_head(Args, Env) -> {Code, Env'}.

compile_head(Args, E) ->
    match_args(Args, 0, E).

%%__________________________________________________________________
%% compile_guard(Guard,Env) -> {Code, Env'}.
%%   should leave nothing extra on the stack      

compile_guard([],E) ->
    {[],E};
compile_guard([H|T],E0) ->
    dformat("compile guard element:~w~n",[H],E0),
    {Code1,E1} = compile_guard_element(H,E0),
    dformat("Guard element Code:~w~n",[Code1],E0),
    {Code2,E2} = compile_guard(T,E1),
    {append(Code1,Code2),E2}.

%%  when compiling a guard elemet we have to be sure we don't accidently
%%  clobber anything on the stack so .......

compile_guard_element({op,_,Op,X,Y},E0) ->
    build_and_call([X,Y],{comp,Op},E0);
compile_guard_element({call,Line,{atom,_,Type},Args},E0) ->
    %% Args is a list of arguments that are to be passed into
    %% the type test  e.g integer(A), arity(T,3) ...
    build_and_call(Args,{test,Type,length(Args)},E0).

%%______________________________________________________________________
%%   compile_seq(Body,Env)

compile_seq([H],E0) ->
    compile_expr(H,E0);
compile_seq([H|T],E0) ->
    {Code1,E1} = compile_expr(H,E0),
    {Code2,E2} = compile_seq(T,E1),
    {append(Code1,[pop|Code2]),E2}.

%%______________________________________________________________________
%% compile_expr(Rhs,Env) -> {Code,Env'}
%%   compile an expression onto the Stack

compile_expr(Rhs,E0) ->
    {Code,E1} = compile_expr1(Rhs,E0),
    dformat("code for:~w = ~w~nalloc=~w~n",[Rhs,Code,alloc(E1)],E1),
    {Code,E1}.

compile_expr1({nil,_},E0) ->
    {[pushNil],E0};
compile_expr1({call,_,{remote,_,{atom,_,erlang},{atom,_,self}}, []}, E0) ->
    {[self], E0};
compile_expr1({call,_,{remote,_,{atom,_,erlang},{atom,_,'hd'}}, [X]}, E0) ->
    build_and_call([X], head, E0);
compile_expr1({call,_,{remote,_,{atom,_,erlang},{atom,_,'tl'}}, [X]}, E0) ->
    build_and_call([X], tail, E0);
compile_expr1({call,_,{remote,_,{atom,_,erlang},{atom,_,'length'}},[X]}, E0) ->
    build_and_call([X], list_length, E0);
compile_expr1({call,_,{remote,_,{atom,_,Module},{atom,_,Function}},Args},E0) ->
    build_and_call(Args,{call,remote,Module,Function,length(Args)},E0);
compile_expr1({call,_,{atom,_,Function},Args},E0) ->
    build_and_call(Args,{call,local,[],Function,length(Args)},E0);
compile_expr1({tuple,_,Args},E0) ->
    build_and_call(Args,{mkTuple,length(Args)},E0);
compile_expr1({cons,_,H,T},E0) ->
    build_and_call([T,H],mkList,E0);
compile_expr1({var,Line,VarName},E0) ->
    case tset_var(VarName,E0) of
	{set, {var,N}, E1} ->
	    {[{pushVar,{VarName,{var,N}}}],E1};
	{set, {arg,N}, E1} ->
	    {[{arg,N}],E1};
	{unset, _, _} ->
	    %% Uuum lint should have got this one
	    %% the real error is in lint but we'll be "user friendly"
	    io:format("*** unset variable:~p in line:~w~n", [VarName, Line]),
	    exit(lint_error)
    end;
compile_expr1({string,_,X},E0) ->
    {[{pushStr,X}], E0};
compile_expr1({atom,_,A},E0) -> 
    {[{pushAtom,A}],E0};
compile_expr1({integer,_,N},E0) ->
    {[{pushInt,N}],E0};
compile_expr1({float,_,N},E0) ->
    {[{pushFloat,N}],E0};
compile_expr1({match,_,Lhs,Rhs},E0) ->
    %% the value of match is the value of Rhs
    %% so .. we first build the Rhs onto the stack
    %% then duplicate it then match it -- the match removes
    %% the top of stack leaving the correct Rhs which was duped ..
    {Code1,E1} = compile_expr(Rhs,E0),
    {Code2,E2} = match(Lhs,E1),
    {append(Code1,[dup|Code2]),E2};
compile_expr1({op, Line, '++', List1, List2}, E0) ->
    %% compile_expr1({call,Line,erlang,'append', [List1, List2]}, E0);
    build_and_call([List1, List2], {call,remote,erlang,append,2}, E0);
compile_expr1({op, Line, '--', List1, List2}, E0) ->
    %% compile_expr1({call,Line,erlang,'subtract', [List1, List2]}, E0).
    build_and_call([List1, List2], {call,remote,erlang,subtract,2}, E0);
compile_expr1({op,Line,'^',X1,X2},E0) ->
    %% compile_expr({call,Line,math,pow,[X1,X2]}, E0);
    build_and_call([X1,X2], {call,remote,math,pow,2}, E0);
compile_expr1({op,_,'!',Id,Mess},E0) ->
    build_and_call([Id,Mess],send,E0);

compile_expr1({op,_,'and',X,Y},E0) ->
    build_and_call([X,Y], {call,remote,erlang,sand,2}, E0);
compile_expr1({op,_,'or',X,Y},E0) ->
    build_and_call([X,Y], {call,remote,erlang,sor,2}, E0);
compile_expr1({op,_,'xor',X,Y},E0) ->
    build_and_call([X,Y], {call,remote,erlang,sxor,2}, E0);
compile_expr1({op,_,'>',X,Y},E0) ->
    build_and_call([X,Y], {call,remote,erlang,sgt,2}, E0);
compile_expr1({op,_,'>=',X,Y},E0) ->
    build_and_call([X,Y], {call,remote,erlang,sge,2}, E0);
compile_expr1({op,_,'<',X,Y},E0) ->
    build_and_call([Y,X], {call,remote,erlang,sgt,2}, E0);
compile_expr1({op,_,'=<',X,Y},E0) ->
    build_and_call([Y,X], {call,remote,erlang,sge,2}, E0);
compile_expr1({op,_,'==',X,Y},E0) ->
    build_and_call([X,Y], {call,remote,erlang,seqeq,2}, E0);
compile_expr1({op,_,'=:=',X,Y},E0) ->
    build_and_call([X,Y], {call,remote,erlang,seq,2}, E0);
compile_expr1({op,_,'/=',X,Y},E0) ->
    build_and_call([X,Y], {call,remote,erlang,sneqeq,2}, E0);
compile_expr1({op,_,'=/=',X,Y},E0) ->
    build_and_call([X,Y], {call,remote,erlang,sneq,2}, E0);

compile_expr1({op,_,Op,X1,X2},E0) ->
    build_and_call([X1,X2],{arith,Op},E0);
compile_expr1({op,_,'bnot',X},E0) ->
    build_and_call([X],{arith,'bnot'},E0);
compile_expr1({op,_,'-',X}, E0) ->
    build_and_call([X],{arith,'neg'},E0);
compile_expr1({op,_,'+',X}, E0) ->
    compile_expr(X, E0);

compile_expr1({op,_,'not',X}, E0) ->
    build_and_call([X], {call,remote,erlang,snot,1}, E0);

compile_expr1({'if',Line,Options},E0) ->
    compile_if({'if',Line,Options},E0);
compile_expr1({'case',Line,Func,Options},E0) ->
    compile_case({'case',Line,Func,Options},E0);
compile_expr1({'receive',Line,Opts,Func,TimeOutBody},E0) ->
    compile_receive({'receive',Line,Opts,Func,TimeOutBody},E0);
compile_expr1({'receive',Line,Opts},E0) ->
    compile_expr1({'receive',Line,Opts, [], []},E0);
compile_expr1({'catch',_,Func},E0) ->
    {Label,E1} = new_label(E0),
    {Code,E2} = compile_expr(Func,E1),
    {append([{pushCatch,Label}|Code],[{label,Label},popCatch]),E2};
compile_expr1({block, _, B}, E0) ->
    compile_seq(B, E0).

%%______________________________________________________________________
%% match_args(List, Start, Env) -> {Code,Env'}.
%%   match elements in List in consequtive Arg registers starting at Start

match_args([H|T], N, E0) ->
    {CodeH,E1} = match(H,E0),
    {CodeT,E2} = match_args(T,N + 1,E1),
    {append([{arg,N}|CodeH],CodeT),E2};
match_args([],_,E) ->
    {[],E}.

%%______________________________________________________________________
%% match_list(L, Env) -> {Code,Env'}
%%   match a list of arguments L

match_list([H|T],E0)->
    {CodeH,E1} = match(H,E0),
    {CodeT,E2} = match_list(T,E1),
    {append(CodeH,CodeT),E2};
match_list([],E) ->
    {[],E}.

%%__________________________________________________________________
%% match(Struct,E) -> {Code,Env'}.
%%   match Struct on stack with environment E

match({nil,_},E) ->
    {[getNil],E};
match({atom,_,X},E) ->
    {[{getAtom,X}],E};
match({var,_,'_'},E) ->
    {[pop],E};
match({var,Line,VarName},E) ->
    case tset_var(VarName,E) of
	{set, Where, E1} ->
	    {[{eqVar,{VarName,Where}}],E1};
	{unset, {arg, N}, E1} ->
	    {[{eqVar,{VarName,{arg,N}}}],E1};
	{unset, {var, N}, E1} ->
	    {[{storeVar,{VarName, {var, N}}}],E1}
    end;
match({integer,_,X},E) ->
    {[{getInt,X}],E};
match({float,_,X},E) ->
    {[{getFloat,X}],E};
match({tuple,_,Args},E0) ->
    {Code,E1} = match_list(Args,E0),
    {[{unpkTuple,length(Args)}|Code],E1};
match({cons,_,H,T},E0) ->
    {Hcode,E1} = match(H,E0),
    {Tcode,E2} = match(T,E1),
    {append([unpkList|Hcode],Tcode),E2};
match({string,_,[]},E0) ->
    {[getNil],E0};
match({string,_,X},E0) ->
    {[{getStr,X}], E0}.


%%______________________________________________________________________
%%
%% match_string([H|T]) ->
%%    [unpkList, {getInt, H}| match_string(T)];
%% match_string([]) ->
%%    [getNil].
%%
%%______________________________________________________________________

build_and_call(Args,Call,E0) ->
    {Code,E1} = build_structure(Args,E0),
    %% io:format('Build and call Here:~w\n',[Code]),
    {append(Code,[Call]),E1}.

%%______________________________________________________________________
%% stack_depth(Code) -> {FinalDepth, MaxDepth}

stack_depth(Code) -> stack_depth(Code, 0, 0).

stack_depth([pushNil|T], N, Max)          -> stack_depth1(T, N + 1, Max);
stack_depth([mkList|T], N, Max)           -> stack_depth1(T, N - 1, Max);
stack_depth([{pushAtom,_}|T], N, Max)     -> stack_depth1(T, N + 1, Max);
stack_depth([{pushInt, _}|T], N, Max)     -> stack_depth1(T, N + 1, Max);
stack_depth([{pushFloat, _}|T], N, Max)   -> stack_depth1(T, N + 1, Max);
stack_depth([{pushStr,_}|T], N, Max)      -> stack_depth1(T, N + 1, Max);
stack_depth([{pushVar,_}|T], N, Max)      -> stack_depth1(T, N + 1, Max);
stack_depth([{pushCatch,_}|T], N, Max)    -> stack_depth1(T, N + 1, Max);
stack_depth([dup|T], N, Max)              -> stack_depth1(T, N + 1, Max);
stack_depth([{mkTuple,S}|T], N, Max)      -> stack_depth1(T, N - S + 1, Max);
stack_depth([{arg,_}|T], N, Max)          -> stack_depth1(T, N + 1, Max);
stack_depth([{arith,_}|T], N, Max)        -> stack_depth1(T, N - 1, Max);
stack_depth([send|T], N, Max)             -> stack_depth1(T, N - 1, Max);
stack_depth([{comp, _}|T], N, Max)        -> stack_depth1(T, N - 1, Max);
stack_depth([{call,_,_,_,A}|T], N, Max)   -> stack_depth1(T, N - A + 5, Max);
stack_depth([unpkList|T], N, Max)         -> stack_depth1(T, N + 1, Max);
stack_depth([{unpkTuple, A}|T], N, Max)   -> stack_depth1(T, N + A - 1, Max);
stack_depth([_|T], N, Max)                -> stack_depth1(T, N, Max);
stack_depth([], N, Max)                   -> {N, Max}.

stack_depth1(T, N, Max) when N > Max      -> stack_depth(T, N, N);
stack_depth1(T, N, Max)                   -> stack_depth(T, N, Max).


%%______________________________________________________________________

build_structure([H|T],E0) ->
    {Code1,E1} = compile_expr(H,E0),
    {Code2,E2} = build_structure(T,E1),
    {append(Code1,Code2),E2};
build_structure([],E0) ->
    {[],E0}.

%%______________________________________________________________________
%% build_string(X) -> Code
%%   build a string onto the stack
%%
%% build_string(X) ->
%%    build_string(X, []).
%%
%% build_string([H|T], L) ->
%%    build_string(T, [{pushInt, H}, mkList|L]);
%% build_string([], L) ->
%%    [pushNil|L].
%%

%%______________________________________________________________________
%%  case ...
%%
%%  {try_me_else,L1}
%%      dup
%%      match code (for Head + Guard)
%%      popCommit  % pop and commit
%%      ...
%%      {goto,Exit}
%%  {label,L1}
%%  ----------------
%%  {try_me_else,L2}
%%      dup
%%      match code
%%      popCommit
%%      ...
%%      {goto,Exit}
%%  {label,L2}
%%  -----------------
%%      failCase
%%  {Label,Exit}

%%______________________________________________________________________
%%  if ...
%%
%%  {try_me_else,L1}
%%   	match code (Guard)
%%      commit
%%      ...
%%      {goto,Exit}
%%  {label,L1}
%%  ----------------
%%  {try_me_else,L2}
%%      match code
%%      commit
%%      ...
%%      {goto,Exit}
%%  {label,L2}
%%  -----------------
%%        failIf
%%  {Label,Exit}

%%______________________________________________________________________
%%  receive ... (With No Timeout Code)
%%
%%  {Label,Receive}
%%  wait  -- on return value is on top of stack
%%      
%%  {try_me_else,L1}
%%      dup
%%      match code (for Head + Guard)
%%      popCommitJoin
%%      ...
%%      {goto,Exit}
%%  {label,L1}
%%  ----------------
%%  {try_me_else,L2}
%%      dup
%%      match code (for Head + Guard)
%%      popCommitJoin
%%      ...
%%      {goto,Exit}
%%  {label,L2}
%%  -----------------
%%      save
%%      {goto,Receive}
%%  {Label,Exit}

%%______________________________________________________________________
%%  receive ... (With Timeout Code)
%%
%%  -- timeout value is on top of stack
%%  setTimeOut
%%  {Label,Receive}
%%  wait(TimeOutLabel)  -- on return value is on top of stack
%%      
%%  {try_me_else,L1}
%%      dup
%%      match code (for Head + Guard)
%%      popCommitJoin
%%      ...
%%      {goto,Exit}
%%  {label,L1}
%%  ----------------
%%  {try_me_else,L2}
%%      dup
%%      match code (for Head + Guard)
%%      popCommitJoin
%%      ...
%%      {goto,Exit}
%%  {label,L2}
%%  -----------------
%%      save
%%      {goto,Receive}
%%  {label, TimeOutLabel}
%%      ...
%%      {goto, Exit}
%%  {Label,Exit}

%%______________________________________________________________________
%% compile_case(Case, Env) -> {Code, Env'}
%%   compile a case 

compile_case({'case',_,Func,Options},E0) ->
    {Code1,E1} = compile_expr(Func,E0),
    {Exit,E2} = new_label(E1),
    {Code2,E3} = 
	compile_option_list(
			    Options,
			    {[dup],[popCommit],[{goto,Exit}],
			     [failCase,{label,Exit}]}, E2),
    {append(Code1,Code2),E3}.

%%______________________________________________________________________
%% compile_if(If, Env) -> {Code, Env'}
%%   compile an if onto the stack

compile_if({'if',_,Options},E0) ->
    {Exit,E1} = new_label(E0),
    compile_option_list(Options,
			{[],[commit],[{goto,Exit}],[failIf,{label,Exit}]},
			E1).

compile_receive({'receive',_,Opts,[],[]},E0) ->
    %% receive 
    %% ...
    %% end
    {ExitLabel,E1} = new_label(E0),
    {Receive,E2} = new_label(E1),
    {Code,E3} = 
	compile_option_list(Opts,
			    {[dup],[popCommitJoin],[{goto,ExitLabel}],
			     [save,{goto,Receive},{label,ExitLabel}]},
			    E2),
    {[{label,Receive},wait|Code],E3};
compile_receive({'receive',_,[],Func,TimeOutBody},E0) ->
    %% receive 
    %% after ...
    %% end
    %% No options in the receive
    {Code1,E1} = compile_expr(Func,E0),
    {TimeOutLabel, E2} = new_label(E1),
    {Code3, E3} = compile_seq(TimeOutBody, E2),
    {ReceiveLabel, E4} = new_label(E3),
    Code2 = append(Code1, [setTimeout,{label, ReceiveLabel},
			   {wait, TimeOutLabel},
			   save,
			   {goto, ReceiveLabel},
			   {label,TimeOutLabel} | Code3]),
    {Code2, E4};
compile_receive({'receive',_,Opts,Func,TimeOutBody},E0) ->
    %% receive 
    %%   ....
    %% after ...
    %% end
    {Code1,E1} = compile_expr(Func,E0),
    {TimeOutLabel, E2} = new_label(E1),
    {ExitLabel,E3} = new_label(E2),
    {Receive,E4} = new_label(E3),
    {Code2,E5} = 
	compile_option_list(Opts,
			    {[dup],[popCommitJoin],[{goto,ExitLabel}],
			     [save,{goto,Receive}]},
			    E4),
    %% compile the 'after...' with the same set of bindings as
    %% Before the Recieve (I.e. from E4)  BUT with updated labels etc. from
    %% E5
    E6 = set_alloc(alloc(E4), E5),
    {Code3, E7} = compile_seq(TimeOutBody, E6),
    Code4 = append([
		    Code1,
		    [setTimeout,{label,Receive},{wait, TimeOutLabel} | Code2],
		    [{label,TimeOutLabel} | Code3],
		    [{label,ExitLabel}]]),
    %% Now we have to merge the allocations from E5 and E7
    %% and set these into E7
    E8 = set_alloc(merge_out_allocs([alloc(E5), alloc(E7)]), E7),
    {Code4, E8}.

%%______________________________________________________________________
%% compile_option_list(Opts,Stuff,Env) -> {Code, Env'}
%%   compile an option list of an if, case or receive

compile_option_list(Opts, Extra, E0) ->
    Init_alloc = alloc(E0),
    {Allocs,Code,E1} = compile_option_list(Opts,Extra,Init_alloc,E0),
    %% first we merge all the allocs for the different branches of the
    %% if/case/receive then we set the alloc in the final environment
    %% we need the final alloc to get the labels etc. correct
    E2 = set_alloc(merge_out_allocs(Allocs),E1),
    {Code,E2}.

%% we have to enter EACH branch of the case option list
%% with the SAME set of allocated and saved variables
%% BUT the code array can be allowed to grow
%% on EXIT the last values of the allocation array should
%% be passed into the code generation

compile_option_list([],{_,_,_,End},_,E) ->
    {[],End,E};
compile_option_list([{clause,Line,G,B}|T],Extra,In_alloc,E0) ->
    compile_option_list([{clause,Line,[],G,B}|T],Extra,In_alloc,E0);
compile_option_list([{clause,Line,H,G,B}|T],Extra,In_alloc,E0) ->
    %% dformat("compile_option_list:~w~nAlloc:~w~nENV:~w~n",
    %% 	[{H,G,B},In_alloc,E0],E0),
    E1 = set_alloc(In_alloc,E0),
    {Code1,E2} = compile_hgb({clause,Line,H,G,B},Extra,E1),
    Alloc1 = alloc(E2),
    {Alloc2,Code2,E3} = compile_option_list(T,Extra,In_alloc,E2),
    {[Alloc1|Alloc2],append(Code1,Code2),E3}.

%%______________________________________________________________________
%% compile_hgb({clause,Line,Head,Guard,Body},Extra,E0) -> {Code, Env'}
%%   compile head, Guard + Body in an option list

compile_hgb({clause,_,H,G,B},{Before,Mid,End,_},E0) ->
    {N,E1} = new_label(E0),
    {CodeH,E2} = match_list(H,E1),
    {CodeG,E3} = compile_guard(G,E2),
    {CodeB,E4} = compile_seq(B,E3),
    {append([
	     [{try_me_else,N}|Before],CodeH,CodeG,Mid,CodeB,End,[{label,N}]]),
     E4}.

%%______________________________________________________________________
%% compile_try(Opts,Errs,Env) -> {Code,Env'}

%% example: compile_try([[a,a],[b],[c,c,c]],[er1,er2],Env)
%%
%%  {try_me_else,L1}
%%     a
%%     a
%%     a
%%     {goto,End}
%%  {label,L1}
%%  {try_me_else,L2}
%%     b
%%     b
%%     {goto,End}
%%  {label,L2}
%%  {try_me_else,L3}
%%     c
%%     c
%%     c
%%     {goto,End}
%%  {label,L3}
%%     er1
%%     er2
%%  {label,End}
%%
%%______________________________________________________________________
%% work out allocations after a 'if' 'case' or 'receive'
%%

%%______________________________________________________________________
%%% merge_out_allocs(Alloc*) -> Alloc'
%%   merge allocation lists after 'if' 'case' or 'receive'
%%   each individual allocation is a list

merge_out_allocs([H|T]) ->
    merge_out_allocs(T,H).

merge_out_allocs([], Acc) ->
    Acc;
merge_out_allocs([H|T], Acc) ->
    merge_out_allocs(T, merge_allocs(H, Acc)).

%%______________________________________________________________________
%% merge_allocs(Alloc1, Alloc2) -> Alloc'
%%    the elements in Alloc' are set only if they are set in BOTH
%%    Alloc1 and Alloc2

merge_allocs(X, Y) -> merge_allocs(X, Y, []).

merge_allocs([], _, L) -> 
    L;
merge_allocs([{VarName,{Where,set}}|T], Y, L) ->
    case is_set(VarName, Y) of
	true  -> merge_allocs(T, Y, [{VarName,{Where,set}}|L]);
        false -> merge_allocs(T, Y, [{VarName,{Where,unset}}|L])
    end;
merge_allocs([H|T], Y, L) ->
    merge_allocs(T, Y, [H|L]).

%%______________________________________________________________________
%% is_set(VarName, Alloc) -> (true | false)
%%    test if variable is set in an allocation

is_set(VarName, [{VarName, {_, set}}|_]) -> true;
is_set(VarName, [_|T]) -> is_set(VarName, T);
is_set(_, []) -> false.

%%______________________________________________________________________
%% dformat(Format,Data,Env)  -> true
%%   debug info (printed if debug_flag is true

dformat(Format, Data, E) -> 
	case debug_flag(E) of
	    true  -> io:format(Format, Data);
	    false -> true
	end.

%%______________________________________________________________________
%%  optimise(Code) -> Code'

optimise(Code) -> 
    jam_optimize:optimse(Code).

%%______________________________________________________________________
%% stack_depth_match(Code) ->  MaxDepth

stack_depth_match(Code) -> stack_depth_match(Code, 0).

stack_depth_match([unpkList|T], Max)         -> stack_depth_match(T, Max + 1);
stack_depth_match([{unpkTuple, A}|T], Max)   -> stack_depth_match(T, Max + A - 1);
stack_depth_match([_|T], Max)                -> stack_depth_match(T, Max);
stack_depth_match([], Max)                   -> Max.

format_error(Other) ->
    io_lib:format("** Serious error in jam_compile~n"
		  " Please send a copy of the program which caused~n"
		  " this error to the Erlang help desk~n"
		  " support@erlang.ericsson.se~n", []).

%%______________________________________________________________________
%%  add_stack_need(Seq) -> Seq'
%%    this splits up the instruction sequence into
%%    basic blocks and adds stack need instructions where necessary

add_stack_need([]) ->
    [];
add_stack_need(Code) ->
    {BasicBlock, Rest} = get_basic_block(Code),
    %% Rest is now guaranted to start with {call, or label ...}
    %% or be []
    CodeH = add_stack_need_to_basic_block(BasicBlock),
    case Rest of 
	[] ->
	    CodeH;
	[Instr|Rest1] ->
	    CodeT = add_stack_need(Rest1),
	    append(CodeH, [Instr|CodeT])
    end.

get_basic_block(Code) ->
    get_basic_block(Code, []).

get_basic_block([H|T], L) ->
    case basic_block_end(H) of
	true ->
	    {lists:reverse(L), [H|T]};
	false ->
	    get_basic_block(T, [H|L])
    end;
get_basic_block([], L) ->
    {lists:reverse(L), []}.

basic_block_end({label, _})   -> true;
basic_block_end({call,_,_,_,_}) -> true;
basic_block_end(_)            -> false.

printit([H|T]) -> io:format("~w\n",[H]), printit(T);
printit([]) -> [].

add_stack_need_to_basic_block([]) -> [];
add_stack_need_to_basic_block(Code) ->
    {_, MaxDepth} = stack_depth(Code),
    %% io:format("Max of:~w = ~w\n",[Code, MaxDepth]),
    add_stack_need_instruction(MaxDepth, Code).


%______________________________________________________________________
%% alloc_vars(Clauses) -> Clauses'

alloc_vars([{clause, Line, Head, Guard, Body}|T]) ->
    ArgVars = head_vars(Head),
    AllVars = real_varsin({Head, Guard, Body}),
    [{clause2, Line, Head, Guard, Body,alloc_extra_vars(AllVars, 0, ArgVars)} |
     alloc_vars(T)];
alloc_vars([]) ->
    [].
%______________________________________________________________________
%%% alloc_extra_vars(Vars, N, Defined_vars) -> {Max, Defined_Vars'}

alloc_extra_vars([VarName|T], N, Alloc) ->
    case lookup(VarName, Alloc) of
	{true, _} ->
	    alloc_extra_vars(T, N, Alloc);
	false ->
	    alloc_extra_vars(T, N+1, [{VarName, {var,N}}|Alloc])
    end;
alloc_extra_vars([], N, Alloc) ->
    {N, Alloc}.


%______________________________________________________________________
%%% head_vars(Head) -> ArgVars*
%%    example: head_vars(X, {a,Y}, P) -> [{X, {arg,0}}, {P, {arg, 2}}]
%%  

head_vars(Head) -> head_vars(Head, 0, []).

head_vars([{var,_,'_'}|T], Arg, V) ->
    head_vars(T, Arg+1, V);
head_vars([{var,_,Name}|T], Arg, V) ->
    case lookup(Name, V) of
	{true, _} -> head_vars(T, Arg+1, V);
        false     -> head_vars(T, Arg+1, [{Name,{arg,Arg}}|V])
    end;
head_vars([_|T], Arg, V) ->
    head_vars(T, Arg+1, V);
head_vars([], _, V) ->
    V.

real_varsin(Form) -> real_varsin(Form,[]).

real_varsin(X, L) when constant(X) 	-> L;
real_varsin([], L)  	-> L;
real_varsin({var,_,'_'}, L) -> L;
real_varsin({var,_,N}, L) when atom(N) -> [N|L];
real_varsin(T, L) when tuple(T) 	-> real_varsin(tuple_to_list(T), L);
real_varsin([H|T], L) 		-> real_varsin(T, real_varsin(H, L)).

	
%%______________________________________________________________________
%% lookup(Var, {Var, Val}*) -> {true, Val} | false

lookup(Var,[{Var,Val}|_]) ->
    {true,Val};
lookup(Var,[H|T]) ->
    lookup(Var,T);
lookup(_,_) ->
    false.
