%% ``The contents of this file are subject to the Erlang Public License,
%% Version 1.0, (the "License"); you may not use this file except in
%% compliance with the License. You may obtain a copy of the License at
%% http://www.erlang.org/EPL1_0.txt
%% 
%% Software distributed under the License is distributed on an "AS IS"
%% basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
%% the License for the specific language governing rights and limitations
%% under the License.
%% 
%% The Original Code is Erlang-4.7.3, December, 1998.
%% 
%% The Initial Developer of the Original Code is Ericsson Telecom
%% AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
%% Telecom AB. All Rights Reserved.
%% 
%% Contributor(s): ______________________________________.''
%%
%% Copyright (C) 1991, Ellemtel Telecommunications Systems Laboratories
%%
%% File    : jam_asm.erl
%% Author  : Joe Armstrong
%% Purpose : Assembler
%% Written : 911101

-module(jam_asm).
-copyright('Copyright (c) 1991-97 Ericsson Telecom AB').
-vsn('$Revision: /main/release/free/1').

-export([module/2, format_error/1]).
-export([as_bytes2/1, as_bytes3/1, as_bytes4/1]).

-import(lists, [reverse/1]).

%% asm:module(Code, Opts) ->  {error, What} | {ok, Bin}

module(Code, Opts) ->
    case catch assemble(Code, Opts) of
	{'EXIT', What} ->
	    {error, [{none, asm, {crashed, What}}]};
	Bin ->
	    {ok, Bin}
    end.

assemble(Code, Opts) ->
    Bytes0 = init_file(),
    dump(Code, Opts, [Bytes0]).

dump([{code,Mod,Name,Arity,Type,Code}|T], Opts, L) ->
    %% io:format("asm::code:~w:~w/~w ~w ~w instructions~n",
    %% [Mod,Name,Arity,Type,length(Code)]),
    %% io:format("~p\n",[Code]),
    %% nlprint(Code),
    Bytes = asm_function(Mod, Name, Arity, Type, Code),
    dump(T, Opts, [Bytes|L]);
dump([], Opts, L) ->
    L1 = reverse(L),
    list_to_binary(L1).


nlprint([H|T]) ->
    io:format("~w~n",[H]),
    nlprint(T);
nlprint([]) ->
    ok.

%%______________________________________________________________________
%% asm:function(Mod, Name, Arity, Type, Code) ->
%%
%% assembles Function 

asm_function(Mod, Name, Arity, Type, Code) ->
    Bytes1 = dump_header(Mod,Type,Name,Arity),
    {Length,Code1,Labels,Patches} = asm1(Code,0,[],[],[]),
    %% io:format("~w:~w/~w Length:~w~n",[Mod,Name,Arity,Length]),
    %% format("code~n",[]),nlprint(lists:reverse(Code1)),
    %% format("Labels~n",[]),nlprint(Labels),
    %% format("Patches~n",[]),nlprint(Patches),
    {Obj,Code2} = asm2(Code1,Labels,[],[]),
    Bytes2 = dump_code({Length,Code2,Obj}),
    %% format("object code:~w~n",[Obj]),
    Len = lists:flat_length(Obj),
    %% format("flat_length:~w~n",[Len]),
    %% io:format("Patches:~w~n",[Patches]),
    Patches1 = merge_patches(Patches),
    %% io:format("Merged Patches~n",[]),nlprint(Patches1),
    Bytes3 = dump_patches(Patches1),
    [Bytes1,Bytes2,Bytes3].


%%______________________________________________________________________
%% asm1(SymCode, Start_Address, AsmCode, Labels, Patches) 
%%   -> {Length,AsmCode,Labels,Patches} 
%%   AsmCode is in reverse order !
%%   

asm1([H|T],Addr,Code,Labels,Patches) ->
    asm1(jam_encode:form(H),H,T,Addr,Code,Labels,Patches);
asm1([],Addr,Code,Labels,Patches) ->
    {Addr,Code,Labels,Patches}.

asm1({Bytes,Len,[]},H,T,Addr,Code,Labels,Patches) ->
    asm1(T,Addr+Len,[{H,Bytes}|Code],Labels,Patches);

asm1({Bytes,Len,[{Offset,{label,LL}} | Ls]},H,T,Addr,Code,Labels,Patches) ->
    {OffsetList, LabelList} =
	make_patch_list([{Offset,{label,LL}} | Ls], [], []),
    asm1(T,Addr+Len,[{H,Bytes,Addr,Offset,OffsetList,LabelList}|Code],
	 Labels,Patches);
asm1({Bytes,Len,[{0,{noteLabel,LL}}]},H,T,Addr,Code,Labels,Patches) ->
    asm1(T,Addr+Len,[{H,Bytes}|Code],[{LL,Addr}|Labels],Patches);
asm1({Bytes,Len,P},H,T,Addr,Code,Labels,Patches) ->
    asm1(T,Addr+Len,[{H,Bytes}|Code],Labels,add_patches(P,Addr,Patches));
asm1(Other,H,T,Addr,Code,Labels,Patches) ->
    io:format("*** asm1 Other:~w~n",[Other]),
    asm1(T,Addr,[{H,[]}|Code],Labels,Patches).

%______________________________________________________________________
%% asm2(Code,Labels,Accumulator,RawAcc) -> {Raw,Code'}

asm2([{X,Y}|T],Labels,Code,Raw) ->
    asm2(T,Labels,[{X,Y}|Code],[Y|Raw]);
asm2([{H,Bytes,Addr,Offset,OffsetList,LabelList}|T],Labels,Code,Raw) ->
    Bytes1 = do_patch(Offset,OffsetList,LabelList,Labels,Addr,Bytes),
    asm2(T,Labels,[{H,Bytes1}|Code],[Bytes1|Raw]);
asm2([],_,Code,Raw) -> {Raw,Code}.


make_patch_list([{Offset,{label,LL}} | Ls], OffsetList, LabelList) ->
    make_patch_list(Ls, [Offset | OffsetList], [LL | LabelList]);
make_patch_list([], OffsetList, LabelList) ->
    {OffsetList, LabelList}.

do_patch(Offset,[Offs|OffsList],[Label|LabelList],Labels,Addr,Bytes) ->
    case lookup(Label,Labels) of
	undefined ->
	    io:format("undefined label:~w~n",[Label]),
	    exit(bad_label);
	LabelAddr ->
	    Val = LabelAddr - Addr - 1,
	    %% format("LabelAddr,Addr,Offset,Val=~w ~w ~w ~w~n",
	    %% [LabelAddr,Addr,Offset,Val]),
	    do_patch(Offset, OffsList, LabelList, Labels, Addr, 
		     patch(Offs,Bytes,Val))
    end;
do_patch(_, [], [], _, _, Bytes) -> Bytes.


patch(Offset,Bytes,Val) ->
%%  patch1(Offset,Bytes,[],as_bytes2(Val)).
    patch1(Offset,Bytes,[],as_bytes3(Val)).

%% patch1(0,[X,Y|T],Before,[B1,B2]) ->
%%   lists:reverse(Before,[B1,B2|T]);
patch1(0,[X,Y,Z|T],Before,[B1,B2,B3]) ->
    lists:reverse(Before,[B1,B2,B3|T]);
patch1(N,[H|T],Before,X) ->
    patch1(N-1,T,[H|Before],X).

add_patches([],_,P) -> P;
add_patches([{Offset,Patch}|T],Addr,P) ->
    add_patches(T,Addr,[{Offset+Addr,Patch}|P]).
%______________________________________________________________________
% patch_object_code(Patch) -> ObjCode

patch_object_code({{const,C},Addr}) ->
    [opcode(patch_const),make_string(C),make_patch_list(Addr)];
patch_object_code({{float,N},Addr}) ->
    [opcode(patch_float),make_float(N),make_patch_list(Addr)];
patch_object_code({{local,Func,Arity},Addr}) ->
    [opcode(patch_local),make_string(Func),[Arity],make_patch_list(Addr)];
patch_object_code({{csa,Mod,Func,Arity},Addr}) ->
    [opcode(csa),make_string(Mod),make_string(Func),[Arity],
      make_patch_list(Addr)];
patch_object_code(Other) ->
    io:format("cant patch_object_code for:~w~n",[Other]),
    [].
%%______________________________________________________________________

dump_patches(P) -> dump_patches(P, []).

dump_patches([H|T], L) ->
    Bytes = patch_object_code(H),
    dump_patches(T, [Bytes|L]);
dump_patches([], L) ->
    reverse(L).
	
%%______________________________________________________________________

merge_patches(X) -> 
    merge_patches(X,[]).

merge_patches([{Addr,Type}|T],L) ->
    case lookup(Type,L) of
	undefined -> 
	    merge_patches(T,[{Type,[Addr]}|L]);
	Locs ->
	    L1 = delete(Type,L),
	    merge_patches(T,[{Type,[Addr|Locs]}|L1])
    end;
merge_patches([H|T],L) ->
    io:format("skipping ...~w~n",[H]),
    merge_patches(T,L);
merge_patches([],L) -> L.

%%______________________________________________________________________

init_file() ->
    [opcode(magic),make_string(jam_encode:magic())].

%%___________________________________________________________________

dump_header(Mod, Type, Name, Arity) ->
    [opcode(define_function), make_string(Mod), make_export(Type),
     make_string(Name), Arity].
	
%%______________________________________________________________________

dump_code({Len,X,Obj}) when Len >= 0, Len =< 16#ffff->
    %% io:format("dump_code: ~p\n",[{Len,Obj}]),
    [opcode(code),as_bytes2(Len)|Obj];
dump_code({Len,X,Obj}) when Len >= 0, Len =< 16#ffffff->
    %% io:format("dump_code: ~p\n",[{Len,Obj}]),
    [opcode(fat_code),as_bytes3(Len)|Obj].

%______________________________________________________________________

make_export(exported) -> opcode(exported);
make_export(local)    -> opcode(local).

make_float(F) ->
    L = float_to_list(F),
    [opcode(string),as_bytes2(length(L)),L].

make_string(C) ->
    L = atom_to_list(C),
    [opcode(string),as_bytes2(length(L)),L].

make_patch_list([]) -> [];
make_patch_list([H|T]) ->
    [opcode(patch_at_address),as_bytes3(H)|make_patch_list(T)]. 

%______________________________________________________________________

opcode(define_function) -> 1;
opcode(string) -> 2;
opcode(exported) -> 3;
opcode(local) -> 4;
opcode(patch_const) -> 5;
opcode(patch_local) -> 6;
opcode(csa) -> 7;
%% opcode(patch_at_address) -> 8;
opcode(patch_at_address) -> 14;
opcode(code)-> 9;
opcode(copyright) -> 10;
opcode(magic) -> 11;
opcode(patch_float) -> 12;
opcode(fat_code)-> 15.

format_error({crashed, Why}) ->
    io_lib:format("asm: EXIT :~p", [Why]).

%______________________________________________________________________
% lookup

lookup(Key,[{Key,Value}|_]) ->
    Value;
lookup(Key,[_|T]) ->
    lookup(Key,T);
lookup(_,[]) ->
    undefined.

delete(Key,L) ->
    delete(Key,L,[]).

delete(Key,[],Before) ->
    Before;
delete(Key,[{Key,_}|T],Before) ->
    lists:reverse(Before,T);
delete(Key,[H|T],Before) ->
    delete(Key,T,[H|Before]).

%% These as-bytes routines don't check their arguments

as_bytes4(I) ->
    B0 = I band 255,
    B1 = (I bsr 8) band 255,
    B2 = (I bsr 16) band 255,
    B3 = (I bsr 24) band 255,
    [B3,B2,B1,B0].

as_bytes3(I) ->
    B0 = I band 255,
    B1 = (I bsr 8) band 255,
    B2 = (I bsr 16) band 255,
    [B2,B1,B0].

as_bytes2(I) ->
    B0 = I band 255,
    B1 = (I bsr 8) band 255,
    [B1,B0].
