%% ``The contents of this file are subject to the Erlang Public License,
%% Version 1.0, (the "License"); you may not use this file except in
%% compliance with the License. You may obtain a copy of the License at
%% http://www.erlang.org/EPL1_0.txt
%% 
%% Software distributed under the License is distributed on an "AS IS"
%% basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
%% the License for the specific language governing rights and limitations
%% under the License.
%% 
%% The Original Code is Erlang-4.7.3, December, 1998.
%% 
%% The Initial Developer of the Original Code is Ericsson Telecom
%% AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
%% Telecom AB. All Rights Reserved.
%% 
%% Contributor(s): ______________________________________.''
%%
%% Copyright (C) 1995, Ellemtel Telecommunications Systems Laboratories
%% File    : compile.erl
%% Author  : Robert Virding
%% Purpose : Run the Erlang compiler.
%% Revision: $Id: compile.erl,v 4.3.1.4 1996/04/11 13:11:21 rv Exp $

-module(compile).
-copyright('Copyright (c) 1991-98 Ericsson Telecom AB').
-vsn('$Revision: /main/release/free/1').
-include("erl_compile.hrl").

%% High-level interface.
-export([file/1,file/2,format_error/1,iofile/1,compile/3]).

%% Internal functions.
-export([internal/4]).

-import(lists, [member/2,reverse/1,keysearch/3]).

%% file(FileName)
%% file(FileName, Options)
%%  Compile the module in file FileName.

file(File) ->
    file(File, [verbose,report_errors,report_warnings]).

file(File,Opts) when atom(Opts) ->
    file(File,[Opts,verbose,report_errors,report_warnings]);
file(File, Opts0) when list(Opts0) ->
    case find_target(Opts0) of
	{Target, Opts1} ->
	    Opts2 = lists:foldr(fun expand_opt/2, [], Opts1),
	    Serv = spawn_link(?MODULE, internal,
			      [self(), Target, 
			       filename:rootname(File, ".erl"), Opts2]),
	    receive
		{Serv, Rep} -> Rep
	    end;
	{error, Reason} ->
	    {error, Reason}
    end.

expand_opt(report, Os) -> [report_errors,report_warnings|Os];
expand_opt(return, Os) -> [return_errors,return_warnings|Os];
expand_opt(O, Os) -> [O|Os].

find_target(Opts1) ->
    J = member(jam, Opts1),
    B = member(beam, Opts1),
    case {member(jam, Opts1), member(beam, Opts1)} of
	{true, false} -> {jam, Opts1};
	{false, true} -> {beam, Opts1};
	{true, true} -> {error, multiple_targets};
	{false, false} ->
	    case erlang:info(machine) of
		"JAM"  -> {jam, [jam|Opts1]};
		"BEAM" -> {beam, [beam|Opts1]}
	    end
    end.

%% format_error(ErrorDescriptor) -> string()

format_error(multiple_targets) ->
    "both beam and jam specified";
format_error({open,E}) ->
    io_lib:format("open error '~s'", [file:format_error(E)]);
format_error(write_error) ->
    "error writing file";
format_error({rename,S}) ->
    io_lib:format("error renaming ~s", [S]);
format_error({parse_transform,M}) ->
    io_lib:format("error in transform '~s'", [M]);
format_error({c_compilation, M}) ->
    io_lib:format("error in C compilation:~n~s", [M]);
format_error({linking, M}) ->
    io_lib:format("error in linking:~n~s", [M]).

%% The compile state record.
-record(compile, {filename="",
		  dir="",
		  base="",
		  ifile="",
		  ofile="",
		  module=[],
		  code=[],
		  options=[],
		  errors=[],
		  warnings=[],
		  compiler,			% jam_compile or beam_compile
		  assembler			% jam_asm or beam_asm_int
		 }).

%% internal(Master, Target, FileName, [Option]) ->
%%	<>

internal(Master, Target, File, Opts) ->
    Master ! {self(),
	      case catch internal(Target, File, Opts) of
		  {'EXIT', Reason} ->
		      {error, Reason};
		  Other ->
		      Other
	      end}.


internal(Target, File, Opts) ->
    Dir = filename:dirname(File),
    Base = filename:basename(File, ".erl"),
    {Compiler, Assembler} =
	case Target of
	    jam -> {jam_compile, jam_asm};
	    beam -> {beam_compile, beam_asm_int}
	end,
    St0 = #compile{filename=File, dir=Dir, base=Base,
		   ifile=erlfile(Dir, Base),
		   ofile=objfile(Base, Target, Opts),
		   options=Opts,
		   compiler=Compiler,
		   assembler=Assembler},
    Passes = [fun remove_file/2, fun parse_module/2, fun transform_module/2,
	      fun lint_module/2, fun expand_module/2,
	      fun compile_module/2, fun asm_module/2],
    case fold_comp(Target, St0, Passes) of
	{ok, St1} ->
	    case member(binary, St1#compile.options) of
		false ->
		    save_binary(St1);
		true ->
		    comp_ret_bin(St1)
	    end;
	{break,St1} ->
	    comp_ret_ok(St1);
	{error,St1} ->
	    comp_ret_err(St1)
    end.

fold_comp(Target, St0, [P|Ps]) ->
    case P(Target, St0) of
	{ok,St1} -> fold_comp(Target, St1, Ps);
	{break,St1} -> {break,St1};
	{error,St1} -> {error,St1}
    end;
fold_comp(_Target, St, []) -> {ok,St}.

find_option([H | T], St) ->
    case member(H, St#compile.options) of
	true -> true;
	false -> find_option(T, St)
    end;
find_option([], _) -> 
    false.

%
%  Remove the target file so we don't have an old one if the compilation fail
%  UNLESS we generate anything else than a beam/jam file
%
remove_file(_, St) ->
    case find_option(['P', 'E', 'S', binary], St) of
	true ->
	    {ok, St};
	false ->
	    file:delete(St#compile.ofile),
	    {ok, St}
    end.

parse_module(_Target, St) ->
    Opts = St#compile.options,
    Cwd = case keysearch(cwd, 1, Opts) of
	      {value, {cwd, Dir}} -> Dir;
	      _ -> "."
	  end,
    IncludePath = [Cwd, St#compile.dir|inc_paths(Opts)],
    case epp:parse_file(St#compile.ifile, IncludePath, pre_defs(Opts)) of
	{ok,Forms} ->
	    {ok,St#compile{code=Forms}};
	{error,E} ->
	    Es = [{St#compile.ifile,[{none,compile,{open,E}}]}],
	    {error,St#compile{errors=St#compile.errors ++ Es}}
    end.

compile_options([{attribute,L,compile,C}|Fs]) when list(C) ->
    C ++ compile_options(Fs);
compile_options([{attribute,L,compile,C}|Fs]) ->
    [C|compile_options(Fs)];
compile_options([_F|Fs]) -> compile_options(Fs);
compile_options([]) -> [].

transforms(Os) -> [ M || {parse_transform,M} <- Os ]. 

transform_module(_Target, St) ->
    %% Extract compile options from code into options field.
    Ts = transforms(St#compile.options ++ compile_options(St#compile.code)),
    foldl_transform(St, Ts).

foldl_transform(St, [T|Ts]) ->
    case catch apply(T, parse_transform, [St#compile.code, St#compile.options]) of
	Forms ->
	    foldl_transform(St#compile{code=Forms}, Ts);
	{'EXIT',R} ->
	    Es = [{St#compile.ifile,[{none,compile,{parse_transform,T}}]}],
	    {error,St#compile{errors=St#compile.errors ++ Es}}
    end;
foldl_transform(St, []) ->
    %% Do listing of "parsed code".
   listing_ret(src_listing(St#compile.code, St, 'P')).

lint_module(_Target, St) ->
    case erl_lint:module(St#compile.code,
			 St#compile.ifile, St#compile.options) of
	{ok,Ws} ->
	    {ok,St#compile{warnings=St#compile.warnings ++ Ws}};
	{error,Es,Ws} ->
	    {error,St#compile{warnings=St#compile.warnings ++ Ws,
			      errors=St#compile.errors ++ Es}}
    end.

%% expand_module(State) -> State'
%%  Do the common preprocessing of the input forms.

expand_module(_Target, St0) ->
    {Mod,Exp,Forms,Opts} = sys_pre_expand:module(St0#compile.code,
						 St0#compile.options),
    %% This is a temporary hack to help the JAM!
    St1 = case member(pj, Opts) of
	      true ->
		  St0#compile{module=Mod,
			      options=Opts,
			      code={Mod,Exp,sys_pre_pj:module(Forms, Opts)}};
	      false ->
		  St0#compile{module=Mod,options=Opts,code={Mod,Exp,Forms}}
	  end,
    listing_ret(src_listing(element(3, St1#compile.code), St1, 'E')).

compile_module(Target, St) ->
    {Mod,Exp,Forms} = St#compile.code,
    Compiler = St#compile.compiler,
    case Compiler:module(Mod, Exp, Forms, St#compile.options) of
	{ok,ACode} ->
	    listing_ret(asm_listing(Target,ACode,St#compile{code=ACode},'S'));
	{error,Es} ->
	    {error,St#compile{errors=St#compile.errors ++
			      [{St#compile.ifile,Es}]}}
    end.

asm_module(_, St) ->
    Assembler = St#compile.assembler,
    case Assembler:module(St#compile.code, St#compile.options) of
	{ok,Bin} ->
	    {ok,St#compile{code=Bin}};
	{error,Es} ->
	    {error,St#compile{errors=St#compile.errors ++
			      [{St#compile.ifile,Es}]}}
    end.

save_binary(St) ->
    Tfile = tmpfile(St#compile.ofile),		%Temp working file
    case write_binary(Tfile, St#compile.code, St) of
	ok ->
	    case file:rename(Tfile, St#compile.ofile) of
		ok ->
		    comp_ret_ok(St);
		{error,E} ->
		    file:delete(Tfile),
		    Es = [{St#compile.ofile,[{none,?MODULE,{rename,Tfile}}]}],
		    comp_ret_err(St#compile{errors=St#compile.errors ++ Es})
	    end;
	{error,E} ->
	    Es = [{Tfile,[{compile,write_error}]}],
	    comp_ret_err(St#compile{errors=St#compile.errors ++ Es})
    end.

write_binary(Name, Bin, St) ->
    Opts = case member(compressed, St#compile.options) of
	       true -> [compressed];
	       false -> []
	   end,
    case file:open(Name, [write, raw|Opts]) of
	{ok, Fd} ->
	    Res = case file:write(Fd, Bin) of
		      ok ->
			  ok;
		      {error, Reason} ->
			  {error, Reason}
		  end,
	    file:close(Fd),
	    Res;
	{error, Reason} ->
	    {error, Reason}
    end.

listing_ret({no,St}) -> {ok,St};
listing_ret({yes,St}) -> {break,St};
listing_ret({error,St}) -> {error,St}.

%% comp_ret_ok(ModuleName, State) -> OkReturn
%% comp_ret_bin(ModuleName, Binary, State) -> OkBinReturn
%% comp_ret_err(State) -> ErrorReturn

comp_ret_ok(St) ->
    report_warnings(St),
    case member(return_warnings, St#compile.options) of
	true -> {ok,St#compile.module,St#compile.warnings};
	false -> {ok,St#compile.module}
    end.

comp_ret_bin(St) ->
    report_warnings(St),
    case member(return_warnings, St#compile.options) of
	true -> {ok,St#compile.module,St#compile.code,St#compile.warnings};
	false -> {ok,St#compile.module,St#compile.code}
    end.

comp_ret_err(St) ->
    report_errors(St),
    report_warnings(St),
    case member(return_errors, St#compile.options) of
	true -> {error,St#compile.errors,St#compile.warnings};
	false -> error
    end.

%% report_errors(State) -> ok
%% report_warnings(State) -> ok

report_errors(St) ->
    case member(report_errors, St#compile.options) of
	true ->
	    lists:foreach(fun ({{F,L},Eds}) -> list_errors(F, Eds);
			      ({F,Eds}) -> list_errors(F, Eds) end,
			  St#compile.errors);
	false -> ok
    end.

report_warnings(St) ->
    case member(report_warnings, St#compile.options) of
	true ->
	    lists:foreach(fun ({{F,L},Eds}) -> list_warnings(F, Eds);
			      ({F,Eds}) -> list_warnings(F, Eds) end,
			  St#compile.warnings);
	false -> ok
    end.

%% list_errors(File, ErrorDescriptors) -> ok

list_errors(F, [{Line,Mod,E}|Es]) ->
    io:fwrite("~s:~w: ~s\n", [F,Line,apply(Mod, format_error, [E])]),
    list_errors(F, Es);
list_errors(F, [{Mod,E}|Es]) ->
    io:fwrite("~s: ~s\n", [F,apply(Mod, format_error, [E])]),
    list_errors(F, Es);
list_errors(F, []) ->
    ok.

%% list_warnings(File, ErrorDescriptors) -> ok

list_warnings(F, [{Line,Mod,E}|Es]) ->
    io:fwrite("~s:~w: Warning: ~s\n", [F,Line,apply(Mod, format_error, [E])]),
    list_warnings(F, Es);
list_warnings(F, [{Mod,E}|Es]) ->
    io:fwrite("~s: Warning: ~s\n", [F,apply(Mod, format_error, [E])]),
    list_warnings(F, Es);
list_warnings(F, []) ->
    ok.

%% verbose_format(State, Format, Args) -> ok.

%verbose_format(St, Format, Args) ->
%    case member(verbose, St#compile.options) of
%	true -> io:format(Format, Args);
%	false -> ok
%    end.

%% erlfile(Dir, Base) -> ErlFile
%% outfile(Base, Extension, Options) -> OutputFile
%% objfile(Base, Target, Options) -> ObjFile
%% tmpfile(ObjFile) -> TmpFile
%%  Work out the correct input and output file names.

iofile(File) when atom(File) ->
    iofile(atom_to_list(File));
iofile(File) ->
    {filename:dirname(File), filename:basename(File, ".erl")}.

erlfile(Dir, Base) ->
    filename:join(Dir, Base++".erl").

outfile(Base, Ext, Opts) when atom(Ext) ->
    outfile(Base, atom_to_list(Ext), Opts);
outfile(Base, Ext, Opts) ->
    Obase = case keysearch(outdir, 1, Opts) of
		{value, {outdir, Odir}} -> filename:join(Odir, Base);
		Other -> Base			% Not found or bad format
	    end,
    Obase++"."++Ext.

objfile(Base, Target, Opts) ->
    outfile(Base, Target, Opts).

tmpfile(Ofile) ->
    reverse([$#|tl(reverse(Ofile))]).

%% pre_defs(Options)
%% inc_paths(Options)
%%  Extract the predefined macros and include paths from the option list.

pre_defs([{d,M,V}|Opts]) ->
    [{M,V}|pre_defs(Opts)];
pre_defs([{d,M}|Opts]) ->
    [M|pre_defs(Opts)];
pre_defs([O|Opts]) ->
    pre_defs(Opts);
pre_defs([]) -> [].

inc_paths(Opts) ->
    [ P || {i,P} <- Opts, list(P) ].

%% src_listing([Form], State, Flag) ->
%%	{yes,State} | {no,State} | {error,State}
%% asm_listing(AsmCode, State, Flag) ->
%%	{yes,State} | {no,State} | {error,State}
%%
%%  The specific listing functions. They return 'yes' or 'no' to indicate
%%  listing whether has been done and {error,State} if there was an error.

src_listing(Forms, St, Flag) ->
    listing(Forms, St, Flag,
	    fun (Lf, Fs) ->
		lists:foreach(fun (F) ->
				  io:put_chars(Lf, [erl_pp:form(F),"\n"]) end,
			      Fs) end).

asm_listing(jam, Asm, St, Flag) ->
    listing(Asm, St, Flag, fun (Lf, As) -> jam_listing:module(Lf, As) end);
asm_listing(beam, Asm, St, Flag) ->
    listing(Asm, St, Flag, fun (Lf, As) -> beam_listing:module(Lf, As) end).

%% listing(Code, State, Flag, ListFun) ->
%%	{yes,State} | {no,State} | {error,State}

listing(Code, St, Flag, ListF) ->
    case member(Flag, St#compile.options) of
	true ->
	    Lfile = outfile(St#compile.base, Flag, St#compile.options),
	    case file:open(Lfile, write) of
		{ok,Lf} -> 
		    ListF(Lf, Code),
		    file:close(Lf),
		    {yes,St};
		{error,E} ->
		    Es = [{Lfile,[{none,compile,write_error}]}],
		    {error,St#compile{errors=St#compile.errors ++ Es}}
	    end;
	false -> {no,St}
    end.


%% compile(AbsFileName, Outfilename, Options)
%%   Compile entry point for erl_compile.

compile(File, _OutFile, Options) ->
    case file(File, make_erl_options(Options)) of
	{ok, _Mod} -> ok;
	Other -> Other
    end.

%% Converts generic compiler options to specific options.

make_erl_options(Opts) ->

    %% This way of extracting will work even if the record passed
    %% has more fields than known during compilation.

    Includes0 = Opts#options.includes,
    Defines = Opts#options.defines,
    Outdir = Opts#options.outdir,
    Warning = Opts#options.warning,
    Verbose = Opts#options.verbose,
    Specific = Opts#options.specific,
    Optimize = Opts#options.optimize,
    OutputType = Opts#options.output_type,
    Cwd = Opts#options.cwd,

    Includes = 
	case Opts#options.ilroot of
	    undefined ->
		Includes0;
	    Ilroot ->
		[Ilroot|Includes0]
	end,

    Options =
	case Verbose of
	    true ->  [verbose];
	    false -> []
	end ++
	case Warning of
	    0 -> [];
	    _ -> [report_warnings]
	end ++
	case Optimize of
	    0 -> [];
	    _ -> [fast]
	end ++
	lists:map(
	      fun ({Name, Value}) ->
		      {d, Name, Value};
		  (Name) ->
		      {d, Name}
	      end,
	      Defines) ++
	case OutputType of
	    undefined -> [];
	    jam -> [jam];
	    beam -> [beam]
	end,

    Options++[report_errors, {cwd, Cwd}, {outdir, Outdir}|
	 lists:map(fun(Dir) -> {i, Dir} end, Includes)]++Specific.
