%% ``The contents of this file are subject to the Erlang Public License,
%% Version 1.0, (the "License"); you may not use this file except in
%% compliance with the License. You may obtain a copy of the License at
%% http://www.erlang.org/EPL1_0.txt
%% 
%% Software distributed under the License is distributed on an "AS IS"
%% basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
%% the License for the specific language governing rights and limitations
%% under the License.
%% 
%% The Original Code is Erlang-4.7.3, December, 1998.
%% 
%% The Initial Developer of the Original Code is Ericsson Telecom
%% AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
%% Telecom AB. All Rights Reserved.
%% 
%% Contributor(s): ______________________________________.''
%%
%%% File    : beam_asm_dict.erl
%%% Author  : Bjorn Gustavsson <bjorn@strider>
%%% Purpose : Maintain atom, import, and export tables for assembler.
%%% Created : 28 Feb 1998 by Bjorn Gustavsson <bjorn@strider>

-module(beam_dict).
-author('bjorn@strider').

-export([new/0, atom/2, export/4, import/4, string/2,
	 atom_table/1, export_table/1, import_table/1, string_table/1]).

-record(asm_dict,
	{atoms = [],				% [{Index, Atom}, ...]
	 exports = [],				% [{F, A, Label}, ...]
	 imports = [],				% [{Index, {M, F, A}, ...]
	 strings = [],				% Deep list of characters
	 next_atom = 0,
	 next_import = 0,
	 string_offset = 0
	}).

new() ->
    #asm_dict{}.

%% Returns the index for an atom (adding it to the atom table if necessary).
%%    atom(Atom, Dict) -> {Index, Dict'}

atom(Atom, Dict) when atom(Atom) ->
    NextIndex = Dict#asm_dict.next_atom,
    case lookup_store(Atom, Dict#asm_dict.atoms, NextIndex) of
	{Index, _, NextIndex} ->
	    {Index, Dict};
	{Index, Atoms, NewIndex} ->
	    {Index, Dict#asm_dict{atoms=Atoms, next_atom=NewIndex}}
    end.

%% Remembers an exported function.
%%    export(Func, Arity, Label, Dict) -> Dict'

export(Func, Arity, Label, Dict0) when atom(Func), integer(Arity), integer(Label) ->
    {Index, Dict1} = atom(Func, Dict0),
    Dict1#asm_dict{exports = [{Index, Arity, Label}| Dict1#asm_dict.exports]}.

%% Returns the index for an import entry (adding it to the import table if necessary).
%%    import(Mod, Func, Arity, Dict) -> {Index, Dict'}

import(Mod, Func, Arity, Dict) when atom(Mod), atom(Func), integer(Arity) ->
    NextIndex = Dict#asm_dict.next_import,
    case lookup_store({Mod, Func, Arity}, Dict#asm_dict.imports, NextIndex) of
	{Index, _, NextIndex} ->
	    {Index, Dict};
	{Index, Imports, NewIndex} ->
	    {_, D1} = atom(Mod, Dict#asm_dict{imports=Imports, next_import=NewIndex}),
	    {_, D2} = atom(Func, D1),
	    {Index, D2}
    end.

%% Returns the index for a string in the string table (adding the string to the
%% table if necessary).
%%    string(String, Dict) -> {Offset, Dict'}

string(Str, Dict) when list(Str) ->
    #asm_dict{strings = Strings, string_offset = NextOffset} = Dict,
    case old_string(Str, Strings) of
	{true, Offset} ->
	    {Offset, Dict};
	false ->
	    NewDict = Dict#asm_dict{strings = Strings++Str,
				    string_offset = NextOffset+length(Str)},
	    {NextOffset, NewDict}
    end.

atom_table(#asm_dict{atoms=Atoms, next_atom=NumAtoms}) ->
    Sorted = lists:sort(Atoms),
    Fun = fun({_, A}) ->
		  L = atom_to_list(A),
		  [length(L)|L]
	  end,
    {NumAtoms, lists:map(Fun, Sorted)}.

%% Returns the export table.
%%    export_table(Dict) -> {NumExports, [{Function, Arity, Label}...]}

export_table(#asm_dict{exports = Exports}) ->
    {length(Exports), Exports}.

import_table(Dict) ->
    #asm_dict{imports = Imports, next_import = NumImports} = Dict,
    Sorted = lists:sort(Imports),
    Fun = fun({_, {Mod, Func, Arity}}) ->
		  {Atom0, _} = atom(Mod, Dict),
		  {Atom1, _} = atom(Func, Dict),
		  {Atom0, Atom1, Arity}
	  end,
    {NumImports, lists:map(Fun, Sorted)}.

string_table(#asm_dict{strings = Strings, string_offset = Size}) ->
    {Size, Strings}.

%%% Local helper functions.

lookup_store(Key, Dict, NextIndex) ->
    case catch lookup_store1(Key, Dict, NextIndex) of
	Index when integer(Index) ->
	    {Index, Dict, NextIndex};
	{Index, NewDict} ->
	    {Index, NewDict, NextIndex+1}
    end.

lookup_store1(Key, [Pair|Dict], NextIndex) when Key > element(2, Pair) ->
    {Index, NewDict} = lookup_store1(Key, Dict, NextIndex),
    {Index, [Pair|NewDict]};
lookup_store1(Key, [{Index, Key}|Dict], NextIndex) ->
    throw(Index);
lookup_store1(Key, Dict, NextIndex) ->
    {NextIndex, [{NextIndex, Key}|Dict]}.

%% Search for string Str in the string pool Pool.
%%   old_string(Str, Pool) -> false | {true, Offset}

old_string(Str, Pool) ->
    old_string(Str, Pool, 0).

old_string([C|Str], [C|Pool], Index) ->
    case lists:prefix(Str, Pool) of
	true ->
	    {true, Index};
	false ->
	    old_string([C|Str], Pool, Index+1)
    end;
old_string(Str, [_|Pool], Index) ->
    old_string(Str, Pool, Index+1);
old_string(Str, [], Index) ->
    false.

    
