%% ``The contents of this file are subject to the Erlang Public License,
%% Version 1.0, (the "License"); you may not use this file except in
%% compliance with the License. You may obtain a copy of the License at
%% http://www.erlang.org/EPL1_0.txt
%% 
%% Software distributed under the License is distributed on an "AS IS"
%% basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
%% the License for the specific language governing rights and limitations
%% under the License.
%% 
%% The Original Code is Erlang-4.7.3, December, 1998.
%% 
%% The Initial Developer of the Original Code is Ericsson Telecom
%% AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
%% Telecom AB. All Rights Reserved.
%% 
%% Contributor(s): ______________________________________.''
%%
%% File    : beam_asm_int.erl
%% Author  : Bjorn Gustavsson
%% Purpose : Assembler for threaded Beam.
%% Written : 930401

-module(beam_asm_int).
-copyright('Copyright (c) 1991-97 Ericsson Telecom AB').
-vsn('$Revision: /main/release/free/1').

-define(beam_magic, "\177BEAM!").

-export([module/2, format_error/1]).
-export([objfile_extension/0]).
-import(lists, [flatmap/2]).

-export([encode/2]).

-include("beam_opcodes.hrl").

%% Format of header for threaded BEAM files:
%%
%% 0	?beam_magic 		6 bytes
%% 6	internal_version	2 bytes
%% 8	flags			4 bytes
%% 12	offset of attributes	4 bytes
%% 16	start of code		4 bytes
%% 20   number of functions     2 bytes
%% 22	number of labels	4 bytes
%% 26   number of atoms		2 bytes
%% 28   number of exports       2 bytes
%% 30   number of imports       2 bytes
%% 32   size of string table    4 bytes
%% 36   offset of atom table	4 bytes
%% 40   offset of export table  4 bytes
%% 44   offset of import table  4 bytes
%% 48   offset of string table  4 bytes

-define(header_size, 52).

module(Code, Opts) ->
    case catch assemble(Code, Opts) of
	{'EXIT', What} ->
	    {error, [{none, ?MODULE, {crashed, What}}]};
	{error, Error} ->
	    {error, [{none, ?MODULE, Error}]};
	Bin when binary(Bin) ->
	    {ok, Bin}
    end.

objfile_extension() ->
    ".beam".

format_error({too_big, Number, Bits}) ->
    io_lib:format("[Internal error] Number '~p' too big to represent in ~p bits",
		  [Number, Bits]);
format_error({crashed, Why}) ->
    io_lib:format("beam_asm_int: EXIT: ~p", [Why]).

assemble({Mod, Exp, Asm, NumLabels}, Opts) ->
    {0, Dict0} = beam_dict:atom(Mod, beam_dict:new()),
    {Code, Dict1} = assemble(Asm, Exp, Dict0, []),
    build_file(Code, Dict1, NumLabels, length(Asm), Opts).

assemble([{function, Name, Arity, Entry, Asm}| T], Exp, Dict0, Acc) ->
    Dict1 = case lists:member({Name, Arity}, Exp) of
		true ->
		    beam_dict:export(Name, Arity, Entry, Dict0);
		false ->
		    Dict0
	    end,
    {Code, Dict2} = assemble_function(Asm, Acc, Dict1),
    assemble(T, Exp, Dict2, Code);
assemble([], Exp, Dict0, Acc) ->
    {IntCodeEnd, Dict1} = make_op(int_code_end, Dict0),
    {list_to_binary(lists:reverse(Acc, [IntCodeEnd])), Dict1}.

assemble_function([H|T], Acc, Dict0) ->
    {Code, Dict} = make_op(H, Dict0),
    assemble_function(T, [Code| Acc], Dict);
assemble_function([], Code, Dict) ->
    {Code, Dict}.

build_file(Code, Dict, NumLabels, NumFuncs, Opts) ->
    {NumAtoms, AtomTab0} = beam_dict:atom_table(Dict),
    AtomTab = list_to_binary(AtomTab0),
    {NumExps, ExpTab0} = beam_dict:export_table(Dict),
    ExpTab = flatten_exports(ExpTab0),
    {StringSize, StringTab0} = beam_dict:string_table(Dict),
    StringTab = list_to_binary(StringTab0),
    {NumImps, ImpTab0} = beam_dict:import_table(Dict),
    ImpTab = flatten_imports(ImpTab0),
    Attributes = build_attributes(Opts),

    %% Calculate offsets.
    ExpTabOffset = ?header_size,
    ImpTabOffset = ExpTabOffset + size(ExpTab),
    CodeOffset = ImpTabOffset + size(ImpTab),
    AtomTabOffset = CodeOffset + size(Code),
    StringTabOffset = AtomTabOffset + size(AtomTab),
    AttributeOffset = StringTabOffset + size(StringTab),

    %% Build the header and the file.
    File = [?beam_magic,
	    opcode_int16(beam_opcodes:format_number()),
	    opcode_int32(build_flags(Opts, 0)),	% Flags (e.g. trace).
	    opcode_int32(AttributeOffset),	% Attributes
	    opcode_int32(CodeOffset),		% Start of code

	    %% Numbers and sizes
	    opcode_int16(NumFuncs),
	    opcode_int32(NumLabels),
	    opcode_int16(NumAtoms),
	    opcode_int16(NumExps),
	    opcode_int16(NumImps),
	    opcode_int32(StringSize),

	    %% Offsets
	    opcode_int32(AtomTabOffset),
	    opcode_int32(ExpTabOffset),
	    opcode_int32(ImpTabOffset),
	    opcode_int32(StringTabOffset),

	    %% Tables and code
	    ExpTab, ImpTab, Code, AtomTab, StringTab, Attributes],
    list_to_binary(File).

build_flags([trace| T], Flags) ->
    build_flags(T, Flags bor 1);
build_flags([_|T], Flags) ->
    build_flags(T, Flags);
build_flags([], Flags) ->
    Flags.

flatten_exports(Exps) ->
    F = fun({F, A, L}) -> [opcode_int16(F), opcode_int16(A), opcode_int32(L)] end,
    list_to_binary(lists:map(F, Exps)).

flatten_imports(Imps) ->
    F = fun({M, F, A}) -> [opcode_int16(M), opcode_int16(F), opcode_int16(A)] end,
    list_to_binary(lists:map(F, Imps)).

build_attributes(Opts) ->
    case lists:keysearch('_atom_attributes_', 1, Opts) of
	{value, {_, Attr}} ->
	    L = [[atom_to_list(N), $=, atom_to_list(V), 0] || {N, V} <- Attr],
	    list_to_binary([L, 0]);
	false ->
	    [0]
    end.

opcode_int32(I) when I > 16#ffffffff ->
    throw({error, {too_big, I, 32}});
opcode_int32(I) ->
    [(I bsr 24) band 16#ff,
     (I bsr 16) band 16#ff,
     (I bsr 8) band 16#ff,
     I band 16#ff].

opcode_int16(I) when I > 16#ffff ->
    throw({error, {too_big, I, 16}});
opcode_int16(I) ->
    [(I bsr 8) band 16#ff, I band 16#ff].

make_op(Op, Dict) when atom(Op) ->
    encode_op(Op, [], Dict);
make_op({arith, {arithBnot, Src, Dest}, W}, Dict) ->
    encode_op(int_bnot, [W, Src, Dest], Dict);
make_op({arith, {Arith, Op, R0, R1, R2}, W}, Dict) ->
    Func = case Op of
	       '+'    -> m_plus;
	       '-'    -> m_minus;
	       '*'    -> m_times;
	       '/'    -> m_div;
	       'div'  -> int_div;
	       'rem'  -> int_rem;
	       'band' -> int_band;
	       'bor'  -> int_bor;
	       'bxor' -> int_bxor;
	       'bsl'  -> int_bsl;
	       'bsr'  -> int_bsr
	   end,
    encode_op(Func, [W, R0, R1, R2], Dict);
make_op({label, N}, Dict) ->
    encode_op(label, [{label, N}], Dict);
make_op({Op, Arg1}, Dict) ->
    encode_op(Op, [Arg1], Dict);

make_op({bif, Bif, Args, Arity}, Dict) ->
    case is_opcode(Bif, Arity) of
	true ->
	    encode_op(Bif, Args, Dict);
	false ->
	    NewArgs = insert_b_after_pf({b, {Bif, Arity}}, Args),
	    BifOp0 =
		case {Arity, length(NewArgs)-2} of
		    {0, 0} -> bif;
		    {0, _} -> gc_bif;
		    {Same, Same} -> gc_bif;
		    {_, _} -> bif
		end,
	    BifOp = list_to_atom(lists:concat([BifOp0, Arity])),
	    encode_op(BifOp, NewArgs, Dict)
    end;
make_op({atom_switch, R, Cases0}, Dict) ->
    Cases = flatmap(fun({A, L}) -> [A, {p, L}] end, Cases0),
    encode_op(atom_switch, [R, length(Cases0)], Cases, Dict);
make_op({jmp_switch, R, Cases0}, Dict) ->
    Cases = [{p, L} || {_, L} <- Cases0],
    encode_op(jmp_switch, [R, length(Cases0)], Cases, Dict);
make_op({lookup_switch, R, Cases0}, Dict) ->
    Cases = flatmap(fun({I, L}) -> [{i, I}, {p, L}] end, Cases0),
    encode_op(lookup_switch, [R, length(Cases0)], Cases, Dict);
make_op({Name, Arg1, Arg2}, Dict) ->
    encode_op(Name, [Arg1, Arg2], Dict);
make_op({Name, Arg1, Arg2, Arg3}, Dict) ->
    encode_op(Name, [Arg1, Arg2, Arg3], Dict);

make_op({callEx, Label, Module, Function, Arity, {Nm,Ar}}, Dict) ->
    encode_op(call_ext, [{extfunc, Module, Function, Arity}], Dict);
make_op({callExLast, Label, Module, Function, Arity, N, {Nm,Ar}}, Dict) ->
    encode_op(call_ext_last, [{extfunc, Module, Function, Arity}, N], Dict);
make_op({Name, Arg1, Arg2, Arg3, Arg4}, Dict) ->
    encode_op(Name, [Arg1, Arg2, Arg3, Arg4], Dict);
make_op({Name, Arg1, Arg2, Arg3, Arg4, Arg5}, Dict) ->
    encode_op(Name, [Arg1, Arg2, Arg3, Arg4, Arg5], Dict);
make_op({Name, Arg1, Arg2, Arg3, Arg4, Arg5, Arg6}, Dict) ->
    encode_op(Name, [Arg1, Arg2, Arg3, Arg4, Arg5, Arg6], Dict).

%% Inserts a {b, Bif} argument after the {p, Label} or {f, Label} argument (if any).

insert_b_after_pf(B, [P| Args]) when element(1, P) == p ->
    [P, B| Args];
insert_b_after_pf(B, [P| Args]) when element(1, P) == f ->
    [P, B|Args];
insert_b_after_pf(B, Args) ->
    [B| Args].
    
%% Returns true if the given bif is implemented as an opcode.

is_opcode(send, 2) -> true;
is_opcode(self, 0) -> true;
is_opcode(hd,1 ) -> true;
is_opcode(tl, 1) -> true;
is_opcode(node, 0) -> true;
is_opcode(make_ref, 0) -> true;
is_opcode(check_process_code, 2) -> true;
is_opcode(process_info, 2) -> true;
is_opcode(element, 2) -> true;
is_opcode(apply, 3) -> true;
is_opcode(_, _) -> false.

encode_op(Name, Args, Dict0) when atom(Name) ->
    {EncArgs, Dict} = encode_args(Args, Dict0),
    {[beam_opcodes:opcode(Name, length(Args))| EncArgs], Dict}.

encode_op(Name, Args, Eargs, Dict0) when atom(Name) ->
    {EncArgs, Dict} = encode_args(Args++Eargs, Dict0),
    {[beam_opcodes:opcode(Name, length(Args))| EncArgs], Dict}.

encode_args([Arg| T], Dict0) ->
    {EncArg, Dict1} = encode_arg(Arg, Dict0),
    {EncTail, Dict2} = encode_args(T, Dict1),
    {[EncArg| EncTail], Dict2};
encode_args([], Dict) ->
    {[], Dict}.

encode_arg({x, 0}, Dict) ->
    {encode(?tag_r, 0), Dict};
encode_arg({x, X}, Dict) ->
    {encode(?tag_x, X), Dict};
encode_arg({y, Y}, Dict) ->
    {encode(?tag_y, Y), Dict};
encode_arg({atom, Atom}, Dict0) when atom(Atom) ->
    {Index, Dict} = beam_dict:atom(Atom, Dict0),
    {encode(?tag_a, Index), Dict};
encode_arg({i, N}, Dict) ->
    {encode(?tag_i, N), Dict};
encode_arg(nil, Dict) ->
    {encode(?tag_n, 0), Dict};
encode_arg({f, W}, Dict) ->
    {encode(?tag_f, W), Dict};
encode_arg({p, W}, Dict) ->
    {encode(?tag_p, W), Dict};
encode_arg({label, N}, Dict) ->
    {encode(?tag_i, N), Dict};
encode_arg({b, {Name, Arity}}, Dict) ->
    case sys_bifs:bif(Name, Arity) of
	{Opcode, _} ->
	    {encode(?tag_i, Opcode), Dict};
	false ->
	    exit({bad_bif, Name, Arity})
    end;
encode_arg({arity, Arity}, Dict) ->
    {encode(?tag_i, Arity), Dict};
encode_arg({string, String}, Dict0) ->
    {Offset, Dict} = beam_dict:string(String, Dict0),
    {encode(?tag_i, Offset), Dict};
encode_arg({bignum, Arity, Sign}, Dict) when 0 =< Sign, Sign =< 1  ->
    {encode(?tag_i, Arity * 2 + Sign), Dict};
encode_arg({bignum_part, Part}, Dict) ->
    {encode(?tag_i, Part), Dict};
encode_arg({extfunc, M, F, A}, Dict0) ->
    {Index, Dict} = beam_dict:import(M, F, A, Dict0),
    {encode(?tag_i, Index), Dict};
encode_arg(Int, Dict) when integer(Int) ->
    {encode(?tag_i, Int), Dict};
encode_arg(Other, Dict) ->
    exit({badarg, encode_arg, [Other]}).

encode(Tag, N) when 0 =< N, N < 16 ->
    (N bsl 4) bor Tag;
encode(Tag, N) when 0 =< N, N < 16#800  ->
    [((N bsr 3) band 2#11100000) bor Tag bor 2#00001000, N band 16#ff];
encode(Tag, N) when N < 0 ->
    [(4 bsl 5) bor 2#00011000 bor Tag,
     (N bsr 24) band 16#ff,
     (N bsr 16) band 16#ff,
     (N bsr 8) band 16#ff,
     N band 16#ff];
encode(Tag, N) when N =< 16#ffffffff ->
    Bytes = to_bytes(N, []),
    Num = length(Bytes),
    [(Num bsl 5) bor 2#00011000 bor Tag| Bytes];
encode(Tag, N) ->
    throw({error, {too_big, N, 32}}).

to_bytes(0, Acc) ->
    Acc;
to_bytes(N, Acc) ->
    to_bytes(N bsr 8, [N band 16#ff| Acc]).
