/* 
   scrollbar.c [2000-11-10]
   (c) 2000 by Dieter Mittelmaier <dieter.mittelmaier@freenet.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <stdlib.h>
#include "drawutil.h"
#include "scrollbar.h"

int slidermousedif;

SCROLLBAR *ScrollBar(Sint16 x, Sint16 y, Uint16 size, int orientation)
{
	SCROLLBAR *s;
	s = (SCROLLBAR *)malloc(sizeof(SCROLLBAR));
	if (s) {
		s->rect.x = x;
		s->rect.y = y;
		if (orientation == VERTICAL) {
			s->rect.w = 15;
			s->rect.h = size;
			s->sliderarea.x = s->rect.x + 2;
			s->sliderarea.y = s->rect.y + 2 + s->rect.w - 4 + 1;
			s->sliderarea.w = s->rect.w - 4;
			s->sliderarea.h = s->rect.h - 4 - (s->rect.w - 4) * 2 - 2;
		} else {
			s->rect.w = size;
			s->rect.h = 15;
			s->sliderarea.x = s->rect.x + 2 + s->rect.h - 4 + 1;
			s->sliderarea.y = s->rect.y + 2;
			s->sliderarea.w = s->rect.w - 4 - (s->rect.h - 4) * 2 - 2;
			s->sliderarea.h = s->rect.h - 4;
		}
		s->orientation = orientation;
		s->linestep = 0;
		s->pagestep = 0;
		s->minvalue = 0;
		s->maxvalue = 0;
		s->curvalue = 0;
		s->isvisible = 0;
		return s;
	} else {
		printf("ScrollBar not allocated\n");
		return NULL;
	}
}

int getValueFromSlider(SCROLLBAR *s, SDL_Surface *surface, Sint16 x, Sint16 y)
{
	int slidermin, slidermax, r;

	if (s->orientation == VERTICAL) {
		slidermin = s->sliderarea.y;
		slidermax = s->sliderarea.y + s->sliderarea.h - s->slider.h;
		s->slider.y = y + slidermousedif;
		if (s->slider.y < slidermin)
			s->slider.y = slidermin;
		if (s->slider.y > slidermax)
			s->slider.y = slidermax;

		r = (s->maxvalue - s->minvalue ) * (s->slider.y - slidermin) /
													(slidermax - slidermin);
	} else {
		slidermin = s->sliderarea.x;
		slidermax = s->sliderarea.x + s->sliderarea.w - s->slider.w;
		s->slider.x = x + slidermousedif;
		if (s->slider.x < slidermin)
			s->slider.x = slidermin;
		if (s->slider.x > slidermax)
			s->slider.x = slidermax;

		r = (s->maxvalue - s->minvalue) * (s->slider.x - slidermin) /
												(slidermax - slidermin);
	}

	SDL_FillRect(surface, &s->sliderarea, back_color);
	drawRaisedWin(surface, &s->slider, light_color, shadow_color, back_color);
	s->curvalue = r;

	return r;
}

void updateScrollBarSlider(SCROLLBAR *s, SDL_Surface *surface, int value)
{
	if (!s->isvisible)
		return;

	if (s->orientation == VERTICAL) {
		int slidermin = s->sliderarea.y;
		int slidermax = s->sliderarea.y + s->sliderarea.h - s->slider.h;
		int r = ((slidermax - slidermin) * 2 * (value - s->minvalue) + 1) /
					((s->maxvalue - s->minvalue) * 2);
		s->curvalue = value;
		s->slider.y = s->sliderarea.y + r;
		SDL_FillRect(surface, &s->sliderarea, back_color);
		drawRaisedWin(surface, &s->slider, light_color, shadow_color, back_color);
	} else {
		int slidermin = s->sliderarea.x;
		int slidermax = s->sliderarea.x + s->sliderarea.w - s->slider.w;
		int r = ((slidermax - slidermin) * 2 * (value - s->minvalue) + 1) /
					((s->maxvalue - s->minvalue) * 2);
		s->curvalue = value;
		s->slider.x = s->sliderarea.x + r;
		SDL_FillRect(surface, &s->sliderarea, back_color);
		drawRaisedWin(surface, &s->slider, light_color, shadow_color, back_color);
	}
}

void drawScrollBarArrows(SCROLLBAR *s, SDL_Surface *surface)
{
	if (!s->isvisible)
		return;

	if (s->orientation == VERTICAL) {
		SDL_Rect r;
		updateScrollBarSlider(s, surface, s->curvalue);
		r.x = s->sliderarea.x;
		r.y = s->sliderarea.y - s->sliderarea.w - 1;
		r.w = s->sliderarea.w;
		r.h = r.w;
		drawUpArrow(surface, &r, light_color, shadow_color, back_color);
		r.y = s->sliderarea.y + s->sliderarea.h;
		drawDownArrow(surface, &r, light_color, shadow_color, back_color);
	} else {
		SDL_Rect r;
		updateScrollBarSlider(s, surface, s->curvalue);
		r.x = s->sliderarea.x - s->sliderarea.h - 1;
		r.y = s->sliderarea.y;
		r.w = s->sliderarea.h;
		r.h = r.w;
		drawLeftArrow(surface, &r, light_color, shadow_color, back_color);
		r.x = s->sliderarea.x + s->sliderarea.w;
		drawRightArrow(surface, &r, light_color, shadow_color, back_color);
	}
}

void drawScrollBar(SCROLLBAR *s, SDL_Surface *surface)
{
	if (!s->isvisible)
		return;

	drawSunkenWin(surface, &s->rect, light_color, shadow_color, back_color);
	drawScrollBarArrows(s, surface);
}

void setScrollBarSize(SCROLLBAR *s, Uint16 size, int visible)
{
	if (s->orientation == VERTICAL) {
		s->rect.h = size;
		s->isvisible = visible;
		s->sliderarea.x = s->rect.x + 2;
		s->sliderarea.y = s->rect.y + 2 + s->rect.w - 4 + 1;
		s->sliderarea.w = s->rect.w - 4;
		s->sliderarea.h = s->rect.h - 4 - (s->rect.w - 4) * 2 - 2;
	} else {
		s->rect.w = size;
		s->isvisible = visible;
		s->sliderarea.x = s->rect.x + 2 + s->rect.h - 4 + 1;
		s->sliderarea.y = s->rect.y + 2;
		s->sliderarea.w = s->rect.w - 4 - (s->rect.h - 4) * 2 - 2;
		s->sliderarea.h = s->rect.h - 4;
	}
}

void setScrollBarRange(SCROLLBAR *s, int linestep, int pagestep, int minvalue, int maxvalue, int value)
{
	if (!s->isvisible)
		return;

	if (s->orientation == VERTICAL) {
		s->slider.x = s->sliderarea.x;
		s->slider.y = s->sliderarea.y;
		s->slider.w = s->sliderarea.w;
		s->slider.h = s->sliderarea.h * (1.0 * pagestep / (maxvalue - minvalue + 1));
		if (s->slider.h < 9)
			s->slider.h = 9;
		if (s->slider.h > s->sliderarea.h)
			s->slider.h = s->sliderarea.h;
		s->linestep = linestep;
		s->pagestep = pagestep;
		s->minvalue = minvalue;
		s->maxvalue = maxvalue;
		s->curvalue = value;
	} else {
		s->slider.x = s->sliderarea.x;
		s->slider.y = s->sliderarea.y;
		s->slider.w = s->sliderarea.w * (1.0 * pagestep / (maxvalue - minvalue + 1));
		s->slider.w = (s->sliderarea.w * 1.0 * pagestep) / (maxvalue - minvalue + pagestep);
		s->slider.h = s->sliderarea.h;
		if (s->slider.w < 9)
			s->slider.w = 9;
		if (s->slider.w > s->sliderarea.w)
			s->slider.w = s->sliderarea.w;
		s->linestep = linestep;
		s->pagestep = pagestep;
		s->minvalue = minvalue;
		s->maxvalue = maxvalue;
		s->curvalue = value;
	}
}

void deleteScrollBar(SCROLLBAR *s)
{
	if (s) {
		free(s);
		s = NULL;
	}
}

int insideScrollBar(Sint16 x, Sint16 y, SDL_Rect *r)
{
	if ((x >= r->x) && (x < r->x + r->w)) {
		if ((y >= r->y) && (y < r->y + r->h)) {
			return 1;
		}
	}

	return 0;
}

int handleScrollBarEvent(SDL_Event *event, SCROLLBAR *s, SDL_Surface *surface)
{
	SDL_Rect arrowUpLeft;
	SDL_Rect arrowDownRight;
	int action = 0;

	arrowUpLeft.x = s->rect.x + 2;
	arrowUpLeft.y = s->rect.y + 2;
	arrowUpLeft.w = 11;
	arrowUpLeft.h = 11;
	if (s->orientation == VERTICAL) {
		arrowDownRight.x = s->rect.x + 2;
		arrowDownRight.y = s->sliderarea.y + s->sliderarea.h + 1;
	} else {
		arrowDownRight.x = s->sliderarea.x + s->sliderarea.w + 1;
		arrowDownRight.y = s->rect.y + 2;
	}
	arrowDownRight.w = 11;
	arrowDownRight.h = 11;
	
	if (!s->isvisible)
		return 0;

	switch (event->type) {
		case SDL_MOUSEBUTTONUP:
		case SDL_MOUSEBUTTONDOWN:
			if ((event->button.state == SDL_PRESSED) && (event->button.button == SDL_BUTTON_LEFT)) {
				if (insideScrollBar(event->button.x, event->button.y, &s->rect)) {
					if (insideScrollBar(event->button.x, event->button.y, &arrowUpLeft)) {
						if (s->orientation == VERTICAL) {
							action = SCROLL_UP;
						} else {
							action = SCROLL_LEFT;
						}
					} else if (insideScrollBar(event->button.x, event->button.y, &arrowDownRight)) {
						if (s->orientation == VERTICAL) {
							action = SCROLL_DOWN;
						} else {
							action = SCROLL_RIGHT;
						}
					} else if (insideScrollBar(event->button.x, event->button.y, &s->slider)) {
						if (s->orientation == VERTICAL) {
							slidermousedif = s->slider.y - event->button.y;
							action = SCROLL_VERTICAL;
						} else {
							slidermousedif = s->slider.x - event->button.x;
							action = SCROLL_HORIZONTAL;
						}
					} else if (insideScrollBar(event->button.x, event->button.y, &s->sliderarea)) {
						if (s->orientation == VERTICAL) {
							if (event->button.y < s->slider.y) {
								action = SCROLL_PAGE_UP;
							} else {
								action = SCROLL_PAGE_DOWN;
							}
						} else {
							if (event->button.x < s->slider.x) {
								action = SCROLL_PAGE_LEFT;
							} else {
								action = SCROLL_PAGE_RIGHT;
							}
						}
					}
				}
			}
			break;
		default:
			break;
	}

	return action;
}

