/* 
   preferences.c [2001-02-14]
   (c) 2000 by Dieter Mittelmaier <dieter.mittelmaier@freenet.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <SDL.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "button.h"
#include "drawutil.h"
#include "listbox.h"
#include "preferences.h"

#define PREF_FILE "/.enjoympegrc"

#define OK				1
#define CANCEL			2

#define GENERAL			0
#define WINSIZE			1
#define FILTERS			2

#define MAX_TAB			3
#define MAX_GEN			4
#define MAX_GEO			7
#define MAX_FIL			3

#define KEEP_ASPECT_RATIO	0
#define PLAY_ON_START		1
#define ALWAYS_LOOP			2
#define SAVE_PREFS			3

typedef struct _tab {
	SDL_Rect rect;
	int active;
	char *name;
}TAB;

TAB tabbar[MAX_TAB];

char *tabtitel[MAX_TAB] = {"General",
						   "Window Size",
						   "Filters"};

char *geometry[MAX_GEO] = {" 320x240",
						   " 352x288",
						   " 384x288",
						   " 640x480",
						   " 768x576",
						   " 800x600",
						   "1024x768"};

BUTTON *genCheck[MAX_GEN];
char *general[MAX_GEN] = {"Keep aspect ratio if video is scaled",
						  "Start playing after video is opened",
						  "Always loop video",
						  "Save settings on exit"};

BUTTON *filCheck[MAX_FIL];
char *filters[MAX_FIL] = {"No Filter",
						  "Bilinear Filter",
						  "Deblocking Filter"};


SDL_Rect dialog;
SDL_Rect tabarea;
static SDL_Surface *backingstore = NULL;
LISTBOX *wsize;
BUTTON *okButton, *cancelButton;

int margin = 4;

char *skipTabsnSpaces(char *start)
{
	while ((*start == ' ') || (*start == '\t')) {
		start++;
	}

	return start;
}

void setdefaultValues(PREFS *p)
{
	p->win_width = 640;
	p->win_height = 480;
	p->keep_aspect_ratio = 1;
	p->loop = 0;
	p->playonstart = 0;
	p->save_prefs = 1;
	p->filter = NULL_FILTER;
}

void savePrefs(PREFS *p)
{
	char file[1024];
	FILE *fp;
	
	strcpy(file, getenv("HOME"));
	strcat(file, PREF_FILE);

	if ((fp = fopen(file, "w")) == NULL) {
		printf("Error creating file: %s", file);
		return;
	}

	fprintf(fp, "#This is the enjoympeg preferences file\n\n");
	fprintf(fp, "WINDOW_SIZE %dx%d\n", p->win_width, p->win_height);
	fprintf(fp, "KEEP_ASPECT_RATIO %d\n", p->keep_aspect_ratio);
	fprintf(fp, "ALWAYS_LOOP %d\n", p->loop);
	fprintf(fp, "PLAY_ON_START %d\n", p->playonstart);
	fprintf(fp, "SAVE_PREFS %d\n", p->save_prefs);
	fprintf(fp, "FILTER %d\n", p->filter);

	fclose(fp);
}

void loadPrefs(PREFS *p)
{
	char file[1024];
	char line[256];
	FILE *fp;
	char *pos = NULL;
	int len = 0;
	
	strcpy(file, getenv("HOME"));
	strcat(file, PREF_FILE);

	if ((fp = fopen(file, "r")) == NULL) {
		setdefaultValues(p);
		savePrefs(p);
		return;
	}

	while (fgets(line, 256 ,fp) != NULL) {
		if (line[0] == '#')
			continue;
		len = strlen(line);
		if (line[len - 1] == '\n') {
			len--;
			line[len] = '\0';
			while ((len >= 0) && ((line[len - 1] == ' ')
								|| (line[len - 1] == '\t'))) {
				line[--len] = '\0';
			}
		}

		if (strlen(skipTabsnSpaces(line)) == 0)
			continue;		

		if (strncmp(line, "WINDOW_SIZE", 11) == 0) {
			pos = skipTabsnSpaces(&line[11]);
			sscanf(pos, "%dx%d", &p->win_width, &p->win_height);
		} else if (strncmp(line, "KEEP_ASPECT_RATIO", 17) == 0) {
			pos = skipTabsnSpaces(&line[17]);
			sscanf(pos, "%d", &p->keep_aspect_ratio);
		} else if (strncmp(line, "ALWAYS_LOOP", 11) == 0) {
			pos = skipTabsnSpaces(&line[11]);
			sscanf(pos, "%d", &p->loop);
		} else if (strncmp(line, "PLAY_ON_START", 13) == 0) {
			pos = skipTabsnSpaces(&line[13]);
			sscanf(pos, "%d", &p->playonstart);
		} else if (strncmp(line, "SAVE_PREFS", 10) == 0) {
			pos = skipTabsnSpaces(&line[10]);
			sscanf(pos, "%d", &p->save_prefs);
		} else if (strncmp(line, "FILTER", 6) == 0) {
			pos = skipTabsnSpaces(&line[6]);
			sscanf(pos, "%d", &p->filter);
		}
	}

	fclose(fp);
}

void drawTabArea(SDL_Surface *surface, int t)
{
	if (t == GENERAL) {
		int i;
		for (i = 0; i < MAX_GEN; i++) {
			drawButton(surface, genCheck[i]);
		}
	} else if (t == FILTERS) {
		int i;
		for (i = 0; i < MAX_FIL; i++) {
			drawButton(surface, filCheck[i]);
		}
	} else if (t == WINSIZE) {
		drawListBox(wsize, surface);
	}
}

int drawTabBar(SDL_Surface *surface)
{
	int t;
	int active = 0;

	drawRaisedWin(surface, &tabarea, light_color, shadow_color, back_color);
	for (t = 0; t < MAX_TAB; t++) {
		if (tabbar[t].active)
			drawLine(surface, tabbar[t].rect.x, tabbar[t].rect.y,
					2, tabbar[t].rect.h, light_color);
		else
			drawLine(surface, tabbar[t].rect.x, tabbar[t].rect.y,
					2, tabbar[t].rect.h, shadow_color);

		if (tabbar[t].active)
			drawLine(surface, tabbar[t].rect.x, tabbar[t].rect.y,
					tabbar[t].rect.w, 2, light_color);
		else
			drawLine(surface, tabbar[t].rect.x, tabbar[t].rect.y,
					tabbar[t].rect.w, 2, shadow_color);

		drawText(surface, tabbar[t].rect.x + margin + 1, tabbar[t].rect.y + margin + 1,
				tabbar[t].name);

		drawLine(surface, tabbar[t].rect.x + tabbar[t].rect.w, tabbar[t].rect.y,
				2, tabbar[t].rect.h, shadow_color);

		if (tabbar[t].active) {
			drawLine(surface, tabbar[t].rect.x + 2, tabbar[t].rect.y + tabbar[t].rect.h,
					tabbar[t].rect.w - 2, 2, back_color);
			active = t;
		}
	}

	drawTabArea(surface, active);
	return active;
}

int insideTab(Sint16 x, Sint16 y)
{
	int i;
	int activetab = 0;

	for (i = 0; i < MAX_TAB; i++) {
		if (tabbar[i].active) {
			activetab = i;
			break;
		}
	}

	for (i = 0; i < MAX_TAB; i++) {
		if ((x >= tabbar[i].rect.x) && (x < tabbar[i].rect.x + tabbar[i].rect.w)) {
			if ((y >= tabbar[i].rect.y) && (y < tabbar[i].rect.y + tabbar[i].rect.h)) {
				if (tabbar[i].active) {
					return 0;
				} else {
					tabbar[activetab].active = 0;
					tabbar[i].active = 1;
					return 1;
				}
			}
		}
	}

	return 0;
}

int handleTabBarEvent(SDL_Event *event)
{
	int handled = 0;

	switch (event->type) {
		case SDL_MOUSEBUTTONUP:
		case SDL_MOUSEBUTTONDOWN:
			if ((event->button.state == SDL_PRESSED)
					&& (event->button.button == SDL_BUTTON_LEFT)) {
				if (insideTab(event->button.x, event->button.y)) {
					handled = 1;
				}
			}
			break;
		default:
			break;
	}
	return handled;
}

void createPrefsdialog(SDL_Surface *surface)
{
	SDL_Rect r;
	SDL_Rect buttonarea;
	SDL_Rect button;
	int i;
	int w = 320;
	int h = 240;
	
	if (surface->w < w)
		w = surface->w;
	if (surface->h < h)
		h = surface->h;

	dialog.x = (surface->w - w) / 2;
	dialog.y = (surface->h - h) / 2;
	dialog.w = w;
	dialog.h = h;

	backingstore = SDL_CreateRGBSurface(SDL_SWSURFACE, dialog.w, dialog.h,
								surface->format->BitsPerPixel,
								surface->format->Rmask, surface->format->Gmask,
								surface->format->Bmask, surface->format->Amask);

	if (backingstore)
		SDL_BlitSurface(surface, &dialog, backingstore, NULL);

	buttonarea.x = dialog.x + 10;
	buttonarea.y = dialog.y + dialog.h - 1 - 20 - (charHeight() + 4 + margin * 2);
	buttonarea.w = dialog.w - 20;
	buttonarea.h = dialog.y + dialog.h - buttonarea.y;

	tabarea.x = dialog.x + 10;
	tabarea.y = dialog.y + 10 + charHeight() + margin * 2;
	tabarea.w = dialog.w - 20 - 1;
	tabarea.h = buttonarea.y - tabarea.y + 1;

	button.x = tabarea.x + 10;
	button.y = tabarea.y + 10;
	button.h = charHeight() + 4 + margin * 2;

	for (i = 0; i < MAX_GEN; i++) {
		button.w = button.h + strlen(general[i]) * charWidth();
		genCheck[i] = Button(&button, i);
		setButtonText(genCheck[i], general[i]);
		setCheckButton(genCheck[i], 0);
		button.y += button.h;
	}

	button.x = tabarea.x + 10;
	button.y = tabarea.y + 10;
	button.h = charHeight() + 4 + margin * 2;

	for (i = 0; i < MAX_FIL; i++) {
		button.w = button.h + strlen(filters[i]) * charWidth();
		filCheck[i] = Button(&button, i);
		setButtonText(filCheck[i], filters[i]);
		setCheckButton(filCheck[i], 0);
		button.y += button.h;
	}

	tabbar[0].rect.x = dialog.x + 10;
	tabbar[0].rect.y = dialog.y + 10;
	tabbar[0].rect.w = strlen(tabtitel[0]) * charWidth() + margin * 2;
	tabbar[0].rect.h = charHeight() + margin * 2;
	tabbar[0].active = 1;
	tabbar[0].name = tabtitel[0];
	
	tabbar[1].rect.x = tabbar[0].rect.x + tabbar[0].rect.w;
	tabbar[1].rect.y = dialog.y + 10;
	tabbar[1].rect.w = strlen(tabtitel[1]) * charWidth() + margin * 2;
	tabbar[1].rect.h = charHeight() + margin * 2;
	tabbar[1].active = 0;
	tabbar[1].name = tabtitel[1];

	tabbar[2].rect.x = tabbar[1].rect.x + tabbar[1].rect.w;
	tabbar[2].rect.y = dialog.y + 10;
	tabbar[2].rect.w = strlen(tabtitel[2]) * charWidth() + margin * 2;
	tabbar[2].rect.h = charHeight() + margin * 2;
	tabbar[2].active = 0;
	tabbar[2].name = tabtitel[2];


	button.x = buttonarea.x;
	button.y = buttonarea.y + 10;
	button.w = (strlen("Cancel") + 2) * charWidth();
	button.h = charHeight() + 4 + margin * 2;
	okButton = Button(&button, OK);
	setButtonText(okButton, "OK");

	button.x += button.w + 10;
	cancelButton = Button(&button, CANCEL);
	setButtonText(cancelButton, "Cancel");

	r.x = tabarea.x + 10;
	r.y = tabarea.y + 10;
	r.w = (tabarea.w - 30) / 2;
	r.h = tabarea.h - 20; //buttonarea.y - r.y;

	wsize = ListBox(r.x, r.y, r.w, r.h, margin, 1);
	for (i = 0; i < MAX_GEO; i++)
		addListBoxItem(wsize, geometry[i], NULL);

	initListBox(wsize, surface);
	setListBoxFocusItem(wsize, 0);
	
}

void prefDialog(SDL_Surface *surface, PREFS *p)
{
	int i;
	int activetab;
	char geo[9];
	int done = 0;
	int cancel = 0;
	okButton = NULL;
	cancelButton = NULL;
	
	loadFont();
	setTransparent(1);
	createPrefsdialog(surface);

	setCheckButton(genCheck[KEEP_ASPECT_RATIO], p->keep_aspect_ratio);
	setCheckButton(genCheck[PLAY_ON_START], p->playonstart);
	setCheckButton(genCheck[ALWAYS_LOOP], p->loop);
	setCheckButton(genCheck[SAVE_PREFS], p->save_prefs);
	setCheckButton(filCheck[p->filter], 1);
	sprintf(geo,"%dx%d",p->win_width, p->win_height);
	for (i = 0; i < MAX_GEO; i++) {
		if (strstr(geometry[i], geo)) {
			setListBoxFocusItem(wsize, i);
			break;
		}
	}
	
	drawRaisedWin(surface, &dialog, light_color, shadow_color, back_color);
	activetab = drawTabBar(surface);
	drawButton(surface, okButton);
	drawButton(surface, cancelButton);
	SDL_UpdateRects(surface, 1, &dialog);
	

	SDL_EnableKeyRepeat(SDL_DEFAULT_REPEAT_DELAY, SDL_DEFAULT_REPEAT_INTERVAL);
	while ( ! done ) {
		SDL_Event event;
		SDL_WaitEvent(&event);
		if (handleTabBarEvent(&event)) {
			activetab = drawTabBar(surface);
			SDL_UpdateRects(surface, 1, &dialog);
			continue;
		} else if (activetab == GENERAL) {
			if ((event.button.state == SDL_PRESSED) && (event.button.button == SDL_BUTTON_LEFT)) {
				int pressed = 0;
				for (i = 0; i < MAX_GEN; i++) {
					if (isButtonPressed(genCheck[i], event.button.x, event.button.y)) {
						drawButton(surface, genCheck[i]);
						SDL_UpdateRects(surface, 1, &dialog);
						pressed = 1;
						break;
					}
				}
				if (pressed) continue;
			}
		} else if (activetab == FILTERS) {
			if ((event.button.state == SDL_PRESSED) && (event.button.button == SDL_BUTTON_LEFT)) {
				/*
				HACK to make radiobuttons out of checkbuttons
				*/
				int pressed = 0;
				for (i = 0; i < MAX_FIL; i++) {
					if (isButtonPressed(filCheck[i], event.button.x, event.button.y)) {
						int d;
						pressed = 1;
						for (d = 0; d < MAX_FIL; d++) {
							setCheckButton(filCheck[d], 0);
							drawButton(surface, filCheck[d]);
						}
						setCheckButton(filCheck[i], 1);
						drawButton(surface, filCheck[i]);
						SDL_UpdateRects(surface, 1, &dialog);
						break;
					}
				}
				if (pressed) continue;
			}
		} else if (activetab == WINSIZE) {
			if (handleListBoxEvent(&event, wsize, surface)) {
				continue;
			}
		}
		switch (event.type) {
			case SDL_KEYDOWN:
				if (event.key.keysym.sym == SDLK_ESCAPE) {
					cancel = 1;
					done = 1;
				} else if (event.key.keysym.sym == SDLK_RETURN) {
						done = 1;
				}
				break;
			case SDL_MOUSEBUTTONUP:
			case SDL_MOUSEBUTTONDOWN:
				if ((event.button.state == SDL_PRESSED) && (event.button.button == SDL_BUTTON_LEFT)) {
					if (isButtonPressed(okButton, event.button.x, event.button.y)) {
						drawButton(surface, okButton);
					} else if (isButtonPressed(cancelButton, event.button.x, event.button.y)) {
						drawButton(surface, cancelButton);
					}
				} else if ((event.button.state == SDL_RELEASED) && (event.button.button == SDL_BUTTON_LEFT)) {
					if (isButtonReleased(okButton, event.button.x, event.button.y)) {
						done = 1;
					} else if (isButtonReleased(cancelButton, event.button.x, event.button.y)) {
						cancel = 1;
						done = 1;
					}
					drawButton(surface, cancelButton);
					drawButton(surface, okButton);
				}
				SDL_UpdateRects(surface, 1, &dialog);
				break;
			default:
				break;
		}
	}
	SDL_EnableKeyRepeat(0, SDL_DEFAULT_REPEAT_INTERVAL);

	if (backingstore) {
		SDL_BlitSurface(backingstore, NULL, surface, &dialog);
		SDL_UpdateRect(surface,dialog.x,dialog.y,dialog.w,dialog.h);
		SDL_FreeSurface(backingstore);
		backingstore = NULL;
	}

	if (!cancel) {
		char *ws;
		int save_at_exit = p->save_prefs;

		p->keep_aspect_ratio = isButtonChecked(genCheck[KEEP_ASPECT_RATIO]);
		p->playonstart = isButtonChecked(genCheck[PLAY_ON_START]);
		p->loop = isButtonChecked(genCheck[ALWAYS_LOOP]);
		for (i = 0; i < MAX_FIL; i++) {
			if(isButtonChecked(filCheck[i])) {
				p->filter = i;
				break;
			}
		}
		i = ListBoxFocusItem(wsize);
		ws = skipTabsnSpaces(geometry[i]);
		sscanf(ws, "%dx%d", &p->win_width, &p->win_height);
		p->save_prefs = isButtonChecked(genCheck[SAVE_PREFS]);
		if (save_at_exit && (!p->save_prefs))
			savePrefs(p);
	}
	
	for (i = 0; i < MAX_GEN; i++) {
		if (genCheck[i]) free(genCheck[i]);
	}
	for (i = 0; i < MAX_FIL; i++) {
		if (filCheck[i]) free(filCheck[i]);
	}
	if (okButton) free(okButton);
	if (cancelButton) free(cancelButton);
	deleteListBox(wsize);

/*
	printf("#This is the enjoympeg preferences file\n\n");
	printf("WINDOW_SIZE %dx%d\n", p->win_width, p->win_height);
	printf("KEEP_ASPECT_RATIO %d\n", p->keep_aspect_ratio);
	printf("ALWAYS_LOOP %d\n", p->loop);
	printf("PLAY_ON_START %d\n", p->playonstart);
	printf("SAVE_PREFS %d\n", p->save_prefs););
	printf("FILTER %d\n", p->filter);
*/

}
