/* 
   listbox.c [2000-11-5]
   (c) 2000 by Dieter Mittelmaier <dieter.mittelmaier@freenet.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <stdlib.h>
#include <string.h>
#include "listbox.h"
#include "drawutil.h"
#include "scrollbar.h"

#define ITEM_ARRAY_SIZE	100

typedef struct _ListItem {
	char *text1;
	char *text2;
}LISTITEM;

typedef struct _ViewPort {
	SDL_Rect rect;
	SDL_Surface *surface;
}VIEWPORT;

struct _ListBox {
	SDL_Rect rect;
	VIEWPORT *viewPort;
	LISTITEM **items;
	int showFocus;
	int itemcount;
	int numitems;
	int maxitemlen;
	int visibleitems;
	int visiblecolumns;
	int focusitem;
	int margin;
	Sint16 offx;
	Sint16 offy;
	Sint16 xstep;
	Sint16 ystep;
	Sint16 xpage;
	Sint16 ypage;
	Uint32 backcolor;
	Uint32 focuscolor;
	SCROLLBAR * hsBar;
	SCROLLBAR * vsBar;
};

LISTBOX *ListBox(Sint16 x, Sint16 y, Uint16 max_w, Uint16 max_h, int margin, int showFocus)
{
	LISTBOX *lb;
	lb = (LISTBOX *)malloc(sizeof(LISTBOX));
	if (lb) {
		lb->viewPort = (VIEWPORT *)malloc(sizeof(VIEWPORT));
		lb->viewPort->surface = NULL;
		lb->rect.x = x;
		lb->rect.y = y;
		lb->rect.w = max_w;
		lb->rect.h = max_h;
		lb->items = NULL;
		lb->showFocus = showFocus;
		lb->itemcount = 0;
		lb->numitems = 0;
		lb->maxitemlen = 0;
		lb->visibleitems = 0;
		lb->focusitem = 0;
		lb->margin = margin;
		lb->offx = 0;
		lb->offy = 0;
		lb->xstep = charWidth();
		lb->ystep = charHeight() + 1;
		lb->backcolor = 0;
		lb->focuscolor = 0;
		lb->viewPort->rect.x = lb->rect.x + 2 + lb->margin;
		lb->viewPort->rect.y = lb->rect.y + 2 + lb->margin;
		lb->viewPort->rect.w = lb->rect.w - 4 - lb->margin * 2;
		lb->viewPort->rect.h = ((lb->rect.h - 4 - lb->margin * 2) / lb->ystep) * lb->ystep;
		lb->rect.h = lb->viewPort->rect.h + 4 + lb->margin * 2;
		lb->hsBar = ScrollBar(lb->rect.x, lb->rect.y  + lb->rect.h - 15,
								lb->rect.w, HORIZONTAL);
		lb->vsBar = ScrollBar(lb->rect.x + lb->rect.w - 15, lb->rect.y,
								lb->rect.h, VERTICAL);
		return lb;
	} else {
		printf("ListBox not allocated\n");
		return NULL;
	}
}
	
void deleteitems(LISTBOX *lb)
{
	int i;

	for ( i = 0; i < lb->itemcount; i++)
		free(lb->items[i]);

	lb->itemcount = 0;
}

void deleteListBox(LISTBOX *lb)
{
	if (lb) {
		if (lb->viewPort) {
			if (lb->viewPort->surface)
				SDL_FreeSurface(lb->viewPort->surface);
			free(lb->viewPort);
		}
		if (lb->items) {
			deleteitems(lb);
			free(lb->items);
		}
		deleteScrollBar(lb->hsBar);
		deleteScrollBar(lb->vsBar);
		free(lb);
		lb = NULL;
	}
}

int addListBoxItem(LISTBOX *lb, char *text1, char *text2)
{
	int itemlen = 0;
	LISTITEM *li = NULL;
	
	if (!lb)
		return 0;

	if (lb->numitems == 0) {
		lb->items = (LISTITEM **) malloc(ITEM_ARRAY_SIZE * sizeof(LISTITEM *));
		if (lb->items == NULL) {
			printf("Malloc ItemArray failed\n");
			return 0;
		}
		lb->numitems = ITEM_ARRAY_SIZE;
	}
	
	if (lb->numitems == lb->itemcount) {
		LISTITEM **newitems;
		lb->numitems += ITEM_ARRAY_SIZE;
		newitems = (LISTITEM **) realloc(lb->items, lb->numitems * sizeof(LISTITEM *));
		if (newitems == NULL) {
			printf("Realloc ItemArray failed\n");
			return 0;
		}
		lb->items = newitems;
	}

	li = (LISTITEM *) malloc(sizeof(LISTITEM));
	if (li == NULL) {
		printf("Malloc LISTITEM failed\n");
		return 0;
	}
	li->text1 = NULL;
	li->text2 = NULL;

	if (text1) {
		itemlen = strlen(text1);
		li->text1 = text1;
	}
		
	if (text2) {
		itemlen += strlen(text2);
		li->text2 = text2;
	}

	if (itemlen > lb->maxitemlen)
		lb->maxitemlen = itemlen;

	lb->items[lb->itemcount] = li;
	lb->itemcount++;

	return 1;
}

void updateViewPort(LISTBOX *lb, SDL_Surface *surface)
{
	SDL_Rect src;
	SDL_Rect dst;
	SDL_Rect focus;
	int i;
	int x;
	int y;


	src.x = 0;
	src.y = 0;
	src.w = lb->viewPort->rect.w;
	src.h = lb->viewPort->rect.h;
	SDL_FillRect(lb->viewPort->surface, &src, lb->backcolor);

	focus.x = 0;
	focus.w = lb->viewPort->rect.w;
	focus.h = lb->ystep;

	x = lb->offx;
	y = 0;
	
	for (i = lb->offy / lb->ystep; i < lb->itemcount; i++) {
		if ((i == lb->focusitem) && lb->showFocus) {
			focus.y = y;
			SDL_FillRect(lb->viewPort->surface, &focus, lb->focuscolor);
		}

		if (lb->items[i]->text1)
			drawText(lb->viewPort->surface, x, y, lb->items[i]->text1);
		if (lb->items[i]->text2)
			drawText(lb->viewPort->surface, x + lb->xstep, y, lb->items[i]->text2);

		y += lb->ystep;
		if (y >= lb->viewPort->rect.h)
			break;
	}

	dst.x = lb->viewPort->rect.x;
	dst.y = lb->viewPort->rect.y;
	dst.w = lb->viewPort->rect.w;
	dst.h = lb->viewPort->rect.h;

	SDL_BlitSurface(lb->viewPort->surface, &src, surface, &dst);
}


void drawListBox(LISTBOX *lb, SDL_Surface *surface)
{
	SDL_Rect l;
	l.x = lb->rect.x;
	l.y = lb->rect.y;
	l.w = lb->rect.w;
	if (lb->vsBar->isvisible)
		l.w -= lb->vsBar->rect.w; 
	l.h = lb->rect.h;
	if (lb->hsBar->isvisible)
		l.h -= lb->hsBar->rect.h; 
	
	SDL_FillRect(surface, &lb->rect, back_color);
	drawSunkenWin(surface, &l, light_color, shadow_color, lb->backcolor);
	drawScrollBar(lb->hsBar, surface);
	drawScrollBar(lb->vsBar, surface);
	updateViewPort(lb, surface);
}

void setListBoxSizes(LISTBOX *lb)
{
	int needVScroll = 0;
	int needHScroll = 0;
	Uint16 hbarsize = lb->rect.w;
	Uint16 vbarsize = lb->rect.h;
	
	lb->viewPort->rect.x = lb->rect.x + 2 + lb->margin;
	lb->viewPort->rect.y = lb->rect.y + 2 + lb->margin;
	lb->viewPort->rect.w = lb->rect.w - 4 - lb->margin * 2;
	lb->viewPort->rect.h = lb->rect.h - 4 - lb->margin * 2;

	lb->visibleitems = lb->viewPort->rect.h / lb->ystep;
	if (lb->itemcount > lb->visibleitems) {
		needVScroll = 1;
		lb->viewPort->rect.w -= 15;
		if (lb->viewPort->rect.w < (lb->xstep * lb->maxitemlen)) {
			needHScroll = 1;
			hbarsize -= 15;
			vbarsize -= 15;
			lb->viewPort->rect.h -= 15;
			lb->visibleitems -= 1;
		}
	} else if (lb->viewPort->rect.w < (lb->xstep * lb->maxitemlen)) {
		needHScroll = 1;
		lb->viewPort->rect.h -= 15;
		lb->visibleitems -= 1;
		if (lb->itemcount > lb->visibleitems) {
			needVScroll = 1;
			hbarsize -= 15;
			vbarsize -= 15;
			lb->viewPort->rect.w -= 15;
		}
	}
	setScrollBarSize(lb->hsBar, hbarsize, needHScroll);
	setScrollBarSize(lb->vsBar, vbarsize, needVScroll);

	lb->visiblecolumns = lb->viewPort->rect.w / lb->xstep;
	
	if (lb->maxitemlen > lb->visiblecolumns) {
		lb->xpage = lb->visiblecolumns * lb->xstep;
	} else {
		lb->xpage = 0;
	}

	if ((lb->focusitem + 1) > lb->visibleitems) {
		lb->offy = (lb->focusitem - lb->visibleitems + 1) * lb->ystep;
	}

	setScrollBarRange(lb->vsBar, 1, lb->visibleitems - 1, 0, lb->itemcount - 1, lb->focusitem);
	setScrollBarRange(lb->hsBar, 1, lb->visiblecolumns, lb->visiblecolumns, lb->maxitemlen, lb->visiblecolumns);

}

int initListBox(LISTBOX *lb, SDL_Surface *surface)
{

	if (lb->viewPort->surface == NULL) {
		lb->viewPort->surface = SDL_CreateRGBSurface(SDL_SWSURFACE,
								lb->viewPort->rect.w, lb->viewPort->rect.h,
								surface->format->BitsPerPixel,
								surface->format->Rmask, surface->format->Gmask,
								surface->format->Bmask, surface->format->Amask);
		if (lb->viewPort->surface == NULL) {
			printf("Cannot create ViewPort_Surface\n");
			return 0;
		}
		lb->backcolor = SDL_MapRGB(lb->viewPort->surface->format, 204, 204, 204);
		lb->focuscolor = SDL_MapRGB(lb->viewPort->surface->format, 255, 255, 255);
	}

	setListBoxSizes(lb);

	return 1;
}

int insideListBox(Sint16 x, Sint16 y, SDL_Rect *r)
{
	if ((x >= r->x) && (x < r->x + r->w)) {
		if ((y >= r->y) && (y < r->y + r->h)) {
			return 1;
		}
	}

	return 0;
}

int itemdoubleClicked(SDL_Rect *r)
{
	SDL_Event event;
	int released = 0;
	int done = 0;
	Uint32 t = SDL_GetTicks();
	
	while (!done) {
		if (SDL_PollEvent(&event)) {
			switch (event.type) {
				case SDL_MOUSEBUTTONUP:
				case SDL_MOUSEBUTTONDOWN:
					if ((event.button.state == SDL_RELEASED) && (event.button.button == SDL_BUTTON_LEFT)) {
						if (insideListBox(event.button.x, event.button.y, r)) {
							released = 1;
							done = 1;
						}
					}
					break;
				default:
					break;
			}
		} else if (SDL_GetTicks() > (t + 200)) {
			done = 1;
		}
	}

	return released;
}

void addKeyPressEvent(int key, int mod)
{
	SDL_Event e;
	e.type = SDL_KEYDOWN;
	e.key.type = SDL_KEYDOWN;
	e.key.state = SDL_PRESSED;
	e.key.keysym.sym = key;
	e.key.keysym.mod = mod;
	SDL_PeepEvents(&e, 1, SDL_ADDEVENT, SDL_KEYDOWNMASK);
}

int handleListBoxEvent(SDL_Event *event, LISTBOX *lb, SDL_Surface *surface)
{
	int update = 0;
	int handled;

	handled = handleScrollBarEvent(event, lb->vsBar, surface);
	if (!handled)
		handled = handleScrollBarEvent(event, lb->hsBar, surface);
	if (handled) {
		int x, y;
		switch (handled) {
			case SCROLL_UP:{
				SDL_Event e;
				while (SDL_GetMouseState(&x,&y) > 0) {
					if (lb->focusitem > 0) {
						if (lb->focusitem > (lb->offy / lb->ystep)) {
							lb->focusitem--;
						} else if(lb->offy > 0) {
							lb->offy -= lb->ystep;
							lb->focusitem--;
						}
					updateScrollBarSlider(lb->vsBar, surface, lb->focusitem);
					updateViewPort(lb, surface);
			    	SDL_UpdateRect(surface,lb->rect.x,lb->rect.y,lb->rect.w,lb->rect.h);
					SDL_Delay(100);
					}
					SDL_PollEvent(&e);
				}
				}
				break;
			case SCROLL_DOWN: {
				SDL_Event e;
				while (SDL_GetMouseState(&x,&y) > 0) {
					if (lb->focusitem < lb->itemcount - 1) {
						if(lb->focusitem < (lb->visibleitems - 1 + lb->offy / lb->ystep)) {
							lb->focusitem++;
						} else if(((lb->offy / lb->ystep) + lb->visibleitems) < lb->itemcount) {
							lb->offy += lb->ystep;
							lb->focusitem++;
						}
					updateScrollBarSlider(lb->vsBar, surface, lb->focusitem);
					updateViewPort(lb, surface);
			    	SDL_UpdateRect(surface,lb->rect.x,lb->rect.y,lb->rect.w,lb->rect.h);
					SDL_Delay(100);
					}
					SDL_PollEvent(&e);
				}
				}
				break;
			case SCROLL_LEFT: {
				SDL_Event e;
				while (SDL_GetMouseState(&x,&y) > 0) {
					if (lb->offx < 0) {
						lb->offx += lb->xstep;
						updateScrollBarSlider(lb->hsBar, surface, -lb->offx/lb->xstep + lb->visiblecolumns);
						updateViewPort(lb, surface);
			    		SDL_UpdateRect(surface,lb->rect.x,lb->rect.y,lb->rect.w,lb->rect.h);
						SDL_Delay(100);
					}
					SDL_PollEvent(&e);
				}
				}
				break;
			case SCROLL_RIGHT: {
				SDL_Event e;
				while (SDL_GetMouseState(&x,&y) > 0) {
					if ((lb->offx + (lb->maxitemlen * lb->xstep)) > lb->viewPort->rect.w) {
						lb->offx -= lb->xstep;
						updateScrollBarSlider(lb->hsBar, surface, -lb->offx/lb->xstep + lb->visiblecolumns);
						updateViewPort(lb, surface);
			    		SDL_UpdateRect(surface,lb->rect.x,lb->rect.y,lb->rect.w,lb->rect.h);
						SDL_Delay(100);
					}
					SDL_PollEvent(&e);
				}
				}
				break;
			case SCROLL_PAGE_UP:
				addKeyPressEvent(SDLK_PAGEUP, 0);
				break;
			case SCROLL_PAGE_DOWN:
				addKeyPressEvent(SDLK_PAGEDOWN, 0);
				break;
			case SCROLL_PAGE_LEFT:
				addKeyPressEvent(SDLK_PAGEUP, KMOD_LCTRL);
				break;
			case SCROLL_PAGE_RIGHT:
				addKeyPressEvent(SDLK_PAGEDOWN, KMOD_LCTRL);
				break;
			case SCROLL_VERTICAL: {
				SDL_Event e;
				int newfocus;
				while (SDL_GetMouseState(&x,&y) > 0) {
					newfocus = getValueFromSlider(lb->vsBar, surface, x, y);
					if (newfocus != lb->focusitem) {
						lb->focusitem = newfocus;
						lb->offy = lb->focusitem * lb->ystep;
						if (lb->focusitem > (lb->itemcount - lb->visibleitems - 1))
							lb->offy = (lb->itemcount - lb->visibleitems) * lb->ystep;
						updateViewPort(lb, surface);
					}
			    	SDL_UpdateRect(surface,lb->rect.x,lb->rect.y,lb->rect.w,lb->rect.h);
					SDL_WaitEvent(&e);
				}
				}
			case SCROLL_HORIZONTAL: {
				SDL_Event e;
				int newoffx;
				while (SDL_GetMouseState(&x,&y) > 0) {
					newoffx = -lb->xstep * getValueFromSlider(lb->hsBar, surface, x, y);
					if (newoffx != lb->offx) {
						lb->offx = newoffx;
						updateViewPort(lb, surface);
					}
			    	SDL_UpdateRect(surface,lb->rect.x,lb->rect.y,lb->rect.w,lb->rect.h);
					SDL_WaitEvent(&e);
				}
				}
			default:
				break;
		}

		return 1;
	}

	handled = 1;
	switch (event->type) {
		case SDL_KEYDOWN:
			if (event->key.keysym.sym == SDLK_HOME) {
				if ((lb->offy > 0) || (lb->focusitem > 0)) {
					lb->focusitem = 0;
					lb->offy = 0;
					update = 1;
				}
			} else if (event->key.keysym.sym == SDLK_END) {
				if (lb->itemcount > lb->visibleitems) {
					lb->focusitem = lb->itemcount - 1;
					lb->offy = (lb->itemcount - lb->visibleitems) * lb->ystep;
					update = 1;
				} else if (lb->focusitem < (lb->itemcount - 1)) {
					lb->focusitem = lb->itemcount - 1;
					lb->offy = 0;
					update = 1;
				}
			} else if (event->key.keysym.sym == SDLK_UP) {
				if (lb->focusitem > (lb->offy / lb->ystep)) {
					lb->focusitem--;
					update = 1;
				} else if(lb->offy > 0) {
					lb->offy -= lb->ystep;
					lb->focusitem--;
					update = 1;
				}
			} else if (event->key.keysym.sym == SDLK_DOWN) {
				if (lb->itemcount > lb->visibleitems) {
					if(lb->focusitem < (lb->visibleitems - 1 + lb->offy / lb->ystep)) {
						lb->focusitem++;
						update = 1;
					}else if(((lb->offy / lb->ystep) + lb->visibleitems) < lb->itemcount) {
						lb->offy += lb->ystep;
						lb->focusitem++;
						update = 1;
					}
				} else if (lb->focusitem < (lb->itemcount - 1)) {
					lb->focusitem++;
					lb->offy = 0;
					update = 1;
				}
			} else if (event->key.keysym.sym == SDLK_PAGEUP) {
				if (event->key.keysym.mod == KMOD_LCTRL) {
					lb->offx += lb->xpage;
					if (lb->offx > 0)
						lb->offx = 0;
					update = 1;
				} else {
					if (lb->itemcount > lb->visibleitems) {
						lb->focusitem -= lb->visibleitems - 1;
						if (lb->focusitem < 0)
							lb->focusitem = 0;
						lb->offy -= lb->viewPort->rect.h - lb->ystep;
						if (lb->offy < 0)
							lb->offy = 0;
						update = 1;
					} else if (lb->focusitem > 0) {
						lb->focusitem = 0;
						lb->offy = 0;
						update = 1;
					}
				}
			} else if (event->key.keysym.sym == SDLK_PAGEDOWN) {
				if (event->key.keysym.mod == KMOD_LCTRL) {
					lb->offx -= lb->xpage;
					if (lb->offx < -(lb->maxitemlen * lb->xstep - lb->xpage))
						lb->offx = -(lb->maxitemlen * lb->xstep - lb->xpage);
					update = 1;
				} else {
					if (lb->itemcount > lb->visibleitems) {
						lb->focusitem += lb->visibleitems - 1;
						if (lb->focusitem > (lb->itemcount - 1))
							lb->focusitem = lb->itemcount - 1;
						lb->offy += lb->viewPort->rect.h - lb->ystep;
						if (lb->offy > ((lb->itemcount - lb->visibleitems) * lb->ystep))
							lb->offy = (lb->itemcount - lb->visibleitems) * lb->ystep;
						update = 1;
					} else if (lb->focusitem < (lb->itemcount - 1)) {
						lb->focusitem = lb->itemcount - 1;
						lb->offy = 0;
						update = 1;
					}
				}
			} else if (event->key.keysym.sym == SDLK_LEFT) {
				if (lb->offx < 0) {
					lb->offx += lb->xstep;
					update = 1;
				}
			} else if (event->key.keysym.sym == SDLK_RIGHT) {
				if ((lb->offx + (lb->maxitemlen * lb->xstep)) > lb->viewPort->rect.w) {
					lb->offx -= lb->xstep;
					update = 1;
				}
			} else {
				handled = 0;
			}
			break;
		case SDL_MOUSEBUTTONUP:
		case SDL_MOUSEBUTTONDOWN:
			if ((event->button.state == SDL_PRESSED) && (event->button.button == SDL_BUTTON_LEFT)) {
				if (insideListBox(event->button.x, event->button.y, &lb->viewPort->rect)) {
					int newfocus = (event->button.y - lb->viewPort->rect.y + lb->offy) / lb->ystep;
					if ((newfocus < lb->itemcount) && (newfocus != lb->focusitem)) {
						lb->focusitem = newfocus;
						update = 1;
					}
				} else {
					handled = 0;
				}
			} else if ((event->button.state == SDL_RELEASED) && (event->button.button == SDL_BUTTON_LEFT)) {
				if (insideListBox(event->button.x, event->button.y, &lb->viewPort->rect)) {
					int newfocus = (event->button.y - lb->viewPort->rect.y + lb->offy) / lb->ystep;
					if (newfocus < lb->itemcount) {
						if (itemdoubleClicked(&lb->viewPort->rect) && lb->showFocus) {
							addKeyPressEvent(SDLK_RETURN, 0);
						} else {
							handled = 0;
						}
					} else {
						handled = 0;
					}
				} else {
					handled = 0;
				}
			}
			break;
		default:
			handled = 0;
			break;
	}

	if (update) {
		update = 0;
		updateViewPort(lb, surface);
		updateScrollBarSlider(lb->hsBar, surface, -lb->offx/lb->xstep + lb->visiblecolumns);
		updateScrollBarSlider(lb->vsBar, surface, lb->focusitem);
	    SDL_UpdateRect(surface,lb->rect.x,lb->rect.y,lb->rect.w,lb->rect.h);
	}

	return handled;
}

int ListBoxFocusItem(LISTBOX *lb)
{
	return lb->focusitem;
}

void setListBoxFocusItem(LISTBOX *lb, int focus)
{
	lb->focusitem = focus;
}

void cleanupListBox(LISTBOX *lb)
{
	deleteitems(lb);
	lb->offx = 0;
	lb->offy = 0;
	lb->focusitem = 0;
	lb->maxitemlen = 0;
}
