/*
 * Copyright (C) 2002,2003 Daniel Heck
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: sound.cc,v 1.18 2003/07/07 21:14:05 reallysoft Exp $
 */
#include "enigma.hh"
#include "options.hh"
#include "oxyd.hh"
#include "sound.hh"
#include "system.hh"
#include "SDL.h"
#include "SDL_mixer.h"
#include "px/dict.hh"
#include "px/tools.hh"
#include <string>
#include <iostream>
#include <cassert>

using namespace px;
using namespace std;

namespace
{
    bool sound_enabled      = true;
    bool sound_enabled_temp = false;
    bool music_enabled      = true;
    bool sound_initialized  = false;

    int    sound_freq     = MIX_DEFAULT_FREQUENCY;
    Uint16 sound_format   = MIX_DEFAULT_FORMAT;
    int    sound_channels = MIX_DEFAULT_CHANNELS;

    Mix_Music *current_music = 0;
    string     current_music_name;

    /* Coordinates of the player in the world.  This is used as the
       reference point for stereo separation.  */
    V2 listener_pos;

    px::Dict<Mix_Chunk*> wav_cache;
}

void
sound::Init()
{
    // Do nothing if user doesn't want sound or mixer lib is already
    // initialized.
    if (!sound_enabled || sound_initialized)
        return;

    // Initialize SDL_mixer lib
    if (Mix_OpenAudio(sound_freq, sound_format, sound_channels, 1024) < 0) {
        fprintf(stderr, "Couldn't open mixer: %s\n", Mix_GetError());
        sound_enabled = false;
        return;
    }
    UpdateVolume();
    sound_initialized = true;
}

void
sound::ClearSoundCache()
{
    for (px::Dict<Mix_Chunk*>::iterator it = wav_cache.begin(); it != wav_cache.end(); ++it)
	Mix_FreeChunk(it->second);
    wav_cache.clear();
}

void
sound::UpdateVolume()
{
    if (!sound_initialized)
        return;                 // SDL_mixer crashes without this check

    int soundvol = int(options::SoundVolume * MIX_MAX_VOLUME);
    Mix_Volume (-1, Clamp (soundvol, 0, MIX_MAX_VOLUME));
    int musicvol = int(options::MusicVolume * MIX_MAX_VOLUME);
    Mix_VolumeMusic (Clamp (musicvol, 0, MIX_MAX_VOLUME));
}

void
sound::Shutdown()
{
    if (sound_initialized) {
	ClearSoundCache();
	Mix_FreeMusic(current_music);
        Mix_CloseAudio();
        sound_initialized = false;
    }
}

void
sound::DisableSound()
{
    if (sound_enabled) {
        sound_enabled = false;
        Shutdown();
    }
}

void
sound::EnableSound()
{
    if (!sound_enabled) {
        sound_enabled = true;
        Init();
    }
}

void
sound::TempDisableSound() {
    sound_enabled_temp = sound_enabled;
    sound_enabled      = false;
}
void
sound::TempReEnableSound() {
    sound_enabled = sound_enabled_temp;
}


void
sound::DisableMusic()
{
    music_enabled = false;
}

static Mix_Chunk *
cache_sound(const std::string &name)
{
    px::Dict<Mix_Chunk*>::iterator i=wav_cache.find(name);
    if (i == wav_cache.end()) {
        string filename = enigma::FindDataFile("sound/" + name + ".wav");
        Mix_Chunk *ch = 0;

        ch = enigma::oxyd::LoadSound(name);
        if (ch == 0)
            ch = Mix_LoadWAV(filename.c_str());

        if (ch != 0)
            wav_cache.insert(name, ch);
        else
	    enigma::Log << "Couldn't load sample: " << Mix_GetError() << endl;
        return ch;
    } else
        return i->second;
}



void
sound::SetListenerPosition (const px::V2 &pos)
{
    listener_pos = pos;
}

void
sound::PlaySound (const char *name, const px::V2 &pos)
{
    // if distance sound origin <= this value, play at full volume
    const double fullvol_range = 0.2;

    if (!sound_enabled)
        return;

    V2 distv = pos-listener_pos;
    double dist = max(0.0, length(distv) - fullvol_range);

    int xdist = int(distv[0] * options::StereoSeparation);
    int left  = px::Clamp (255 - xdist, 0, 255);
    int right = px::Clamp (255 + xdist, 0, 255);

    double range = 50;         // How far can sound travel?
    double volume = 1 - dist/range;

    if (Mix_Chunk *chunk = cache_sound(name)) {
	int channel = -1; //Mix_GroupOldest(-1);
	int mixvol = int(volume * options::SoundVolume * MIX_MAX_VOLUME);

        channel = Mix_PlayChannel(channel, chunk, 0);
        Mix_SetPanning(channel, left, right);
	Mix_Volume(channel, px::Clamp(mixvol, 0, MIX_MAX_VOLUME));
    }
}

void
sound::PlaySound (const char *name)
{
    PlaySound (name, listener_pos);
}

void
sound::FadeoutMusic()
{
    while (Mix_FadingMusic() != MIX_NO_FADING)
        SDL_Delay(10);

    if (Mix_PlayingMusic()) {
        Mix_FadeOutMusic(500);
        SDL_Delay(400);
    }
    while (Mix_PlayingMusic())
        SDL_Delay(10);
}

void
sound::PlayMusic(const char *name)
{
    if (!sound_enabled || !music_enabled || name==current_music_name)
        return;

    FadeoutMusic();

    string fname = enigma::FindDataFile(name);

    if (current_music)
	Mix_FreeMusic(current_music);

    current_music = Mix_LoadMUS(fname.c_str());

    if (current_music) {
        Mix_PlayMusic(current_music,-1);
	UpdateVolume();
        current_music_name = name;
    }
    else
        current_music_name = "";
}

void
sound::StopMusic()
{
    Mix_HaltMusic();
    Mix_FreeMusic(current_music);
    current_music = 0;
    current_music_name = "";
}

void
sound::StopMusic(const char *name)
{
    if (name==current_music_name)
        StopMusic();
}

/* SDL_ConvertAudio is not very good at audio resampling since it is
   only capable of changing the sound frequency by integer powers of 2
   (i.e., by a factor of ... 1/4 1/2 1 2 ...).  The sound files used
   by Oxyd are sampled at 6kHz which we must convert to roughly 22kHz.
   This function resamples between any two frequencies using simple
   linear interpolation.  It is not capable of changing the sample
   format or dealing with more than one channel.
*/
namespace
{
    Sint8 *resample (const Sint8 *data, Uint32 len, int oldfreq, int newfreq, Uint32 *newlen)
    {
        assert (data);
        assert (len>0);
        assert (oldfreq > 0);
        assert (newfreq > 0);
        const int sample_size = 1;    //  8bit sample data

        double ratio = double(oldfreq)/newfreq;
        *newlen = int(len/ratio + sample_size);
        Sint8 *newdata = (Sint8*) malloc (sample_size * (*newlen));
        if (!newdata)
            return 0;

        const Sint8 *src = data;
        Sint8 *dst = newdata;

        float srcpos = 0;
        for (unsigned i=0; i<*newlen; ++i) {
            int srcidx = static_cast<int>(floor(srcpos));
            float a2 = srcpos - srcidx;
            float a1 = 1.0f - a2;
            dst[i] = static_cast<Sint8> ((a1*src[srcidx] + a2*src[srcidx+1])/2);
            srcpos += ratio;
        }

        return newdata;
    }

#ifndef HAVE_MIX_QUICKLOAD

#endif

}




Mix_Chunk *
sound::ChunkFromRaw (const Uint8 *buf, Uint32 len,
                     int sfreq, int sformat, int schannels)
{
    if (!sound_initialized || !buf)
        return 0;

    // get destination format
    int dfreq, dchannels;
    Uint16 dformat;
    Mix_QuerySpec (&dfreq, &dformat, &dchannels);
    enigma::Log << "converting: (" << sfreq <<","<<schannels
                << ") to (" << dfreq <<","<<dchannels << endl;

    // resample
    Uint32 newlen=0;
    Uint8 *newbuf = reinterpret_cast<Uint8*>(resample(reinterpret_cast<Sint8*>(const_cast<Uint8*>(buf)),
                                                      len, sfreq, dfreq, &newlen));


    // convert audio data
    SDL_AudioCVT cvt;
    if (!SDL_BuildAudioCVT (&cvt, sformat, schannels, dfreq,
                            dformat, dchannels, dfreq))
        return 0;

    cvt.buf = (Uint8*) malloc(newlen * cvt.len_mult);
    cvt.len = newlen;
    memcpy(cvt.buf, newbuf, newlen);
    free(newbuf);

    SDL_ConvertAudio(&cvt);

    Mix_Chunk *chunk = Mix_QuickLoad_RAW(cvt.buf, cvt.len_cvt);
    chunk->allocated = 1;
    return chunk;
}

