/*
 * Copyright (C) 2002 Daniel Heck
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */
#ifndef OPTIONS_HH
#define OPTIONS_HH

#include <ctime>
#include <string>

namespace options
{
    using std::string;

    extern int    BitsPerPixel;
    extern double MouseSpeed;
    extern bool   WizardMode;
    extern bool   FullScreen;
    extern bool   ShowFPS;
    extern bool   UseAlpha;
    extern bool   Nograb;
    extern bool   Nozoom;
    extern bool   InGameMusic;
    extern double SoundVolume;
    extern double MusicVolume;
    extern double StereoSeparation;
    extern string MenuMusicFile;
    extern string LevelMusicFile;
    extern int    SoundSet;
    extern bool   SkipSolvedLevels;
    extern int    Difficulty;
    extern int    VideoMode;
    extern int    LevelMenuPosition;

    // True if any LevelStatus differs from last saved state.
    extern bool LevelStatusChanged;

    // An option was changed that will not take effect until Enigma is
    // restarted.
    extern bool MustRestart;

    // An option was changed that makes it necessary to restart the
    // current level (e.g. Difficulty changed while playing level).
    // When 'MustRestartLevel' is set to true, the current level
    // will restart w/o any further questions.
    extern bool MustRestartLevel;

    struct LevelStatus {
        LevelStatus(int easy=-1, int hard=-1, int fin=0, std::time_t solv = 0);
        bool operator != (const LevelStatus& other) const;

        // Variables.
        int  par_easy, par_hard; // Best time in seconds (-1: use level default)
        int  finished;          // User completed this level?
                                // 0 = no,1=easy,2=hard,3=easy&hard
        std::time_t solved_at; // date when level was solved last time
    };

    // --------------------------------------------------------------------------------
    // Contants :

    const int minMouseSpeed = 1;
    const int maxMouseSpeed = 15;

    // --------------------------------------------------------------------------------

    /* Get the status of a particular level in a particular pack.
       Returns 0 if no record for this level exists. */
    LevelStatus *GetLevelStatus(const string &levelpack,
                                const string &levelname);

    /* Set the status of a particular level.  The previous entry (if
       available) for this level is discarded. */
    void SetLevelStatus(const string &levelpack,
                        const string &levelname,
                        const LevelStatus &stat);

    /* Returns true if time is new record for this level. */
    bool SetLevelTime(const string &pack, const string &level, int difficulty, int time);

    void SetLevelFinished(const string &pack, const string &level, int difficulty);

    /* Save the current set of options to the user's configuration
       file. (Either ~/.enigmarc on Unix systems or ./enigmarc.lua on
       Win32. */
    bool Save ();

    /* Try to load the user's configuration file.  Returns true if
       successful. */
    bool Load ();
}
#endif
