//======================================================================
// Copyright (C) 2002 Daniel Heck
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//  
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//  
// You should have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//======================================================================
#ifndef WORLD_HH_INCLUDED
#define WORLD_HH_INCLUDED

#include "enigma.hh"
#include "px/math.hh"
#include "tools.hh"

#include <string>
#include <iosfwd>

namespace world
{
    using enigma::GridPos;
    using enigma::Direction;

    class ActorInfo {
    public:
        ActorInfo(Actor *a, px::V3 p, px::V3 v);
        void dispose();

        Actor *actor;
        px::V3 pos, vel;
        px::V3 oldpos, oldvel;
        std::vector<Stone*> stone_contacts;
        bool grabbed;
    };

    enum StoneResponse {
        STONE_PASS,
        STONE_REBOUND
    };

    struct StoneContact {
        StoneContact(const ActorInfo *a, Stone *s, GridPos p, 
                     const px::V3 &cp, const px::V3 &n)
            : actorinfo(a), st(s), stpos(p), cpoint(cp), normal(n)
        {}
        
        const ActorInfo *actorinfo;
        Stone           *st;
        GridPos         stpos;
        px::V3          cpoint; // contact point
        px::V3          normal; // face normal at contact point
        StoneResponse   stresponse;
    };

    using std::string;
    typedef const std::string cstring;
}

//----------------------------------------------------------------------
// Global variables
//----------------------------------------------------------------------
namespace world
{
    extern tools::Timer g_timer;
}

namespace world
{
    void SetMouseForce(const px::V3 &f);

    void NameObject(Object *obj, const std::string &name);
    Object *GetNamedObject(const std::string &name);

//----------------------------------------
// Actor access
//----------------------------------------
    void SetActor(double x, double y, Actor* a);
    void GrabActor(Actor *a);
    void ReleaseActor(Actor *a);
    const ActorInfo &GetActorInfo(Actor *a);

//----------------------------------------
// Stone access
//----------------------------------------
    void    SetStone(GridPos p, Stone* st);
    Stone   *GetStone(GridPos p);
    Stone   *YieldStone(GridPos p);
    void    KillStone(GridPos p);

    void MaybeMoveStone(GridPos p, Direction dir);
    void SwapStones(GridPos p, GridPos newp);

//----------------------------------------
// Item access
//----------------------------------------
    void    SetItem(GridPos p, Item* st);
    void    KillItem(GridPos p);

    // Get a pointer to the item at (x,y).
    Item*   GetItem(GridPos p);
    
    // Remove item from the world and yield it to the caller.  This is
    // used for moving items to an inventory.
    Item*   YieldItem(GridPos p);

//----------------------------------------
// Floor access
//----------------------------------------
    void    SetFloor(GridPos p, Floor* st);
    Floor*  GetFloor(GridPos p);
    void    KillFloor(GridPos p);

    void Shutdown();

    // Create a new, empty world.
    void Create(int w, int h);
    void Load(const std::string &name);
    void Tick(double dtime);
}
#endif
