//======================================================================
// Copyright (C) 2002 Daniel Heck
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//  
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//  
// You should have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//======================================================================
#ifndef TOOLS_HH_INCLUDED
#define TOOLS_HH_INCLUDED

#include "px/pxfwd.hh"
#include "px/tools.hh"
#include <iosfwd>
#include <string>
#include <list>
#include <stddef.h>

namespace tools
{
    class TypeError {};

    class Value {
    public:
        enum Type { NIL, DOUBLE, STRING, VECTOR3 };

        Value() : type(NIL) {}
        Value(double d) : type(DOUBLE) { val.dval = d; }
        Value(const char* str);
        ~Value() { clear(); }

        Value(const Value& v);
        Value& operator=(const Value& v);

        void assign(double d) { clear(); type=DOUBLE; val.dval=d; }
        void assign(const char* s);
    
        Type    get_type() const { return type; }
        double  get_double() const throw(TypeError);
        const char* get_string() const throw(TypeError);
    private:
        void clear();

        Type type;
        union {
            double dval;
            char* str;
        } val;
    };

    px::Buffer& operator<<(px::Buffer& buf, const Value& val);
    px::Buffer& operator>>(px::Buffer& buf, Value& val);
    
    std::ostream& operator<<(std::ostream& os, const Value& val);

    bool to_bool(const Value &v);
    int to_int(const Value &v);
    double to_double(const Value &v);
    const char *to_string(const Value &v);

     // Itor

    template <class T> 
    class Itor {
    public:
        virtual ~Itor() {}
        virtual T* first() = 0;
        virtual T* next() = 0;
    };

     // ValueMap

    class UnboundName {};
    
    class ValueMap : public px::Nocopy {
    public:
        ValueMap();
        ~ValueMap();

        void assign(const std::string& name, const Value& val);
        const Value& value(const std::string& name) const;
        bool boundp(const std::string& name) const;

        typedef Itor<const std::string> Itor;
        Itor* itor() const;
    private:
        class ValueMap_Impl* pimpl;
    };
}

namespace tools
{
    class TimeHandler {
    public:
        virtual ~TimeHandler() {}
        virtual void tick(double dtime) {}
        virtual void alarm() {}
    };

    class Timer {
    public:
        void deactivate(TimeHandler* th);
        void set_alarm(TimeHandler* th, double interval, bool repeatp = false);
        void tick(double dtime);
    private:
        class Alarm {
            TimeHandler* handler;
            double interval;
            double timeleft;
            bool repeatp;
        public:
            Alarm(TimeHandler* h, double i, bool r);
            void tick(double dtime);
            bool expired() const { return timeleft <= 0; }
        };

        std::list<TimeHandler*>  handlers;
        std::list<Alarm>         alarms;
    };
}

#endif
