//======================================================================
// Copyright (C) 2002 Daniel Heck
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//  
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//  
// You should have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//======================================================================
#include "enigma.hh"
#include "sound.hh"
#include <SDL/SDL_mixer.h>
#include <string>
#include <map>
#include <iostream>

using namespace std;

namespace
{
    bool sound_enabled = true;
    bool sound_initialized = false;

    int sound_freq = MIX_DEFAULT_FREQUENCY;
    Uint16 sound_format = MIX_DEFAULT_FORMAT;
    int sound_channels = MIX_DEFAULT_CHANNELS;

    map<string, Mix_Chunk*> wav_cache;
}

void sound::Init()
{
    // Do nothing if user doesn't want sound or mixer lib is already
    // initialized.
    if (!sound_enabled || sound_initialized)
        return;

    // Initialize SDL_mixer lib
    if (Mix_OpenAudio(sound_freq, sound_format, sound_channels, 512) < 0) {
        cerr << "Couldn't open mixer: " << Mix_GetError() << endl;
        sound_enabled = false;
        return;
    }
    sound_initialized = true;
}

void sound::Shutdown()
{
    if (sound_initialized) {
        Mix_CloseAudio();
    }
    sound_initialized = false;
}

void sound::DisableSound() {sound_enabled = false;}
void sound::EnableSound() {sound_enabled = true;}

Mix_Chunk *
cache_sound(const std::string &name)
{
    map<string, Mix_Chunk*>::iterator i=wav_cache.find(name);
    if (i == wav_cache.end()) {
        string filename = enigma::FindDataFile("sound/" + name + ".wav");
        Mix_Chunk *ch = Mix_LoadWAV(filename.c_str());
        if (ch != 0)
            wav_cache[name] = ch;
        else
            cerr << "Couldn't load sample: "<<Mix_GetError()<<endl;
        return ch;
    } else
        return i->second;
}

void 
sound::PlaySound(const std::string &name)
{
    if (!sound_enabled)
        return;
    if (Mix_Chunk *ch = cache_sound(name)) {
        Mix_PlayChannel(-1, ch, 0);
    }
}

namespace
{
    Mix_Music *current_music = 0;
}

void
sound::PlayMusic(const std::string &name)
{
    if (!sound_enabled)
        return;

    StopMusic();

//     string fname = enigma::FindDataFile(name);
//     current_music = Mix_LoadMUS(fname.c_str());
//     if (current_music)
//         Mix_PlayMusic(current_music,-1);
}

void 
sound::StopMusic()
{
    Mix_HaltMusic();
    Mix_FreeMusic(current_music);
    current_music = 0;
}
