//======================================================================
// Copyright (C) 2002 Daniel Heck
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//  
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//  
// You should have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//======================================================================
#ifndef VIDEO_HH_INCLUDED
#define VIDEO_HH_INCLUDED

#include "pxfwd.hh"
#include "drawable.hh"
#include <SDL/SDL.h>

//----------------------------------------
// Colors
//----------------------------------------
namespace px
{
    struct RGBA_Mask {
        RGBA_Mask(Uint32 rr=0, Uint32 gg=0, Uint32 bb=0, Uint32 aa=0)
            : r(rr), g(gg), b(bb), a(aa)
        {}
        Uint32 r,g,b,a;
    };
    struct RGB {
        RGB(char rr=0, char gg=0, char bb=0)
            : r(rr), g(gg), b(bb) 
        {}
        char r,g,b;
    };
    struct RGBA {
        RGBA(char rr=0, char gg=0, char bb=0, char aa=0)
            : r(rr), g(gg), b(bb), a(aa)
        {}
        char r,g,b,a;
    };
}

//----------------------------------------
// Surface
//----------------------------------------
namespace px
{
    class Surface : public Drawable {
    public:
	virtual ~Surface();

	// Lowlevel framebuffer access.
	void* scanline_pointer(int y);
	void* pixel_pointer(int x, int y);
	void set_color_key(Uint32 color);
        void set_alpha(int a);

	// Drawable interface.
	void lock();
	void unlock();
        Uint32 map_color(int r, int g, int b, int a);
        Uint32 map_color(int r, int g, int b);
        void box(int x, int y, int w, int h, Uint32 color);
        void blit(int x, int y, Surface* src, const Rect& sr);
        void blit(int x, int y, Surface* src);
        Rect size() const;

	// Accessors.
	inline int bypp() const { return m_surface->format->BytesPerPixel; }
	inline int bipp() const { return m_surface->format->BitsPerPixel; }
	inline Uint32 pitch() const { return m_pitch; }
	inline int width() const { return m_surface->w; }
	inline int height() const { return m_surface->h; }

	SDL_Surface* get_surface() const { return m_surface; }

    protected:
	// Constructors.
	explicit Surface(SDL_Surface* sfc = NULL);
	Surface(int width, int height, int bipp, const RGBA_Mask &mask);
	Surface(void* data, int w, int h, int bipp, int pitch, 
                const RGBA_Mask &mask);
	
	// SDL surface access.
	void set_surface(SDL_Surface* sfc);
	
	friend Surface* DisplayFormat(Surface*);
	
    private:
	SDL_Surface*	m_surface;
	void**		m_scanline_ptr;
	Uint32		m_pitch;

        void init_scanlines();
	
	Surface(const Surface&);
	Surface& operator=(const Surface&);
    };

    inline void* Surface::scanline_pointer(int y) 
    { 
        if (!m_scanline_ptr)
            init_scanlines();
        return m_scanline_ptr[y]; 
    }
    
    inline void* Surface::pixel_pointer(int x, int y) 
    {
        return static_cast<Uint8*>(scanline_pointer(y)) + x*bypp(); 
    }
}    

//----------------------------------------
// TSurface
//----------------------------------------
namespace px
{
    template <class PIXELTYPE, int BIPP>
    class TSurface : virtual public Surface {
    public:
	// Constructors.
	explicit TSurface(SDL_Surface* sfc) : Surface(sfc) {}
	TSurface(int w, int h, const RGBA_Mask &mask = RGBA_Mask())
	    : Surface(w, h, BIPP, mask)
        {}

	// Create a surface from data that is already somewhere in memory.
	TSurface(void* data, int w, int h, int pitch, const RGBA_Mask &mask)
	    : Surface(data, w, h, BIPP, mask)
        {}

        PIXELTYPE* scanline_pointer(int y) 
        { 
            return static_cast<PIXELTYPE*>(Surface::scanline_pointer(y)); 
        }
        
        PIXELTYPE* pixel_pointer(int x, int y) 
        { 
            return (scanline_pointer(y) + x); 
        }

        Uint32 get_pixel(int x, int y)
        {
            return *pixel_pointer(x, y);
        }

	// Drawable interface.
	void set_pixel(int x, int y, Uint32 color);
	void set_pixels(int n, int* xlist, int* ylist, Uint32 color);
	void hline(int x, int y, int w, Uint32 color);
	void vline(int x, int y, int h, Uint32 color);
    protected:
	TSurface() {}
    private:
	TSurface(const TSurface<PIXELTYPE,BIPP>&);
	TSurface<PIXELTYPE,BIPP>& operator=(const TSurface<PIXELTYPE,BIPP>&);
    };

    typedef TSurface<Uint16, 16> Surface16;
    typedef TSurface<Uint32, 32> Surface32;
    typedef TSurface<Uint8, 8> Surface8;

    // 24bit surfaces are inherently slow, so don't use them for
    // things that need to be done quickly!
    class Surface24 : public Surface {
    public:
	// Constructors.
	explicit Surface24(SDL_Surface* sfc) : Surface(sfc) {}
	Surface24(int width, int height, const RGBA_Mask &mask)
	    : Surface(width, height, 24, mask)
        {}
	Surface24(void* data, int w, int h, int pitch, const RGBA_Mask &mask)
	    : Surface(data, w, h, 24, pitch, mask)
        {}

	// Drawable interface.
        Uint32 get_pixel(int x, int y);
        void set_pixel(int x, int y, Uint32 color);
    };
}

//----------------------------------------
// Screen
//----------------------------------------
namespace px
{
    class Screen : virtual public Surface {
    public:
	Screen(int bipp);
        ~Screen();
	void update_all();
	void update_rect(const Rect& r);
	void flush_updates();
	void set_caption(const char* str);
	void open(int w, int h);
    private:
	RectList& m_dirtyrects;
	int       m_bipp;

	Screen(const Screen&);
	Screen& operator=(const Screen&);
    };


    template <class PIXELTYPE, int BIPP>
    class TScreen : public TSurface<PIXELTYPE, BIPP>, public Screen {
    public:
	TScreen() : Screen(BIPP) {}
	TScreen(SDL_Surface* s) : Screen(BIPP) {set_surface(s);}
    };

    typedef TScreen<Uint16, 16> Screen16;
    typedef TScreen<Uint32, 32> Screen32;
    typedef TScreen<Uint8, 8> Screen8;
}

//----------------------------------------
// Functions
//----------------------------------------
namespace px
{
    Screen* CreateScreen(SDL_Surface *);
    Screen* DisplayFormat(Screen* s);

    Surface *Duplicate(Surface *s);
    Surface* CreateSurface(SDL_Surface*);
    Surface* DisplayFormat(Surface* s);

    Surface* LoadImage(const char* filename);
}

#endif /* !VIDEO_HH_INCLUDED */
