//======================================================================
// Copyright (C) 2002 Daniel Heck
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//  
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//  
// You should have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//======================================================================
#include "video.hh"
#include "error.hh"
#include "geom.hh"
#include "tools.hh"

#include <SDL/SDL_image.h>
#include <SDL/SDL_syswm.h>

#include <cassert>
#include <string>

using namespace std;
using namespace px;

 // Surface implementation

Surface::Surface(SDL_Surface* sfc)
    : m_surface(0)
    , m_scanline_ptr(0)
    , m_pitch(0)
{
    if (sfc != 0)
	set_surface(sfc);
}

// Create a new empty surface with the desired dimension and surface
// format.
Surface::Surface(int w, int h, int bipp, const RGBA_Mask &mask)
    : m_surface(0)
    , m_scanline_ptr(0)
    , m_pitch(0)
{
    SDL_Surface* sfc;
    sfc = SDL_CreateRGBSurface(SDL_SWSURFACE, w, h, bipp,
                               mask.r, mask.g, mask.g, mask.a);
    if (sfc == 0)
        throw XVideo(string("Couldn't create surface: ") + SDL_GetError());
    set_surface(sfc);
}

/* Create a surface from image data that is already somewhere in
   memory.  */
Surface::Surface(void* data, int w, int h, int bipp, int pitch,
                 const RGBA_Mask &mask)
    : m_surface	(0)
    , m_scanline_ptr(0)
    , m_pitch(0)
{
    SDL_Surface* sfc;
    sfc = SDL_CreateRGBSurfaceFrom(SDL_SWSURFACE, w, h, bipp,
                                   pitch, mask.r, mask.g, mask.b, mask.a);
    if (sfc == 0)
        throw XVideo(string("Couldn't create surface:") + SDL_GetError());
    set_surface(sfc);
}	

Uint32 Surface::map_color(int r, int g, int b)
{
    return SDL_MapRGB(m_surface->format, r, g, b);
}

Uint32 Surface::map_color(int r, int g, int b, int a)
{
    return SDL_MapRGBA(m_surface->format, r, g, b, a);
}

Rect Surface::size() const
{
    return Rect(0, 0, width(), height());
}

Surface::~Surface() 
{
    SDL_FreeSurface(m_surface);
    delete[] m_scanline_ptr;
}

void 
Surface::set_color_key(Uint32 color) 
{
    SDL_SetColorKey(m_surface, SDL_SRCCOLORKEY | SDL_RLEACCEL, color); 
}

void
Surface::set_alpha(int a)
{
    SDL_SetAlpha(get_surface(), SDL_SRCALPHA, a);
}

void 
Surface::set_surface(SDL_Surface* sfc)
{
    assert(sfc != 0);
    SDL_FreeSurface(m_surface);
    m_surface = sfc;

    delete[] m_scanline_ptr;
    m_scanline_ptr = 0;

    m_pitch = m_surface->pitch/bypp();
}

// Calculate new scanline pointers
void Surface::init_scanlines()
{
    m_scanline_ptr = new void*[m_surface->h];
    Uint8* p = static_cast<Uint8*>(m_surface->pixels);

    for (int i=0; i<m_surface->h; i++) {
        m_scanline_ptr[i] = p;
        p += m_surface->pitch;
    }
}


void Surface::lock() 
{
    assert(m_surface != 0);
    if (SDL_MUSTLOCK(m_surface))
        SDL_LockSurface(m_surface);
}

void Surface::unlock() 
{
    assert(m_surface != 0);
    if (SDL_MUSTLOCK(m_surface))
        SDL_UnlockSurface(m_surface);
}

void
Surface::box(int x, int y, int w, int h, Uint32 color)
{
    SDL_Rect dr;
    dr.x = x; dr.y = y; dr.w = w; dr.h = h;
    SDL_FillRect(m_surface, &dr, color);
}

void 
Surface::blit(int x, int y, Surface* src, const Rect& sr) 
{
    SDL_Rect r1;
    SDL_Rect r2;
    r1.x = sr.x; r1.y = sr.y;
    r1.w = sr.w; r1.h = sr.h;
    
    r2.x = x; r2.y = y;
    SDL_BlitSurface(src->m_surface, &r1, m_surface, &r2);
}

void 
Surface::blit(int x, int y, Surface* src) 
{
    SDL_Rect dr;
    dr.x = x; dr.y = y;
    SDL_BlitSurface(src->m_surface, 0, m_surface, &dr);
}

//----------------------------------------------------------------------
// Surface24 implementation.
//----------------------------------------------------------------------
void Surface24::set_pixel(int x, int y, Uint32 color) 
{
    Uint8* p = static_cast<Uint8*>(pixel_pointer(x, y));
    SDL_GetRGB(color, get_surface()->format, p,p+1,p+2);
}

Uint32 Surface24::get_pixel(int x, int y)
{
    Uint8* p = static_cast<Uint8*>(pixel_pointer(x, y));
    return SDL_MapRGB(get_surface()->format, p[0],p[1],p[2]);
}

 // Screen implementation.

// `Xlib.h' also defines a type named `Screen' so we have to specify
// the namespace explicitly and cannot simple use a using-declaration.

px::Screen::Screen(int bipp) : m_dirtyrects(*(new RectList)), m_bipp(bipp)
{}

px::Screen::~Screen()
{
    delete &m_dirtyrects;
}

void 
px::Screen::update_all() 
{
    SDL_UpdateRect(get_surface(), 0, 0, 0, 0); 
}
    
void 
px::Screen::update_rect(const Rect& r) 
{
    if (m_dirtyrects.size() < 40)
        m_dirtyrects.push_back(r);
}
    
void 
px::Screen::set_caption(const char* str) 
{
    SDL_WM_SetCaption(str, 0);
}
    
void 
px::Screen::open(int w, int h) 
{
    SDL_Surface* sfc = SDL_SetVideoMode(w, h, m_bipp, SDL_SWSURFACE);
    set_surface(sfc);
}
    
void 
px::Screen::flush_updates() 
{
    if (m_dirtyrects.empty()) return;
    if (m_dirtyrects.size() < 40) 
    {
        m_dirtyrects.intersect(Rect(0, 0, width(), height()));

        RectList::iterator i;
        for (i=m_dirtyrects.begin(); i != m_dirtyrects.end(); ++i)
            SDL_UpdateRect(get_surface(), i->x, i->y, i->w, i->h);
    } 
    else
        SDL_UpdateRect(get_surface(), 0, 0, 0, 0);	// update everything
    m_dirtyrects.clear();
}


//----------------------------------------------------------------------
// Template instantiations
//----------------------------------------------------------------------

// A macro to make the function definitions a lot more readable.
#define VOID_TSURFACE                           \
	template <class PIXELTYPE, int BIPP>    \
	void px::TSurface<PIXELTYPE, BIPP>

VOID_TSURFACE::set_pixel(int x, int y, Uint32 color) 
{
    *(pixel_pointer(x, y)) = color;
}

VOID_TSURFACE::set_pixels(int n, int* xlist, int* ylist, Uint32 color)
{
    int *xp = xlist, *yp = ylist;
    for (int i=n; i > 0; --i) 
    {
        int x = *xp++, y = *yp++;
        *(pixel_pointer(x, y)) = color;
    }
}

VOID_TSURFACE::hline(int x, int y, int w, Uint32 color) 
{
    PIXELTYPE* dst = pixel_pointer(x, y);
    for (; w > 0; --w)
        *dst++ = color;
}
    
VOID_TSURFACE::vline(int x, int y, int h, Uint32 color) 
{
    PIXELTYPE* dst = pixel_pointer(x, y);
    for (; h > 0; --h) 
    {
        *dst = color;
        dst += pitch();
    }
}
#undef VOID_TSURFACE


namespace px 
{
    template class TScreen<Uint8, 8>;
    template class TScreen<Uint16, 16>;
    template class TScreen<Uint32, 32>;
}

 // Functions

// Convert the surface to the native surface format.  A pointer to a
// new surface is returned; the old one must be deleted by hand if it
// is no longer needed. 

Surface*
px::DisplayFormat(Surface* s) 
{
    if (SDL_Surface* s2 = SDL_DisplayFormat(s->m_surface))
    {
        switch (s2->format->BitsPerPixel) {
        case 8: return new Surface8(s2);
        case 16: return new Surface16(s2);
        case 24: return new Surface24(s2);
        case 32: return new Surface32(s2);
        }
    }
    return 0;
}

px::Screen*
px::CreateScreen(SDL_Surface* s)
{
    if (s == 0) 
        return 0;
    switch (s->format->BitsPerPixel) {
    case 8:  return new Screen8 (s);
    case 16: return new Screen16(s);
    case 32: return new Screen32(s);
    default: return 0;
    }
}

Surface* 
px::CreateSurface(SDL_Surface* s)
{
    if (s == 0) 
        return 0;
    switch (s->format->BitsPerPixel) {
    case 8:  return new Surface8 (s);
    case 16: return new Surface16(s);
    case 32: return new Surface32(s);
    default: return 0;
    }
}

Surface *
px::Duplicate(Surface *s)
{
    if (s==0) return 0;
    SDL_Surface *sdls = s->get_surface();
    SDL_Surface *copy = SDL_ConvertSurface(sdls, sdls->format, SDL_SWSURFACE);
    return CreateSurface(copy);
}

Surface* 
px::LoadImage(const char* filename)
{
    if (SDL_Surface *tmp = IMG_Load(filename))
    {
        SDL_Surface* img;

        if (tmp->flags & SDL_SRCALPHA) {
            img = SDL_DisplayFormatAlpha(tmp);
        }
        else if (tmp->flags & SDL_SRCCOLORKEY) {
            SDL_SetColorKey(tmp, SDL_SRCCOLORKEY | SDL_RLEACCEL,
                            tmp->format->colorkey);
            img = SDL_DisplayFormat(tmp);
        }
        else
            img = SDL_DisplayFormat(tmp);
        if (img == 0)
            return CreateSurface(tmp);
        SDL_FreeSurface(tmp);
        return CreateSurface(img);
    }
    return 0;
}

#if 0

static void 
tint16_ck(SDL_Surface *s, SDL_Rect *r, int rtint, int gtint, int btint)
{
    int i, j;
    Uint8 r1, g1, b1;
    Uint8 *fb  = (Uint8*)s->pixels + r->y*s->pitch + r->x*2;

    for (j=r->h; j; --j) {
	Uint16 *p = (Uint16*) fb;
	for (i=r->w; i; --i, ++p) {
	    if (*p == s->format->colorkey)
		continue;
	    SDL_GetRGB(*p, s->format, &r1, &g1, &b1);
	    *p = SDL_MapRGB(s->format,
			    CLAMP(r1 + rtint, 0, 255),
			    CLAMP(g1 + gtint, 0, 255),
			    CLAMP(b1 + btint, 0, 255));
	}
	fb += s->pitch;
    }
}

static void 
tint16(SDL_Surface *s, SDL_Rect *r, int rtint, int gtint, int btint)
{
    int i, j;
    Uint8 r1, g1, b1;
    Uint8 *fb  = ((Uint8*)s->pixels + r->y*s->pitch) + r->x*2;

    for (j=r->h; j; --j) {
	Uint16 *p = (Uint16*) fb;
	for (i=r->w; i; --i, ++p) {
	    SDL_GetRGB(*p, s->format, &r1, &g1, &b1);
	    *p = SDL_MapRGB(s->format,
			    CLAMP(r1 + rtint, 0, 255),
			    CLAMP(g1 + gtint, 0, 255),
			    CLAMP(b1 + btint, 0, 255));
	}
	fb += s->pitch;
    }

}

void 
px::TintRegion(Surface *s, const Rect &r, const RGB &tintcolor);
{
    SDL_Rect r;

    if (rect == 0) {
	r.x = r.y = 0;
	r.w = s->w;
	r.h = s->h;
    } else
	r = *rect;

    switch (s->format->BytesPerPixel) {
    case 2:
	if (s->flags & SDL_SRCCOLORKEY)
	    tint16_ck(s, &r, rtint, gtint, btint);
	else
	    tint16(s, &r, rtint, gtint, btint);
	break;
    default:
	printf("tint_region: unsupported video format.\n");
    }
}
#endif
