//======================================================================
// Copyright (C) 2002 Daniel Heck
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//  
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//  
// You should have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//======================================================================
#ifndef PX_GEOM_HH
#define PX_GEOM_HH

#include <iosfwd>
#include <vector>

//----------------------------------------
// Point2 / Point3
//----------------------------------------
namespace px
{
    class Point2 {
    public:
        int x, y;
        Point2(int xx, int yy) : x(xx),y(yy) 
        {}
    };

    class Point3 {
    public:
        int x, y, z;
        Point3(int xx, int yy, int zz) : x(xx),y(yy),z(zz) 
        {}
    };
}

//----------------------------------------
// Rect
//----------------------------------------
namespace px
{
    class Rect {
    public:
        int x, y, w, h;
        Rect(int xx = 0, int yy = 0, int ww = 0, int hh = 0);
        void move(int xx, int yy) { x = xx; y = yy; }
        void resize(int ww, int hh) { w = ww; h = hh; }
        void assign(int xx, int yy, int ww, int hh);
        void intersect(const Rect& r);
        bool contains(int xx, int yy) const;
        bool overlaps(const Rect& r) const;
        bool empty() const { return !(w > 0 && h > 0); }
    };

    inline bool operator==(const Rect &r, const Rect &s)
    {
        return (r.x==s.x && r.y==s.y && r.w==s.w && r.h==s.h);
    }
    
    inline Rect::Rect(int xx, int yy, int ww, int hh)
        : x(xx), y(yy), w(ww), h(hh)
    {}
    
    inline void 
    Rect::assign(int xx, int yy, int ww, int hh) 
    {
        x = xx; y = yy;
        w = ww; h = hh;
    }

    inline bool 
    Rect::contains(int xx, int yy) const 
    {
        return (xx >= x && xx < x+w && yy >= y && yy<y+h);
    }

    // Center one rectangle inside another.
    inline Rect center(const Rect &outer, const Rect &inner)
    {
        return Rect(outer.x + (outer.w-inner.w)/2,
                    outer.y + (outer.h-inner.h)/2,
                    inner.w, inner.h);
    }

    Rect intersect(const Rect& r1, const Rect& r2);

    std::ostream& operator<<(std::ostream& os, const Rect& r);

    extern Rect EmptyRect;
}

//----------------------------------------
// RectList
//----------------------------------------
namespace px
{
    class RectList {
    public:
        typedef Rect value_type;
        typedef std::vector<Rect>::iterator iterator;
        typedef std::vector<Rect>::const_iterator const_iterator;

        RectList() {}
        RectList(const RectList&);
        RectList& operator= (const RectList&);

        iterator       begin()      { return m_rectangles.begin(); }
        const_iterator begin() const { return m_rectangles.begin(); }
        iterator       end()        { return m_rectangles.end(); }
        const_iterator end() const  { return m_rectangles.end(); }

        void clear()                { m_rectangles.clear(); }
        int  size() const           { return m_rectangles.size(); }
        bool empty() const          { return m_rectangles.empty(); }
        void push_back(const Rect& r);
        void intersect(const Rect& r2);
        void add(const Rect& r);
        void sub(const Rect& r2);
        void merge(const RectList& rl);

        void swap(RectList &rl);
    private:
        std::vector<Rect> m_rectangles;
    };
}

#endif
