//======================================================================
// Copyright (C) 2002 Daniel Heck
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//  
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//  
// You should have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//======================================================================
#ifndef PX_ARRAY2_HH
#define PX_ARRAY2_HH

#include <memory>

namespace px
{
    // exception safe implementation
    // Implemented with a little help from TC++PL.
    
    template <class T, class A = std::allocator<T> >
    struct Array2Base {
        A alloc;                // allocator
        T *first, *last;        // start/end of allocated space

        Array2Base(const A &a, typename A::size_type n)
            : alloc(a), first(alloc.allocate(n)), last(first+n)
        {}
        ~Array2Base() { 
            // allocate(0) return 0 on GCC 2.95 -- standard?
            if(first) 
                alloc.deallocate(first, last-first); 
        }
    };

    template <class T, class A=std::allocator<T> >
    class Array2 : private Array2Base<T,A> {
      A alloc;
      public:
        typedef T value_type;
        typedef T *iterator;
        typedef const T *const_iterator;
        typedef T &reference;
        typedef const T &const_reference;
        typedef typename A::size_type size_type;

        Array2(int ww, int hh, const T& val=T(), const A& a=A())
            : Array2Base<T,A>(a, ww*hh), w(ww), h(hh)
        {
            std::uninitialized_fill(first, last, val);
        }
        Array2(const Array2<T,A> &a)
            : Array2Base<T,A>(a.alloc, a.last-a.first)
        {
            std::uninitialized_copy(a.begin(), a.end(), first);
        }
        Array2<T,A> &operator=(Array2<T,A> a2); // call by value!

        // Destructor
        ~Array2() { destroy_elements(); }

        iterator begin() { return first; }
        iterator end() { return last; }
        const_iterator begin() const { return first; }
        const_iterator end() const { return last; }
        iterator row_begin(size_type y) { return first + y*w; }
        iterator row_end(size_type y) { return first + y*w + w; }
        const_iterator row_begin(size_type y) const { return first + y*w; }
        const_iterator row_end(size_type y) const { return first + y*w + w; }


        void swap(Array2<T,A> &a2);

        size_type width() const { return w; }
        size_type height()const { return h; }


        T&	 get(size_type x, size_type y) { return first[y*w+x]; }
        const T& get(size_type x, size_type y) const { return first[y*w+x]; }
        T& 	 operator()(size_type x, size_type y) { return get(x,y); }
        const T& operator()(size_type x, size_type y) const { return get(x,y); }
        void 	 set(size_type x, size_type y, const T& val) 
        { first[y*w+x]=val; }

        void clear() {
            destroy_elements();
            std::uninitialized_fill(first, last, val);
        }


      private:
        size_type w, h;

        void destroy_elements() {
            for (T* p=first; p!=last; ++p)
                p->~T(); 
        }
    };

    template <class T, class A>
    void Array2<T,A>::swap(Array2<T,A> &a2) 
    {
        std::swap(first, a2.first);
        std::swap(last, a2.last);
        std::swap(w, a2.w);
        std::swap(h, a2.h);
    }

    template <class T, class A>
    void swap(Array2<T,A> &a, Array2<T,A> &b)
    {
        a.swap(b);
    }

    template <class T, class A> Array2<T,A>& 
    Array2<T,A>::operator=(Array2<T,A> a2)
    {
        px::swap(*this, a2);
        return *this;
    }
}

#endif
