//======================================================================
// Copyright (C) 2002 Daniel Heck
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//  
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//  
// You should have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//======================================================================
#include "player.hh"
#include "tools.hh"
#include "sound.hh"
#include "px/tools.hh"
#include <algorithm>
#include <iostream>

using namespace player;
using namespace std;
using namespace tools;

using world::GridPos;


class PlayerInfo {
public:
    string name;
    Inventory inventory;
    Actor *main_actor;
};

typedef vector<PlayerInfo> PlayerList;


 // Local variables
namespace
{
    PlayerList players;
    int icurrent_player = 0;
    bool current_player_inhibit_pickup = false;
}

 // Inventory implementation

unsigned const Inventory::max_items = 12;

Inventory::Inventory()
{}

Inventory::~Inventory() 
{ 
    px::delete_sequence(items.begin(), items.end()); 
}

Item *
Inventory::get_item(int idx)
{
    return idx >= size() ? 0 : items[idx];
}

void 
Inventory::add_item(Item *i)
{
    items.insert(items.begin(), i);
    string mname = i->get_inventory_model();
    display::AddInventoryItem(mname);
}

void Inventory::rotate()
{
    if (!items.empty())
    {
        std::rotate(items.begin(), items.begin()+1, items.end());
        display::RotateInventory();
    }
}

void Inventory::activate_first()
{
    if (!items.empty())
    {
        Item *it = items[0];
        Actor *ac = players[icurrent_player].main_actor;
        const world::ActorInfo &ai = GetActorInfo(ac);

        GridPos p(ai.pos[0], ai.pos[1]);

        switch (it->activate(ac, p)) {
        case world::ITEM_DROP:
            // only drop if no item underneath and floor allows it ###
            if (world::GetItem(p) == 0) {
                display::RemoveInventoryItem(0);
                items.erase(items.begin());
                world::SetItem(p, it);
                it->on_drop(ac);
            }
            break;
        case world::ITEM_KILL:
            delete it;
            display::RemoveInventoryItem(0);
            items.erase(items.begin());
            break;
        case world::ITEM_KEEP:
            break;
        }

    }
}

 // Functions

void 
player::NewWorld(int nplayers)
{
    players.clear();
    players.resize(nplayers);
}

Inventory *
player::GetInventory(int iplayer)
{
    return &players[iplayer].inventory;
}

void 
player::Suicide()
{
    SendMessage(players[icurrent_player].main_actor, "shatter");
}

void 
player::SetMainActor(int iplayer, Actor *a)
{
    assert(iplayer >= 0 && (unsigned)iplayer < players.size());
    players[iplayer].main_actor = a;
}

bool 
player::AllActorsDead()
{
    for (unsigned i=0; i<players.size(); ++i) {
        if (!players[i].main_actor->is_dead())
            return false;
    }
    return true;
}

void 
player::InhibitPickup(bool fl)
{
    current_player_inhibit_pickup = fl;
}

void 
player::PickupItem(Actor *a, GridPos p)
{
    int iplayer = a->int_attrib("player");
    if (iplayer < 0 || (unsigned)iplayer >= players.size()) {
        cerr << "PickupItem: illegal 'player' entry\n";
        return;
    }

    if (iplayer==icurrent_player && current_player_inhibit_pickup)
    {
        // do not pick up items if current player doesn't want to
        return;
    }

    Inventory &inv = players[iplayer].inventory;
    if (!inv.is_full())
    {
        Item *item = world::YieldItem(p);
        inv.add_item(item);
    }
}

void player::ActivateItem()
{
    Inventory &inv = players[icurrent_player].inventory;
    inv.activate_first();
}

void player::RotateInventory()
{
    sound::PlaySound("invrotate");
    Inventory &inv = players[icurrent_player].inventory;
    inv.rotate();
}
