//======================================================================
// Copyright (C) 2002 Daniel Heck
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//  
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//  
// You should have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//======================================================================

#include "config.h"
#include "enigma.hh"
#include "display.hh"
#include "world.hh"
#include "objects.hh"
#include "player.hh"
#include "lua.hh"
#include "px/px.hh"
#include "sound.hh"
#include "cache.hh"
#include "system.hh"

#include <SDL/SDL.h>

#include <cstdio>
#include <cstring>
#include <cstdlib>
#include <string>
#include <cassert>
#include <iostream>
#include <algorithm>

using namespace std;
using namespace px;
using namespace enigma;

namespace {
    void optionsmenu();
}

void 
GridPos::move(Direction dir) 
{ 
    switch(dir) { 
    case NORTH: y--; break; case SOUTH: y++; break;
    case EAST: x++; break; case WEST: x--; break;
    case NODIR: break;
    }
}

GridPos 
enigma::move(GridPos p, Direction dir) 
{
    GridPos tmp = p;
    tmp.move(dir);
    return tmp;
}

bool 
enigma::operator!=(GridPos a, GridPos b)
{
    return (a.x!=b.x || a.y!=b.y);
}

Direction 
enigma::reverse(Direction d)
{
    switch(d) {
    case NORTH: return SOUTH;
    case SOUTH: return NORTH;
    case WEST: return EAST;
    case EAST: return WEST;
    default: return NODIR;
    }
}




//----------------------------------------
// Video engine
//----------------------------------------
namespace
{
    class Video_SDL {
        SDL_Surface*    sdlScreen;
        bool            fullScreen;
        string          caption;
        px::Screen*     screen;
        bool            initialized;
    public:
        Video_SDL();
        ~Video_SDL();

        void init();
        void toggle_inputgrab();
        void toggle_fullscreen();
        void set_fullscreen(bool on_off);
        void set_caption(const std::string &str);
        px::Screen *get_screen() { return screen; }
    };
}

Video_SDL::Video_SDL()
    : sdlScreen(0), fullScreen(false), screen(0), initialized(false)
{}

Video_SDL::~Video_SDL()
{
    SDL_WM_GrabInput(SDL_GRAB_OFF);
    if (sdlScreen != 0 && fullScreen)
        SDL_WM_ToggleFullScreen(sdlScreen);
    delete screen;
}

void Video_SDL::set_caption(const string &str)
{
    if (initialized)
        SDL_WM_SetCaption(str.c_str(), 0);
}

void Video_SDL::init()
{
    SDL_WM_SetCaption(caption.c_str(), 0);

    int w = 640;
    int h = 480;
    int bpp = 16;

    sdlScreen = SDL_SetVideoMode(w, h, bpp, 0);
    if (sdlScreen == 0)
    {
        cerr << "couldn't open screen: " << SDL_GetError() << endl;
        exit(1);
    }
    screen = px::CreateScreen(sdlScreen);
    assert(screen != 0);

    SDL_ShowCursor(0);
    SDL_EnableKeyRepeat(SDL_DEFAULT_REPEAT_DELAY / 2,
                        SDL_DEFAULT_REPEAT_INTERVAL / 2);

    initialized = true;
}

void Video_SDL::toggle_inputgrab()
{
    if (SDL_WM_GrabInput(SDL_GRAB_QUERY) == SDL_GRAB_ON)
        SDL_WM_GrabInput(SDL_GRAB_OFF);
    else
        SDL_WM_GrabInput(SDL_GRAB_ON);
}

void Video_SDL::set_fullscreen(bool on_off)
{
    if (on_off != fullScreen)
    {
        toggle_fullscreen();
        fullScreen = on_off;
    }
}

void Video_SDL::toggle_fullscreen()
{
    SDL_WM_ToggleFullScreen(sdlScreen);
    fullScreen = !fullScreen;
}


//----------------------------------------------------------------------
// Key bindings
//----------------------------------------------------------------------
namespace
{
    class KeyBinding {
    private:
        px::Callback *cb;
    };
}

void GlobalSetKey()
{
}

void LocalSetKey()
{
}


//======================================================================
// MAIN PROGRAM
//======================================================================

namespace
{
    Video_SDL *video_engine;

    vector<string>  args;       // list of command line arguments
    bool quitFlag;
    bool level_finished;

    struct LevelInfo {
        LevelInfo(string fn, string n, string a)
            : filename(fn), name(n), author(a)
        {
            hidden=false;
            playable=true;
        }

        string filename;
        string name;
        string author;
        bool hidden;
        bool playable;
    };

    vector<LevelInfo> levels;
    unsigned icurrent_level;

}

//----------------------------------------
// MouseCursor
//----------------------------------------
namespace
{
    class MouseCursor {
    public:
        MouseCursor(const string &fname, int hx, int hy) 
	    : background(0), cursor(enigma::LoadImage(fname)) 
	{
            x = oldx = 0;
	    y = oldy = 0;
	    hotx = hx;
	    hoty = hy;
	    visible = 1;
	    changedp = true;
        }
        ~MouseCursor() {
            delete background;
            delete cursor;
        }
	
        void redraw(Screen *screen) {
	    if (visible>0 && cursor) {
		GC gc(screen);
		int scrx=x-hotx;
		int scry=y-hoty;
		blit(gc, scrx, scry, cursor);
		screen->update_rect(
		    Rect(scrx,scry,cursor->width(), cursor->height()));
	    }
	    changedp=false;	    
        }
        void move(int newx, int newy) {
            x=newx;
            y=newy;
	    changedp=true;
        }
        void show() {
	    if (++visible == 1)
		changedp=true;
        }
        void hide() {
	    if (--visible == 0)
		changedp=true;
        }
	bool has_changed() { return changedp; }
    private:
        Surface *background;
        Surface *cursor;
        int x, y;
        int oldx, oldy;
	int hotx, hoty;
	int visible;
	bool changedp;
    };
}


//
// Fadein and fadeout of screen regions
//
namespace
{
    enum FadeMode { FADEIN, FADEOUT };

    px::Screen *get_screen() { return video_engine->get_screen(); } 
    
    void fade(FadeMode mode)
    {
        const double fadesec = 0.8;
        double v = 255/fadesec;
        
        px::Screen *screen = get_screen();
        px::Surface *buffer = Duplicate(screen);
        double dt;

        double a = mode==FADEIN ? 0 : 255;
        while (true) {
            Uint32 otime = SDL_GetTicks();

            screen->box(0,0,640,480, 0);
            buffer->set_alpha(int(a));
            screen->blit(0,0,buffer);
            screen->update_all();

            dt = (SDL_GetTicks()-otime)/1000.0;
            if (mode==FADEIN && (a+=v*dt) > 255)
                break;
            else if (mode==FADEOUT && (a-=v*dt) < 0)
                break;
        }

        if (mode==FADEIN) {
            buffer->set_alpha(255);
            screen->blit(0,0,buffer);
        } else
            screen->box(0,0,640,480, 0);
        screen->update_all();
        delete buffer;
    }

    void make_screenshot(const string &fname)
    {
        SDL_SaveBMP(get_screen()->get_surface(), fname.c_str());
    }

    void flush_events()
    {
        SDL_Event e;
        while (SDL_PollEvent(&e)) ;
    }
}

void 
enigma::SetCaption(const std::string &str)
{
    video_engine->set_caption(str);
}

void 
enigma::ClearLevelList()
{
    levels.clear();
}

void 
enigma::AddLevel(const std::string &filename,
                 const std::string &name)
{
    levels.push_back(LevelInfo(filename, name, ""));
}

static void 
load_level(int ilevel) {
    icurrent_level=ilevel;
    level_finished = false;
    world::Load(levels[icurrent_level].filename);
}

void 
StartLevel(unsigned ilevel) {
    sound::PlayMusic("sound/Emilie.xm");
    load_level(ilevel);
}

void 
NextLevel()
{
    if (1+icurrent_level == levels.size()) 
        quitFlag=true;
    else {
        load_level(1+icurrent_level);

        fade(FADEOUT);
        display::Redraw(get_screen(), false);
        fade(FADEIN);
        flush_events();
    }
}

void enigma::FinishLevel()
{
    level_finished=true;
}

void
enigma::RestartLevel()
{
    bool reload_level = true;

    if (reload_level)
        load_level(icurrent_level);
    else {
        level_finished = false;
    }
    fade(FADEOUT);
    display::Redraw(get_screen(), false);
    fade(FADEIN);
    flush_events();
}

static void
game_on_keydown(SDL_Event &e)
{
    switch (e.key.keysym.sym) {
    case SDLK_ESCAPE:
        quitFlag = true;
        break;
    case SDLK_F11: video_engine->toggle_inputgrab(); break;
    case SDLK_F12: video_engine->toggle_fullscreen(); break;
    case SDLK_F10:
        make_screenshot(levels[icurrent_level].filename + ".bmp");
        break;
    case SDLK_F3:
        player::Suicide();
        break;
    case SDLK_n:
        enigma::FinishLevel();
        break;
    case SDLK_l:
        load_level(icurrent_level);
        break;
    default:
        break;
    }
}

static void
game_on_mousebutton(SDL_Event &e)
{
    if (e.button.state == SDL_PRESSED)
    {
        if (e.button.button == 1) {
            // left mousebutton -> activate first item in inventory
            player::ActivateItem();
            player::InhibitPickup(true);
        }
        else if (e.button.button == 3) {
            // right mousebutton -> rotate inventory
            player::RotateInventory();
            player::InhibitPickup(true);
        }
    } 
    else {
        // mouse button released -> allow picking up items
        int b = SDL_GetMouseState(0, 0);
        if (!(b & SDL_BUTTON(1)) &&  !(b & SDL_BUTTON(3))) {
            player::InhibitPickup(false);
        }
    }
}

static void 
game_handle_events()
{
    SDL_Event e;
    while (SDL_PollEvent(&e)) 
    {
        switch (e.type) {
        case SDL_KEYDOWN:
            game_on_keydown(e);
            break;
        case SDL_MOUSEMOTION: 
            {
                V3 force;
                force[0] = e.motion.xrel;
                force[1] = e.motion.yrel;
                double sens = lua::get_num("Sensitivity");
                world::SetMouseForce(force*sens);
            }
            break;
        case SDL_MOUSEBUTTONDOWN:
        case SDL_MOUSEBUTTONUP:
            game_on_mousebutton(e);
            break;
        case SDL_QUIT:
            quitFlag = true;
            break;
        }
    }
}

static Rect smaller(const Rect &r) 
{
    return Rect(r.x+1, r.y+1, r.w-2, r.h-2);
}

namespace
{
    class ImageAlloc {
    public:
        Surface *acquire(const std::string &name) {
            return px::LoadImage(name.c_str());
        }
        void release(Surface *s) { delete s; }
    };

    typedef cache::Cache<Surface*, ImageAlloc> ImageCache;

    class LevelMenu {
    public:
        LevelMenu();
        int manage();
    private:
        void paint();

        ImageCache cache;

        int ifirst, iselected;
        int width, height;
        Font &smallfnt;
	MouseCursor cursor;
    };
}

LevelMenu::LevelMenu()
    : ifirst(0), iselected(0),
      width(3), height(3),
      smallfnt(enigma::LoadFont("levelmenu")),
      cursor("it-magicwand", 4,4)
{
    cursor.hide();
}


void
LevelMenu::paint()
{
    Screen *screen = get_screen();
    GC gc(screen);

    set_color(gc, 0,0,0);
    box(gc, screen->size());

    vector<Rect> areas;

    const int imgw = 160;
    const int imgh = 104;
    const int hgap = 20, vgap=40;

    const int x0 = (screen->width() - width*(imgw+hgap)+hgap)/2;
    const int y0 = 30;

    unsigned i=ifirst;
    for (int y=0; y<height; y++)
        for (int x=0; x<width; x++, i++) {
            if (i >= levels.size())
                goto done_painting;

            int xpos = x0 + x*(imgw + hgap);
            int ypos = y0 + y*(imgh + vgap);
            areas.push_back(Rect(xpos-hgap/2, ypos-hgap/2, 
                                 imgw+hgap, imgh+hgap+vgap/2));

            string fname = string("levels/") + levels[i].filename + ".png";
            if (Surface *img = cache.get(enigma::FindDataFile(fname)))
                screen->blit(xpos, ypos, img);
            
            char x[10];
            sprintf(x, "(%d)   ", i);

            string tstr(x);
            tstr += levels[i].name;
            auto_ptr<Surface> title(smallfnt.render(tstr));
            screen->blit(xpos+(imgw-title->width())/2,
                         ypos+imgh+1,
                         title.get());
        }
  done_painting:

    set_color(gc, 255,0,0);
    Rect a=areas[iselected-ifirst];
    frame(gc, a);
    frame(gc, smaller(a));
    frame(gc, smaller(smaller(a)));

    screen->update_all();
}

int
LevelMenu::manage()
{
    SDL_Event e;
    bool redraw=true;
    while (true) {
        if (redraw) {
	    cursor.hide();
            paint();
	    cursor.show();
            redraw=false;
        }
	if (cursor.has_changed()) {
	    cursor.redraw(get_screen());
	    get_screen()->flush_updates();
	}
        SDL_Delay(10);

        if (SDL_PollEvent(&e)) {
	    if (e.type == SDL_MOUSEMOTION) {
		cursor.move(e.motion.x, e.motion.y);
	    } 
	    else if (e.type == SDL_MOUSEBUTTONDOWN) {
	    
	    }
	    else if (e.type == SDL_KEYDOWN) {
                int newsel = iselected;
                int newfirst = ifirst;

                switch (e.key.keysym.sym) {
                case SDLK_LEFT: newsel--; break;
                case SDLK_RIGHT: newsel++; break;
                case SDLK_DOWN: newsel += width; break;
                case SDLK_UP: newsel -= width; break;
                case SDLK_PAGEDOWN: 
                    newfirst += width*height; 
                    newsel   += width*height;
                    break;
                case SDLK_PAGEUP:
                    newfirst -= width*height; 
                    newsel   -= width*height;
                    break;
                case SDLK_HOME: newsel = 0; break;
                case SDLK_END: newsel = levels.size()-1; break;
                case SDLK_RETURN:
                    return iselected;
                case SDLK_ESCAPE:
                    return -1;
                default:
                    break;
                }

                if (newfirst!=ifirst && newfirst>=0 && newfirst<(int)levels.size())
                {
                    ifirst = newfirst;
                    iselected = Max(Min(newsel,(int)levels.size()-1), 0);
                    redraw=true;
                }
                else if (newsel!=iselected && newsel>=0 && newsel<(int)levels.size())
                {
                    while (newsel < ifirst)
                        ifirst -= width;
                    while (newsel >= ifirst+width*height)
                        ifirst += width;
                    iselected = newsel;
                    redraw = true;
                }
            }
        }

    }
    cache.clear();
}

static void
game()
{
    if (levels.empty()) {
        cerr << "no levels defined\n";
        return;
    }

    LevelMenu lm;
    int ilevel = lm.manage();
    if (ilevel == -1)
        return;                 // return to main menu
    StartLevel(ilevel);

    display::Redraw(get_screen());

    Uint32 old_time = SDL_GetTicks();
    Uint32 new_time;
    quitFlag = false;

    while (!quitFlag) {
        double dtime = 0;
        do {
            new_time = SDL_GetTicks();
            dtime = (new_time - old_time) / 1000.0;
        } while (dtime < 4/1000.0);

        game_handle_events();
        world::Tick(dtime);
        display::Tick(dtime);


        display::Redraw(get_screen());
        old_time = new_time;

        if (player::AllActorsDead()) {
            enigma::RestartLevel();
            old_time = SDL_GetTicks();
        }
        if (level_finished) {
            NextLevel();
            old_time = SDL_GetTicks();
        }

// ?? movements become jerky with this delay on. maybe make it an option?
//        SDL_Delay(10);
    }
    sound::StopMusic();
}

namespace
{
    class Widget {
    public:
        virtual ~Widget() {}
        void draw(px::GC &gc);
        Rect get_area() const { return area; }
    private:
        virtual void on_draw(px::GC &gc) = 0;
        Rect area;
    };

    class TextWidget : public Widget {
        string text;
    public:
        TextWidget(const string &str) : text(str) {}
        
    private:
    };

    class WidgetLayouter {
    public:

    };
}


//----------------------------------------------------------------------
 // Menu handling
//----------------------------------------------------------------------

class MenuEntry {
public:
    virtual ~MenuEntry() {}
    virtual void handle_event(SDL_Event *e) {}
    virtual Rect size() = 0;
    virtual void draw(GC &gc, const Rect &r, bool isactive) = 0;
};

//----------------------------------------
// TextME
//----------------------------------------
class TextME : public MenuEntry {
    string text;
    Font &menufont, &menufontsel;
public:
    TextME(const string &t) 
        : text(t), 
          menufont(enigma::LoadFont("menufont")),
          menufontsel(enigma::LoadFont("menufontsel"))
    {}

    Rect size() { 
        return Rect(0,0,menufont.get_width(text), menufont.get_height()); 
    }
    void draw(GC &gc, const Rect &r, bool isactive) {
        auto_ptr<Surface> s;
        if (isactive)
            s.reset(menufontsel.render(text));
        else
            s.reset(menufont.render(text));
        blit(gc, r.x, r.y, s.get());
    }
};

//----------------------------------------
// Menu
//----------------------------------------
namespace
{
    class Menu {
    public:
        Menu();
        virtual ~Menu();
        void add_entry(MenuEntry *e) { entries.push_back(e); }
        Rect size();
        void set_active(unsigned idx);
        virtual int manage(Screen *scr);
    protected:
        virtual void draw(GC &gc, const Rect &r);
        void quit_menu(int retcode) { 
            quit_menu_p = true;
            return_code = retcode; 
        }
        virtual void on_event(SDL_Event *e);
    private:
        // Variables
        bool quit_menu_p, must_redraw;
        int return_code;
        vector<MenuEntry *> entries;
        unsigned iactive_entry;
        auto_ptr<Surface16> background; // background image
    };
}

Menu::Menu() 
    : iactive_entry(0)
{}
Menu::~Menu() 
{ 
    delete_sequence(entries.begin(), entries.end()); 
}

void Menu::set_active(unsigned idx)
{
    if (iactive_entry != idx) {
        iactive_entry = idx;
        must_redraw = true;
    }
}

Rect Menu::size() 
{
    int h=0;
    int maxw=0;
    for (unsigned i=0; i<entries.size(); ++i) {
        Rect s = entries[i]->size();
        h += s.h;
        if (s.w > maxw) maxw=s.w;
    }
    return Rect(0,0,maxw,h);
}

void Menu::draw(GC &gc, const Rect &r)
{
    clip(gc,r);

    blit(gc, r.x, r.y, background.get());
    int yoff=0;
    for (unsigned i=0; i<entries.size(); ++i) {
        Rect entryarea = center(r, entries[i]->size());
        entryarea.y = r.y + yoff;
        entries[i]->draw(gc, entryarea, i==iactive_entry);
        yoff += entryarea.h;
    }
    must_redraw = false;
}

int Menu::manage(Screen *scr)
{
    Rect r = center(scr->size(), this->size());

    background.reset(new Surface16(r.w, r.h));
    background->blit(0, 0, scr, r);

    GC gc(scr);

    must_redraw = true;
    quit_menu_p = false;
    SDL_Event e;
    while (!quit_menu_p) {
        if (must_redraw) {
            draw(gc, r);
            scr->update_rect(r);
            scr->flush_updates();
        }

        SDL_Delay(10);
        while (SDL_PollEvent(&e)) {
            on_event(&e);
        }
    }
    scr->blit(r.x, r.y, background.get());
    return return_code;
}

void 
Menu::on_event(SDL_Event *e)
{
    if (e->type == SDL_KEYDOWN) {
        switch (e->key.keysym.sym) {
        case SDLK_ESCAPE:
            quit_menu(-1);
            break;
        case SDLK_RETURN:
            quit_menu(iactive_entry);
            break;
        case SDLK_DOWN:
            if (iactive_entry < entries.size()-1)
                set_active(iactive_entry +1);
            break;
        case SDLK_UP:
            if (iactive_entry >0)
                set_active(iactive_entry-1);
            break;
        default:
            ;
        }
    }
    else if (e->type == SDL_MOUSEMOTION) {

    }
    else if (e->type == SDL_MOUSEBUTTONDOWN) {

    }
}

//----------------------------------------------------------------------
// Main menu
//----------------------------------------------------------------------
namespace
{
    class MainMenu : public Menu {
        auto_ptr<Surface> logo;
    public:
        MainMenu()
            : logo (enigma::LoadImage("enigma_logo2"))
	{
            assert(logo.get());
	    add_entry(new TextME("New Game"));
//	    add_entry(new TextME("Options"));
	    add_entry(new TextME("Quit"));
	}
	
        int manage(Screen *scr) {
            scr->box(0, 0, scr->width(), scr->height(), 12421);
            Rect logoarea = center(scr->size(), logo->size());
            logoarea.y = 40;
            scr->blit(logoarea.x,logoarea.y, logo.get());
            scr->update_all();
            return Menu::manage(scr);
        }
    private:
        void on_event(SDL_Event *e) {
            if (e->type == SDL_KEYDOWN) {
                switch (e->key.keysym.sym) {
                case SDLK_F11:
                    video_engine->toggle_inputgrab(); 
                    return;
                case SDLK_F12:
                    video_engine->toggle_fullscreen();
                    return;
                default: ;
                }
            } 
            else if (e->type == SDL_QUIT) {
                quit_menu(-1);
                return;
            }
            // pass unhandled events to the parent class
            Menu::on_event(e);
        }
    };

    void mainmenu()
    {
        MainMenu m;

        while (true) {
            enigma::SetCaption("Enigma - Main Menu");
            try {
                switch (m.manage(get_screen())) {
                case 0: game(); break;
//                case 1: optionsmenu(); break;
                case 1: case -1: return;
                default: break;
                }
            } catch (lua::Error) {
                cerr << "Returned to main menu...\n";
            }
        }
    }
}

//----------------------------------------------------------------------
// Options menu
//----------------------------------------------------------------------

namespace 
{
    class OptionsMenu : public Menu {
    public:
        OptionsMenu() {
	    add_entry(new TextME("Back"));
        }
        int manage(Screen *scr) {
            scr->box(0, 0, scr->width(), scr->height(), 0);
            scr->update_all();
            return Menu::manage(scr);
        }

    };

    void optionsmenu()
    {
        enigma::SetCaption("Enigma - Options");
        OptionsMenu m;
        m.manage(get_screen());
    }
}


//----------------------------------------------------------------------
// Startup
//----------------------------------------------------------------------

namespace
{
    void load_config_file()
    {
        // First try a configuration file in the current directory.
        string fname = "enigma_conf.lua";
        try {
            lua::Dofile(fname);
        } catch (lua::Error) {
            cerr << "Could not load configuration file\n";
        }
    }

    void save_config_file()
    {
    }
}

static void 
init()
{
    bool nosound_flag = false;

    lua::Init();

    bool show_help = false;
    for (unsigned i=0; i < ::args.size(); ++i) 
    {
        string& arg = ::args[i];

        if (arg == "--help" || arg == "-h")
            show_help = true;
	else if (arg == "--nosound")
	    nosound_flag = true;
        else
            ; // ignore unknown arguments
    }
    if (show_help) {
        cout << "Available command line options for enigma:\n\n"
             << "    --nosound      disable music and sound\n"
             << "    --help -h      show this help\n"
             << endl;
    }

    // load configuration file
    load_config_file();

    int sdl_flags = SDL_INIT_VIDEO | SDL_INIT_NOPARACHUTE;
    if (!nosound_flag) {
	sdl_flags |= SDL_INIT_AUDIO;
    } else {
        sound::DisableSound();
    }

    if (SDL_Init(sdl_flags) < 0) 
    {
        cerr << "Couldn't init SDL: " << SDL_GetError() << endl;
        exit(1);
    }

    video_engine = new Video_SDL();
    video_engine->init();
//    video_engine->set_fullscreen(true);

    display::Init();
    sound::Init();

    // load level index
    lua::Dofile(enigma::FindDataFile("levels/init.lua"));
}

static void 
shutdown()
{
    delete video_engine;
    display::Shutdown();
    world::Shutdown();
    save_config_file();
}

template <class OutIt>
void split_copy(const std::string &str, char sep, OutIt out)
{
    string::const_iterator i,j;
    i=str.begin();
    while (true) {
        j = std::find(i, str.end(), sep);
        *out = string(i,j);
        ++out;
        if (j==str.end())
            break;
        i=j+1;
    }
}

namespace
{
    string datapath=DEFAULT_DATA_PATH;
    vector<string> datapaths;

    void init_datapaths()
    {
        datapaths.clear();
        split_copy(datapath, ':', back_inserter(datapaths));

        for_each(datapaths.begin(), datapaths.end(), &sysdep::expand_path);
    }
}

std::string 
enigma::FindDataFile(const std::string &filename)
{
    if (datapaths.empty())
        init_datapaths();

    for (unsigned i=0; i<datapaths.size(); ++i) {
        string complete_name = datapaths[i] + sysdep::path_separator + filename;
        if (FILE *fp = fopen(complete_name.c_str(), "rb")) {
            fclose(fp);
            return complete_name;
        }
    }
    cerr << "FindDataFile: file not found: " << filename << endl;
    return filename;
}


namespace
{
    class FontAlloc {
    public:
        Font *acquire(const std::string &name) {
            return px::LoadBitmapFont(
                enigma::FindDataFile(string("fonts/") + name + ".png"),
                enigma::FindDataFile(string("fonts/") + name + ".bmf"));
        }
        void release(Font *f) { delete f; }
    };

    cache::Cache<Font *, FontAlloc> font_cache;
}

px::Font &
enigma::LoadFont(const string &name)
{
    return *font_cache.get(name);
}

px::Surface *
enigma::LoadImage(const std::string &name)
{
    string filename = FindDataFile(string("gfx/") + name + ".png");
    return px::LoadImage(filename.c_str());
}


int main(int argc, char** argv)
{
    try {
        copy(argv+1, argv+argc, back_inserter(::args));
        init();
        mainmenu();
        shutdown();
    }
    catch (px::XGeneric& e) {
        cerr << "main: caught px exception: " << e.get_string() << endl;
        return 1;
    }
    catch (std::exception& e) {
        cerr << "main: caught exception: " << e.what() << endl;
        return 1;
    }
    return 0;
}
