//======================================================================
// Copyright (C) 2002 Daniel Heck
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//  
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//  
// You should have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//======================================================================

// This file contains the code responsible for the complete
// in-game visual rendering.  This includes displaying the current
// landscape with all its objects and the inventory at the bottom
// of the screen.

#include "lua.hh"
#include "display.hh"
#include "tools.hh"
#include "cache.hh"
#include "px/tools.hh"
#include "px/array2.hh"
#include "px/video.hh"
#include "px/font.hh"

#include <vector>
#include <algorithm>
#include <functional>
#include <list>
#include <iostream>
#include <map>
#include <cmath>

using namespace std;
using namespace px;
using namespace display;

namespace
{
    class View {
    public:
        virtual bool has_changed() = 0;
        virtual void draw(GC &gc, Rect r) = 0;
        virtual void draw_changes(GC &gc, Rect r) = 0;
    };
    class WorldView : public View {
    public:
    private:
    };
    class InventoryView : public View {
    public:
    private:
    };

}


namespace
{
    typedef px::Handle<Surface> SurfaceHandle;

    class SurfaceAlloc {
    public:
        SurfaceHandle acquire(const std::string &name) {
            return SurfaceHandle(enigma::LoadImage(name));
        }
        void release(SurfaceHandle sh) {}
    };
        
    typedef cache::Cache<SurfaceHandle, SurfaceAlloc> SurfaceCache;
}



namespace
{
    SurfaceCache surface_cache;
}

//----------------------------------------
// Function prototypes
//----------------------------------------
namespace
{
    class Model2d;

    Model2d *make_model2d(const string &name);
    void define_model(const string &name, Model2d *prototype);

    void world_to_screen (const V3 & pos, int *x, int *y);
    void init_lua_funcs();

    void add_active_model(Model2d *m);
    void remove_active_model(Model2d *m);
}


//======================================================================
 // 2D models
//======================================================================
namespace
{
    typedef unsigned int uint;

    class Model2d : public Animation {
    public:
        virtual void draw(px::GC &gc, int x, int y) = 0;
        virtual void draw_shade(px::GC &gc, int x, int y) = 0;

        virtual void activate(double worldx, double worldy) {}
        virtual void deactivate() {}
        virtual bool is_garbage() const { return false; }
        virtual void tick(double dtime) {}
        virtual bool has_changed(Rect &changed_region) { return false; }

        virtual Model2d *clone()=0;
    };
}

//----------------------------------------------------------------------
 // Image
//----------------------------------------------------------------------
namespace
{
    class ImageRep {
    public:
        ImageRep(px::Surface *i, int xo, int yo)
            : image(i), xoff(xo), yoff(yo), refcount(1)
        {}
        ~ImageRep() { delete image; }
        px::Surface *image;
        int xoff, yoff;     // relative origin of the image
        int refcount;
    };

    class Image : public Model2d {
        ImageRep* rep;
    private:
        Image(ImageRep *r) : rep(r) {rep->refcount++;}
    public:
        Image(px::Surface *sfc, int xoff, int yoff) {
            rep = new ImageRep(sfc,xoff,yoff);
        }
        ~Image() {
            if (--rep->refcount == 0)
                delete rep;
        }
        void draw(px::GC &gc, int x, int y) {
            if (rep->image)
                blit(gc, x+rep->xoff, y+rep->yoff, rep->image);
        }
        void draw_shade(px::GC &gc, int x, int y) {}
        Model2d *clone() {return new Image(rep);}
    };
}

//----------------------------------------------------------------------
// ShadedModel
//----------------------------------------------------------------------
namespace
{
    class ShadedModel : public Model2d {
    public:
        ShadedModel(Model2d *m, Model2d *sh) {model=m; shade=sh;}
        ~ShadedModel() { delete model; delete shade; }

        // Model2d interface
        void activate(double worldx, double worldy) {
            model->activate(worldx, worldy);
        }
        void deactivate() { model->deactivate(); }
	void set_callback(AnimCallback *cb) { model->set_callback(cb); }
	void reverse() { model->reverse(); }
        void draw(px::GC &gc, int x, int y) { 
            if(model) 
                model->draw(gc,x,y); 
        }
        void draw_shade(px::GC &gc, int x, int y) { 
            if(shade) 
                shade->draw(gc,x,y); 
        }
        Model2d *clone() {
            return new ShadedModel(model->clone(), shade->clone());
        }
    private:
        Model2d *model, *shade;
    };
}

//----------------------------------------------------------------------
// CompositeModel
//
// This kind of model consists of two layers: one background layer and
// some foreground layer that is drawn over the background.  Both
// layers may be animated.  This model make it easy to compose complex
// animations from simple ones; see models-2d.lua for a few examples.
//----------------------------------------------------------------------
namespace 
{
    class CompositeModel : public Model2d {
        Model2d *bg, *fg;
    public:
        CompositeModel(Model2d *b, Model2d *f) : bg(b), fg(f) {}
        ~CompositeModel() { 
            delete bg; delete fg; 
        }

        // Animation interface
        void set_callback(AnimCallback *cb) {
            bg->set_callback(cb);
            fg->set_callback(cb);
        }
        double duration() const { return 0; }
        void reverse() {
            bg->reverse();
            fg->reverse();
        }

        // Model2d interface
        void activate(double worldx, double worldy) {
            bg->activate(worldx, worldy);
            fg->activate(worldx, worldy);
        }
        void deactivate() {
            bg->deactivate();
            fg->deactivate();
        }
        void draw(px::GC &gc, int x, int y) { 
            bg->draw(gc,x,y);
            fg->draw(gc,x,y);
        }
        void draw_shade(px::GC &gc, int x, int y) {
            bg->draw_shade(gc,x,y);
        }
        Model2d *clone() {
            return new CompositeModel(bg->clone(), fg->clone());
        }
    };
}

//----------------------------------------------------------------------
// RandomModel
//----------------------------------------------------------------------
namespace
{
    class RandomModel : public Model2d {
        vector<string> modelnames;
    public:
        void draw(px::GC &gc, int x, int y) {}
        void draw_shade(px::GC &gc, int x, int y) {}
        void add_model(const string &name) {modelnames.push_back(name);}

        // Model2d interface
        Model2d *clone() {
            if (!modelnames.empty()) {
                int r = int(modelnames.size() * (rand()/(RAND_MAX+1.0)));
                return make_model2d(modelnames[r]);
            } else {
                cerr << "display_2d.cc: empty RandomModel\n";
                return 0;
            }
        }
    };
}

//----------------------------------------------------------------------
 // Animation model
//----------------------------------------------------------------------
namespace
{
    class AnimFrame : public px::Nocopy {
    public:
        AnimFrame(Model2d *m, double dur)
            : model(m), duration(dur)
        {}
        ~AnimFrame() { delete model; }
        Model2d *model;
        double  duration;
    };

    class AnimRep {
    public:
        AnimRep(bool l) : loop(l), refcount(1) {}
        ~AnimRep() { delete_sequence(frames.begin(), frames.end()); }
        double duration() const {
            double d = 0;
            for (uint i=0; i<frames.size(); i++)
                d += frames[i]->duration;
            return d;
        }
        vector<AnimFrame*>  frames;
        bool                loop;
        int                 refcount;
    };

    class Anim2d : public Model2d, public px::Nocopy {
    public:
        Anim2d(bool loop) : rep(new AnimRep(loop)) {}
        ~Anim2d();
        double duration() const;
        void set_callback(AnimCallback *cb) { callback = cb; }

        void add_frame(Model2d *m, double duration);

        // Model2d interface
        void draw(px::GC &gc, int x, int y);
        void draw_shade(px::GC &gc, int x, int y);
        Model2d *clone() { return new Anim2d(rep); }
        void reverse() { reversep = !reversep; }

        void activate(double worldx, double worldy);
        void deactivate();
        void tick(double dtime);
        bool has_changed(Rect &changed_region);
        bool is_garbage() const { return finishedp; }
    private:
        Anim2d(AnimRep *r);

        AnimRep *rep;
        uint    curframe;       // current frame number
        double  frametime;      // elapsed time since frame was activated
        bool    finishedp;      // animation has finished
        bool    changedp;       // model state has changed since redraw
        bool    reversep;       // play the animation in reverse direction

        double posx, posy;
        AnimCallback *callback;
    };
}

Anim2d::Anim2d(AnimRep *r) 
    : rep(r), curframe(0), frametime(0), 
      finishedp(false), changedp(false), reversep(false),
      callback(0)
{
    rep->refcount++;
}

Anim2d::~Anim2d() 
{
    if (--rep->refcount == 0)
        delete rep;
    deactivate();
}

void Anim2d::add_frame(Model2d *m, double duration)
{
    rep->frames.push_back(new AnimFrame(m, duration));
}

double Anim2d::duration() const
{
    return rep->duration();
}

void Anim2d::draw(px::GC &gc, int x, int y) 
{
    if (!finishedp) {
        AnimFrame *f =rep->frames[curframe];
        f->model->draw(gc,x,y);
        changedp = false;
    }
}
        
void Anim2d::draw_shade(px::GC &gc, int x, int y) 
{
    if (!finishedp) {
        AnimFrame *f =rep->frames[curframe];
        f->model->draw_shade(gc,x,y);
    }
}

void Anim2d::activate(double worldx, double worldy)
{
    add_active_model(this);
    posx=worldx;
    posy=worldy;
}

void Anim2d::deactivate()
{
    remove_active_model(this);
}

bool Anim2d::has_changed(Rect &changed_region)
{
    if (changedp)
        changed_region = Rect(int(posx-1), int(posy-1), 3, 3);
    return changedp;
}

void Anim2d::tick (double dtime) 
{
    assert(curframe >= 0);
    assert(curframe < rep->frames.size());
    frametime += dtime;
    double framedur = rep->frames[curframe]->duration;

    if (frametime >= framedur) {
        frametime -= framedur;
        changedp = true;

        if (reversep) {
            if (curframe >= 1)
                curframe--;
            else if (rep->loop)
                curframe = rep->frames.size()-1;
            else
                finishedp = true;
        } 
        else {
            if (curframe+1 < rep->frames.size())
                curframe++;
            else if (rep->loop)
                curframe = 0;
            else 
                finishedp = true;
        }
        if (finishedp && callback!=0)
            callback->animation_finished();
    }
}

//======================================================================
// DEFINING AND CREATING MODELS
//======================================================================
namespace
{
    map<string, Model2d*> model_templates;

    void define_model(const string &name, Model2d *m)
    {
        if (model_templates.find(name) != model_templates.end()) {
            cerr << "redefinition of model " << name << endl;
            delete m;
        }
        else
            model_templates[name] = m;
    }

    // Create a new model of type `name'
    Model2d * make_model2d(cstring &name)
    {
        map<string,Model2d*>::iterator i=model_templates.find(name);
        if (i != model_templates.end()) 
            return i->second->clone();
        else {
            cerr << "unknown model " << name << endl;
            return 0;
        }
    }
}



//======================================================================
// DISPLAY ENGINE
//======================================================================

namespace
{
    struct Field {
        static const int NUM_LAYERS = 3;
    
        Field() { fill_n(layers, NUM_LAYERS, (Model2d*)0); }
        ~Field() { delete_sequence(layers, layers+NUM_LAYERS); }
        Model2d* layers[NUM_LAYERS];
    };
    class Sprite : public Nocopy {
    public:
        Model2d *model;
        V3 pos;
        SpriteLayer layer;

        Sprite(const V3 & p, SpriteLayer l, Model2d *m) 
            : model(m), pos(p), layer(l)
        {}
        ~Sprite() { delete model; }
    };

    typedef Array2<Field>   FieldArray;
    typedef list<Model2d*>   ModelList;

    //typedef map<SpriteId, Sprite*> SpriteList;
    typedef vector<Sprite*> SpriteList;
}

//----------------------------------------
// TextDisplay
//----------------------------------------
namespace 
{
    class TextDisplay {
    public:
        TextDisplay(Rect a, Font &f);

        void set_text(const string &t, TextMode m);

        void deactivate() { active = false; }
        bool is_active() const { return active; }

        void tick(double dtime);
        bool has_changed() { return changedp; }

        void draw(px::GC &gc, const Rect &r);
    private:
        string text;
        bool active, changedp;
        TextMode mode;
        Rect area;
        double xoff;
        const double scrollspeed; // pixels per second
        std::auto_ptr<Surface> textsurface;
        px::Font &font;
        double time, maxtime;
    };
}

TextDisplay::TextDisplay(Rect a, Font &f) 
    : text(), active(false), changedp(false), 
      mode(TEXT_SCROLLING),
      area(a), xoff(0), scrollspeed(200),
      textsurface(0), font(f)
{
    time = maxtime = 0;
}

void 
TextDisplay::set_text(const string &t, TextMode m) 
{ 
    text = t;
    mode = m;
    textsurface.reset(font.render(text.c_str()));

    time = 0;
    maxtime = 1e20;             // this should be close ``indefinitely''
    switch (mode) {
    case TEXT_SCROLLING:
        xoff = -area.w;
        break;
    case TEXT_2SECONDS:
        maxtime = 2.0;          // 2 seconds
        // fall through
    case TEXT_STATIC:
        xoff = -(area.w - textsurface->width())/2;
        break;
    }
    changedp = active = true;
}

void 
TextDisplay::tick(double dtime) 
{
    time += dtime;
    if (time > maxtime) {
        active = false;
        changedp = true;
    } 
    else {
        int oldxoff = int(xoff);
        if (mode == TEXT_SCROLLING) {
            xoff += dtime * scrollspeed;
            changedp = int(xoff) != oldxoff;
            if (xoff >= textsurface->width()) {
                active = false;
                changedp = true;
            }
        }
    }
}

void 
TextDisplay::draw(px::GC &gc, const Rect &r) 
{
    if (active) {
        clip(gc, intersect(area, r));
        set_color(gc, 0,0,0);
        box(gc, area);
        blit(gc, area.x-int(xoff), area.y, textsurface.get());
    }
}

//======================================================================
 // LOCAL VARIABLES
//======================================================================

namespace
{
    // Do not change the order of the following two variables!
    ModelList       active_models;
    FieldArray      fields(0,0);
    SpriteList      sprites;

    // Parameters that affect the appearance on the screen
    DisplayFlags display_flags = SHOW_ALL;
    int          scrollx, scrolly;

    // Width and height of a tile
    int tile_w = 32;
    int tile_h = 32;


    // Inventory
    vector<Model2d*> invmodels;
    bool invchanged = false;


    bool         redraw_everything;
    vector<Rect> changelist;

    // Screen layout
    Rect gamearea(0,0,640, 480-64);
    Rect inventoryarea(0, 480-64, 640, 64);
    Rect itemarea(142, 480-64+5, 640-147, 64-10);
    Rect textarea(147, 480-64+20, 640-157, 32);

    TextDisplay *text_display;
}

void 
display::Init()
{
    text_display = new TextDisplay(textarea, enigma::LoadFont("dreamorp24"));
    init_lua_funcs();
    cout << "initializing 2d models... " << flush;
    lua::Dofile(enigma::FindDataFile("models-2d.lua"));
    cout << "done\n";
    cout << "# of models: " << model_templates.size() << endl;
}

void 
display::Shutdown()
{
    delete_sequence(sprites.begin(), sprites.end());
    active_models.clear();
    delete text_display;

    delete_map(model_templates.begin(),
               model_templates.end());
}

void 
display::NewWorld(int w, int h)
{
    delete_sequence(invmodels.begin(), invmodels.end());
    invmodels.clear();

    fields = FieldArray(w, h);
    scrollx = scrolly = 0;

    delete_sequence(sprites.begin(), sprites.end());
    sprites.clear();
    redraw_everything = true;
    text_display->deactivate();
}

//----------------------------------------------------------------------
// Sprite following code
//----------------------------------------------------------------------
namespace
{
    const SpriteId magic_spriteid = 1000000;
    FollowMode follow_mode = FOLLOW_SCREEN;
    SpriteId follow_sprite = magic_spriteid;

    void follow_doit()
    {
        if (follow_sprite != magic_spriteid)
        {
            Sprite *s = sprites[follow_sprite];
        
            int sx, sy;
            world_to_screen(s->pos, &sx, &sy);
	
            if (sx < gamearea.x + tile_w/2) {
                scrollx -= 19;
                redraw_everything = true;
            }
            if (sy < gamearea.y + tile_h/2) {
                scrolly -= 12;
                redraw_everything = true;
            }

            if (sx >= gamearea.x + gamearea.w-tile_w/2) {
                scrollx += 19;
                redraw_everything = true;
            }
            if (sy >= gamearea.y + gamearea.h-tile_h/2) {
                scrolly += 12;
                redraw_everything = true;
            }
        }
    }
}

void 
display::FollowSprite(SpriteId id)
{
    follow_sprite = id;
    follow_doit();
}

void 
display::SetFollowMode(FollowMode m)
{
    follow_mode = m;
}

//----------------------------------------------------------------------
 // Inventory management
//----------------------------------------------------------------------

void 
display::AddInventoryItem(cstring &mname)
{
    Model2d *m2 = make_model2d(mname);
    invmodels.insert(invmodels.begin(), m2);
    invchanged = true;
}

void 
display::RemoveInventoryItem(int idx)
{
    delete invmodels[idx];
    invmodels.erase(invmodels.begin()+idx);
    invchanged = true;
}

void 
display::RotateInventory()
{
    rotate(invmodels.begin(), invmodels.begin()+1,
           invmodels.end());
    text_display->deactivate();
    invchanged = true;
}

void 
display::ShowText(const std::string &str, TextMode m)
{
    text_display->set_text(str, m);
}

namespace
{
    void mark_change(const Rect & r)
    {
        Rect visible_rect(scrollx-1, scrolly-1, 
                          gamearea.w/tile_w + 2,
                          gamearea.h/tile_h + 2);
        if (visible_rect.overlaps(r))
            changelist.push_back(r);
    }
}

//----------------------------------------
// Handling active models
//----------------------------------------

namespace 
{
    void add_active_model(Model2d *m)
    {
        list<Model2d*> &am = ::active_models;
        if (find(am.begin(), am.end(), m) ==am.end()) {
            // if not already in the list...
            am.push_back(m);
        }
    }

    void remove_active_model(Model2d *m)
    {
        ::active_models.remove(m);
    }

    void maybe_redraw_model(Model2d *m)
    {
        Rect r;
        if (m->has_changed(r))
            mark_change(r);
    }

    void redraw_sprite_region(SpriteId id)
    {
        Sprite *s = sprites[id];
        int x = int(s->pos[0]);
        int y = int(s->pos[1]);
        mark_change (Rect(x-1, y-1, 3, 3));
    }
}

void 
display::Tick(double dtime)
{
    ModelList &am = active_models;
    am.remove_if(mem_fun(&Model2d::is_garbage));

    // Watch out! Animation callbacks invoked during a 'tick' often
    // delete the animation object that called them in the first
    // place.  Because this will also remove the model from the list
    // active models, we cannot use the standard for_each algorithm
    // (which assumes that the sequence is not modified) to traverse
    // the list.
    for (ModelList::iterator i=am.begin(); i!=am.end(); ) {
        ModelList::iterator n=next(i);
        (*i)->tick(dtime);
        i=n;
    }
    for_each(am.begin(), am.end(), &maybe_redraw_model);

    follow_doit();
    
    // update text display
    if (text_display->is_active()) {
        text_display->tick(dtime);
        invchanged = text_display->has_changed();
    }
}

ModelId 
display::SetModel(GridLayer layer, int x, int y, cstring &modelname)
{
    Field& f = fields(x, y);
    Model2d *m2 = make_model2d(modelname);
    delete f.layers[layer];
    f.layers[layer] = m2;
    if (m2)
        m2->activate(double(x), double(y));
    if (layer == GRID_STONES)
        mark_change(Rect(x, y, 2, 2));
    else
        mark_change(Rect(x,y,1,1));
    return ModelId(layer,x,y);
}

void
display::KillModel(GridLayer l, int x, int y)
{
    KillModel(ModelId(l,x,y));
}

void
display::KillModel(ModelId id)
{
    Field& f = fields(id.x, id.y);
    delete f.layers[id.layer];
    f.layers[id.layer] = 0;
    mark_change(Rect(id.x, id.y, 2, 2));
}

void 
display::SetCallback(ModelId id, AnimCallback *cb)
{
    if (Animation *a = GetAnimation(id))
        a->set_callback(cb);
}

void 
display::ReverseAnimation(ModelId id)
{
    if (Animation *a = GetAnimation(id))
        a->reverse();
}


SpriteId
display::AddSprite(SpriteLayer l, const V3& pos, cstring &modelname)
{
    SpriteId id;
    Model2d *m2 = make_model2d(modelname);

    // find first empty entry
    SpriteList::iterator i = find(sprites.begin(),sprites.end(), (Sprite*)0);
    if (i == sprites.end()) {
        id = sprites.size();
        sprites.push_back(new Sprite(pos, l, m2));
    }
    else {
        id = distance(sprites.begin(), i);
        *i = new Sprite(pos, l, m2);
    }
    m2->activate(pos[0], pos[1]);
    redraw_sprite_region(id);
    return id;
}

void
display::MoveSprite(SpriteId id, const px::V3& newpos)
{
    Sprite *sprite = sprites[id];
    if ((fabs(newpos[0]-sprite->pos[0]) > 1.0/tile_w) ||
        (fabs(newpos[1]-sprite->pos[1]) > 1.0/tile_h))
    {
        redraw_sprite_region(id);
        sprite->pos = newpos;
        redraw_sprite_region(id);
    }
}

void 
display::ReplaceSprite(SpriteId id, cstring &modelname)
{
    Sprite *sprite = sprites[id];
    delete sprite->model;
    sprite->model = make_model2d(modelname);
    sprite->model->activate(sprite->pos[0], sprite->pos[1]);
    redraw_sprite_region(id);
}


void
display::KillSprite(SpriteId id) 
{
    if (Sprite *sprite = sprites[id]) {
        sprites[id] = 0;
        delete sprite;
    }
}

void 
display::SetCallback(SpriteId id, AnimCallback *cb)
{
    if (Animation *a = GetAnimation(id))
        a->set_callback(cb);
}

Animation *
display::GetAnimation(ModelId id)
{
    Field& f = fields(id.x, id.y);
    return f.layers[id.layer];
}

Animation *
display::GetAnimation(SpriteId id)
{
    return sprites[id]->model;
}


//======================================================================
// LUA INTERFACE CODE
//======================================================================

namespace
{
    int def_image(lua_State *L)
    {
        const char *name=lua_tostring(L, 1);
        const char *fname=lua_tostring(L, 2);
        int xoff=int(lua_tonumber(L, 3));
        int yoff=int(lua_tonumber(L, 4));

        if (!name) lua_error(L, "invalid model name");
        if (!fname) lua_error(L, "invalid file name");

        px::Surface *sfc = enigma::LoadImage(fname);
        if (sfc) { 
            Image *img=new Image(sfc, xoff, yoff);
            define_model(name, img);
        }
        return 0;
    }

    int def_randmodel(lua_State *L)
    {
        RandomModel *m = new RandomModel();
        const char *name     = lua_tostring(L, 1);
        int n          = lua_getn(L, 2);

        for (int i=1; i<=n; i++) {
            lua_rawgeti(L, 2, i);
            m->add_model(lua_tostring(L, -1));
            lua_pop(L, 1);
        }
        define_model(name, m);
        return 0;
    }

    int def_shmodel(lua_State *L)
    {
        const char *name = lua_tostring(L, 1);
        const char *model = lua_tostring(L, 2);
        const char *shade = lua_tostring(L, 3);

        if (!name) lua_error(L, "invalid model name");
        if (!model) lua_error(L, "invalid foreground model");
        if (!shade) lua_error(L, "invalid shade model");

        define_model(name, 
                     new ShadedModel(make_model2d(model),
                                     make_model2d(shade)));
        return 0;
    }


    string anim_templ_name;
    Anim2d *anim_templ = 0;


    int def_anim (lua_State *L)
    {
        const char *name = lua_tostring(L, 1);
        bool loop = bool(lua_tonumber(L, 2));

        if (!name) lua_error(L, "invalid model name");

        anim_templ = new Anim2d(loop);
        anim_templ_name = name;
        define_model(anim_templ_name, anim_templ);
        return 0;
    }

    int add_frame(lua_State *L)
    {
        const char *name = lua_tostring(L, 1);
        const char *model= lua_tostring(L, 2);
        double time_sec = lua_tonumber(L, 3) / 1000.0;
    
        if (anim_templ_name != name) 
            lua_error(L, "cannot add frames to completed animations");
        if (!model) lua_error(L, "invalid model name");

        anim_templ->add_frame(make_model2d(model), 
                              time_sec);
        return 0;
    }

    int set_tile_size(lua_State *L)
    {
        int w = int(lua_tonumber(L, 1));
        int h = int(lua_tonumber(L, 2));

        tile_w = w;
        tile_h = h;
        return 0;
    }

    int def_composite(lua_State *L)
    {
        const char *name = lua_tostring(L, 1);
        const char *bgname = lua_tostring(L, 2);
        const char *fgname = lua_tostring(L, 3);

        if (!name) lua_error(L, "invalid model name");
        if (!bgname) lua_error(L, "invalid background model");
        if (!fgname) lua_error(L, "invalid foreground model");

        define_model(name, 
                     new CompositeModel(make_model2d(bgname),
                                        make_model2d(fgname)));
        return 0;
    }

    // Create an image by overlaying several other images
    int def_overlay_image(lua_State *L)
    {
        const char *name = lua_tostring(L, 1); // name of the new model
        int n          = lua_getn(L, 2);

        Surface *sfc=0;
        SurfaceHandle overlay;

        for (int i=1; i<=n; i++) {
            lua_rawgeti(L, 2, i);
            const char *fname = lua_tostring(L, -1);
            lua_pop(L, 1);

            if (sfc == 0) {
                sfc = enigma::LoadImage(fname);
            } else {
                overlay = surface_cache.get(fname);
                sfc->blit(0,0,overlay.get());
            }
        }
        if (sfc != 0)
            define_model(name, new Image(sfc, 0,0));
        return 0;
    }

    lua::CFunction luafuncs[] = {
        { def_image,        "DefineImage" },
        { def_randmodel,    "DefineRandmodel" },
        { def_shmodel,      "DefineShmodel" },
        { def_anim,         "DefineAnim" },
        { def_composite,    "DefineComposite" },
        { def_overlay_image, "DefineOverlayImage", },
        { add_frame,        "AddFrame" },
        { set_tile_size,    "SetTileSize" },
        { 0,0 }
    };

    void init_lua_funcs()
    {
        lua::RegisterFuncs(luafuncs);
    }
}

//======================================================================
 // DRAWING CODE
//======================================================================

namespace
{
    void world_to_screen (const V3 & pos, int *x, int *y)
    {
        *x = int((pos[0] - ::scrollx)*tile_w);
        *y = int((pos[1] - ::scrolly)*tile_h);
    }

    void
    draw_layer(GridLayer layer, GC& gc,
               int xmin, int xmax, int ymin, int ymax, bool shade)
    {
        Array2<Field>::iterator iter;
        for (int y=ymin; y < ymax; y++)
        {
            iter = ::fields.row_begin(y) + xmin;
            for (int x = xmin; x<xmax; ++x, ++iter)
            {
                if (Model2d* m = iter->layers[layer])
                {
                    int xpos = (x - ::scrollx)*tile_w;
                    int ypos = (y - ::scrolly)*tile_h;
                    if (shade)
                        m->draw_shade(gc, xpos, ypos);
                    else
                        m->draw(gc, xpos, ypos);
                }
            }
        }
    }

    void draw_sprites(SpriteLayer l, bool shades, GC &gc)
    {
        for (SpriteList::iterator i=::sprites.begin(); i != ::sprites.end(); ++i)
        {
            Sprite *s = *i;
            if (!(s && s->layer == l && s->model))
                continue;

            if (s->model->is_garbage()) {
                delete s;
                *i = 0;
            } 
            else {
                int sx, sy;
                world_to_screen(s->pos, &sx, &sy);
                if (shades)
                    s->model->draw_shade(gc, sx, sy);
                else
                    s->model->draw(gc, sx, sy);
            }
        }
    }


    void draw_inventory(px::GC &gc, const px::Rect &r)
    {
        SurfaceHandle frame = surface_cache.get("inventory");
        SurfaceHandle logo = surface_cache.get("enigma_logo2_small");

        clip(gc, intersect(inventoryarea, r));

        blit(gc, inventoryarea.x, inventoryarea.y, frame.get());
        blit(gc, inventoryarea.x+5, inventoryarea.y+10, logo.get());
        if (text_display->is_active())
            text_display->draw(gc, r);
        else {
            for (unsigned i=0; i<invmodels.size(); ++i)
            {
                Model2d *m2 = invmodels[i];
                m2->draw(gc, itemarea.x + 10+i*40, itemarea.y + 11);
            }
        }
    }

    // This functions draws a region of the world on the screen.  The
    // region `r' denotes a region on the screen and not in the world.

    void redraw_region(px::GC &gc, const px::Rect &r)
    {
        int worldw = ::fields.width();
        int worldh = ::fields.height();

        // only redraw the gridpoints that lie at least partially inside `r'.
        int xmin = max(0, r.x/tile_w + scrollx-1);
        int ymin = max(0, r.y/tile_h + scrolly-1);
        int xmax = min(worldw, xmin + r.w/tile_w + 2);
        int ymax = min(worldh, ymin + r.h/tile_h + 2);


        // this is the region occupied by the world in screen coordinates
        Rect levelregion(-scrollx*tile_w, -scrolly*tile_h,
                         worldw*tile_w, worldh*tile_h);

        // Blank everything that is not inside the world
        set_color(gc, 130, 30, 130);
        if (!(display_flags & SHOW_FLOOR))
            box(gc, r);
        else {
            RectList rl;
            rl.push_back(r);
            rl.sub(levelregion);
            for (RectList::iterator i=rl.begin(); i != rl.end(); ++i)
                box(gc, *i);
        }

        clip(gc, intersect(levelregion, r));

        if (display_flags & SHOW_FLOOR)
            draw_layer(GRID_FLOOR, gc, xmin, xmax, ymin, ymax, false);

        if (display_flags & SHOW_ITEMS)
            draw_layer(GRID_ITEMS, gc, xmin, xmax, ymin, ymax, false);

        if (display_flags & SHOW_STONES)
            draw_layer(GRID_STONES, gc, xmin, xmax, ymin, ymax, true);

        if (display_flags & SHOW_SPRITES)
        {
            draw_sprites(SPRITE_ACTOR, true, gc);
            draw_sprites(SPRITE_ACTOR, false, gc);
        }
        if (display_flags & SHOW_STONES)
            draw_layer(GRID_STONES, gc, xmin, xmax, ymin, ymax, false);

        draw_sprites(SPRITE_EFFECT, false, gc);
        draw_inventory(gc, r);
    }
}

namespace {
    bool operator<(const Rect a, const Rect b) {
        return (a.x+(a.y<<16) < b.x+(b.y<<16)) || a.w<b.w || a.h<b.h;
    }
}


void
display::Redraw(Screen *screen, bool update_screenp)
{
    GC gc(screen);

    if (redraw_everything) {
        redraw_region(gc, screen->size());
        if (update_screenp)
            screen->update_all();
        redraw_everything = false;
    }
    else if (!changelist.empty()) {
        // remove duplicates
//         sort(changelist.begin(), changelist.end());
//         unsigned before=changelist.size();
        changelist.erase(unique(changelist.begin(), 
                                changelist.end()),
                         changelist.end());

        for (vector<Rect>::iterator i=changelist.begin();
             i != changelist.end(); ++i)
        {
            Rect r = Rect(tile_w * (i->x - ::scrollx), tile_h * (i->y - ::scrolly),
                          tile_w * i->w, tile_h * i->h);
            redraw_region(gc, r);
            screen->update_rect(r);
        }
    }
    // redraw inventory
    if (invchanged) {
        draw_inventory(gc, inventoryarea);
        screen->update_rect(inventoryarea);
        invchanged = false;
    }
    if (update_screenp)
        screen->flush_updates();
    changelist.clear();
}
