//======================================================================
// Copyright (C) 2002 Daniel Heck
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//  
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//  
// You should have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//======================================================================
#ifndef DISPLAY_HH
#define DISPLAY_HH

#include "fwd.hh"
#include "enigma.hh"
#include "px/math.hh"
#include "px/video.hh"
#include <string>

namespace display
{
    typedef const std::string cstring;

    // Animations can be prompted to invoke a callback on completion.
    // These callbacks must be derived from this class, and the method
    // being invoked is 'animation_finished'.
    //
    // Please note that you may not delete or replace models other
    // than the one that induced the callback from inside
    // 'animation_finished' -- use a timer or a flag to do this.
    class AnimCallback {
    public:
        virtual ~AnimCallback() {}
        virtual void animation_finished() = 0;
    };

    class Animation {
    public:
        virtual ~Animation() {}
        virtual double duration() const { return 0; }
        virtual void set_callback(AnimCallback *cb) {}
        virtual void reverse() {}
    };

    enum GridLayer 
    {
        GRID_FLOOR, GRID_ITEMS, GRID_STONES, GRID_COUNT
    };

    enum SpriteLayer
    {
        SPRITE_ACTOR, SPRITE_EFFECT, SPRITE_COUNT
    };

    enum DisplayFlags 
    {
        SHOW_FLOOR     = 0x01,
        SHOW_STONES    = 0x02,
        SHOW_ITEMS     = 0x04,
        SHOW_SHADES    = 0x08,
        SHOW_SPRITES   = 0x10,
	SHOW_TIME      = 0x20,
	SHOW_INVENTORY = 0x40,
        SHOW_ALL       = 0x7f
    };

    struct ModelId {
        GridLayer layer;
        int x, y;
        ModelId() {}
        ModelId(GridLayer l, int xx, int yy) : layer(l), x(xx),y(yy) {}
    };

    typedef unsigned int SpriteId;
}

// This part of the interface is implemented in display_2d.cc
namespace display
{
    void Init();
    void Shutdown();

    void NewWorld (int w, int h);
    void Redraw(px::Screen *screen, bool updatescreenp = true);
    void Tick(double dtime);

    //
    // Model management
    // 
    ModelId SetModel(GridLayer l, int x,int y, cstring &modelname);
    void KillModel(ModelId id);
    void KillModel(GridLayer l, int x, int y);

    SpriteId AddSprite (SpriteLayer l, const px::V3 &pos, cstring &modelname);
    void MoveSprite (SpriteId id, const px::V3 &newpos);
    void ReplaceSprite(SpriteId id, cstring &modelname);
    void KillSprite (SpriteId id);

    Animation *GetAnimation(ModelId id);
    Animation *GetAnimation(SpriteId id);

    void SetCallback(ModelId id, AnimCallback *cb);
    void SetCallback(SpriteId id, AnimCallback *cb);
    void ReverseAnimation(ModelId id);

    //
    // Inventory management
    //
    void AddInventoryItem(cstring &modelname);
    void RemoveInventoryItem(int idx);
    void RotateInventory();

    //
    // Sprite following
    // 
    enum FollowMode { 
        FOLLOW_SCROLLING,       // scroll the screen
        FOLLOW_SCREEN           // flip the screen region
    };
    void FollowSprite(SpriteId id);
    void SetFollowMode(FollowMode m);

    //
    // Text display
    // 
    enum TextMode {
        TEXT_SCROLLING,         // scroll text from right to left
        TEXT_2SECONDS,          // show text centered, for 2 seconds
        TEXT_STATIC             // show text centered, indefinitely
    };
    void ShowText(const std::string &str, TextMode m=TEXT_SCROLLING);
}
#endif
