/*
  mxDateTime -- Generic date/time types

  Copyright (c) 1997-2000, Marc-Andre Lemburg; mailto:mal@lemburg.com
  Copyright (c) 2000-2001, eGenix.com Software GmbH; mailto:info@egenix.com
  See the documentation for further copyright information or contact
  the author (mailto:mal@lemburg.com).

  XXX Update the "mixed" coercion hack when the number protocol changes
      to introduce new binary&ternary functions.

*/

/* Define this to aid in finding memory leaks */
/*#define MAL_MEM_DEBUG*/
/*#define MAL_DEBUG*/

/* Logging file used by debugging facility */
#ifndef MAL_DEBUG_OUTPUTFILE
# define MAL_DEBUG_OUTPUTFILE "mxDateTime.log"
#endif

/* We want all our symbols to be exported */
#define MX_BUILDING_MXDATETIME

/* Uncomment this to make the old interfaces available too */
/*#define OLD_INTERFACE*/

/* mx.DateTime can use its own API for querying the current time from
   the OS, or reuse the Python time.time() function. The latter is
   more portable, but slower. Define the following symbol to use the
   faster native API.  */
/*#define USE_FAST_GETCURRENTTIME*/

/* Some additional switches are needed on some platforms to make
   strptime() and timegm() available. */
#ifndef _GNU_SOURCE
# define _GNU_SOURCE 1
#endif

#include "mx.h"
#include "mxDateTime.h"

#ifdef USE_FAST_GETCURRENTTIME

/* Additional includes needed for native mxDateTime_GetCurrentTime()
   API */
# if defined(HAVE_SYS_TIME_H) && defined(TIME_WITH_SYS_TIME)
#  include <sys/time.h>
#  include <time.h>
# else
#  include <time.h>
# endif
# ifdef HAVE_UNISTD_H
#  include <unistd.h>
# endif

#endif


/* Version number: Major.Minor.Patchlevel */
#define MXDATETIME_VERSION "2.0.2"

/* The module makes use of two functions called strftime() and
   strptime() for the conversion between strings and date/time
   values. Since not all C compilers know about these functions,
   you can turn these features on/off be defining the following
   symbols (if you ran the Python configuration script then it will
   provide this information for you -- on other platforms than
   Unix you may have to define them manually). */
/*#define HAVE_STRFTIME*/
/*#define HAVE_STRPTIME*/
/*#define HAVE_TIMEGM*/

/* The start size for output from strftime. */
#define STRFTIME_OUTPUT_SIZE	1024

/* Define these to have the module use free lists (saves malloc calls) */
#define MXDATETIME_FREELIST
#define MXDATETIMEDELTA_FREELIST

/* Define this to enable the copy-protocol (__copy__, __deepcopy__) */
#define COPY_PROTOCOL

/* Define to have the seconds part rounded when assigning to tm_sec in
   tm structs; note that rounding can result in the seconds part to be
   displayed as 60 instead of wrapping to the next minute, hour,
   etc. */
/*#define ROUND_SECONDS_IN_TM_STRUCT*/

/* --- module helpers ----------------------------------------------------- */

/* Seconds in a day (as double) */
#define SECONDS_PER_DAY ((double) 86400.0)

/* Test for negativeness of doubles */
#define DOUBLE_IS_NEGATIVE(x) ((x) < (double) 0.0)

/* --- module doc-string -------------------------------------------------- */

static char *Module_docstring = 

 MXDATETIME_MODULE" -- Generic date/time types. Version "MXDATETIME_VERSION"\n\n"

 "Copyright (c) 1997-2000, Marc-Andre Lemburg; mailto:mal@lemburg.com\n"
 "Copyright (c) 2000-2001, eGenix.com Software GmbH; mailto:info@egenix.com\n\n"
 "                 All Rights Reserved\n\n"
 "See the documentation for further information on copyrights,\n"
 "or contact the author."
;

/* --- module globals ----------------------------------------------------- */

static PyObject *mxDateTime_Error;		/* Error Exception
						   object */
static PyObject *mxDateTime_RangeError;		/* RangeError
						   Exception object */

static PyObject *mxDateTime_GregorianCalendar;	/* String 'Gregorian' */
static PyObject *mxDateTime_JulianCalendar;	/* String 'Julian' */

static int mxDateTime_POSIXConform = 0;		/* Does the system use POSIX
						   time_t values ? */

/* Table with day offsets for each month (0-based, without and with leap) */
static int month_offset[2][13] = {
    { 0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334, 365 },
    { 0, 31, 60, 91, 121, 152, 182, 213, 244, 274, 305, 335, 366 }
};

/* Table of number of days in a month (0-based, without and with leap) */
static int days_in_month[2][12] = {
    { 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 },
    { 31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 }
};

/* Free lists for DateTime and DateTimeDelta objects */
#ifdef MXDATETIME_FREELIST
static mxDateTimeObject *mxDateTime_FreeList = NULL;
#endif
#ifdef MXDATETIMEDELTA_FREELIST
static mxDateTimeDeltaObject *mxDateTimeDelta_FreeList = NULL;
#endif

/* This must be a callable function that returns the current local
   time in Unix ticks. It is set by the mxDateTime/__init__.py module
   to the standard Python time.time() function. */
static PyObject *mxDateTime_nowapi = NULL;

/* --- forward declarations ----------------------------------------------- */

staticforward PyTypeObject mxDateTime_Type;
staticforward PyMethodDef mxDateTime_Methods[];

staticforward PyTypeObject mxDateTimeDelta_Type;
staticforward PyMethodDef mxDateTimeDelta_Methods[];

staticforward
PyObject *mxDateTimeDelta_FromDaysEx(long days,
				     double seconds);

staticforward
PyObject *mxDateTimeDelta_FromSeconds(double seconds);

/* --- internal macros ---------------------------------------------------- */

#define _mxDateTime_Check(v) \
        (((mxDateTimeObject *)(v))->ob_type == &mxDateTime_Type)

#define _mxDateTimeDelta_Check(v) \
        (((mxDateTimeDeltaObject *)(v))->ob_type == \
	 &mxDateTimeDelta_Type)

/* --- module helpers ----------------------------------------------------- */

/* Create an exception object, insert it into the module dictionary
   under the given name and return the object pointer; this is NULL in
   case an error occurred. base can be given to indicate the base
   object to be used by the exception object. It should be NULL
   otherwise */

static 
PyObject *insexc(PyObject *moddict,
		 char *name,
		 PyObject *base)
{
    PyObject *v;
    char fullname[256];
    
    sprintf(fullname,MXDATETIME_MODULE".%s",name);
    v = PyErr_NewException(fullname, base, NULL);
    if (v == NULL)
	return NULL;
    if (PyDict_SetItemString(moddict,name,v))
	return NULL;
    return v;
}

/* Helper for adding integer constants to a dictionary. Check for
   errors with PyErr_Occurred() */
static 
void insint(PyObject *dict,
	    char *name,
	    int value)
{
    PyObject *v = PyInt_FromLong((long)value);
    PyDict_SetItemString(dict, name, v);
    Py_XDECREF(v);
}

#if 0
/* Helper for adding string constants to a dictionary. Check for
   errors with PyErr_Occurred() */
static 
void insstr(PyObject *dict,
	    char *name,
	    char *value)
{
    PyObject *v = PyString_FromString(value);
    PyDict_SetItemString(dict, name, v);
    Py_XDECREF(v);
}
#endif

/* Helper for adding objects to dictionaries. Check for errors with
   PyErr_Occurred() */
static 
void insobj(PyObject *dict,
	    char *name,
	    PyObject *v)
{
    PyDict_SetItemString(dict, name, v);
    Py_XDECREF(v);
}

static
PyObject *notimplemented1(PyObject *v)
{
    Py_Error(PyExc_TypeError,
	     "operation not implemented");
 onError:
    return NULL;
}

#if 0
static
PyObject *notimplemented2(PyObject *v, PyObject *w)
{
    Py_Error(PyExc_TypeError,
	     "operation not implemented");
 onError:
    return NULL;
}

static
PyObject *notimplemented3(PyObject *u, PyObject *v, PyObject *w)
{
    Py_Error(PyExc_TypeError,
	     "operation not implemented");
 onError:
    return NULL;
}
#endif

/* --- DateTime Object -------------------------------------------------*/

/* --- allocation --- */

static
mxDateTimeObject *mxDateTime_New(void)
{
    mxDateTimeObject *datetime;

#ifdef MXDATETIME_FREELIST
    if (mxDateTime_FreeList) {
	datetime = mxDateTime_FreeList;
	mxDateTime_FreeList = *(mxDateTimeObject **)mxDateTime_FreeList;
	datetime->ob_type = &mxDateTime_Type;
	_Py_NewReference(datetime);
    }
    else
#endif 
	 {
	datetime = PyObject_NEW(mxDateTimeObject,&mxDateTime_Type);
	if (datetime == NULL)
	    return NULL;
    }

    /* Init vars */
    datetime->argument = 0;

    return datetime;
}

/* --- deallocation --- */

static
void mxDateTime_Free(mxDateTimeObject *datetime)
{
    Py_XDECREF(datetime->argument);
#ifdef MXDATETIME_FREELIST
    /* Append to free list */
    *(mxDateTimeObject **)datetime = mxDateTime_FreeList;
    mxDateTime_FreeList = datetime;
    _Py_ForgetReference(datetime);
#else
    PyMem_DEL(datetime);
#endif
}

/* --- internal functions --- */

#ifdef USE_FAST_GETCURRENTTIME

/* Returns the current time in Unix ticks.

   The function tries to use the gettimeofday() API in BSD systems and
   falls back to time() for all others.

   -1.0 is returned in case of an error.

*/

static
double mxDateTime_GetCurrentTime(void)
{
#ifdef HAVE_GETTIMEOFDAY
    struct timeval tv;

# ifdef GETTIMEOFDAY_NO_TZ
    if (!gettimeofday(&tv))
# else
    if (!gettimeofday(&tv, 0))
# endif
	return ((double)tv.tv_sec + (double)tv.tv_usec * 1e-6);
    else
	return -1.0;
#else
    time_t ticks;

    time(&ticks);
    return (double) ticks;
#endif
}

#else

/* Get the current time in Unix ticks. 

   This function reuses the time.time() of the Python time module,
   which is more portable but is slower than the above implementation.

*/

static
double mxDateTime_GetCurrentTime(void)
{
    double fticks;
    PyObject *v;

    /* Call the mxDateTime_nowapi function and use its return
       value as basis for the DateTime instance's value. */
    if (mxDateTime_nowapi == NULL)
	Py_Error(mxDateTime_Error,
		 "no current time API set");
    v = PyEval_CallObject(mxDateTime_nowapi, NULL);
    if (!v)
	goto onError;
    fticks = PyFloat_AsDouble(v);
    Py_DECREF(v);
    if (fticks == -1 && PyErr_Occurred())
	goto onError;

#if 0
    /* Round to the nearest micro-second: we use the same approach
       taken in the builtin round(). */
    fticks *= 1e6;
    if (DOUBLE_IS_NEGATIVE(fticks))
	fticks = ceil(fticks - 0.5);
    else
	fticks = floor(fticks + 0.5);
    fticks /= 1e6;
#endif    

    return fticks;

 onError:
    return -1.0;
}

#endif

/* This function checks whether the system uses the POSIX time_t rules
   (which do not support leap seconds) or a time package with leap
   second support enabled. Return 1 if it uses POSIX time_t values, 0
   otherwise.

   POSIX: 1986-12-31 23:59:59 UTC == 536457599

   With leap seconds:		  == 536457612

   (since there were 13 leapseconds in the years 1972-1985 according
   to the tz package available from ftp://elsie.nci.nih.gov/pub/)

*/

static
int mxDateTime_POSIX(void)
{
    time_t ticks = 536457599;
    struct tm *tm;

    memset(&tm,0,sizeof(tm));
    tm = gmtime(&ticks);
    if (tm == NULL)
	return 0;
    if (tm->tm_hour == 23 &&
	tm->tm_min == 59 &&
	tm->tm_sec == 59 &&
	tm->tm_mday == 31 &&
	tm->tm_mon == 11 &&
	tm->tm_year == 86)
	return 1;
    else
	return 0;
}

#ifndef HAVE_TIMEGM
/* Calculates the conversion of the datetime instance to Unix ticks.

   For instances pointing to localtime, localticks will hold the
   corresponding Unix ticks value. In case the instance points to GMT
   time, gmticks will hold the correct ticks value.

   In both cases, gmtoffset will hold the GMT offset (local-GMT).

   Returns -1 (and sets an exception) to indicate errors; 0
   otherwise. 

   Note:

   There's some integer rounding error in the mktime() function that
   triggers near MAXINT on Solaris. The error was reported by Joe Van
   Andel <vanandel@ucar.edu> among others:

    Ooops: 2038-01-18 22:52:31.00 t = 2147467951 diff = -4294857600.0

   On 64-bit Alphas running DEC OSF, Tony Ibbs <tony@lsl.co.uk>
   reports:

    Ooops: 1901-12-13 21:57:57.00 t = 2147487973 diff = -4294967296.0
    ...(the diffs stay the same)...
    Ooops: 1969-12-31 10:10:54.00 t = 4294917550 diff = -4294967296.0

   Note the years ! Some rollover is happening near 2^31-1 even
   though Alphas happen to use 64-bits. This could be a bug in this
   function or in DEC's mktime() implementation.

*/

static
int mxDateTime_CalcTicks(mxDateTimeObject *datetime,
			 double *localticks,
			 double *gmticks,
			 double *gmtoffset)
{
    struct tm tm;
    struct tm *gmt;
    time_t ticks;
    double offset;

    Py_Assert(datetime->calendar == MXDATETIME_GREGORIAN_CALENDAR,
	      mxDateTime_Error,
	      "can only convert the Gregorian calendar to ticks");
    
    /* Calculate floor()ed ticks value  */
    memset(&tm,0,sizeof(tm));
    tm.tm_hour = (int)datetime->hour;
    tm.tm_min = (int)datetime->minute;
    tm.tm_sec = (int)datetime->second;
    tm.tm_mday = (int)datetime->day;
    tm.tm_mon = (int)datetime->month - 1;
    tm.tm_year = (int)datetime->year - 1900;
    tm.tm_wday = ((int)datetime->day_of_week + 1) % 7;
    tm.tm_yday = (int)datetime->day_of_year - 1;
    tm.tm_isdst = -1; /* unkown */
    ticks = mktime(&tm);
    if (ticks == (time_t)-1) {
	/* XXX Hack to allow conversion during DST switching. */
        tm.tm_hour = 0;
	tm.tm_min = 0;
	tm.tm_sec = 0;
	ticks = mktime(&tm);
	if (ticks == (time_t)-1)
	    Py_Error(mxDateTime_Error,
		     "cannot convert value to a Unix ticks value");
	ticks += ((int)datetime->hour * 3600
		  + (int)datetime->minute * 60
		  + (int)datetime->second);
    }

    /* Add fraction for localticks */
    *localticks = ((double)ticks
		   + (datetime->abstime - floor(datetime->abstime)));
    
    /* Now compare local time and GMT time */
    gmt = gmtime(&ticks);
    if (gmt == NULL)
	Py_Error(mxDateTime_Error,
		 "cannot convert value to a Unix ticks value");

    /* Check whether we have the same day and prepare offset */
    if (gmt->tm_mday != tm.tm_mday) {
	double localdate = (tm.tm_year * 10000 + 
			    tm.tm_mon *  100 +
			    tm.tm_mday);
	double gmdate = (gmt->tm_year * 10000 +
			 gmt->tm_mon * 100 + 
			 gmt->tm_mday);
	if (localdate < gmdate)
	    offset = -SECONDS_PER_DAY;
	else
	    offset = SECONDS_PER_DAY;
    }
    else
	offset = 0.0;

    /* Calculate difference in seconds */
    offset += ((datetime->hour - gmt->tm_hour) * 3600.0
	       + (datetime->minute - gmt->tm_min) * 60.0
	       + (floor(datetime->second) - (double)gmt->tm_sec));
    *gmtoffset = offset;
    *gmticks = *localticks + offset;
    return 0;
    
 onError:
    return -1;
}
#endif

/* These functions work for positive *and* negative years for
   compilers which round towards zero and ones that always round down
   to the nearest integer. */

/* Return 1/0 iff year points to a leap year in calendar. */

static
int mxDateTime_Leapyear(register long year,
			int calendar)
{
    if (calendar == MXDATETIME_GREGORIAN_CALENDAR)
	return (year % 4 == 0) && ((year % 100 != 0) || (year % 400 == 0));
    else
	return (year % 4 == 0);
}

/* Return the day of the week for the given absolute date. */

static
int mxDateTime_DayOfWeek(register long absdate)
{
    int day_of_week;

    if (absdate >= 1)
	day_of_week = (absdate - 1) % 7;
    else
	day_of_week = 6 - ((-absdate) % 7);
    return day_of_week;
}

/* Return the year offset, that is the absolute date of the day
   31.12.(year-1) in the given calendar.

   Note:
   For the Julian calendar we shift the absdate (which is measured
   using the Gregorian Epoch) value by two days because the Epoch
   (0001-01-01) in the Julian calendar lies 2 days before the Epoch in
   the Gregorian calendar.

*/

static 
int mxDateTime_YearOffset(register long year,
			  int calendar)
{
    year--;
    if (calendar == MXDATETIME_GREGORIAN_CALENDAR) {
	if (year >= 0 || -1/4 == -1)
	    return year*365 + year/4 - year/100 + year/400;
	else
	    return year*365 + (year-3)/4 - (year-99)/100 + (year-399)/400;
    }
    else if (calendar == MXDATETIME_JULIAN_CALENDAR) {
	if (year >= 0 || -1/4 == -1)
	    return year*365 + year/4 - 2;
	else
	    return year*365 + (year-3)/4 - 2;
    }
    Py_Error(mxDateTime_Error,
	     "unknown calendar");
 onError:
    return -1;
}

/* Sets the date part of the DateTime object using the indicated
   calendar. 

   XXX This could also be done using some integer arithmetics rather
       than with this iterative approach...

*/

static
int mxDateTime_SetFromAbsDate(register mxDateTimeObject *datetime,
			      long absdate,
			      int calendar)
{
    register long year;
    long yearoffset;
    int leap,dayoffset;
    int *monthoffset;

    DPRINTF("mxDateTime_SetFromAbsDate(datetime=%x,absdate=%i,calendar=%i)\n",
	    datetime,absdate,calendar);

    /* Approximate year */
    if (calendar == MXDATETIME_GREGORIAN_CALENDAR)
	year = (long)(((double)absdate) / 365.2425);
    else if (calendar == MXDATETIME_JULIAN_CALENDAR)
	year = (long)(((double)absdate) / 365.25);
    else
	Py_Error(mxDateTime_Error,
		 "unknown calendar");
    if (absdate > 0)
	year++;

    /* Apply corrections to reach the correct year */
    while (1) {
	/* Calculate the year offset */
	yearoffset = mxDateTime_YearOffset(year,calendar);
	if (yearoffset == -1 && PyErr_Occurred())
	    goto onError;
	DPRINTF(" trying year = %i yearoffset = %i \n",
		(int)year,yearoffset);

	/* Backward correction: absdate must be greater than the
	   yearoffset */
	if (yearoffset >= absdate) {
	    year--;
	    DPRINTF(" backward correction\n");
	    continue;
	}

	dayoffset = absdate - yearoffset;
	leap = mxDateTime_Leapyear(year,calendar);

	/* Forward correction: non leap years only have 365 days */
	if (dayoffset > 365 && !leap) {
	    year++;
	    DPRINTF(" forward correction\n");
	    continue;
	}
	DPRINTF(" using year = %i leap = %i dayoffset = %i\n",
		year,leap,dayoffset);

	break;
    }

    datetime->year = year;
    datetime->calendar = calendar;

    /* Now iterate to find the month */
    monthoffset = month_offset[leap];
    {
	register int month;
	
	for (month = 1; month < 13; month++)
	    if (monthoffset[month] >= dayoffset)
		break;
	datetime->month = month;
	datetime->day = dayoffset - month_offset[leap][month-1];
    }
    
    datetime->day_of_week = mxDateTime_DayOfWeek(absdate);
    datetime->day_of_year = dayoffset;
    
    return 0;

 onError:
    return -1;
}

/* Sets the time part of the DateTime object. */

static
int mxDateTime_SetFromAbsTime(mxDateTimeObject *datetime,
			      double abstime)
{
    int inttime;
    int hour,minute;
    double second;

    DPRINTF("mxDateTime_SetFromAbsTime(datetime=%x,abstime=%f)\n",
	    datetime,abstime);

    inttime = (int)abstime;
    hour = inttime / 3600;
    minute = (inttime % 3600) / 60;
    second = abstime - (double)(hour*3600 + minute*60);

    datetime->hour = hour;
    datetime->minute = minute;
    datetime->second = second;

    return 0;
}

/* Set the instance's value using the given date and time. calendar
   may be set to the flags: MXDATETIME_GREGORIAN_CALENDAR,
   MXDATETIME_JULIAN_CALENDAR to indicate the calendar to be used. */

static
int mxDateTime_SetFromDateAndTime(mxDateTimeObject *datetime,
				  int year,
				  int month,
				  int day,
				  int hour,
				  int minute,
				  double second,
				  int calendar)
{
    double comdate;
    
    if (datetime == NULL) {
	PyErr_BadInternalCall();
	goto onError;
    }

    /* Calculate the absolute date */
    {
	int leap;
	long yearoffset,absdate;

	/* Range check */
	Py_AssertWithArg(year > -(LONG_MAX / 366) && year < (LONG_MAX / 366),
			 mxDateTime_RangeError,
			 "year out of range: %i",
			 year);

	/* Is it a leap year ? */
	leap = mxDateTime_Leapyear(year,calendar);

	/* Negative month values indicate months relative to the years end */
	if (month < 0)
	    month += 13;
	Py_AssertWithArg(month >= 1 && month <= 12,
			 mxDateTime_RangeError,
			 "month out of range (1-12): %i",
			 month);

	/* Negative values indicate days relative to the months end */
	if (day < 0)
	    day += days_in_month[leap][month - 1] + 1;
	Py_AssertWithArg(day >= 1 && day <= days_in_month[leap][month - 1],
			 mxDateTime_RangeError,
			 "day out of range: %i",
			 day);

	yearoffset = mxDateTime_YearOffset(year,calendar);
	if (yearoffset == -1 && PyErr_Occurred())
	    goto onError;

	absdate = day + month_offset[leap][month - 1] + yearoffset;

	DPRINTF("mxDateTime_SetFromDateAndTime("
		"datetime=%x year=%i month=%i day=%i "
		"hour=%i minute=%i second=%f calendar=%i)\n",
		datetime,year,month,day,hour,minute,second,calendar);
	DPRINTF("mxDateTime_SetFromDateAndTime: "
		"yearoffset=%li leap=%i absdate=%li\n",
		yearoffset,leap,absdate);

	datetime->absdate = absdate;
	
	datetime->year = year;
	datetime->month = month;
	datetime->day = day;

	datetime->day_of_week = mxDateTime_DayOfWeek(absdate);
	datetime->day_of_year = (short)(absdate - yearoffset);

	datetime->calendar = calendar;

	comdate = (double)absdate - 693594.0;
    }

    /* Calculate the absolute time */
    {
	Py_AssertWithArg(hour >= 0 && hour <= 23,
			 mxDateTime_RangeError,
			 "hour out of range (0-23): %i",
			 hour);
	Py_AssertWithArg(minute >= 0 && minute <= 59,
			 mxDateTime_RangeError,
			 "minute out of range (0-59): %i",
			 minute);
	Py_AssertWithArg(second >= (double)0.0 && 
			 (second < (double)60.0 || 
			  (hour == 23 && minute == 59 && 
			   second < (double)61.0)),
			 mxDateTime_RangeError,
			 "second out of range (0.0 - <60.0; <61.0 for 23:59): %f",
			 second);

	datetime->abstime = (double)(hour*3600 + minute*60) + second;

	datetime->hour = hour;
	datetime->minute = minute;
	datetime->second = second;

	if (DOUBLE_IS_NEGATIVE(comdate))
	    comdate -= datetime->abstime / SECONDS_PER_DAY;
	else
	    comdate += datetime->abstime / SECONDS_PER_DAY;
	datetime->comdate = comdate;
    }
    return 0;
 onError:
    return -1;
}

/* Set the instance's value using the given absolute date and
   time. The calendar used is the Gregorian. */

static
int mxDateTime_SetFromAbsDateTime(mxDateTimeObject *datetime,
				  long absdate,
				  double abstime,
				  int calendar)
{
    if (datetime == NULL) {
	PyErr_BadInternalCall();
	goto onError;
    }
    
    DPRINTF("mxDateTime_SetFromAbsDateTime(datetime=%x,"
	    "absdate=%i,abstime=%f,calendar=%i)\n",
	    datetime,absdate,abstime,calendar);

    /* Bounds check */
    Py_AssertWithArg(abstime >= 0.0 && abstime <= SECONDS_PER_DAY,
		     mxDateTime_RangeError,
		     "abstime out of range (0.0 - 86400.0): %f",
		     abstime);

    datetime->absdate = absdate;
    datetime->abstime = abstime;
    
    /* Calculate COM date */
    {
	register double comdate;
	
	comdate = (double)(datetime->absdate - 693594);
	if (DOUBLE_IS_NEGATIVE(comdate))
	    comdate -= datetime->abstime / SECONDS_PER_DAY;
	else
	    comdate += datetime->abstime / SECONDS_PER_DAY;
	datetime->comdate = comdate;
    }

    /* Calculate the date */
    if (mxDateTime_SetFromAbsDate(datetime,
				  absdate,
				  calendar))
	goto onError;
    
    /* Calculate the time */
    if (mxDateTime_SetFromAbsTime(datetime,
				  abstime))
	goto onError;

    return 0;
 onError:
    return -1;
}

/* Set the instance's value using the given Windows COM date.  The
   calendar used is the Gregorian. */

static
int mxDateTime_SetFromCOMDate(mxDateTimeObject *datetime,
			      double comdate)
{
    long absdate;
    double abstime;

    if (datetime == NULL) {
	PyErr_BadInternalCall();
	goto onError;
    }
    
    datetime->comdate = comdate;

    /* XXX should provide other means to calculate the broken down
       values for these huge values. */
    Py_AssertWithArg(-(double)LONG_MAX <= comdate &&
		     comdate <= (double)LONG_MAX,
		     mxDateTime_RangeError,
		     "DateTime COM date out of range: %f",
		     comdate);

    absdate = (long)comdate;
    abstime = (comdate - (double)absdate) * SECONDS_PER_DAY;
    if (DOUBLE_IS_NEGATIVE(abstime))
	abstime = -abstime;
    absdate += 693594;

    DPRINTF("mxDateTime_SetFromCOMDate: absdate=%li abstime=%f\n",
	    absdate,abstime);

    datetime->absdate = absdate;
    datetime->abstime = abstime;
    
    /* Calculate the date */
    if (mxDateTime_SetFromAbsDate(datetime,
				  absdate,
				  MXDATETIME_GREGORIAN_CALENDAR))
	goto onError;
    
    /* Calculate the time */
    if (mxDateTime_SetFromAbsTime(datetime,
				  abstime))
	goto onError;

    return 0;
 onError:
    return -1;
}

/* --- API functions --- */

static
PyObject *mxDateTime_FromDateAndTime(int year,
				     int month,
				     int day,
				     int hour,
				     int minute,
				     double second)
{
    mxDateTimeObject *datetime;
    
    datetime = mxDateTime_New();
    if (datetime == NULL)
	return NULL;
    if (mxDateTime_SetFromDateAndTime(datetime,
				      year,month,day,
				      hour,minute,second,
				      MXDATETIME_GREGORIAN_CALENDAR))
	goto onError;

    return (PyObject *)datetime;
 onError:
    mxDateTime_Free(datetime);
    return NULL;
}

/* Alias */
#define mxDateTime_FromGregorianDateAndTime mxDateTime_FromDateAndTime

static
PyObject *mxDateTime_FromJulianDateAndTime(int year,
					   int month,
					   int day,
					   int hour,
					   int minute,
					   double second)
{
    mxDateTimeObject *datetime;
    
    datetime = mxDateTime_New();
    if (datetime == NULL)
	return NULL;
    if (mxDateTime_SetFromDateAndTime(datetime,
				      year,month,day,
				      hour,minute,second,
				      MXDATETIME_JULIAN_CALENDAR))
	goto onError;

    return (PyObject *)datetime;
 onError:
    mxDateTime_Free(datetime);
    return NULL;
}

static
PyObject *mxDateTime_FromAbsDateAndTime(long absdate,
					double abstime)
{
    mxDateTimeObject *datetime;
    
    datetime = mxDateTime_New();
    if (datetime == NULL)
	return NULL;
    if (mxDateTime_SetFromAbsDateTime(datetime,
				      absdate,
				      abstime,
				      MXDATETIME_GREGORIAN_CALENDAR))
	goto onError;

    return (PyObject *)datetime;
 onError:
    mxDateTime_Free(datetime);
    return NULL;
}

static
PyObject *mxDateTime_FromAbsDateTime(long absdate,
				     double abstime,
				     int calendar)
{
    mxDateTimeObject *datetime;
    
    datetime = mxDateTime_New();
    if (datetime == NULL)
	return NULL;
    if (mxDateTime_SetFromAbsDateTime(datetime,
				      absdate,
				      abstime,
				      calendar))
	goto onError;

    return (PyObject *)datetime;
 onError:
    mxDateTime_Free(datetime);
    return NULL;
}

/* Creates a new DateTime instance using datetime as basis by adding
   the given offsets to the value of datetime and then re-normalizing
   them.

   The resulting DateTime instance will use the same calendar as
   datetime.

*/

static
PyObject *mxDateTime_FromDateTimeAndOffset(mxDateTimeObject *datetime,
					   long absdate_offset,
					   double abstime_offset)
{
    mxDateTimeObject *dt;
    long absdate = datetime->absdate;
    double abstime = datetime->abstime;
    long days;

    absdate += absdate_offset;
    abstime += abstime_offset;

    /* Normalize */
    if (abstime < 0 && abstime >= -SECONDS_PER_DAY) {
	abstime += SECONDS_PER_DAY;
	absdate -= 1;
    }
    if (abstime >= SECONDS_PER_DAY && abstime < 2*SECONDS_PER_DAY) {
	abstime -= SECONDS_PER_DAY;
	absdate += 1;
    }
    while (DOUBLE_IS_NEGATIVE(abstime)) {
	days = (long)(-abstime / SECONDS_PER_DAY) + 1;
	abstime += days * SECONDS_PER_DAY;
	absdate -= days;
    }
    while (abstime >= SECONDS_PER_DAY) {
	days = (long)(abstime / SECONDS_PER_DAY);
	abstime -= days * SECONDS_PER_DAY;
	absdate += days;
    }
    
    dt = mxDateTime_New();
    if (dt == NULL)
	return NULL;
    if (mxDateTime_SetFromAbsDateTime(dt,
				      absdate,
				      abstime,
				      datetime->calendar))
	goto onError;

    return (PyObject *)dt;

 onError:
    mxDateTime_Free(dt);
    return NULL;
}

static
PyObject *mxDateTime_FromAbsDays(double absdays)
{
    mxDateTimeObject *datetime;
    long absdate;
    double abstime,fabsdays;
    
    datetime = mxDateTime_New();
    if (datetime == NULL)
	return NULL;
    fabsdays = floor(absdays);
    Py_AssertWithArg(fabsdays > -LONG_MAX && fabsdays < LONG_MAX,
		     mxDateTime_RangeError,
		     "absdays out of range: %f",
		     absdays);
    absdate = (long)fabsdays + 1;
    abstime = (absdays - fabsdays) * SECONDS_PER_DAY;
    if (mxDateTime_SetFromAbsDateTime(datetime,
				      absdate,
				      abstime,
				      MXDATETIME_GREGORIAN_CALENDAR))
	goto onError;

    return (PyObject *)datetime;

 onError:
    mxDateTime_Free(datetime);
    return NULL;
}

static
PyObject *mxDateTime_FromTuple(PyObject *v)
{
    mxDateTimeObject *datetime = 0;
    int year,month,day,hour,minute;
    double second;

    if (!PyTuple_Check(v)) {
	PyErr_BadInternalCall();
	return NULL;
    }
    if (!PyArg_ParseTuple(v,
	"iiiiid;need a date/time 6-tuple (year,month,day,hour,minute,second)",
			  &year,&month,&day,&hour,&minute,&second))
	return NULL;
    
    datetime = mxDateTime_New();
    if (datetime == NULL)
	return NULL;
    if (mxDateTime_SetFromDateAndTime(datetime,
				      year,month,day,
				      hour,minute,second,
				      MXDATETIME_GREGORIAN_CALENDAR))
	goto onError;

    return (PyObject *)datetime;

 onError:
    mxDateTime_Free(datetime);
    return NULL;
}

static
PyObject *mxDateTime_FromTmStruct(struct tm *tm)
{
    mxDateTimeObject *datetime = 0;
    
    datetime = mxDateTime_New();
    if (datetime == NULL)
	return NULL;
    
    if (mxDateTime_SetFromDateAndTime(datetime,
				      tm->tm_year + 1900,
				      tm->tm_mon + 1,
				      tm->tm_mday,
				      tm->tm_hour,
				      tm->tm_min,
				      (double)tm->tm_sec,
				      MXDATETIME_GREGORIAN_CALENDAR))
	goto onError;

    return (PyObject *)datetime;

 onError:
    mxDateTime_Free(datetime);
    return NULL;
}

static
PyObject *mxDateTime_FromTicks(double ticks)
{
    mxDateTimeObject *datetime = 0;
    struct tm *tm;
    double seconds;
    time_t tticks = (time_t)ticks;
    
    datetime = mxDateTime_New();
    if (datetime == NULL)
	return NULL;

    /* Conversion is done to local time */
    tm = localtime(&tticks);
    /* Add fraction */
    seconds = floor((double)tm->tm_sec) + (ticks - floor(ticks));

    if (mxDateTime_SetFromDateAndTime(datetime,
				      tm->tm_year + 1900,
				      tm->tm_mon + 1,
				      tm->tm_mday,
				      tm->tm_hour,
				      tm->tm_min,
				      seconds,
				      MXDATETIME_GREGORIAN_CALENDAR))
	goto onError;

    return (PyObject *)datetime;

 onError:
    mxDateTime_Free(datetime);
    return NULL;
}

static
PyObject *mxDateTime_FromGMTicks(double ticks)
{
    mxDateTimeObject *datetime = 0;
    struct tm *tm;
    double seconds;
    time_t tticks = (time_t)ticks;
    
    datetime = mxDateTime_New();
    if (datetime == NULL)
	return NULL;
    /* Conversion is done to GMT time */
    tm = gmtime(&tticks);
    /* Add fraction */
    seconds = floor((double)tm->tm_sec) + (ticks - floor(ticks));
    if (mxDateTime_SetFromDateAndTime(datetime,
				      tm->tm_year + 1900,
				      tm->tm_mon + 1,
				      tm->tm_mday,
				      tm->tm_hour,
				      tm->tm_min,
				      seconds,
				      MXDATETIME_GREGORIAN_CALENDAR))
	goto onError;

    return (PyObject *)datetime;

 onError:
    mxDateTime_Free(datetime);
    return NULL;
}

static
PyObject *mxDateTime_FromCOMDate(double comdate)
{
    mxDateTimeObject *datetime = 0;

    datetime = mxDateTime_New();
    if (datetime == NULL)
	return NULL;
    if (mxDateTime_SetFromCOMDate(datetime,comdate))
	goto onError;

    return (PyObject *)datetime;

 onError:
    mxDateTime_Free(datetime);
    return NULL;
}

static
struct tm *mxDateTime_AsTmStruct(mxDateTimeObject *datetime,
				 struct tm *tm)
{
    memset(tm,0,sizeof(tm));
    tm->tm_hour = (int)datetime->hour;
    tm->tm_min = (int)datetime->minute;
#if ROUND_SECONDS_IN_TM_STRUCT
    tm->tm_sec = (int)(datetime->second + 0.5); /* Round the value */
#else
    tm->tm_sec = (int)datetime->second;
#endif
    tm->tm_mday = (int)datetime->day;
    tm->tm_mon = (int)datetime->month - 1;
    tm->tm_year = (int)datetime->year - 1900;
    tm->tm_wday = ((int)datetime->day_of_week + 1) % 7;
    tm->tm_yday = (int)datetime->day_of_year - 1;
    tm->tm_isdst = -1; /* unkown */
    return tm;
}

static
double mxDateTime_AsCOMDate(mxDateTimeObject *datetime)
{
    return datetime->comdate;
}

/* This global is set to
   -1 if mktime() auto-corrects the value of the DST flag to whatever the
      value should be for the given point in time (which is bad)
    0 if the global has not yet been initialized
    1 if mktime() does not correct the value and returns proper values
 */

static int mktime_works = 0;

static
int init_mktime_works(void)
{
    struct tm tm;
    time_t a,b;
    
    /* Does mktime() in general and specifically DST = -1 work ? */
    memset(&tm,0,sizeof(tm));
    tm.tm_mday = 1;
    tm.tm_mon = 5;
    tm.tm_year = 98;
    tm.tm_isdst = -1;
    a = mktime(&tm);
    Py_Assert(a != (time_t)-1,
	      PyExc_SystemError,
	      "mktime() returned an error (June)");
    memset(&tm,0,sizeof(tm));
    tm.tm_mday = 1;
    tm.tm_mon = 0;
    tm.tm_year = 98;
    tm.tm_isdst = -1;
    a = mktime(&tm);
    Py_Assert(a != (time_t)-1,
	      PyExc_SystemError,
	      "mktime() returned an error (January)");

    /* Some mktime() implementations return (time_t)-1 when setting
       DST to anything other than -1. Others adjust DST without
       looking at the given setting. */

    /* a = (Summer, DST = 0) */
    memset(&tm,0,sizeof(tm));
    tm.tm_mday = 1;
    tm.tm_mon = 5;
    tm.tm_year = 98;
    tm.tm_isdst = 0;
    a = mktime(&tm);
    if (a == (time_t)-1) {
	mktime_works = -1;
	return 0;
    }

    /* b = (Summer, DST = 1) */
    memset(&tm,0,sizeof(tm));
    tm.tm_mday = 1;
    tm.tm_mon = 5;
    tm.tm_year = 98;
    tm.tm_isdst = 1;
    b = mktime(&tm);
    if (a == (time_t)-1 || a == b) {
	mktime_works = -1;
	return 0;
    }
    
    /* a = (Winter, DST = 0) */
    memset(&tm,0,sizeof(tm));
    tm.tm_mday = 1;
    tm.tm_mon = 0;
    tm.tm_year = 98;
    tm.tm_isdst = 0;
    a = mktime(&tm);
    if (a == (time_t)-1) {
	mktime_works = -1;
	return 0;
    }

    /* b = (Winter, DST = 1) */
    memset(&tm,0,sizeof(tm));
    tm.tm_mday = 1;
    tm.tm_mon = 0;
    tm.tm_year = 98;
    tm.tm_isdst = 1;
    b = mktime(&tm);
    if (a == (time_t)-1 || a == b) {
	mktime_works = -1;
	return 0;
    }
    
    mktime_works = 1;
    return 0;
 onError:
    return -1;
}

/* Returns the ticks value for datetime assuming it stores a datetime
   value in local time. 
   
   offsets is subtracted from the resulting ticks value (this can be
   used to implement DST handling). 

   dst is passed to the used mktime() C lib API and can influence the
   calculation: dst == 1 means that the datetime value should be
   interpreted with DST on, dst == 0 with DST off. Note that this
   doesn't work on all platforms. dst == -1 means: use the DST value
   in affect at the given point in time.

*/

static
double mxDateTime_AsTicksWithOffset(mxDateTimeObject *datetime,
				    double offset,
				    int dst)
{
    struct tm tm;
    time_t tticks;
    double ticks;
    
    Py_Assert(datetime->calendar == MXDATETIME_GREGORIAN_CALENDAR,
	      mxDateTime_Error,
	      "can only convert the Gregorian calendar to ticks");
    
    memset(&tm,0,sizeof(tm));
    tm.tm_hour = (int)datetime->hour;
    tm.tm_min = (int)datetime->minute;
    tm.tm_sec = (int)datetime->second;
    tm.tm_mday = (int)datetime->day;
    tm.tm_mon = (int)datetime->month - 1;
    tm.tm_year = (int)datetime->year - 1900;
    tm.tm_wday = ((int)datetime->day_of_week + 1) % 7;
    tm.tm_yday = (int)datetime->day_of_year - 1;
    tm.tm_isdst = dst;
    /* mktime uses local time ! */
    tticks = mktime(&tm);
    Py_Assert(tticks != (time_t)-1,
	      mxDateTime_Error,
	      "cannot convert value to a time value");
    /* Check if mktime messes up DST */
    if (dst >= 0 && mktime_works <= 0) {
	if (mktime_works == 0) {
	    if (init_mktime_works() < 0)
		goto onError;
	}
	if (mktime_works < 0)
	    Py_Error(PyExc_SystemError,
	       "mktime() doesn't support setting DST to anything but -1");
    }
    /* Add fraction and turn into a double and subtract offset */
    ticks = (double)tticks
	    + (datetime->abstime - floor(datetime->abstime))
	    - offset;
    return ticks;
 onError:
    return -1.0;
}

static
double mxDateTime_AsTicks(mxDateTimeObject *datetime)
{
    return mxDateTime_AsTicksWithOffset(datetime,0,-1);
}

/* Returns the ticks value for datetime assuming it stores a UTC
   datetime value. 

   offsets is subtracted from the resulting ticks value before
   returning it. This is useful to implement time zone handling.

*/

static
double mxDateTime_AsGMTicksWithOffset(mxDateTimeObject *datetime,
				      double offset)
{
    Py_Assert(datetime->calendar == MXDATETIME_GREGORIAN_CALENDAR,
	      mxDateTime_Error,
	      "can only convert the Gregorian calendar to ticks");

    /* For POSIX style calculations there's nothing much to do... */
    if (mxDateTime_POSIXConform) {
	return ((datetime->absdate - 719163) * SECONDS_PER_DAY 
		+ datetime->abstime
		- offset);
    }

#ifdef HAVE_TIMEGM
    {
	/* Use timegm() API */
	struct tm tm;
	time_t tticks;

	/* Use timegm() if not POSIX conform: the time package knows about
	   leap seconds so we use that information too. */
	memset(&tm,0,sizeof(tm));
	tm.tm_hour = (int)datetime->hour;
	tm.tm_min = (int)datetime->minute;
	tm.tm_sec = (int)datetime->second;
	tm.tm_mday = (int)datetime->day;
	tm.tm_mon = (int)datetime->month - 1;
	tm.tm_year = (int)datetime->year - 1900;
	tm.tm_wday = ((int)datetime->day_of_week + 1) % 7;
	tm.tm_yday = (int)datetime->day_of_year - 1;
	tm.tm_isdst = 0;
	/* timegm uses UTC ! */
	tticks = timegm(&tm);
	Py_Assert(tticks != (time_t)-1,
		  mxDateTime_Error,
		  "cannot convert value to a time value");
	/* Add fraction and turn into a double */
	return ((double)tticks
		+ (datetime->abstime - floor(datetime->abstime))
		- offset);
    }
#else
    {
	/* Work around with a trick... */
	double localticks,gmticks,gmtoffset;

	if (mxDateTime_CalcTicks(datetime,
				 &localticks,&gmticks,&gmtoffset))
	    goto onError;
	return gmticks - offset;
    }
#endif

 onError:
    return -1.0;
}

static
double mxDateTime_AsGMTicks(mxDateTimeObject *datetime)
{
    return mxDateTime_AsGMTicksWithOffset(datetime,0);
}

/* Returns the UTC offset at the given time; assumes local time is
   stored in the instance. */

static
double mxDateTime_GMTOffset(mxDateTimeObject *datetime)
{
    double gmticks,ticks;
    
    gmticks = mxDateTime_AsGMTicks(datetime);
    if (gmticks == -1.0 && PyErr_Occurred())
	goto onError;
    ticks = mxDateTime_AsTicksWithOffset(datetime,0,-1);
    if (ticks == -1.0 && PyErr_Occurred())
	goto onError;
    return gmticks - ticks;

 onError:
    return -1.0;
}

/* Return the instance's value in absolute days: days since 0001-01-01
   0:00:00 using fractions for parts of a day. */

static
double mxDateTime_AsAbsDays(mxDateTimeObject *datetime)
{
    return ((double)(datetime->absdate - 1) + 
	    datetime->abstime / SECONDS_PER_DAY);
}

/* Return broken down values of the instance. This call returns the
   values as stored in the instance regardeless of the used
   calendar. */

static
int mxDateTime_BrokenDown(mxDateTimeObject *datetime,
			  long *year,
			  int *month,
			  int *day,
			  int *hour,
			  int *minute,
			  double *second)
{
    if (year)
	*year = (long)datetime->year;
    if (month)
	*month = (int)datetime->month;
    if (day)
	*day = (int)datetime->day;
    if (hour)
	*hour = (int)datetime->hour;
    if (minute)
	*minute = (int)datetime->minute;
    if (second)
	*second = (double)datetime->second;
    return 0;
}

/* Return the instance's value as broken down values using the Julian
   calendar. */

static
int mxDateTime_AsJulianDate(mxDateTimeObject *datetime,
			    long *pyear,
			    int *pmonth,
			    int *pday,
			    int *phour,
			    int *pminute,
			    double *psecond,
			    int *pday_of_week,
			    int *pday_of_year)
{
    long absdate = datetime->absdate;
    int year,month,day,dayoffset;

    /* Get the date in the Julian calendar */
    if (datetime->calendar != MXDATETIME_JULIAN_CALENDAR) {
	mxDateTimeObject temp;

	/* Recalculate the date from the absdate value */
	if (mxDateTime_SetFromAbsDate(&temp,
				      absdate,
				      MXDATETIME_JULIAN_CALENDAR))
	    goto onError;
	year = temp.year;
	month = temp.month;
	day = temp.day;
	dayoffset = temp.day_of_year;
    }
    else {
	year = datetime->year;
	month = datetime->month;
	day = datetime->day;
	dayoffset = datetime->day_of_year;
    }
    
    if (pyear)
	*pyear = (long)year;
    if (pmonth)
	*pmonth = (int)month;
    if (pday)
	*pday = (int)day;

    if (phour)
	*phour = (int)datetime->hour;
    if (pminute)
	*pminute = (int)datetime->minute;
    if (psecond)
	*psecond = (double)datetime->second;

    if (pday_of_week)
	*pday_of_week = mxDateTime_DayOfWeek(absdate);
    if (pday_of_year)
	*pday_of_year = (int)dayoffset;

    return 0;

 onError:
    return -1;
}

/* Return the instance's value as broken down values using the Gregorian
   calendar. */

static
int mxDateTime_AsGregorianDate(mxDateTimeObject *datetime,
			    long *pyear,
			    int *pmonth,
			    int *pday,
			    int *phour,
			    int *pminute,
			    double *psecond,
			    int *pday_of_week,
			    int *pday_of_year)
{
    long absdate = datetime->absdate;
    int year,month,day,dayoffset;

    /* Recalculate the date in the Gregorian calendar */
    if (datetime->calendar != MXDATETIME_GREGORIAN_CALENDAR) {
	mxDateTimeObject temp;

	/* Recalculate the date  from the absdate value */
	if (mxDateTime_SetFromAbsDate(&temp,
				      absdate,
				      MXDATETIME_GREGORIAN_CALENDAR))
	    goto onError;
	year = temp.year;
	month = temp.month;
	day = temp.day;
	dayoffset = temp.day_of_year;
    }
    else {
	year = datetime->year;
	month = datetime->month;
	day = datetime->day;
	dayoffset = datetime->day_of_year;
    }
    
    if (pyear)
	*pyear = (long)year;
    if (pmonth)
	*pmonth = (int)month;
    if (pday)
	*pday = (int)day;

    if (phour)
	*phour = (int)datetime->hour;
    if (pminute)
	*pminute = (int)datetime->minute;
    if (psecond)
	*psecond = (double)datetime->second;

    if (pday_of_week)
	*pday_of_week = mxDateTime_DayOfWeek(absdate);
    if (pday_of_year)
	*pday_of_year = (int)dayoffset;

    return 0;

 onError:
    return -1;
}

/* Returns a Python string containing the locale's timezone name for
   the given DateTime instance (assuming it refers to local time).
   "???"  is returned in case it cannot be determined.  */

static
PyObject *mxDateTime_TimezoneString(mxDateTimeObject *datetime)
{
    struct tm tm;
    time_t ticks;
    char tz[255];

    if (datetime->calendar != MXDATETIME_GREGORIAN_CALENDAR)
	return PyString_FromString("???");
#ifndef HAVE_STRFTIME
    return PyString_FromString("???");
#else
    memset(&tm,0,sizeof(tm));
    tm.tm_hour = (int)datetime->hour;
    tm.tm_min = (int)datetime->minute;
    tm.tm_sec = (int)datetime->second;
    tm.tm_mday = (int)datetime->day;
    tm.tm_mon = (int)datetime->month - 1;
    tm.tm_year = (int)datetime->year - 1900;
    tm.tm_isdst = -1;
    ticks = mktime(&tm);
    if (ticks == (time_t)-1)
	return PyString_FromString("???");
    strftime(tz,sizeof(tm),"%Z",&tm);
    return PyString_FromString(tz);
#endif
}

/* Returns the DST setting for the given DateTime instance assuming it
   refers to local time. -1 is returned in case it cannot be
   determined, 0 if it is not active, 1 if it is. For calendars other
   than the Gregorian the function always returns -1. 

   XXX If mktime() returns -1 for isdst, try harder using the hack in
       timegm.py.

*/

static
int mxDateTime_DST(mxDateTimeObject *datetime)
{
    struct tm tm;
    time_t ticks;
    
    if (datetime->calendar != MXDATETIME_GREGORIAN_CALENDAR)
	return -1;

    memset(&tm,0,sizeof(tm));
    tm.tm_hour = (int)datetime->hour;
    tm.tm_min = (int)datetime->minute;
    tm.tm_sec = (int)datetime->second;
    tm.tm_mday = (int)datetime->day;
    tm.tm_mon = (int)datetime->month - 1;
    tm.tm_year = (int)datetime->year - 1900;
    tm.tm_isdst = -1;
    ticks = mktime(&tm);
    if (ticks == (time_t)-1)
	return -1;
    return tm.tm_isdst;
}

/* Returns the ISO week notation for the given DateTime instance as
   tuple (year,isoweek,isoday). The algorithm also Works for negative
   dates.

   XXX Check this algorithm for the Julian calendar.
   
*/

static
PyObject *mxDateTime_ISOWeekTuple(mxDateTimeObject *datetime)
{
    int week;
    int year = datetime->year;
    int day;

    /* Estimate */
    week = (datetime->day_of_year-1) - datetime->day_of_week + 3;
    if (week >= 0)
	week = week / 7 + 1;
    day = datetime->day_of_week + 1;
    DPRINTF("mxDateTime_ISOWeekTuple: estimated year, week, day = %i, %i, %i\n",
	    year,week,day);

    /* Verify */
    if (week < 0) {
	/* The day lies in last week of the previous year */
	year--;
	if ((week > -2) || 
	    (week == -2 && mxDateTime_Leapyear(year,datetime->calendar)))
	    week = 53;
	else	    
	    week = 52;
    }
    else if (week == 53) {
	/* Check if the week belongs to year or year+1 */
	if (31-datetime->day + datetime->day_of_week < 3) {
	    week = 1;
	    year++;
	}
    }
    DPRINTF("mxDateTime_ISOWeekTuple: corrected year, week, day = %i, %i, %i\n",
	    year,week,day);
    return Py_BuildValue("iii",year,week,day);
}

/* Return a string identifying the used calendar. */

static
PyObject *mxDateTime_CalendarString(mxDateTimeObject *datetime)
{
    PyObject *v = mxDateTime_GregorianCalendar;
    
    switch (datetime->calendar) {
    case MXDATETIME_GREGORIAN_CALENDAR: 
	break;
    case MXDATETIME_JULIAN_CALENDAR:
	v = mxDateTime_JulianCalendar;
	break;
    default:
	Py_Error(PyExc_SystemError,
		 "Internal error in mxDateTime: wrong calendar value");
    }
    
    Py_INCREF(v);
    return v;

 onError:
    return NULL;
}

/* Writes a string representation to buffer. If the string does not
   fit the buffer, nothing is written. */

static
void mxDateTime_AsString(mxDateTimeObject *self,
			 char *buffer,
			 int buffer_len)
{
    double second;

    if (!buffer || buffer_len < 50)
	return;
    /* When showing the seconds value, display the value truncated to 2
       decimal places -- not rounded, as this can cause the value of
       60.00 to show up even when the indictated time does not point
       to a leap second. */
    second = floor(self->second * 100.0) / 100.0;
    if (self->year >= 0)
	sprintf(buffer,"%04li-%02i-%02i %02i:%02i:%05.2f",
		(long)self->year,(int)self->month,(int)self->day,
		(int)self->hour,(int)self->minute,(float)second);
    else
	sprintf(buffer,"-%04li-%02i-%02i %02i:%02i:%05.2f",
		(long)-self->year,(int)self->month,(int)self->day,
		(int)self->hour,(int)self->minute,(float)second);
}

/* Returns a string indicating the date in ISO format. */

static
PyObject *mxDateTime_DateString(mxDateTimeObject *self)
{
    char buffer[50];

    if (self->year >= 0)
	sprintf(buffer,"%04li-%02i-%02i",
		(long)self->year,(int)self->month,(int)self->day);
    else
	sprintf(buffer,"-%04li-%02i-%02i",
		(long)-self->year,(int)self->month,(int)self->day);

    return PyString_FromString(buffer);
}

/* Returns a string indicating the time in ISO format. */

static
PyObject *mxDateTime_TimeString(mxDateTimeObject *self)
{
    char buffer[50];
    double second;

    /* When showing the seconds value, display the value truncated to 2
       decimal places -- not rounded, as this can cause the value of
       60.00 to show up even when the indictated time does not point
       to a leap second. */
    second = floor(self->second * 100.0) / 100.0;
    sprintf(buffer,"%02i:%02i:%05.2f",
	    (int)self->hour,(int)self->minute,(float)second);

    return PyString_FromString(buffer);
}

/* --- methods --- */

#define datetime ((mxDateTimeObject*)self)

#ifdef HAVE_STRFTIME
Py_C_Function( mxDateTime_strftime,
	       "strftime(formatstr)")
{
    PyObject *v;
    char *fmt = 0;
    char *output = 0;
    int len_output,size_output = STRFTIME_OUTPUT_SIZE;
    struct tm tm;

    Py_GetArg("|s",fmt);
    
    if (!fmt)
	/* We default to the locale's standard date/time format */
	fmt = "%c";

    /* Init tm struct */
    memset(&tm,0,sizeof(tm));
    tm.tm_mday = (int)datetime->day;
    tm.tm_mon = (int)datetime->month - 1;
    tm.tm_year = (int)datetime->year - 1900;
    tm.tm_hour = (int)datetime->hour;
    tm.tm_min = (int)datetime->minute;
#if ROUND_SECONDS_IN_TM_STRUCT
    tm.tm_sec = (int)(datetime->second + 0.5); /* Round the value */
#else
    tm.tm_sec = (int)datetime->second;
#endif
    tm.tm_wday = ((int)datetime->day_of_week + 1) % 7;
    tm.tm_yday = (int)datetime->day_of_year - 1;
    tm.tm_isdst = -1; /* unkown */

    output = new(char,size_output);

    while (1) {
	if (output == NULL) {
	    PyErr_NoMemory();
	    goto onError;
	}
    	len_output = strftime(output,size_output,fmt,&tm);
	if (len_output == size_output) {
	    size_output *= 2;
	    output = resize(output,char,size_output);
	}
	else
	    break;
    }
    v = PyString_FromStringAndSize(output,len_output);
    if (v == NULL)
	goto onError;
    free(output);
    return v;

 onError:
    if (output)
	free(output);
    return NULL;
}
#endif

Py_C_Function( mxDateTime_tuple,
	       "tuple()\n"
	       "Return a (year,month,day,hour,minute,second,day_of_week,\n"
	       "day_of_year,dst) tuple.")
{
    int dst;
    
    Py_NoArgsCheck();
    dst = mxDateTime_DST(datetime);
    return Py_BuildValue("iiiiidiii",
			 (int)datetime->year,
			 (int)datetime->month,
			 (int)datetime->day,
			 (int)datetime->hour,
			 (int)datetime->minute,
			 (double)datetime->second,
			 (int)datetime->day_of_week,
			 (int)datetime->day_of_year,
			 dst);
 onError:
    return NULL;
}

Py_C_Function( mxDateTime_Julian,
	       "Julian()\n"
	       "Return an instance pointing to the same date and time,\n"
	       "but using the Julian calendar. If the instance already\n"
	       "uses the Julian calendar, a new reference to it is returned."
	       )
{
    long year;
    int month,day,hour,minute,day_of_week,day_of_year;
    double second;

    Py_NoArgsCheck();

    if (datetime->calendar == MXDATETIME_JULIAN_CALENDAR) {
	Py_INCREF(datetime);
	return (PyObject *)datetime;
    }
    if (mxDateTime_AsJulianDate(datetime,
				&year,
				&month,
				&day,
				&hour,
				&minute,
				&second,
				&day_of_week,
				&day_of_year))
	goto onError;
    return mxDateTime_FromJulianDateAndTime(year,
					    month,
					    day,
					    hour,
					    minute,
					    second);
 onError:
    return NULL;
}

Py_C_Function( mxDateTime_Gregorian,
	       "Gregorian()\n"
	       "Return an instance pointing to the same date and time,\n"
	       "but using the Gregorian calendar. If the instance already\n"
	       "uses the Gregorian calendar, a new reference to it is returned."
	       )
{
    long year;
    int month,day,hour,minute,day_of_week,day_of_year;
    double second;

    Py_NoArgsCheck();

    if (datetime->calendar == MXDATETIME_GREGORIAN_CALENDAR) {
	Py_INCREF(datetime);
	return (PyObject *)datetime;
    }
    if (mxDateTime_AsGregorianDate(datetime,
				   &year,
				   &month,
				   &day,
				   &hour,
				   &minute,
				   &second,
				   &day_of_week,
				   &day_of_year))
	goto onError;
    return mxDateTime_FromGregorianDateAndTime(year,
					       month,
					       day,
					       hour,
					       minute,
					       second);
 onError:
    return NULL;
}

#ifdef COPY_PROTOCOL
Py_C_Function( mxDateTime_copy,
	       "copy([memo])\n\n"
	       "Return a new reference for the instance. This function\n"
	       "is used for the copy-protocol. Real copying doesn't take\n"
	       "place, since the instances are immutable.")
{
    PyObject *memo;
    
    Py_GetArg("|O",memo);
    Py_INCREF(datetime);
    return (PyObject *)datetime;
 onError:
    return NULL;
}
#endif

Py_C_Function( mxDateTime_ticks,
	       "ticks([offset=0.0,dst=-1])\n\n"
	       "Return a time.time()-like value, representing the objects\n"
	       "value assuming it is local time. The conversion is done\n"
	       "using mktime() with the DST flag set to dst. offset is\n"
	       "subtracted from the resulting ticks value.")
{
    double ticks,offset = 0.0;
    int dst = -1;
    
    Py_Get2Args("|di",offset,dst);
    ticks = mxDateTime_AsTicksWithOffset(datetime,offset,dst);
    if (ticks == -1.0 && PyErr_Occurred())
	goto onError;
    return PyFloat_FromDouble(ticks);
 onError:
    return NULL;
}

Py_C_Function( mxDateTime_gmticks,
	       "gmticks(offset=0.0)\n\n"
	       "Return a time.time()-like value, representing the objects\n"
	       "value assuming it is UTC time. offset is subtracted from\n"
	       "the resulting ticks value.")
{
    double ticks,offset = 0.0;
    
    Py_GetArg("|d",offset)
    ticks = mxDateTime_AsGMTicksWithOffset(datetime,offset);
    if (ticks == -1.0 && PyErr_Occurred())
	goto onError;
    return PyFloat_FromDouble(ticks);
 onError:
    return NULL;
}

Py_C_Function( mxDateTime_gmtoffset,
	       "gmtoffset()\n\n"
	       "Returns a DateTimeDelta instance representing the UTC offset\n"
	       "for datetime assuming that the stored values refer to local\n"
	       "time. If you subtract this value from datetime, you'll get\n"
	       "UTC time."
	       )
{
    double offset;
    
    Py_NoArgsCheck();
    offset = mxDateTime_GMTOffset(datetime);
    if (offset == -1.0 && PyErr_Occurred())
	goto onError;
    return mxDateTimeDelta_FromSeconds(offset);
    
 onError:
    return NULL;
}

Py_C_Function( mxDateTime_gmtime,
	       "gmtime()\n\n"
	       "Returns a DateTime instance representing datetime in UTC\n"
	       "time assuming that the stored values refer to local time."
	       )
{
    double offset;
    
    Py_NoArgsCheck();
    offset = mxDateTime_GMTOffset(datetime);
    if (offset == -1.0 && PyErr_Occurred())
	goto onError;
    return mxDateTime_FromDateTimeAndOffset(datetime, 0, -offset);
    
 onError:
    return NULL;
}

Py_C_Function( mxDateTime_localtime,
	       "localtime()\n\n"
	       "Returns a DateTime instance representing datetime in local\n"
	       "time assuming that the stored values refer to UTC time."
	       )
{
    double gmticks;

    Py_NoArgsCheck();
    gmticks = mxDateTime_AsGMTicks(datetime);
    if (gmticks == -1.0 && PyErr_Occurred())
	goto onError;
    
    return mxDateTime_FromTicks(gmticks);

 onError:
    return NULL;
}

Py_C_Function( mxDateTime_COMDate,
	       "COMDate()\n\n"
	       "Return a float where the whole part is the number of days\n"
	       "in the Gregorian calendar since 30.12.1899 and the fraction\n"
	       "part equals abstime/86400.0.")
{
    double comdate;
    
    Py_NoArgsCheck();
    comdate = mxDateTime_AsCOMDate(datetime);
    return PyFloat_FromDouble(comdate);
 onError:
    return NULL;
}

Py_C_Function( mxDateTime_absvalues,
	       "absvalues()")
{
    Py_NoArgsCheck();
    Py_Return2("ld",datetime->absdate,datetime->abstime);
 onError:
    return NULL;
}

#undef datetime

/* --- slots --- */

static
PyObject *mxDateTime_AsFloat(mxDateTimeObject *self)
{
    double ticks;

    ticks = mxDateTime_AsTicksWithOffset(self,0,-1);
    if (ticks == -1.0 && PyErr_Occurred())
	goto onError;
    return PyFloat_FromDouble(ticks);
 onError:
    return NULL;
}

static
PyObject *mxDateTime_AsInt(mxDateTimeObject *self)
{
    double ticks;

    ticks = mxDateTime_AsTicksWithOffset(self,0,-1);
    if (ticks == -1.0 && PyErr_Occurred())
	goto onError;
    return PyInt_FromLong((long)ticks);
 onError:
    return NULL;
}

static
int mxDateTime_NonZero(mxDateTimeObject *self)
{
    /* A mxDateTime instance can never be zero */
    return 1;
}

static
PyObject *mxDateTime_Str(mxDateTimeObject *self)
{
    char s[50];

    mxDateTime_AsString(self,s,sizeof(s));
    return PyString_FromString(s);
}

static
PyObject *mxDateTime_Repr(mxDateTimeObject *self)
{
    char t[100];
    char s[50];

    mxDateTime_AsString(self,s,sizeof(s));
    sprintf(t,"<DateTime object for '%s' at %lx>",s,(long)self);
    return PyString_FromString(t);
}

static
PyObject *mxDateTime_Getattr(mxDateTimeObject *self,
			     char *name)
{
    if (Py_WantAttr(name,"year"))
	return PyInt_FromLong(self->year);

    else if (Py_WantAttr(name,"month"))
	return PyInt_FromLong((long)self->month);

    else if (Py_WantAttr(name,"day"))
	return PyInt_FromLong((long)self->day);

    else if (Py_WantAttr(name,"hour"))
	return PyInt_FromLong((long)self->hour);

    else if (Py_WantAttr(name,"minute"))
	return PyInt_FromLong((long)self->minute);

    else if (Py_WantAttr(name,"second"))
	return PyFloat_FromDouble((double)self->second);

    else if (Py_WantAttr(name,"absdays"))
	return PyFloat_FromDouble(
		      (double)(self->absdate - 1) + 
		      self->abstime / SECONDS_PER_DAY);

    else if (Py_WantAttr(name,"absdate"))
	return PyInt_FromLong(self->absdate);

    else if (Py_WantAttr(name,"abstime"))
	return PyFloat_FromDouble((double)self->abstime);

    else if (Py_WantAttr(name,"date"))
	return mxDateTime_DateString(self);

    else if (Py_WantAttr(name,"time"))
	return mxDateTime_TimeString(self);

    else if (Py_WantAttr(name,"yearoffset")) {
	long yearoffset = mxDateTime_YearOffset(self->year,self->calendar);
	if (yearoffset == -1 && PyErr_Occurred())
	    goto onError;
	return PyInt_FromLong(yearoffset);
    }
    
    else if (Py_WantAttr(name,"is_leapyear"))
	return PyInt_FromLong(mxDateTime_Leapyear(self->year,self->calendar));
    
    else if (Py_WantAttr(name,"day_of_week"))
	return PyInt_FromLong((long)self->day_of_week);

    else if (Py_WantAttr(name,"day_of_year"))
	return PyInt_FromLong((long)self->day_of_year);

    else if (Py_WantAttr(name,"days_in_month"))
	return PyInt_FromLong(
	       days_in_month[mxDateTime_Leapyear(self->year,self->calendar)]\
	                    [self->month - 1]);
    
    else if (Py_WantAttr(name,"iso_week"))
	return mxDateTime_ISOWeekTuple(self);

    else if (Py_WantAttr(name,"tz"))
	return mxDateTime_TimezoneString(self);

    else if (Py_WantAttr(name,"dst"))
	return PyInt_FromLong(mxDateTime_DST(self));

    else if (Py_WantAttr(name,"mjd"))
	return PyFloat_FromDouble((double)(self->absdate - 678576) + 
				  self->abstime / SECONDS_PER_DAY);

    else if (Py_WantAttr(name,"tjd"))
	return PyFloat_FromDouble((double)((self->absdate - 678576) % 10000) + 
				  self->abstime / SECONDS_PER_DAY);

    else if (Py_WantAttr(name,"tjd_myriad"))
	return PyInt_FromLong((self->absdate - 678576) / 10000 + 240);

    else if (Py_WantAttr(name,"jdn"))
	return PyFloat_FromDouble((double)self->absdate + 1721424.5 + 
				  self->abstime / SECONDS_PER_DAY);

    else if (Py_WantAttr(name,"calendar"))
	return mxDateTime_CalendarString(self);

    else if (Py_WantAttr(name,"__members__"))
	return Py_BuildValue("["
			     "sss"
			     "sss"
			     "sss"
			     "sss"
			     "sss"
			     "sss"
			     "ss"
			     "ss"
			     "]",
			     "year","month","day",
			     "hour","minute","second",
			     "absdays","absdate","abstime",
			     "yearoffset","is_leapyear","day_of_week",
			     "day_of_year","days_in_month","tz",
			     "dst","iso_week","mjd",
			     "tjd","tjd_myriad",
			     "jdn","calendar"
			     );

    return Py_FindMethod(mxDateTime_Methods,
			 (PyObject *)self,name);

 onError:
    return NULL;
}

static
int mxDateTime_Compare(mxDateTimeObject *self,
		       mxDateTimeObject *other)
{
    if (self == other) {
	if (other->argument) {
	    if (PyNumber_Check(other->argument)) {
		/* cmp(DateTime,Number) -- compare ticks */
		double t1 = PyFloat_AsDouble(other->argument);
		double t0 = mxDateTime_AsTicksWithOffset(self,0,-1);

		if ((t0 == -1.0 || t1 == -1.0) 
		    && PyErr_Occurred())
		    goto onError;

		/* clear temp var */
		Py_DECREF(other->argument);
		other->argument = 0;
	    
		return (t0 < t1) ? -1 : (t0 > t1) ? 1 : 0;
	    }
	    /* clear temp var */
	    Py_DECREF(other->argument);
	    other->argument = 0;
	}
	else
	    return 0;
    }
    else if (_mxDateTime_Check(other)) {
	long d0 = self->absdate, d1 = other->absdate;
	double t0 = self->abstime, t1 = other->abstime;

	return (d0 < d1) ? -1 : (d0 > d1) ? 1 :
	       (t0 < t1) ? -1 : (t0 > t1) ? 1 : 0;
    }
    Py_ErrorWithArg(PyExc_TypeError,
		    "can't compare other type (%s) to DateTime",
		    ((PyObject *)other)->ob_type->tp_name);
 onError:
    return -1;
}

static
long mxDateTime_Hash(mxDateTimeObject *self)
{
    long x = 0;
    long z[sizeof(double)/sizeof(long)+1];
    register int i;
    
    /* Clear z */
    for (i = sizeof(z) / sizeof(long) - 1; i >= 0; i--)
	z[i] = 0;

    /* Copy the double onto z */
    *((double *)z) = self->absdate * SECONDS_PER_DAY + self->abstime;

    /* Hash the longs in z together using XOR */
    for (i = sizeof(z) / sizeof(long) - 1; i >= 0; i--)
	x ^= z[i];

    /* Errors are indicated by returning -1, so we have to fix
       that hash value by hand. */
    if (x == -1)
	x = 19980427;

    return x;
}

/* We use a hack to make "mixed" coercion possible: if e.g. you
   write 3*DateTime(1), then the coercion of 3 to the DateTime
   object will fail, but coercion of the object to 3 will succeed.

   Since there is no way to figure out the order in which the operation
   will be executed, we simply return the same object twice and store the
   other object in a temporary variable argument of the DateTime 
   object. The operation can then decide what to do by looking at that
   temporary variable. In any case it will have to release the reference
   stored there and reset the variable to NULL.
   
   Unfortunately this doesn't always work: If coercion succeeds but
   the operation is implemented, then the reference is not cleared
   until the next operation is tried.  If this operation is one where
   the same object is used for both arguments, then the coercion
   function will not be called at all and the argument from the
   previous failed operation will be used for this operation --
   resulting in wrong values.  

   Moreover there is no way of telling the order of the operands, a-b
   could be executed as b-a after "mixed" coercion. Hopefully the need
   for this hack will be eliminated in one of the next Python
   releases.  */

static 
int mxDateTime_Coerce(PyObject **pv,
		      PyObject **pw)
{
    /* Only DateTime op DateTimeDelta and DateTime op Number are
       allowed; if you add any other type combination, you must adapt
       all numeric operations ! */
    if (_mxDateTime_Check(*pv)) {
	if (_mxDateTimeDelta_Check(*pw)) {
	    DPRINTF("mxDateTime_Coerce: datetime op delta\n");
	    Py_INCREF(*pv);
	    Py_INCREF(*pw);
	    return 0;
	}
	else if (PyNumber_Check(*pw)) {
	    register mxDateTimeObject *self = (mxDateTimeObject *)*pv;
	    
	    DPRINTF("mxDateTime_Coerce: datetime op number\n");
	    Py_INCREF(*pw);
	    Py_XDECREF(self->argument);
	    self->argument = *pw;
	    *pw = *pv;
	    Py_INCREF(*pv);
	    Py_INCREF(*pw);
	    return 0;
	}
    }
    DPRINTF("mxDateTime_Coerce: failed\n");
    return 1;
}

static
PyObject *mxDateTime_Add(mxDateTimeObject *self,
			 mxDateTimeObject *other)
{
    long absdate = self->absdate;
    double abstime = self->abstime;
    int calendar = self->calendar;

    if (_mxDateTimeDelta_Check(other)) {
	/* DateTime + DateTimeDelta */
	abstime += ((mxDateTimeDeltaObject *)other)->seconds;
    }
    else if (_mxDateTime_Check(other)) {
	if (self == other && other->argument) {
	    /* DateTime + Number (meaning days or fractions thereof) */
	    double value = PyFloat_AsDouble(other->argument);

	    /* clear temp var */
	    Py_DECREF(other->argument);
	    other->argument = 0;

	    if (value == -1 && PyErr_Occurred())
		goto onError;
	
	    abstime += value * SECONDS_PER_DAY;
	}
	else
	    /* DateTime + DateTime ... not supported */
	    Py_Error(PyExc_TypeError,
		     "DateTime + DateTime is not supported");
    }
    else
	Py_Error(PyExc_TypeError,
		 "unknown combination of types for addition");
    
    /* Normalize the time value */
    if (abstime >= SECONDS_PER_DAY) {
	long days = (long)(abstime / SECONDS_PER_DAY);
	absdate += days;
	abstime -= SECONDS_PER_DAY * (double)days;
    }
    else if (DOUBLE_IS_NEGATIVE(abstime)) {
	long days = (long)((-abstime-1) / SECONDS_PER_DAY) + 1;
	absdate -= days;
	abstime += SECONDS_PER_DAY * (double)days;
    }
    return mxDateTime_FromAbsDateTime(absdate,abstime,calendar);

 onError:
    return NULL;
}

static
PyObject *mxDateTime_Sub(mxDateTimeObject *self,
			 mxDateTimeObject *other)
{
    long absdate = self->absdate;
    double abstime = self->abstime;
    int calendar = self->calendar;

    if (_mxDateTimeDelta_Check(other)) {
	/* DateTime - DateTimeDelta */
	abstime -= ((mxDateTimeDeltaObject *)other)->seconds;
    }
    else if (_mxDateTime_Check(other)) {
	if (self == other && other->argument) {
	    /* DateTime - Number (meaning days or fractions thereof) */
	    double value = PyFloat_AsDouble(other->argument);

	    /* clear temp var */
	    Py_DECREF(other->argument);
	    other->argument = 0;

	    if (value == -1 && PyErr_Occurred())
		goto onError;
	
	    abstime -= value * SECONDS_PER_DAY;
	}
	else {
	    /* DateTime - DateTime */
	    absdate -= other->absdate;
	    abstime -= other->abstime;
	    return mxDateTimeDelta_FromDaysEx(absdate,abstime);
	}
    }
    else
	Py_Error(PyExc_TypeError,
		 "unknown combination of types for subtraction");
    
    /* Normalize the time value */
    if (abstime >= SECONDS_PER_DAY) {
	long days = (long)(abstime / SECONDS_PER_DAY);
	absdate += days;
	abstime -= SECONDS_PER_DAY * (double)days;
    }
    else if (DOUBLE_IS_NEGATIVE(abstime)) {
	long days = (long)((-abstime-1) / SECONDS_PER_DAY) + 1;
	absdate -= days;
	abstime += SECONDS_PER_DAY * (double)days;
    }
    return mxDateTime_FromAbsDateTime(absdate,abstime,calendar);

 onError:
    return NULL;
}

/* These two functions are necessary since the "mixed" coercion hack
   forces us to be very careful about what happens to the object after
   it was coerced. */

static
PyObject *mxDateTime_notimplemented2(mxDateTimeObject *v, PyObject *w)
{
    if (v->argument) {
	Py_DECREF(v->argument);
	v->argument = 0;
    }
    Py_Error(PyExc_TypeError,
	     "operation not implemented");
 onError:
    return NULL;
}

static
PyObject *mxDateTime_notimplemented3(mxDateTimeObject *v, 
				     PyObject *w, PyObject *u)
{
    if (v->argument) {
	Py_DECREF(v->argument);
	v->argument = 0;
    }
    Py_Error(PyExc_TypeError,
	     "operation not implemented");
 onError:
    return NULL;
}

/* Python Type Tables */

static
PyNumberMethods mxDateTime_TypeAsNumber = {

    /* These slots are not NULL-checked, so we must provide dummy functions */
    (binaryfunc)mxDateTime_Add,			/*nb_add*/
    (binaryfunc)mxDateTime_Sub,			/*nb_subtract*/
    (binaryfunc)mxDateTime_notimplemented2,	/*nb_multiply*/
    (binaryfunc)mxDateTime_notimplemented2,	/*nb_divide*/
    (binaryfunc)mxDateTime_notimplemented2,	/*nb_remainder*/
    (binaryfunc)mxDateTime_notimplemented2,	/*nb_divmod*/
    (ternaryfunc)mxDateTime_notimplemented3,	/*nb_power*/
    (unaryfunc)notimplemented1,			/*nb_negative*/
    (unaryfunc)notimplemented1,			/*nb_positive*/

    /* Everything below this line EXCEPT nb_nonzero (!) is NULL checked */
    (unaryfunc)0,				/*nb_absolute*/
    (inquiry)mxDateTime_NonZero,		/*nb_nonzero*/
    (unaryfunc)0,				/*nb_invert*/
    (binaryfunc)mxDateTime_notimplemented2,	/*nb_lshift*/
    (binaryfunc)mxDateTime_notimplemented2,	/*nb_rshift*/
    (binaryfunc)mxDateTime_notimplemented2,	/*nb_and*/
    (binaryfunc)mxDateTime_notimplemented2,	/*nb_xor*/
    (binaryfunc)mxDateTime_notimplemented2,	/*nb_or*/
    (coercion)mxDateTime_Coerce,		/*nb_coerce*/
    (unaryfunc)mxDateTime_AsInt,		/*nb_int*/
    (unaryfunc)0,				/*nb_long*/
    (unaryfunc)mxDateTime_AsFloat,		/*nb_float*/
    (unaryfunc)0,				/*nb_oct*/
    (unaryfunc)0,				/*nb_hex*/
};

statichere
PyTypeObject mxDateTime_Type = {
    PyObject_HEAD_INIT(0)		/* init at startup ! */
    0,			  		/*ob_size*/
    "DateTime",	  			/*tp_name*/
    sizeof(mxDateTimeObject),      	/*tp_basicsize*/
    0,			  		/*tp_itemsize*/
    /* slots */
    (destructor)mxDateTime_Free,		/*tp_dealloc*/
    (printfunc)0,		  	/*tp_print*/
    (getattrfunc)mxDateTime_Getattr,  	/*tp_getattr*/
    (setattrfunc)0,		  	/*tp_setattr*/
    (cmpfunc)mxDateTime_Compare, 		/*tp_compare*/
    (reprfunc)mxDateTime_Repr,	  	/*tp_repr*/
    &mxDateTime_TypeAsNumber, 		/*tp_as_number*/
    0,					/*tp_as_sequence*/
    0,					/*tp_as_mapping*/
    (hashfunc)mxDateTime_Hash,		/*tp_hash*/
    (ternaryfunc)0,			/*tp_call*/
    (reprfunc)mxDateTime_Str,		/*tp_str*/
    (getattrofunc)0, 			/*tp_getattro*/
    (setattrofunc)0, 			/*tp_setattro*/
    0,					/*tp_as_buffer*/
    0,					/*tp_xxx4*/
    (char*) 0				/*tp_doc*/
};

/* Python Method Table */

statichere
PyMethodDef mxDateTime_Methods[] =
{   
#ifdef HAVE_STRFTIME
    Py_MethodListEntry("strftime",mxDateTime_strftime),
#endif
    Py_MethodListEntryNoArgs("tuple",mxDateTime_tuple),
    Py_MethodListEntryNoArgs("Julian",mxDateTime_Julian),
    Py_MethodListEntryNoArgs("Gregorian",mxDateTime_Gregorian),
    Py_MethodListEntryNoArgs("COMDate",mxDateTime_COMDate),
    Py_MethodListEntry("ticks",mxDateTime_ticks),
    Py_MethodListEntryNoArgs("absvalues",mxDateTime_absvalues),
#ifdef HAVE_STRFTIME
    Py_MethodListEntry("Format",mxDateTime_strftime),
#endif
#ifdef COPY_PROTOCOL
    Py_MethodListEntry("__deepcopy__",mxDateTime_copy),
    Py_MethodListEntry("__copy__",mxDateTime_copy),
#endif
    Py_MethodListEntry("gmticks",mxDateTime_gmticks),
    Py_MethodListEntryNoArgs("gmtoffset",mxDateTime_gmtoffset),
    Py_MethodListEntryNoArgs("gmtime",mxDateTime_gmtime),
    Py_MethodListEntryNoArgs("localtime",mxDateTime_localtime),
#ifdef OLD_INTERFACE
    /* Old interface */
    Py_MethodListEntryNoArgs("as_tuple",mxDateTime_tuple),
    Py_MethodListEntryNoArgs("as_COMDate",mxDateTime_COMDate),
    Py_MethodListEntryNoArgs("as_ticks",mxDateTime_ticks),
#endif
    {NULL,NULL} /* end of list */
};

/* --- DateTimeDelta Object -----------------------------------------*/

/* --- allocation --- */

static
mxDateTimeDeltaObject *mxDateTimeDelta_New(void)
{
    mxDateTimeDeltaObject *delta;

#ifdef MXDATETIMEDELTA_FREELIST
    if (mxDateTimeDelta_FreeList) {
	delta = mxDateTimeDelta_FreeList;
	mxDateTimeDelta_FreeList = \
	    *(mxDateTimeDeltaObject **)mxDateTimeDelta_FreeList;
	delta->ob_type = &mxDateTimeDelta_Type;
	_Py_NewReference(delta);
    }
    else
#endif	
	 {
	delta = PyObject_NEW(mxDateTimeDeltaObject,&mxDateTimeDelta_Type);
	if (delta == NULL)
	    return NULL;
    }

    /* Init vars */
    delta->argument = 0;

    return delta;
}

/* --- deallocation --- */

static
void mxDateTimeDelta_Free(mxDateTimeDeltaObject *delta)
{
    /* Part of the "mixed" coerion hack */
    Py_XDECREF(delta->argument);
#ifdef MXDATETIMEDELTA_FREELIST
    /* Append to free list */
    *(mxDateTimeDeltaObject **)delta = mxDateTimeDelta_FreeList;
    mxDateTimeDelta_FreeList = delta;
    _Py_ForgetReference(delta);
#else
    PyMem_DEL(delta);
#endif
}

/* --- internal functions --- */

/* We may have a need for this in the future: */

#define mxDateTimeDelta_SetFromDaysEx(delta,days,seconds) \
        mxDateTimeDelta_SetFromSeconds(delta, SECONDS_PER_DAY*days + seconds)

static
int mxDateTimeDelta_SetFromSeconds(mxDateTimeDeltaObject *delta,
				   double seconds)
{
    if (delta == NULL) {
	PyErr_BadInternalCall();
	goto onError;
    }

    /* Store the internal seconds value as-is */
    delta->seconds = seconds;

    /* The broken down values are always positive: force seconds to be
       positive. */
    if (DOUBLE_IS_NEGATIVE(seconds))
	seconds = -seconds;

    /* Range check */
    Py_AssertWithArg(seconds <= SECONDS_PER_DAY * (double)LONG_MAX,
		     mxDateTime_RangeError,
		     "DateTimeDelta value out of range: %f seconds",
		     seconds);

    /* Calculate the broken down time */
    {
	register long wholeseconds;
	long day = 0;
	short hour,minute;
	double second;
	
	/* Calculate day part and then normalize seconds to be in the
	   range 0 <= seconds < 86400.0 */ 
	day += (long)(seconds / SECONDS_PER_DAY);
	seconds -= SECONDS_PER_DAY * (double)day;
	/* Some compilers (e.g. gcc 2.95.3 on Mandrake) have troubles
	   with getting rounding right even though 86400.0 IS exactly
	   representable using IEEE floats... */
	if (seconds >= SECONDS_PER_DAY) {
	    day++;
	    seconds -= SECONDS_PER_DAY;
	}
	/* Calculate the other parts based on the normalized seconds
           value */
	wholeseconds = (long)seconds;
	hour = wholeseconds / 3600;
	minute = (wholeseconds % 3600) / 60;
	second = seconds - (double)(hour*3600 + minute*60);
	/* Fix a possible rounding error */
	if (DOUBLE_IS_NEGATIVE(second))
	    second = 0.0;

	DPRINTF("mxDateTimeDelta_SetFromSeconds: "
		"seconds=%f day=%li hour=%i minute=%i second=%f\n",
		delta->seconds,day,hour,minute,second);

	delta->day = day;
	delta->hour = (signed char)hour;
	delta->minute = (signed char)minute;
	delta->second = second;
    }
    return 0;

 onError:
    return -1;
}

/* --- API functions --- */

statichere
PyObject *mxDateTimeDelta_FromDaysEx(long days,
				     double seconds)
{
    mxDateTimeDeltaObject *delta;
    
    delta = mxDateTimeDelta_New();
    if (delta == NULL)
	return NULL;

    if (mxDateTimeDelta_SetFromDaysEx(delta,days,seconds))
	goto onError;

    return (PyObject *)delta;
 onError:
    mxDateTimeDelta_Free(delta);
    return NULL;
}

statichere
PyObject *mxDateTimeDelta_FromSeconds(double seconds)
{
    mxDateTimeDeltaObject *delta;
    
    delta = mxDateTimeDelta_New();
    if (delta == NULL)
	return NULL;

    if (mxDateTimeDelta_SetFromSeconds(delta,seconds))
	goto onError;

    return (PyObject *)delta;
 onError:
    mxDateTimeDelta_Free(delta);
    return NULL;
}

static
PyObject *mxDateTimeDelta_FromDays(double days)
{
    mxDateTimeDeltaObject *delta;
    
    delta = mxDateTimeDelta_New();
    if (delta == NULL)
	return NULL;

    if (mxDateTimeDelta_SetFromSeconds(delta,days * SECONDS_PER_DAY))
	goto onError;

    return (PyObject *)delta;
 onError:
    mxDateTimeDelta_Free(delta);
    return NULL;
}

static
PyObject *mxDateTimeDelta_FromTime(int hours,
				   int minutes,
				   double seconds)
{
    mxDateTimeDeltaObject *delta;
    
    delta = mxDateTimeDelta_New();
    if (delta == NULL)
	return NULL;

    seconds += (double)(hours * 3600 + minutes * 60);

    if (mxDateTimeDelta_SetFromSeconds(delta,seconds))
	goto onError;

    return (PyObject *)delta;
 onError:
    mxDateTimeDelta_Free(delta);
    return NULL;
}

static
PyObject *mxDateTimeDelta_FromTuple(PyObject *v)
{
    mxDateTimeDeltaObject *delta = 0;
    int days;
    double seconds;

    if (!PyTuple_Check(v)) {
	PyErr_BadInternalCall();
	return NULL;
    }
    if (!PyArg_ParseTuple(v,
	"id;need a 2-tuple (days,seconds)",
			  &days,&seconds))
	return NULL;
    
    delta = mxDateTimeDelta_New();
    if (delta == NULL)
	return NULL;

    if (mxDateTimeDelta_SetFromDaysEx(delta,days,seconds))
	goto onError;

    return (PyObject *)delta;

 onError:
    mxDateTimeDelta_Free(delta);
    return NULL;
}

static
PyObject *mxDateTimeDelta_FromTimeTuple(PyObject *v)
{
    mxDateTimeDeltaObject *delta = 0;
    double hours,minutes,seconds;

    if (!PyTuple_Check(v)) {
	PyErr_BadInternalCall();
	return NULL;
    }
    if (!PyArg_ParseTuple(v,
	"ddd;need a 3-tuple (hours,minutes,seconds)",
			  &hours,&minutes,&seconds))
	return NULL;
    
    delta = mxDateTimeDelta_New();
    if (delta == NULL)
	return NULL;

    if (mxDateTimeDelta_SetFromSeconds(delta,
	     hours * 3600.0 + minutes * 60.0 + seconds))
	goto onError;

    return (PyObject *)delta;

 onError:
    mxDateTimeDelta_Free(delta);
    return NULL;
}

static
double mxDateTimeDelta_AsDouble(mxDateTimeDeltaObject *delta)
{
    return delta->seconds;
}

static
double mxDateTimeDelta_AsDays(mxDateTimeDeltaObject *delta)
{
    return delta->seconds / SECONDS_PER_DAY;
}

static
int mxDateTimeDelta_BrokenDown(mxDateTimeDeltaObject *delta,
			       long *day,
			       int *hour,
			       int *minute,
			       double *second)
{
    if (day)
	*day = (long)delta->day;
    if (hour)
	*hour = (int)delta->hour;
    if (minute)
	*minute = (int)delta->minute;
    if (second)
	*second = (double)delta->second;
    return 0;
}

/* Writes a string representation to buffer. If the string does not
   fit the buffer, nothing is written. */

static
void mxDateTimeDelta_AsString(mxDateTimeDeltaObject *self,
			      char *buffer,
			      int buffer_len)
{
    double second;

    if (!buffer || buffer_len < 50)
	return;
    /* When showing the seconds value, show the value truncated to 2
       decimal places -- not rounded, as this an cause the value of
       60.00 to show up even though DateTimeDelta instances don't
       support this value. */
    second = floor(self->second * 100.0) / 100.0;
    if (self->day != 0) {
	if (self->seconds >= 0.0)
	    sprintf(buffer,"%li:%02i:%02i:%05.2f",
		    (long)self->day,(int)self->hour,
		    (int)self->minute,(float)second);
	else
	    sprintf(buffer,"-%li:%02i:%02i:%05.2f",
		    (long)self->day,(int)self->hour,
		    (int)self->minute,(float)second);
    }
    else {
	if (self->seconds >= 0.0)
	    sprintf(buffer,"%02i:%02i:%05.2f",
		    (int)self->hour,(int)self->minute,(float)second);
	else
	    sprintf(buffer,"-%02i:%02i:%05.2f",
		    (int)self->hour,(int)self->minute,(float)second);
    }
}

/* --- methods --- (should have lowercase extension) */

#define delta ((mxDateTimeDeltaObject*)self)

#ifdef HAVE_STRFTIME
Py_C_Function( mxDateTimeDelta_strftime,
	       "strftime(formatstr)\n\n"
	       "Returns a formatted string of the time (ignoring the sign).\n"
	       "Of course, it only makes sense to use time related\n"
	       "specifiers. The delta sign is not taken into account.\n"
	       "All values are shown positive.")
{
    PyObject *v;
    struct tm tm;
    char *fmt;
    char *output = 0;
    int len_output,size_output = STRFTIME_OUTPUT_SIZE;

    Py_GetArg("s",fmt);
    
    /* Init to the epoch */
    memset(&tm,0,sizeof(tm));
    tm.tm_year = 0;

    /* Store the values in their appropriate places */
    tm.tm_mday = (int)delta->day;
    tm.tm_hour = (int)delta->hour;
    tm.tm_min = (int)delta->minute;
    tm.tm_sec = (int)delta->second;

    output = new(char,size_output);

    while (1) {
	if (output == NULL) {
	    PyErr_NoMemory();
	    goto onError;
	}
    	len_output = strftime(output,size_output,fmt,&tm);
	if (len_output == size_output) {
	    size_output *= 2;
	    output = resize(output,char,size_output);
	}
	else
	    break;
    }
    v = PyString_FromStringAndSize(output,len_output);
    if (v == NULL)
	goto onError;
    free(output);
    return v;

 onError:
    if (output)
	free(output);
    return NULL;
}
#endif

Py_C_Function( mxDateTimeDelta_absvalues,
	       "absvalues()\n\n"
	       "Return a (absdays,absseconds) tuple. The absseconds part is\n"
	       "normalized in such way that it is always < 86400.0. The\n"
	       "values can be used to do date/time calculations.\n"
	       "Both are signed.")
{
    long days;
    double seconds;
    
    Py_NoArgsCheck();

    seconds = delta->seconds;
    days = (long)(seconds / SECONDS_PER_DAY);
    seconds = seconds - SECONDS_PER_DAY * (double)days;
    return Py_BuildValue("ld",
			 days,
			 seconds);
    
 onError:
    return NULL;
}

Py_C_Function( mxDateTimeDelta_tuple,
	       "tuple()\n\n"
	       "Return a (day,hour,minute,second) tuple. The values are all\n"
	       "signed and use the same conventions as the attributes of\n"
	       "the same name.")
{
    Py_NoArgsCheck();
    if (!DOUBLE_IS_NEGATIVE(delta->seconds))
	return Py_BuildValue("iiid",
			     (int)delta->day,
			     (int)delta->hour,
			     (int)delta->minute,
			     delta->second);
    else
	return Py_BuildValue("iiid",
			     -(int)delta->day,
			     -(int)delta->hour,
			     -(int)delta->minute,
			     -delta->second);
    
 onError:
    return NULL;
}

#ifdef COPY_PROTOCOL
Py_C_Function( mxDateTimeDelta_copy,
	       "copy([memo])\n"
	       "Return a new reference for the instance. This function\n"
	       "is used for the copy-protocol. Real copying doesn't take\n"
	       "place, since the instances are immutable.")
{
    PyObject *memo;
    
    Py_GetArg("|O",memo);
    Py_INCREF(delta);
    return (PyObject *)delta;
 onError:
    return NULL;
}
#endif

#ifdef OLD_INTERFACE

/* Old interface */
Py_C_Function( mxDateTimeDelta_as_timetuple,
               "as_timetuple()\n\n"
               "Return a (hour,minute,second) tuple. The values are all\n"
               "signed.")
{
    Py_NoArgsCheck();
    if (!DOUBLE_IS_NEGATIVE(delta->seconds))
        return Py_BuildValue("iid",
                             (int)delta->hour,
                             (int)delta->minute,
                             delta->second);
    else
        return Py_BuildValue("iid",
                             -(int)delta->hour,
                             -(int)delta->minute,
                             -delta->second);
    
 onError:
    return NULL;
}

Py_C_Function( mxDateTimeDelta_as_ticks,
               "as_ticks()\n\n"
               "Return the objects value in seconds. Days are converted\n"
               "using 86400.0 seconds. The returned value is signed.")
{
    Py_NoArgsCheck();
    return PyFloat_FromDouble(mxDateTimeDelta_AsDouble(delta));
 onError:
    return NULL;
}
#endif

#undef delta

/* --- slots --- */

static
PyObject *mxDateTimeDelta_AsFloat(mxDateTimeDeltaObject *self)
{
    return PyFloat_FromDouble(mxDateTimeDelta_AsDouble(self));
}

static
PyObject *mxDateTimeDelta_AsInt(mxDateTimeDeltaObject *self)
{
    return PyInt_FromLong((long)mxDateTimeDelta_AsDouble(self));
}

static
PyObject *mxDateTimeDelta_Str(mxDateTimeDeltaObject *self)
{
    char s[50];

    mxDateTimeDelta_AsString(self,s,sizeof(s));
    return PyString_FromString(s);
}

static
PyObject *mxDateTimeDelta_Repr(mxDateTimeDeltaObject *self)
{
    char t[100];
    char s[50];

    mxDateTimeDelta_AsString(self,s,sizeof(s));
    sprintf(t,"<DateTimeDelta object for '%s' at %lx>",s,(long)self);
    return PyString_FromString(t);
}

static
PyObject *mxDateTimeDelta_Getattr(mxDateTimeDeltaObject *self,
				  char *name)
{
    if (Py_WantAttr(name,"hour")) {
	if (!DOUBLE_IS_NEGATIVE(self->seconds))
	    return PyInt_FromLong((long)self->hour);
	else
	    return PyInt_FromLong(-(long)self->hour);
    }
    else if (Py_WantAttr(name,"minute")) {
	if (!DOUBLE_IS_NEGATIVE(self->seconds))
	    return PyInt_FromLong((long)self->minute);
	else
	    return PyInt_FromLong(-(long)self->minute);
    }
    else if (Py_WantAttr(name,"second")) {
	if (!DOUBLE_IS_NEGATIVE(self->seconds))
	    return PyFloat_FromDouble(self->second);
	else
	    return PyFloat_FromDouble(-self->second);
    }
    else if (Py_WantAttr(name,"day")) {
	if (!DOUBLE_IS_NEGATIVE(self->seconds))
	    return PyInt_FromLong((long)self->day);
	else
	    return PyInt_FromLong(-(long)self->day);
    }
    else if (Py_WantAttr(name,"seconds"))
	return PyFloat_FromDouble(self->seconds);

    else if (Py_WantAttr(name,"minutes"))
	return PyFloat_FromDouble(self->seconds / 60.0);

    else if (Py_WantAttr(name,"hours"))
	return PyFloat_FromDouble(self->seconds / 3600.0);

    else if (Py_WantAttr(name,"days"))
	return PyFloat_FromDouble(self->seconds / SECONDS_PER_DAY);
    
    else if (Py_WantAttr(name,"__members__"))
	return Py_BuildValue("[ssssssss]",
			     "hour","minute","second",
			     "day","seconds","minutes",
			     "hours","days");

    return Py_FindMethod(mxDateTimeDelta_Methods,
			 (PyObject *)self,name);
}

static
int mxDateTimeDelta_Compare(mxDateTimeDeltaObject *self,
			    mxDateTimeDeltaObject *other)
{
    double i = self->seconds;
    double j = other->seconds;
    if (self == other) {
	if (other->argument) {
	    if (PyNumber_Check(other->argument)) {
		/* cmp(DateTimeDelta,Number) -- compare seconds */
		double t1 = PyFloat_AsDouble(other->argument);
		double t0 = mxDateTimeDelta_AsDouble(self);

		if ((t0 == -1.0 || t1 == -1.0) 
		    && PyErr_Occurred())
		    goto onError;

		/* clear temp var */
		Py_DECREF(other->argument);
		other->argument = 0;
	    
		return (t0 < t1) ? -1 : (t0 > t1) ? 1 : 0;
	    }
	    /* clear temp var */
	    Py_DECREF(other->argument);
	    other->argument = 0;
	}
	else
	    return 0;
	Py_ErrorWithArg(PyExc_TypeError,
			"can't compare other type (%s) to DateTimeDelta",
			((PyObject *)other)->ob_type->tp_name);
    }
    else
	return (i < j) ? -1 : (i > j) ? 1 : 0;

 onError:
    return -1;
}

static
long mxDateTimeDelta_Hash(mxDateTimeDeltaObject *self)
{
    long x = 0;
    long z[sizeof(double)/sizeof(long)+1];
    register int i;
    
    /* Clear z */
    for (i = sizeof(z) / sizeof(long) - 1; i >= 0; i--)
	z[i] = 0;

    /* Copy the double onto z */
    *((double *)z) = self->seconds;

    /* Hash the longs in z together using XOR */
    for (i = sizeof(z) / sizeof(long) - 1; i >= 0; i--)
	x ^= z[i];

    /* Errors are indicated by returning -1, so we have to fix
       that hash value by hand. */
    if (x == -1)
	x = 19980428;

    return x;
}

/* Same hack as above: */

static 
int mxDateTimeDelta_Coerce(PyObject **pv,
			   PyObject **pw)
{
    if (_mxDateTimeDelta_Check(*pv)) {
	/* Delta op DateTime: can't do it */
	if (_mxDateTime_Check(*pw))
	    Py_Error(PyExc_TypeError,
		     "only DateTime op DateTimeDelta is supported");
	/* Delta op Number: use hack */
	if (PyNumber_Check(*pw)) {
	    register mxDateTimeDeltaObject *self = \
		     (mxDateTimeDeltaObject *)*pv;
	    
	    DPRINTF("mxDateTimeDelta_Coerce: delta op number\n");
	    Py_INCREF(*pw);
	    Py_XDECREF(self->argument);
	    self->argument = *pw;
	    *pw = *pv;
	    Py_INCREF(*pv);
	    Py_INCREF(*pw);
	    return 0;
	}
	/* NOTE: If you add any other type combination, you must also
	   adapt all numeric operations ! */
    }
    DPRINTF("mxDateTimeDelta_Coerce: failed\n");
    return 1;

 onError:
    DPRINTF("mxDateTimeDelta_Coerce: failed\n");
    return -1;
}

static
PyObject *mxDateTimeDelta_Negative(mxDateTimeDeltaObject *self)
{
    return mxDateTimeDelta_FromSeconds(-self->seconds);
}

static
PyObject *mxDateTimeDelta_Positive(mxDateTimeDeltaObject *self)
{
    Py_INCREF(self);
    return (PyObject *)self;
}

static
PyObject *mxDateTimeDelta_Abs(mxDateTimeDeltaObject *self)
{
    if (self->seconds >= 0) {
	Py_INCREF(self);
	return (PyObject *)self;
    }
    else
	return mxDateTimeDelta_FromSeconds(-self->seconds);
}

static
int mxDateTimeDelta_NonZero(mxDateTimeDeltaObject *self)
{
    return (self->seconds != (double)0.0);
}

static
PyObject *mxDateTimeDelta_Add(mxDateTimeDeltaObject *self,
			      mxDateTimeDeltaObject *other)
{
    if (self == other && other->argument) {
	/* Delta + Number (meaning seconds) */
	double value = PyFloat_AsDouble(other->argument);

	/* clear temp var */
	Py_DECREF(other->argument);
	other->argument = 0;

	if (value == -1 && PyErr_Occurred())
	    goto onError;
	
	return mxDateTimeDelta_FromSeconds(self->seconds + value);
    }
    else
	/* Delta + Delta */
	return mxDateTimeDelta_FromSeconds(self->seconds + other->seconds);
 onError:
    return NULL;
}

static
PyObject *mxDateTimeDelta_Sub(mxDateTimeDeltaObject *self,
				 mxDateTimeDeltaObject *other)
{
    if (self == other && other->argument) {
	/* Delta - Number (meaning seconds), Number - Seconds */
	double value = PyFloat_AsDouble(other->argument);

	/* clear temp var */
	Py_DECREF(other->argument);
	other->argument = 0;

	if (value == -1 && PyErr_Occurred())
	    goto onError;
	
	return mxDateTimeDelta_FromSeconds(self->seconds - value);
    }
    else
	/* Delta - Delta */
	return mxDateTimeDelta_FromSeconds(self->seconds - other->seconds);
 onError:
    return NULL;
}

static
PyObject *mxDateTimeDelta_Multiply(mxDateTimeDeltaObject *self,
				   mxDateTimeDeltaObject *other)
{
    if (self == other && other->argument) {
	/* Delta * Number, Number * Delta */
	double value = PyFloat_AsDouble(other->argument);

	/* clear temp var */
	Py_DECREF(other->argument);
	other->argument = 0;

	if (value == -1 && PyErr_Occurred())
	    goto onError;
	
	return mxDateTimeDelta_FromSeconds(self->seconds * value);
    }
    else
	/* Delta * Delta */
	Py_Error(PyExc_TypeError,
		 "DateTimeDelta * DateTimeDelta not supported");
 onError:
    return NULL;
}

static
PyObject *mxDateTimeDelta_Divide(mxDateTimeDeltaObject *self,
				 mxDateTimeDeltaObject *other)
{
    if (self == other && other->argument) {
	/* Delta / Number, Number / Delta */
	double value = PyFloat_AsDouble(other->argument);

	/* clear temp var */
	Py_DECREF(other->argument);
	other->argument = 0;

	if (value == -1 && PyErr_Occurred())
	    goto onError;
	Py_Assert(value != 0.0,
		  PyExc_ZeroDivisionError,
		  "DateTimeDelta division");
	return mxDateTimeDelta_FromSeconds(self->seconds / value);
    }
    else {
	/* Delta / Delta */
	Py_Assert(other->seconds != 0.0,
		  PyExc_ZeroDivisionError,
		  "DateTimeDelta division");
	return PyFloat_FromDouble(self->seconds / other->seconds);
    }
 onError:
    return NULL;
}

/* Same note as above. */

static
PyObject *mxDateTimeDelta_notimplemented2(mxDateTimeDeltaObject *v, 
					  PyObject *w)
{
    if (v->argument) {
	Py_DECREF(v->argument);
	v->argument = 0;
    }
    Py_Error(PyExc_TypeError,
	     "operation not implemented");
 onError:
    return NULL;
}

static
PyObject *mxDateTimeDelta_notimplemented3(mxDateTimeDeltaObject *v, 
					  PyObject *w, PyObject *u)
{
    if (v->argument) {
	Py_DECREF(v->argument);
	v->argument = 0;
    }
    Py_Error(PyExc_TypeError,
	     "operation not implemented");
 onError:
    return NULL;
}

/* Python Type Tables */

static 
PyNumberMethods mxDateTimeDelta_TypeAsNumber = {

    /* These slots are not NULL-checked, so we must provide dummy functions */
    (binaryfunc)mxDateTimeDelta_Add,			/*nb_add*/
    (binaryfunc)mxDateTimeDelta_Sub,			/*nb_subtract*/
    (binaryfunc)mxDateTimeDelta_Multiply,		/*nb_multiply*/
    (binaryfunc)mxDateTimeDelta_Divide,			/*nb_divide*/
    (binaryfunc)mxDateTimeDelta_notimplemented2,	/*nb_remainder*/
    (binaryfunc)mxDateTimeDelta_notimplemented2,	/*nb_divmod*/
    (ternaryfunc)mxDateTimeDelta_notimplemented3,	/*nb_power*/
    (unaryfunc)mxDateTimeDelta_Negative,		/*nb_negative*/
    (unaryfunc)mxDateTimeDelta_Positive,		/*nb_positive*/

    /* Everything below this line EXCEPT nb_nonzero (!) is NULL checked */
    (unaryfunc)mxDateTimeDelta_Abs,			/*nb_absolute*/
    (inquiry)mxDateTimeDelta_NonZero,			/*nb_nonzero*/
    (unaryfunc)0,					/*nb_invert*/
    (binaryfunc)mxDateTimeDelta_notimplemented2,	/*nb_lshift*/
    (binaryfunc)mxDateTimeDelta_notimplemented2,	/*nb_rshift*/
    (binaryfunc)mxDateTimeDelta_notimplemented2,	/*nb_and*/
    (binaryfunc)mxDateTimeDelta_notimplemented2,	/*nb_xor*/
    (binaryfunc)mxDateTimeDelta_notimplemented2,	/*nb_or*/
    (coercion)mxDateTimeDelta_Coerce,			/*nb_coerce*/
    (unaryfunc)mxDateTimeDelta_AsInt,			/*nb_int*/
    (unaryfunc)0,					/*nb_long*/
    (unaryfunc)mxDateTimeDelta_AsFloat,			/*nb_float*/
    (unaryfunc)0,					/*nb_oct*/
    (unaryfunc)0,					/*nb_hex*/
};

statichere
PyTypeObject mxDateTimeDelta_Type = {
    PyObject_HEAD_INIT(0)		/* init at startup ! */
    0,			  		/*ob_size*/
    "DateTimeDelta",	  		/*tp_name*/
    sizeof(mxDateTimeDeltaObject),   	/*tp_basicsize*/
    0,			  		/*tp_itemsize*/
    /* slots */
    (destructor)mxDateTimeDelta_Free,	/*tp_dealloc*/
    (printfunc)0,  			/*tp_print*/
    (getattrfunc)mxDateTimeDelta_Getattr,  	/*tp_getattr*/
    (setattrfunc)0,		  	/*tp_setattr*/
    (cmpfunc)mxDateTimeDelta_Compare,  		/*tp_compare*/
    (reprfunc)mxDateTimeDelta_Repr,		/*tp_repr*/
    &mxDateTimeDelta_TypeAsNumber, 		/*tp_as_number*/
    0,					/*tp_as_sequence*/
    0,					/*tp_as_mapping*/
    (hashfunc)mxDateTimeDelta_Hash,		/*tp_hash*/
    (ternaryfunc)0,			/*tp_call*/
    (reprfunc)mxDateTimeDelta_Str,		/*tp_str*/
    (getattrofunc)0, 			/*tp_getattro*/
    (setattrofunc)0, 			/*tp_setattro*/
    0,					/*tp_as_buffer*/
    0,					/*tp_xxx4*/
    (char*) 0				/*tp_doc*/
};

/* Python Method Table */

statichere
PyMethodDef mxDateTimeDelta_Methods[] =
{   
    Py_MethodListEntryNoArgs("absvalues",mxDateTimeDelta_absvalues),
    Py_MethodListEntryNoArgs("tuple",mxDateTimeDelta_tuple),
#ifdef HAVE_STRFTIME
    Py_MethodListEntry("strftime",mxDateTimeDelta_strftime),
#endif
#ifdef COPY_PROTOCOL
    Py_MethodListEntry("__deepcopy__",mxDateTimeDelta_copy),
    Py_MethodListEntry("__copy__",mxDateTimeDelta_copy),
#endif
#ifdef OLD_INTERFACE
    /* Old interface */
    Py_MethodListEntryNoArgs("as_tuple",mxDateTimeDelta_absvalues),
    Py_MethodListEntryNoArgs("as_timetuple",mxDateTimeDelta_as_timetuple),
    Py_MethodListEntryNoArgs("as_ticks",mxDateTimeDelta_as_ticks),
#endif
    {NULL,NULL} /* end of list */
};

/* --- Other functions ----------------------------------------------------- */

Py_C_Function( mxDateTime_DateTime,
	       "DateTime(year,month=1,day=1,hour=0,minute=0,second=0.0)\n\n"
	       "Returns a DateTime-object reflecting the given date\n"
	       "and time. Seconds can be given as float to indicate\n"
	       "fractions."
	       )
{
    int year,
	month = 1,
	day = 1;
    int hour = 0,
	minute = 0;
    double second = 0.0;
    
    Py_Get6Args("i|iiiid",year,month,day,hour,minute,second);
    return mxDateTime_FromDateAndTime(year,month,day,hour,minute,second);
 onError:
    return NULL;
}

Py_C_Function( mxDateTime_JulianDateTime,
	       "JulianDateTime(year,month=1,day=1,hour=0,minute=0,second=0.0)\n\n"
	       "Returns a DateTime-object reflecting the given Julian date\n"
	       "and time. Seconds can be given as float to indicate\n"
	       "fractions."
	       )
{
    int year,
	month = 1,
	day = 1;
    int hour = 0,
	minute = 0;
    double second = 0.0;
    
    Py_Get6Args("i|iiiid",year,month,day,hour,minute,second);
    return mxDateTime_FromJulianDateAndTime(year,month,day,hour,minute,second);
 onError:
    return NULL;
}

Py_C_Function( mxDateTime_DateTimeFromAbsDateTime,
	       "DateTimeFromAbsDateTime(absdate[,abstime=0.0])\n\n"
	       "Returns a DateTime-object for the given absolute values.")
{
    long absdate;
    double abstime = 0.0;

    Py_Get2Args("l|d",absdate,abstime);

    return mxDateTime_FromAbsDateAndTime(absdate,abstime);
 onError:
    return NULL;
}

#ifdef HAVE_STRPTIME
Py_C_Function( mxDateTime_strptime,
	       "strptime(str,formatstr,default=None)\n\n"
	       "Returns a DateTime-object reflecting the parsed\n"
	       "date and time; default can be given to set default values\n"
	       "for parts not given in the string. If not given,\n"
	       "1.1.0001 0:00:00.00 is used instead."
	       )
{
    char *str;
    char *fmt;
    char *lastchr;
    int len_str,pos;
    struct tm tm;
    PyObject *defvalue = NULL;

    Py_Get3Args("ss|O",str,fmt,defvalue);
    
    len_str = strlen(str);
    if (defvalue) {
	Py_Assert(_mxDateTime_Check(defvalue),
		  PyExc_TypeError,
		  "default must be a DateTime instance");
	mxDateTime_AsTmStruct((mxDateTimeObject *)defvalue,&tm);
    }
    else {
	/* Init to 1.1.0001 0:00:00.00 */
	memset(&tm,0,sizeof(tm));
	tm.tm_mday = 1;
	tm.tm_year = -1899;
    }
    /* Parse */
    lastchr = strptime(str,fmt,&tm);
    Py_Assert(lastchr != NULL,
	      mxDateTime_Error,
	      "strptime() parsing error");
    pos = (int)(lastchr - str);
    if (pos != len_str)
	Py_ErrorWithArg(mxDateTime_Error,
			"strptime() parsing error at character %i",pos);
    return mxDateTime_FromTmStruct(&tm);
    
 onError:
    return NULL;
}
#endif

Py_C_Function( mxDateTime_DateTimeFromCOMDate,
	       "DateTimeFromCOMDate(comdate)\n\n"
	       "Returns a DateTime-object reflecting the given date\n"
	       "and time.")
{
    double comdate;
    
    Py_GetArg("d",comdate);
    return mxDateTime_FromCOMDate(comdate);
 onError:
    return NULL;
}

#ifdef OLD_INTERFACE
Py_C_Function( mxDateTime_DateTimeFromTicks,
	       "DateTimeFromTicks(ticks)\n\n"
	       "Returns a DateTime-object reflecting the given time\n"
	       "value. Conversion is done to local time (similar to\n"
	       "time.localtime()).")
{
    double ticks;
    
    Py_GetArg("d",ticks);
    return mxDateTime_FromTicks(ticks);
 onError:
    return NULL;
}
#endif

Py_C_Function( mxDateTime_DateTimeFromAbsDays,
	       "DateTimeFromAbsDays(absdays)\n\n"
	       "Returns a DateTime-object reflecting the given time\n"
	       "value (days since the epoch).")
{
    double absdays;
    
    Py_GetArg("d",absdays);
    return mxDateTime_FromAbsDays(absdays);
 onError:
    return NULL;
}

Py_C_Function( mxDateTime_DateTimeDelta,
	       "DateTimeDelta(days[,hours=0.0,minutes=0.0,seconds=0.0])\n\n"
	       "Returns a DateTimeDelta-object reflecting the given\n"
	       "day and time delta."
	       )
{
    PyObject *v;
    double days,
	hours = 0, 
	minutes = 0,
	seconds = 0.0;
    
    Py_Get4Args("d|ddd",days,hours,minutes,seconds);

    v = mxDateTimeDelta_FromSeconds(
			   days*SECONDS_PER_DAY
			   + hours*3600.0
			   + minutes*60.0
			   + seconds);
    if (v == NULL)
	goto onError;

    return v;
 onError:
    return NULL;
}

#if OLD_INTERFACE
Py_C_Function( mxDateTime_TimeDelta,
	       "TimeDelta(hours[,minutes=0,seconds=0.0])\n\n"
	       "Returns a DateTimeDelta-object reflecting the given\n"
	       "time delta. Seconds can be given as float to indicate\n"
	       "fractions."
	       )
{
    double hour, 
	minute=0.0,
	second=0.0;
    
    Py_Get3Args("d|dd",hour,minute,second);
    return mxDateTimeDelta_FromSeconds(hour*3600.0
				       + minute*60.0
				       + second);
 onError:
    return NULL;
}
#endif

Py_C_Function( mxDateTime_DateTimeDeltaFromSeconds,
	       "DateTimeDeltaFromSeconds(seconds)\n\n"
	       "Returns a DateTimeDelta-object reflecting the given time\n"
	       "value.")
{
    double seconds;
    
    Py_GetArg("d",seconds);
    return mxDateTimeDelta_FromSeconds(seconds);
 onError:
    return NULL;
}

Py_C_Function( mxDateTime_DateTimeDeltaFromDays,
	       "DateTimeDeltaFromDays(days)\n\n"
	       "Returns a DateTimeDelta-object reflecting the given time\n"
	       "value given in days (fractions are allowed).")
{
    double days;
    
    Py_GetArg("d",days);
    return mxDateTimeDelta_FromDays(days);
 onError:
    return NULL;
}

Py_C_Function( mxDateTime_now,
	       "now()\n\n"
	       "Returns a DateTime-object reflecting the current local time."
	       )
{
    double fticks;

    Py_NoArgsCheck();
    fticks = mxDateTime_GetCurrentTime();
    if (fticks == -1 && PyErr_Occurred())
	goto onError;
    return mxDateTime_FromTicks(fticks);

 onError:
    return NULL;
}

Py_C_Function( mxDateTime_utc,
	       "utc()\n\n"
	       "Returns a DateTime-object reflecting the current UTC time."
	       )
{
    double fticks;

    Py_NoArgsCheck();
    fticks = mxDateTime_GetCurrentTime();
    if (fticks == -1 && PyErr_Occurred())
	goto onError;
    return mxDateTime_FromGMTicks(fticks);

 onError:
    return NULL;
}

Py_C_Function( mxDateTime_cmp,
	       "cmp(value1,value2[,accuracy=0.0])\n\n"
	       "Compares two DateTime[Delta] objects. If accuracy is\n"
	       "given, then equality will result in case the absolute\n"
	       "difference between the two values is less than or equal\n"
	       "to accuracy.")
{
    PyObject *a,*b;
    double acc = 0.0;
    
    Py_Get3Args("OO|d",a,b,acc);

    if (_mxDateTime_Check(a) && _mxDateTime_Check(b)) {
	/* cmp(DateTime,DateTime) */
	register long datediff = ((mxDateTimeObject *)b)->absdate -
	                         ((mxDateTimeObject *)a)->absdate;
	register double timediff = ((mxDateTimeObject *)b)->abstime -
	                         ((mxDateTimeObject *)a)->abstime;

	if ((datediff >= 0 && datediff <= (long)(acc / SECONDS_PER_DAY)) ||
	    (datediff < 0 && -datediff <= (long)(acc / SECONDS_PER_DAY))) {
	    if ((!DOUBLE_IS_NEGATIVE(timediff) && timediff <= acc) ||
		(DOUBLE_IS_NEGATIVE(timediff) && -timediff <= acc))
		return PyInt_FromLong(0l);
	    else if (DOUBLE_IS_NEGATIVE(timediff))
		return PyInt_FromLong(1l);
	    else
		return PyInt_FromLong(-1l);
	}
	else if (datediff < 0)
	    return PyInt_FromLong(1l);
	else
	    return PyInt_FromLong(-1l);
    }

    else if (_mxDateTimeDelta_Check(a) && _mxDateTimeDelta_Check(b)) {
	/* cmp(DateTimeDelta,DateTimeDelta) */
	register double timediff = ((mxDateTimeDeltaObject *)b)->seconds -
	                           ((mxDateTimeDeltaObject *)a)->seconds;
	
	if ((!DOUBLE_IS_NEGATIVE(timediff) && timediff <= acc) ||
	    (DOUBLE_IS_NEGATIVE(timediff) && -timediff <= acc))
	    return PyInt_FromLong(0l);
	else if (DOUBLE_IS_NEGATIVE(timediff))
	    return PyInt_FromLong(1l);
	else
	    return PyInt_FromLong(-1l);
    }

    else
	Py_Error(PyExc_TypeError,
		 "objects must be DateTime[Delta] instances");

 onError:
    return NULL;
}

Py_C_Function( mxDateTime_setnowapi,
	       "setnowapi(fct)\n\n"
	       "Sets the current time API used by now(). This must be\n"
	       "a callable function which returns the current local time in\n"
	       "Unix ticks."
	       )
{
    PyObject *v;

    Py_GetArg("O",v);

    Py_Assert(PyCallable_Check(v),
	      PyExc_TypeError,
	      "argument must be callable");

    Py_INCREF(v);
    mxDateTime_nowapi = v;

    Py_ReturnNone();
 onError:
    return NULL;
}

/* --- module interface ---------------------------------------------------- */

/* Python Method Table */

static 
PyMethodDef Module_methods[] =
{   
    Py_MethodListEntryNoArgs("now",mxDateTime_now),
    Py_MethodListEntry("DateTime",mxDateTime_DateTime),
    Py_MethodListEntry("DateTimeDelta",mxDateTime_DateTimeDelta),
#ifdef HAVE_STRPTIME
    Py_MethodListEntry("strptime",mxDateTime_strptime),
#endif
    Py_MethodListEntry("DateTimeFromCOMDate",mxDateTime_DateTimeFromCOMDate),
    Py_MethodListEntry("DateTimeFromAbsDateTime",mxDateTime_DateTimeFromAbsDateTime),
    Py_MethodListEntry("DateTimeFromAbsDays",mxDateTime_DateTimeFromAbsDays),
    Py_MethodListEntry("DateTimeDeltaFromSeconds",mxDateTime_DateTimeDeltaFromSeconds),
    Py_MethodListEntry("DateTimeDeltaFromDays",mxDateTime_DateTimeDeltaFromDays),
    Py_MethodListEntry("cmp",mxDateTime_cmp),
    Py_MethodListEntryNoArgs("utc",mxDateTime_utc),
    Py_MethodListEntry("JulianDateTime",mxDateTime_JulianDateTime),
    Py_MethodListEntry("setnowapi",mxDateTime_setnowapi),
#ifdef OLD_INTERFACE
    Py_MethodListEntry("Date",mxDateTime_DateTime),
    Py_MethodListEntry("Time",mxDateTime_TimeDelta),
    Py_MethodListEntry("Timestamp",mxDateTime_DateTime),
    Py_MethodListEntry("TimeDelta",mxDateTime_TimeDelta),
    Py_MethodListEntry("DateTimeFromTicks",mxDateTime_DateTimeFromTicks),
    Py_MethodListEntry("DateTimeDeltaFromTicks",mxDateTime_DateTimeDeltaFromSeconds),
#endif
    {NULL,NULL} /* end of list */
};

/* C API table - always add new things to the end for binary
   compatibility. */
static
mxDateTimeModule_APIObject mxDateTimeModuleAPI =
{
    &mxDateTime_Type,
    mxDateTime_FromAbsDateAndTime,
    mxDateTime_FromTuple,
    mxDateTime_FromDateAndTime,
    mxDateTime_FromTmStruct,
    mxDateTime_FromTicks,
    mxDateTime_FromCOMDate,
    mxDateTime_AsTmStruct,
    mxDateTime_AsTicks,
    mxDateTime_AsCOMDate,
    &mxDateTimeDelta_Type,
    mxDateTimeDelta_FromDaysEx,
    mxDateTimeDelta_FromTime,
    mxDateTimeDelta_FromTuple,
    mxDateTimeDelta_FromTimeTuple,
    mxDateTimeDelta_AsDouble,
    mxDateTime_FromAbsDays,
    mxDateTime_AsAbsDays,
    mxDateTimeDelta_FromDays,
    mxDateTimeDelta_AsDays,
    mxDateTime_BrokenDown,
    mxDateTimeDelta_BrokenDown
};

/* Cleanup function */
static 
void mxDateTimeModule_Cleanup(void)
{
#ifdef MXDATETIME_FREELIST
    {
	mxDateTimeObject *d = mxDateTime_FreeList;
	while (d != NULL) {
	    mxDateTimeObject *v = d;
	    d = *(mxDateTimeObject **)d;
	    free(v);
	}
    }
#endif
#ifdef MXDATETIMEDELTA_FREELIST
    {
	mxDateTimeDeltaObject *d = mxDateTimeDelta_FreeList;
	while (d != NULL) {
	    mxDateTimeDeltaObject *v = d;
	    d = *(mxDateTimeDeltaObject **)d;
	    free(v);
	}
    }
#endif
    /* Drop reference to the now API callable. */
    Py_XDECREF(mxDateTime_nowapi);
    mxDateTime_nowapi = NULL;
}

/* Create PyMethodObjects and register them in the module's dict */
MX_EXPORT(void) 
     initmxDateTime(void)
{
    PyObject *module, *moddict;
    static int initialized = 0;

    if (initialized)
	Py_Error(PyExc_SystemError,
		 "can't initialize "MXDATETIME_MODULE" more than once");

    /* Init type objects */
    PyType_Init(mxDateTime_Type);
    PyType_Init(mxDateTimeDelta_Type);

    /* Init globals */
    mxDateTime_POSIXConform = mxDateTime_POSIX();

    /* Create module */
    module = Py_InitModule4(MXDATETIME_MODULE, /* Module name */
			    Module_methods, /* Method list */
			    Module_docstring, /* Module doc-string */
			    (PyObject *)NULL, /* always pass this as *self */
			    PYTHON_API_VERSION); /* API Version */
    if (module == NULL)
	goto onError;

    /* Register cleanup function */
    if (Py_AtExit(mxDateTimeModule_Cleanup)) {
	/* XXX what to do if we can't register that function ??? */
	DPRINTF("* Failed to register mxDateTime cleanup function\n");
    }

    /* Add some constants to the module's dict */
    moddict = PyModule_GetDict(module);
    if (moddict == NULL)
	goto onError;
    insobj(moddict,"__version__",PyString_FromString(MXDATETIME_VERSION));
    insint(moddict,"POSIX",mxDateTime_POSIXConform);

    /* Calendar string constants */
    if (!(mxDateTime_GregorianCalendar = PyString_FromString("Gregorian")))
	goto onError;
    PyString_InternInPlace(&mxDateTime_GregorianCalendar);
    PyDict_SetItemString(moddict,"Gregorian",mxDateTime_GregorianCalendar);
    if (!(mxDateTime_JulianCalendar = PyString_FromString("Julian")))
	goto onError;
    PyString_InternInPlace(&mxDateTime_JulianCalendar);
    PyDict_SetItemString(moddict,"Julian",mxDateTime_JulianCalendar);

    /* Errors */
    if (!(mxDateTime_Error = insexc(moddict,"Error",PyExc_StandardError)))
	goto onError;
    if (!(mxDateTime_RangeError = insexc(moddict,"RangeError",
					 mxDateTime_Error)))
	goto onError;

    /* Type objects */
    Py_INCREF(&mxDateTime_Type);
    PyDict_SetItemString(moddict,"DateTimeType",
			 (PyObject *)&mxDateTime_Type);
    Py_INCREF(&mxDateTimeDelta_Type);
    PyDict_SetItemString(moddict,"DateTimeDeltaType",
			 (PyObject *)&mxDateTimeDelta_Type);

    /* Export C API; many thanks to Jim Fulton for pointing this out to me */
    insobj(moddict,MXDATETIME_MODULE"API",
	   PyCObject_FromVoidPtr((void *)&mxDateTimeModuleAPI, NULL));

    DPRINTF("* Loaded "MXDATETIME_MODULE" C extension at 0x%0x.\n",
	    (int)module);
    DPRINTF("Notes: "
	    "sizeof(time_t) = %i, sizeof(int) = %i, sizeof(long) = %i\n",
	    sizeof(time_t),sizeof(int), sizeof(long));
    initialized = 1;

 onError:
    /* Check for errors and report them */
    if (PyErr_Occurred())
	Py_ReportModuleInitError(MXDATETIME_MODULE);
    return;
}
