/* $Id: notebook.cc,v 1.17 2002/02/13 03:53:56 bergo Exp $ */

/*

    eboard - chess client
    http://eboard.sourceforge.net
    Copyright (C) 2000-2002 Felipe Paulo Guazzi Bergo
    bergo@seul.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <iostream.h>
#include "eboard.h"
#include "global.h"
#include "notebook.h"

#include "dead.xpm"

Page::Page() {
  global.debug("Page","Page","1");
  number=0;
  content=0;
  title[0]=0;
  naughty=false;
  level=IM_ZERO;
  label=gtk_label_new("<none>");
  gtk_widget_show(label);
}

Page::Page(int _number,GtkWidget *_content, char *_title) {
  global.debug("Page","Page","2");
  number=_number;
  content=_content;
  title[255]=0;
  level=IM_ZERO;
  naughty=false;
  strncpy(title,_title,255);
  label=gtk_label_new(title);
  gtk_widget_show(label);
}

int Page::operator==(int n) {
  return(n==number);
}

int Page::under(Importance imp) {
  return(level < imp);
}

int Page::above(Importance imp) {
  return(level > imp);
}

void Page::setLevel(Importance imp) {
  level=imp;
}

void Page::setTitle(char *_title) {
  title[255]=0;
  strncpy(title,_title,255);
  gtk_label_set_text(GTK_LABEL(label),title);
  gtk_widget_queue_resize(label);
}

char * Page::getTitle() {
  return(title);
}

void Page::renumber(int newid) {
  char z[258];
  number=newid;
  sprintf(z,"{%s}",title);
  setTitle(z);
}

void Page::dump() {
  cerr.setf(ios::hex,ios::basefield);
  cerr.setf(ios::showbase);
  cerr << "[page " << ((unsigned int)this) << "] ";
  cerr.setf(ios::dec,ios::basefield);
  cerr << "pageid=" << number << " ";
  cerr << "title=" << title << " ";
  cerr << "level=" << level << endl;
}

Notebook::Notebook() {
  global.debug("Notebook","Notebook");
  pcl=0;
  widget=gtk_notebook_new();
  gtk_notebook_set_tab_pos(GTK_NOTEBOOK(widget),GTK_POS_RIGHT);
  gtk_signal_connect(GTK_OBJECT(widget),"switch_page",
		     GTK_SIGNAL_FUNC(notebook_switch_page),
		     (gpointer)this);
}

void Notebook::addPage(GtkWidget *what,char *title,int id) {
  Page *pg;
  global.debug("Notebook","addPage",title);
  pg=new Page(id,what,title);
  pages.push_back(pg);  
  gtk_notebook_append_page(GTK_NOTEBOOK(widget),what,pg->label);

  /* to test graphics only, adds dead.xpm to console tab
  if (pages.size() == 2)
    addBombToTab(pg);
  */
}

void Notebook::removePage(int id) {
  vector<Page *>::iterator pi;
  int i;
  global.debug("Notebook","removePage");
  for(i=0,pi=pages.begin();pi!=pages.end();pi++,i++)
    if ( (*(*pi)) == id ) {
      gtk_notebook_remove_page(GTK_NOTEBOOK(widget),i);
      delete(*pi);
      pages.erase(pi);
      return;
    }  
}

int Notebook::getCurrentPage() {
  return(gtk_notebook_get_current_page(GTK_NOTEBOOK(widget)));
}

int Notebook::getCurrentPageId() {
  vector<Page *>::iterator pi;
  int i;
  i=getCurrentPage();
  if (i >= pages.size())
    return 0;
  return( pages[i]->number );
}

void Notebook::setTabColor(int pageid,int color,Importance imp=IM_TOP) {
  vector<Page *>::iterator pi;
  int i;
  global.debug("Notebook","setTabColor","1");
  for(i=0,pi=pages.begin();pi!=pages.end();pi++,i++)
    if ( (*(*pi)) == pageid ) {
      setTabColor(i,color,0,imp);
      return;
    }
}

void Notebook::setTabColor(int page_num,int color,int poly,
			   Importance imp=IM_TOP) {
  vector<Page *>::iterator pi;
  GdkColor nc;
  GtkStyle *style;
  int i;
  i=page_num;

  global.debug("Notebook","setTabColor","2");
  for(pi=pages.begin();i;pi++,i--) ;

  if ((*pi)->above(imp))
    return;

  if (imp==IM_RESET) imp=IM_ZERO;
  (*pi)->setLevel(imp);

  nc.red=((color>>16)<<8);
  nc.green=(color&0x00ff00);
  nc.blue=((color&0xff)<<8);
  style=gtk_style_new();
  style->fg[0]=nc;
  style->fg[1]=nc;
  style->fg[2]=nc;
  style->fg[3]=nc;
  style->fg[4]=nc;
  gtk_widget_set_style( (*pi)->label, style );
  gtk_widget_queue_draw( (*pi)->label );
}

void Notebook::goToPageId(int id) {
  vector<Page *>::iterator pi;
  int i;
  for(i=0,pi=pages.begin();pi!=pages.end();pi++,i++)
    if ( (*(*pi)) == id ) {
      gtk_notebook_set_page(GTK_NOTEBOOK(widget),i);
      return;
    }
}

void Notebook::setTabPosition(GtkPositionType pos) {
  gtk_notebook_set_tab_pos(GTK_NOTEBOOK(widget),pos);
}

void Notebook::dump() {
  vector<Page *>::iterator pi;
  for(pi=pages.begin();pi!=pages.end();pi++)
    (*pi)->dump();
}

void Notebook::goToNext() {
  gtk_notebook_next_page(GTK_NOTEBOOK(widget));
}

void Notebook::goToPrevious() {
  gtk_notebook_prev_page(GTK_NOTEBOOK(widget));
}

void Notebook::renumberPage(int oldid,int newid) {
  vector<Page *>::iterator i;
  for(i=pages.begin();i!=pages.end();i++)
    if ( (*(*i)) == oldid )
      (*i)->renumber(newid);
}

void Notebook::setListener(PaneChangeListener *listener) {
  pcl=listener;
}

void Notebook::setNaughty(int pageid, bool val) {
  vector<Page *>::iterator i;
  for(i=pages.begin();i!=pages.end();i++)
    if ( (*(*i)) == pageid ) {

      if ( (val)&&(! (*i)->naughty) )
	addBombToTab(*i);
      (*i)->naughty=val;

    }
}

void Notebook::getNaughty(vector<int> &evil_ids) {
  vector<Page *>::iterator i;
  evil_ids.clear();
  for(i=pages.begin();i!=pages.end();i++)
    if ( (*i)->naughty )
      evil_ids.push_back( (*i)->number );
}

bool Notebook::hasNaughty() {
  vector<Page *>::iterator i;
  for(i=pages.begin();i!=pages.end();i++)
    if ( (*i)->naughty )
      return true;
  return false;
}

// SOMEONE SET UP US THE BOMB!
void Notebook::addBombToTab(Page *pg) {
  GtkWidget *hb,*label,*pm;
  GdkPixmap *d;
  GdkBitmap *b;
  GtkStyle *style;

  style=gtk_widget_get_default_style();

  d = gdk_pixmap_create_from_xpm_d (widget->window, &b,
				    &style->bg[GTK_STATE_NORMAL],
				    (gchar **) dead_xpm);
  pm=gtk_pixmap_new(d,b);

  label=gtk_label_new(pg->getTitle());
  hb=gtk_hbox_new(FALSE,2);
  
  gtk_box_pack_start(GTK_BOX(hb),label,FALSE,FALSE,4);
  gtk_box_pack_start(GTK_BOX(hb),pm,FALSE,FALSE,4);

  gtk_widget_show(label);
  gtk_widget_show(pm);
  gtk_widget_show(hb);
  
  pg->label=label;
  gtk_notebook_set_tab_label(GTK_NOTEBOOK(widget),pg->content,hb);
}

void Notebook::pretendPaneChanged() {
  if (pcl)
    pcl->paneChanged( getCurrentPage(), getCurrentPageId() );
}

void
notebook_switch_page(GtkNotebook *notebook,
		     GtkNotebookPage *page,
		     gint page_num,
		     gpointer data) {
  Notebook *me;
  me=(Notebook *)data;
  me->setTabColor(page_num,0,0,IM_RESET);
  if ( (me->pcl) && (me->pages.size() > page_num) )
    me->pcl->paneChanged(page_num, me->pages[page_num]->number);
}

// ================================================================

NotebookInsider::NotebookInsider() {
  mynotebook=0;
  pgid=0;
}

void NotebookInsider::setNotebook(Notebook *nb,int pageid) {
  pgid=pageid;
  mynotebook=nb;
}

void NotebookInsider::contentUpdated() {
  int color=0xff0000;
  Importance maxi=IM_NORMAL;
  global.debug("NotebookInsider","contentUpdated");
  if (!impstack.empty()) {
    maxi=IM_IGNORE;
    while(!impstack.empty()) {
      if (impstack.top() > maxi) maxi=impstack.top();
      impstack.pop();
    }
    switch(maxi) {
    case IM_IGNORE:
      return;
    case IM_NORMAL:
      color=0xff0000;
      break;
    case IM_PERSONAL:
      color=0x0000ff;
      break;
    }
  }
  if (!mynotebook)
    return;
  if (mynotebook->getCurrentPageId() != pgid)
    mynotebook->setTabColor(pgid,color,maxi);
}

void NotebookInsider::pop() {
  global.debug("NotebookInsider","pop");
  if (!mynotebook)
    return;
  mynotebook->goToPageId(pgid);
}

void NotebookInsider::pushLevel(Importance imp) {
  impstack.push(imp);
}

int NotebookInsider::getPageId() {
  return(pgid);
}
