/* $Id: global.cc,v 1.48 2002/03/17 20:51:47 bergo Exp $ */

/*

    eboard - chess client
    http://eboard.sourceforge.net
    Copyright (C) 2000-2002 Felipe Paulo Guazzi Bergo
    bergo@seul.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <iostream.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <dirent.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <time.h>
#include <signal.h>

#define GLOBAL_CC 1

#include "global.h"
#include "text.h"
#include "config.h"
#include "chess.h"
#include "tstring.h"
#include "notebook.h"
#include "board.h"
#include "quickbar.h"

Global global;

Global::Global() {
  input=0;
  output=0;
  network=0;
  status=0;
  protocol=0;
  chandler=0;
  promotion=0;
  ebook=0;
  skgraph=0;
  inputhistory=0;
  bmlistener=0;
  qbcontainer=0;
  quickbar=0;

  LastScratch = 0;

  Quitting=0;
  Version=VERSION;
  SelfInputColor = 0xc0c0ff;

  HilightLastMove=0;
  AnimateMoves=0;
  Premove=1;

  PasswordMode = 0;

  TabPos=0;

  CommLog=0;
  DebugLog=0;
  PauseLog=0;

  MainLevel=0;
  QuitPending=0;

  ScrollBack=1000;
  FicsAutoLogin=1;
  BeepWhenOppMoves=0;
  EnableSounds=0;
  PopupSecondaryGames=1;
  SmartDiscard=0;
  ShowCoordinates=0;

  PlainSquares=0;
  LightSqColor=0xe7cf93;
  DarkSqColor=0x9e8661;

  AntialiasPieces=1;
  ShowRating=1;

  VectorPieces=0;
  CheckLegality=0;
  DrawHouseStock=1;

  AppendPlayed=0;
  AppendObserved=0;
  strcpy(AppendFile,"~/.eboard/mygames.pgn");

  IcsSeekGraph=1;
  HideSeeks=0;

  SplitChannels=0;
  ChannelsToConsoleToo=0;

  ShowQuickbar=1;

  sndevents[1].Pitch=650;
  sndevents[1].Duration=350;

  sndevents[2].Pitch=900;
  sndevents[2].Duration=80;

  sndevents[3].Pitch=444;
  sndevents[3].Duration=40;
  sndevents[3].Count=3;

  strcpy(ClockFont,DEFAULT_FONT_CLOK);
  strcpy(PlayerFont,DEFAULT_FONT_PLYR);
  strcpy(InfoFont,DEFAULT_FONT_INFO);
  strcpy(ConsoleFont,DEFAULT_FONT_CONS);

  RCKeys.push_back("HilightLastMove"); //0
  RCKeys.push_back("AnimateMoves");
  RCKeys.push_back("Premove");         //2
  RCKeys.push_back("PieceSet");
  RCKeys.push_back("TabPos");          //4
  RCKeys.push_back("ClockFont");
  RCKeys.push_back("PlayerFont");      //6
  RCKeys.push_back("InfoFont");
  RCKeys.push_back("PlainSquares");    //8
  RCKeys.push_back("LightSqColor");
  RCKeys.push_back("DarkSqColor");     //10
  RCKeys.push_back("Host");
  RCKeys.push_back("Antialias");       //12
  RCKeys.push_back("ShowRating");
  RCKeys.push_back("ScrollBack");      //14
  RCKeys.push_back("FicsAutoLogin");
  RCKeys.push_back("BeepOpp");         //16
  RCKeys.push_back("SoundEvent");
  RCKeys.push_back("EnableSounds");    //18
  RCKeys.push_back("VectorPieces");
  RCKeys.push_back("CheckLegality");   //20
  RCKeys.push_back("AppendPlayed");
  RCKeys.push_back("AppendObserved");  //22
  RCKeys.push_back("AppendFile");
  RCKeys.push_back("ConsoleFont");     //24
  RCKeys.push_back("SeekGraph");
  RCKeys.push_back("HideSeeks");       //26
  RCKeys.push_back("SplitChannels");
  RCKeys.push_back("ChSplitAndCons");  //28
  RCKeys.push_back("DrawHouseStock");
  RCKeys.push_back("SquareSet");       //30
  RCKeys.push_back("PopupSecondaryGames");
  RCKeys.push_back("SmartDiscard");    //32
  RCKeys.push_back("ShowCoordinates");
  RCKeys.push_back("TerminalColors");  //34
  RCKeys.push_back("DesktopState");
  RCKeys.push_back("DesktopStateDC");  //36
  RCKeys.push_back("ShowQuickbar");
  RCKeys.push_back("QuickbarButton");  //38

  PopupHelp = false;
}

void Global::dropQuickbarButtons() {
  int i;
  for(i=0;i<QuickbarButtons.size();i++)
    delete(QuickbarButtons[i]);
  QuickbarButtons.clear();
}

void Global::clearDupes(ChessGame *cg) {
  list<ChessGame *>::iterator gi;
  for(gi=GameList.begin();gi!=GameList.end();gi++)
    if ( (*(*gi)) == cg->GameNumber )
      renumberGame(*gi,nextFreeGameId(8000));
}

void Global::deleteGame(ChessGame *cg) {
  list<ChessGame *>::iterator gi;
  for(gi=GameList.begin();gi!=GameList.end();gi++)
    if ( (*gi) == cg ) {
      GameList.erase(gi);
      return;
    }
}

void Global::appendGame(ChessGame *cg,bool RenumberDupes=true) {
  if (RenumberDupes) clearDupes(cg);
  GameList.push_back(cg);
}
void Global::prependGame(ChessGame *cg, bool RenumberDupes=true) {
  if (RenumberDupes) clearDupes(cg);
  GameList.push_front(cg);
}

void Global::renumberGame(ChessGame *cg,int id) {
  int oldid;
  oldid=cg->GameNumber;
  cg->GameNumber=id;
  // renumber notebook references
  if (ebook) ebook->renumberPage(oldid,id);
}

void Global::removeBoard(Board *b) {
  for(BLi=BoardList.begin();BLi!=BoardList.end();BLi++)
    if ( (*BLi) == b ) {
      BoardList.erase(BLi);
      return;
    }
  //  cerr << "<Global::removeBoard> ** board not found\n";
}

void Global::statOS() {
  FILE *p;
  p=popen("uname -s","r");
  if (!p) p=popen("/bin/uname -s","r");
  if (!p) p=popen("/sbin/uname -s","r");
  if (!p) p=popen("/usr/bin/uname -s","r");
  if (!p) p=popen("/usr/sbin/uname -s","r");
  if (!p) { strcpy(SystemType,"unknown"); return; }
  SystemType[63]=0;
  fgets(SystemType,64,p);
  pclose(p);
  if (SystemType[strlen(SystemType)-1]=='\n')
    SystemType[strlen(SystemType)-1]=0;
}

void Global::ensureDirectories() {
  char *home;
  char z[256];
  DIR *tdir;
  home=getenv("HOME");
  if (!home) {
    cerr << "[eboard] ** no $HOME" << endl;
    return;
  }
  sprintf(z,"%s/.eboard",home);

  tdir=opendir(z);
  if (tdir==NULL)
    PopupHelp = true;
  else
    closedir(tdir);
    
  if (createDir(z)) return;
  sprintf(z,"%s/.eboard/craftylog",home);
  createDir(z);
  sprintf(z,"%s/.eboard/eng-out",home);
  createDir(z);
  sprintf(z,"%s/.eboard/scripts",home);
  createDir(z);
}

int Global::createDir(char *z) {
  DIR *tdir;
  tdir=opendir(z);
  if (tdir)
    closedir(tdir);
  else
    if (mkdir(z,0755)) {
      cerr << "[eboard] ** failed to create directory " << z << endl;
      return -1;
    }
  return 0;
}

void Global::readRC() {
  TString t(512);
  static char *sep=" \n\t,:\r";
  static char *sep2="\n\t,:\r";
  char z[512],*p;
  FILE *f;
  HostBookmark *hbm;
  int i;
  QButton *qb;

  sprintf(z,"%s/.eboard/eboard.conf",getenv("HOME"));

  f=fopen(z,"r");
  if (!f)
    return;

  while(fgets(z,511,f)) {
    t.set(z);
    p=t.token(sep);
    if (!p) continue;
    if (p[0]=='#') continue;
    if (!strcmp(p,RCKeys[0])) {HilightLastMove=atoi(t.token(sep)); continue;}
    if (!strcmp(p,RCKeys[1])) {AnimateMoves=atoi(t.token(sep)); continue;}
    if (!strcmp(p,RCKeys[2])) {Premove=atoi(t.token(sep)); continue;}
    if (!strcmp(p,RCKeys[3])) {
      Board::setCurrentPieceSet(t.token("\n\r\t: "),true,true); 
      continue;
    }
    if (!strcmp(p,RCKeys[4])) {
      if (p=t.token(sep))
	switch(p[0]) {
	case 'R': TabPos=0; break;
	case 'L': TabPos=1; break;
	case 'T': TabPos=2; break;
	case 'B': TabPos=3; break;
	}
      continue;
    }
    if (!strcmp(p,RCKeys[5]))  { strncpy(ClockFont,t.token(sep2),96); continue; }
    if (!strcmp(p,RCKeys[6]))  { strncpy(PlayerFont,t.token(sep2),96); continue; }
    if (!strcmp(p,RCKeys[7]))  { strncpy(InfoFont,t.token(sep2),96); continue; }
    if (!strcmp(p,RCKeys[8]))  { PlainSquares=atoi(t.token(sep)); continue; }
    if (!strcmp(p,RCKeys[9]))  { LightSqColor=strtol(t.token(sep),0,16); continue; }
    if (!strcmp(p,RCKeys[10])) { DarkSqColor=strtol(t.token(sep),0,16); continue; }

    if (!strcmp(p,RCKeys[11])) {
      hbm=new HostBookmark();
      strcpy(hbm->host,t.token(",\n\r"));
      hbm->port=atoi(t.token(",\n\r"));
      strcpy(hbm->protocol,t.token(",\n\r"));
      HostHistory.push_back(hbm);
    }

    if (!strcmp(p,RCKeys[12])) { AntialiasPieces=atoi(t.token(sep)); continue; }
    if (!strcmp(p,RCKeys[13])) { ShowRating=atoi(t.token(sep)); continue; }
    if (!strcmp(p,RCKeys[14])) { ScrollBack=atoi(t.token(sep)); continue; }
    if (!strcmp(p,RCKeys[15])) { FicsAutoLogin=atoi(t.token(sep)); continue; }
    if (!strcmp(p,RCKeys[16])) { BeepWhenOppMoves=atoi(t.token(sep)); continue; }

    // sound events
    if (!strcmp(p,RCKeys[17])) {
      i=atoi(t.token(sep));
      if (i<4)
	sndevents[i].read(t);
      continue;
    }

    if (!strcmp(p,RCKeys[18])) { EnableSounds=atoi(t.token(sep)); continue; }
    if (!strcmp(p,RCKeys[19])) { VectorPieces=atoi(t.token(sep)); continue; }
    if (!strcmp(p,RCKeys[20])) { CheckLegality=atoi(t.token(sep)); continue; }
    if (!strcmp(p,RCKeys[21])) { AppendPlayed=atoi(t.token(sep)); continue; }
    if (!strcmp(p,RCKeys[22])) { AppendObserved=atoi(t.token(sep)); continue; }
    if (!strcmp(p,RCKeys[23])) { strncpy(AppendFile,t.token(sep),128); continue; }
    if (!strcmp(p,RCKeys[24])) { strncpy(ConsoleFont,t.token(sep2),96); continue; }
    if (!strcmp(p,RCKeys[25])) { IcsSeekGraph=atoi(t.token(sep)); continue; }
    if (!strcmp(p,RCKeys[26])) { HideSeeks=atoi(t.token(sep)); continue; }
    if (!strcmp(p,RCKeys[27])) { SplitChannels=atoi(t.token(sep)); continue; }
    if (!strcmp(p,RCKeys[28])) { ChannelsToConsoleToo=atoi(t.token(sep)); continue; }
    if (!strcmp(p,RCKeys[29])) { DrawHouseStock=atoi(t.token(sep)); continue; }
    if (!strcmp(p,RCKeys[30])) {
      strcpy(z,t.token("\n\r\t: "));
      if (strcmp(Board::getCurrentSquareSet(),z))
	Board::setCurrentPieceSet(z,false,true);
      continue;
    }
    if (!strcmp(p,RCKeys[31])) { PopupSecondaryGames=atoi(t.token(sep)); continue; }
    if (!strcmp(p,RCKeys[32])) { SmartDiscard=atoi(t.token(sep)); continue; }
    if (!strcmp(p,RCKeys[33])) { ShowCoordinates=atoi(t.token(sep)); continue; }
    if (!strcmp(p,RCKeys[34])) { Colors.read(t); continue; }
    if (!strcmp(p,RCKeys[35])) { Desk.read(t); continue; }
    if (!strcmp(p,RCKeys[36])) { Desk.readConsole(t); continue; }
    if (!strcmp(p,RCKeys[37])) { ShowQuickbar=atoi(t.token(sep)); continue; }
    if (!strcmp(p,RCKeys[38])) { 
      qb=new QButton();
      qb->icon=atoi(t.token("\n\r\t:"));
      qb->caption=t.token("\n\r\t:");
      qb->command=t.token("\n\r\t:");
      QuickbarButtons.push_back(qb);
    }

  }
  fclose(f);  
}

void Global::writeRC() {
  char z[512];
  FILE *f;
  list<HostBookmark *>::iterator bi;
  static char *tabpos="RLTB";
  int i;

  sprintf(z,"%s/.eboard/eboard.conf",getenv("HOME"));

  f=fopen(z,"w");
  if (!f)
    return;

  fprintf(f,"%s::%d\n",RCKeys[0],HilightLastMove);
  fprintf(f,"%s::%d\n",RCKeys[1],AnimateMoves);
  fprintf(f,"%s::%d\n",RCKeys[2],Premove);
  fprintf(f,"%s::%s\n",RCKeys[3],Board::getCurrentPieceSet());
  fprintf(f,"%s::%c\n",RCKeys[4],tabpos[TabPos%4]);
  fprintf(f,"%s::%s\n",RCKeys[5],ClockFont);
  fprintf(f,"%s::%s\n",RCKeys[6],PlayerFont);
  fprintf(f,"%s::%s\n",RCKeys[7],InfoFont);

  fprintf(f,"%s::%d\n",RCKeys[8],PlainSquares);
  fprintf(f,"%s::%x\n",RCKeys[9],LightSqColor);
  fprintf(f,"%s::%x\n",RCKeys[10],DarkSqColor);

  for(bi=HostHistory.begin();bi!=HostHistory.end();bi++)
    fprintf(f,"%s,%s,%d,%s\n",RCKeys[11],
	    (*bi)->host,(*bi)->port,(*bi)->protocol);

  fprintf(f,"%s::%d\n",RCKeys[12],AntialiasPieces);
  fprintf(f,"%s::%d\n",RCKeys[13],ShowRating);
  fprintf(f,"%s::%d\n",RCKeys[14],ScrollBack);
  fprintf(f,"%s::%d\n",RCKeys[15],FicsAutoLogin);
  fprintf(f,"%s::%d\n",RCKeys[16],BeepWhenOppMoves);

  for(i=0;i<4;i++)
    sndevents[i].write(f,RCKeys[17],i);

  fprintf(f,"%s::%d\n",RCKeys[18],EnableSounds);
  fprintf(f,"%s::%d\n",RCKeys[19],VectorPieces);
  fprintf(f,"%s::%d\n",RCKeys[20],CheckLegality);
  fprintf(f,"%s::%d\n",RCKeys[21],AppendPlayed);
  fprintf(f,"%s::%d\n",RCKeys[22],AppendObserved);
  fprintf(f,"%s::%s\n",RCKeys[23],AppendFile);

  fprintf(f,"%s::%s\n",RCKeys[24],ConsoleFont);
  fprintf(f,"%s::%d\n",RCKeys[25],IcsSeekGraph);
  fprintf(f,"%s::%d\n",RCKeys[26],HideSeeks);
  fprintf(f,"%s::%d\n",RCKeys[27],SplitChannels);
  fprintf(f,"%s::%d\n",RCKeys[28],ChannelsToConsoleToo);
  fprintf(f,"%s::%d\n",RCKeys[29],DrawHouseStock);
  fprintf(f,"%s::%s\n",RCKeys[30],Board::getCurrentSquareSet());
  fprintf(f,"%s::%d\n",RCKeys[31],PopupSecondaryGames);
  fprintf(f,"%s::%d\n",RCKeys[32],SmartDiscard);
  fprintf(f,"%s::%d\n",RCKeys[33],ShowCoordinates);
  fprintf(f,"%s::",RCKeys[34]); Colors.write(f);
  fprintf(f,"%s::",RCKeys[35]); Desk.write(f);
  Desk.writeConsoles(f,RCKeys[36]);
  fprintf(f,"%s::%d\n",RCKeys[37],ShowQuickbar);
  for(i=0;i<QuickbarButtons.size();i++)
    fprintf(f,"%s::%d:%s:%s\n",RCKeys[38], 
	    QuickbarButtons[i]->icon,
	    QuickbarButtons[i]->caption.c_str(),
	    QuickbarButtons[i]->command.c_str());

  fclose(f);
}

ChessGame * Global::getGame(int num) {
  list<ChessGame *>::iterator gi;
  for(gi=GameList.begin();gi!=GameList.end();gi++)
    if ( (*(*gi)) == num )
      return(*gi);
  return 0;
}

int Global::nextFreeGameId(int base) {
  int v;
  for(v=base;getGame(v)!=0;v++) ;
  return v;
}

void Global::WrappedMainIteration() {
  MainLevel++;
  gtk_main_iteration();
  MainLevel--;
  if ((!MainLevel)&&(QuitPending))
    Global::WrappedMainQuit();
}

void Global::WrappedMainQuit() {
  if (MainLevel) {
    QuitPending++;
    return;
  }
  QuitPending=0;
  signal(SIGCHLD,SIG_DFL); // prevent the crash reported by gcp
  gtk_main_quit();
}

void Global::addAgent(NetConnection *ag) {
  Agents.push_back(ag);
}

void Global::removeAgent(NetConnection *ag) {
  list<NetConnection *>::iterator ni;
  for(ni=Agents.begin();ni!=Agents.end();ni++)
    if ( (*ni) == ag ) {
      Agents.erase(ni);
      return;
    }
}

void Global::agentBroadcast(char *z) {
  list<NetConnection *>::iterator ni;
  if (Agents.empty())
    return;
  for(ni=Agents.begin();ni!=Agents.end();ni++)
    if ((*ni)->isConnected())
      (*ni)->writeLine(z);
}

int  Global::receiveAgentLine(char *dest,int limit) {
  list<NetConnection *>::iterator ni;
  global.debug("Global","receiveAgentLine");
  if (Agents.empty())
    return 0;
  for(ni=Agents.begin();ni!=Agents.end();ni++)
    if ( (*ni)->isConnected())
      if ((*ni)->readLine(dest,limit)==0)
	return 1;
  return 0;
}

void Global::opponentMoved() {
  if (BeepWhenOppMoves) {
    if (AnimateMoves)
      SoundStack.push(0);
    else
      sndevents[0].safePlay();
  }
}

/*
void Global::clearSoundStack() {
  while(!SoundStack.empty())
    SoundStack.pop();
}
*/

void Global::flushSound() {
  if (!SoundStack.empty()) {
    sndevents[SoundStack.top()].safePlay();
    SoundStack.pop();
  }
}

void Global::drawOffered() {
  if (EnableSounds)
    sndevents[1].safePlay();
}

void Global::privatelyTold() {
  if (EnableSounds)
    sndevents[2].safePlay();
}

void Global::challenged() {
  if (EnableSounds)
    sndevents[3].safePlay();
}

void Global::repaintAllBoards() {
  for(BLi=BoardList.begin();BLi!=BoardList.end();BLi++)
    (*BLi)->refreshPieceSet();
}

bool Global::hasSoundFile(char *p) {
  vector<string>::iterator vi;
  for(vi=SoundFiles.begin();vi!=SoundFiles.end();vi++)
    if ( (*vi) == p )
      return true;
  return false;
}

void Global::setPasswordMode(int pm) {
  list<DetachedConsole *>::iterator i;
  PasswordMode = pm;
  for(i=Consoles.begin();i!=Consoles.end();i++)
    (*i)->setPasswordMode(pm);
}

void Global::debug(char *klass,char *method,char *data=0) {
  FILE *f;
  char z[256],rm[512];
  time_t now;

  if (!DebugLog)
    return;

  sprintf(z,"%s/DEBUG.eboard",getenv("HOME"));
  f=fopen(z,"a");
  if (!f) return;

  sprintf(rm,"+ %s::%s",klass,method);
  if (data) {
    strcat(rm," [");
    strcat(rm,data);
    strcat(rm,"]");
  }

  now=time(0);
  strftime(z,255,"%Y-%b-%d %H:%M:%S",localtime(&now));
  fprintf(f,"%s [%d] %s\n",z,getpid(),rm);
  fclose(f);
}

void Global::LogAppend(char *msg) {
  FILE *f;
  char z[256],rm[512],*p,*q;
  static char hexa[17]="0123456789abcdef";
  time_t now;

  if (PauseLog)
    msg="(message obfuscated -- password mode ?)";

  if (CommLog) {
    sprintf(z,"%s/LOG.eboard",getenv("HOME"));
    f=fopen(z,"a");
    if (!f) return;
    p=msg; q=rm;
    memset(rm,0,512);
    for(;*p;p++) {
      if ( *p == '\n' ) { *q++='\\'; *q++='n'; continue; }
      if ( *p == '\r' ) { *q++='\\'; *q++='r'; continue; }
      if ( *p < 32 ) { 
	*q++='('; 
	*q++='0';
	*q++='x';
	*q++=hexa[(*p)>>4];
	*q++=hexa[(*p)&0x0f];
	*q++=')';
      } else
	*q++=*p;
    }
    now=time(0);
    strftime(z,255,"%Y-%b-%d %H:%M:%S",localtime(&now));
    fprintf(f,"%s [%d] %s\n",z,getpid(),rm);
    fclose(f);
  }
}

void Global::dumpGames() {
  cerr.setf(ios::dec,ios::basefield);
  cerr << " GAME LIST (" << GameList.size() << " elements)\n";
  cerr << "--------------------------------------------------------------------------\n";
  for(GLi=GameList.begin();GLi!=GameList.end();GLi++)
    (*GLi)->dump();
  cerr << "--------------------------------------------------------------------------\n";
}

void Global::dumpBoards() {
  cerr.setf(ios::dec,ios::basefield);
  cerr << " BOARD LIST (" << BoardList.size() << " elements)\n";
  cerr << "--------------------------------------------------------------------------\n";
  for(BLi=BoardList.begin();BLi!=BoardList.end();BLi++)
    (*BLi)->dump();
  cerr << "--------------------------------------------------------------------------\n";
}

void Global::dumpPanes() {
  cerr.setf(ios::dec,ios::basefield);
  cerr << " PANE LIST\n";
  cerr << "--------------------------------------------------------------------------\n";
  ebook->dump();
  cerr << "--------------------------------------------------------------------------\n";
}

void Global::addHostBookmark(HostBookmark *hbm) {
  list<HostBookmark *>::iterator bi;

  for(bi=HostHistory.begin();bi!=HostHistory.end();bi++)
    if ( (*(*bi)) == hbm ) {
      delete hbm;
      return;
    }
  HostHistory.push_front(hbm);
  if (HostHistory.size() > 16) {
    delete(HostHistory.back());
    HostHistory.pop_back();
  }
    
  writeRC();
  if (bmlistener != 0) bmlistener->updateBookmarks();
}

void Global::updateScrollBacks() {
  output->updateScrollBack();
  updateChannelScrollBacks();
}

Notebook * Global::getNotebook() {
  return(ebook);
}

void Global::gatherConsoleState() {
  list<DetachedConsole *>::iterator i;

  // please make Desk.consoles empty before calling this. Thanks.

  for(i=Consoles.begin();i!=Consoles.end();i++)
    Desk.addConsole(*i);
}

// malloc has the stupid idea of segfaulting when
// allocating a word-incomplete size
void * Global::safeMalloc(int nbytes) {
  return(malloc(nbytes + (nbytes % 4)));
}

GdkFont * Global::loadFont(EboardFont i) {
  GdkFont *f;

  switch(i) {

  case EF_PlayerFont: // player font (default 18 pt)
    f=gdk_font_load(PlayerFont);
    if (!f) f=gdk_font_load("-*-helvetica-bold-r-normal--17-*-*-*-*-*-*");
    if (!f) f=gdk_font_load("-*-helvetica-bold-r-normal--14-*-*-*-*-*-*");
    if (!f) f=gdk_font_load("helvetica");
    break;       

  case EF_ClockFont: // clock font (default 34 pt)
    f=gdk_font_load(ClockFont);
    if (!f) f=gdk_font_load("-*-helvetica-bold-r-normal--34-*-*-*-*-*-*");
    if (!f) f=gdk_font_load("-*-helvetica-bold-r-normal--25-*-*-*-*-*-*");
    if (!f) f=gdk_font_load("-*-helvetica-bold-r-normal--24-*-*-*-*-*-*");
    if (!f) f=gdk_font_load("helvetica");
    break;

  case EF_InfoFont: // info font (default 12 pt)
    f=gdk_font_load(InfoFont);
    if (!f) f=gdk_font_load("-*-helvetica-medium-r-normal--11-*-*-*-*-*-*");
    if (!f) f=gdk_font_load("-*-helvetica-medium-r-normal--10-*-*-*-*-*-*");
    if (!f) f=gdk_font_load("helvetica");
    break;
  default:
    f=NULL;
  }
  return f;

}

// ----------

int HostBookmark::operator==(HostBookmark *hbm) {
  if (strcmp(host,hbm->host)) return 0;
  if (port!=hbm->port) return 0;
  if (strcmp(protocol,hbm->protocol)) return 0;
  return 1;
}

// -------------------------------- channel splitting

void ChannelSplitter::ensurePane(int ch) {
  int i,j;
  j=panes.size();
  for(i=0;i<j;i++)
    if (numbers[i]==ch)
      return; // already exists
  createPane(ch);
}

void ChannelSplitter::createPane(int ch) {
  Notebook *nb;
  Text *op;
  char z[64];
  nb=getNotebook();
  if (!nb) return;  
  op=new Text();
  sprintf(z,"Channel %d",ch);
  op->show();
  nb->addPage(op->widget,z,-200-ch);
  op->setNotebook(nb,-200-ch);
  numbers.push_back(ch);
  panes.push_back(op);
}

void ChannelSplitter::appendToChannel(int ch,char *msg,int color,Importance im=IM_NORMAL) {
  int i,j;
  ensurePane(ch);
  j=panes.size();
  for(i=0;i<j;i++)
    if (numbers[i]==ch) {
      panes[i]->append(msg,color,im);
      panes[i]->contentUpdated();
      return;
    }  
}

void ChannelSplitter::updateChannelScrollBacks() {
  int i,j;
  j=panes.size();
  for(i=0;i<j;i++)
    panes[i]->updateScrollBack();
}

TerminalColor::TerminalColor() {
  TextDefault   = 0xeeeeee;
  TextBright    = 0xffffff;
  PrivateTell   = 0xffff00;
  NewsNotify    = 0xff8080;
  Mamer         = 0xffdd00;
  KibitzWhisper = 0xd38fd3;
  Shouts        = 0xddffdd;
  Seeks         = 0x80ff80;
  ChannelTell   = 0x3cd9d1;
  Engine        = 0xc0ff60;
  Background    = 0;
}

void TerminalColor::read(TString &t) {
  static char *comma=",:\n\r \t";  

  TextDefault   = strtol(t.token(comma), 0, 16);
  TextBright    = strtol(t.token(comma), 0, 16);
  PrivateTell   = strtol(t.token(comma), 0, 16);
  NewsNotify    = strtol(t.token(comma), 0, 16);
  Mamer         = strtol(t.token(comma), 0, 16);
  KibitzWhisper = strtol(t.token(comma), 0, 16);
  Shouts        = strtol(t.token(comma), 0, 16);
  Seeks         = strtol(t.token(comma), 0, 16);
  ChannelTell   = strtol(t.token(comma), 0, 16);
  Engine        = strtol(t.token(comma), 0, 16);
  Background    = strtol(t.token(comma), 0, 16);
}

void TerminalColor::write(FILE *f) {
  fprintf(f,"%x,%x,%x,%x,%x,%x,%x,%x,%x,%x,%x\n",
	  TextDefault,TextBright,PrivateTell,NewsNotify,
	  Mamer,KibitzWhisper,Shouts,Seeks,ChannelTell,
	  Engine,Background);
}

// ---- desktop saving

WindowGeometry::WindowGeometry(int a,int b,int c,int d) {
  X=a; Y=b; W=c; H=d;
}

WindowGeometry::WindowGeometry() {
  setNull();
}

void WindowGeometry::retrieve(GtkWidget *w) {
  gint a[7];  
  gdk_window_get_geometry(w->window,a,a+1,a+2,a+3,a+4);
  gdk_window_get_origin(w->window,a+5,a+6);
  X=a[5]-a[0];
  Y=a[6]-a[1];
  W=a[2];
  H=a[3];
}

bool WindowGeometry::isNull() {
  return( (X==0)&&(Y==0)&&(W==0)&&(H==0) );
}

void WindowGeometry::setNull() {
  X=Y=W=H=0;
}

void WindowGeometry::write(FILE *f) {
  fprintf(f,"(%d,%d,%d,%d)",X,Y,W,H);
}

void WindowGeometry::read(TString &t) {
  static char *sep=":,()\n\t\r ";
  X=atoi(t.token(sep));
  Y=atoi(t.token(sep));
  W=atoi(t.token(sep));
  H=atoi(t.token(sep));
}

// --------

void Desktop::clear() {
  vector<WindowGeometry *>::iterator i;
  vector<string *>::iterator j;

  wMain.setNull();
  wGames.setNull();
  wLocal.setNull();
  wAds.setNull();

  for(i=consoles.begin();i!=consoles.end();i++)
    delete(*i);

  for(j=cfilters.begin();j!=cfilters.end();j++)
    delete(*j);

  consoles.clear();
  cfilters.clear();
}

void Desktop::write(FILE *f) {
  wMain.write(f);    fprintf(f,",");
  wGames.write(f);   fprintf(f,",");
  wLocal.write(f);   fprintf(f,",");
  wAds.write(f);     fprintf(f,"\n");
}

void Desktop::read(TString &t) {
  wMain.read(t);
  wGames.read(t);
  wLocal.read(t);
  wAds.read(t);
}

void Desktop::writeConsoles(FILE *f, const char *key) {
  int i,j;
  j=consoles.size();
  
  for(i=0;i<j;i++) {
    fprintf(f,"%s::",key);
    consoles[i]->write(f);
    fprintf(f,"%s\n",cfilters[i]->c_str());
  }
}

void Desktop::readConsole(TString &t) {
  WindowGeometry *wg;
  string *s;
  static char *sep="\n\r";
  char *p;

  wg=new WindowGeometry();
  wg->read(t);

  s=new string();
  p=t.token(sep);
  if (p) (*s)=p;

  consoles.push_back(wg);
  cfilters.push_back(s);
}

void Desktop::addConsole(DetachedConsole *dc) {
  WindowGeometry *wg;
  wg=new WindowGeometry();
  wg->retrieve(dc->widget);
  consoles.push_back(wg);
  cfilters.push_back(new string(dc->getFilter()));
}

void Desktop::spawnConsoles(TextSet *ts) {
  int i,j;
  char tmp[512];
  DetachedConsole *dc;
  j=consoles.size();

  for(i=0;i<j;i++) {
    dc=new DetachedConsole(ts,0);
    dc->show();
    dc->restorePosition(consoles[i]);
    if (cfilters[i]->size()) {
      strcpy(tmp,cfilters[i]->c_str());
      dc->setFilter(tmp);
    }
  }
}

// ------- ah, the zombies

ZombieHunter::ZombieHunter() {
  signal(SIGCHLD,zh_sigchild_handler);
}

ZombieHunter::~ZombieHunter() {
  pids.clear();
  handlers.clear();
}

void ZombieHunter::add(int pid, SigChildHandler *sigh) {
  pids.push_back(pid);
  handlers.push_back(sigh);
}

void ZombieHunter::handleSigChild() {
  pid_t epid;
  int i,s;
  
  while ( ( epid = waitpid(-1,&s,WNOHANG) ) > 0 ) {
    for(i=0;i<pids.size();i++)
      if (pids[i] == epid) {
	if (handlers[i] != 0) handlers[i]->ZombieNotification(epid);
	pids.erase(pids.begin() + i);
	handlers.erase(handlers.begin() + i);
	break;
      }
  }
}

void zh_sigchild_handler(int sig) {
  if (sig == SIGCHLD)
    global.zombies.handleSigChild();
}
