/* misc.c - 2000/06/28 */
/*
 *  EasyTAG - Tag editor for MP3 and Ogg Vorbis files
 *  Copyright (C) 2000-2003  Jerome Couderc <j.couderc@ifrance.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>

#include "misc.h"
#include "easytag.h"
#include "msgbox.h"
#include "id3_tag.h"
#include "browser.h"
#include "setting.h"
#include "bar.h"
#include "prefs.h"
#include "scan.h"
#include "genres.h"
#include "i18n.h"


/***************
 * Declaration *
 ***************/
// Playlist window
GtkWidget *WritePlaylistWindow = NULL;
GtkWidget *playlist_use_mask_name;
GtkWidget *PlayListNameEntry;
GtkWidget *playlist_use_dir_name;
GtkWidget *playlist_full_path;
GtkWidget *playlist_relative_path;
GtkWidget *playlist_content_none;
GtkWidget *playlist_content_filename;
GtkWidget *playlist_content_mask;
GtkWidget *PlayListContentMaskEntry;

// Search file window
GtkWidget *SearchFileWindow = NULL;
GtkWidget *SearchStringEntry;
GtkWidget *SearchInFilename;
GtkWidget *SearchInTag;
GtkWidget *SearchCaseSensitive;
GtkWidget *SearchResultList;
GtkWidget *SearchStatusBar;
guint      SearchStatusBarContext;

// Load filename window
GtkWidget *LoadFilenameWindow  = NULL;
GtkWidget *FileToLoad;
GtkWidget *LoadFileContentList;
GtkWidget *LoadFileNameList;


GdkColor LIGHT_GREY = {0, 0xd2d2, 0xd2d2, 0xd2d2};


/**************
 * Prototypes *
 **************/
void     Open_Write_Playlist_Window      (void);
void     Write_Playlist_Window_Key_Press (GtkWidget *window, GdkEvent *event);
void     Destroy_Write_Playlist_Window   (void);
void     Playlist_Write_Button_Pressed   (void);
gboolean Write_Playlist                  (gchar *play_list_name);
gboolean Playlist_Check_Content_Mask     (GtkObject *widget_to_show_hide, GtkEntry *widget_source);
void     Playlist_Convert_Forwardslash_Into_Backslash (gchar *string);

void Open_Search_File_Window           (void);
void Destroy_Search_File_Window        (void);
void Search_File_Window_Key_Press      (GtkWidget *window, GdkEvent *event);
void Search_File                       (GtkWidget *search_button);
void Add_Row_To_Search_Result_List     (ET_File *ETFile,gchar *string_to_search);
void Search_Result_List_Row_Selected   (GtkCList *clist, gint row, gint column, GdkEventButton *event, gpointer data);
void Search_Result_List_Row_Unselected (GtkCList *clist, gint row, gint column, GdkEventButton *event, gpointer data);


void Open_Load_Filename_Window      (void);
void Destroy_Load_Filename_Window   (void);
void Load_Filename_Window_Key_Press (GtkWidget *window, GdkEvent *event);
void Load_Filename_Clist_Key_Press  (GtkWidget *clist, GdkEvent *event);
void Load_File_Content (GtkWidget *file_entry);
void Load_File_List    (void);
void Load_Filename_Select_Row_In_Other_Clist (GtkWidget *clist_target, gint row, gint column, GdkEventButton *event, gpointer clist_emit);
void Load_Filename_Set_Filenames (void);
void Button_Load_Set_Sensivity (GtkWidget *button, GtkWidget *entry);
GtkWidget *Create_Load_Filename_Popup_Menu      (GtkCList *clist);
void Load_Filename_Clist_Insert_Blank_Line      (GtkCList *clist);
void Load_Filename_Clist_Delete_Line            (GtkCList *clist);
void Load_Filename_Clist_Delete_All_Blank_Lines (GtkCList *clist);
void Load_Filename_Clist_Reload                 (GtkCList *clist);
void Load_Filename_Update_Text_Line             (GtkWidget *clist, GtkWidget *entry);
void Load_Filename_Edit_Text_Line               (GtkWidget *entry, gint row, gint column, GdkEventButton *event, GtkWidget *clist);



/*************
 * Functions *
 *************/

/******************************
 * Functions managing pixmaps *
 ******************************/
/*
 * Buttons creation with pixmap
 */
#include "../pixmaps/apply.xpm"
#include "../pixmaps/browse.xpm"
#include "../pixmaps/cancel.xpm"
#include "../pixmaps/close.xpm"
#include "../pixmaps/execute.xpm"
#include "../pixmaps/ok.xpm"
#include "../pixmaps/no.xpm"
#include "../pixmaps/save.xpm"
#include "../pixmaps/search.xpm"
#include "../pixmaps/yes.xpm"
GtkWidget *Create_Button_With_Pixmap (guint button_type)
{
    GtkWidget *Button;
    GtkWidget *HBox;
    GtkWidget *Label;
    GtkWidget *PixmapIcon;
    GdkPixmap *pixmap;
    GdkBitmap *mask;

    gtk_widget_realize(MainWindow);
    switch (button_type)
    {
        case BUTTON_OK:
            Label = gtk_label_new(_(" OK "));
            pixmap = gdk_pixmap_create_from_xpm_d(MainWindow->window,&mask,NULL,ok_xpm);
            break;

        case BUTTON_YES:
            Label = gtk_label_new(_(" Yes "));
            pixmap = gdk_pixmap_create_from_xpm_d(MainWindow->window,&mask,NULL,yes_xpm);
            break;

        case BUTTON_NO:
            Label = gtk_label_new(_(" No "));
            pixmap = gdk_pixmap_create_from_xpm_d(MainWindow->window,&mask,NULL,no_xpm);
            break;

        case BUTTON_APPLY:
            Label = gtk_label_new(_(" Apply "));
            pixmap = gdk_pixmap_create_from_xpm_d(MainWindow->window,&mask,NULL,apply_xpm);
            break;

        case BUTTON_SAVE:
            Label = gtk_label_new(_(" Save "));
            pixmap = gdk_pixmap_create_from_xpm_d(MainWindow->window,&mask,NULL,save_xpm);
            break;

        case BUTTON_CANCEL:
            Label = gtk_label_new(_(" Cancel "));
            pixmap = gdk_pixmap_create_from_xpm_d(MainWindow->window,&mask,NULL,cancel_xpm);
            break;

        case BUTTON_CLOSE:
            Label = gtk_label_new(_(" Close "));
            pixmap = gdk_pixmap_create_from_xpm_d(MainWindow->window,&mask,NULL,close_xpm);
            break;

        case BUTTON_WRITE:
            Label = gtk_label_new(_(" Write "));
            pixmap = gdk_pixmap_create_from_xpm_d(MainWindow->window,&mask,NULL,save_xpm);
            break;

        case BUTTON_EXECUTE:
            Label = gtk_label_new(_(" Execute "));
            pixmap = gdk_pixmap_create_from_xpm_d(MainWindow->window,&mask,NULL,execute_xpm);
            break;

        case BUTTON_SEARCH:
            Label = gtk_label_new(_(" Search "));
            pixmap = gdk_pixmap_create_from_xpm_d(MainWindow->window,&mask,NULL,search_xpm);
            break;

        case BUTTON_BROWSE:
            Label = gtk_label_new(_(" Browse... "));
            pixmap = gdk_pixmap_create_from_xpm_d(MainWindow->window,&mask,NULL,browse_xpm);
            break;

        default:
            Button = gtk_button_new_with_label("Unknown button");
            return Button;
            break;
    }

    /* Create Icon */
    PixmapIcon = gtk_pixmap_new(pixmap,mask);
    gdk_pixmap_unref(pixmap);
    gdk_pixmap_unref(mask);

    Button = gtk_button_new();
    HBox = gtk_hbox_new(FALSE,0);
    gtk_container_add(GTK_CONTAINER(Button),HBox);
    /* Add items in button */
    gtk_container_add(GTK_CONTAINER(HBox),PixmapIcon);
    gtk_container_add(GTK_CONTAINER(HBox),Label);
    /* Alignment of items */
    gtk_misc_set_alignment(GTK_MISC(PixmapIcon),1,0.5);
    gtk_misc_set_alignment(GTK_MISC(Label),0,0.5);

    return Button;
}



/*
 * Return a widget with a pixmap
 * Note: for pixmap 'pixmap.xpm', pixmap_name is 'pixmap_xpm'
 */
GtkWidget *Create_Pixmap_Icon (gchar **pixmap_name)
{
    GtkWidget *PixmapIcon;
    GdkPixmap *pixmap;
    GdkBitmap *mask;

    if (!pixmap_name) return NULL;

    gtk_widget_realize(MainWindow);
    pixmap = gdk_pixmap_create_from_xpm_d(MainWindow->window,&mask,NULL,pixmap_name);
    PixmapIcon = gtk_pixmap_new(pixmap,mask);
    gdk_pixmap_unref(pixmap);
    gdk_pixmap_unref(mask);

    return PixmapIcon;
}


/*
 * Create an icon into an event box to allow some events (as to display tooltips).
 */
GtkWidget *Create_Pixmap_Icon_With_Event_Box (gchar **pixmap_name)
{
    GtkWidget *icon;
    GtkWidget *EventBox;

    EventBox = gtk_event_box_new();
    if (pixmap_name)
    {
        icon = Create_Pixmap_Icon(pixmap_name);
        gtk_container_add(GTK_CONTAINER(EventBox),icon);
    }

    return EventBox;
}



/*
 * Return a button with an icon and a label
 */
GtkWidget *Create_Button_With_Icon_And_Label (gchar **pixmap_name, gchar *label)
{
    GtkWidget *Button;
    GtkWidget *HBox;
    GtkWidget *Label;
    GtkWidget *PixmapIcon;
    GdkPixmap *pixmap;
    GdkBitmap *mask;


    Button = gtk_button_new();
    HBox = gtk_hbox_new(FALSE,0);
    gtk_container_add(GTK_CONTAINER(Button),HBox);

    /* Add a pixmap if not null */
    if (pixmap_name != NULL)
    {
        gtk_widget_realize(MainWindow);
        pixmap = gdk_pixmap_create_from_xpm_d(MainWindow->window,&mask,NULL,pixmap_name);
        PixmapIcon = gtk_pixmap_new(pixmap,mask);
        gdk_pixmap_unref(pixmap);
        gdk_pixmap_unref(mask);
        gtk_container_add(GTK_CONTAINER(HBox),PixmapIcon);
    }

    /* Add a label if not null */
    if (label != NULL)
    {
        Label = gtk_label_new(label);
        gtk_container_add(GTK_CONTAINER(HBox),Label);
    }    

    /* Display a warning message if the both parameters are NULL */
    if (pixmap_name==NULL && label==NULL)
        g_warning("Empty button created 'adr=%p' (no icon and no label)!",Button);

    return Button;
}




/*
 * Attach an history list to a combobox
 */
void Add_To_Combo_Box_History(GtkObject *combobox)
{
    GList *list, *tmplist;
    
    if (!GTK_IS_COMBO(combobox) || strlen(gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(combobox)->entry)))==0)
        return;

    tmplist = list = gtk_object_get_data(GTK_OBJECT(combobox),"History");
    list = Add_String_To_Glist(list,gtk_editable_get_chars(GTK_EDITABLE(GTK_COMBO(combobox)->entry),0,-1)); // PS: gtk_editable... allocate memory
    if (list)
        gtk_combo_set_popdown_strings(GTK_COMBO(combobox),list);
    else
        gtk_list_clear_items(GTK_LIST(GTK_COMBO(combobox)->list),0,-1);

    /* Attach data only if there was no data attached */
    if (!tmplist)
        gtk_object_set_data(GTK_OBJECT(combobox),"History",list);
}


/*
 * Add the 'string' passed in parameter to the 'list'.
 * If this string already exists into list, it doesn't add it.
 */
GList *Add_String_To_Glist (GList *list, gchar *string)
{
    GList *tmp_list;
    gint HISTORY_MAX_LENGTH = 15;

    if (!string || strlen(string)<=0 )
        return list;

    /* We search and remove duplicated items (matching with string) */    
    tmp_list = list = g_list_first(list);
    while (tmp_list)
    {
        if ( strcmp(tmp_list->data,string)==0 )
        {
            list = g_list_remove(list,tmp_list->data);
            tmp_list = g_list_first(list);
            continue;
        }
        if (tmp_list->next == NULL) break;
        tmp_list = g_list_next(tmp_list);
    }

    /* We add the string to the beginning of the list */
    list = g_list_prepend(list,string);
    
    /* Cut end of list if length exceeds HISTORY_MAX_LENGTH items */
    if (g_list_length(list)>(guint)HISTORY_MAX_LENGTH)
    {
        tmp_list = g_list_nth(list,HISTORY_MAX_LENGTH);
        tmp_list->prev->next = NULL;
        tmp_list->prev = NULL;
        g_list_free(tmp_list);
    }

    return list;
}





/*
 * Sort the list of files by ascending filenames.
 */
void Sort_File_list_By_Ascending_Filename (void)
{
    if (!ETFileList) return;
    
    ET_Save_File_Data_From_UI(ETFileDisplayed);

    /* Go to the first item of the list */
    ET_File_List_First();
    ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Ascending_Filename);
    
    /* Reload files in browser list */
    Browser_List_Load_Files();

    ET_Display_File_Data_To_UI(ETFileDisplayed);

    Update_Command_Buttons_Sensivity();
}

/*
 * Sort the list of files by descending filenames.
 */
void Sort_File_list_By_Descending_Filename (void)
{
    if (!ETFileList) return;
    
    ET_Save_File_Data_From_UI(ETFileDisplayed);

    /* Go to the first item of the list */
    ET_File_List_First();
    ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Descending_Filename);
    
    /* Reload files in browser list */
    Browser_List_Load_Files();

    ET_Display_File_Data_To_UI(ETFileDisplayed);

    Update_Command_Buttons_Sensivity();
}





/*
 * Event attached to an entry to disable an other widget (for example: a button)
 * when the entry is empty
 */
void Entry_Changed_Disable_Object(GtkObject *widget_to_disable, GtkEditable *source_widget)
{
    gchar *text = NULL;
    
    if (!widget_to_disable || !source_widget) return;
    
    text = gtk_editable_get_chars(GTK_EDITABLE(source_widget),0,-1);
    if (!text || strlen(text)<1)
        gtk_widget_set_sensitive(GTK_WIDGET(widget_to_disable),FALSE);
    else
        gtk_widget_set_sensitive(GTK_WIDGET(widget_to_disable),TRUE);
        
    if (text) g_free(text);
}    




/*
 * Personnal function to create tooltips.
 */
GtkTooltips *gtk_tooltips_new_1 (void)
{
    GtkTooltips *tooltips;
    
    tooltips = gtk_tooltips_new();
    Set_Tips_Color(tooltips);
    
    return tooltips;
}

/*
 * Set the color of tooltips to yellow if the option have been activated
 */
void Set_Tips_Color (GtkTooltips *tips)
{
    GtkStyle *style;
    GdkColor YELLOW = {0, 0xffff, 0xffff, 0xcccc};

    if (USE_COLORED_TOOLTIPS)
    {
        /* Tips coloration */
        gtk_tooltips_force_window(tips);
        style = gtk_style_copy(gtk_widget_get_style(tips->tip_window));
        style->bg[GTK_STATE_NORMAL] = YELLOW;
        gtk_widget_set_style(tips->tip_window,style);
    }
}




/*
 * To insert only digits in an entry. If the text contains only digits: returns it,
 * else only first digits.
 */
#include <ctype.h>
void Insert_Only_Digit (GtkEditable *editable, const gchar *inserted_text, gint length,
                        gint *position, gpointer data)
{
    gint i, i_start = 0;
    gchar *wid_text = NULL;
    gchar *result = g_new (gchar, length);

    if (length<=0 || !inserted_text)
        return;

    // To insert negatives values ('-' only for the first character)
    wid_text = gtk_editable_get_chars(GTK_EDITABLE(editable),0,-1);
    if ( inserted_text[0]=='-' && wid_text && strlen(wid_text)<1 )
    {
        result[0] = inserted_text[0];
        i_start = 1;
    }
    if (wid_text) g_free(wid_text);
    
    for (i=i_start; i<length; i++)
        result[i] = isdigit(inserted_text[i])?inserted_text[i]:(gchar)NULL;
    gtk_signal_handler_block_by_func(GTK_OBJECT(editable),GTK_SIGNAL_FUNC(Insert_Only_Digit),data);
    gtk_editable_insert_text (editable, result, length, position);
    gtk_signal_handler_unblock_by_func(GTK_OBJECT(editable),GTK_SIGNAL_FUNC(Insert_Only_Digit),data);
    gtk_signal_emit_stop_by_name(GTK_OBJECT(editable),"insert_text");
    g_free(result);
}




/*
 * Help the user to select the genre: search and propose the genre which 
 * may correspond to the entered first characters.
 *
 * Function needs improvments...
 */
void Parse_Genre (GtkEntry *entry, GdkEvent *event, GtkCombo *combo)
{
    GdkEventKey *kevent;
    gint cursor_position;
    gchar *typed_string = NULL; // Characters entered by the user
    GtkList *list;
    GList *glist;
    GtkListItem *list_item;
    GtkWidget *label;
    gchar *label_string;
    gchar *string_to_add = NULL;


    // Exits if not a key pressed...
    if (!event || event->type != GDK_KEY_PRESS)
        return;

    kevent = (GdkEventKey *)event;
    switch(kevent->keyval)
    {
        case GDK_Left:
        case GDK_Right:
        case GDK_Delete:
        case GDK_BackSpace:
        case GDK_Shift_L:
        case GDK_Shift_R:
        case GDK_Control_L:
        case GDK_Control_R:
        case GDK_Alt_L:
        case GDK_Alt_R:
            return;
        case GDK_Tab:
            gtk_widget_grab_focus(GTK_WIDGET(GenreMButton)); // Give focus to the next widget
        case GDK_KP_Enter:
        case GDK_Return:
            gtk_editable_select_region(GTK_EDITABLE(entry),strlen(gtk_entry_get_text(entry)),-1);
            gtk_editable_set_position(GTK_EDITABLE(entry),strlen(gtk_entry_get_text(entry)));
            //gtk_widget_event(MainWindow,(GdkEvent *)event);
            return;
    }

    // Current position of the cursor
    cursor_position = gtk_editable_get_position(GTK_EDITABLE(entry));

    // Don't use autocompletion if cursor is not at the end.
    if ( (cursor_position < (gint)strlen(gtk_entry_get_text(entry))) )
        return;

    typed_string = gtk_editable_get_chars(GTK_EDITABLE(entry),0,cursor_position);

    // Get adress of the combo list
    list = GTK_LIST(combo->list);
    glist = list->children;
    // Search if a genre may correspong in the list
    while (glist)
    {
        list_item = (GtkListItem *)glist->data;
        label = GTK_BIN(list_item)->child;
        gtk_label_get(GTK_LABEL(label),&label_string);
        if (strncasecmp(label_string,typed_string,strlen(typed_string))==0)
        {
            string_to_add = ( strlen(label_string)>strlen(typed_string) ) ? (label_string+strlen(typed_string)) : NULL;
            if (string_to_add)
            {
                gint pos = cursor_position;
                
                gtk_entry_set_text(entry,label_string); // To select item on the list
                gtk_editable_delete_text(GTK_EDITABLE(entry),cursor_position,-1);
                gtk_editable_insert_text(GTK_EDITABLE(entry),string_to_add,strlen(string_to_add),&pos);
                gtk_editable_set_position(GTK_EDITABLE(entry),cursor_position);
                gtk_editable_select_region(GTK_EDITABLE(entry),cursor_position,-1);
            }
            break;    
        }
        glist = g_list_next(glist);
    }

    if (typed_string) g_free(typed_string);
}




/*
 * Parse and auto complete date entry if you don't type the 4 digits.
 */
#include <time.h>
#include <stdlib.h>
#include <math.h>
void Parse_Date (void)
{
    gchar *year;
    gchar *tmp, *tmp1;
    gchar current_year[5];
    time_t t;
    struct tm t0;

    if (!DATE_AUTO_COMPLETION) return;

    /* Get the info entered by user */
    year = gtk_entry_get_text(GTK_ENTRY(YearEntry));

    if ( strcmp(year,"")!=0 && strlen(year)<4 )
    {
        t = time(NULL);
        /* Get the current date */
        memcpy(&t0, localtime(&t), sizeof(struct tm));
        /* Put the current year in 'current_year' tab */
        sprintf(current_year,"%d",1900+t0.tm_year);

        tmp = &current_year[4-strlen(year)];
        if ( atoi(year) <= atoi(tmp) )
        {
            sprintf(current_year,"%d",atoi(current_year)-atoi(tmp));
            tmp1 = g_strdup_printf("%d",atoi(current_year)+atoi(year));
            gtk_entry_set_text(GTK_ENTRY(YearEntry),tmp1);
            g_free(tmp1);

        }else
        {
            sprintf(current_year,"%d",atoi(current_year)-atoi(tmp)-(gint)pow(10,strlen(year)));
            tmp1 = g_strdup_printf("%d",atoi(current_year)+atoi(year));
            gtk_entry_set_text(GTK_ENTRY(YearEntry),tmp1);
            g_free(tmp1);
        }
    }
}




/*
 * Load the genres list to the combo, and sorts it
 */
int Compare_Two_Genres (gchar *genre1,gchar *genre2)
{
    return strcmp(genre1,genre2);
}
void Load_Genres_List_To_UI(void)
{
    GList *genrelist = NULL;
    guint i;

    if (!GenreEntry) return;

    /* Create GList */
    if (!genrelist)
    {
        /* Add these elements for: no genre / unknown */
        genrelist = g_list_append(genrelist,"");
        genrelist = g_list_append(genrelist,"Unknown");
        /* Load others genres */
        for (i=0; i<GENRE_MAX; i++)
            genrelist = g_list_insert_sorted(genrelist,id3_genres[i],(GCompareFunc)Compare_Two_Genres);
    }
    /* Connect the combo and the glist */
    gtk_combo_set_popdown_strings(GTK_COMBO(GenreEntry),genrelist);
}




/*
 * Load the track numbers (minimum 30) into the track combo list
 */
void Load_Track_List_To_UI (void)
{
    GList *tracklist = NULL;
    guint len;
    guint i;

    if (!ETFileList || !TrackEntry) return;

    if ((len=ETFileList_Length) < 30)
        len = 30;
    
    // Create list of tracks
    if (!tracklist)
    {
        tracklist = g_list_append(tracklist," ");
        for (i=1; i<=len; i++)
            if (NUMBER_TRACK_FORMATED)
            {
                tracklist = g_list_append(tracklist,g_strdup_printf("%.*d",NUMBER_TRACK_FORMATED_SPIN_BUTTON,i));
            }else
            {
                tracklist = g_list_append(tracklist,g_strdup_printf("%.2d",i));
            }
    }

    // Connect the combo and the list
    gtk_combo_set_popdown_strings(GTK_COMBO(TrackEntry),tracklist);
}





/*
 * Change mouse cursor
 */
void Init_Mouse_Cursor (void)
{
    MouseCursor = NULL;
}
void Destroy_Mouse_Cursor (void)
{
    if (MouseCursor)
    {
        gdk_cursor_destroy(MouseCursor);
        MouseCursor = NULL;
    }
}
void Set_Busy_Cursor (void)
{
    /* If still built, destroy it to avoid memory leak */
    Destroy_Mouse_Cursor();
    /* Create the new cursor */
    MouseCursor = gdk_cursor_new (GDK_WATCH);
    gdk_window_set_cursor(MainWindow->window,MouseCursor);
}
void Set_Unbusy_Cursor (void)
{
    /* Back to standard cursor */
    gdk_window_set_cursor(MainWindow->window,NULL);
    Destroy_Mouse_Cursor();
}



/*
 * Callback to the event "focus-in-event" when an entry receive the focus : we select all the text
 */
void Focus_Select_Text_In_Editable (GtkWidget *widget)
{
    gtk_editable_select_region(GTK_EDITABLE(widget),0,-1);
}



/*
 * Run the audio player and load files of the current dir
 */
void Run_Audio_Player_Using_File_List (AP_Play_Type type)
{
    pid_t pid;

    // Exit if no program selected...
    if (strlen(g_strstrip(AUDIO_FILE_PLAYER))<1)
        return;

    pid = fork();
    switch(pid)
    {
        case -1:
            g_warning(_("Can't fork another process!\n"));
            break;
        case 0:
        {
            gchar **argv;
            gint    argv_index = 0;
            gchar **argv_user;
            gint    argv_user_number;
            GList *etfilelist;
            ET_File *etfile;
            gchar *filename;

            argv_user = g_strsplit(AUDIO_FILE_PLAYER," ",0); // the string may contains arguments, space is the delimiter
            // Number of arguments into 'argv_user'
            for (argv_user_number=0;argv_user[argv_user_number];argv_user_number++);

            // The list of files to play
            if (type == PLAY_DIRECTORY)
                etfilelist = g_list_first(ETFileList);
            else
                etfilelist = ET_File_List_Get_Selection();

            argv = g_new0(gchar *,argv_user_number + g_list_length(etfilelist) + 1);

            // Load 'user' arguments (program name and more...)
            while (argv_user[argv_index])
            {
                argv[argv_index] = argv_user[argv_index];
                argv_index++;
            }
            // Load files as arguments
            while (etfilelist)
            {
                etfile   = (ET_File *)etfilelist->data;
                filename = ((File_Name *)etfile->FileNameCur->data)->value;
                argv[argv_index++] = filename;
                etfilelist = etfilelist->next;
            }
            argv[argv_index] = NULL; // Ends the list of arguments
            if (execvp(argv[0],argv) == -1)
            {
                g_warning(_("Can't execute %s (%s)!\n"),argv[0],g_strerror(errno));
            }
            g_free(argv);
            g_strfreev(argv_user);
            _exit(1);
            break;
        }
        default:
            break;
    }
}
void Run_Audio_Player_Using_Directory (void)
{
    Run_Audio_Player_Using_File_List(PLAY_DIRECTORY);
}
void Run_Audio_Player_Using_Selection (void)
{
    Run_Audio_Player_Using_File_List(PLAY_SELECTION);
}





gchar *Convert_Size (gfloat size)
{
    gchar *data=NULL;
    /* Units Tab of file size (bytes,kilobytes,...) */
    gchar *Units_Tab[] = { N_("B"), N_("KB"), N_("MB"), N_("GB"), N_("TB")};
    gint i = 0;

    while ( (gint)size/1024 && i<(gint)(sizeof(Units_Tab)/sizeof(Units_Tab[0])-1) )
    {
        size = size/1024;
        i++;
    }
    return data = g_strdup_printf("%.1f %s",size,_(Units_Tab[i]));    
}
gchar *Convert_Size_1 (gfloat size)
{
    gchar *data=NULL;
    /* Units Tab of file size (bytes,kilobytes,...) */
    gchar *Units_Tab[] = { N_("B"), N_("KB"), N_("MB"), N_("GB"), N_("TB")};
    guint i = 0;

    while ( (gint)size/1024 && i<(sizeof(Units_Tab)/sizeof(Units_Tab[0])-1) )
    {
        size = size/1024;
        i++;
    }
    if (i >= 2) // For big values : display 3 number afer the separator (coma or point)
        return data = g_strdup_printf("%.3f %s",size,_(Units_Tab[i]));
    else
        return data = g_strdup_printf("%.1f %s",size,_(Units_Tab[i]));
}


gchar *Convert_Duration (gulong duration)
{
    guint hour=0;
    guint minute=0;
    guint second=0;
    gchar *data = NULL;

    if (duration<=0)
        return g_strdup_printf("%d:%.2d",minute,second);

    hour   = duration/3600;
    minute = (duration%3600)/60;
    second = (duration%3600)%60;

    if(hour)
        data = g_strdup_printf("%d:%.2d:%.2d",hour,minute,second);
    else
        data = g_strdup_printf("%d:%.2d",minute,second);
    
    return data;
}


/*
 * Returns the size of a file in bytes
 */
gulong Get_File_Size (gchar *filename)
{
    struct stat statbuf;

    if (filename)
    {
        stat(filename,&statbuf);
        return statbuf.st_size;
    }else
    {
        return 0;
    }
}




/*
 * Delete spaces at the end and the beginning of the string 
 */
void Strip_String (gchar *string)
{
    if (!string) return;
    string = g_strstrip(string);
}



/*
 * Get text of an Entry or a ComboBox
 */
gchar *gtk_entry_get_text_1 (GtkWidget *widget)
{
    if (GTK_IS_COMBO(widget))
    {
        return gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(widget)->entry));
    }else if (GTK_IS_ENTRY(widget))
    {
        return gtk_entry_get_text(GTK_ENTRY(widget));
    }else
    {
        return NULL;
    }
}

/*
 * Set text of an Entry or a ComboBox
 */
void gtk_entry_set_text_1 (GtkWidget *widget, gchar *text)
{
    if (GTK_IS_COMBO(widget))
    {
        gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(widget)->entry),text);
    }else if(GTK_IS_ENTRY(widget))
    {
        gtk_entry_set_text(GTK_ENTRY(widget),text);
    }
}




/*******************************
 * Writting playlist functions *
 *******************************/
/*
 * The window to write playlists.
 */
#include "../pixmaps/mask.xpm"
#include "../pixmaps/forbidden.xpm"
void Open_Write_Playlist_Window (void)
{
    GtkWidget *Frame;
    GtkWidget *VBox;
    GtkWidget *vbox, *hbox;
    GtkWidget *ButtonBox;
    GtkWidget *Button;
    GtkWidget *Separator;
    GtkWidget *Label;
    GtkWidget *Icon;
    GtkWidget *MaskStatusIconBox, *MaskStatusIconBox1;
    GtkTooltips *Tips;
    GList *History_List;
    gint mw_x, mw_y;


    if (WritePlaylistWindow != NULL) 
    {
        gdk_window_raise(WritePlaylistWindow->window);
        return;
    }
    
    Tips = gtk_tooltips_new_1();

    WritePlaylistWindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(WritePlaylistWindow),_("Generate a playlist"));
    gtk_window_set_transient_for(GTK_WINDOW(WritePlaylistWindow),GTK_WINDOW(MainWindow));
    gtk_window_set_policy(GTK_WINDOW(WritePlaylistWindow),FALSE,TRUE,TRUE);
    gtk_signal_connect(GTK_OBJECT(WritePlaylistWindow),"destroy",(GtkSignalFunc)Destroy_Write_Playlist_Window,NULL);
    gtk_signal_connect(GTK_OBJECT(WritePlaylistWindow),"delete_event",(GtkSignalFunc)Destroy_Write_Playlist_Window,NULL);
    gtk_signal_connect(GTK_OBJECT(WritePlaylistWindow),"key_press_event",(GtkSignalFunc)Write_Playlist_Window_Key_Press,NULL);

    Frame = gtk_frame_new(NULL);
    gtk_container_add(GTK_CONTAINER(WritePlaylistWindow),Frame);
    gtk_container_set_border_width(GTK_CONTAINER(Frame),4);

    VBox = gtk_vbox_new(FALSE,2);
    gtk_container_add(GTK_CONTAINER(Frame),VBox);
    gtk_container_border_width(GTK_CONTAINER(VBox),4);

    /* Playlist name */
    Frame = gtk_frame_new(_("M3U Playlist Name"));
    gtk_box_pack_start(GTK_BOX(VBox),Frame,TRUE,TRUE,0);
    vbox = gtk_vbox_new(FALSE,0);
    gtk_container_add(GTK_CONTAINER(Frame),vbox);
    gtk_container_border_width(GTK_CONTAINER(vbox),4);

    playlist_use_mask_name = gtk_radio_button_new_from_widget(NULL);
    gtk_box_pack_start(GTK_BOX(vbox),playlist_use_mask_name,FALSE,FALSE,0);
    // Set a label and a combobox in the 1rst radio button
    hbox = gtk_hbox_new(FALSE,0);
    gtk_container_add(GTK_CONTAINER(playlist_use_mask_name),hbox);
    Label = gtk_label_new(_("Use mask :"));
    gtk_box_pack_start(GTK_BOX(hbox),Label,FALSE,FALSE,0);
    PlayListNameEntry = gtk_combo_new();
    gtk_combo_set_case_sensitive(GTK_COMBO(PlayListNameEntry),TRUE);
    gtk_box_pack_start(GTK_BOX(hbox),PlayListNameEntry,TRUE,TRUE,4);
    playlist_use_dir_name = gtk_radio_button_new_with_label_from_widget(
        GTK_RADIO_BUTTON(playlist_use_mask_name),_("Use directory name"));
    gtk_box_pack_start(GTK_BOX(vbox),playlist_use_dir_name,FALSE,FALSE,0);
    // History list
    History_List = Load_Play_List_Name_List();
    History_List = Add_String_To_Glist(History_List,PLAYLIST_NAME);
    if (History_List)
        gtk_combo_set_popdown_strings(GTK_COMBO(PlayListNameEntry),History_List);
    gtk_object_set_data(GTK_OBJECT(PlayListNameEntry),"History",History_List);
    gtk_signal_connect_object(GTK_OBJECT(GTK_ENTRY(GTK_COMBO(PlayListNameEntry)->entry)),"activate",
        Add_To_Combo_Box_History,GTK_OBJECT(PlayListNameEntry));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(playlist_use_mask_name),PLAYLIST_USE_MASK_NAME);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(playlist_use_dir_name),PLAYLIST_USE_DIR_NAME);

    // Mask status icon
    MaskStatusIconBox = Create_Pixmap_Icon_With_Event_Box(forbidden_xpm);
    gtk_box_pack_start(GTK_BOX(hbox),MaskStatusIconBox,FALSE,FALSE,0);
    gtk_tooltips_set_tip(Tips,MaskStatusIconBox,_("Invalid Scanner Mask"),NULL);
    // Signal connection to check if mask is correct into the mask entry
    gtk_signal_connect_object(GTK_OBJECT(GTK_COMBO(PlayListNameEntry)->entry),"changed",
        GTK_SIGNAL_FUNC(Playlist_Check_Content_Mask),GTK_OBJECT(MaskStatusIconBox));

    // Button for Mask editor
    Button = gtk_button_new();
    Icon = Create_Pixmap_Icon(mask_xpm);
    gtk_container_add(GTK_CONTAINER(Button),Icon);
    gtk_box_pack_start(GTK_BOX(hbox),Button,FALSE,FALSE,0);
    gtk_button_set_relief(GTK_BUTTON(Button),GTK_RELIEF_NONE);
    gtk_tooltips_set_tip(Tips,Button,_("Edit Masks"),NULL);
    // The masks will be edited into a tab of the preferences window. In the future...
    //gtk_signal_connect(GTK_OBJECT(Button),"clicked",(GtkSignalFunc)???,NULL);
// FIX ME : edit the masks
gtk_widget_set_sensitive(GTK_WIDGET(Button),FALSE);


    /* Playlist options */
    Frame = gtk_frame_new(_("Playlist Options"));
    gtk_box_pack_start(GTK_BOX(VBox),Frame,TRUE,TRUE,0);
    vbox = gtk_vbox_new(FALSE,0);
    gtk_container_add(GTK_CONTAINER(Frame),vbox);
    gtk_container_border_width(GTK_CONTAINER(vbox),4);
    playlist_full_path = gtk_radio_button_new_with_label(NULL,_("Use full path for files in playlist"));
    gtk_box_pack_start(GTK_BOX(vbox),playlist_full_path,FALSE,FALSE,0);
    playlist_relative_path = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(playlist_full_path),
        _("Use relative path for files in playlist"));
    gtk_box_pack_start(GTK_BOX(vbox),playlist_relative_path,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(playlist_full_path),PLAYLIST_FULL_PATH);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(playlist_relative_path),PLAYLIST_RELATIVE_PATH);
    // Separator line
    Separator = gtk_hseparator_new();
    gtk_box_pack_start(GTK_BOX(vbox),Separator,FALSE,FALSE,0);

    // Create playlist in parent directory
    playlist_create_in_parent_dir = gtk_check_button_new_with_label(_("Create playlist in the parent directory"));
    gtk_box_pack_start(GTK_BOX(vbox),playlist_create_in_parent_dir,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(playlist_create_in_parent_dir),PLAYLIST_CREATE_IN_PARENT_DIR);
    gtk_tooltips_set_tip(Tips,playlist_create_in_parent_dir,_("If activated, the playlist will be created "
        "in the parent directory."),NULL);

    // DOS Separator
    playlist_use_dos_separator = gtk_check_button_new_with_label(_("Use DOS directory separator"));
    gtk_box_pack_start(GTK_BOX(vbox),playlist_use_dos_separator,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(playlist_use_dos_separator),PLAYLIST_USE_DOS_SEPARATOR);
    gtk_tooltips_set_tip(Tips,playlist_use_dos_separator,_("This option replaces the UNIX directory "
        "separator '/' into DOS separator '\\'."),NULL);

    /* Playlist content */
    Frame = gtk_frame_new(_("Playlist Content"));
    gtk_box_pack_start(GTK_BOX(VBox),Frame,TRUE,TRUE,0);
    vbox = gtk_vbox_new(FALSE,0);
    gtk_container_add(GTK_CONTAINER(Frame),vbox);
    gtk_container_border_width(GTK_CONTAINER(vbox),4);
    playlist_content_none = gtk_radio_button_new_with_label(NULL,_("Write only list of files"));
    gtk_box_pack_start(GTK_BOX(vbox),playlist_content_none,FALSE,FALSE,0);
    playlist_content_filename = gtk_radio_button_new_with_label_from_widget(
        GTK_RADIO_BUTTON(playlist_content_none),_("Write info using filename"));
    gtk_box_pack_start(GTK_BOX(vbox),playlist_content_filename,FALSE,FALSE,0);
    playlist_content_mask = gtk_radio_button_new_from_widget(GTK_RADIO_BUTTON(playlist_content_none));
    gtk_box_pack_start(GTK_BOX(vbox),playlist_content_mask,FALSE,FALSE,0);
    // Set a label, a combobox and un editor button in the 3rd radio button
    hbox = gtk_hbox_new(FALSE,0);
    gtk_container_add(GTK_CONTAINER(playlist_content_mask),hbox);
    Label = gtk_label_new(_("Write info using :"));
    gtk_box_pack_start(GTK_BOX(hbox),Label,FALSE,FALSE,0);
    PlayListContentMaskEntry = gtk_combo_new();
    gtk_box_pack_start(GTK_BOX(hbox),PlayListContentMaskEntry,TRUE,TRUE,4);
    // History list
    History_List = Load_Playlist_Content_Mask_List();
    History_List = Add_String_To_Glist(History_List,PLAYLIST_CONTENT_MASK_VALUE);
    if (History_List)
        gtk_combo_set_popdown_strings(GTK_COMBO(PlayListContentMaskEntry),History_List);
    gtk_object_set_data(GTK_OBJECT(PlayListContentMaskEntry),"History",History_List);
    gtk_signal_connect_object(GTK_OBJECT(GTK_ENTRY(GTK_COMBO(PlayListContentMaskEntry)->entry)),"activate",
        Add_To_Combo_Box_History,GTK_OBJECT(PlayListContentMaskEntry));

    // Mask status icon
    MaskStatusIconBox1 = Create_Pixmap_Icon_With_Event_Box(forbidden_xpm);
    gtk_box_pack_start(GTK_BOX(hbox),MaskStatusIconBox1,FALSE,FALSE,0);
    gtk_tooltips_set_tip(Tips,MaskStatusIconBox1,_("Invalid Scanner Mask"),NULL);
    // Signal connection to check if mask is correct into the mask entry
    gtk_signal_connect_object(GTK_OBJECT(GTK_COMBO(PlayListContentMaskEntry)->entry),"changed",
        GTK_SIGNAL_FUNC(Playlist_Check_Content_Mask),GTK_OBJECT(MaskStatusIconBox1));

    // Button for Mask editor
    Button = gtk_button_new();
    Icon = Create_Pixmap_Icon(mask_xpm);
    gtk_container_add(GTK_CONTAINER(Button),Icon);
    gtk_box_pack_start(GTK_BOX(hbox),Button,FALSE,FALSE,0);
    gtk_button_set_relief(GTK_BUTTON(Button),GTK_RELIEF_NONE);
    gtk_tooltips_set_tip(Tips,Button,_("Edit Masks"),NULL);
    // The masks will be edited into a tab of the preferences window. In the future...
    //gtk_signal_connect(GTK_OBJECT(Button),"clicked",(GtkSignalFunc)???,NULL);
// FIX ME : edit the masks
gtk_widget_set_sensitive(GTK_WIDGET(Button),FALSE);

    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(playlist_content_none),    PLAYLIST_CONTENT_NONE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(playlist_content_filename),PLAYLIST_CONTENT_FILENAME);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(playlist_content_mask),    PLAYLIST_CONTENT_MASK);


    /* Separator line */
    Separator = gtk_hseparator_new();
    gtk_box_pack_start(GTK_BOX(VBox),Separator,FALSE,FALSE,0);

    ButtonBox = gtk_hbutton_box_new ();
    gtk_box_pack_start(GTK_BOX(VBox),ButtonBox,FALSE,FALSE,0);
    gtk_button_box_set_layout(GTK_BUTTON_BOX(ButtonBox),GTK_BUTTONBOX_END);
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(ButtonBox),10);

    /* Button to write the playlist */
    Button = Create_Button_With_Pixmap(BUTTON_WRITE);
    gtk_container_add(GTK_CONTAINER(ButtonBox),Button);
    GTK_WIDGET_SET_FLAGS(Button,GTK_CAN_DEFAULT);
    gtk_signal_connect_object(GTK_OBJECT(Button),"clicked",(GtkSignalFunc)Playlist_Write_Button_Pressed,NULL);
    gtk_signal_connect_object(GTK_OBJECT(Button),"clicked",Add_To_Combo_Box_History,GTK_OBJECT(PlayListNameEntry));
    gtk_signal_connect_object(GTK_OBJECT(Button),"clicked",Add_To_Combo_Box_History,GTK_OBJECT(PlayListContentMaskEntry));

    /* Button to cancel */
    Button = Create_Button_With_Pixmap(BUTTON_CLOSE);
    gtk_container_add(GTK_CONTAINER(ButtonBox),Button);
    GTK_WIDGET_SET_FLAGS(Button, GTK_CAN_DEFAULT);
    gtk_widget_grab_default(Button);
    gtk_signal_connect(GTK_OBJECT(Button),"clicked",(GtkSignalFunc)Destroy_Write_Playlist_Window,NULL);

    gtk_widget_show_all(WritePlaylistWindow);

    gdk_window_get_position(MainWindow->window,&mw_x,&mw_y);
    gtk_window_reposition(GTK_WINDOW(WritePlaylistWindow),
        mw_x + (MainWindow->allocation.width/2)  - (WritePlaylistWindow->allocation.width)/2,
        mw_y + (MainWindow->allocation.height/2) - (WritePlaylistWindow->allocation.height)/2);

    // To avoid/minimize 'fliker', but problem to center correctly the window
    //gtk_widget_show_all(WritePlaylistWindow);

    /* To initialize the mask status icon and visibility */
    gtk_signal_emit_by_name(GTK_OBJECT(GTK_COMBO(PlayListNameEntry)->entry),"changed");
    gtk_signal_emit_by_name(GTK_OBJECT(GTK_COMBO(PlayListContentMaskEntry)->entry),"changed");
}
void Destroy_Write_Playlist_Window (void)
{
    if (WritePlaylistWindow)
    {
        GList *list;
        
        /* Save combobox history lists before exit */
        list = gtk_object_get_data(GTK_OBJECT(PlayListNameEntry),"History");
        Save_Play_List_Name_List(list);
        list = gtk_object_get_data(GTK_OBJECT(PlayListContentMaskEntry),"History");
        Save_Playlist_Content_Mask_List(list);
 
        gtk_widget_destroy(WritePlaylistWindow);
        WritePlaylistWindow = (GtkWidget *)NULL;
    }
}
void Write_Playlist_Window_Key_Press (GtkWidget *window, GdkEvent *event)
{
    GdkEventKey *kevent;

    if (event && event->type == GDK_KEY_PRESS)
    {
        kevent = (GdkEventKey *)event;
        switch(kevent->keyval)
        {
            case GDK_Escape:
                Destroy_Write_Playlist_Window();
                break;
        }
    }
}
void Playlist_Write_Button_Pressed (void)
{
    gchar *playlist_name = NULL;
    gchar *playlist_path, *playlist_path_tmp;
    FILE  *file;
    gchar *msg;
    GtkWidget *msgbox;
    gint msgbox_button=0;


    // Check if playlist name was filled
    if ( gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(playlist_use_mask_name)) 
    &&   strlen(gtk_entry_get_text_1(PlayListNameEntry))<=0 )
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(playlist_use_dir_name),TRUE);

    /* Also set in the function 'Apply_Changes' */
    PLAYLIST_NAME                 = g_strdup(gtk_entry_get_text_1(PlayListNameEntry));
    PLAYLIST_USE_MASK_NAME        = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(playlist_use_mask_name));
    PLAYLIST_USE_DIR_NAME         = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(playlist_use_dir_name));

    PLAYLIST_FULL_PATH            = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(playlist_full_path));
    PLAYLIST_RELATIVE_PATH        = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(playlist_relative_path));
    PLAYLIST_CREATE_IN_PARENT_DIR = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(playlist_create_in_parent_dir));
    PLAYLIST_USE_DOS_SEPARATOR    = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(playlist_use_dos_separator));

    PLAYLIST_CONTENT_NONE         = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(playlist_content_none));
    PLAYLIST_CONTENT_FILENAME     = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(playlist_content_filename));
    PLAYLIST_CONTENT_MASK         = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(playlist_content_mask));
    PLAYLIST_CONTENT_MASK_VALUE   = g_strdup(gtk_entry_get_text_1(PlayListContentMaskEntry));

    // Path of the playlist file
    // Note : End of path must contains '/' and playlist_path musn't be freed
    playlist_path_tmp = g_strdup(Browser_Get_Current_Path());
    if (PLAYLIST_CREATE_IN_PARENT_DIR)
    {
        if ( (strcmp(playlist_path_tmp,"/") != 0) )
        {
            gchar *tmp;
            if (playlist_path_tmp[strlen(playlist_path_tmp)-1]=='/')
                playlist_path_tmp[strlen(playlist_path_tmp)-1] = '\0';
            if ( (tmp=strrchr(playlist_path_tmp,'/')) != NULL )
                *(tmp + 1) = '\0';
        }
    }
    playlist_path = g_strdup(playlist_path_tmp);
    if (playlist_path_tmp) g_free(playlist_path_tmp);
    
    // Build the playlist file name
    if (PLAYLIST_USE_MASK_NAME)
    {
        gchar *filename_generated = NULL;

        if (!ETFileList) return;
        
        // Generate filename from tag of the current selected file (hummm FIX ME)
        filename_generated = Scan_Generate_New_Filename_From_Mask(ETFileDisplayed,PLAYLIST_NAME);

        // Replace Characters (with scanner)
        if (CONVERT_UNDERSCORE_AND_P20_INTO_SPACE)
        {
            Scan_Convert_Underscore_Into_Space(filename_generated);
            Scan_Convert_P20_Into_Space(filename_generated);
        }
        if (CONVERT_SPACE_INTO_UNDERSCORE)
        {
            Scan_Convert_Space_Into_Undescore(filename_generated);
        }

        playlist_name = g_strconcat(playlist_path,filename_generated,".m3u",NULL);

        if (filename_generated) g_free(filename_generated);
    }else // PLAYLIST_USE_DIR_NAME
    {
        gchar *extract_dir_name;
        
        if ( strcmp(playlist_path,"/")==0 )
        {
            extract_dir_name = g_strdup("playlist");
        }else
        {
            gchar *tmp_string = g_strdup(playlist_path);
            if (tmp_string[strlen(tmp_string)-1]=='/')
                tmp_string[strlen(tmp_string)-1] = '\0';
            extract_dir_name = g_strdup(g_basename(tmp_string));
            g_free(tmp_string);
        }
        playlist_name = g_strconcat(playlist_path,extract_dir_name,".m3u",NULL);
        g_free(extract_dir_name);
    }

    // Check if file exists
    if (CONFIRM_WRITE_PLAYLIST)
    {
        if ( (file=fopen(playlist_name,"r")) != NULL )
        {
            fclose(file);
            msg = g_strdup_printf(_("Playlist file '%s' already exists!\nOverwrite?"),playlist_name); 
            msgbox = msg_box_new(_("Write Playlist..."),msg,MSG_QUESTION,BUTTON_YES,BUTTON_NO,0);
            msg_box_hide_check_button(MSG_BOX(msgbox));
            msgbox_button = msg_box_run(MSG_BOX(msgbox));
            gtk_widget_destroy(msgbox);
            g_free(msg);
        }
    }

    // Writing playlist if ok
    if (msgbox_button==0 || msgbox_button==BUTTON_YES )
    {
        if ( Write_Playlist(playlist_name) == FALSE )
        {
            // Writing fails...
            msg = g_strdup_printf(_("Can't write playlist file '%s'!\n(%s)"),playlist_name,g_strerror(errno)); 
            msgbox = msg_box_new(_("Error..."),msg,MSG_ERROR,BUTTON_OK,0);
            msg_box_hide_check_button(MSG_BOX(msgbox));
            msg_box_run(MSG_BOX(msgbox));
            gtk_widget_destroy(msgbox);
        }else
        {
            msg = g_strdup_printf(_("Written playlist file '%s'"),playlist_name); 
            //msgbox = msg_box_new(_("Information..."),msg,MSG_INFO,BUTTON_OK,0);
            Statusbar_Message(msg,TRUE);
        }
        g_free(msg);
    }

    if (playlist_name) g_free(playlist_name);
}
gboolean Playlist_Check_Content_Mask (GtkObject *widget_to_show_hide, GtkEntry *widget_source)
{
    gchar *tmp  = NULL;
    gchar *mask = NULL;


    if (!widget_to_show_hide || !widget_source)
        goto Bad_Mask;

    mask = g_strdup(gtk_entry_get_text(GTK_ENTRY(widget_source)));
    if (!mask || strlen(mask)<1)
        goto Bad_Mask;

    while (mask)
    {
        if ( (tmp=strrchr(mask,'%'))==NULL )
        {
            /* There is no more code. */
            /* No code in mask is accepted. */
            goto Good_Mask;
        }
        if (strlen(tmp)>1 && (tmp[1]=='t' || tmp[1]=='a' || tmp[1]=='b' || tmp[1]=='y' ||
                              tmp[1]=='g' || tmp[1]=='n' || tmp[1]=='l' || tmp[1]=='c' || tmp[1]=='i'))
        {
            /* The code is valid. */
            /* No separator is accepted. */
            *(mask+strlen(mask)-strlen(tmp)) = '\0';
        }else
        {
            goto Bad_Mask;
        }
    }

    Bad_Mask:
        if (mask) g_free(mask);
        gtk_widget_show(GTK_WIDGET(widget_to_show_hide));
        return FALSE;

    Good_Mask:
        if (mask) g_free(mask);
        gtk_widget_hide(GTK_WIDGET(widget_to_show_hide));
        return TRUE;
}

/*
 * Function to replace UNIX ForwardSlash with a DOS BackSlash
 */
void Playlist_Convert_Forwardslash_Into_Backslash (gchar *string)
{
    gchar *tmp;

    while ((tmp=strchr(string,'/'))!=NULL)
        *tmp = '\\';
}


/*
 * Write a playlist
 */
gboolean Write_Playlist (gchar *play_list_name)
{
    FILE  *file;
    ET_File *etfile;
    GList *etfilelist;
    gchar *filename;
    gchar *basedir;
    gint   duration;

    if ((file = fopen(play_list_name,"wb")) != NULL)
    {
        /* 'base directory' where is located the playlist. Used also to write file with a
         * relative path for file located in this directory and sub-directories
         */
        basedir = g_dirname(play_list_name);

        // 1) First line of the file (if playlist content is not set to "write only list of files")
        if (!PLAYLIST_CONTENT_NONE)
        {
            fprintf(file,"#EXTM3U\n");
        }
        
        etfilelist = g_list_first(ETFileList);
        while (etfilelist)
        {
            etfile   = (ET_File *)etfilelist->data;
            filename = ((File_Name *)etfile->FileNameCur->data)->value;
            duration = ((ET_File_Info *)etfile->ETFileInfo)->duration;
            if (PLAYLIST_RELATIVE_PATH)
            {
                // Keep only files in this directory and sub-dirs
                if ( strncmp(filename,basedir,strlen(basedir))==0 )
                {
                    // 2) Write the header
                    if (PLAYLIST_CONTENT_NONE)
                    {
                        // No header written
                    }else if (PLAYLIST_CONTENT_FILENAME)
                    {
                        // Header uses only filename
                        fprintf(file,"#EXTINF:%d,%s\n",duration,g_basename(filename));
                    }else if (PLAYLIST_CONTENT_MASK)
                    {
                        // Header uses generated filename from a mask
                        gchar *mask = g_strdup(gtk_entry_get_text_1(PlayListContentMaskEntry));
                        gchar *filename_generated = Scan_Generate_New_Filename_From_Mask(etfile,mask);
                        fprintf(file,"#EXTINF:%d,%s\n",duration,filename_generated);
                        if (mask)               g_free(mask);
                        if (filename_generated) g_free(filename_generated);
                    }
                    
                    // 3) Write the file path
                    if (PLAYLIST_USE_DOS_SEPARATOR)
                    {
                        gchar *filename_conv = g_strdup(filename+strlen(basedir)+1);
                        Playlist_Convert_Forwardslash_Into_Backslash(filename_conv);
                        fprintf(file,"%s\n",filename_conv);
                        if (filename_conv) g_free(filename_conv);
                    }else
                    {
                        fprintf(file,"%s\n",filename+strlen(basedir)+1);
                    }
                }
            }else // PLAYLIST_FULL_PATH
            {
                // 2) Write the header
                if (PLAYLIST_CONTENT_NONE)
                {
                    // No header written
                }else if (PLAYLIST_CONTENT_FILENAME)
                {
                    // Header uses only filename
                    fprintf(file,"#EXTINF:%d,%s\n",duration,g_basename(filename));
                }else if (PLAYLIST_CONTENT_MASK)
                {
                    // Header uses generated filename from a mask
                    gchar *mask = g_strdup(gtk_entry_get_text_1(PlayListContentMaskEntry));
                    gchar *filename_generated = Scan_Generate_New_Filename_From_Mask(etfile,mask);
                    fprintf(file,"#EXTINF:%d,%s\n",duration,filename_generated);
                    if (mask)               g_free(mask);
                    if (filename_generated) g_free(filename_generated);
                }
                
                // 3) Write the file path
                if (PLAYLIST_USE_DOS_SEPARATOR)
                {
                    gchar *filename_conv = g_strdup(filename);
                    Playlist_Convert_Forwardslash_Into_Backslash(filename_conv);
                    fprintf(file,"%s\n",filename_conv);
                    if (filename_conv) g_free(filename_conv);
                }else
                {
                    fprintf(file,"%s\n",filename);
                }
            }
            etfilelist = etfilelist->next;
        }
        g_free(basedir);
        fclose(file);

        return TRUE;
    }else
    {
        g_print(_("ERROR while opening file: '%s' (%s).\n\a"),play_list_name,g_strerror(errno));
        return FALSE;
    }
}




/*****************************
 * Searching files functions *
 *****************************/
/*
 * The window to search keywords in the list of files.
 */
void Open_Search_File_Window (void)
{
    GtkWidget *VBox;
    GtkWidget *Frame;
    GtkWidget *Table;
    GtkWidget *Label;
    GtkWidget *Button;
    GtkWidget *Separator;
    GList *History_List;
    gint mw_x, mw_y;
    GtkWidget *ScrollWindow;
    gint i;
    gchar *SearchResultList_Titles[] = { N_("File Name"), N_("Title"), N_("Artist"), N_("Album"),
                                         N_("Year"), N_("Track"), N_("Genre"), N_("Comment")};


    if (SearchFileWindow != NULL) 
    {
        gdk_window_raise(SearchFileWindow->window);
        return;
    }
    
    SearchFileWindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(SearchFileWindow),_("Search a file"));
    //gtk_window_set_transient_for(GTK_WINDOW(SearchFileWindow),GTK_WINDOW(MainWindow));
    gtk_window_set_policy(GTK_WINDOW(SearchFileWindow),FALSE,TRUE,TRUE);
    gtk_signal_connect(GTK_OBJECT(SearchFileWindow),"destroy",(GtkSignalFunc)Destroy_Search_File_Window,NULL);
    gtk_signal_connect(GTK_OBJECT(SearchFileWindow),"delete_event",(GtkSignalFunc)Destroy_Search_File_Window,NULL);
    gtk_signal_connect(GTK_OBJECT(SearchFileWindow),"key_press_event",(GtkSignalFunc)Search_File_Window_Key_Press,NULL);

    VBox = gtk_vbox_new(FALSE,0);
    gtk_container_add(GTK_CONTAINER(SearchFileWindow),VBox);
    gtk_container_border_width(GTK_CONTAINER(VBox),1);

    Frame = gtk_frame_new(NULL);
    //gtk_container_add(GTK_CONTAINER(SearchFileWindow),Frame);
    gtk_box_pack_start(GTK_BOX(VBox),Frame,TRUE,TRUE,0);
    gtk_container_set_border_width(GTK_CONTAINER(Frame),2);

    Table = gtk_table_new(3,6,FALSE);
    gtk_container_add(GTK_CONTAINER(Frame),Table);
    gtk_container_border_width(GTK_CONTAINER(Table),2);
    gtk_table_set_row_spacings(GTK_TABLE(Table),4);
    gtk_table_set_col_spacings(GTK_TABLE(Table),4);

    // Words to search
    Label = gtk_label_new(_("Search :"));
    gtk_misc_set_alignment(GTK_MISC(Label),1.0,0.5);
    gtk_table_attach(GTK_TABLE(Table),Label,0,1,0,1,GTK_FILL,GTK_FILL,0,0);
    SearchStringEntry = gtk_combo_new();
    gtk_combo_disable_activate(GTK_COMBO(SearchStringEntry));
    gtk_combo_set_value_in_list(GTK_COMBO(SearchStringEntry),FALSE,FALSE);
    gtk_combo_set_case_sensitive(GTK_COMBO(SearchStringEntry),TRUE);
    gtk_widget_set_usize(GTK_WIDGET(SearchStringEntry),200,-1);
    gtk_table_attach(GTK_TABLE(Table),SearchStringEntry,1,5,0,1,GTK_EXPAND|GTK_FILL,GTK_FILL,0,0);
    // History List
    History_List = Load_Search_File_List();
    if (History_List)
        gtk_combo_set_popdown_strings(GTK_COMBO(SearchStringEntry),History_List);
    gtk_object_set_data(GTK_OBJECT(SearchStringEntry),"History",History_List);
    gtk_signal_connect_object(GTK_OBJECT(GTK_ENTRY(GTK_COMBO(SearchStringEntry)->entry)),"activate",
        Add_To_Combo_Box_History,GTK_OBJECT(SearchStringEntry));
    gtk_entry_set_text_1(SearchStringEntry,"");

    // Set content of the clipboard if available
    gtk_editable_paste_clipboard(GTK_EDITABLE(GTK_COMBO(SearchStringEntry)->entry));
    gtk_signal_connect(GTK_OBJECT(GTK_COMBO(SearchStringEntry)->entry),"grab-focus",Focus_Select_Text_In_Editable,NULL);

    // Where...
    Label = gtk_label_new(_("In :"));
    gtk_misc_set_alignment(GTK_MISC(Label),1.0,0.5);
    gtk_table_attach(GTK_TABLE(Table),Label,0,1,1,2,GTK_FILL,GTK_FILL,0,0);
    SearchInFilename = gtk_check_button_new_with_label(_("File name"));
    SearchInTag = gtk_check_button_new_with_label(_("Tag"));
    gtk_table_attach(GTK_TABLE(Table),SearchInFilename,1,2,1,2,GTK_FILL,GTK_FILL,0,0);
    gtk_table_attach(GTK_TABLE(Table),SearchInTag,2,3,1,2,GTK_FILL,GTK_FILL,0,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(SearchInFilename),TRUE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(SearchInTag),TRUE);

    Separator = gtk_vseparator_new();
    gtk_table_attach(GTK_TABLE(Table),Separator,3,4,1,2,GTK_FILL,GTK_FILL,0,0);

    // Property of the search
    SearchCaseSensitive = gtk_check_button_new_with_label(_("Case sensitive"));
    gtk_table_attach(GTK_TABLE(Table),SearchCaseSensitive,4,5,1,2,GTK_EXPAND|GTK_FILL,GTK_FILL,0,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(SearchCaseSensitive),FALSE);

    // Results list
    ScrollWindow = gtk_scrolled_window_new(NULL,NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(ScrollWindow),GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
    gtk_widget_set_usize(GTK_WIDGET(ScrollWindow),-1,130);
    gtk_table_attach(GTK_TABLE(Table),ScrollWindow,0,6,2,3,GTK_EXPAND|GTK_FILL,GTK_EXPAND|GTK_FILL,0,0);
    SearchResultList = gtk_clist_new_with_titles(8,SearchResultList_Titles);
    gtk_container_add(GTK_CONTAINER(ScrollWindow),SearchResultList);
    //gtk_clist_set_column_auto_resize(GTK_CLIST(BrowserList),0,TRUE);
    gtk_clist_set_reorderable(GTK_CLIST(SearchResultList),FALSE);
    //gtk_clist_set_selection_mode(GTK_CLIST(SearchResultList),GTK_SELECTION_SINGLE);
    gtk_clist_set_selection_mode(GTK_CLIST(SearchResultList),GTK_SELECTION_EXTENDED);
    gtk_clist_column_titles_passive(GTK_CLIST(SearchResultList));
    gtk_widget_set_sensitive(GTK_WIDGET(SearchResultList),FALSE);
    gtk_signal_connect(GTK_OBJECT(SearchResultList),"select_row",(GtkSignalFunc)Search_Result_List_Row_Selected,NULL);
    gtk_signal_connect(GTK_OBJECT(SearchResultList),"unselect_row",(GtkSignalFunc)Search_Result_List_Row_Unselected,NULL);

    for (i=0;i<8;i++)
    {
        gtk_clist_set_column_auto_resize(GTK_CLIST(SearchResultList),i,TRUE);
        //gtk_clist_optimal_column_width(GTK_CLIST(SearchResultList),i);
        //gtk_clist_set_column_resizeable(GTK_CLIST(SearchResultList),i,TRUE);
    }

    // Button to run the search
    Button = Create_Button_With_Pixmap(BUTTON_SEARCH);
    gtk_table_attach(GTK_TABLE(Table),Button,5,6,0,1,GTK_FILL,GTK_FILL,0,0);
    GTK_WIDGET_SET_FLAGS(Button,GTK_CAN_DEFAULT);
    gtk_widget_grab_default(Button);
    gtk_signal_connect(GTK_OBJECT(Button),"clicked",(GtkSignalFunc)Search_File,NULL);
    gtk_signal_connect(GTK_OBJECT(GTK_COMBO(SearchStringEntry)->entry),"activate",(GtkSignalFunc)Search_File,NULL);
    gtk_signal_connect_object(GTK_OBJECT(Button),"clicked",Add_To_Combo_Box_History,GTK_OBJECT(SearchStringEntry));

    // Button to cancel
    Button = Create_Button_With_Pixmap(BUTTON_CLOSE);
    gtk_table_attach(GTK_TABLE(Table),Button,5,6,1,2,GTK_FILL,GTK_FILL,0,0);
    gtk_signal_connect(GTK_OBJECT(Button),"clicked",(GtkSignalFunc)Destroy_Search_File_Window,NULL);

    // Status bar
    SearchStatusBar = gtk_statusbar_new();
    //gtk_table_attach(GTK_TABLE(Table),SearchStatusBar,0,6,3,4,GTK_FILL,GTK_FILL,0,0);
    gtk_box_pack_start(GTK_BOX(VBox),SearchStatusBar,FALSE,TRUE,0);
    SearchStatusBarContext = gtk_statusbar_get_context_id(GTK_STATUSBAR(SearchStatusBar),"Messages");
    gtk_statusbar_push(GTK_STATUSBAR(SearchStatusBar),SearchStatusBarContext,_("Ready to search..."));


    gtk_widget_show_all(SearchFileWindow);

    gdk_window_get_position(MainWindow->window,&mw_x,&mw_y);
    gtk_window_reposition(GTK_WINDOW(SearchFileWindow),
        mw_x + (MainWindow->allocation.width/2)  - (SearchFileWindow->allocation.width)/2,
        mw_y + (MainWindow->allocation.height/2) - (SearchFileWindow->allocation.height)/2);

    // To avoid/minimize 'fliker', but problem to center correctly the window
    //gtk_widget_show_all(SearchFileWindow);
}
void Destroy_Search_File_Window (void)
{
    if (SearchFileWindow)
    {
        GList *list;
        
        /* Save combobox history lists before exit */
        list = gtk_object_get_data(GTK_OBJECT(SearchStringEntry),"History");
        Save_Search_File_List(list);
 
        gtk_widget_destroy(SearchFileWindow);
        SearchFileWindow = (GtkWidget *)NULL;
    }
}
void Search_File_Window_Key_Press (GtkWidget *window, GdkEvent *event)
{
    GdkEventKey *kevent;

    if (event && event->type == GDK_KEY_PRESS)
    {
        kevent = (GdkEventKey *)event;
        switch(kevent->keyval)
        {
            case GDK_Escape:
                Destroy_Search_File_Window();
                break;
        }
    }
}

void Search_File (GtkWidget *search_button)
{
    gchar *string_to_search = NULL;
    GList *etfilelist;
    ET_File *ETFile;
    gchar *msg;


    if (!SearchStringEntry || !SearchInFilename || !SearchInTag || !SearchResultList)
        return;

    string_to_search = gtk_entry_get_text_1(SearchStringEntry);
    if (!string_to_search)
        return;

    gtk_widget_set_sensitive(GTK_WIDGET(search_button),FALSE);
    gtk_clist_clear(GTK_CLIST(SearchResultList));
    gtk_statusbar_push(GTK_STATUSBAR(SearchStatusBar),SearchStatusBarContext,"");
    
    etfilelist = g_list_first(ETFileList);
    while (etfilelist)
    {
        ETFile = (ET_File *)etfilelist->data;
        
        // Search in the filename
        if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(SearchInFilename)))
        {
            gchar *filename2 = g_strdup(((File_Name *)ETFile->FileNameNew->data)->value);
            gchar *string_to_search2 = g_strdup(string_to_search);

            // To search without case sensivity
            if (!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(SearchCaseSensitive)))
            {
                g_strdown(filename2);
                g_strdown(string_to_search2);
            }

            if ( filename2 && strstr(g_basename(filename2),string_to_search2) )
            {
                Add_Row_To_Search_Result_List(ETFile,string_to_search);
                etfilelist = etfilelist->next;
                g_free(filename2);
                g_free(string_to_search2);
                continue;
            }
            g_free(filename2);
            g_free(string_to_search2);
        }

        // Search in the tag
        if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(SearchInTag)))
        {
            File_Tag *FileTag   = (File_Tag *)ETFile->FileTag->data;
            gchar *title2       = g_strdup(FileTag->title);
            gchar *artist2      = g_strdup(FileTag->artist);
            gchar *album2       = g_strdup(FileTag->album);
            gchar *year2        = g_strdup(FileTag->year);
            gchar *track2       = g_strdup(FileTag->track);
            gchar *track_total2 = g_strdup(FileTag->track_total);
            gchar *genre2       = g_strdup(FileTag->genre);
            gchar *comment2     = g_strdup(FileTag->comment);
            gchar *string_to_search2 = g_strdup(string_to_search);

            // To search without case sensivity
            if (!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(SearchCaseSensitive)))
            {
                // Convert to lower case
                if (title2)       g_strdown(title2);
                if (artist2)      g_strdown(artist2);
                if (album2)       g_strdown(album2);
                if (year2)        g_strdown(year2);
                if (track2)       g_strdown(track2);
                if (track_total2) g_strdown(track_total2);
                if (genre2)       g_strdown(genre2);
                if (comment2)     g_strdown(comment2);
                g_strdown(string_to_search2);
            }

            if ( (title2       && strstr(title2,      string_to_search2) )
             ||  (artist2      && strstr(artist2,     string_to_search2) )
             ||  (album2       && strstr(album2,      string_to_search2) )
             ||  (year2        && strstr(year2,       string_to_search2) )
             ||  (track2       && strstr(track2,      string_to_search2) )
             ||  (track_total2 && strstr(track_total2,string_to_search2) )
             ||  (genre2       && strstr(genre2,      string_to_search2) )
             ||  (comment2     && strstr(comment2,    string_to_search2) ) )
            {
                Add_Row_To_Search_Result_List(ETFile,string_to_search);
            }
            if (title2)       g_free(title2);
            if (artist2)      g_free(artist2);
            if (album2)       g_free(album2);
            if (year2)        g_free(year2);
            if (track2)       g_free(track2);
            if (track_total2) g_free(track_total2);
            if (genre2)       g_free(genre2);
            if (comment2)     g_free(comment2);
            g_free(string_to_search2);
        }
        etfilelist = etfilelist->next;
    }

    gtk_widget_set_sensitive(GTK_WIDGET(search_button),TRUE);
    // Display the number of files in the statusbar
    msg = g_strdup_printf(_("Found : %d file(s)"),GTK_CLIST(SearchResultList)->rows);
    gtk_statusbar_push(GTK_STATUSBAR(SearchStatusBar),SearchStatusBarContext,msg);
    g_free(msg);

    // Disable result list if no row inserted
    if ( GTK_CLIST(SearchResultList)->rows > 0 )
        gtk_widget_set_sensitive(GTK_WIDGET(SearchResultList),TRUE);
    else
        gtk_widget_set_sensitive(GTK_WIDGET(SearchResultList),FALSE);
}

void Add_Row_To_Search_Result_List(ET_File *ETFile,gchar *string_to_search)
{
    gchar *SearchResultList_Text[8];
    gchar *track, *track_total;
    gboolean case_sensitive;
    gint new_row;
    gint column;

    if (!ETFile || !string_to_search)
        return;

    case_sensitive = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(SearchCaseSensitive));
    
    // Filename
    SearchResultList_Text[0] = g_basename( ((File_Name *)ETFile->FileNameNew->data)->value );
    // Title
    SearchResultList_Text[1] = ((File_Tag *)ETFile->FileTag->data)->title;
    // Artist
    SearchResultList_Text[2] = ((File_Tag *)ETFile->FileTag->data)->artist;
    // Album
    SearchResultList_Text[3] = ((File_Tag *)ETFile->FileTag->data)->album;
    // Year
    SearchResultList_Text[4] = ((File_Tag *)ETFile->FileTag->data)->year;

    // Track
    track       = ((File_Tag *)ETFile->FileTag->data)->track;
    track_total = ((File_Tag *)ETFile->FileTag->data)->track_total;
    if (track)
    {
        if (track_total)
            SearchResultList_Text[5] = g_strconcat(track,"/",track_total,NULL);
        else
            SearchResultList_Text[5] = g_strdup(track);
    }else
    {
        SearchResultList_Text[5] = NULL;
    }
    //SearchResultList_Text[5] = ((File_Tag *)ETFile->FileTag->data)->track;

    //Genre
    SearchResultList_Text[6] = ((File_Tag *)ETFile->FileTag->data)->genre;
    // Comment
    SearchResultList_Text[7] = ((File_Tag *)ETFile->FileTag->data)->comment;

    // Load the row in the list
    new_row = gtk_clist_append(GTK_CLIST(SearchResultList),SearchResultList_Text);
    gtk_clist_set_row_data(GTK_CLIST(SearchResultList),new_row,ETFile);

    // Highlight the keywords in the result list
    for (column=0;column<8;column++)
    {
        if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(SearchCaseSensitive)))
        {
            if ( SearchResultList_Text[column] && strstr(SearchResultList_Text[column],string_to_search) )
            {
                GdkColor RED = {0, 0xffff, 0x0000, 0x0000};
                GtkStyle *style = gtk_style_copy(gtk_widget_get_style(SearchResultList));
                style->fg[GTK_STATE_NORMAL] = RED;
                style->bg[GTK_STATE_SELECTED] = RED;
                gtk_clist_set_cell_style(GTK_CLIST(SearchResultList),new_row,column,style);
            }
        }else
        {
            // Search wasn't case sensitive
            gchar *list_text = NULL;
            gchar *string_to_search2 = g_strdup(string_to_search);

            if (SearchResultList_Text[column])
                list_text = g_strdup(SearchResultList_Text[column]);

            if (list_text) g_strdown(list_text);
            g_strdown(string_to_search2);

            if ( list_text && strstr(list_text,string_to_search2) )
            {
                GdkColor RED = {0, 0xffff, 0x0000, 0x0000};
                GtkStyle *style = gtk_style_copy(gtk_widget_get_style(SearchResultList));
                style->fg[GTK_STATE_NORMAL] = RED;
                style->bg[GTK_STATE_SELECTED] = RED;
                gtk_clist_set_cell_style(GTK_CLIST(SearchResultList),new_row,column,style);
            }
            if (list_text) g_free(list_text);
            g_free(string_to_search2);
        }
    }

    // Frees allocated data
    if (SearchResultList_Text[5]) g_free(SearchResultList_Text[5]);
}

/*
 * Callback to select-row event
 */
void Search_Result_List_Row_Selected (GtkCList *clist, gint row, gint column, GdkEventButton *event, gpointer data)
{
    GList   *SelectedRow = NULL;
    ET_File *ETFile;
    
    Browser_List_Unselect_All_Files();

    SelectedRow = ((GtkCList*)clist)->selection;
    while (SelectedRow)
    {
        ETFile = gtk_clist_get_row_data(GTK_CLIST(clist),GPOINTER_TO_INT(SelectedRow->data));
        Browser_List_Select_File(ETFile,TRUE);
        Action_Select_Nth_File_By_Etfile(ETFile);
        SelectedRow = SelectedRow->next;
    }
}
/*
 * Callback to unselect-row event
 */
void Search_Result_List_Row_Unselected (GtkCList *clist, gint row, gint column, GdkEventButton *event, gpointer data)
{
    ET_File *ETFile;
    
    ETFile = gtk_clist_get_row_data(GTK_CLIST(clist),row);
    Browser_List_Unselect_File(ETFile);
}




/********************************************
 * Load Filenames from a TXT file functions *
 ********************************************/
/*
 * The window to load the filenames from a txt.
 */
#include "../pixmaps/add.xpm"
void Open_Load_Filename_Window (void)
{
    GtkWidget *VBox, *hbox;
    GtkWidget *Frame;
    GtkWidget *Label;
    GtkWidget *ButtonBox;
    GtkWidget *Button;
    GtkWidget *Entry;
    GtkWidget *ButtonLoad;
    GtkWidget *Separator;
    GtkWidget *ScrollWindow;
    GtkWidget *Table;
    GtkTooltips *Tips;
    GList *History_List;
    gchar *path;
    gint mw_x, mw_y;


    if (LoadFilenameWindow != NULL) 
    {
        gdk_window_raise(LoadFilenameWindow->window);
        return;
    }
    
    LoadFilenameWindow  = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(LoadFilenameWindow),_("Load the filenames from a TXT file"));
    gtk_window_set_transient_for(GTK_WINDOW(LoadFilenameWindow),GTK_WINDOW(MainWindow));
    gtk_window_set_policy(GTK_WINDOW(LoadFilenameWindow),FALSE,TRUE,TRUE);
    gtk_signal_connect(GTK_OBJECT(LoadFilenameWindow),"destroy",(GtkSignalFunc)Destroy_Load_Filename_Window,NULL);
    gtk_signal_connect(GTK_OBJECT(LoadFilenameWindow),"delete_event",(GtkSignalFunc)Destroy_Load_Filename_Window,NULL);
    gtk_signal_connect(GTK_OBJECT(LoadFilenameWindow),"key_press_event",(GtkSignalFunc)Load_Filename_Window_Key_Press,NULL);

    Tips = gtk_tooltips_new_1();

    Frame = gtk_frame_new(NULL);
    gtk_container_add(GTK_CONTAINER(LoadFilenameWindow),Frame);
    gtk_container_set_border_width(GTK_CONTAINER(Frame),2);

    VBox = gtk_vbox_new(FALSE,4);
    gtk_container_add(GTK_CONTAINER(Frame),VBox);
    gtk_container_border_width(GTK_CONTAINER(VBox),2);

    hbox = gtk_hbox_new(FALSE,4);
    gtk_box_pack_start(GTK_BOX(VBox),hbox,FALSE,TRUE,0);

    // File to load
    Label = gtk_label_new(_("File :"));
    gtk_misc_set_alignment(GTK_MISC(Label),1.0,0.5);
    gtk_box_pack_start(GTK_BOX(hbox),Label,FALSE,FALSE,0);
    FileToLoad = gtk_combo_new();
    gtk_combo_disable_activate(GTK_COMBO(FileToLoad));
    gtk_combo_set_value_in_list(GTK_COMBO(FileToLoad),FALSE,FALSE);
    gtk_combo_set_case_sensitive(GTK_COMBO(FileToLoad),TRUE);
    gtk_widget_set_usize(GTK_WIDGET(FileToLoad),200,-1);
    gtk_box_pack_start(GTK_BOX(hbox),FileToLoad,TRUE,TRUE,0);
    // History List
    History_List = Load_File_To_Load_List();
    if (History_List)
        gtk_combo_set_popdown_strings(GTK_COMBO(FileToLoad),History_List);
    gtk_object_set_data(GTK_OBJECT(FileToLoad),"History",History_List);
    gtk_signal_connect_object(GTK_OBJECT(GTK_ENTRY(GTK_COMBO(FileToLoad)->entry)),"activate",
        Add_To_Combo_Box_History,GTK_OBJECT(FileToLoad));
    // Initial value
    if ((path=Browser_Get_Current_Path())!=NULL)
        gtk_entry_set_text_1(FileToLoad,path);
    // the 'changed' signal is attached below to enable/disable the button to load
    // Button 'browse'
    Button = Create_Button_With_Pixmap(BUTTON_BROWSE);
    gtk_box_pack_start(GTK_BOX(hbox),Button,FALSE,FALSE,0);
    gtk_object_set_data(GTK_OBJECT(FileToLoad),"Parent_Window",LoadFilenameWindow);
    gtk_signal_connect_object(GTK_OBJECT(Button),"clicked",(GtkSignalFunc)File_Selection_Window_For_File,GTK_OBJECT(FileToLoad));
    // Button 'load'
    // the signal attached to this button, to load the file, is placed after the LoadFileContentList definition
    ButtonLoad = Create_Button_With_Icon_And_Label(add_xpm,_(" Load "));
    //ButtonLoad = gtk_button_new_with_label(_(" Load "));
    gtk_box_pack_start(GTK_BOX(hbox),ButtonLoad,FALSE,FALSE,0);
    gtk_signal_connect_object(GTK_OBJECT(GTK_COMBO(FileToLoad)->entry),"changed",(GtkSignalFunc)Button_Load_Set_Sensivity,GTK_OBJECT(ButtonLoad));

    // Separator line
    Separator = gtk_hseparator_new();
    gtk_box_pack_start(GTK_BOX(VBox),Separator,FALSE,FALSE,0);


    Table = gtk_table_new(2,2,FALSE);
    gtk_box_pack_start(GTK_BOX(VBox),Table,TRUE,TRUE,0);

    // Label of file content
    Label = gtk_label_new(_("Loaded File Content :"));
    gtk_table_attach(GTK_TABLE(Table),Label,0,1,0,1,GTK_FILL,GTK_FILL,0,2);

    // Content of the loaded file
    ScrollWindow = gtk_scrolled_window_new(NULL,NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(ScrollWindow),GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
    gtk_widget_set_usize(GTK_WIDGET(ScrollWindow),250,200);
    gtk_table_attach_defaults(GTK_TABLE(Table),ScrollWindow,0,1,1,2);
    LoadFileContentList = gtk_clist_new(1);
    gtk_container_add(GTK_CONTAINER(ScrollWindow),LoadFileContentList);
    gtk_clist_set_column_auto_resize(GTK_CLIST(LoadFileContentList),0,TRUE);
    gtk_clist_set_reorderable(GTK_CLIST(LoadFileContentList),TRUE);
    gtk_clist_set_selection_mode(GTK_CLIST(LoadFileContentList),GTK_SELECTION_SINGLE);
    //gtk_clist_set_selection_mode(GTK_CLIST(LoadFileContentList),GTK_SELECTION_EXTENDED); // Needs to fix some problems
    // Signal to automatically load the file
    gtk_signal_connect_object(GTK_OBJECT(ButtonLoad),"clicked",(GtkSignalFunc)Load_File_Content,GTK_OBJECT(FileToLoad));
    gtk_signal_connect(GTK_OBJECT(LoadFileContentList),"key-press-event",(GtkSignalFunc)Load_Filename_Clist_Key_Press,NULL);


    // Label of current list
    Label = gtk_label_new(_("Files Name List :"));
    gtk_table_attach(GTK_TABLE(Table),Label,1,2,0,1,GTK_FILL,GTK_FILL,0,2);

    // List of current filenames
    ScrollWindow = gtk_scrolled_window_new(NULL,NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(ScrollWindow),GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
    gtk_widget_set_usize(GTK_WIDGET(ScrollWindow),250,200);
    gtk_table_attach_defaults(GTK_TABLE(Table),ScrollWindow,1,2,1,2);
    LoadFileNameList = gtk_clist_new(1);
    gtk_container_add(GTK_CONTAINER(ScrollWindow),LoadFileNameList);
    gtk_clist_set_column_auto_resize(GTK_CLIST(LoadFileNameList),0,TRUE);
    gtk_clist_set_reorderable(GTK_CLIST(LoadFileNameList),TRUE);
    gtk_clist_set_selection_mode(GTK_CLIST(LoadFileNameList),GTK_SELECTION_SINGLE);
    gtk_signal_connect(GTK_OBJECT(LoadFileNameList),"key-press-event",(GtkSignalFunc)Load_Filename_Clist_Key_Press,NULL);

    // Signals to 'select' the same row into the other list (to show the corresponding filenames)
    gtk_signal_connect_object(GTK_OBJECT(LoadFileContentList),"select_row",GTK_SIGNAL_FUNC(Load_Filename_Select_Row_In_Other_Clist),GTK_OBJECT(LoadFileNameList));
    gtk_signal_connect_object(GTK_OBJECT(LoadFileNameList),"select_row",GTK_SIGNAL_FUNC(Load_Filename_Select_Row_In_Other_Clist),GTK_OBJECT(LoadFileContentList));


    // Load the list of files in the list widget
    Load_File_List();

    // Create popup menus
    Create_Load_Filename_Popup_Menu(GTK_CLIST(LoadFileContentList));
    Create_Load_Filename_Popup_Menu(GTK_CLIST(LoadFileNameList));

    // Entry to edit a line into the clist
    Entry = gtk_entry_new();
    gtk_box_pack_start(GTK_BOX(VBox),Entry,FALSE,TRUE,0);
    gtk_signal_connect_object_after(GTK_OBJECT(Entry),"changed",(GtkSignalFunc)Load_Filename_Update_Text_Line,GTK_OBJECT(LoadFileContentList));
    // Signal to load the line text in the editing entry
    gtk_signal_connect_object(GTK_OBJECT(LoadFileContentList),"select-row",(GtkSignalFunc)Load_Filename_Edit_Text_Line,GTK_OBJECT(Entry));


    // Separator line
    Separator = gtk_hseparator_new();
    gtk_box_pack_start(GTK_BOX(VBox),Separator,FALSE,FALSE,0);

    LoadFileRunScanner = gtk_check_button_new_with_label(_("Run the current scanner for each file"));
    gtk_box_pack_start(GTK_BOX(VBox),LoadFileRunScanner,FALSE,TRUE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(LoadFileRunScanner),LOAD_FILE_RUN_SCANNER);
    gtk_tooltips_set_tip(Tips,LoadFileRunScanner,_("When activating this option, after loading the "
        "filenames, the current selected scanner will be ran (the scanner window must be opened)."),NULL);

    // Separator line
    Separator = gtk_hseparator_new();
    gtk_box_pack_start(GTK_BOX(VBox),Separator,FALSE,FALSE,0);

    ButtonBox = gtk_hbutton_box_new ();
    gtk_box_pack_start(GTK_BOX(VBox),ButtonBox,FALSE,FALSE,0);
    gtk_button_box_set_layout(GTK_BUTTON_BOX(ButtonBox),GTK_BUTTONBOX_END);
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(ButtonBox),10);

    // Button to load filenames
    Button = Create_Button_With_Pixmap(BUTTON_APPLY);
    gtk_container_add(GTK_CONTAINER(ButtonBox),Button);
    GTK_WIDGET_SET_FLAGS(Button,GTK_CAN_DEFAULT);
    gtk_signal_connect(GTK_OBJECT(Button),"clicked",(GtkSignalFunc)Load_Filename_Set_Filenames,NULL);
    gtk_signal_connect_object(GTK_OBJECT(Button),"clicked",Add_To_Combo_Box_History,GTK_OBJECT(FileToLoad));

    // Button to cancel
    Button = Create_Button_With_Pixmap(BUTTON_CLOSE);
    gtk_container_add(GTK_CONTAINER(ButtonBox),Button);
    GTK_WIDGET_SET_FLAGS(Button, GTK_CAN_DEFAULT);
    gtk_widget_grab_default(Button);
    gtk_signal_connect(GTK_OBJECT(Button),"clicked",(GtkSignalFunc)Destroy_Load_Filename_Window,NULL);


    // To initialize 'ButtonLoad' sensivity
    gtk_signal_emit_by_name(GTK_OBJECT(GTK_COMBO(FileToLoad)->entry),"changed");

    gtk_widget_show_all(LoadFilenameWindow);

    gdk_window_get_position(MainWindow->window,&mw_x,&mw_y);
    gtk_window_reposition(GTK_WINDOW(LoadFilenameWindow),
        mw_x + (MainWindow->allocation.width/2)  - (LoadFilenameWindow->allocation.width)/2,
        mw_y + (MainWindow->allocation.height/2) - (LoadFilenameWindow->allocation.height)/2);

    // To avoid/minimize 'fliker', but problem to center correctly the window
    //gtk_widget_show_all(LoadFilenameWindow);
}
void Destroy_Load_Filename_Window (void)
{
    if (LoadFilenameWindow)
    {
        GList *list;
        
        /* Save combobox history lists before exit */
        list = gtk_object_get_data(GTK_OBJECT(FileToLoad),"History");
        Save_File_To_Load_List(list);
 
        gtk_widget_destroy(LoadFilenameWindow);
        LoadFilenameWindow = (GtkWidget *)NULL;
    }
}
void Load_Filename_Window_Key_Press (GtkWidget *window, GdkEvent *event)
{
    GdkEventKey *kevent;

    if (event && event->type == GDK_KEY_PRESS)
    {
        kevent = (GdkEventKey *)event;
        switch(kevent->keyval)
        {
            case GDK_Escape:
                Destroy_Load_Filename_Window();
                break;
        }
    }
}
/*
 * To enable/disable sensivity of the button 'Load'
 */
void Button_Load_Set_Sensivity (GtkWidget *button, GtkWidget *entry)
{
    struct stat statbuf;
    gchar *path;

    if (!entry || !button)
        return;

    path = gtk_entry_get_text_1(entry);
    if ( path && stat(path,&statbuf)==0 && S_ISREG(statbuf.st_mode))
        gtk_widget_set_sensitive(GTK_WIDGET(button),TRUE);
    else
        gtk_widget_set_sensitive(GTK_WIDGET(button),FALSE);
}

void Load_Filename_Clist_Key_Press (GtkWidget *clist, GdkEvent *event)
{
    if (event && event->type == GDK_KEY_PRESS)
    {
        GdkEventKey *kevent = (GdkEventKey *)event;

        switch(kevent->keyval)
        {
            case GDK_Delete:
                Load_Filename_Clist_Delete_Line((GtkCList *)clist);
                break;
            case GDK_I:
            case GDK_i:
                Load_Filename_Clist_Insert_Blank_Line((GtkCList *)clist);
                break;
        }
    }
}

/*
 * Load content of the file into the LoadFileContentList clist
 */
void Load_File_Content (GtkWidget *file_entry)
{
    FILE *file;
    gchar *file_path = NULL;
    gchar buffer[MAX_STRING_LEN];
    gchar *clist_text[1];
    gint new_row;

    if (!file_entry)
        return;
    
    // The file to read
    file_path = g_strdup(gtk_entry_get_text_1(file_entry));
    
    if ( (file=fopen(file_path,"r"))==0 )
    {
        g_print(_("Can't open file '%s' (%s)\n"),file_path,g_strerror(errno));
    }else
    {
        gtk_clist_clear(GTK_CLIST(LoadFileContentList));
        while(fgets(buffer,sizeof(buffer),file))
        {
            if (buffer[strlen(buffer)-1]=='\n')
                buffer[strlen(buffer)-1]='\0';
            if (buffer[strlen(buffer)-1]=='\r')
                buffer[strlen(buffer)-1]='\0';
            clist_text[0] = &buffer[0];
            new_row = gtk_clist_append(GTK_CLIST(LoadFileContentList),clist_text);
        }
        fclose(file);
    }
    if (file_path) g_free(file_path);
}
/*
 * Load the names of the current list of files
 */
void Load_File_List (void)
{
    GList *etfilelist;
    ET_File *etfile;
    gchar *filename;
    gchar *pos;
    gint new_row;
    
    gtk_clist_clear(GTK_CLIST(LoadFileNameList));
    etfilelist = g_list_first(ETFileList);
    while (etfilelist)
    {
        etfile   = (ET_File *)etfilelist->data;
        filename = g_strdup(g_basename(((File_Name *)etfile->FileNameNew->data)->value));
        // Remove the extension ('filename' must be allocated to don't affect the initial value)
        if ((pos=strrchr(filename,'.'))!=NULL) *pos = 0;
        new_row = gtk_clist_append(GTK_CLIST(LoadFileNameList),&filename);
        gtk_clist_set_row_data(GTK_CLIST(LoadFileNameList),new_row,etfile);
        g_free(filename);
        etfilelist = etfilelist->next;
    }
}
/*
 * To select the corresponding row in the other clist
 */
void Load_Filename_Select_Row_In_Other_Clist (GtkWidget *clist_target, gint row, gint column, GdkEventButton *event, gpointer clist_emit)
{
    gchar *text;
    GtkAdjustment *ct_adj, *ce_adj;

    if (!clist_emit || !clist_target)
        return;

    gtk_clist_unselect_all(GTK_CLIST(clist_target));

    // Synchronize the two clist
    ce_adj = gtk_clist_get_vadjustment(GTK_CLIST(clist_emit));
    ct_adj = gtk_clist_get_vadjustment(GTK_CLIST(clist_target));

    if (GTK_ADJUSTMENT(ct_adj)->upper >= GTK_ADJUSTMENT(ct_adj)->page_size
    &&  GTK_ADJUSTMENT(ce_adj)->upper >= GTK_ADJUSTMENT(ce_adj)->page_size)
    {
        // Rules are displayed in the both clist
        if (GTK_ADJUSTMENT(ce_adj)->value <= GTK_ADJUSTMENT(ct_adj)->upper - GTK_ADJUSTMENT(ct_adj)->page_size)
        {
            gtk_adjustment_set_value(GTK_ADJUSTMENT(ct_adj),GTK_ADJUSTMENT(ce_adj)->value);
        }else
        {
            gtk_adjustment_set_value(GTK_ADJUSTMENT(ct_adj),GTK_ADJUSTMENT(ct_adj)->upper - GTK_ADJUSTMENT(ct_adj)->page_size);
            if (row <= GTK_CLIST(clist_target)->rows-1)
                gtk_adjustment_set_value(GTK_ADJUSTMENT(ce_adj),GTK_ADJUSTMENT(ct_adj)->value);
        }
    }else if (GTK_ADJUSTMENT(ct_adj)->upper < GTK_ADJUSTMENT(ct_adj)->page_size) // Target Clist rule not visible
    {
        if (row <= GTK_CLIST(clist_target)->rows-1)
            gtk_adjustment_set_value(GTK_ADJUSTMENT(ce_adj),GTK_ADJUSTMENT(ct_adj)->value);
    }

    // Must block the select signal of the target to avoid looping
    gtk_signal_handler_block_by_func(GTK_OBJECT(clist_target),GTK_SIGNAL_FUNC(Load_Filename_Select_Row_In_Other_Clist),GTK_OBJECT(clist_emit));
    gtk_clist_get_text(GTK_CLIST(clist_emit),row,column,&text);
    gtk_clist_select_row(GTK_CLIST(clist_target),row,column);
    gtk_signal_handler_unblock_by_func(GTK_OBJECT(clist_target),GTK_SIGNAL_FUNC(Load_Filename_Select_Row_In_Other_Clist),GTK_OBJECT(clist_emit));
}
/*
 * Set the new file name of each file
 */
void Load_Filename_Set_Filenames (void)
{
    gint row;
    ET_File   *ETFile;
    File_Name *FileName;
    gchar *clist_text;
    gchar *clist_text_tmp = NULL;
    gchar *filename_new;
    
    if ( !ETFileList || !LoadFileContentList || !LoadFileNameList)
        return;

    // Save current file
    ET_Save_File_Data_From_UI(ETFileDisplayed);

    for (row=0;(row<GTK_CLIST(LoadFileContentList)->rows)&&(row<GTK_CLIST(LoadFileNameList)->rows);row++)
    {
        ETFile = gtk_clist_get_row_data(GTK_CLIST(LoadFileNameList),row);
        gtk_clist_get_text(GTK_CLIST(LoadFileContentList),row,0,&clist_text);
        if (ETFile && clist_text && strlen(clist_text)>0)
        {
            clist_text_tmp = g_strdup(clist_text);
            ET_File_Name_Convert_Character(clist_text_tmp); // Replace invalid characters

            /* Build the filename with the path */
            filename_new = ET_Generate_File_Name(ETFile,clist_text_tmp);
            g_free(clist_text_tmp);

            /* Set the new filename */
            // Create a new 'File_Name' item
            FileName = ET_File_Name_Item_New();
            // Save changes of the 'File_Name' item
            ET_Set_Field_File_Name_Item(&FileName->value,filename_new);
            ET_Manage_Changes_Of_File_Data(ETFile,FileName,NULL);
            
            if (filename_new) g_free(filename_new);

            // Then run current scanner if asked...
            if (ScannerWindow && gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(LoadFileRunScanner)) )
                Scan_Select_Mode_And_Run_Scanner(ETFile);
        }
    }
    Browser_List_Refresh_Whole_List();
    ET_Display_File_Data_To_UI(ETFileDisplayed);
}
/*
 * Create and attach a popup menu on the two clist of the LoadFileWindow
 */
gboolean Load_Filename_Popup_Menu_Handler (GtkMenu *menu, GdkEventButton *event)
{
    if (event && (event->type==GDK_BUTTON_PRESS) && (event->button==3))
    {
        gtk_menu_popup(menu,NULL,NULL,NULL,NULL,event->button,event->time);
        return TRUE;
    }
    return FALSE;
}
GtkWidget *Create_Load_Filename_Popup_Menu (GtkCList *clist)
{
    GtkWidget *BrowserPopupMenu;
    GtkWidget *MenuItem;


    BrowserPopupMenu = gtk_menu_new();
    gtk_signal_connect_object(GTK_OBJECT(clist),"button_press_event",(GtkSignalFunc)Load_Filename_Popup_Menu_Handler,GTK_OBJECT(BrowserPopupMenu));

    MenuItem = gtk_menu_item_new_with_label(_("Insert a blank line"));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Load_Filename_Clist_Insert_Blank_Line,GTK_OBJECT(clist));

    MenuItem = gtk_menu_item_new_with_label(_("Delete this line"));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Load_Filename_Clist_Delete_Line,GTK_OBJECT(clist));

    MenuItem = gtk_menu_item_new_with_label(_("Delete all blank lines"));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Load_Filename_Clist_Delete_All_Blank_Lines,GTK_OBJECT(clist));

    MenuItem = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);

    MenuItem = gtk_menu_item_new_with_label(_("Reload"));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Load_Filename_Clist_Reload,GTK_OBJECT(clist));

    gtk_widget_show_all(BrowserPopupMenu);
    return BrowserPopupMenu;
}

/*
 * Insert a blank line before the selected line in the clist
 */
void Load_Filename_Clist_Insert_Blank_Line (GtkCList *clist)
{
    GList *SelectedRow;
    gint row = 0;
    gchar *text = {""};

    if (!clist) return;

    if ( (SelectedRow=((GtkCList*)clist)->selection) )
        row = GPOINTER_TO_INT(SelectedRow->data);
    gtk_clist_insert(GTK_CLIST(clist),row,&text);
    gtk_clist_set_background(GTK_CLIST(clist),row,&LIGHT_GREY);
}

/*
 * Delete all blank lines in the clist
 */
void Load_Filename_Clist_Delete_All_Blank_Lines (GtkCList *clist)
{
    gchar *text;
    gint   row, nb_rows;

    if (!clist) return;

    nb_rows = GTK_CLIST(clist)->rows - 1;
    row = nb_rows;
    while (row >= 0)
    {
        gtk_clist_get_text(GTK_CLIST(clist),row,0,&text);
        if (text==NULL || strlen(text)==0)
            gtk_clist_remove(GTK_CLIST(clist),row);
        row--;
    }
}

/*
 * Delete the selected line in the clist
 */
void Load_Filename_Clist_Delete_Line (GtkCList *clist)
{
    GList *SelectedRow;

    if (!clist) return;

    if ( (SelectedRow=((GtkCList*)clist)->selection) )
        gtk_clist_remove(GTK_CLIST(clist),GPOINTER_TO_INT(SelectedRow->data));
}

void Load_Filename_Clist_Reload (GtkCList *clist)
{
    if (!clist) return;

    if (clist == (GtkCList *)LoadFileContentList)
    {
        Load_File_Content(FileToLoad);
    }else if (clist == (GtkCList *)LoadFileNameList)
    {
        Load_File_List();
    }
}

/*
 * Update the text of the selected line into the clist, with the text entered into the entry
 */
void Load_Filename_Update_Text_Line (GtkWidget *clist, GtkWidget *entry)
{
    GList *SelectedRow = NULL;

    if (!clist || !entry) return;
    
    SelectedRow = ((GtkCList*)clist)->selection;
    if (SelectedRow && SelectedRow->data!=NULL)
    {
        gint row = GPOINTER_TO_INT(SelectedRow->data);
        gchar *text = gtk_entry_get_text_1(entry);

        gtk_clist_set_text(GTK_CLIST(clist),row,0,text);
        if (strlen(text)>0)
            gtk_clist_set_background(GTK_CLIST(clist),row,NULL);
        else
            gtk_clist_set_background(GTK_CLIST(clist),row,&LIGHT_GREY);
    }

}

/*
 * Set the text of the selected line of the clist into the entry
 */
void Load_Filename_Edit_Text_Line (GtkWidget *entry, gint row, gint column, GdkEventButton *event, GtkWidget *clist)
{
    gchar *text;

    if (!clist || !entry) return;

//    gtk_signal_handler_block_by_func(GTK_OBJECT(entry),GTK_SIGNAL_FUNC(Update_Text_Line),NULL);
    gtk_clist_get_text(GTK_CLIST(clist),row,column,&text);
    gtk_entry_set_text_1(entry,text);
//    gtk_signal_handler_unblock_by_func(GTK_OBJECT(entry),GTK_SIGNAL_FUNC(Update_Text_Line),NULL);
}
