#include <iostream.h>
#include <sys/time.h>
#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/socket.h>
#include <sys/resource.h>
#include <sys/utsname.h>
#include <unistd.h>
#include <netdb.h>
#include <signal.h>
#include <netinet/in.h>
#ifndef __CYGWIN32__
#include <netinet/tcp.h>
#endif
#include <arpa/inet.h>
#ifndef __CYGWIN32__
#include <sys/un.h>
#endif
#include <errno.h>
#ifndef __CYGWIN32__
#include <X11/Xproto.h>
#else
#include "Xproto-cygwin.h"
#endif
#include <pwd.h>
#ifdef _AIX
#include <strings.h>
#endif /* _AIX */

#include "constants.H"
#include "util.H"
#include "ClientMultiplexer.H"
#include "ServerMultiplexer.H"

#include "Compresser.H"

#if defined(__EMX__ ) || defined(__CYGWIN32__)

struct sockaddr_un
{
  u_short sun_family;		/* socket family: AF_UNIX */
  char sun_path[108];		/* path name (not used) */
};

#endif


#ifdef _AIX
#include <sys/select.h>
#endif /* _AIX */

#if defined(hpux) && !defined(RLIM_INFINITY)
/* HP-UX hides this define */
#define	RLIM_INFINITY	0x7fffffff
#endif


static void Usage(const char ** argv);
static void Cleanup();
static void HandleSignal(int);
static int AwaitConnection(int portNum);
static int ConnectToRemote(char *remoteHost, int portNum);
static void DaemonInit(unsigned int);
static void KillDaemon(unsigned int);
static void MakeLockFileName(char*, unsigned int);

static int VersionNumbersMatch(int sockfd);
static int ReadDataCh(int fd, char *buf, int maxlen, char stop);
static int parseRemoteOptions(char *opts);

// This variable tells whether or not the client should initiate the
// connection.  (see -w option)
static int ClientInitiateConnection = 0;

// Maximum number of open file descriptors for this process
static unsigned int maxNumFDs = 0;

// Variable to tell output routines whether they should be quiet or not
// Added for -f option
// This variable is used by other files
int silent = 0;

// Variable to indicate whether the user wants backing store turned on
// to help reduce network traffic at some cost in server memory.
// This variable is used by other files.
int wantBackingStore = 0;

// These two variables are needed for the forking
// We don't fork by default since old versions didn't
static int dofork = 0;
static char lockfilename[512] = {0};

// And the default lockfilename - this goes in the user's $HOME dir
static const char *LOCK_FILE_NAME = ".dxpc.pid";

// Now statistics can go to a file
char logfilename[1024] = {0};
ostream *logofs;

// Controls miniLZO PutImage compression: used by other files.
int compressImages = -1;

// Info about sockets used by the client proxy (global so that
// the cleanup signal handler can access it)
static char udomSocketPathname[100];
static int useUnixDomainSocket = 1;
sockaddr_in serverAddr;

// dxpc runs in client mode or server mode
enum ProxyMode
{
  PROXY_CLIENT, PROXY_SERVER
}
proxyMode = PROXY_SERVER;

// Macro is TRUE if we should initiate the connection.
#define WE_INITIATE_CONNECTION \
    ((proxyMode == PROXY_SERVER && !ClientInitiateConnection) \
  || (proxyMode == PROXY_CLIENT && ClientInitiateConnection))


// #define COREDUMPS_ENABLED

#ifdef COREDUMPS_ENABLED
int enableCoredumps(void)
{
     rlimit rlim;
     if (getrlimit(RLIMIT_CORE, &rlim))
     {
	cerr << "Cannot read RLIMIT_CORE: " << strerror(errno) << endl;
	return -1;
     }

     if (rlim.rlim_cur < rlim.rlim_max)
     {
        rlim.rlim_cur = rlim.rlim_max;
	if (setrlimit(RLIMIT_CORE, &rlim))
        {
	     cerr << "Cannot set RLIMIT_CORE: " << strerror(errno) << endl;
             return -2;
        }
     }
     cerr << "Set RLIMIT_CORE to "<< rlim.rlim_max << endl;
     return 0;
}
#endif

int
main(int argc, char **argv)
{
  (void) SaveCopyrightFromOptimizer();

  udomSocketPathname[0] = 0;

  unsigned int 	proxyPort = DEFAULT_PROXY_PORT;
  unsigned int 	displayNum = DEFAULT_DISPLAY_NUM;
  unsigned int 	statisticsLevel = 0;
  int 		useTCPSocket = 1;
  char 		*remoteHost = NULL;
  const char 	*displaySrc = 0;

#ifdef COREDUMPS_ENABLED
  enableCoredumps();
#endif

  // used to use getopt here, but it caused too many
  // portability problems
  for (int argi = 1; argi < argc; argi++)
  {
    char *nextArg = argv[argi];
    if (*nextArg == '-')
    {
      switch (nextArg[1])
      {
      case 'b':
	{
	  const char *arg = GetArg (argi, argc, (const char **)argv);
	  if (!arg)
	     Usage ((const char **)argv);
	  if (*arg == 'a')
	    wantBackingStore = 2 /* Always */;
	  else if (*arg == 'm' || *arg == 'w')
	    wantBackingStore = 1 /* WhenMapped */;
	  else if (*arg == 'n')
	    wantBackingStore = 0;
	  else
	    Usage ((const char **)argv);
	}
	break;

      case 'd':
	{
	  const char *arg = GetArg(argi, argc, (const char **)argv);
	  if (arg == NULL)
	    Usage((const char **)argv);
	  else
	    displayNum = atoi(arg);
	}
	break;
      case 'D':
	displaySrc = GetArg(argi, argc, (const char **)argv);
	if (!displaySrc)
	{
	    Usage((const char **)argv);
	}
	break;
      case 'f':
	dofork = 1;
	break;
      case 'k':
	KillDaemon(proxyPort);		// This function doesn't return

	break;
      case 'l':
	{
	  const char *arg = GetArg(argi, argc, (const char **)argv);
	  if (!arg)
	    Usage((const char **)argv);
	  else
	    strcpy(logfilename, arg);
	}
	break;
      case 'p':
	{
	  const char *arg = GetArg(argi, argc, (const char **)argv);
	  if (arg == NULL)
	    Usage((const char **)argv);
	  else
	    proxyPort = atoi(arg);
	}
	break;
      case 's':
	{
	  const char *arg = GetArg(argi, argc, (const char **)argv);
	  if (arg == NULL)
	    Usage((const char **)argv);
	  else
	    statisticsLevel = atoi(arg);
	}
	break;
      case 't':
	useTCPSocket = 0;
	break;
      case 'u':
	useUnixDomainSocket = 0;
	break;
      case 'v':
	PrintVersionInfo();
	exit(0);
      case 'w':
	ClientInitiateConnection = 1;
	break;
      
      case 'i':
          {
	    const char *arg = GetArg(argi, argc, (const char **)argv);
	    if (arg == NULL)
	       Usage((const char **)argv);
	     else
	        compressImages = atoi(arg);
	  }	
	break;
      default:
	Usage((const char **)argv);
      }
    }
    else
    {
      if (remoteHost != NULL)
	Usage((const char **)argv);
      else
	remoteHost = nextArg;
    }
  }

  if (logfilename[0] != '\0')
  {
    logofs = new ofstream(logfilename, ios::out);
  }
  else
  {
    logofs = &cout;
  }

  int xServerAddrFamily = AF_INET;
  sockaddr *xServerAddr = NULL;
  unsigned int xServerAddrLength = 0;

  if ((!remoteHost && !ClientInitiateConnection) ||
      (remoteHost && ClientInitiateConnection))
  {
    cout << "dxpc proxy running in CLIENT mode" << endl;
    proxyMode = PROXY_CLIENT;
  }
  else
  {
    cout << "dxpc proxy running in SERVER mode" << endl;
    proxyMode = PROXY_SERVER;

    // Keep Cleanup() from deleting the real X server's pipe
    // when we exit...
    useUnixDomainSocket = 0;

    // $DISPLAY is the X server for which we'll act as a proxy
    if (!displaySrc)
    {
	displaySrc = getenv("DISPLAY");
    }
    if ((displaySrc == NULL) || (*displaySrc == 0))
    {
      cerr << "$DISPLAY is not set" << endl;
      Cleanup();
    }
    char *display = new char[strlen(displaySrc) + 1];
    if (!display)
    {
      cerr << "Out of memory duping DISPLAY" << endl;
      Cleanup();
    }
    strcpy(display, displaySrc);
    char *separator = strchr(display, ':');
    if ((separator == NULL) || !isdigit(*(separator + 1)))
    {
      cerr << "invalid DISPLAY '" << display << "'" << endl;
      Cleanup();
    }
    *separator = 0;
    int displayNum = atoi(separator + 1);
    if ((separator == display) || !strcmp(display, "unix"))
    {
      // UNIX domain port
      xServerAddrFamily = AF_UNIX;
      sockaddr_un *xServerAddrUNIX = new sockaddr_un;
      xServerAddrUNIX->sun_family = AF_UNIX;
      sprintf(udomSocketPathname, "/tmp/.X11-unix/X%d", displayNum);
      struct stat statInfo;
      if (stat(udomSocketPathname, &statInfo) == -1)
      {
#if (defined(__hpux) || defined(hpux)) && !defined(PGID_USE_PID)
	sprintf(udomSocketPathname, "/usr/spool/sockets/X11/%d", displayNum);
	if (stat(udomSocketPathname, &statInfo) == -1)
	{
#endif
	  cerr << "cannot open UNIX domain connection to X server" << endl;
	  Cleanup();
#if (defined(__hpux) || defined(hpux)) && !defined(PGID_USE_PID)
	}
#endif
      }
      strcpy(xServerAddrUNIX->sun_path, udomSocketPathname);
      xServerAddr = (sockaddr *) xServerAddrUNIX;
      //      xServerAddrLength = strlen(udomSocketPathname) + 2;
      xServerAddrLength = sizeof(sockaddr_un);
    }
    else
    {
      // TCP port
      xServerAddrFamily = AF_INET;
      int ipAddr;
      hostent *hostAddr = gethostbyname(display);
      if (hostAddr == NULL)
      {
	// on some UNIXes, gethostbyname doesn't accept IP addresses,
	// so try inet_addr:
	ipAddr = (int) inet_addr(display);
	if (ipAddr == -1)
	{
	  cerr << "Unknown host '" << display << "'" << endl;
	  Cleanup();
	}
      }
      else
	ipAddr = *(int *) hostAddr->h_addr_list[0];
      sockaddr_in *xServerAddrTCP = new sockaddr_in;
      xServerAddrTCP->sin_family = AF_INET;
      xServerAddrTCP->sin_port = htons(X_TCP_PORT + displayNum);
      xServerAddrTCP->sin_addr.s_addr = ipAddr;
      xServerAddr = (sockaddr *) xServerAddrTCP;
      xServerAddrLength = sizeof(sockaddr_in);
    }
    if (display)
    {
      delete [] display;
      display = 0;
    }
  }
  // Sanity check compressImages.
  if (WE_INITIATE_CONNECTION)
  {
      if (compressImages != -1)
      {
       	  cerr << "warning: image compression option (-i) ignored "
	       << "when initiating connection"
	       << endl;
	  compressImages = -1;
      }
  }
  else
  {
      // We will accept connections; we control image compression.
      if (compressImages == -1)
      {
	 // Default image compression level.
	 compressImages = DEFAULT_IMAGE_COMPRESSION_LEVEL;
      }
      else if (compressImages && 
	       !Compresser::isValidCompressionLevel(compressImages))
      {
	 cerr << "warning: invalid image compression level " 
	      << compressImages << ": image compression disabled"
	      << endl;
	 compressImages = 0;
      }
      if (!silent)
      {
	cout << "using image compression level " 
	     << compressImages << endl;
      }
  }

  // Increase the max # of open file descriptors for this process

  maxNumFDs = 0;
#if defined(RLIMIT_NOFILE)
  rlimit limits;
  if (getrlimit(RLIMIT_NOFILE, &limits) == 0)
  {
    if (limits.rlim_max == RLIM_INFINITY)
      maxNumFDs = 0;
    else
      maxNumFDs = (unsigned int) limits.rlim_max;
  }
#endif /* RLIMIT_NOFILE */

#if defined(_SC_OPEN_MAX)
  if (maxNumFDs == 0)
    maxNumFDs = sysconf(_SC_OPEN_MAX);
#endif

#if defined(FD_SETSIZE)
  if (maxNumFDs > FD_SETSIZE)
    maxNumFDs = FD_SETSIZE;
#endif /* FD_SETSIZE */

#if defined(RLIMIT_NOFILE)
  if (limits.rlim_cur < maxNumFDs)
  {
    limits.rlim_cur = maxNumFDs;
    setrlimit(RLIMIT_NOFILE, &limits);
  }
#endif /* RLIMIT_NOFILE */

  if (maxNumFDs == 0)
  {
    cerr <<
      "cannot determine number of available file descriptors, exiting!"
      << endl;
    return 1;
  }
  // Install some signal handlers for graceful shutdown
  signal(SIGHUP, HandleSignal);
  signal(SIGINT, HandleSignal);
  signal(SIGTERM, HandleSignal);

  signal(SIGPIPE, (void (*)(int)) SIG_IGN);


  // If running as client proxy, open sockets that mimic an
  // X display to which X clients can connect (e.g., unix:8
  // and <hostname>:8)
  int tcpFD = -1;
  int unixFD = -1;
  if (proxyMode == PROXY_CLIENT)
  {
    if (useTCPSocket)
    {
      // Open TCP socket for display
      tcpFD = socket(AF_INET, SOCK_STREAM, PF_UNSPEC);
      if (tcpFD == -1)
      {
	cerr << "socket() failed for TCP socket, errno=" <<
	  errno << endl;
	Cleanup();
      }
      int flag = 1;
      if (setsockopt(tcpFD, SOL_SOCKET, SO_REUSEADDR, (char *) &flag,
		     sizeof(flag)) < 0)
      {
	cerr << "setsockopt(SO_REUSEADDR) failed for TCP socket, errno = " 
	     << errno << endl;
      }
      sockaddr_in tcpAddr;
      tcpAddr.sin_family = AF_INET;
      unsigned int xPortTCP = X_TCP_PORT + displayNum;
      tcpAddr.sin_port = htons(xPortTCP);
      tcpAddr.sin_addr.s_addr = htonl(INADDR_ANY);
      if (bind(tcpFD, (sockaddr *) & tcpAddr, sizeof(tcpAddr)) == -1)
      {
	cerr << "bind() failed for TCP port " << xPortTCP <<
	  ", errno=" << errno << endl;
	Cleanup();
      }
      if (listen(tcpFD, 5) == -1)
      {
	cerr << "listen() failed for TCP port " << xPortTCP <<
	  ", errno=" << errno << endl;
	Cleanup();
      }
    }
    if (useUnixDomainSocket)
    {
      // Open UNIX domain socket for display
      unixFD = socket(AF_UNIX, SOCK_STREAM, PF_UNSPEC);
      if (unixFD == -1)
      {
	cerr << "socket() failed for UNIX domain socket, errno=" <<
	  errno << endl;
	Cleanup();
      }
      sockaddr_un unixAddr;
      unixAddr.sun_family = AF_UNIX;
      struct stat dirStat;
      if ((stat("/tmp/.X11-unix", &dirStat) == -1) && (errno == ENOENT))
      {
	mkdir("/tmp/.X11-unix", 0777);
	chmod("/tmp/.X11-unix", 0777);
      }
      sprintf(udomSocketPathname, "/tmp/.X11-unix/X%d", displayNum);
      strcpy(unixAddr.sun_path, udomSocketPathname);
      if (bind(unixFD, (sockaddr *) & unixAddr,
	       strlen(udomSocketPathname) + 2) == -1)
      {
	cerr << "bind() failed for UNIX domain socket " <<
	  udomSocketPathname << ", errno=" << errno << endl;
	cerr << "This probably means you do not have sufficient rights to "
	     << "write to /tmp/.X11-unix/." << endl
	     << "Either use the -u option or obtain the necessary rights."
	     << endl;
	Cleanup();
      }
      if (listen(unixFD, 5) == -1)
      {
	cerr << "listen() failed for UNIX domain socket " <<
	  udomSocketPathname << ", errno=" << errno << endl;
	Cleanup();
      }
    }
  }
  if (dofork)
  {
    DaemonInit(proxyPort);
  }
  // Set up low-bandwidth connection between the proxies
  int proxyFD = -1;
  if (WE_INITIATE_CONNECTION)
  {
      proxyFD = ConnectToRemote(remoteHost, proxyPort);
      // Compare the version number sent by the host to our version. If we don't
      // get back an identical string we exit.
      if (!VersionNumbersMatch(proxyFD))
      {
	Cleanup();
      }
      if (!silent)
      {
	*logofs << "connected to " 
	        << ((proxyMode == PROXY_CLIENT) ? "server" : "client") 
	        << " proxy\nready" << endl;
      }
   }
   else
   {
      proxyFD = AwaitConnection(proxyPort);

      if (!silent)
      {
	*logofs << "connected to "
		<< ((proxyMode == PROXY_CLIENT) ? "server" : "client")
	        << " proxy\nready" << endl;
      }
   }

  // Create multiplexer
  Multiplexer *multiplexer;
  if (proxyMode == PROXY_SERVER)
    multiplexer = new ServerMultiplexer(proxyFD, xServerAddrFamily,
					xServerAddr, xServerAddrLength,
					statisticsLevel);
  else
    multiplexer = new ClientMultiplexer(proxyFD, statisticsLevel);


  if (compressImages)
  {
    int err = lzo_init();
    if (err != LZO_E_OK)
    {
       cerr << "warning: cannot initialize image compression library"
	    << " (error " << err << ")" << endl;
       compressImages = 0;
    }
  }

  // Loop endlessly, reading from all of the open file descriptors
  for (;;)
  {
    fd_set readSet;
    FD_ZERO(&readSet);
    FD_SET(proxyFD, &readSet);
    unsigned int numFDsToSelect = proxyFD + 1;
    if (proxyMode == PROXY_CLIENT)
    {
      if (useTCPSocket)
      {
	FD_SET(tcpFD, &readSet);
	if (tcpFD >= (int) numFDsToSelect)
	  numFDsToSelect = tcpFD + 1;
      }
      if (useUnixDomainSocket)
      {
	FD_SET(unixFD, &readSet);
	if (unixFD >= (int) numFDsToSelect)
	  numFDsToSelect = unixFD + 1;
      }
    }
    multiplexer->setSelectFDs(&readSet, numFDsToSelect);

    timeval delay;
    delay.tv_sec = 600;
    delay.tv_usec = 0;
    // PGID_USE_PID, defined in <sys/types.h>, is specific to HP-UX 10.x
#if (defined(__hpux) || defined(hpux)) && !defined(PGID_USE_PID)
    int result = select(numFDsToSelect, (int *) &readSet, NULL, NULL, &delay);
#else
    int result = select(numFDsToSelect, &readSet, NULL, NULL, &delay);
#endif
    if (result == -1)
    {
      if (errno == EINTR)
	continue;
      cerr << "select() failed, errno=" << errno << endl;
      Cleanup();
    }
    for (unsigned int j = 0; j < numFDsToSelect; j++)
    {
      if (!(FD_ISSET(j, &readSet)))
	continue;
      if (proxyMode == PROXY_CLIENT)
      {
	if (((int) j == tcpFD) && useTCPSocket)
	{
	  sockaddr_in newAddr;
	  ACCEPT_SOCKLEN_T addrLen = sizeof(sockaddr_in);
	  int newFD = accept(tcpFD, (sockaddr *) & newAddr, &addrLen);
	  if (newFD == -1)
	  {
	    cerr << "accept() failed, errno=" << errno << endl;
	    Cleanup();
	  }
	  multiplexer->createNewConnection(newFD);
	  continue;
	}
	if (((int) j == unixFD) && useUnixDomainSocket)
	{
	  sockaddr_un newAddr;
	  ACCEPT_SOCKLEN_T addrLen = sizeof(sockaddr_un);
	  int newFD = accept(unixFD, (sockaddr *) & newAddr, &addrLen);
	  if (newFD == -1)
	  {
	    cerr << "accept() failed, errno=" << errno << endl;
	    Cleanup();
	  }
	  multiplexer->createNewConnection(newFD);
	  continue;
	}
      }
      if (!multiplexer->handleSelect(j))
	Cleanup();
    }
  }

  return 0;
}

static void
Usage(const char ** argv)
{
cerr << "Usage: " << argv[0] << 
" [common options] [client options | server options] [connect options]" << endl
<< endl; cerr
<< "[common options]" << endl
<< "    -p port_num		Specifies the TCP port on which the LISTENING process" << endl
<< "			listens, or to which the CONNECTING process connects" << endl
<< " 			(default is "<< DEFAULT_PROXY_PORT << ")." << endl
<< "    -f			fork (starts process in background)." << endl
<< "    -k			kill (kills backgrounded process started via -f)." << endl
<< "    -v			Print version info and exit." << endl
<< "    -s (1|2)		Print compression stats; -s 1 prints a summary on" << endl
<< "			exit, -s 2 prints details for each X application." << endl
<< "    -l log_file	write output to a logfile instead of stdout." << endl
<< endl; cerr
<< "[client options] (only meaningful to the CLIENT process)" << endl
<< "    -i compression_lvl  Specify image bitmap compression level (0-9,99," << endl
<< "			999). 0 disables bitmap compression; 999 is maximal;" << endl
<< "			other values are tradeoffs of speed vs. compression." << endl
<< "			Default is "<< DEFAULT_IMAGE_COMPRESSION_LEVEL << "." << endl
<< "    -d display_num	Specify display number to emulate. Default is 8." << endl
<< "    -u			Do not attempt to open UNIX domain socket for" << endl
<< "			proxied server." << endl
<< "    -t			Do not attempt to open TCP socket for proxied server." << endl
<< endl; cerr
<< "[server options] (only meaningful to the SERVER process)" << endl
<< "    -D			Specify X host on which to display proxied" << endl
<< "			applications. Defaults to value of the DISPLAY" << endl
<< "			environment variable." << endl
<< "    -b (a|w)        force backing store (-ba = always, -bw = when mapped)" << endl
<< endl; cerr
<< "[connect options]" << endl
<< "     hostname		The name of the machine on which the LISTENING" << endl
<< "			process is running. The presence of the hostname" << endl
<< "			argument specifies that this is the CONNECTING" << endl
<< "			process. Its absence specifies that this is the" << endl
<< "			LISTENING process." << endl
<< "     -w			If this is the CONNECTING process, specifies that it" << endl
<< "			is the CLIENT process (by default, the CONNECTING" << endl
<< "			process is the SERVER process). If this is the" << endl
<< "			LISTENING process, specifies that it is the SERVER" << endl
<< "			process (by default, the LISTENING process is the" << endl
<< "			CLIENT process)." << endl
<< endl; cerr
<< "Notes on modes:" << endl
<< endl; cerr
<< "dxpc has two modes; the connection mode, which is either LISTENING or" << endl
<< "CONNECTING; and the X mode, which is either CLIENT or SERVER." << endl
<< endl; cerr
<< "The LISTENING process waits for a CONNECTING process to initiate the TCP" << endl
<< "connection between the two processes. The LISTENING process must always be" << endl
<< "started first. The CONNECTING process initiates the connection to the" << endl
<< "LISTENING process. dxpc will run as the CONNECTING process if a hostname" << endl
<< "argument is supplied (see connect options, above). Otherwise it will run as" << endl
<< "the LISTENING process." << endl
<< endl; cerr
<< "The SERVER process is typically located on the same machine as the real X" << endl
<< "server, and is responsible for displaying the output of applications. The" << endl
<< "CLIENT process is typically located on the same machine as the X" << endl
<< "applications, and is responsible for forwarding the output of those" << endl
<< "applications to the SERVER process. By default, dxpc runs as the CLIENT" << endl
<< "process if it is the LISTENING process (due to the lack of a hostname" << endl
<< "argument) and the SERVER process if it is the CONNECTING process, but the -w" << endl
<< "switch reverses this (see connect options, above)." << endl
<< endl; cerr
<< "For example, the command 'dxpc myhost.work.com' starts dxpc as the" << endl
<< "CONNECTING process (because a host name is supplied) and the SERVER process" << endl
<< "(because it is the CONNECTING process and -w is not supplied). The command" << endl
<< "'dxpc -w' starts dxpc as the LISTENING process (because no hostname is" << endl
<< "supplied) and the SERVER process (because it is the LISTENING process, and" << endl
<< "-w reverses the usual logic)." << endl;

  exit(1);
}

static void
Cleanup()
{
  if (!silent)
    *logofs << "Closing all file descriptors and shutting down..." << endl;

  if (dofork)
    if (remove(lockfilename))
      perror("Unable to remove lockfile");

  if (useUnixDomainSocket)
    unlink(udomSocketPathname);

  for (unsigned int i = 0; i < maxNumFDs; i++)
    (void) close(i);

  exit(1);
}

static void
HandleSignal(int)
{
  Cleanup();
}

// Open TCP socket to listen for server proxy; block until server
// proxy connects, then close listener socket and return FD of socket
// on which server proxy is connected
//
static int
AwaitConnection(int portNum)
{
  int proxyFD = socket(AF_INET, SOCK_STREAM, 0);
  if (proxyFD == -1)
  {
    cerr << "socket() failed for TCP socket, errno=" << errno << endl;
    Cleanup();
  }
  int flag = 1;
  if (setsockopt(proxyFD, SOL_SOCKET, SO_REUSEADDR, (char *) &flag,
		 sizeof(flag)) < 0)
  {
    cerr << "setsockopt(SO_REUSEADDR) failed for proxy port, errno=" <<
      errno << endl;
  }
  sockaddr_in tcpAddr;
  tcpAddr.sin_family = AF_INET;
  tcpAddr.sin_port = htons(portNum);
  tcpAddr.sin_addr.s_addr = htonl(INADDR_ANY);
  if (bind(proxyFD, (sockaddr *) & tcpAddr, sizeof(tcpAddr)) == -1)
  {
    cerr << "bind() failed for TCP port " << portNum <<
      ", errno=" << errno << endl;
    Cleanup();
  }
  if (listen(proxyFD, 1) == -1)
  {
    cerr << "listen() failed for TCP port " << portNum <<
      ", errno=" << errno << endl;
    Cleanup();
  }
  for (;;)
  {
    fd_set readSet;
    FD_ZERO(&readSet);
    FD_SET(proxyFD, &readSet);
#if (defined(__hpux) || defined(hpux)) && !defined(PGID_USE_PID)
    int result = select(proxyFD + 1, (int *) &readSet, NULL, NULL, NULL);
#else
    int result = select(proxyFD + 1, &readSet, NULL, NULL, NULL);
#endif
    if (result == -1)
    {
      if (errno == EINTR)
	continue;
      cerr << "select() failed, errno=" << errno << endl;
      Cleanup();
    }
    if (FD_ISSET(proxyFD, &readSet))
    {
      sockaddr_in newAddr;
      ACCEPT_SOCKLEN_T addrLen = sizeof(sockaddr_in);
      int newFD = accept(proxyFD, (sockaddr *) & newAddr, &addrLen);
      if (newFD == -1)
      {
	cerr << "accept() failed, errno=" << errno << endl;
	Cleanup();
      }
      // Now we send our version number.
      char version[40];
      if (!WE_INITIATE_CONNECTION)
      {
      	sprintf(version, "DXPC %i.%i/ci=%d", 
	        DXPC_VERSION_MAJOR, DXPC_VERSION_MINOR,
                compressImages);
      }
      else
      {
	sprintf(version, "DXPC %i.%i", DXPC_VERSION_MAJOR, DXPC_VERSION_MINOR);
      }
      write(newFD, version, strlen(version) + 1);

      // If the client doesn't like our version it will simply close the
      // socket.

      close(proxyFD);
      return newFD;
    }
  }
}


// Connect to remote proxy.  If successful, return FD of connection;
// if unsuccessful, return -1
//
static int
ConnectToRemote(char *remoteHost, int portNum)
{
  int remoteIPAddr;
  hostent *hostAddr = gethostbyname(remoteHost);
  if (hostAddr == NULL)
  {
    // on some UNIXes, gethostbyname doesn't accept IP addresses,
    // so try inet_addr:
    remoteIPAddr = (int) inet_addr(remoteHost);
    if (remoteIPAddr == -1)
    {
      cerr << "Unknown host '" << remoteHost << "'" << endl;
      Cleanup();
    }
  }
  else
    remoteIPAddr = *(int *) hostAddr->h_addr_list[0];

  int remoteProxyFD = socket(AF_INET, SOCK_STREAM, PF_UNSPEC);
  if (remoteProxyFD == -1)
  {
    cerr << "socket() failed, errno=" << errno << endl;
    Cleanup();
  }
  int flag = 1;
  if (setsockopt(remoteProxyFD, SOL_SOCKET, SO_REUSEADDR, (char *) &flag,
		 sizeof(flag)) < 0)
  {
    cerr << "setsockopt(SO_REUSEADDR) failed for proxy port, errno=" <<
      errno << endl;
  }
  if (!silent)
  {
    *logofs << "trying to connect to remote proxy..." << endl;
  }
  sockaddr_in addr;
  addr.sin_family = AF_INET;
  addr.sin_port = htons(portNum);
  addr.sin_addr.s_addr = remoteIPAddr;
  if (connect(remoteProxyFD, (sockaddr *) & addr, sizeof(sockaddr_in)) == -1)
  {
    cerr << "connect() failed, errno=" << errno << endl;
    close(remoteProxyFD);
    Cleanup();
  }
  flag = 1;
  setsockopt(remoteProxyFD, IPPROTO_TCP, TCP_NODELAY, (char *) &flag,
	     sizeof(int));

  return remoteProxyFD;
}

static int
VersionNumbersMatch(int sockfd)
{
  // We consider the version numbers to match if the major and minor
  // numbers match.  Different patch levels should by definition be
  // compatible with each other.

  char version[20];
  char recvmsg[40];
  char *opts;

  sprintf(version, "DXPC %i.%i", DXPC_VERSION_MAJOR, DXPC_VERSION_MINOR);

  if (ReadDataCh(sockfd, recvmsg, sizeof(recvmsg), '\0') < 0)
  {
    return 0;
  }

  // Split any available options off from version number.
  if ((opts = strchr(recvmsg, '/')) != NULL)
  {
	*opts++ = 0;	
  } 

  if (strncmp(recvmsg, version, strlen(version)))
  {
    // Make sure recvmsg will be printable
    unsigned int ctr;
    for (ctr = 0;
	 ctr < strlen(recvmsg) &&
	 (isgraph(recvmsg[ctr]) || isspace(recvmsg[ctr]));
	 ctr++);

    recvmsg[ctr] = 0;

    cerr << "Error version numbers don't match!" << endl
      << "Local version: " << version << endl
      << "Remote version: " << recvmsg << endl;

    return 0;
  }

  if (parseRemoteOptions(opts))
  {
	// Not strictly a version number mismatch, but fail anyway.
	return 0;
  }

  // We initiated the connection; if we didn't learn a compression level, 
  // we're screwed.
  if (compressImages == -1)
  {
	cerr << "error: host didn't specify image compression level"
	     << endl; 
        return 0;
  }
  return 1;
}

static int
ReadDataCh(int fd, char *buf, int maxlen, char stop)
{
  int ctr = 0;
  int result;

  while (ctr < maxlen)
  {
    if ((result = read(fd, buf + ctr, 1)) == -1 || !result)
    {
      return -1;
    }
    if (result && *(buf + ctr++) == stop)
    {
      return ctr;
    }
  }

  return 0;
}

static void
DaemonInit(unsigned int displayNum)
{
#ifdef __EMX__
  cerr << "The daemon option is disabled under OS/2" << endl;
  exit(-1);
#else

  switch (fork())
  {
  case -1:
    perror("dxpc");
    Cleanup();
  case 0:
    break;
  default:
    exit(0);
  }
  pid_t pid;

  if ((pid = setsid()) == -1)
  {
    cerr << "Error setsid() failed." << endl;
    Cleanup();
  }
  MakeLockFileName(lockfilename, displayNum);

  // First we try to open the file to see if dxpc is already running
  FILE *pidfile = fopen(lockfilename, "r");

  if (pidfile)
  {
    // The open was successful
    // So we try and read a pid out of it.
    char oldpid[10];
    switch (fread(oldpid, 1, sizeof(oldpid), pidfile))
    {
    case 0:
      cerr << "Found empty pidfile " << lockfilename
	<< ".  Overriding." << endl;
      break;
    case -1:
      cerr << "Error reading from old pidfile " << lockfilename
	<< ".  Overriding." << endl;
      break;
    default:
      // Do a sanity check on the returned data
      if (!isdigit((int) ((unsigned char) oldpid[0])))
      {
	cerr << "Invalid data in pidfile " << lockfilename
	  << ".  Aborting." << endl;

	fclose(pidfile);
	Cleanup();
      }
      long oldpidval = atoi(oldpid);

      int override = 0;
      switch (kill(oldpidval, 0))
      {
      case ESRCH:
      case EPERM:
	// Either the pid doesn't exist or is owned by someone
	// else.  It's probably safe to override.
	cerr << "Stale pidfile found.  Overriding." << endl;
	override = 1;
	break;
      default:
	cerr << "Error.  It appears another dxpc is running at pid "
	     << oldpidval << endl
	     << "If this isn't correct, then delete " << lockfilename
	     << endl;
      }
      if( override ) break;
      fclose(pidfile);
      dofork = 0;		// So Cleanup() won't delete the lockfile

      Cleanup();
      }

    fclose(pidfile);
  }
  if (!(pidfile = fopen(lockfilename, "w")))
  {
    perror("dxpc");
    Cleanup();
  }
  fprintf(pidfile, "%lu", (unsigned long)pid);
  fclose(pidfile);

  // Now turn off all non-error output to the console
  silent = 1;

#endif
  return;
}

void
KillDaemon(unsigned int displayNum)
{
  char lockfilename[512];

  MakeLockFileName(lockfilename, displayNum);

  FILE *pidfile = fopen(lockfilename, "r");

  if (pidfile)
  {
    // The open was successful
    // So we try and read a pid out of it.
    char pid[10];
    switch (fread(pid, 1, sizeof(pid), pidfile))
    {
    case 0:
    case -1:
      cerr << "Error reading pid from " << lockfilename << endl
	<< "You will have to manually kill the daemon." << endl;

      break;
    default:
      fclose(pidfile);

      // Do a sanity check on the returned data
      if (!isdigit((int) ((unsigned char) pid[0])))
      {
	cerr << "Invalid data in pidfile " << lockfilename
	  << ".  Aborting." << endl;

	exit(1);
      }
      long pidval = atoi(pid);

      cout << "Killing dxpc at pid " << pidval << endl;

      if (kill(pidval, SIGTERM) == -1)
      {
	perror("dxpc");
	cerr << "Leaving pidfile intact." << endl;
	exit(1);
      }
    }
    exit(0);
  }
  else
  {
    cerr << "No daemon is running." << endl;
    exit(1);
  }
}


static void
MakeLockFileName(char* lockfilename, unsigned int displayNum)
{
  char *homedir = getenv("HOME");
  if (!homedir)
  {
    cerr << "You have no environment variable HOME!" << endl
	 << "How did that happen?" << endl;
    exit(1);
  }
  strcpy(lockfilename, homedir);
  strcat(lockfilename, "/");
  strcat(lockfilename, LOCK_FILE_NAME);	// constants.h
  struct utsname unameInfo;
  if(uname(&unameInfo) == -1)
  {
    unameInfo.nodename[0] = 0;
  }
  else
  {
    char* dotptr = strchr( unameInfo.nodename, '.' );
    if( dotptr ) *dotptr = 0;
    strcat(lockfilename, "-");
    strcat(lockfilename, unameInfo.nodename);
  }

  struct passwd* passdata;
  if( (passdata = getpwuid(getuid())) != 0)
  {
    strcat(lockfilename, "-");
    strcat(lockfilename, passdata->pw_name);
  }
  else
  {
    strcat(lockfilename, "");
  }
  sprintf(lockfilename + strlen(lockfilename), "-%u", displayNum);
}

// Parse the option string passed from the remote end at startup.
static int parseRemoteOptions(char *opts)
{
    char *name, *value;

    // The options string is intended to be a series of name/value
    // tuples in the form name=value seperated by the '/' character.  
    name = strtok(opts, "=");
    while (name)
    {
        value = strtok(NULL, "/");
	if (!value)
	{
	    cerr << "error in remote option " << name 
		 << ": no value found" << endl;
	    return -1;
	}

        // Currently we only have one known option...
        if (!strcmp(name, "ci"))
        {
	    if (WE_INITIATE_CONNECTION)
	    {
		// The far end just told us what level of image compression
		// to use.
		compressImages = atoi(value);
		if (!silent)
		{
		    cout << "using image compression level " 
		         << compressImages << endl;
	        }
	    }
	    else
	    {
		cerr << "image compression option from initiating side"
		        " ignored" << endl;
	    }	
        }
	else
	{
	    cerr << "unknown remote option: " << name << " = " << value
		 << endl;
	    return -1;
	}

	name = strtok(NULL, "=");	
    }
    return 0;
}
