/* Copyright (C) 2006 G.P. Halkes
   Licensed under the Open Software License version 2.0 */

#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include "definitions.h"

static const char resetColor[] = "\033[0m";

/** Start the diff program. */
static FILE *startDiff(void) {
	FILE *diff;
	char *command;
	size_t length;
	
	/* Allocate memory to hold the diff command with arguments. */
	length = strlen(option.ignoreCase ? DIFF_COMMAND_CI : DIFF_COMMAND) + 1 + strlen(option.oldFile.tokens->name) + 1 + strlen(option.newFile.tokens->name) + 1;
	if (option.diffOption != NULL)
		length += strlen(option.diffOption) + 1;
	if ((command = malloc(length)) == NULL) {
		fatal(_("Could not allocate memory: %s\n"), strerror(errno));
	}
	/* Compose the diff command. */
	strcpy(command, option.ignoreCase ? DIFF_COMMAND_CI : DIFF_COMMAND);
	strcat(command, " ");
	if (option.diffOption != NULL) {
		strcat(command, option.diffOption);
		strcat(command, " ");
	}
	strcat(command, option.oldFile.tokens->name);
	strcat(command, " ");
	strcat(command, option.newFile.tokens->name);
	
	/* Start diff */
	if ((diff = popen(command, "r")) == NULL)
		fatal(_("Failed to execute diff: %s\n"), strerror(errno));
	free(command);
	return diff;
}

typedef enum {ADD, DEL, COMMON} Mode;

/** Skip or print the next bit of whitespace from @a file.
	@param file The file with whitespace.
	@param print Skip or print.
	@param mode What type of output to generate.
*/
static void handleNextWhitespace(TempFile *file, bool print, Mode mode) {
	int c;
	while ((c = getc(file->file)) != EOF) {
		if (c == option.whitespaceDelimiter)
			return;
		if (print) {
			/* Less mode also over-strikes whitespace */
			if (option.less && c != '\n') {
				if (mode == DEL) {
					putchar('_');
					putchar('\010');
				}
			}
			putchar(c);
		}
	}
}

/** Skip or print the next token from @a file.
	@param file The file with tokens.
	@param print Skip or print.
	@param mode What type of output to generate.
*/
static void handleNextToken(TempFile *file, bool print, Mode mode) {
	int c;
	while ((c = getc(file->file)) != EOF) {
		if (c == '\n')
			return;
		if (print) {
			/* Re-transliterate the characters, if necessary. */
			if (option.transliterate && c == '\\') {
				c = getc(file->file);
				switch (c) {
					case 'n':
						c = '\n';
						break;
					case '\\':
						break;
#ifdef NO_MINUS_A
					default:
						if (isxdigit(c)) {
							int value = (c <= '9' ? c - '0' : c - 'A' + 10) << 4;
							c = getc(file->file);
							if (c == EOF)
								fatal(_("Error reading back input\n"));
							c = value + (c <= '9' ? c - '0' : c - 'A');
						} else {
							fatal(_("Error reading back input\n"));
						}
#else
					default:
						fatal(_("Error reading back input\n"));
#endif
				}
			}
			/* Printer mode and less mode do special stuff, per character. */
			if ((option.printer || option.less) && c != '\n') {
				if (mode == DEL) {
					putchar('_');
					putchar('\010');
				} else if (mode == ADD) {
					putchar(c);
					putchar('\010');
				}
			}
			putchar(c);
		}
	}
}

/** Skip or print the next whitespace and tokens from @a file.
	@param file The @a InputFile to use.
	@param idx The last word to print or skip.
	@param print Skip or print.
	@param mode What type of output to generate.
*/
static void handleWord(InputFile *file, int idx, bool print, Mode mode) {
	while (file->lastPrinted < idx) {
		handleNextWhitespace(file->whitespace, print, mode);
		handleNextToken(file->tokens, print, mode);
		file->lastPrinted++;
	}
}

/** Print (or skip if the user doesn't want to see) the common words.
	@param idx The last word to print (or skip).
*/
void printToCommonWord(int idx) {
	handleWord(&option.newFile, idx, option.printCommon, COMMON);
}

/** Print (or skip if the user doesn't want to see) words from @a file.
	@param range The range of words to print (or skip).
	@param file The @a InputFile to print from.
	@param mode Either ADD or DEL, uded for printing of start/stop markers.
*/
static void printWords(int *range, InputFile *file, bool print, Mode mode) {
	/* Skip to bit to print */
	handleWord(file, range[0] - 1, false, mode);
	
	/* Print the firs word. As we need to add the markers AFTER the first bit of
	   white space, we can't just ust handleWord */

	/* Print preceding whitespace. Should not be overstriken, so print as common */
	handleNextWhitespace(file->whitespace, print, COMMON);
	/* Print start marker */
	if (print && option.needStartStop) {
		if (mode == ADD) {
			if (option.colorMode)
				fwrite(option.addColor, 1, option.addColorLen, stdout);
			fwrite(option.addStart, 1, option.addStartLen, stdout);
		} else {
			if (option.colorMode)
				fwrite(option.delColor, 1, option.delColorLen, stdout);
			fwrite(option.delStart, 1, option.delStartLen, stdout);
		}
	}
	/* Print first word */
	handleNextToken(file->tokens, print, mode);
	file->lastPrinted++;
	/* Print following words */
	handleWord(file, range[range[1] >= 0 ? 1: 0], print, mode);
	/* Print stop marker */
	if (print && option.needStartStop) {
		if (mode == ADD)
			fwrite(option.addStop, 1, option.addStopLen, stdout);
		else
			fwrite(option.delStop, 1, option.delStopLen, stdout);

		if (option.colorMode)
			fwrite(resetColor, 1, sizeof(resetColor) - 1, stdout);
	}
}

/** Print (or skip if the user doesn't want to see) deleted words.
	@param range The range of words to print (or skip).
*/
void printDeletedWords(int *range) {
	printWords(range, &option.oldFile, option.printDeleted, DEL);
}

/** Print (or skip if the user doesn't want to see) inserted words.
	@param range The range of words to print (or skip).
*/
void printAddedWords(int *range) {
	printWords(range, &option.newFile, option.printAdded, ADD);
}

/** Print (or skip if the user doesn't want to see) the last (common) words of both files. */
void printEnd(void) {
	if (!option.printCommon)
		return;
	while(!feof(option.newFile.tokens->file)) {
		handleNextWhitespace(option.newFile.whitespace, true, COMMON);
		handleNextToken(option.newFile.tokens, true, COMMON);
	}
}

/** Read the output of the diff command, and call the appropriate print routines */
void doDiff(void) {
	int c, command, currentIndex, range[4];
	FILE *diff;
	
	diff = startDiff();

	if (option.needMarkers)
		puts("======================================================================");
	
	while ((c = getc(diff)) != EOF) {
		if (c == '<' || c == '>' || c == '-') {
			/* Skip all lines showing differences */
			while ((c = getc(diff)) != EOF && c != '\n') {}
			if (c == EOF)
				break;
		} else if (isdigit(c)) {
			/* Initialise values for next diff */
			currentIndex = 0;
			memset(range, 0xff, sizeof(range));
			range[0] = c - '0';
			command = 0;
			/* Read until the end of the line, or the end of the input */
			while ((c = getc(diff)) != EOF && c != '\n') {
				if (isdigit(c)) {
					/* Line number */
					range[currentIndex] = range[currentIndex] * 10 + (c - '0');
				} else if (c == ',') {
					/* Line number separator */
					if (currentIndex == 3 || currentIndex == 1)
						fatal(_("Error parsing diff output\n"));
					range[++currentIndex] = 0;
				} else if (strchr("acd", c) != NULL) {
					/* Command code */
					if (currentIndex == 3 || command != 0)
						fatal(_("Error parsing diff output\n"));
					command = c;
					currentIndex = 2;
					range[currentIndex] = 0;
				} else {
					fatal(_("Error parsing diff output\n"));
				}
			}
			/* Check that a command is set */
			if (command == 0)
				fatal(_("Error parsing diff output\n"));
			differences = 1;
			if (currentIndex != 2 && currentIndex != 3)
				fatal(_("Error parsing diff output\n"));
			/* Print common words. For delete commands we need to take into
			   account that the words were BEFORE the named line. */
			printToCommonWord(command != 'd' ? range[2] - 1 : range[2]);
			if (command != 'a')
				printDeletedWords(range);
			if (command != 'd')
				printAddedWords(range + 2);
			if (option.needMarkers)
				puts("\n======================================================================");

			/* Update statistics */
			switch (command) {
				case 'a':
					statistics.added += range[3] >= 0 ? range[3] - range[2] + 1 : 1;
					break;
				case 'd':
					statistics.deleted += range[1] >= 0 ? range[1] - range[0] + 1 : 1;
					break;
				case 'c':
					statistics.newChanged += range[3] >= 0 ? range[3] - range[2] + 1 : 1;
					statistics.oldChanged += range[1] >= 0 ? range[1] - range[0] + 1 : 1;
					break;
				default:
					PANIC();
			}
			if (c == EOF)
				break;
		} else {
			fatal(_("Error parsing diff output\n"));
		}
	}
	pclose(diff);
	printEnd();
}
