 /*
 * wirerdata.c
 *
 * Extension to wire.c, specified to do the conversion of rdata's for
 * every type of rr
 *
 * (c) NLnet Labs, 2004
 *
 * See the file LICENSE for the license
 *
 */
 
#include "common.h"

/**
 * Returns the size of the canonical domain name (in bytes, including dots)
 * todo: move this to better file?
 */
size_t
getnamesize(uint8_t *buf, size_t offset)
{
	size_t namelength = 0;
	size_t labeloffset = offset;
	uint8_t labelsize = buf[labeloffset];
	size_t pos;
	int i;
	uint16_t int16;
	uint8_t *minibuf;

	size_t pointers[256];
	int nrofpointers = 0;

	/* just calculate the name length first */
	while (labelsize > 0) {
		/* compression check */
		if (labelsize >= 192) {
			minibuf = xmalloc(2);  
			minibuf[0] = buf[labeloffset];
			minibuf[1] = buf[labeloffset+1];
			set_bit(&minibuf[0], 7, 0);
			set_bit(&minibuf[0], 6, 0);
			memcpy(&int16, &minibuf[0], 2);
			labeloffset = ntohs(int16);
			xfree(minibuf);

			/* check for looping pointers etc */
			if (nrofpointers > 256) {
				error("Too many label pointers in dname");
				return namelength;
			} else {
				for (i=0; i < nrofpointers; i++) {
					if (pointers[i] == labeloffset) {
						error("Compression pointer loop detected in dname\n");
						return namelength;
					}
				}
				pointers[i] = labeloffset;
				nrofpointers++;
			}

		} else if (labelsize > 63) {
			/* labelsize too large */
			error("label size (63) exceeded");
			return namelength;
		} else {
			for (pos = 1; pos <= labelsize; pos++) {
				/* add extra size for escape characters */
				if (buf[labeloffset+pos] == 0) {
					namelength += 4;
				} else if (buf[labeloffset+pos] == '.') {
					namelength += 2;
				} else if (isascii(buf[labeloffset+pos])) {
					namelength++;
				} else {
					namelength += 4;
				}
			}
			labeloffset+=labelsize;
			labeloffset++;
			namelength++;
		}
		labelsize = buf[labeloffset];
	}
	return namelength+1;
}

/**
 * Puts the dname starting at position offset on the wire buf into the rr
 * (canonical form)
 * returns the number of bytes the dname used on the wire (to update offset)
 */
size_t
wire2rdata_dname(uint8_t *buf, size_t offset, struct t_rr *rr)
{
	size_t origoffset = offset;
	uint8_t *data;
	size_t namelength;

	namelength = getnamesize(buf, offset);
	data = xmalloc(namelength);
	memset(data, 0, namelength);
	offset += wire2namestr(buf, offset, (char *) data);
	rr_add_rdata(rdata_create(data, namelength-1), rr);
	xfree(data);
	return offset - origoffset;
}

/**
 * Puts the 48 bit integer from the wire at position offset in the given rr
 */
size_t
wire2rdata_int48(uint8_t *buf, size_t offset, struct t_rr *rr)
{
	/* there is no type for 48 bits int (is there?)*/
	uint64_t int64;
	uint8_t *data;
	
	memcpy(&int64, &buf[offset], 6);
	data = xmalloc(48);
	snprintf((char *) data, 48, "%lu", (unsigned long) ntohl((u_long)int64));
	rr_add_rdata(rdata_create(data, strlen((char *) data)), rr);
	xfree(data);
	return 6;
}

/**
 * Puts the 32 bit integer from the wire at position offset in the given rr
 */
size_t
wire2rdata_int32(uint8_t *buf, size_t offset, struct t_rr *rr)
{
	uint32_t int32;
	uint8_t *data;
	
	memcpy(&int32, &buf[offset], 4);
	data = xmalloc(24);
	snprintf((char *) data, 24, "%lu", (unsigned long) ntohl((u_long)int32));
	rr_add_rdata(rdata_create(data, strlen((char *) data)), rr);
	xfree(data);
	return 4;
}

/**
 * Puts the 16 bit integer from the wire at position offset in the given rr
 */
size_t
wire2rdata_int16(uint8_t *buf, size_t offset, struct t_rr *rr)
{
	uint16_t int16;
	uint8_t *data;
	
	memcpy(&int16, &buf[offset], 2);
	data = xmalloc(12);
	snprintf((char *) data, 12, "%d", (int) ntohs(int16));
	rr_add_rdata(rdata_create(data, strlen((char *) data)), rr);
	xfree(data);
	return 2;
}

/**
 * Puts the 8 bit integer from the wire at position offset in the given rr
 */
size_t
wire2rdata_int8(uint8_t *buf, size_t offset, struct t_rr *rr)
{
	uint8_t *data;
	
	data = xmalloc(4);
	snprintf((char *) data, 4, "%d", (int) buf[offset]);
	rr_add_rdata(rdata_create(data, strlen((char *) data)), rr);
	xfree(data);
	return 1;
}

/**
 * Puts the 32-bit Internet address from the wire at position offset in the given rr
 */
size_t
wire2rdata_a(uint8_t *buf, size_t offset, struct t_rr *rr)
{
	uint8_t *data;
	
	data = (uint8_t *) bin2ipv4(&buf[offset]);
	rr_add_rdata(rdata_create(data, 15), rr);
	xfree(data);
	return 4;
}

/**
 * Puts the 128-bit Internet address from the wire at position offset in the given rr
 */
size_t
wire2rdata_aaaa(uint8_t *buf, size_t offset, struct t_rr *rr)
{
	uint8_t *data;
	struct t_rdata *rdata;
	
	data = (uint8_t *) bin2ipv6(&buf[offset]);
	rdata = rdata_create(data, strlen((char *) data));
	rr_add_rdata(rdata, rr);
	xfree(data);
	return 16;
}

/**
 * Places the given number of bytes from the wire as an rdata in the rr
 * (as a string containing hex numbers)
 */
size_t 
wire2rdata_hex(uint8_t *buf, size_t offset, struct t_rr *rr, size_t len)
{
	char *str;
	size_t i;
	uint8_t int8;
	
	str = xmalloc((len * 2 + 1)*sizeof(char));
	for (i=0; i<len; i++) {
		int8 = buf[offset+i];
		snprintf(&str[i*2], 3, "%02x", (unsigned int) int8);
	}
	rr_add_rdata(rdata_create((uint8_t *) str, (len*2) * sizeof(char)), rr);
	xfree(str);
	return (len*2) * sizeof(char);
}

/**
 * Places the given number of bytes from the wire as an rdata in the rr
 * (as unknown rdata, see rfc 3597)
 */
size_t 
wire2rdata_unknown(uint8_t *buf, size_t offset, struct t_rr *rr, size_t len)
{
	char *str;
	size_t i;
	uint8_t int8;
	size_t tmp;
	
	str = xmalloc((len * 2 + 10)*sizeof(char));
	
	snprintf(str, 10, "\\# %d ", (int) len);
	tmp = strlen(str);
	
	for (i = 0; i < len; i++) {
		int8 = buf[offset + i];
		snprintf(&str[tmp + i * 2], 3, "%02x", (unsigned int) int8);
	}
	rr_add_rdata(rdata_create((uint8_t *) str, (len * 2 + tmp) * sizeof(char)), rr);
	xfree(str);
	return (len * 2 + tmp) * sizeof(char);	
}



/**
 * Copies the string from the wire at the specified offset and puts it in the given rr
 * format is <size>[<byte>]*
 */
size_t
wire2rdata_str(uint8_t *buf, size_t offset, struct t_rr *rr)
{
	//int len = buf[offset];
	// escape escapable chars!
	size_t i, j = 0;

	// max len 2*the string (if every char should be escaped)
	uint8_t *rdstr = xmalloc(buf[offset]*2);

	for (i=0; i < (size_t) buf[offset]; i++) {
		if (
			buf[offset+i+1] == '\\' ||
			buf[offset+i+1] == '\"' ||
			buf[offset+i+1] == '\''
		) {
			rdstr[j] = (uint8_t) '\\';
			j++;
			rdstr[j] = buf[offset+i+1];
			j++;
		} else if (!isprint((int) buf[offset+i+1])) {
			rdstr[j] = (uint8_t) '\\';
			j++;
			snprintf((char *) &rdstr[j], 4, "%03d", (int) buf[offset+i+1]);
			j += 3;
		} else {
			rdstr[j] = buf[offset+i+1];
			j++;
		}
	}
	rr_add_rdata(rdata_create(rdstr, j), rr);

	xfree(rdstr);
	return i+1;
}

/**
 * Base64-encodes the data at the given position and adds it to the rr
 */
size_t
wire2rdata_base64(uint8_t *buf, size_t offset, struct t_rr *rr, size_t len)
{
	unsigned char *str;
	str = base64_encode((unsigned char *) &buf[offset], (int) len);
	rr_add_rdata(rdata_create((uint8_t *) str, strlen((char *) str)), rr);
	xfree(str);
	return len;
}

/** 
 * Convert the nsec rdata to a printable list of rr types 
 */
size_t
wire2rdata_nsec(uint8_t *buf, size_t offset, struct t_rr *rr, size_t len)
{
	/*
	 * window block # | # octect | bitlist
	 *
	 * actual types are wb# * 256
	 * then follow #octects octets with the bits
	 * each bit on is a type
	 * so: type = wb# * 256 + placement of bit
	 *
	 * everything is relative we only now where the first
	 * window block starts
	 */
	
	unsigned int wb;
	unsigned int octets;
	unsigned int i;
	char *nsec_str =  NULL;
	char *tmpstr;
	short b; /* has to be able to go < 0 */
	uint8_t j;
	uint16_t bit;
	unsigned int bit_pos;
	size_t memsize = 100;
	size_t strsize = 0;
	size_t partsize;

	wb      = (unsigned int) buf[offset]; /* the first window block */
	octets	= (unsigned int) buf[offset + 1]; /* how many octect are there */

	offset += 2; /* start of the actual window blocks */
	
	/* XXX TODO multiple windows */
	nsec_str = xmalloc(memsize);
	nsec_str[0] = '\0';

	bit_pos = 0;
	for(i = 0; i < octets; ++i) {
		j = buf[offset + i];
		/* reverse the sequence, network bit order starts at the
		 * left. XXX port error right here....
		 */
		for(b = 7; b >=  0; --b) {
			bit = bit_set(&j, (unsigned short) b) ;
			bit = (bit * bit_pos) + (wb * 256);
			if (bit) {
				tmpstr = typebyint(bit, ztypes);
				partsize = strlen(tmpstr);
				while (strsize + partsize + 1 > memsize) {
					memsize = memsize+memsize;
					nsec_str = xrealloc(nsec_str, memsize);
				}
				nsec_str = strcat(nsec_str, tmpstr);
				nsec_str = strcat(nsec_str, " ");
				strsize += partsize + 1;
				/*xfree(tmpstr);*/
			}
			bit_pos++;
		}
	}

	rr_add_rdata(rdata_create((uint8_t *)nsec_str, strlen(nsec_str)), rr);
	xfree(nsec_str);
	return len; 
}


/** 
 * Convert the wks rdata to a printable list of rr types 
 */
size_t
wire2rdata_wks(uint8_t *buf, size_t offset, struct t_rr *rr, size_t len)
{
	uint8_t int8;
	char *wks_str;
	char *tmp_str;
	size_t memsize = 100;
	size_t strsize = 0;
	size_t bit_nr = 0;
	size_t byte_nr = 0;
	unsigned short cur_bit_nr = 0;
	struct protoent *proto;
	struct servent *serv;
	
	int8 = buf[offset];
	offset++;
	
	wks_str = xmalloc(memsize);
	
	/* try getprotobynumber */
	proto = getprotobynumber((int) int8);
	if (proto && proto->p_name) {
		snprintf(wks_str, memsize, "%s ", proto->p_name);
	} else {
		snprintf(wks_str, memsize, "%u ", (unsigned int) int8);
	}
	
	strsize += strsize + strlen(wks_str);
	for (byte_nr = 0; byte_nr < len - 1; byte_nr++) {
		for (cur_bit_nr=7; cur_bit_nr < 8; cur_bit_nr--) {
			if (bit_set(&buf[offset + byte_nr], cur_bit_nr)) {
				tmp_str = xmalloc(memsize);
				
				if (proto && proto->p_name) {
					serv = getservbyport((int) htons(bit_nr), proto->p_name);
				} else {
					serv = getservbyport((int) htons(bit_nr), NULL);
				}
				if (serv && serv->s_name) {
					snprintf(tmp_str, memsize, "%s ", serv->s_name);
				} else {
					snprintf(tmp_str, memsize, "%u ", (unsigned int) bit_nr);
				}
				strsize += strlen(tmp_str);
				if (strsize >= memsize) {
					memsize = memsize * 2;
					wks_str = xrealloc(wks_str, memsize);
				}
				wks_str = strcat(wks_str, tmp_str);
				xfree(tmp_str);

			}
			bit_nr++;
		}
	}
	rr_add_rdata(rdata_create((uint8_t *)wks_str, strlen(wks_str)), rr);
	xfree(wks_str);
	return len;
}

/**
 */
size_t 
wire2rdata_nsap(uint8_t *buf, size_t offset, struct t_rr *rr, size_t len)
{
	char *str;
	size_t i;
	uint8_t int8;
	
	str = xmalloc((len * 2 + 2)*sizeof(char));
	str[0] = '0';
	str[1] = 'x';
	for (i=0; i<len; i++) {
		int8 = buf[offset+i];
		snprintf(&str[i*2 + 2], 3, "%02x", (unsigned int) int8);
	}
	rr_add_rdata(rdata_create((uint8_t *) str, (len*2 + 2) * sizeof(char)), rr);
	xfree(str);
	return (len*2) * sizeof(char);
}

static inline long
power(long a, long b) {
        long result = 1;
        while (b > 0) {
                if (b & 1) {
                        result *= a;
                        if (b == 1) {
                                return result;
                        }
                }
                a *= a;
                b /= 2;
        }
        return result;
}


size_t 
wire2rdata_loc(uint8_t *buf, size_t offset, struct t_rr *rr, size_t len)
{
	/* we could do checking (ie degrees < 90 etc)? */
	uint8_t version = buf[offset];
	uint8_t size;
	uint8_t horizontal_precision;
	uint8_t vertical_precision;
	uint32_t int32;
	uint32_t longitude;
	uint32_t latitude;
	uint32_t altitude;
	char northerness;
	char easterness;
	uint32_t h;
	uint32_t m;
	double s;
	long value, unit, meters;
	char str[100];
	char *total = xmalloc(100);
	uint32_t equator = (uint32_t) power(2, 31);
	size_t str_len;

	memset(str, 0, 100);
	total[0] = '\0';
	
	if (version == 0) {
		size = buf[offset + 1];
		horizontal_precision = buf[offset + 2];
		vertical_precision = buf[offset + 3];
		
		memcpy(&int32, &buf[offset + 4], 4);
		latitude = ntohl(int32);
		memcpy(&int32, &buf[offset + 8], 4);
		longitude = ntohl(int32);
		memcpy(&int32, &buf[offset + 12], 4);
		altitude = ntohl(int32);
		
		if (latitude > equator) {
			northerness = 'N';
			latitude = latitude - equator;
		} else {
			northerness = 'S';
			latitude = equator - latitude;
		}
		h = latitude / (1000 * 60 * 60);
		latitude = latitude % (1000 * 60 * 60);
		m = latitude / (1000 * 60);
		latitude = latitude % (1000 * 60);
		s = (double) latitude / 1000.0;
		snprintf(str, 17, "%u %u %0.3f %c ", (unsigned int) h, (unsigned int)  m, s, northerness);
		total = strcat(total, str);

		if (longitude > equator) {
			easterness = 'E';
			longitude = longitude - equator;
		} else {
			easterness = 'W';
			longitude = equator - longitude;
		}
		h = longitude / (1000 * 60 * 60);
		longitude = longitude % (1000 * 60 * 60);
		m = longitude / (1000 * 60);
		longitude = longitude % (1000 * 60);
		s = (double) longitude / (1000.0);
		snprintf(str, 17, "%u %u %0.3f %c ", (unsigned int)  h, (unsigned int)  m, s, easterness);
		total = strcat(total, str);

		meters = (long) altitude - 10000000;
		snprintf(str, 16, "%u", (unsigned int)  meters / 100);
		total = strcat(total, str);
		if (meters % 100 != 0) {
			snprintf(str, 16, ".%02u",  (unsigned int) meters % 100);
			total = strcat(total, str);
		}
		snprintf(str, 3, "m ");
		total = strcat(total, str);


		value = (short) ((size & 0xf0) >> 4);
		unit = (short) (size & 0x0f);
		meters = value * power(10, unit);
		snprintf(str, 16, "%u", (unsigned int)  meters / 100);
		total = strcat(total, str);
		if (meters % 100 != 0) {
			snprintf(str, 16, ".%02u", (unsigned int)  meters % 100);
			total = strcat(total, str);
		} 
		snprintf(str, 16, "m ");
		total = strcat(total, str);

		value = (short) ((horizontal_precision & 0xf0) >> 4);
		unit = (short) (horizontal_precision & 0x0f);
		meters = value * power(10, unit);
		snprintf(str, 16, "%u",  (unsigned int) meters / 100);
		total = strcat(total, str);
		if (meters % 100 != 0) {
			snprintf(str, 16, ".%02u",  (unsigned int) meters % 100);
			total = strcat(total, str);
		}
		snprintf(str, 3, "m ");
		total = strcat(total, str);

		value = (long) ((vertical_precision & 0xf0) >> 4);
		unit = (long) (vertical_precision & 0x0f);
		meters = value * power(10, unit);
		snprintf(str, 16, "%u",  (unsigned int) meters / 100);
		total = strcat(total, str);
		if (meters % 100 != 0) {
			snprintf(str, 16, ".%02u",  (unsigned int) meters % 100);
			total = strcat(total, str);
		} 
		snprintf(str, 3, "m ");
		total = strcat(total, str);

		str_len = strlen(total) + 1;

		rr_add_rdata(rdata_create((uint8_t *) total, str_len * sizeof(char)), rr);
		xfree(total);
		return len;
	} else {
		return 0;
	}



}

size_t 
wire2rdata_apl(uint8_t *buf, size_t offset, struct t_rr *rr, size_t len)
{
	char str[100];
	char *total;
	size_t pos;
	uint16_t address_family;
	uint8_t prefix;
	uint8_t negation;
	uint8_t afdlength;
	short i;
	total = xmalloc(100);
	total[0] = '\0';
	pos = 0;
	while (pos < len - 4) {
		memcpy(&address_family, &buf[offset + pos], 2);
		address_family = ntohs(address_family);
		
		if (address_family == 1) {
			prefix = buf[offset + pos + 2];
			negation = (buf[offset + pos + 3] & 0x80) >> 7;
			afdlength = buf[offset + pos + 3] & 0x7f;
			pos += 4;
			
			if (negation) {
				total = strcat(total, "!");
			}
			
			snprintf(str, 4, "%u:", (unsigned int) address_family);
			total = strcat(total, str);
			
			for (i = 0; i < (short) afdlength; i++) {
				if (i > 0) {
					total = strcat(total, ".");
				}
				snprintf(str, 4, "%u", (unsigned int) buf[offset + pos + i]);
				total = strcat(total, str);
			}
			while (i < 4) {
				if (i > 0) {
					total = strcat(total, ".");
				}
				total = strcat(total, "0");
				i++;
			}

			snprintf(str, 6, "/%u ", (unsigned int) prefix);
			total = strcat(total, str);

			pos += afdlength;
		} else if (address_family == 2) {
			prefix = buf[offset + pos + 2];
			negation = (buf[offset + pos + 3] & 0x80) >> 7;
			afdlength = buf[offset + pos + 3] & 0x7f;
			pos += 4;
			
			if (negation) {
				total = strcat(total, "!");
			}
			
			snprintf(str, 4, "%u:", (unsigned int)address_family);
			total = strcat(total, str);
			
			for (i = 0; i < (short) afdlength; i++) {
				if (i % 2 == 0 && i > 0) {
					total = strcat(total, ":");
				}
				snprintf(str, 4, "%02x", (unsigned int)buf[offset + pos + i]);
				total = strcat(total, str);
			}
			while (i < 16) {
				if (i % 2 == 0 && i > 0) {
					total = strcat(total, ":");
				}
				total = strcat(total, "00");
				i++;
			}

			snprintf(str, 6, "/%u ", (unsigned int)prefix);
			total = strcat(total, str);

			pos += afdlength;
			
		} else {
			fprintf(stderr, "Unknown address family used in APL data\n");
			goto error;
		}
		
	}
	rr_add_rdata(rdata_create((uint8_t *) total, strlen(total)), rr);
	xfree(total);
	return (len*2) * sizeof(char);

	error:
	xfree(total);
	return 0;
}




/**
 * Converts wire data to rdata
 * returns the length of the data on the buffer
 *
 * The data is currently stored as follows:
 * - Integers (16 and 32 bits) are converted to host byte order
 * - Strings are copied (the length is stored in rdata->length, the string itself in rdata->data
 * - dnames are read and stored as strings (canonical)
 * - bitmaps and 8-bit ints are copied as-is
 * - Inet addresses are stored as strings (for instance ipv4 as "127.0.0.1")
 */
size_t
wire2rdata(uint8_t *buf, size_t offset, struct t_rr *rr)
{
	uint16_t int16, length;
	size_t origoffset = offset;
	size_t i = 0;
	
	assert(rr != NULL);
	assert(rr->type);
	assert(buf != NULL);
	
	/* read the length */
	memcpy(&int16, &buf[offset], 2);
	length = ntohs(int16);
	
	offset += 2;
	/* depending on the type, the data should be stored differently */
	while (offset - origoffset < length+2) {
		switch(rr_rdata_types[rr->type][i]) {
			case RDATA_TYPE_DNAME:
				offset += wire2rdata_dname(buf, offset, rr);
				break;
			case RDATA_TYPE_INT48:
				offset += wire2rdata_int48(buf, offset, rr);
				break;
			case RDATA_TYPE_INT32:
				offset += wire2rdata_int32(buf, offset, rr);
				break;
			case RDATA_TYPE_INT16:
			case RDATA_TYPE_TYPE:
			case RDATA_TYPE_CLASS:
			case RDATA_TYPE_CERTTYPE:
				offset += wire2rdata_int16(buf, offset, rr);
				break;
			case RDATA_TYPE_INT8:
			case RDATA_TYPE_ALGO:
				offset += wire2rdata_int8(buf, offset, rr);
				break;
			case RDATA_TYPE_A:
				offset += wire2rdata_a(buf, offset, rr);
				break;
			case RDATA_TYPE_AAAA:
				offset += wire2rdata_aaaa(buf, offset, rr);
				break;
			case RDATA_TYPE_ANY:
				warning("ANY type rdata?!?");
				exit(EXIT_FAILURE);
				break;
			case RDATA_TYPE_STR:
				offset += wire2rdata_str(buf, offset, rr);
				break;
			case RDATA_TYPE_BASE64:
				offset += wire2rdata_base64(buf, offset, rr, length-(offset-origoffset)+2);
				break;
			case RDATA_TYPE_HEX:
				offset += wire2rdata_hex(buf, offset, rr, length-(offset-origoffset)+2);
				break;
			case RDATA_TYPE_NSEC:
				offset += wire2rdata_nsec(buf, offset, rr, length-(offset-origoffset)+2);
				break;
			case RDATA_TYPE_WKS:
				offset += wire2rdata_wks(buf, offset, rr, length-(offset-origoffset)+2);
				break;
			case RDATA_TYPE_NSAP:
				offset += wire2rdata_nsap(buf, offset, rr, length-(offset-origoffset)+2);
				break;
			case RDATA_TYPE_LOC:
				offset += wire2rdata_loc(buf, offset, rr, length-(offset-origoffset)+2);
				break;
			case RDATA_TYPE_APL:
				offset += wire2rdata_apl(buf, offset, rr, length-(offset-origoffset)+2);
				break;
			default:
				warning("Unknown rdata type (value %d, rrtype: %d, rdata nr: %d at %d)\n",
						rr_rdata_types[rr->type][i], rr->type,
						i, offset);
				offset += wire2rdata_unknown(buf, offset, rr, length-(offset-origoffset)+2);
				
		}
		i++;
	}
	return length+2;
}

size_t 
rdata2wire_dname(struct t_rdata *rdata, uint8_t *buf, size_t offset, const size_t buf_len)
{
	return name2wire(rdata, buf, offset, buf_len);
}

size_t
rdata2wire_int48(struct t_rdata *rdata, uint8_t *buf, size_t offset, const size_t buf_len)
{
	if (offset + 6 <= buf_len) {
		uint64_t int64 = rdata2uint64(rdata);
		int64 = htonl(int64);
		/* 64 bits is 2 bytes too much, in network order thats the first 2 bytes */
		memcpy(&buf[offset], &int64+2, 6);
		return 6;
	} else {
		fprintf(stderr, "Buffer too small in rdata2wire_int48\n");
		exit(1);
	}
}

size_t
rdata2wire_int32(struct t_rdata *rdata, uint8_t *buf, size_t offset, const size_t buf_len)
{
	if (offset + 4 <= buf_len) {
		uint32_t int32 = rdata2uint32(rdata);
		int32 = htonl(int32);
		memcpy(&buf[offset], &int32, 4);
		return 4;
	} else {
		fprintf(stderr, "Buffer too small in rdata2wire_int32\n");
		exit(1);
	}
}

size_t
rdata2wire_int16(struct t_rdata *rdata, uint8_t *buf, size_t offset, const size_t buf_len)
{
	if (offset + 2 <= buf_len) {
		uint16_t int16 = rdata2uint16(rdata);
		int16 = htons(int16);
		memcpy(&buf[offset], &int16, 2);
		return 2;
	} else {
		fprintf(stderr, "Buffer too small in rdata2wire_int16\n");
		exit(1);
	}
}

size_t
rdata2wire_int8(struct t_rdata *rdata, uint8_t *buf, size_t offset, const size_t buf_len)
{
	if (offset + 1 <= buf_len) {
		uint8_t int8 = rdata2uint8(rdata);
		buf[offset] = int8;
		return 1;
	} else {
		fprintf(stderr, "Buffer too small in rdata2wire_int8\n");
		exit(1);
	}
}

size_t
rdata2wire_a(struct t_rdata *rdata, uint8_t *buf, size_t offset, const size_t buf_len)
{
	char *str, *o_str;
	uint8_t a, b, c, d;

	if (offset + 4 <= buf_len) {
		o_str = rdata2str(rdata);
		
		str = o_str;
		a = (uint8_t) atoi(str);
		str = index(str, '.')+1;
		b = (uint8_t) atoi(str);
		str = index(str, '.')+1;
		c = (uint8_t) atoi(str);
		str = index(str, '.')+1;
		d = (uint8_t) atoi(str);

		buf[offset] = a;
		buf[offset+1] = b;
		buf[offset+2] = c;
		buf[offset+3] = d;
		
		xfree(o_str);
		return 4;
	} else {
		fprintf(stderr, "Buffer too small in rdata2wire_a\n");
		exit(1);
	}
}

size_t
rdata2wire_aaaa(struct t_rdata *rdata, uint8_t *buf, size_t offset, const size_t buf_len)
{
	/* TODO */
	return 0;
}

size_t
rdata2wire_hex(struct t_rdata *rdata, uint8_t *buf, size_t offset, const size_t buf_len)
{
	return hexstr2bin((char *) rdata->data, (int) rdata->length, buf, offset, buf_len);
}

size_t
rdata2wire_nsec(struct t_rdata *rdata, uint8_t *buf, size_t offset, const size_t buf_len)
{
	size_t omemsize;
	size_t memsize = 100;
	uint16_t largest = 0;
	uint16_t current;
	uint8_t *bitmap = xmalloc(memsize);
	uint8_t window_block;
	uint8_t window_len;
	uint8_t cur;
	
	size_t orig_offset = offset;
	char *substr;
	size_t start = 0;
	size_t end = 0;
	
	memset(bitmap, 0, memsize);

	while (end < rdata->length) {
		while (end < rdata->length && rdata->data[end] != ' ') {
			end++;
		}

		substr = xstrndup((char *) &(rdata->data[start]), end-start);

		current = intbyname(substr, ztypes);
		
		if (current > largest) {
			largest = current;
		}
		
		/*printf("nr: %d\n", current);*/
		if ((size_t) (current / 8) > memsize) {
			omemsize = memsize;
			while ((size_t) (current / 8) > memsize) {
				memsize = memsize * 2;
			}
			bitmap = xrealloc(bitmap, memsize);
			memset(&bitmap[omemsize], 0, memsize - omemsize);
		}
		
		set_bit(&bitmap[current / 8], (unsigned short) (7 - (current % 8)), 1);
		
		xfree(substr);
		if ( end < rdata->length) {
			start = end + 1;
			end = start;
		}
	}
	
	for (window_block = 0; (uint16_t) window_block * 256 <= largest; window_block++) {
		window_len = 0;
		
		for (cur = 0; cur < 255 && cur + window_block*256 < memsize; cur++) {

			if (bit_set(&bitmap[((256*window_block)+cur)/8], (unsigned short) (cur % 8))) {
				window_len = (cur / 8) + 1;
			}
		}
		if (window_len > 0) {
			/*printf("window block %d!\n", window_block);*/
			if (offset + 2 + window_len < buf_len) {
				buf[offset] = window_block;
				offset++;
				buf[offset] = window_len;
				offset++;
				memcpy(&buf[offset], &bitmap[32*window_block], window_len);
				offset += window_len;
			} else {
				fprintf(stderr, "Buffer too small in rdata2wire_nsec\n");
				exit(1);
			}
		}
	}
	return offset - orig_offset;
}

size_t
rdata2wire_wks(struct t_rdata *rdata, uint8_t *buf, size_t offset, const size_t buf_len)
{
	struct protoent *protocol;
	struct servent *service;

	size_t omemsize;
	size_t memsize = 100;
	int largest = 0;
	int current;
	uint8_t *bitmap = xmalloc(memsize);

	char *protostr;
	char *substr;
	size_t start = 0;
	size_t end = 0;
	
	memset(bitmap, 0, memsize);
	
	while (end < rdata->length && rdata->data[end] != ' ') {
		end++;
	}

	protostr = xstrndup((char *) &(rdata->data[start]), end-start);
	/*printf("PROTOSTR: %s\n", protostr);*/
	protocol = getprotobyname(protostr);
	
	if ( end < rdata->length) {
		start = end + 1;
		end = start;
	}
	
	while (end < rdata->length) {
		while (end < rdata->length && rdata->data[end] != ' ') {
			end++;
		}

		substr = xstrndup((char *) &(rdata->data[start]), end-start);
		/*printf("SUBSTR: %s\n", substr);*/
		service = getservbyname(substr, protostr);
		/*printf("service: %d\n", htons(service->s_port));*/
		current = (int) htons((uint16_t) service->s_port);
		
		if (current > largest) {
			largest = current;
		}
		if ((size_t) (current / 8) > memsize) {
			omemsize = memsize;
			while ((size_t) (current / 8) > memsize) {
				memsize = memsize * 2;
			}
			bitmap = xrealloc(bitmap, memsize);
			memset(&bitmap[omemsize], 0, memsize - omemsize);
		}
		
		set_bit(&bitmap[current / 8], (unsigned short) (7 - (current % 8)), 1);
		
		xfree(substr);
		if ( end < rdata->length) {
			start = end + 1;
			end = start;
		}
	}

	if (offset + 1 + ((largest / 8) + 2) <= buf_len) {
		buf[offset] = (uint8_t) protocol->p_proto;
		offset++;
		xfree(protostr);
		
		/* +1 for protocol byte, +1 for /8-off-by-1 */
		memcpy(&buf[offset], bitmap, (size_t) ((largest / 8) + 2));
		return (size_t) (largest / 8) + 2;
	} else {
		fprintf(stderr, "Buffer too small in rdata2wire_wks\n");
		exit(1);
	}
}

size_t
rdata2wire_certtype(struct t_rdata *rdata, uint8_t *buf, size_t offset, const size_t buf_len)
{
	char *certstr = rdata2str(rdata);
	uint16_t int16 = intbyname(certstr, cert_types);
	if (offset + 2 <= buf_len) {
		int16 = htons(int16);
		memcpy(&buf[offset], &int16, 2);
		return 2;
	} else {
		fprintf(stderr, "Buffer too small in rdata2wire_certtype\n");
		exit(1);
	}

}

size_t
rdata2wire_unknown(struct t_rdata *rdata, uint8_t *buf, size_t offset, const size_t buf_len)
{
	int size;
	size_t result;
	char *str = rdata2str(rdata);
	
	if (rdata->length < 4) {
		error("Unknown rdata < 4, which should not happen: %s", str);
		return 0;
	}
	if (!(
		rdata->data[0] == '\\' &&
		rdata->data[1] == '#' &&
		rdata->data[2] == ' '
	)) {
		warning("Unknown rdata does not start with \\# <size>: %s", str);
		return 0;
	}
	
	size = atoi(&str[3]);
	result = hexstr2bin(&str[5], size*2, buf, offset, buf_len);

	xfree(str);
	return result;
}

size_t
rdata2wire_str(struct t_rdata *rdata, uint8_t *buf, size_t offset, const size_t buf_len)
{
	size_t result;
	char *str = rdata2str(rdata);
	size_t len;
	/* we don't enforce 256 anywhere as the max */
	/* this function needs MAJOR redesign */
	len = strlen(str);
	if (len > 255) {
		len = 255;
	}
	
	if (offset + 1 + len <= buf_len) {
		buf[offset] = len;
		offset++;
		strncpy((char *) &buf[offset], str, buf_len-offset);
		result = 1 + len;
		xfree(str);
		return result;
	} else {
		fprintf(stderr, "Buffer too small in rdata2wire_str\n");
		exit(1);
	}
}

size_t
rdata2wire_base64(struct t_rdata *rdata, uint8_t *buf, size_t offset, const size_t buf_len)
{
	char *str = rdata2str(rdata);
	char *dec;
	size_t size;
	dec = base64_decode((unsigned char *) str, (int) strlen(str), &size);
	if (offset + size <= buf_len) {
		memcpy(&buf[offset], dec, size);
	} else {
		fprintf(stderr, "Buffer too small in rdata2wire_base64\n");
		exit(1);
	}
	xfree(str);
	xfree(dec);
	return size;
}

size_t
rdata2wire_nsap(struct t_rdata *rdata, uint8_t *buf, size_t offset, const size_t buf_len)
{

	char *str = rdata2str(rdata);
	size_t size = 0;
	
	if (str[0] == '0' && str[1] == 'x') {
		size = hexstr2bin(str + 2, (int) (strlen(str) - 2), buf, offset, buf_len);
	} else {
		fprintf(stderr, "Parse error in rdata2wire_nsap, element does not start with 0x");
	}
	xfree(str);
	return size;
}
	

/* not the most efficient way to do it and it could use some more
   error checking */
size_t
rdata2wire_loc(struct t_rdata *rdata, uint8_t *buf, size_t offset, const size_t buf_len)
{
	char *str = rdata2str(rdata);
	
	uint8_t version = 0;
	uint8_t loc_size;
	uint8_t horizontal_precision;
	uint8_t vertical_precision;
	uint32_t latitude;
	uint32_t longitude;
	uint32_t altitude;
	
	uint32_t equator = (uint32_t) power(2, 31);
	
	unsigned int a,b,e,f,i,j,k,l;
	float c,g;
	char d,h;
	uint8_t tmp1, tmp2;
	uint32_t int32;

	/* not the most efficient way to do it and it could be more lenient */
	tmp1 = (uint8_t) sscanf(str, "%u %u %f %c %u %u %f %c %um %um %um %um", &a, &b, &c, &d, &e, &f, &g, &h, &i, &j, &k, &l);
	/*printf("FOUND: %u %u %f %c %u %u %f %c %um %um %um %um\n", a,b,c,d,e,f,g,h,i,j,k,l);*/

	if (tmp1 < 12) {
		fprintf(stderr, "Parse error in rdata2wire_loc, not enough elements read (all values should be specified)\n");
		goto error;
	}

	latitude = a * (1000*60*60) + b * (1000*60) + ((uint32_t) (c*1000));
	if (d == 'N') {
		latitude += equator;
	} else if (d == 'S') {
		latitude = equator - latitude;
	} else {
		fprintf(stderr, "Parse error in wire2rdata_loc, N/S not read\n");
		goto error;
	}

	longitude = e * (1000*60*60) + f * (1000*60) + ((uint32_t) (g*1000));
	if (h == 'E') {
		longitude += equator;
	} else if (h == 'W') {
		longitude = equator - longitude;
	} else {
		fprintf(stderr, "Parse error in wire2rdata_loc, E/W not read\n");
		goto error;
	}
	
	altitude = (uint32_t) i + 10000000L;
	
	/* size, hp and mp are encoded in 2 nibbles, as a * 10 ^ b
	 * a and b must be 0-9
	 */
	j = j * 100;
	tmp2 = 0;
	while (j % 10 == 0) {
		tmp2++;
		j = j / 10;
	}
	tmp1 = (uint8_t) j;
	
	loc_size = ((tmp1 & 0x0f) << 4) | (tmp2 & 0x0f);
	
	k = k * 100;
	tmp2 = 0;
	while (k % 10 == 0) {
		tmp2++;
		k = k / 10;
	}
	tmp1 = (uint8_t) k;
	horizontal_precision = ((tmp1 & 0x0f) << 4) | (tmp2 & 0x0f);

	l = l * 100;
	tmp2 = 0;
	while (l % 10 == 0) {
		tmp2++;
		l = l / 10;
	}
	tmp1 = (uint8_t) l;
	vertical_precision = ((tmp1 & 0x0f) << 4) | (tmp2 & 0x0f);
	
	if (offset + 16 < buf_len) {
		buf[offset] = version;
		buf[offset + 1] = loc_size;
		buf[offset + 2] = horizontal_precision;
		buf[offset + 3] = vertical_precision;
		int32 = htonl(latitude);
		memcpy(&buf[offset + 4], &int32, 4);
		int32 = htonl(longitude);
		memcpy(&buf[offset + 8], &int32, 4);
		int32 = htonl(altitude);
		memcpy(&buf[offset + 12], &int32, 4);
	} else {
		fprintf(stderr, "Buffer too small in rdata2wire_loc\n");
		exit(1);
	}

	xfree(str);
	return 16;

	error:
	xfree(str);
	return 0;
}

size_t
rdata2wire_apl(struct t_rdata *rdata, uint8_t *buf, size_t offset, const size_t buf_len)
{

	char *str = rdata2str(rdata);
	size_t pos = 0;
	size_t len = strlen(str);
	size_t orig_offset = offset;
	
	//uint16_t address_family;
	unsigned int address_family;
	//uint16_t prefix;
	unsigned int prefix;
	uint8_t negation;
	uint8_t afdlength;
	unsigned int data_a[4];
	int data_aaaa[16];
	int tmp;
	
	while (pos < len) {
		if (str[pos] == '!') {
			negation = 1;
			pos++;
		} else {
			negation = 0;
		}
		
		tmp = sscanf(&str[pos], "%u:", &address_family);
		if (tmp != 1 && tmp != 2) {
			fprintf(stderr, "Unknown address family in apl rdata\n");
			goto error;
		}

		if (address_family == 1) {
			memset(data_a, 0, 4);
			
			address_family = (unsigned int) htons(address_family);
			while(str[pos] != ':') {
				pos++;
			}
			pos++;
			
			tmp = sscanf(&str[pos], "%u.%u.%u.%u", &data_a[0], &data_a[1], &data_a[2], &data_a[3]);
			if (tmp < 1 || tmp > 5) {
				fprintf(stderr, "Not enough data in A address (%d elements read in \'%s\'\n", tmp, &str[pos]);
				goto error;
			} else {
				afdlength = (uint8_t) tmp;
				while(afdlength > 0 && data_a[afdlength - 1] == 0) {
					afdlength--;
				}
			}
			
			while(str[pos] != '/') {
				pos++;
			}
			pos++;
			tmp = sscanf(&str[pos], "%u", &prefix);
			if (tmp != 1) {
				fprintf(stderr, "Prefix not found\n");
				goto error;
			}

			while(str[pos] != ' ') {
				pos++;
			}
			pos++;
			
			if (offset + 4 + afdlength <= buf_len) {
				buf[offset] = 0;
				buf[offset + 1] = 0x01;
				buf[offset + 2] = (uint8_t) prefix;
				buf[offset + 3] = (negation << 7) | afdlength;
				offset += 4;
				for (tmp = 0; tmp < (int) afdlength; tmp++) {
					buf[offset + tmp] = (uint8_t) data_a[tmp];
				}
				offset += afdlength;
			} else {
				fprintf(stderr, "Buffer too small in rdata2wire_apl\n");
				exit(1);
			}
		} else {
			memset(data_aaaa, 0, 4);
			
			address_family = (unsigned int) htons(address_family);
			while(str[pos] != ':') {
				pos++;
			}
			pos++;
			
			tmp = sscanf(&str[pos], "%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:", &data_aaaa[0], &data_aaaa[1], &data_aaaa[2], &data_aaaa[3], &data_aaaa[4], &data_aaaa[5], &data_aaaa[6], &data_aaaa[7], &data_aaaa[8], &data_aaaa[9], &data_aaaa[10], &data_aaaa[11], &data_aaaa[12], &data_aaaa[13], &data_aaaa[14], &data_aaaa[15]);
			if (tmp == 0 || tmp > 16) {
				fprintf(stderr, "Not enough data in AAAA address\n");
				goto error;
			} else {
				afdlength = (uint8_t) tmp;
				while(afdlength > 0 && data_aaaa[afdlength - 1] == 0) {
					afdlength--;
				}
			}
			
			while(str[pos] != '/') {
				pos++;
			}
			pos++;
			
			sscanf(&str[pos], "%u", &prefix);
			
			while(str[pos] != ' ') {
				pos++;
			}
			pos++;
			
			if (offset + 4 + afdlength <= buf_len) {
				buf[offset] = 0;
				buf[offset + 1] = 0x02;
				buf[offset + 2] = prefix;
				buf[offset + 3] = (negation << 7) | afdlength;
				offset += 4;
				for (tmp = 0; tmp < (int) afdlength; tmp++) {
					buf[offset + tmp] = (uint8_t) data_aaaa[tmp];
				}
				offset += afdlength;
			} else {
				fprintf(stderr, "Buffer too small in rdata2wire_apl\n");
				exit(1);
			}
		}
	}
	xfree(str);
	return offset - orig_offset;

	error: 
	fprintf(stderr, "Parse error in rdata2wire_apl\n");
	xfree(str);
	return 0;
}
	


/**
 * Converts the given t_rdata to wire format and places it in the buffer at
 * the specified offset
 * returns the total length of the wire data
 */
size_t
rdata2wire(struct t_rr *rr, uint8_t *buf, size_t offset, const size_t buf_len)
{
	size_t origoffset = offset;
	uint16_t int16;
	uint16_t i;

	offset += 2;
	for (i=0; i < rr->rdcount; i++) {
		switch(rr_rdata_types[rr->type][i]) {
			case RDATA_TYPE_DNAME:
				offset += rdata2wire_dname(rr->rdata[i], buf, offset, buf_len);
				break;
				
			case RDATA_TYPE_INT48:
				offset += rdata2wire_int48(rr->rdata[i], buf, offset, buf_len);
				break;
			
			case RDATA_TYPE_INT32:
				offset += rdata2wire_int32(rr->rdata[i], buf, offset, buf_len);
				break;
			
			case RDATA_TYPE_INT16:
			case RDATA_TYPE_TYPE:
			case RDATA_TYPE_CLASS:
				offset += rdata2wire_int16(rr->rdata[i], buf, offset, buf_len);
				break;
			
			case RDATA_TYPE_INT8:
			case RDATA_TYPE_ALGO:
				offset += rdata2wire_int8(rr->rdata[i], buf, offset, buf_len);
				break;
			
			case RDATA_TYPE_A:
				offset += rdata2wire_a(rr->rdata[i], buf, offset, buf_len);
				break;
				
			case RDATA_TYPE_AAAA:
				offset += rdata2wire_aaaa(rr->rdata[i], buf, offset, buf_len);
				break;
			
			case RDATA_TYPE_ANY:
				error("ANY type rdata?!?");
				exit(EXIT_FAILURE);
				break;
				
			case RDATA_TYPE_STR:
				offset += rdata2wire_str(rr->rdata[i], buf, offset, buf_len);
				break;
				
			case RDATA_TYPE_BASE64:
				offset += rdata2wire_base64(rr->rdata[i], buf, offset, buf_len);
				break;
				
			case RDATA_TYPE_HEX:
				offset += rdata2wire_hex(rr->rdata[i], buf, offset, buf_len);
				break;
			
			case RDATA_TYPE_NSEC:
				offset += rdata2wire_nsec(rr->rdata[i], buf, offset, buf_len);
				break;

			case RDATA_TYPE_WKS:
				offset += rdata2wire_wks(rr->rdata[i], buf, offset, buf_len);
				break;

			case RDATA_TYPE_CERTTYPE:
				offset += rdata2wire_certtype(rr->rdata[i], buf, offset, buf_len);
				break;

			case RDATA_TYPE_NSAP:
				offset += rdata2wire_nsap(rr->rdata[i], buf, offset, buf_len);
				break;

			case RDATA_TYPE_LOC:
				offset += rdata2wire_loc(rr->rdata[i], buf, offset, buf_len);
				break;

			case RDATA_TYPE_APL:
				offset += rdata2wire_apl(rr->rdata[i], buf, offset, buf_len);
				break;

			default:
				warning("Unknown rdata type (value %d, rr type %d (%s), rdata nr %d)", rr_rdata_types[rr->type][i], rr->type, namebyint(rr->type, ztypes), i);
				warning("\t\tdata: %s", rdata2str(rr->rdata[i]));
				offset += rdata2wire_unknown(rr->rdata[i], buf, offset, buf_len);
		}
	}
	/* set the length of the rdata at the beginning */
	int16 = offset - origoffset - 2;
	int16 = htons(int16);
	memcpy(&buf[origoffset], &int16, 2);
	
	return offset - origoffset;
}
