/* util.c
 *
 * contains some generic helper function, xmalloc, etc
 *
 * (c) NLnet Labs
 *
 * See the file LICENSE for the license
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include "common.h"

static void * xmalloc_ex(size_t, const char *, int);
static void * xrealloc_ex(void *, size_t, const char *, int);
static void xfree_ex(void *, const char *, int);
static char * xstrdup_ex(const char*, const char*, int);
static char * xstrndup_ex(const char*, size_t, const char*, int);

#ifdef DEBUGMEM
#define MAX_POINTERS 10000
char *pointers[10000];
int nr_of_pointers = 0;
long total_allocated = 0;
#endif

static void *
xmalloc_ex(size_t size, const char *file, int line)
{
	void *p;
	#ifdef DEBUGMEM
	int i;
	#endif
	
	p = malloc(size);

	if (p == NULL) {
		exit(EXIT_FAILURE);
	}

	#ifdef DEBUGMEM
	if (nr_of_pointers >= MAX_POINTERS) {
		error("Out of possible pointers... (%d, see util.c)", MAX_POINTERS);
		exit(EXIT_FAILURE);
	}

	for (i=0; i<nr_of_pointers; i++) {
		if (pointers[i] == p) {
			error("%p already allocated!", p);
			exit(EXIT_FAILURE);
		}
	}

	pointers[nr_of_pointers] = p;
	nr_of_pointers++;
	
	total_allocated += size;
	
	if (getenv("SHOWMEM")) {
		mesg("allocated %p (size: %d) at %s:%d", p, (int) size, (char *) file, line);
	}
	/* flush pipes for gdb readability */
	fflush(stdout);

	#endif
	return p;
}

static void *
xrealloc_ex(void *p, size_t size, const char *file, int line)
{
	char *np;

	#ifdef DEBUGMEM
	char *op = p;
	#endif
	
	np = realloc(p, size);
	
	#ifdef DEBUGMEM
	if (np != op) {
		int i;
		int found = 0;
		for (i = 0; i < nr_of_pointers; i++) {
			if (pointers[i] == op) {
				found = 1;
				pointers[i] = np;
			}
		}
		if (!found && p != NULL) {
			error("realloc: %p was not allocated or already freed (at %s:%u)", op, file, line);
			memcpy(0x0, "a", 1);
			exit(EXIT_FAILURE);
		} else if (!found) {
			np = malloc(size);

			if (np == NULL) {
				exit(EXIT_FAILURE);
			}
			pointers[nr_of_pointers] = np;
			nr_of_pointers++;
		}
	
	}
	/* flush pipes for gdb readability */
	if (getenv("SHOWMEM")) {
		mesg("reallocated %p to %p at %s:%u\n", op, np, file, line);
	}
	(void)fflush(stdout);
	
	#endif
	return np;
}

static void 
xfree_ex(void *ptr, const char *file, int line)
{
	#ifdef DEBUGMEM
	int i;
	int found = 0;
	
	for (i=0; i<nr_of_pointers; i++) {
		if (pointers[i] == ptr) {
			found = 1;
			for(; i<nr_of_pointers-1; i++) {
				pointers[i] = pointers[i+1];
			}
		}
	}
	if (!found) 
		warning("%p was not allocated with xmalloc (or already freed) called from %s:%d", ptr, file, line);
	else {
		nr_of_pointers--;
		/* flush pipes for gdb readability */
		if (getenv("SHOWMEM")) {
			mesg("freeing %p at %s:%d", ptr, file, line);
		}
	}
	fflush(stdout);
	
	#endif
	free(ptr);
	fflush(stdout);
}


static char *
xstrdup_ex(const char *s, const char *file, int line)
{
	return xstrndup_ex(s, strlen(s), file, line);
}

static char *
xstrndup_ex(const char *s, size_t n, const char *file, int line)
{
	char *res = xmalloc_ex(n + 1, file, line);
	size_t i;
	memset(res, 0, n + 1);
 	
	for (i=0; i < n; i++) {
		res[i] = s[i];
		if (s[i] == '\0') {
			i = n;
		}
	}
	return (char *) res;
}



#ifndef DEBUGMEM
/**
 * Allocates a size bytes
 * bail out if something fails here
 */
void *
xmalloc(size_t size)
{
	return xmalloc_ex(size, "", 0);
}

/**
 * Reallocate size bytes, with error checking
 */
void *
xrealloc(void *p, size_t size)
{
	return xrealloc_ex(p, size, "", 0);
}
/**
 * Frees the data
 */
void
xfree(void *ptr)
{
	xfree_ex(ptr, "", 0);
}

char *
xstrdup(const char *s)
{
	return xstrdup_ex(s, "", 0);
}

char *
xstrndup(const char *s, size_t n)
{
	return xstrndup_ex(s, n, "", 0);
}
#endif

void
gdbhook(void)
{}


/**
 * Prints the contents of the allocation list to stdout (for debugging
 * purposes only)
 */
void
show_mem(void)
{
	#ifdef DEBUGMEM
	if (getenv("SHOWMEM")) {
		int i;
		mesg("Memory stats:");
		mesg("%d unfreed pointers", nr_of_pointers);
		
		for(i=0; i<nr_of_pointers; i++) {
			mesg("%p %s", pointers[i], pointers[i]);
		}
		if (nr_of_pointers > 1) {
			error("you had pointers left...");
			exit(EXIT_FAILURE);
		}
	}
	#endif
}


